
// ****************************************************************************
//
//                        Page R - resistor meter
//
// ****************************************************************************
// Pins:
//  PA4 ........... R1 220
//  PA5 ........... R2 2K2
//  PD7 ........... R3 22K
//  PB3 (SWCLK) ... R4 220K
//  PD1 (SWIO) .... R5 2M2
//  PD6 ........... RC input, ADC6

// Measure range: 0.1 ohm - 100 Mohm, resolution 0.1 ohm, precision 5%
// At ranges of 8Mohm and above, accuracy may be worse than 5%.
// It can also be used to test diode polarity and LEDs. The RC output has positive
// polarity. The LED flashes at a frequency of 2 Hz.

// Do not leave the programmer pins connected during capacitor and resistor measurements.

#ifndef _PAGE_R_H
#define _PAGE_R_H

#ifdef __cplusplus
extern "C" {
#endif

// resistor pins
#define R1_GPIO		PA4
#define R2_GPIO		PA5
#define R3_GPIO		PD7
#define R4_GPIO		PB3
#define R5_GPIO		PD1

#define RC_GPIO		PD6

// resistor real value in 0.1 ohm
#define R1_RAW		2200		// 220
#define R2_RAW		22000		// 2K2
#define R3_RAW		220000		// 22K
#define R4_RAW		2200000		// 220K
#define R5_RAW		22000000	// 2M2

#define R0_RAW		216		// MCU internal resistor value in 0.1 ohm (= 21.6 ohm)

#define R_IN_SCALE	(1ll << 50)	// scale to subtract input impedance
#define R_IN		3500000000ll	// MCU ADC input impedance in 0.1 ohm, range 5-8 Mohm (= 500 Mohm)
#define R_IN_REC	(R_IN_SCALE / R_IN) // reciprocal value of input impedance
#define R_IN2		1400000000ll	// MCU ADC input impedance in 0.1 ohm, range >= 8 Mohm (= 140 Mohm)
#define R_IN2_REC	(R_IN_SCALE / R_IN2) // reciprocal value of input impedance

#define RC_ADC		6		// RC ADC input channel

// timing in [us]
//#define PERIOD	(16667*HCLK_PER_US) // 60 Hz (USA) ... 17 ms
#define PERIOD		(20000*HCLK_PER_US) // 50 Hz (Europe) ... 20 ms
					// measure period in ticks - It should be
					// equal to the period of the power grid,
					// to compensate for interference.

#define WAIT_STAB	(5*HCLK_PER_US)	// wait time to stabilise signal (in ticks)

// resistor index
enum {
	R1_INX = 0,
	R2_INX,
	R3_INX,
	R4_INX,
	R5_INX,

	R_NUM			// number of resistors
};

// resistor bit mask
#define R1_MASK		(1<<R1_INX)
#define R2_MASK		(1<<R2_INX)
#define R3_MASK		(1<<R3_INX)
#define R4_MASK		(1<<R4_INX)
#define R5_MASK		(1<<R5_INX)

// Resistor GPIO
extern const u8 R_Gpio[R_NUM];

// Resistor value in 0.1 ohm
extern const s32 R_Val[R_NUM];

// set all resistors OFF
void R_SetAllResOff();

// setup one resistor OFF by index
void R_SetResOff(int inx);

// setup one resistor HIGH by index
void R_SetResH(int inx);

// setup one resistor LOW by index
void R_SetResL(int inx);

// get ADC sample fast inline
INLINE u16 R_GetADC()
{
	// software start conversion
	ADC1_SwStart();

	// wait end of conversion
	while (!ADC1_EndGet()) {}

	// get conversion result
	return ADC1_Data();
}

// R initialize
void R_Init();

// R terminate
void R_Term();

// Page R (returns key PREV/NEXT)
u8 PageR();

#ifdef __cplusplus
}
#endif

#endif // _PAGE_R_H
