; ****************************************************************************
;
;                                   Memory
;
; ****************************************************************************

#include "include.inc"

	.text

; ----------------------------------------------------------------------------
;                          Memory copy in UP direction
; ----------------------------------------------------------------------------
; INPUT: R31:R30 (Z) = destination address
;	 R27:R26 (X) = source address
;	 R25:R24 = length (>= 0)
; OUTPUT: R31:R30 (Z) = destination address + length
;	  R27:R26 (X) = source address + length
; DESTROYS: R25, R24, R0
; STACK: 2 bytes
; ----------------------------------------------------------------------------
; MemCpySmall -  copy small block by 1 byte (more optimised for small blocks < 4 bytes)
; MemCpy - copy big block by 4 bytes (more optimised for big blocks >= 4 bytes)

; ----- [16] copy 4 bytes

MemCpy2:
	ld	r0,X+		; [2] byte 1
	st	Z+,r0		; [2]

	ld	r0,X+		; [2] byte 2
	st	Z+,r0		; [2]

	ld	r0,X+		; [2] byte 3
	st	Z+,r0		; [2]

	ld	r0,X+		; [2] byte 4
	st	Z+,r0		; [2]

; ----- [3,4] faster copy by 4 bytes
; [20] per loop, [5] per byte, big copy by 4 bytes [N/4*20+N%4*8+14]
;   0:[14] 1:[22] 2:[30] 3:[38] 4:[34] 5:[42] 6:[50] 7:[58] 8:[54] 9:[62] 10:[70] 11:[78] 12:[74]

.global MemCpy
MemCpy:
	sbiw	r24,4		; [2] data counter
	brpl	MemCpy2         ; [1,2] next 4 bytes

; ----- [4] jump to copy rest of data

	adiw	r24,4		; [2] return counter
	rjmp	MemCpySmall	; [2]

; ----- [4] copy 1 byte
; [8] per loop, [8] per byte, small copy by 1 byte [N*8+7]
;   0:[7] 1:[15] 2:[23] 3:[31] 4:[39] 5:[47] 6:[55] 7:[63] 8:[71] 9:[79] 10:[87]

MemCpy4:
	ld	r0,X+		; [2]
	st	Z+,r0		; [2]

; ----- [4,7] copy by 1 byte

.global MemCpySmall
MemCpySmall:
	sbiw	r24,1		; [2] data counter
	brpl	MemCpy4		; [1,2] next 1 byte
	ret			; [4]

; ----------------------------------------------------------------------------
;                          Memory copy in DOWN direction
; ----------------------------------------------------------------------------
; INPUT: R31:R30 (Z) = destination address + length
;	 R27:R26 (X) = source address + length
;	 R25:R24 = length (>= 0)
; OUTPUT: R31:R30 (Z) = destination address
;	  R27:R26 (X) = source address
; DESTROYS: R25, R24, R0
; STACK: 2 bytes
; ----------------------------------------------------------------------------
; MemCpyDSmall -  copy small block by 1 byte (more optimised for small blocks < 4 bytes)
; MemCpyD - copy big block by 4 bytes (more optimised for big blocks >= 4 bytes)

; ----- [16] copy 4 bytes

MemCpyD2:
	ld	r0,-X		; [2] byte 1
	st	-Z,r0		; [2]

	ld	r0,-X		; [2] byte 2
	st	-Z,r0		; [2]

	ld	r0,-X		; [2] byte 3
	st	-Z,r0		; [2]

	ld	r0,-X		; [2] byte 4
	st	-Z,r0		; [2]

; ----- [3,4] faster copy by 4 bytes
; [20] per loop, [5] per byte, big copy by 4 bytes [N/4*20+N%4*8+14]
;   0:[14] 1:[22] 2:[30] 3:[38] 4:[34] 5:[42] 6:[50] 7:[58] 8:[54] 9:[62] 10:[70] 11:[78] 12:[74]

.global MemCpyD
MemCpyD:
	sbiw	r24,4		; [2] data counter
	brpl	MemCpyD2	; [1,2] next 4 bytes

; ----- [4] jump to copy rest of data

	adiw	r24,4		; [2] return counter
	rjmp	MemCpyDSmall	; [2]

; ----- [4] copy 1 byte
; [8] per loop, [8] per byte, small copy by 1 byte [N*8+7]
;   0:[7] 1:[15] 2:[23] 3:[31] 4:[39] 5:[47] 6:[55] 7:[63] 8:[71] 9:[79] 10:[87]

MemCpyD4:
	ld	r0,-X		; [2]
	st	-Z,r0		; [2]

; ----- [7,8] copy by 1 byte

.global MemCpyDSmall
MemCpyDSmall:
	sbiw	r24,1		; [2] data counter
	brpl	MemCpyD4	; [1,2] next 1 byte
	ret			; [4]

; ----------------------------------------------------------------------------
;                          Get max. end of memory
; ----------------------------------------------------------------------------
; OUTPUT: R25:R24 = max. end of allocable memory
; STACK: 2 bytes
; ----------------------------------------------------------------------------

.global MemEnd
MemEnd:
	in	r24,_SFR_IO_ADDR(SPL)
	in	r25,_SFR_IO_ADDR(SPH) ; get stack pointer
	subi	r24,lo8(STACKRES)
	sbci	r25,hi8(STACKRES)
	ret

; ----------------------------------------------------------------------------
;                              Get free memory
; ----------------------------------------------------------------------------
; OUTPUT: R25:R24 = free memory
; DESTROYS: R0
; STACK: 4 bytes
; ----------------------------------------------------------------------------

.global MemFree
MemFree:
	rcall	MemEnd		; get max. end of memory
	ldd	r0,Y+DATA_MEMTOP ; get top of allocated memory
	sub	r24,r0		; free memory
	ldd	r0,Y+DATA_MEMTOP+1
	sbc	r25,r0
	ret

; ----------------------------------------------------------------------------
;                              Check free memory
; ----------------------------------------------------------------------------
; INPUT: R24 = required free memory
; OUTPUT: R25:R24 = new free memory (<0 if error)
;	  flag N set if memory error (use BRMI instruction)
; DESTROYS: R1, R0
; STACK: 6 bytes
; ----------------------------------------------------------------------------

.global MemCheck
MemCheck:
	mov	r1,r24		; R1 <- required free memory
	rcall	MemFree		; get free memory -> R25:R24
	sub	r24,r1
	sbc	r25,R_ZERO
MemCheck2:
	ret

; ----------------------------------------------------------------------------
;                Check space in calculator stack to take new value
; ----------------------------------------------------------------------------
; DESTROYS: R25, R24, R1, R0
; STACK: 8 bytes
; ----------------------------------------------------------------------------

;.global CalcCheck
;CalcCheck:
;	ldi	r24,5		; size of number

; MemCheckErr must follow

; ----------------------------------------------------------------------------
;                  Check free memory with error report
; ----------------------------------------------------------------------------
; INPUT: R24 = required free memory
; DESTROYS: R25, R24, R1, R0
; STACK: 8 bytes
; ----------------------------------------------------------------------------

.global MemCheckErr
MemCheckErr:
	rcall	MemCheck	; check memory
	brpl	MemCheck2	
	rjmp	MemErr		; memory error

; ----------------------------------------------------------------------------
;                     Repair pointers in allocable memory
; ----------------------------------------------------------------------------
; INPUT: R25:R24 = number of bytes to insert (>= 0) or delete (< 0)
;	 R26 = start offset of first pointer to update (OFF_CURLINE,...)
; DESTROYS: R27, R26, R1, R0
; STACK: 2 bytes
; ----------------------------------------------------------------------------

MemRep:

; ----- X <- start of pointers

	clr	r27
	subi	r26,lo8(-(PointBeg))	; X <- first pointer
	sbci	r27,hi8(-(PointBeg))
	rjmp	MemRep4

; ----- load pointer

MemRep2:ld	r0,X+
	ld	r1,X

; ----- shift pointer

	add	r0,r24
	adc	r1,r25

; ----- save new pointer

	sbiw	r26,1
	st	X+,r0
	st	X+,r1

; ----- next pointer

MemRep4:
	cpi	r26,lo8(PointEnd)
	brne	MemRep2
MemRep6:
	ret

; ----------------------------------------------------------------------------
;                       Insert space into allocable memory
; ----------------------------------------------------------------------------
; INPUT: R31:R30 (Z) = start address
;	 R25:R24 = number of bytes to insert (>= 0)
;	 R26 = start offset of first pointer to update (OFF_CURLINE,...)
; OUTPUT: R31:R30 (Z) = start address + number of bytes
;	  R27:R26 (X) = start address
; DESTROYS: R25, R24, R1, R0
; STACK: 6 bytes
; ----------------------------------------------------------------------------

.global MemIns
MemIns:

; ----- zero number of bytes

	mov	r0,r24
	or	r0,r25
	breq	MemRep6

; ----- save number of bytes -> R27:R1

	mov	r27,r25
	mov	r1,r24

; ----- R25:R24 <- get available memory

	rcall	MemFree		; DESTROYS: R0 (stack 4 bytes)

; ----- check required size

	cp	r24,r1		; check required size
	cpc	r25,r27
	brcc	MemIns2		; memory is OK

; ----- memory error

	ldi	r24,ERR_MEM
	rcall	Error

; ----- return number of bytes -> R25:R24

MemIns2:
	mov	r24,r1
	mov	r25,r27

; ----- repair pointers (R26=offset of first pointer)

	rcall	MemRep		; DESTROYS: R27, R26, R1, R0

; ----- X <- current end of memory (source address)

	ldd	r26,Y+DATA_MEMTOP ; get top of allocated memory
	ldd	r27,Y+DATA_MEMTOP+1

; ----- R1:R0 <- size of moved data

	movw	r0,r26		; top of memory
	sub	r0,r30
	sbc	r1,r31		; size of data

; ----- Z <- new end of memory (destination address)

	movw	r30,r26		; Z <- current end of memory
	add	r30,r24
	adc	r31,r25		; Z <- new end of memory

; ----- save new top of memory

	std	Y+DATA_MEMTOP,r30
	std	Y+DATA_MEMTOP+1,r31

; ----- move memory (in back direction)
; INPUT: R31:R30 (Z) = destination address + length
;	 R27:R26 (X) = source address + length
;	 R25:R24 = length (>= 0)
; OUTPUT: R31:R30 (Z) = destination address
;	  R27:R26 (X) = source address
; DESTROYS: R25, R24, R0

	movw	r24,r0		; R25:R24 <- size of moved data
	rjmp	MemCpyD

; ----------------------------------------------------------------------------
;                      Negate 16-bit number
; ----------------------------------------------------------------------------
; INPUT: R25:R24 = number
; OUTPUT: R25:R24 = -number
;	  C = set if number is != 0, clear if number is = 0
; STACK: 2 bytes
; ----------------------------------------------------------------------------

.global Negate
Negate:
	com	r25
	neg	r24		; set C if number is != 0
	sbci	r25,0xff
Negate8:
	ret

; ----------------------------------------------------------------------------
;                    Delete space from allocable memory
; ----------------------------------------------------------------------------
; INPUT: R31:R30 (Z) = start address
;	 R25:R24 = number of bytes to delete (>= 0)
;	 R26 = start offset of first pointer to update (OFF_CURLINE,...)
; OUTPUT: R31:R30 (Z) = new end address
;	  R27:R26 (X) = old end address
; DESTROYS: R25, R24, R1, R0
; STACK: 4 bytes
; ----------------------------------------------------------------------------

.global MemDel
MemDel:

; ----- zero number of bytes

	mov	r0,r24
	or	r0,r25
	breq	Negate8

; ----- repair pointers

	rcall	Negate		; negate number of bytes
	rcall	MemRep		; DESTROYS: R27, R26, R1, R0
	rcall	Negate		; negate number of bytes

; ----- X <- new start of data (source address)

	movw	r26,r30		; X <- old start address
	add	r26,r24
	adc	r27,r25		; X <- new start address

; ----- R25:R24 <- size of moved data

	ldd	r24,Y+DATA_MEMTOP ; get top of allocated memory
	ldd	r25,Y+DATA_MEMTOP+1
	sub	r24,r26
	sbc	r25,r27		; size of data

; ----- move memory
; INPUT: R31:R30 (Z) = destination address
;	 R27:R26 (X) = source address
;	 R25:R24 = length (>= 0)
; OUTPUT: R31:R30 (Z) = destination address + length
;	  R27:R26 (X) = source address + length
; DESTROYS: R25, R24, R0

	rcall	MemCpy

; ----- save new top of memory

	std	Y+DATA_MEMTOP,r30
	std	Y+DATA_MEMTOP+1,r31
	ret
