; ****************************************************************************
;
;                                   EEPROM
;
; ****************************************************************************

#include "include.inc"

	.text

; ----------------------------------------------------------------------------
;                           Read EEPROM byte
; ----------------------------------------------------------------------------
; INPUT: R25:R24 = source address
; OUTPUT: R23 = data
; DESTROYS: nothing
; STACK: 2 bytes
; ----------------------------------------------------------------------------

; ----- set up address

.global EERead
EERead:
	out	_SFR_IO_ADDR(EEARH),r25
	out	_SFR_IO_ADDR(EEARL),r24

; ----- start read operation

	sbi	_SFR_IO_ADDR(EECR),EERE

; ----- read data

	in	r23,_SFR_IO_ADDR(EEDR)
	ret

; ----------------------------------------------------------------------------
;             Read EEPROM word from word index (EEPROM max. 512 B)
; ----------------------------------------------------------------------------
; INPUT: R24 = word index
; OUTPUT: R23 = 1st byte
;	  R22 = 2nd byte
; DESTROYS: nothing
; STACK: 5 bytes
; ----------------------------------------------------------------------------

.global EEReadW
EEReadW:

; ----- push registers

	push	r25

; ----- convert word index to byte offset + 1 -> R25:R24

	clr	r25
	sec
	adc	r24,r24
	adc	r25,r25

; ----- read 2nd byte -> R22
; INPUT: R25:R24 = source address
; OUTPUT: R23 = data
; DESTROYS: nothing
; STACK: 2 bytes
	rcall	EERead
	mov	r22,r23

; ----- read 1st byte -> R23

	sbiw	r24,1
	rcall	EERead

; ----- convert byte offset back to word index -> R24

	lsr	r25
	ror	r24

; ----- pop registers

	pop	r25
	ret	

; ----------------------------------------------------------------------------
;                              Read EEPROM data
; ----------------------------------------------------------------------------
; INPUT: R31:R30 = destination address in RAM
;	 R25:R24 = source address in EEPROM
;	 R20 = number of bytes
; OUTPUT: R31:R30 = new destination address
;	  R25:R24 = new source address
; DESTROYS: R23, R20
; STACK: 4 bytes
; ----------------------------------------------------------------------------

EEReadData2:
; INPUT: R25:R24 = source address
; OUTPUT: R23 = data
; DESTROYS: nothing
; STACK: 2 bytes
	rcall	EERead		; read byte -> R23
	st	Z+,r23		; save data byte
	adiw	r24,1		; increase source address

.global EEReadData
EEReadData:
	subi	r20,1		; decrement number of bytes
	brcc	EEReadData2	; next byte
	ret

; ----------------------------------------------------------------------------
;                           Write EEPROM byte
; ----------------------------------------------------------------------------
; INPUT: R25:R24 = destination address
;	 R22 = data
; DESTROYS: nothing
; STACK: 5 bytes
; ----------------------------------------------------------------------------

; ----- check old content

.global EEWrite
EEWrite:
	push	r23

; INPUT: R25:R24 = source address
; OUTPUT: R23 = data
; DESTROYS: nothing
; STACK: 2 bytes
	rcall	EERead		; read old byte -> R23
	cp	r23,r22
	breq	EEWrite4	; data already set

; ----- set up address and data

	out	_SFR_IO_ADDR(EEARH),r25
	out	_SFR_IO_ADDR(EEARL),r24
	out	_SFR_IO_ADDR(EEDR),r22

; ----- start write operation

#ifdef MCU8
	sbi	_SFR_IO_ADDR(EECR),EEMWE ; from now, 4 clock cycles to write EEWE
	sbi	_SFR_IO_ADDR(EECR),EEWE
#else
	sbi	_SFR_IO_ADDR(EECR),EEMPE ; from now, 4 clock cycles to write EEWE
	sbi	_SFR_IO_ADDR(EECR),EEPE
#endif

; ----- wait operation

	rcall	EEWait
EEWrite4:
	pop	r23
	ret

; ----------------------------------------------------------------------------
;             Write EEPROM word to word index (EEPROM max. 512 B)
; ----------------------------------------------------------------------------
; INPUT: R24 = word index
; 	 R23 = 1st byte
;	 R22 = 2nd byte
; DESTROYS: nothing
; STACK: 8 bytes
; ----------------------------------------------------------------------------

.global EEWriteW
EEWriteW:

; ----- push registers

	push	r25

; ----- convert word index to byte offset + 1 -> R25:R24

	clr	r25
	sec
	adc	r24,r24
	adc	r25,r25

; ----- write 2nd byte R22
; INPUT: R25:R24 = destination address
;	 R22 = data
; DESTROYS: nothing
; STACK: 5 bytes
	rcall	EEWrite

; ----- write 1st byte R23

	push	r22
	mov	r22,r23
	sbiw	r24,1
	rcall	EEWrite
	pop	r22

; ----- convert byte offset back to word index -> R24

	lsr	r25
	ror	r24

; ----- pop registers

	pop	r25
	ret	

; ----------------------------------------------------------------------------
;                           Write EEPROM data
; ----------------------------------------------------------------------------
; INPUT: R31:R30 = source address in RAM
;	 R25:R24 = destination address in EEPROM
;	 R20 = number of bytes
; OUTPUT: R31:R30 = shifted source address
;	  R25:R24 = shifted destination address
; DESTROYS: R23, R22
; STACK: 7 bytes
; ----------------------------------------------------------------------------

EEWriteData2:
; INPUT: R25:R24 = destination address
;	 R22 = data
; DESTROYS: nothing
; STACK: 5 bytes
	ld	r22,Z+		; R22 <- data byte
	rcall	EEWrite		; write byte
	adiw	r24,1		; increase destination address

.global EEWriteData
EEWriteData:
	subi	r20,1		; decrement number of bytes
	brcc	EEWriteData2	; next byte
	ret

; ----------------------------------------------------------------------------
;                           Wait EEPROM operation
; ----------------------------------------------------------------------------
; DESTROYS: nothing
; STACK: 2 bytes
; ----------------------------------------------------------------------------

.global EEWait
EEWait:
#ifdef MCU8
	sbic	_SFR_IO_ADDR(EECR),EEWE
#else
	sbic	_SFR_IO_ADDR(EECR),EEPE
#endif
	rjmp	EEWait
	ret

