StackSize	equ	0x2000		; stack size (paragraph aligned)
SECTION		.text
		org	0
%ifdef DOS
EXEStart:	db	4Dh,5Ah		; EXE file signature
		dw	AllocSize % 512
		dw	(AllocSize + 511) / 512
		dw	0		; number of relocation entries
		dw	EXEHead/16	; header size in paragraphs
		dw	StackSize/16+4	; min extra memory
		dw	StackSize/16+4	; max extra memory
		dw	AllocSize/16	; initial SS
		dw	StackSize	; initial SP
		dw	0		; checksum
		dw	Start		; initial IP
		dw	-EXEHead/16	; initial CS
		dw	0		; file offset of the relocation table
		dw	0		; overlay number
		align	16,db 0
EXEEnd:
EXEHead		equ	(EXEEnd - EXEStart) ; EXE head size
%else
EXEHead		equ	0
%endif

; ----------------------------------------------------------------------------
;             Start of program at 0080:0000 (or CS:0020 in case of EXE program)
; ----------------------------------------------------------------------------

; ------------- Identification

Start:		jmp	Init

Ident		dw	8A25h		; identification word

; ------------- Init segments

Init:		cli			; disable interrupts
		mov	ax,cs		; AX <- program segment
		mov	ds,ax		; DS <- program segment
		add	ax,(Graphics - Start + EXEHead)/16 ; add code size
		mov	[GraphSeg],ax	; segment with VGA graphics
		mov	word [GraphOff],BMPData ; offset with VGA graphics
		add	ax,DataSize/16	; add graphics size
		mov	ss,ax		; SS <- stack segment
		mov	sp,StackSize	; SP <- end of stack
		sti			; enable interrupts

; ------------- Init random generator

		call	RandomInit	; init random generator

; ------------- Install new interrupt handlers

		call	IntInit		; install interrupt handlers

; ------------- Push old videomode

		mov	ah,0fh		; AH <- function code
		call	Int10		; call Int 10h interrupt
		mov	[OldVMode],al	; store old video mode

; ------------- Init Graphics

		call	InitGraph	; init graphics
		jnc	GameStart	; game start

; ------------- Error, quit program

		mov	ah,0		; AH <- function code
		mov	al,[OldVMode]	; AL <- old video mode
		call	Int10		; call Int 10h interrupt

; ------------- No VGA, display error message

		mov	si,ErrorNoVGA	; error text - cannot find VGA card
		call	DispText	; display error message

; ------------- Wait for a key press

		mov	ah,10h		; AH <- function code
		int	16h		; input character

; ------------- End program/Repeat booting

Quit:		call	IntTerm		; deinstall interrupt handlers
%ifdef	DOS
		mov	ax,4C00h
		int	21h
%else
		int	19h		; repeat booting
%endif
; ------------- Display background

GameStart:	call	SetPalettes	; set palettes

		call	DispGraph	; display background image
		call	KeyInit		; key init
		mov	al,11111b	; switch all bulbs on
		call	SetBulbs	; set bulbs
		call	RedrawReels	; redraw all reels
		jmp	short GameStart3

GameStart2:	mov	dl,3		; number of win lines
		call	GetPos		; get new reel position
		inc	word [GameIn]	; add debug Game-In

		push	dx
		mov	dl,111b		; direction down
		mov	dh,1		; whole turns
		call	DebugPrint	; print debug informations
		call	TurnReels	; animation
		pop	dx

		call	TestSymbol	; test symbols
		call	GetPrize	; get prize of win
		or	ax,ax		; is any win?
		jz	GameStart22	; it is not a win
		add	[GameOut],ax	; add debug Game-Out
		add	[Bank],ax	; add win to bank
GameStart22:	call	DebugPrint	; print debug informations

GameStart3:	call	DrawCredit	; draw credit
		call	DrawBet		; draw bet
		call	DrawWin		; draw win
		call	DrawBank	; draw bank
		dec	word [Credit]	; decrement credits

		mov	ah,10h		; AH <- function code
		int	16h		; input character
		cmp	al,32		; is it Spacebar?
		je	GameStart2	; next turn

		mov	ah,0		; AH <- function code
		mov	al,[OldVMode]	; AL <- old video mode
		call	Int10		; call Int 10h interrupt

		jmp	short Quit	; quit the program

; ----------------------------------------------------------------------------
;                       Print debug informations
; ----------------------------------------------------------------------------
; INPUT:	BX = current reel position
; ----------------------------------------------------------------------------

%define		DEBUG
%include	"MACROS.ASM"		; macros

DebugPrint:	DebugPosition bx	; debug display position
		DebugBonus		; debug display bonus
		DebugMode		; debug display winning mode
		DebugGameIn		; debug display Game-In
		DebugGameOut		; debug display Game-Out
		DebugGameProfit		; debug display Profit
		ret

; ----------------------------------------------------------------------------
;                          Set cursor position
; ----------------------------------------------------------------------------
; INPUT:	DL = column
;		DH = row
; ----------------------------------------------------------------------------

SetCursor:	push	ax		; push AX
		push	bx		; push BX
		mov	ah,2		; AH <- 2 function code
		mov	bh,0		; page
		call	Int10		; call Int 10h
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                        Display number with BIOS
; ----------------------------------------------------------------------------
; INPUT:	AX = number
; ----------------------------------------------------------------------------

; ------------- Push registers

DispNum:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX

; ------------- Convert number to digits

		mov	bx,10		; BX <- 10 divider
		xor	cx,cx		; CX <- 0 counter
DispNum2:	xor	dx,dx		; DX <- 0 number HIGH
		div	bx		; divide number with 10
		push	dx		; push character in DL
		inc	cx		; increment counter in CX
		or	ax,ax		; was it last digit?
		jnz	DispNum2	; convert next digit

; ------------- Display digits

DispNum3:	pop	ax		; AL <- digit
		add	al,"0"		; AL <- convert to character
		call	DispChar	; display character
		loop	DispNum3	; display next digit

; ------------- Pop registers

		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                           Display text with BIOS
; ----------------------------------------------------------------------------
; INPUT:	DS:SI = error message ASCIIZ (ends with zero)
; ----------------------------------------------------------------------------

DispText:	push	ax		; push AX
		push	si		; push SI
		cld			; set direction up
DispText2:	lodsb			; AL <- load next character from DS:SI
		or	al,al		; AL == 0? is it end of text?
		jz	DispText3	; it is end of text
		call	DispChar	; display character in AL
		jmp	short DispText2	; next character
DispText3:	pop	si		; pop SI
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                            Init random generator
; ----------------------------------------------------------------------------

;  ------------ Push registers

RandomInit:	push	ax		; push AX
		push	ds		; push DS

; ------------- Init random generator

		xor	ax,ax		; AX <- 0
		mov	ds,ax		; DS <- 0
		mov	ax,[46ch]	; AX <- timer LOW
		mov	[cs:RandSeed],ax; random seed LOW
		mov	ax,[46ch+2]	; AX <- timer HIGH
		mov	[cs:RandSeed+2],ax ; random seed HIGH

; ------------- Pop registers

		pop	ds		; pop DS
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                           Random generator
; ----------------------------------------------------------------------------
; OUTPUT:	AX = random number
; ----------------------------------------------------------------------------

RandSeed:	dd	5623489		; random seed
RandCoef1:	dd	214013		; random coefficient 1
RandCoef2:	dd	2531011		; random coefficient 2

; General algorithm (new_number = old_number*coef1 + coef2) for the random
; function is taken from interpreted BASIC and is described in Microsoft
; Knowledgebase article Q28150. Other possible coefficients are:
; 214013/2531011, 17405/10395331, 214013/13737667, 214013/10395331

; ------------- Push registers

Random:		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	ds		; push DS

; ------------- Prepare registers

		push	cs		; push CS
		pop	ds		; DS <- CS

; ------------- New random seed (randseed = randseed * coef1 + coef2)

		mov	ax,[RandSeed]	; AX <- random seed LOW
		push	ax		; push AX
		mul	word [RandCoef1]; * coef1 LOW
		xchg	ax,bx		; BX <- push multiple LOW
		mov	cx,dx		; CX <- push multiple HIGH
		pop	ax		; pop AX
		mul	word [RandCoef1+2] ; * coef HIGH
		add	cx,ax		; add multiple LOW
		mov	ax,[RandSeed+2]	; AX <- random seed HIGH
		mul	word [RandCoef1]; * coef1 LOW
		add	ax,cx		; AX <- multiple HIGH
		add	bx,[RandCoef2]	; + coef2 LOW
		adc	ax,[RandCoef2+2]; + coef2 HIGH
		mov	[RandSeed],bx	; new random seed LOW
		mov	[RandSeed+2],ax	; new random seed HIGH

; ------------- Pop registers (AX=new random seed HIGH)

		pop	ds		; pop DS
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		ret

; ----------------------------------------------------------------------------
;                        Display one character with BIOS
; ----------------------------------------------------------------------------
; INPUT:	AL = character to display
; ----------------------------------------------------------------------------

DispChar:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		mov	ah,0eh		; AH <- 0Eh function code
		mov	bx,7		; BL <- color of text, BH <- page 0
		call	Int10		; call Int 10h
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                         Initialize interrupts
; ----------------------------------------------------------------------------

; ------------- Push registers

IntInit:	push	ax		; push AX
		push	dx		; push DX
		push	ds		; push DS

; ------------- Prepare registers

		cli			; disable interrupts
		xor	ax,ax		; AX <- 0
		mov	ds,ax		; DS <- 0

; ------------- Install Int 08h handler

		mov	ax,[8*4]	; old handler LOW
		mov	[cs:Old08],ax	; save old handler LOW
		mov	ax,[8*4+2]	; old handler HIGH
		mov	[cs:Old08+2],ax	; save old handler HIGH
		mov	word [8*4],Int08; new offset of Int 08h
		mov	[8*4+2],cs	; new segment of Int 08h

; ------------- Turn off diskette motor

		mov	dx,3f2h		; floppy control port
		mov	al,0ch		; AL <- control byte
		out	dx,al		; turn off diskette motor
		and	byte [43fh],0f0h ; turn off motor runnings bits

; ------------- Set clock rate to 1/8 (e.g. 1193182/8192=145.652 Hz, 7 ms)

		mov	al,34h		; AL <- command code
		out	43h,al		; set command for rate setting
		mov	al,0		; AL <- rate LOW
		out	40h,al		; set rate LOW
		mov	al,32		; AL <- rate HIGH
		out	40h,al		; set rate HIGH

; ------------- Enable interrupts

		sti			; enbale interrupts

; ------------- Pop registers

		pop	ds		; pop DS
		pop	dx		; pop DX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                         Deinitialize interrupts
; ----------------------------------------------------------------------------

; ------------- Push registers

IntTerm:	push	ax		; push AX
		push	ds		; push DS

; ------------- Prepare registers

		cli			; disable interrupts
		xor	ax,ax		; AX <- 0
		mov	ds,ax		; DS <- 0

; ------------- Return clock rate

		mov	al,34h		; AL <- command code
		out	43h,al		; set command for rate setting
		mov	al,0		; AL <- rate LOW
		out	40h,al		; set rate LOW
		out	40h,al		; set rate HIGH

; ------------- Return old Int 08h handler

		mov	ax,[cs:Old08]	; old handler LOW
		mov	[8*4],ax	; return old handler LOW
		mov	ax,[cs:Old08+2]	; old handler HIGH
		mov	[8*4+2],ax	; return old handler HIGH

; ------------- Enable interrupts

		sti			; enbale interrupts

; ------------- Pop registers

		pop	ds		; pop DS
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                                 Includes
; ----------------------------------------------------------------------------

%include	"GAME.ASM"		; game services
%include	"GAMETAB.INC"		; winning tables
%include	"GRAPHIC.ASM"		; graphic
%include	"REELS.ASM"		; reels
%include	"DIGITS.ASM"		; digits
%include	"KEYS.ASM"		; keys
%include	"SOUND.ASM"		; sound

; ------------- Added graphics

SECTION		.text

		align	16
Graphics:

AllocSize	equ	(Graphics - Start) + DataSize
