; ============================================================================
;
;                            VEGASLOT - Reels
;
; ============================================================================

SECTION		.text

; ------------- Timing constants

%ifdef FAST
SLOWSPEED	equ	100		; speed for slow rotation
STARTDEL	equ	1		; start delay of reels
STARTSPEED	equ	1000		; start speed
ACCEL		equ	1		; steps for acceleration
STOPSPEED	equ	1000		; stop speed
DECEL		equ	1		; steps for deceleration
OVERSTEPS	equ	1		; overshoot steps
%else
SLOWSPEED	equ	10		; speed for slow rotation
STARTDEL	equ	15		; start delay of reels
STARTSPEED	equ	10		; start speed
ACCEL		equ	80		; steps for acceleration
STOPSPEED	equ	20		; stop speed
DECEL		equ	70		; steps for deceleration
OVERSTEPS	equ	120		; overshoot steps
%endif

; ----------------------------------------------------------------------------
;                        Init offsets of symbols
; ----------------------------------------------------------------------------
; INPUT:	DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

InitSymbOff:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI

; ------------- Symbol 1

		mov	bx,[WindowL]	; left coordinate of reel window
		mov	cx,[ScreenH]	; height of screen
		shr	cx,1		; middle of screen
		call	CalcSrc		; calculate offset of symbol 1
		mov	[SymbOff+1*4+0],si ; offset LOW of symbol 1
		mov	[SymbOff+1*4+2],ax ; offset HIGH of symbol 1

; ------------- Symbol 0

		sub	cx,[SymbolH]	; top coordinate of symbol 0
		call	CalcSrc		; calculate offset of symbol 0
		mov	[SymbOff+0*4+0],si ; offset LOW of symbol 0
		mov	[SymbOff+0*4+2],ax ; offset HIGH of symbol 0

; ------------- Symbol 2

		add	bx,[SymbolW]	; left coordinate of symbol 2
		call	CalcSrc		; calculate offset of symbol 2
		mov	[SymbOff+2*4+0],si ; offset LOW of symbol 2
		mov	[SymbOff+2*4+2],ax ; offset HIGH of symbol 2

; ------------- Symbol 3

		add	cx,[SymbolH]	; top coordinate of symbol 3
		call	CalcSrc		; calculate offset of symbol 3
		mov	[SymbOff+3*4+0],si ; offset LOW of symbol 3
		mov	[SymbOff+3*4+2],ax ; offset HIGH of symbol 3

; ------------- Symbol 5

		add	bx,[SymbolW]	; left coordinate of symbol 5
		call	CalcSrc		; calculate offset of symbol 5
		mov	[SymbOff+5*4+0],si ; offset LOW of symbol 5
		mov	[SymbOff+5*4+2],ax ; offset HIGH of symbol 5

; ------------- Symbol 4

		sub	cx,[SymbolH]	; top coordinate of symbol 4
		call	CalcSrc		; calculate offset of symbol 4
		mov	[SymbOff+4*4+0],si ; offset LOW of symbol 4
		mov	[SymbOff+4*4+2],ax ; offset HIGH of symbol 4

; ------------- Pop registers

		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                       Turn reels to new position
; ----------------------------------------------------------------------------
; INPUT:	DL = directions (3 bits: 1=forward, 0=back)
;		DH = 1 use whole turns, 0 use short turns
;		BX = new position of reels (3*5 bits)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

TurnReels:	push	ax		; push AX
		push	cx		; push CX
		push	dx		; push DX
		push	bp		; push BP

; ------------- Calculate parameters for turning reels

		call	CalcReel	; calculate parameters

; ------------- Play sound

		mov	ax,TurnSound	; AX <- turning sound
		or	dl,dl		; is turn back?
		jnz	TurnReels2	; it is turn forward
		mov	ax,BonusSound	; AX <- bonus turning sound
TurnReels2:	call	PlayRepeat	; play turning sound

; ------------- Set keys

		mov	byte [Turning],1 ; set flag of turning reels
		mov	bp,4		; BP <- 4: 3 reels turn
TurnReels3:	call	SetKeys		; set keys

; ------------- Check, if animation ends (must be done aforetime)

		mov	cx,4		; CX=4: 3 reels turn
		cmp	word [ReelAddr],Int08Stop
		jne	TurnReels4	; 3 reels turn
		dec	cx		; CX=3: 2 reels turn
		cmp	word [ReelAddr+2],Int08Stop
		jne	TurnReels4	; 2 reels turn
		dec	cx		; CX=2: 1 reel turns
		cmp	word [ReelAddr+4],Int08Stop
		jne	TurnReels4	; 1 reel turns
		dec	cx		; CX=1: no reel turns

; ------------- Play stop sound

TurnReels4:	cmp	cx,bp		; stop any reel?
		je	TurnReels5	; reels continue
		mov	bp,cx		; BP <- push next state
		mov	ax,StopTurnSound; AX <- stop sound
		call	PlaySound	; play stop sound

; ------------- Animate reels

TurnReels5:	call	RedrawReels	; redraw all reels

; ------------- Animation ends?

		loop	TurnReels3	; wait for end of animation

; ------------- Set keys

		mov	byte [Turning],0 ; reset flag of turning reels
		call	SetKeys		; set keys

; ------------- Pop registers

		pop	bp		; pop BP
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                  Calculate parameters for turning reels
; ----------------------------------------------------------------------------
; INPUT:	DL = directions (3 bits: 1=forward, 0=back)
;		DH = 1 fast, 0 slow
;		BX = new position of reels (3*5 bits)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

CalcReel:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	bp		; push BP

; ------------- Initialize remaining steps to whole turns

		mov	al,POS		; AL <- number of symbols per reel
		mul	dh		; AX <- symbols per turn
		xchg	al,ah		; AL <- 0, AH <- symbols per turn
		mov	cl,ah		; CL <- symbols per turn
		mov	[ReelRem],ax	; remaining steps 0 (1x)
		add	ah,cl		; increase number of turns
		mov	[ReelRem+2],ax	; remaining steps 1 (2x)
		add	ah,cl		; increase number of turns
		mov	[ReelRem+4],ax	; remaining steps 2 (3x)

; ------------- Initialize start delays of reels

		mov	word [ReelTime],1 ; start delay of reel 0
		mov	word [ReelTime+2],STARTDEL+1 ; start delay of reel 1
		mov	word [ReelTime+4],STARTDEL*2+1 ; start delay of reel 2

; ------------- Prepare registers

		mov	bp,3		; BP <- number of reels
		xor	si,si		; SI <- offset in reel tables

; ------------- Set direction

CalcReel1:	mov	al,dl		; AL <- direction
		and	al,1		; AL = direction
		mov	[ReelDir+si],al	; direction of turn

; ------------- Required end position -> AL

		mov	al,bl		; AL <- new position
		and	al,POSMASK	; AL <- position of one reel
		mov	[ReelEnd+si+1],al ; end position of reel

; ------------- Current position -> AH

		mov	ah,[ReelPos+si+1]; AH <- current position

; ------------- Distinguish direction

		test	dl,1		; is direction forward?
		jnz	CalcReel2	; it is direction forward
		xchg	ah,al		; exchange new and old positon

; ------------- Calculate remaining steps

CalcReel2:	sub	al,ah		; AL <- different
		add	al,POS		; + number of steps per turn
		cmp	al,POS		; is it more than 1 turn?
		jb	CalcReel3	; it is OK
		sub	al,POS		; correction	
CalcReel3:	add	[ReelRem+si+1],al ; remaining steps

; ------------- Check, if this reel moves

		cmp	word [ReelRem+si],0 ; any steps?
		je	CalcReel9	; this reel doesn't run

; ------------- Slow turn

		or	dh,dh		; is it slow turn?
		jnz	CalcReel4	; it is not slow turn
		mov	word [ReelAddr+si],Int08Slow ; turn forward
		jmp	short CalcReel9	; next reel

; ------------- Pasting steps

CalcReel4:	mov	word [ReelPast+si],0 ; pasting steps

; ------------- Overshoot steps

		add	word [ReelRem+si],OVERSTEPS; overshoot steps
		mov	word [ReelOver+si],OVERSTEPS ; overshoot steps

; ------------- Start reel moving

		mov	word [ReelAddr+si],Int08Wait ; start waiting

; ------------- Next reel

CalcReel9:	shr	dl,1		; shift directions
		inc	si		; SI <- offset of next reel
		inc	si		; SI <- offset of next reel
		mov	cl,5		; CL <- number of rotation
		shr	bx,cl		; BX <- position of next reel
		dec	bp		; counter of reels
		jz	CalcReelA	; it was last reel
		jmp	CalcReel1	; calculate next reel

; ------------- Pop registers

CalcReelA:	pop	bp		; pop BP
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                      Interrupt 08h handler (timer)
; ----------------------------------------------------------------------------

; ------------- Sound interrupt routine

Int08:		call	SoundInt	; call sound interrupt routine

; ------------- AutoStart timer

		inc	byte [cs:AutoStartTime] ; AutoStart timer

; ------------- Push registers

		push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	bp		; push BP
		push	ds		; push DS

; ------------- Prepare registers for reel animations

		push	cs		; push CS
		pop	ds		; DS <- data segment
		xor	si,si		; SI <- 0 offset of reel
		mov	bp,3		; BP <- 3 number of reels

; ------------- Jump to reel routine

Int082:		jmp	near [ReelAddr+si] ; jump to reel routine

; ------------- Slow turn

Int08Slow:	mov	ax,SLOWSPEED	; slow speed
		cmp	ax,[ReelRem+si]	; remains enough steps?
		jl	Int08Slow2	; there is enough steps
Int08Slow1:	mov	ax,[ReelEnd+si]	; reel end position
		mov	[ReelPos+si],ax	; new current position
		mov	word [ReelAddr+si],Int08Stop ; stop turn
		jmp	Int08Stop	; next reel

Int08Slow2:	sub	[ReelRem+si],ax; decrease remaining steps
		jmp	short Int08Move	; move reel

; ------------- Wait for start moving

Int08Wait: 	dec	word [ReelTime+si] ; wait timer
		jnz	Int08Wait2	; it is not end of waiting
		mov	word [ReelAddr+si],Int08In ; turning IN
Int08Wait2:	jmp	short Int08Stop	; next reel

; ------------- Turning IN

Int08In:	mov	ax,[ReelPast+si]; pasting steps
		xor	dx,dx		; DX <- 0
		mov	cx,ACCEL	; CX <- acceleration
		div	cx		; AX <- new acceleration
		add	ax,STARTSPEED	; AX <- current speed
		mov	dx,ax		; DX <- current speed
		add	dx,[ReelPast+si]; DX <- new pasting steps
		mov	[ReelPast+si],dx; new pasting steps
		sub	[ReelRem+si],ax ; new remainging steps
		cmp	dx,[ReelRem+si] ; is it after half?
		jl	Int08In2	; it is first half
		mov	word [ReelAddr+si],Int08Out ; turning OUT
Int08In2:	jmp	short Int08Move	; move reels

; ------------- Turning OUT

Int08Out:	mov	ax,[ReelRem+si] ; remaining steps
		xor	dx,dx		; DX <- 0
		mov	cx,DECEL	; CX <- deceleration
		div	cx		; AX <- new acceleration
		add	ax,STOPSPEED	; AX <- current speed
		cmp	ax,[ReelRem+si]	; remain enough steps?
		jl	Int08Out2	; remain enough steps
		mov	ax,[ReelRem+si]	; limit steps
		mov	word [ReelAddr+si],Int08Over
Int08Out2:	sub	[ReelRem+si],ax	; decrease remaining steps
		jmp	short Int08Move	; move reels

; ------------- Overshoot

Int08Over:	mov	ax,STOPSPEED	; stop speed
                cmp	ax,[ReelOver+si] ; remain enough steps?
		jge	Int08Slow1	; end turning
		sub	[ReelOver+si],ax ; decrease remainging steps
		neg	ax		; negative direction

; ------------- Move reel (AX=shift)

Int08Move:	mov	dx,[ReelPos+si]	; DX <- current reel position
		cmp	byte [ReelDir+si],0 ; is direction back?
		jne	Int08Move2	; it is direction forward
		neg	ax		; direction back
Int08Move2:	add	dx,ax		; move reel forward
		and	dh,POSMASK	; mask new reel position
		mov	[ReelPos+si],dx	; new reel position

; ------------- Next reel

Int08Stop:	inc	si		; increase offset of reel
		inc	si		; increase offset of reel
		dec	bp		; decrement reel counter
		jz	Int088		; end
		jmp	Int082		; next reel

; ------------- Pop registers (now without AX)

Int088:		pop	ds		; pop DS
		pop	bp		; pop BP
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX

; ------------- Counting original service

		dec	byte [cs:Int08Next] ; will be original Int 08h?
		jnz	Int089		; will be original Int 08h
		mov	byte [cs:Int08Next],8 ; new counter

; ------------- End of interrupt

		mov	al,20h		; AL <- end of interrupt mask
		out	20h,al		; set end of interrupt
		pop	ax		; pop AX
		iret

; ------------- Jump to original Int 08h handler

Int089:		pop	ax		; pop AX
		jmp	far [cs:Old08]	; continue with old interrupt

; ----------------------------------------------------------------------------
;                         Redraw all reels
; ----------------------------------------------------------------------------
; INPUT:	DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

RedrawReels:	push	ax		; push AX
		push	dx		; push DX

; ------------- Redraw reels

		mov	dl,0		; DL <- 0 reel 0
		mov	ax,[ReelPos]	; AX <- position of reel 0
		call	DrawReel	; draw reel 0
		mov	dl,1		; DL <- 1 reel 1
		mov	ax,[ReelPos+2]	; AX <- position of reel 1
		call	DrawReel	; draw reel 1
		mov	dl,2		; DL <- 2 reel 2
		mov	ax,[ReelPos+4]	; AX <- position of reel 2
		call	DrawReel	; draw reel 2

; ------------- Pop registers

		pop	dx		; pop DX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                             Draw one reel
; ----------------------------------------------------------------------------
; INPUT:	AH = index of symbol in line 0 (0 to 31)
;		AL = offset in one symbol (0 to 255)
;		DL = reel (0 to 2)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

DrawReel:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	bp		; push BP

; ------------- Index of symbol -> BP

		push	ax		; push AX
		mov	al,ah		; AL <- index of symbol
		mov	ah,0		; AX = index of symbol
		add	ax,2		; AX <- index of first symbol
		xchg	ax,bp		; BP <- index of symbol
		pop	ax		; pop AX

; ------------- Prepare top line -> BX

		push	dx		; push DX
		mov	ah,0		; AX = offset in one symbol
		mul	word [SymbolH]	; AX <- relative height
		mov	al,ah
		mov	ah,dl		; AX <- offset in lines
		add	ax,[WindowT]	; add top of reel window
		sub	ax,[SymbolH]	; AX <- top line of symbol
		xchg	ax,bx		; BX <- top line of symbol
		pop	dx		; pop DX

; ------------- Addres of reel tab -> SI

		mov	al,POS		; AL <- number of symbols per reel
		mul	dl		; AX <- offset of reel tab
		add	ax,ReelTab	; AX <- address of reel tab
		xchg	ax,si		; SI <- address of reel tab

; ------------- Draw one symbol

DrawReel5:	and	bp,POSMASK	; mask position of symbol
		mov	al,[cs:bp+si]	; AL <- symbol
		mov	ah,dl		; AH <- reel
		call	DrawSymbol	; draw one symbol
		
; ------------- Prepare next symbol

		dec	bp		; increase index of symbol
		cmp	bx,[WindowW]	; is it below reel window?
		jb	DrawReel5	; draw next symbol

; ------------- Pop registers

		pop	bp		; pop BP
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret		

; ----------------------------------------------------------------------------
;                             Draw one symbol
; ----------------------------------------------------------------------------
; INPUT:	AL = symbol (0 to 5)
;		AH = reel (0 to 2)
;		BX = top line
;		DS = data segment
; OUTPUT:	BX + SymbolH (shift to next row of items)
; ----------------------------------------------------------------------------

; ------------- Push registers

DrawSymbol:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push 	di		; push DI
		push	bp		; push BP
		push	es		; push ES

; ------------- Prepare darkness table -> ES

		push	ax
		cmp	ah,1		; is it reel 1?
                mov	ax,Darkness0	; darkness table 0
		jb	DrawSymbol0	; it is reel 0
                mov	ax,Darkness2	; darkness table 2
		ja	DrawSymbol0	; it is reel 2
		mov	ax,Darkness1	; darkness table 1
DrawSymbol0:	mov	es,ax		; ES <- darkness table
		pop	ax

; ------------- Prepare height (-> CX) and top line (-> BP) of symbol

		mov	cx,[SymbolH]	; CX <- height of one symbol
		xor	bp,bp		; BP <- 0 top line in one symbol
		mov	si,[WindowT]	; SI <- top line of reel window
		sub	si,bx		; SI <- distance from top line of symbol
		jle	DrawSymbol1	; top edge is OK below top of window
		mov	bx,[WindowT]	; BX <- new destination line
		mov	bp,si		; DX <- top line in one symbol
		sub	cx,si		; CX <- decrease height of symbol
		jle	DrawSymbol9	; symbol is whole above window
DrawSymbol1:	mov	si,[WindowB]	; bottom line of reel window
		sub	si,bx		; height of space
		jle	DrawSymbol9	; symbol is whole below window
		cmp	si,cx		; check if there is enough space
		jae	DrawSymbol2	; there is enough space
		mov	cx,si		; CX <- limit height of symbol

; ------------- Prepare destination offset (-> DX:DI)

DrawSymbol2:	push	cx		; push CX, height of symbol
		push	bx		; push BX, destination line

		push	ax		; push AX, symbol, reel

		mov	al,ah		; AL <- index of reel
		mov	ah,0		; AX = index of reel
		mul	word [SymbolW]	; AX <- offset X in reel window
		add	ax,[WindowL]	; AX <- offset X on screen
		xchg	ax,bx		; BX <- X coordinate, AX <- line
		xchg	ax,cx		; CX <- line X
		call	CalcDest	; calculate destination offset

; ------------- Prepare source offset (-> AX:SI)

		push	dx		; push DX
		mov	ax,[ScreenW]	; AX <- width of screen
		mul	bp		; DX:AX <- source offset of top line
		xchg	ax,bp		; BP <- offset LOW
		mov	bx,dx		; BX <- offset HIGH
		pop	dx		; pop DX

		pop	ax		; pop AX, symbol, reel

		mov	ah,0		; AX = index of item
		shl	ax,1
		shl	ax,1		; AX = offset in table
		add	ax,SymbOff	; AX = address in table
		xchg	ax,si		; SI <- address in table
		mov	ax,[si+2]	; AX <- offset HIGH
		mov	si,[si]		; SI <- offset LOW
		add	si,bp		; add offset LOW
		adc	ax,bx		; add offset HIGH

		pop	bp		; pop BP, destination line
		pop	cx		; pop CX, height of symbol

; ------------- Calculate top position in darkness table (-> BP)

		push	ax		; push AX
		mov	ax,es		; AX <- darkness table
		add	bp,ax		; BP = darkness table + first line
		pop	ax		; pop AX

; ------------- Draw symbol

		mov	bx,[SymbolW]	; width of symbol
DrawSymbol3:	push	bp		; push BP, darkness

		push	ax		; push AX
		mov	ah,[cs:bp]	; AL <- level
		mov	al,0
		mov	bp,ax		; AX <- offset of table
		cmp	ah,DarkLev	; is it valid level?
		pop	ax
		jae	DrawSymbol4	; it is not valid level

		add	bp,DarkTab	; BP <- address of dark table
		call	DrawDark	; draw dark line
		jmp	short DrawSymbol5

DrawSymbol4:	call	DrawLine	; draw one line

DrawSymbol5:	pop	bp		; pop BP, darkness
		inc	bp
		loop	DrawSymbol3	; draw next line

; ------------- Pop registers

DrawSymbol9:	pop	es		; pop ES
		pop	bp		; pop BP
		pop	di		; pop DI
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		add	bx,[SymbolH]	; shift to next row of items
		ret

; ----------------------------------------------------------------------------
;                               Set bulbs
; ----------------------------------------------------------------------------
; INPUT:	AL = mask of lines b0 to b4 (1=shines)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

SetBulbs:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	di		; push DI
		push	es		; push ES

; ------------- Prepare registers

		push	ds		; push DS
		pop	es		; ES <- data segment
		cld			; direction up

; ------------- Clear bulb buffer

		push	ax		; push AX
		mov	di,Bulbs	; DI <- buffer of bulbs
		mov	cx,ROWS*REELS	; CX <- size of bulb buffer
		mov	al,0		; AL <- 0 clear byte
		rep	stosb		; clear bulb buffer
		pop	ax		; pop AX

; ------------- Set bulb buffer - lines

		mov	si,Line		; SI <- offset of lines
		mov	cl,LINES	; CL <- number of lines
SetBulbs1:	test	al,1		; shines this line?
          	jz	SetBulbs2	; this bulb doesn't shine
		mov	di,[si]		; DI <- offset of bulb of reel 0
		mov	byte [Bulbs+1+di],1 ; shine this bulb on reel 0
		mov	di,[si+2]	; DI <- offset of bulb of reel 1
		mov	byte [Bulbs+4+di],1 ; shine this bulb on reel 1
		mov	di,[si+4]	; DI <- offset of bulb of reel 2
		mov	byte [Bulbs+7+di],1 ; shine this bulb on reel 2
SetBulbs2:      add	si,6		; SI <- offset of next line
		shr	al,1		; AL <- next line
		loop	SetBulbs1	; set net line

; ------------- Copy darkness tables of reels

		mov	si,Darkness	; darkness table
		mov	di,Darkness0	; darkness table of reel 0
		mov	cx,[ScreenH]	; height of screen
		rep	movsb		; copy tables for reels
		mov	si,Darkness	; darkness table
		mov	di,Darkness1	; darkness table of reel 1
		mov	cx,[ScreenH]	; height of screen
		rep	movsb		; copy tables for reels
		mov	si,Darkness	; darkness table
		mov	di,Darkness2	; darkness table of reel 2
		mov	cx,[ScreenH]	; height of screen
		rep	movsb		; copy tables for reels

; ------------- Loop through reels

		mov	si,Bulbs+2	; table of bulbs
		mov	di,Darkness0	; darkness table of reel 0
		add	di,[WindowT]	; top line of reel window
		mov	bl,3		; number of reels
SetBulbs3:	push	di		; push DI
		mov	bh,3		; number of bulbs
SetBulbs4:     	cmp	byte [si],0	; shines this bulb?
		jne	SetBulbs8	; this bulb shines

; ------------- Darkness this bulb

		push	di
		mov	cx,[SymbolH]	; height of one symbol
SetBulbs5:	mov	al,[di]		; AL <- current dark level
		cmp	al,DarkLev	; is it valid dark level?
		jb	SetBulbs6	; this level is valid
		mov	al,-1		; prepare next level
SetBulbs6:	inc	ax		; increase dark level
		inc	ax		; increase dark level
		inc	ax		; increase dark level
		cmp	al,DarkLev	; is it valid dark level?
		jb	SetBulbs7	; it is valid dark level
		mov	al,DarkLev-1	; limit dark level
SetBulbs7:	mov	[di],al		; new dark level
		inc	di		; next dark level
		loop	SetBulbs5	; next dark level
		pop	di

; ------------- Next bulb

SetBulbs8:	add	di,[SymbolH]	; next bulb
		dec	si		; next bulb
		dec	bh		; bulb counter
		jnz	SetBulbs4	; next bulb

; ------------- Next reel

		add	si,3+3		; bulb in next reel
		pop	di		; pop DI
		add	di,Darkness1-Darkness0 ; next darkness table
		dec	bl		; reel counter
		jnz	SetBulbs3	; next reel		

; ------------- Pop registers

		pop	es		; pop ES
		pop	di		; pop DI
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX

; ------------- Redraw all reels

		call	RedrawReels	; redraw all reels
		ret

; ----------------------------------------------------------------------------
;                                    Data
; ----------------------------------------------------------------------------

Turning:	db	0		; 1=flag of turning reels

; ------------- Symbols

SymbolW:	dw	64		; width of one symbol
SymbolH:	dw	40		; height of one symbol
WindowW:	dw	192		; width of reel window
WindowH:	dw	120		; height of reel window
WindowL:	dw	64		; left coordinate of reel window
WindowT:	dw	40		; top coordinate of reel window
WindowB:	dw	160		; bottom coordinate of reel window

Border:		dw	5		; height of dark border
BorderT:	dw	0		; line of top border
BorderB:	dw	0		; line of bottom border

; ------------- Interrupt INT 08h

Old08:		dd	0		; old INT 08 interrupt
Int08Next:	db	8		; internal counter for Int 08h

; ------------- Address of reel routine

ReelAddr:	dw	Int08Stop,Int08Stop,Int08Stop

; ------------- Current position of reels (HIGH=symbol, LOW=partial 0 to 255)

ReelPos:	dw	0,0,0

; ------------- End position of reels

ReelEnd		dw	0,0,0

; ------------- Direction of turning (1=forward, 0=back)

ReelDir:	dw	0,0,0

; ------------- Start time of reels

ReelTime:	dw	0,0,0

; ------------- Pasting steps

ReelPast:	dw	0,0,0

; ------------- Remaining steps

ReelRem:	dw	0,0,0

; ------------- Overshoot steps

ReelOver:	dw	0,0,0

; ------------- Bulbs, 1=lights

Bulbs:		db	0,0,0		; reel 0
		db	0,0,0		; reel 1
		db	0,0,0		; reel 2

; ----------------------------------------------------------------------------
;          Uninitialized data (60 KB area in place of VGA graphics)
; ----------------------------------------------------------------------------

SECTION		.bss

; ------------- Offsets of symbols

SymbOff:	resd	SYMBOLS		; offset of symbols
