; ============================================================================
;
;                          Calculate Game Table
;
; ============================================================================

; ------------- Setup parameters

SYMBOLS		equ	6		; number of types of symbols
SYMMASK		equ	7		; mask of symbol index

C		equ	0		; shortcut of symbol "cherry"
A		equ	1		; shortcut of symbol "apple"
P		equ	2		; shortcut of symbol "plum"
E		equ	3		; shortcut of symbol "pear"
G		equ	4		; shortcut of symbol "grape"
B		equ	5		; shortcut of symbol "bell"

REELS		equ	3		; number of reels
POS		equ	32		; number of positions on one reel
POSMASK		equ	1fh		; mask of reel position
LINES		equ	5		; number of winning lines

PROFIT		equ	9400		; profit of game * 100

WINVALUES	equ	SYMBOLS		; number of winning values

; ------------- Structure of item of winning table (8 bytes)

struc		WinItem

win_value	resw 	1		; value of win prize (1 to 750)
win_type	resw	1		; winning types
					;  3 bits: symbol 1
					;  3 bits: number of lines with symbol 1
					;  3 bits: symbol 2
					;  3 bits: number of lines with symbol 2
					;  3 bits: symbol 3
					;  1 bit:  number of lines with symbol 3
win_num		resw	1		; number of wins
win_rand	resw	1		; randomness (0 to 0ffffh)

endstruc

WINITEM_SIZE	equ	8

; ------------- Start of program

SECTION		.text

		org	100h		; start of COM program

; ------------- Print help 1

Start:		mov	ax,TextHelp1	; AX <- help 1
		call	PrintText	; print help 1

; ------------- Print symbols on reels

		mov	al,0		; AL <- 0 first reel
		call	PrintReel	; print symbols on reel 0
		mov	ax,TextHelp2	; AX <- help 2
		call	PrintText	; print help 2
		mov	al,1		; AL <- 1 second reel
		call	PrintReel	; print symbols on reel 1
		mov	ax,TextHelp2	; AX <- help 2
		call	PrintText	; print help 2
		mov	al,2		; AL <- 2 third reel
		call	PrintReel	; print symbols on reel 2

; ------------- Print help 3

		mov	ax,TextHelp3	; AX <- help 3
		call	PrintText	; print help 3

; ------------- Print winning values

		cld			; direction up
		mov	cx,SYMBOLS-1	; number of symbols - 1
		mov	si,WinVal	; table of winning values
Start1:		lodsw			; AX <- value of one symbol
		call	PrintNum	; print value of one symbol
		mov	al,","		; AL <- comma character
		call	PrintChar	; print comma character
		loop	Start1		; print next value
		lodsw			; AX <- value of last symbol
		call	PrintNum	; print value of last symbol

; ------------- Print help 4

		mov	ax,TextHelp4	; AX <- help 4
		call	PrintText	; print help 4

; ------------- Initialize winning tables

		mov	dl,1		; DL <- 1 winning lines
Start2:		call	CalcWinTab	; calculate winning table
		call	SortWinTab	; sort winning table

; ------------- Calculate rough game profit

		call	CalcWinRand	; calculate game randomness
		call	CalcWinProf	; calculate game profit

; ------------- First correction of game profit - direct proportion

		push	dx		; push DX (winning lines)
		mov	ax,PROFIT	; required profit
		mul	word [WinTabMul] ; profit * multiplier
		div	word [WinTabProf] ; / current profit
		mov	[WinTabMul],ax	; new multiplier
		pop	dx		; pop DX (winning lines)

; ------------- Second correction of game profit - loop correction

		mov	cx,1000		; CX <- number of iterations
Start3:		call	CalcWinRand	; calculate game randomness
		call	CalcWinProf	; calculate game profit
		mov	ax,PROFIT	; AX <- required profit
		cmp	ax,[WinTabProf]	; is profit correct?
		je	Start7		; profit matches
		mov	ax,1		; correction up
		jg	Start4		; do correction up
		neg	ax		; else correction down
Start4:		add	[WinTabMul],ax	; correction of multiplier
		loop	Start3		; try next calculation

; ------------- Third correction of game profit - correction of first win

		mov	cx,[WinTab+win_rand] ; randomness of first win
		shr	cx,1		; CX = number of iterations
		jcxz	Start7		; what?
Start5:		mov	ax,PROFIT	; AX <- required profit
		cmp	ax,[WinTabProf]	; is profit correct?
		je	Start7		; profit matches
		mov	ax,1		; correction up
		jg	Start6		; do correction up
		neg	ax		; else correction down
Start6:		add	[WinTab+win_rand],ax ; correction of first win
		add	[WinTabRand],ax	; correction of total randomness
		call	CalcWinProf	; calculate game profit
		loop	Start5		; try next calculation

; ------------- Print winning table
		
Start7:		call	PrintWinTab	; print winning table
		inc	dx		; increase number of winning lines
		cmp	dl,LINES	; were all winning lines?
		jbe	Start2		; next winning line

; ------------- Print help 5

		mov	ax,TextHelp5	; AX <- help 5
		call	PrintText	; print help 5

		int	20h		; end of program

; ----------------------------------------------------------------------------
;                            Print symbols on one reel
; ----------------------------------------------------------------------------
; INPUT:	AL = reel (0 to 2)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

PrintReel:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	si		; push SI
		
; ------------- Prepare registers

		cld			; direction up
		mov	cx,POS		; CX <- positions on one reel
		mul	cl		; AX <- offset of reel tab
		add	ax,ReelTab	; AX <- addres of reel tab
		xchg	ax,si		; SI <- addres of reel tab

; ------------- Display one symbol

PrintReel2:	xor	ax,ax		; AX <- 0
		lodsb			; AL <- one symbol
		xchg	ax,bx		; BX <- symbol
		mov	al,[bx+TextSymb]; AL <- character of symbol
		call	PrintChar	; print symbol

; ------------- Print comma

		cmp	cl,1		; is it last symbol?
		je	PrintReel3	; it is last symbol
		mov	al,","		; comma character
		call	PrintChar	; print comma character
PrintReel3:	loop	PrintReel2	; print next symbol

; ------------- Pop registers

		pop	si		; pop SI
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                            Print Winning Table
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

PrintWinTab:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push 	si		; push SI

; ------------- Prepare registers

		cld			; set direction up
		mov	si,WinTab	; SI <- winning table
		mov	cx,[WinTabNum]	; CX <- number of wintab items

; ------------- Print label

		mov	ax,TextWinTab	; AX <- text of label
		call	PrintText	; print text of label
		xor	ax,ax		; AX <- 0
		mov	al,dl		; AL <- number of lines
		call	PrintNum	; print number of label

; ------------- Print prefix of line

		mov	ax,TextDW2	; AX <- text ": dw"
		call	PrintText	; print text ": dw"

; ------------- Print zero profit

		xor	ax,ax		; AX <- 0
		call	PrintNumCom	; print zero profit with comma

; ------------- Print number of items

		mov	ax,cx		; AX <- number of items
		call	PrintNumCom	; print number of items with comma

; ------------- Print total wins

		mov	ax,[WinTabTot]	; AX <- total wins
		call	PrintNumCom	; print number of total wins with comma

; ------------- Print total randomness

		mov	ax,[WinTabRand]	; AX <- total randomness
		call	PrintNum	; print total randomness

; ------------- Print remark separator

		mov	ax,TextRemark	; AX <- text "; "
		call	PrintText	; print text "; "
		mov	ax,TextProfit	; AX <- "Profit"
		call	PrintText	; print text "Profit"

; ------------- Print profit

		push	dx		; push DX
		mov	ax,[WinTabProf]	; AX <- profit
		xor	dx,dx		; DX <- 0
		mov	bx,100		; BX <- 100 divisor
		div	bx		; AX <- percents, DX <- hundredths
		call	PrintNum	; print percents
		mov	al,"."		; decimal point
		call	PrintChar	; print decimal point
		xchg	ax,dx		; AX <- hundredths, DX <- percents
		mov	bl,10           ; BL <- 10 divisor
		div	bl		; AL <- decimals, AH <- hundreths
		or	ax,"00"		; correction to characters
		call	PrintChar	; print decimals
		mov	al,ah		; AL <- hundreths
		call	PrintChar	; print hundreths
		pop	dx		; pop DX
		mov	al,"%"		; percentage character
		call	PrintChar	; print percentage

; ------------- Print end of line

		call	PrintCRLF	; print end of line
		jcxz	PrintWinTab9	; there is no wintab item

; ------------- Print prefix of line

PrintWinTab2:	mov	ax,TextDW	; AX <- text "dw"
		call	PrintText	; print text "dw"

; ------------- Print winning prize

		mov	ax,[si+win_value] ; AX <- winning prize
		call	PrintNumCom	; print winning prize with comma

; ------------- Print winning type

		mov	ax,[si+win_type] ; AX <- winning type
		call	PrintNumCom	; print winning type with comma

; ------------- Print number of wins

		mov	ax,[si+win_num]	; AX <- number of wins
		call	PrintNumCom	; print number of wins with comma

; ------------- Print randomness

		mov	ax,[si+win_rand]; AX <- randomness
		call	PrintNum	; print randomness

; ------------- Print remark separator

		mov	ax,TextRemark	; AX <- text "; "
		call	PrintText	; print text "; "

; ------------- Winning types

		mov	dx,[si+win_type] ; DX <- winning types

; ------------- Print winning value of symbol

PrintWinTab3:	mov	ax,dx		; AX <- winning symbol
		shr	dx,1		; DX >> 1
		shr	dx,1		; DX >> 1
		shr	dx,1		; DX >> 1
		and	ax,7		; AX = symbol
		shl	ax,1		; AX = symbol * 2
		xchg	ax,bx		; BX <- offset in table
		mov	ax,[bx+WinVal]	; AX <- winning value
		call	PrintNum	; print winning value

; ------------- Print number of lines with symbol

		mov	ax,dx		; AX <- number of lines
		shr	dx,1		; DX >> 1
		shr	dx,1		; DX >> 1
		shr	dx,1		; DX >> 1
		and	ax,7		; AX = number of symbol
		cmp	ax,1		; is there 2 or more lines?
		jbe	PrintWinTab4	; there is only 1 line
		push	ax		; push AX
		mov	al,"x"		; character "x"
		call	PrintChar	; print character "x"
		pop	ax		; pop AX
		call	PrintNum	; print number of lines

; ------------- Test, if there will be next symbol

PrintWinTab4:	test	dx,7 << 3	; will be next symbol?
		jz	PrintWinTab6	; it was last symbol
		mov	al," "		; separator (space)
		call	PrintChar	; print separator
		jmp	short PrintWinTab3 ; print next symbol

; ------------- Print end of line

PrintWinTab6:	call	PrintCRLF	; print end of line

; ------------- Next winning prize

		add	si,WINITEM_SIZE	; next winning item
		loop	PrintWinTab2	; nest winning item

; ------------- pop registers

PrintWinTab9:	pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop 	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                         Print number with comma
; ----------------------------------------------------------------------------
; INPUPT:	AX = number to print
;		DS = data segment
; ----------------------------------------------------------------------------

PrintNumCom:	call	PrintNum	; print number
                push	ax		; push AX
		mov	ax,TextComma	; AX <- text ", "
		call	PrintText	; print text ", "
		pop	ax		; pop AX
		ret
		
; ----------------------------------------------------------------------------
;                             Print number
; ----------------------------------------------------------------------------
; INPUPT:	AX = number to print
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

PrintNum:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX

; ------------- Convert number do digits

		xor	cx,cx		; CX <- 0 character counter
		mov	bx,10		; BX <- 10 divisor
PrintNum2:	xor	dx,dx		; DX <- 0
		div	bx		; divide DX:AX with 10
		xchg	ax,dx		; AX <- remainder
		add	al,'0'		; convert to printable character
		push	ax		; push character to stack
		inc	cx		; increase character counter
		xchg	ax,dx		; AX <- quotient
		or	ax,ax		; is it zero?
		jnz	PrintNum2	; it is not zero yet

; ------------- Display digits

PrintNum3:	pop	ax		; pop character from stack
		call	PrintChar	; print character
		loop	PrintNum3	; print next character

; ------------- Pop registers

		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                             Print ASCIIZ text
; ----------------------------------------------------------------------------
; INPUPT:	DS:AX = text to print
; ----------------------------------------------------------------------------

; ------------- Push registers

PrintText:	push	ax		; push AX 
		push	si		; push SI

; ------------- Print text

		cld			; direction up
		xchg	ax,si		; SI <- text to print
PrintText2:	lodsb			; AL <- character to print
		or	al,al		; is it end of text?
		jz	PrintText4	; it is end of text
		call	PrintChar	; print character
		jmp	short PrintText2 ; next character

; ------------- Pop registers

PrintText4:	pop	si		; pop SI
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                        Print tabulator to position
; ----------------------------------------------------------------------------
; INPUT:	AL = required position
;		DS = data segment
; ----------------------------------------------------------------------------

PrintTabPos:	call	PrintTab	; print tabulator
		cmp	al,[CharPos]	; is it required position?
		ja	PrintTabPos	; print next tabulator
		ret
		
; ----------------------------------------------------------------------------
;                             Print tabulator
; ----------------------------------------------------------------------------
;		DS = data segment
; ----------------------------------------------------------------------------

PrintTab:	push	ax		; push AX
		mov	al,9		; AL <- TAB character
		call	PrintChar	; print TAB character
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                            Print end of line
; ----------------------------------------------------------------------------
;		DS = data segment
; ----------------------------------------------------------------------------

PrintCRLF:	push	ax		; push AX
		mov	al,13		; AL <- CR character
		call	PrintChar	; print CR character
		mov	al,10		; AL <- LF character
		call	PrintChar	; print LD character
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                             Print character
; ----------------------------------------------------------------------------
; INPUPT:	AL = character to print
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

PrintChar:	push	ax		; push AX
		push	dx		; push DX

; ------------- End of line (CR character)

		cmp	al,13		; is it CR character?
		jne	PrintChar2	; it is not CR character
		mov	byte [CharPos],0 ; reset character position
		jmp	short PrintChar6 ; print character

; ------------- Tabulator (TAB character)

PrintChar2:	cmp	al,9		; is it TAB character?
		jne	PrintChar4	; it is not TAB character
		add	byte [CharPos],8 ; round position up
		and	byte [CharPos],~7 ; tab position
		jmp	short PrintChar6 ; print character

; ------------- Other characters

PrintChar4:	inc	byte [CharPos]	; increase character position

; ------------- Print character

PrintChar6:	mov	dl,al		; DL <- character to print
		mov	ah,2		; AH <- function code
		int	21h		; print character

; ------------- Pop registers

		pop	dx		; pop DX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                         Calculate randomness
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

CalcWinRand:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX

; ------------- Prepare registers

		mov	word [WinTabRand],0 ; reset total randomness
		mov	bx,WinTab	; BX <- winning table
		mov	cx,[WinTabNum]	; CX <- number of winning table items

; ------------- Calculate one value

CalcWinRand2:	mov	ax,[WinTabMul]	; AX <- multiplier for game profit
		xor	dx,dx		; DX <- 0
		div     word [bx+win_value] ; / winning value
		mov	dx,[bx+win_num]	; number of wins
		shr	dx,1		; number of wins / 2
		shr	dx,1		; number of wins / 4
		add	ax,dx		; AX += additional correction
		mov	[bx+win_rand],ax ; randomness of this win
		add	[WinTabRand],ax	; add to total randomness
		add	bx,WINITEM_SIZE	; BX <- next winning item
		loop	CalcWinRand2	; calculate next item

; ------------- Pop registers

		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                     Calculate winning game profit
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

CalcWinProf:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	di		; push DI

; ------------- Prepare registers

		push	dx		; push number of lines

		mov	bx,WinTab	; BX <- winning table
		xor	si,si		; SI <- 0 accumulator LOW
		xor	di,di		; DI <- 0 accumulator HIGH
		mov	cx,[WinTabNum]	; CX <- number of winning table items

; ------------- Add value of one item

CalcWinProf2:	mov	ax,[bx+win_value] ; winning value
		mul	word [bx+win_rand] ; profit of this item
		add	si,ax		; add accumulator LOW
		adc	di,dx		; add accumulator HIGH

; ------------- Next item

		add	bx,WINITEM_SIZE	; BX <- address of next item
		loop	CalcWinProf2	; next item

		pop	bx		; BL <- number of lines

; ------------- Calculate game profit per mille (10000 * win_sum / (bet * 65536))

		mov	ax,10000	; profit multiplier
		mul	si		; calculate profit per mille LOW
		xchg	dx,si		; SI <- multiple HIGH
		mov	ax,10000	; profit multiplier
		mul	di		; calculate profit per mille HIGH
		add	ax,si		; carry from LOW
		adc	dx,0		; carry
		mov	bh,0		; BX = number of lines
		mov	bl,[bx+BetTab-1]; BX <- bet
		div	bx		; divide with bet
		mov	[WinTabProf],ax	; game profit per mille

; ------------- Pop registers

		pop	di		; pop DI
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                     Sort winning table (slow bubble sort)
; ----------------------------------------------------------------------------
; INPUT:	DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

SortWinTab:	push	ax		; push AX
		push	cx		; push CX
		push	si		; push SI

; ------------- Start one passage

		cld			; direction up
SortWinTab2:	mov	si,WinTab	; SI <- winning table
		mov	cx,[WinTabNum]	; CX <- number of winning table items
		jcxz	SortWinTab7	; not enough items
		dec	cx		; CX <- number of item pairs
		jcxz	SortWinTab7	; not enough items
		
; ------------- Compare two items

SortWinTab3:	mov	ax,[si+win_value] ; winning value
		cmp	ax,[si+WINITEM_SIZE+win_value] ; compare with next item
		jne	SortWinTab4	; winning value is not the same
		mov	ax,[si+win_type] ; winning type
		cmp	ax,[si+WINITEM_SIZE+win_type] ; compare with next item	
SortWinTab4:	jbe	SortWinTab6	; items are in right order

; ------------- Exchange items

		mov	cx,WINITEM_SIZE/2 ; CX <- number of words
SortWinTab5:	lodsw			; AX <- load 1 word of item
		xchg	ax,[si+WINITEM_SIZE-2] ; exchange 1 word
		mov	[si-2],ax	; store 1 word
		loop	SortWinTab5	; next word of item
		jmp	short SortWinTab2 ; repeat sorting

; ------------- Next item

SortWinTab6:	add	si,WINITEM_SIZE	; SI <- next item
		loop	SortWinTab3	; sort next item

; ------------- Pop registers

SortWinTab7:	pop	si		; pop SI
		pop	cx		; pop CX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                         Calculate winning table
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DS = data segment
; ----------------------------------------------------------------------------

; ------------- Push registers

CalcWinTab:	push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	di		; push DI
		push	bp		; push BP
		push	es		; push ES

; ------------- Prepare registers

		xor	bx,bx		; BX <- 0 reel position pointer
		mov	[WinTabNum],bx	; reset counter of wintab items
		mov	[WinTabTot],bx	; reset counter of total wins
		mov	di,WinTab	; DI <- winning table
		push	ds		; push DS
		pop	es		; ES <- data segment
		cld			; set direction up

; ------------- Test winning symbol -> DH mask, BP symbols

CalcWinTab2:	call	TestSymbol	; test winning symbol

; ------------- Calculate winning prize -> AX

		call	GetPrize	; calculate winning prize
		or	ax,ax		; is any winning prize?
		jz	CalcWinTab6	; there is no winning prize
		inc	word [WinTabTot] ; increase counter of total wins

; ------------- Calculate winning types -> BP

		push	ax		; push AX
		call	GetTypes	; calculate winning types
		xchg	ax,bp		; BP <- winning types
		pop	ax		; pop AX

; ------------- Search, if this win already exists

		mov	si,WinTab	; SI <- winning table
		mov	cx,[WinTabNum]	; CX <- number of winning table items
		jcxz	CalcWinTab5	; there is no wintab item
CalcWinTab3:	cmp	ax,[si+win_value]; check winning prize
		jne	CalcWinTab4	; it is not this case
		cmp	bp,[si+win_type]; check winning types
		jne	CalcWinTab4	; it is not this case
		inc	word [si+win_num]; increase number of wins
		jmp	short CalcWinTab6 ; next reel position

CalcWinTab4:	add	si,WINITEM_SIZE	; next wintable item
		loop	CalcWinTab3	; test next item

; ------------- Create new winning table item

CalcWinTab5:	stosw			; store winning prize
		xchg	ax,bp		; AX <- winning types
		stosw			; store winning types
		mov	ax,1		; AX <- 1 initial number of wins
		stosw			; store number of wins
		stosw			; store randomness
		inc	word [WinTabNum]; increase number of wintable items

; ------------- Next reel position

CalcWinTab6:	inc	bx		; increase reel position
		jns	CalcWinTab2	; next position

; ------------- Pop registers

		pop	es		; pop ES
		pop	bp		; pop BP
		pop	di		; pop DI
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ----------------------------------------------------------------------------
;                          Calculate winning types
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DH = winning mask, bit 0 to 4: there is win on line 0 to 4
;		BP = winning symbols on lines
;			bit  0 to  2: symbol in line 0
;			bit  3 to  5: symbol in line 1
;			bit  6 to  8: symbol in line 2
;			bit  9 to 11: symbol in line 3
;			bit 12 to 14: symbol in line 4
;		DS = data segment
; OUTPUT:	AX = winning types
;			bit  0 to  2: symbol 1
;			bit  3 to  5: number of lines with symbol 1
;			bit  6 to  8: symbol 2
;			bit  9 to 11: number of lines with symbol 2
;			bit 12 to 14: symbol 3
;			bit 15:       number of lines with symbol 3
; ----------------------------------------------------------------------------

; ------------- Push registers

GetTypes:	push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		push	di		; push DI
		push	bp		; push BP

; ------------- Prepare registers

		xor	si,si		; SI <- 0 number of lines LOW
		xor	di,di		; DI <- 0 number of lines HIGH

; ------------- Test, if there is a win on this line

GetTypes2:	test	dh,1		; is there a win?
        	jz	GetTypes5	; there is not a win

; ------------- Winning symbol complement -> CX

		mov	cx,SYMBOLS-1	; CX <- number of symbols-1
		sub	cx,bp		; CX <- winning symbol complement
		and	cx,SYMMASK	; mask winning symbol complement
		mov	ax,cx		; AX <- winning symbol complement
		add	cx,ax		; CX <- symbol * 2
		add	cx,ax		; CX <- symbol * 3

; ------------- Prepare mask -> BX:AX

		mov	ax,1		; AX <- 1 mask LOW
		xor	bx,bx		; BX <- 0 mask HIGH
		jcxz	GetTypes4	; no rotation
GetTypes3:	shl	ax,1		; rotate AX left
		rcl	bx,1		; rotate BX left with carry
		loop	GetTypes3	; rotate next bit

; ------------- Increase symbol counter

GetTypes4:	add	si,ax		; increase counter LOW
		adc	di,bx		; increase counter HIGH

; ------------- Next line

GetTypes5:	shr	dh,1		; rotate winning mask
		mov	cl,3		; CL <- 3 number of rotation
		shr	bp,cl		; rotate winning symbols on lines
		dec	dl		; will there be next line?
		jnz	GetTypes2	; next line

; ------------- Pack symbol counters

		xor	ax,ax		; AX <- 0 winning types
		mov	dx,SYMBOLS-1	; DX <- number of symbols-1
GetTypes6:	mov	bx,si		; BX <- counter LOW
		and	bx,7		; mask number of lines
		jz	GetTypes7	; this symbol has 0 lines

; ------------- Add this symbol to winning types

		shl	ax,cl		; shift winning types left
		or	ax,bx		; AX << number of lines
		shl	ax,cl		; shift winning types left
		or	ax,dx		; AX << symbol number

; ------------- Next symbol

GetTypes7:	shr	di,1		; rotate lines counter HIGH
		rcr	si,1		; rotate lines counter LOW
		shr	di,1		; rotate lines counter HIGH
		rcr	si,1		; rotate lines counter LOW
		shr	di,1		; rotate lines counter HIGH
		rcr	si,1		; rotate lines counter LOW
		sub	dx,1		; number of symbol
		jnc	GetTypes6	; next symbol

; ------------- Pop registers

		pop	bp		; pop BP
		pop	di		; pop DI
		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		ret

; ----------------------------------------------------------------------------
;                          Calculate winning prize
; ----------------------------------------------------------------------------
; INPUT:	DL = number of lines (1 to 5)
;		DH = winning mask, bit 0 to 4: there is win on line 0 to 4
;		BP = winning symbols on lines
;			bit  0 to  2: symbol in line 0
;			bit  3 to  5: symbol in line 1
;			bit  6 to  8: symbol in line 2
;			bit  9 to 11: symbol in line 3
;			bit 12 to 14: symbol in line 4
;		DS = data segment
; OUTPUT:	AX = winning prize
; ----------------------------------------------------------------------------

; ------------- Push registers

GetPrize:	push	bx		; push BX
		push	dx		; push DX
		push	bp		; push BP

; ------------- Prepare registers

		xor	ax,ax		; AX <- 0 prize accumulator
		
; ------------- Test, if there is a win on this line

GetPrize2:	test	dh,1		; is there a win?
        	jz	GetPrize3	; there is not a win

; ------------- Add winning value

		mov	bx,bp		; bx <- symbol
		and	bx,SYMMASK	; mask one symbol
		shl	bx,1		; index of symbol * 2
		add	ax,[bx+WinVal]	; add winning value

; ------------- Next line

GetPrize3:	shr	dh,1		; rotate winning mask
		shr	bp,1
		shr	bp,1
		shr	bp,1		; rotate symbols on lines
		dec	dl		; will there be next line?
		jnz	GetPrize2	; next line

; ------------- Pop registers

		pop	bp		; pop BP
		pop	dx		; pop DX
		pop	bx		; pop BX
		ret

; ----------------------------------------------------------------------------
;                            Test winning symbols
; ----------------------------------------------------------------------------
; INPUT:	BX = reel positions
;			bit  0 to  4: position of reel 0
;			bit  5 to  9: position of reel 1
;			bit 10 to 14: position of reel 2
;		DS = data segment
; OUTPUT:	DH = winning mask, bit 0 to 4: there is win on line 0 to 4
;		BP = winning symbols on lines
;			bit  0 to  2: symbol in line 0
;			bit  3 to  5: symbol in line 1
;			bit  6 to  8: symbol in line 2
;			bit  9 to 11: symbol in line 3
;			bit 12 to 14: symbol in line 4
; ----------------------------------------------------------------------------

; ------------- Push registers

TestSymbol:	push	ax		; push AX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI

; ------------- Prepare registers

		cld			; direction up
		mov	si,Lines	; SI <- offsets of lines
		xor	bp,bp		; BP <- 0 winning symbols
		mov	cx,100h		; CH <- 1 bit mask, CL <- 0 rotations
		mov	dx,LINES	; DH <- 0 winning mask, DL <- lines

; ------------- Get symbol on reel 0 -> BP, CH

TestSymb2:	push	cx		; push CX
		push	dx		; push DX

		lodsw			; AX <- offset of symbol
		push	bx		; push BX
		add	bx,ax		; add offset of symbol
		and	bx,POSMASK	; mask of position of reel 0
		mov	al,[bx+Reel0]	; symbol on reel 0
		mov	ah,0		; AH <- 0
		mov	ch,al		; CH <- save symbol on reel 0
		shl	ax,cl		; rotate symbol on position
		or	bp,ax		; BP <- save symbol in this line
		pop	bx		; pop BX

; ------------- Get symbol on reel 1 -> DL

		lodsw			; AX <- offset of symbol
		push	bx		; push BX
		mov	cl,5		; CL <- number of rotations
		shr	bx,cl		; rotate position of reel 1
		add	bx,ax		; add offset of symbol
		and	bx,POSMASK	; mask of position of reel 1
		mov	dl,[bx+Reel1]	; symbol on reel 1
		pop	bx		; pop BX
		
; ------------- Get symbol on reel 2 -> DH

		lodsw			; AX <- offset of symbol
		push	bx		; push BX
		mov	cl,10		; CL <- number of rotations
		shr	bx,cl		; rotate position of reel 2
		add	bx,ax		; add offset of symbol
		and	bx,POSMASK	; mask of position of reel 2
		mov	dh,[bx+Reel2]	; symbol on reel 2
		pop	bx		; pop BX

; ------------- Compare symbols if there are identical
		
		cmp	ch,dl		; are symbols 0 and 1 identical?
		jne	TestSymb3	; there are not identical
		cmp	ch,dh		; are symbols 0 and 2 identical?

TestSymb3:	pop	dx		; pop DX
		pop	cx		; pop CX
		jne	TestSymb4	; symbols are not identical

; ------------- Set flag this is winning position	

		or	dh,ch		; DH <- set winning mask

; ------------- Next line

TestSymb4:	add	cl,3		; increase number of rotation of symbol
		shl	ch,1		; shift winning mask
		dec	dl		; decrement line counter
		jnz	TestSymb2	; next line

; ------------- Pop registers

		pop	si		; pop SI
		mov	ch,dh		; CH <- winning mask
		pop	dx		; pop DX
		mov	dh,ch		; DH <- winning mask
		pop	cx		; pop CX
		pop	ax		; pop AX		
		ret

; ----------------------------------------------------------------------------
;                                Data
; ----------------------------------------------------------------------------

CharPos:	db	0			; character position

; ------------- Texts

TextHelp1:	db	"; Don't edit this file, it is generated by program."
		db	13,10,13,10
		db	"SECTION",9,9,".text",13,10,13,10
		db	"; ------------- Symbols on reels",13,10
		db	"; C=cherry, A=apple, P=plum, E=pear, G=grape, B=bell"
		db	13,10,13,10,"ReelTab: db ",0

TextHelp2:	db	13,10,9," db ",0

TextHelp3:	db	13,10,13,10
		db	"; ------------- Winning values (3 equal symbols)"
		db	13,10,13,10,"WinVal:",9,9,"dw",9,0

TextHelp4:	db	13,10,13,10
		db	"; ------------- Winning tables",13,10
		db	"; Head of one table: zero win, items in"
		db	" table, total wins, total randomness",13,10
		db	"; One item of table: win value, types,"
		db	" wins, randomness",13,10,0

TextHelp5:	db	13,10
		db	"; ------------- List of winning tables",13,10,13,10
		db	"WinTabs:",9,"dw",9,"WinTab1",13,10
		db	9,9,"dw",9,"WinTab2",13,10
		db	9,9,"dw",9,"WinTab3",13,10
		db	9,9,"dw",9,"WinTab4",13,10
		db	9,9,"dw",9,"WinTab5",13,10,0

TextWinTab:	db	13,10,"WinTab",0
TextDW:		db	9,9,"dw",9,0
TextDW2:	db	":",9,"dw",9,0
TextComma:	db	",",9,0
TextRemark:	db	9,"; ",0
TextProfit:	db	"Profit ",0
TextSymb:	db	"CAPEGB"

; ------------- Location of symbols on reels
	;    C=cherry, A=apple, P=plum, E=pear, G=grape, B=bell
	;                       1 1 1 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 2 2 3 3
	;   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
ReelTab:
Reel0:	db  C,C,C,C,C,A,A,P,P,E,P,A,C,P,A,C,G,E,A,E,G,C,A,G,E,G,A,P,P,B,C,A
Reel1:	db  C,C,C,A,C,A,A,P,E,P,A,E,P,A,C,G,E,G,E,P,C,P,A,G,E,G,C,A,E,B,A,C
Reel2:	db  C,C,C,G,C,A,A,A,P,C,G,P,E,P,C,E,G,E,C,E,B,G,P,G,E,G,C,E,G,B,P,A

; ------------- Offsets of lines

Lines:
Line1:		dw	 0,  0,  0		; 1: middle line
Line2:		dw	 1,  1,  1		; 2: top line
Line3:		dw	-1, -1, -1		; 3: bottom line
Line4:		dw	 1,  0, -1		; 4: top diagonal
Line5:		dw	-1,  0,  1		; 5: bottom diagonal

; ------------- Winning values (3 equal symbols)

WinVal:		dw	10,20,50,100,200,750	; C,A,P,E,G,B

; ------------- Bet table (per number of lines)

BetTab:		db	1,2,3,4,5

; ------------- Winning table

WinTabRand:	dw	0			; total randomness of wins
WinTabTot:	dw	0			; total wins
WinTabProf:	dw	0			; game profit per mille
WinTabMul:	dw	8000			; multiplier for game profit

WinTabNum:	dw	0			; number of items

WinTab:		times 1000 db 0			; winning table
