; ============================================================================
;
;                   MicroDOS - Micro Disk Operating System
;
; ============================================================================

;%define	NOHD			; uncomment this to not use hard drive
;%define	DOS			; use this flag to compile it as COM

%ifndef DOS
BOOTSEG		EQU	60h		; boot segment of the system
%endif

; ------------- Bits

B0		EQU	0000000000000001b
B1		EQU	0000000000000010b
B2		EQU	0000000000000100b
B3		EQU	0000000000001000b
B4		EQU	0000000000010000b
B5		EQU	0000000000100000b
B6		EQU	0000000001000000b
B7		EQU	0000000010000000b
B8		EQU	0000000100000000b
B9		EQU	0000001000000000b
B10		EQU	0000010000000000b
B11		EQU	0000100000000000b
B12		EQU	0001000000000000b
B13		EQU	0010000000000000b
B14		EQU	0100000000000000b
B15		EQU	1000000000000000b

; ------------- Variables in data segment 0

CurPhantom	EQU	504h		; (1) current phantom disk (0 or 1)

; ------------- Character constants

BREAK		EQU	3		; break character (Ctrl+C)
TAB		EQU	9		; tabulator character
CR		EQU	0Dh		; Carriage Return ASCII character
LF		EQU	0Ah		; Line Feed ASCII character
PRINT		EQU	10h		; print character (Ctrl+P)
EOF		EQU	1Ah		; End of File character (Ctrl+Z)
SPACE		EQU	20h		; space character (first printable)
PAUSE		EQU	13h		; pause character (Ctrl+S)
DELETE		EQU	7fh		; delete character
CTRL_PRINTSCR	EQU	7200h		; Ctrl+Print Screen key

; ------------- Limits

SECTORSIZE	EQU	512		; disk sector size

MAXDISK		EQU	4		; number of disks (A: - D:)

; ------------- System start

SECTION		.text

%ifdef DOS
		org	100h		; compile it as COM program
%else
		org	0
%endif

; ----------------------------------------------------------------------------
;                               Jump Vector Table
; ----------------------------------------------------------------------------

JVTInit:	jmp	Init		; 0: jump to system initialization
JVTTestChar:	jmp	TestChar	; 3: test character from keyboard
JVTGetChar:	jmp	GetChar		; 6: get character from keyboard
JVTDispChar:	jmp	DispChar	; 9: display character to display
JVTPrintChar:	jmp	PrintChar	; 0ch: print character to printer
JVTRecChar:	jmp	RecChar		; 0fh: receive character from COM
JVTSendChar:	jmp	SendChar	; 12h: send character to COM
JVTReadDisk:	jmp	MyInt25		; 15h: read sectors from disk (Int 25h)
JVTWriteDisk:	jmp	MyInt26		; 18h: write sectors to disk (Int 26h)

; ----------------------------------------------------------------------------
;                       Divide with zero service
; ----------------------------------------------------------------------------

; ------------- Push registers

MyInt00:	push	dx		; push DX

; ------------- Display error text

		mov	dx,MyInt0Text	; DX <- error text
		call	DispText	; display error text

; ------------- Pop registers

		pop	dx		; pop DX

; ------------- Interrupt program

		int	23h		; interrupt program
MyInt01:
MyInt03:
MyInt04:	iret

; ----------------------------------------------------------------------------
;                               Data
; ----------------------------------------------------------------------------

MyInt0Text:	db	CR,LF,'Divide overflow',CR,LF,0

; ----------------------------------------------------------------------------
;                                 Includes
; ----------------------------------------------------------------------------

%include	"IO_CON.ASM"		; console device
%include	"IO_AUX.ASM"		; serial COM port
%include	"IO_PRN.ASM"		; printer device
%include	"IO_DSK.ASM"		; disk device
%include	"DOS.ASM"		; Int 21h services
%include	"DOS_CON.ASM"		; DOS console
%include	"DOS_CLK.ASM"		; DOS clock
%include	"DOS_AUX.ASM"		; DOS serial COM port
%include	"DOS_PRN.ASM"		; DOS printer
%include	"DOS_DSK.ASM"		; DOS disk
%include	"DOS_PAR.ASM"		; parser
%include	"DOS_FIL.ASM"		; files
%include	"DOS_EXE.ASM"		; program
%include	"INIT.ASM"		; initialization (must be last!)

; ------------- Data buffers (aprox. 32 KB)

Int21Stack1	EQU	SystemEnd+80h	; end of internal stack for INT 21h
Int21Stack2	EQU	Int21Stack1+80h	; end of internal stack for console
DataBuffer	EQU	Int21Stack2	; (SECTORSIZE) disk data buffer
DiskDDPBA	EQU	DataBuffer+SECTORSIZE; (DDPB_SIZE) DDPB of A:
DiskDDPBB	EQU	DiskDDPBA+DDPB_SIZE ; (DDPB_SIZE) DDPB of B:
DiskDDPBC	EQU	DiskDDPBB+DDPB_SIZE ; (DDPB_SIZE) DDPB of C:
DiskDDPBD	EQU	DiskDDPBC+DDPB_SIZE ; (DDPB_SIZE) DDPB of D:
DiskBuff	EQU	DiskDDPBD+DDPB_SIZE ; (SECTORSIZE) disk sector buffer
DiskParBPB	EQU	DiskBuff+SECTORSIZE ; (MAXDISK*BPB_SIZE) disk tables
RootBuf		EQU	DiskParBPB+MAXDISK*BPB_SIZE; (SECTORSIZE) Root sector
ConBuff		EQU	RootBuf+SECTORSIZE ; (84h) console input buffer

DataBuffersEnd	EQU	ConBuff+84h	; end of data buffers
