; ============================================================================
;
;                               Litux Loader
;          Boot sector for FAT12 or FAT16 (only for 512 B sectors!)
;
; ============================================================================

; Compilation: Use symbol "FAT16" to use FAT16 instead of FAT12

BASE 	equ	7C00h			; base start address
SYSSEG	equ	800h			; start segment to load system
MAXSEC	equ	4000h			; max. sector size

; File directory entry

struc	Dir

	dir_filename:	resb	8	; filename
	dir_ext:	resb	3	; extension
	dir_attribute:	resb	1	; attributes
	dir_res:	resb	10	; reserved space (create and read time)
	dir_time:	resw	1	; time of last write
	dir_date:	resw	1	; date of last write
	dir_cluster:	resw	1	; start cluster
	dir_size:	resd	1	; file size
	
endstruc

; Local variables

absolute	BASE - 20 - 12 - MAXSEC
; - end of stack (=STACK)
; - SECSIZE sector buffer (for reading root dir and FAT) (=BUFFER)
; - 12 bytes disk param table (=DISKTAB)
; - 20 bytes local variables
; - start of boot sector (=BASE)

STACK		resd	1		; end of stack

BUFFER		resb	MAXSEC		; buffer to read one sector
	
DISKTAB		resb	12		; disk param table
DIRBASE		resd	1		; directory base sector
DIRSIZE		resd	1		; number of sectors per root
DIRENTRY	resw	1		; number of dir entries per sector
OLDFAT		resw	1		; index of old FAT sector

; Boot sector entries (offsets to entries in head of boot sector)

absolute	0			; starting at offset 0

Start_off:	resb	3		; jump to starting routine
OEMIdent_off:	resb	8		; OEM identifier (8 characters)
SectorSize_off:	resw	1		; sector size in bytes (always 512)
ClusterSize_off:resb	1		; cluster size in sectors
Reserved_off:	resw	1		; reserved sectors in front of FAT
FatNum_off:	resb	1		; number of FAT tables
Root_off:	resw	1		; number of root directory entries
TotalSect_off:	resw	1		; total number of sectors
MediaDescr_off:	resb	1		; media descriptor
FatSectors_off:	resw	1		; sectors per one FAT table
Sectors_off:	resw	1		; number of sectors per track
Heads_off:	resw	1		; number of heads
Hidden_off:	resd	1		; number of hidden sectors
BigTotal_off:	resd	1		; total number of sectors DWORD
Disk_off:	resb	1		; disk (0=floppy, 0x80=hard disk)
Head_off:	resb	1		; head
Magic_off:	resb	1		; extension head magic
Serial_off:	resd	1		; disk serial number
Label_off:	resb	11		; disk label (11 characters)
FatName_off:	resb	8		; name of the disk file system (8 char)
	
; ----------------------------------------------------------------------------
;      Start of the boot sector loader (starting at 0000:7C00 address)
; ----------------------------------------------------------------------------

SECTION		.text

		org	7C00h		; booting address of boot sector

Start:		jmp	Init		; jump to starting routine
	
; ----------------------------------------------------------------------------
;                    Head of the FAT12/FAT16 file system
; ----------------------------------------------------------------------------

OEMIdent	db	"LITUX100"	; OEM identifier (8 characters)
SectorSize	dw	512		; sector size in bytes (always 512)
ClusterSize	db	1		; cluster size in sectors
Reserved	dw	1		; reserved sectors in front of FAT
FatNum		db	2		; number of FAT tables (2)
Root		dw	16*14		; number of root directory entries
TotalSectors	dw	80*18*2		; total number of sectors
					; (if > 65535 this entry is zero
					; and BigTotalSectors will be used)
MediaDescriptor	db	0xF0		; media descriptor
					; (FF=320K, FE=160K, FD=360K, FC=180K,
					; F9=1.2M and 720K, F8=HD, F0=1.44M)
FatSectors	dw	7		; sectors per one FAT table
Sectors		dw	18		; number of sectors per track
Heads		dw	2		; number of heads
Hidden		dd	0		; number of hidden sectors
BigTotalSectors	dd	0		; total number of sectors DWORD
					; (used for sectors > 65535)
Disk		db	0		; disk (0=floppy, 0x80=hard disk)
Head		db	0		; head
Magic		db	0x29		; extension head magic
Serial		dd	0		; disk serial number
Label		db	"NO NAME    "	; disk label (11 characters)
%ifdef FAT16
FatName		db	"FAT16   "	; name of the disk file system (8 char)
%else
FatName		db	"FAT12   "	; name of the disk file system (8 char)
%endif

; ----------------------------------------------------------------------------
;                      Boot sector starting routine
; ----------------------------------------------------------------------------

; ------------- Init stack below beginning of the loader
	
Init:		cli			; disable interrupt
		xor	cx,cx		; CX <- 0
		mov	ss,cx		; SS <- 0
		mov	sp,STACK	; SP <- end of stack
		mov	bp,BASE		; BP <- boot base address

; ------------- Load disk param table

		mov	es,cx		; ES <- 0
		mov	bx,4*1eh	; address of INT 1Eh vector
		lds	si,[es:bx]	; DS:SI <- address of disk param table
		push	si		; push old address SI of table
		push	ds		; push old address DS of table
		cld			; direction flag "up"
		mov	cl,11		; CX <- length of disk param table
		mov	di,DISKTAB	; DI <- buffer for disk param table
		rep	movsb		; load disk param table

; From this point DS:SI (original disk param table) is pushed in the stack!!!
		
; ------------- Set address of new disk param table

		mov	ds,cx		; DS <- 0		
		mov	word [bx],DISKTAB; offset of new disk table
		mov	[bx+2],cx	; segment of new disk table
		
; ------------- Set new parameters of disk param table

		mov	al,[bp+Sectors_off]; number of sectors per track
		mov	[di-7],al	; number of last sector on track
		mov	byte [di-2],15	; start-up time for floppy
		sti			; enable interrupt

; ------------- Disk reset

		xor	ax,ax		; AX <- 0
		mov	dl,[bp+Disk_off]; DL <- disk to reset
		int	13h		; reset of disk system

; ------------- Calculate number of directory entries per sector

		mov	bx,[bp+SectorSize_off] ; sector size
		shr	bx,5		; BX/32 = directory entries per sector
		mov	[bp+DIRENTRY-BASE],bx ; store entries per sector

; ------------- Prepare number of sectors per root

		mov	ax,[bp+Root_off]; root entries
		add	ax,bx		; round up to full sector
		dec	ax
		xor	dx,dx		; DX <- 0
		div	bx		; calculate number of sectors
		mov	[bp+DIRSIZE-BASE],ax ; sectors per root
		
; ------------- Prepare sector of first directory entry

		mov	al,[bp+FatNum_off]; number of FATs
		cbw			; AX <- number of FATs
		mul	word [bp+FatSectors_off] ; DX:AX = sector per all FATs
		mov	[bp+DIRBASE-BASE],ax ; store base sector of directory
		mov	[bp+DIRBASE+2-BASE],dx
		
; ------------- Prepare to scan directory

		mov	cx,[bp+DIRSIZE-BASE] ; number of root sectors
		push	cx		; push sector counter

; ------------- Read 1 sector of root directory (DX:AX = sector number)
		
Scan1:		mov	bx,BUFFER	; buffer to read sector
		mov	cx,1		; number of sectors
		push	bx		; pointer to directory entry
		call	Read		; read 1 sector
		pop	di		; DI <- pointer to directory entry

; ------------- Scan directory entries in 1 sector

		mov	bx,[bp+DIRENTRY-BASE] ; number of directory entries
Scan2:		mov	cl,11		; length of filename
		mov	si,Filename	; name of search system file
		cmp	byte [si],0	; is it end of root directory?
		je	ScanError	; is it end = ERROR
		push	di		; push directory entry
		repe	cmpsb		; compare filename
		pop	di		; pop directory entry
		je	ScanOK		; directory entry found
		lea	di,[di + Dir_size] ; next directory entry
		dec	bx		; entry counter
		jnz	Scan2		; scan next entry
	
; ------------- Prepare to next sector of root

		pop	cx		; pop sector counter
		dec	cx		; sector counter
		jnz	Scan1		; scan next sector of root

; ------------- Errro - no system

ScanError:	jmp	Error
		
; ------------- Directory entry found

ScanOK:		pop	cx			; pop sector counter

		mov	ax,SYSSEG		; starting segment
		mov	es,ax			; segment to load data
		mov	word [bp+OLDFAT-BASE],0xFFFF
		mov	ax,[di+dir_cluster]	; first cluster of file
NextClust:	xor	bx,bx
		call	ReadClust
		push	es
%ifdef FAT16
		mov	cx,4			; nibbles in FAT16
%else
		mov	cx,3			; nibbles in FAT12
%endif
		mul	cx			; DX:AX = nibble index
		shr	dx,1
		rcr	ax,1			; DX:AX = byte index, CF=odd
		pushf
		
		mov	bx,BUFFER
;		mov	si,SECSIZE
		div	si			; AX = sector, DX = offset
		mov	di,dx			; DI <- offset
		cmp	ax,[bp+OLDFAT-BASE]
		je	NextClus2
		
		push	dx
		xor	dx,dx
		mov	[bp+OLDFAT-BASE],ax
		push	ds
		pop	es
		mov	cx,1
		call	Read
		pop	dx

NextClus2:	mov	dh,[bx+di]
		popf
		
%ifdef FAT16
		mov	cl,0
%else
		mov	cl,4
%endif
		jc	NextClus3
		shl	dx,cl

NextClus3:	shr	dx,cl

		mov	si,0xFFFF
		shr	si,cl
		and	si,0xFFF8
		
		pop	ax
;		add	ax,SECSIZE/16
		mov	es,ax
		
		xchg	ax,dx
		
		cmp	ax,si
		jb	NextClust





; ------------- Continue booting with next device

		int	19h		; continue booting

; ----------------------------------------------------------------------------
;                         Read data from one cluster
; ----------------------------------------------------------------------------
; Entry: ReadClust
; Input: AX = cluster number
;	 ES:BX = buffer
; ----------------------------------------------------------------------------

; ------------- Push registers

ReadClust:	push	ax
		push	cx
		push	dx

; ------------- Calculate first sector of cluster

		mov	ch,0
		mov	cl,[bp+ClusterSize_off] ; CX <- sectors per cluster
		dec	ax
		dec	ax		; AX <- base cluster 0
		mul	cx		; calculate relative sector
		add	ax,[bp+DIRSIZE-BASE]
		adc	dx,byte 0
		add	ax,[bp+DIRBASE-BASE]
		adc	dx,[bp+DIRBASE+2-BASE]

; ------------- Read cluster
		
		call	Read

; ------------- Pop registers
		
		pop	dx
		pop	cx
		pop	ax
		ret

; ----------------------------------------------------------------------------
;                         Read sectors from disk
; ----------------------------------------------------------------------------
; Entry: Read
; Input: DX:AX = absolute sector (without hidden and reserved sectors)
;        CX = number of sectors to read
;        ES:BX = buffer
; Output: DX:AX = new absolute sector
;         CX = 0
;         ES:BX = new address in buffer
; Modifies: SI
; ----------------------------------------------------------------------------

; ------------- Push registers

Read:		push	ax		; push register AX (sector LOW)
		push	dx		; push register DX (sector HIGH)
		push	cx		; push register CX (number of sectors)

; ------------- Add hidden sectors (start of partition)

		add	ax,[bp+Hidden_off]	; add hidden sectors LOW
		adc	dx,[bp+Hidden_off+2]	; add hidden sectors HIGH
		
; ............. Add reserved sectors (= boot sector)

		add	ax,[bp+Reserved_off]
		adc	dx,byte 0

; ------------- Divide by sectors per track, calculate first sector

		push	ax		; push absolute sector LOW
		xchg	ax,dx		; AX <-> DX absolute sector
		cwd			; DX:AX = absolute sector HIGH
		mov	si,[bp+Sectors_off] ; number of sectors per track
		div	si		; DX:AX / sectors_per_track
		xchg	ax,cx		; CX <- dividend HIGH
		pop	ax		; AX <- absolute sector LOW
		div	si		; DX:AX / sectors_per_track
		inc	dx		; DX = first sector in track
		push	dx		; push first sector

; ------------- Divide by sides, calculate first head

		mov	dx,cx		; DX:AX = number of track
		div	word [bp+Heads_off] ; DX:AX / heads
		mov	cl,6		; number of shifts
		shl	ah,cl		; AH = cylinder HIGH
		pop	cx		; pop first sector
		or	cl,ah		; CL = first sector and cylinder HIGH
		mov	ch,al		; CH <- cylinder LOW
		mov	dh,dl		; DH <- head

; ------------- Determine number of sectors to read

		xchg	ax,si		; sectors per track
		sub	al,cl		; maximum number of sectors
		pop	si		; SI <- sectors to read
		push	si		; back to stack
		cmp	ax,si		; is less sectors required?
		jb	Read2		; read all sectors to end of track		
		xchg	ax,si		; AX <- required nuber of sectors
Read2:		push	ax		; push number of sectors to read

; ------------- Read group of sectors

		mov	ah,2		; function to read sectors
		mov	dl,[bp+Disk_off]; DL <- disk to read from
		int	13h		; read sectors
		jc	Error		; error reading sectors
		
; ------------- Pop registers

		pop	si		; SI <- readed sectors
		pop	cx		; CX <- sectors to read
		pop	dx		; DX <- sector HIGH
		pop	ax		; AX <- sector LOW
		
; ------------- Shift registers

		add	ax,si		; shift absolute sector LOW
		jnc	Read3
		inc	dx		; shift absolute sector HIGH

Read3:		sub	cx,si		; shift number of sectors
Read4:		add	bx,[bp+SectorSize_off]; shift address
		dec	si		; number of sectors
		jnz	Read4		; next shift
		
; ------------- Next blok of sectors

		or	cx,cx
		jnz	Read		; next sectors		
		ret
		
; ----------------------------------------------------------------------------
;                                 Error
; ----------------------------------------------------------------------------

; ------------- Display error message

Error:		mov	si,ErrorMessage	; error text
Error2:		lodsb			; AL <- load next character from DS:SI
		or	al,al		; AL == 0? (is it end of text?)
		jz	Error3		; end of text
		mov	ah,0eh		; AH <- function code		
		mov	bx,7		; BL <- color of text, BH <- page 0
		int	10h		; print character
		jmp	short Error2	; next character

; ------------- Wait for a key press

Error3:		xor	ax,ax		; AX <- 0
		int	16h
		
; ------------- Reboot system

		int	19h

; ----------------------------------------------------------------------------
;                                   Strings
; ----------------------------------------------------------------------------
	
; ------------- Error message

ErrorMessage:	db	13,10,"NO SYSTEM",0

; ------------- System filename

Filename:	db	"KERNEL  SYS"

; ------------- Align to sector size

times 512 - 2 - ($ - Start) db	0

		dw	0aa55h
	
