; =============================================================================
;
;                      Litos - Unicode character type
;
; =============================================================================

		CODE_SECTION

; ------------- Macro - Area of Unicode character types - invalid characters
; %1 = number of characters

%macro		UNIINV	1
		%rep	%1
		UNITP	CHNONE, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Area of Unicode character types - control
; %1 = number of characters

%macro		UNICTR	1
		%rep	%1
		UNITP	CHCTR, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Area of initialized Unicode character types - control
; %1 = starting character, %2 = number of characters

%macro		UNICTRCH 2
		%assign CTRL_CH %1
		%rep	%2
		UNITP	CHCTR, CTRL_CH, 0
		%assign CTRL_CH CTRL_CH + 1
		%endrep
%endmacro

; ------------- Macro - Area of Unicode character types - space
; %1 = number of characters

%macro		UNISPC	1
		%rep	%1
		UNITP	CHSPC, " ", 0
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - mark
; %1 = ASCII alternative

%macro		UNIMRK	1
		UNITP	CHMRK, %1, 0
%endmacro

; ------------- Macro - Area of Unicode character types - mark
; %1 = number of characters

%macro		UNIMRKNUM 1
		%rep	%1
		UNITP	CHMRK, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - sign
; %1 = ASCII alternative

%macro		UNISIG	1
		UNITP	CHSIG, %1, 0
%endmacro

; ------------- Macro - Area of Unicode character types - sign
; %1 = number of characters

%macro		UNISIGNUM 1
		%rep	%1
		UNITP	CHSIG, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Area of initialized Unicode character types - sign
; %1 = character, %2 = number of characters

%macro		UNISIGCH 2
		%rep	%2
		UNITP	CHSIG, %1, 0
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - digit
; %1 = ASCII alternative

%macro		UNIDIG	1
		UNITP	CHDIG, %1, 0
%endmacro

; ------------- Macro - Area of Unicode character types - digit
; %1 = number of characters

%macro		UNIDIGNUM 1
		%rep	%1
		UNITP	CHDIG, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Area of initialized Unicode character types - digit
; %1 = starting character, %2 = number of characters

%macro		UNIDIGCH 2
		%assign UNIDIG_CH %1
		%rep	%2
		UNITP	CHDIG, UNIDIG_CH, 0
		%assign	UNIDIG_CH UNIDIG_CH + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - modifier
; %1 = ASCII alternative

%macro		UNIMOD	1
		UNITP	CHMOD, %1, 0
%endmacro

; ------------- Macro - Area of Unicode character types - modifier
; %1 = number of characters

%macro		UNIMODNUM 1
		%rep	%1
		UNITP	CHMOD, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - capital
; %1 = ASCII alternative, %2 = small letter

%macro		UNICAP	2
		UNITP	CHCAP,%1,%2
%endmacro

; ------------- Macro - Area of Unicode character types - capital
; %1 = number of characters

%macro		UNICAPNUM 1
		%rep	%1
		UNITP	CHCAP, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode character type - small
; %1 = ASCII alternative, %2 = capital letter

%macro		UNISMA	2
		UNITP	CHSMA,%1,%2
%endmacro

; ------------- Macro - Area of Unicode character types - small
; %1 = number of characters

%macro		UNISMANUM 1
		%rep	%1
		UNITP	CHSMA, UNINOASC, 0
		%endrep
%endmacro

; ------------- Macro - Area of initialized Unicode character types - small
; %1 = starting character, %2 = number of characters

%macro		UNISMACH 2
		%assign UNISMA_CH %1
		%rep	%2
		UNITP	CHSMA, UNISMA_CH, 0
		%assign UNISMA_CH UNISMA_CH + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode capital/small pairs
; %1 = starting index, %2 = number of character pairs

%macro		UNICAPSMA 2
		%assign CAPSMA_INX %1
		%rep	%2
		%assign CAPSMA_INX CAPSMA_INX + 1
		UNICAP	UNINOASC,CAPSMA_INX
		%assign CAPSMA_INX2 CAPSMA_INX - 1
		UNISMA	UNINOASC,CAPSMA_INX2
		%assign CAPSMA_INX CAPSMA_INX + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode capital with small index
; %1 = starting index of small, %2 = number of characters

%macro		UNICAPINX 2
		%assign CAPSMA_INX %1
		%rep	%2
		UNICAP	UNINOASC,CAPSMA_INX
		%assign CAPSMA_INX CAPSMA_INX + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode capital with small index
; %1 = starting character, %2 = starting small, %3 = number of characters

%macro		UNICAPINXCH 3
		%assign UNICAP_CH %1
		%assign CAPSMA_INX %2
		%rep	%3
		UNICAP	UNICAP_CH,CAPSMA_INX
		%assign UNICAP_CH UNICAP_CH + 1
		%assign CAPSMA_INX CAPSMA_INX + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode small with capital index
; %1 = starting index of capital, %2 = number of characters

%macro		UNISMAINX 2
		%assign CAPSMA_INX %1
		%rep	%2
		UNISMA	UNINOASC,CAPSMA_INX
		%assign CAPSMA_INX CAPSMA_INX + 1
		%endrep
%endmacro

; ------------- Macro - Initialized Unicode small with capital index
; %1 = starting character, %2 = starting capital, %3 = number of characters

%macro		UNISMAINXCH 3
		%assign UNISMA_CH %1
		%assign CAPSMA_INX %2
		%rep	%3
		UNISMA	UNISMA_CH,CAPSMA_INX
		%assign UNISMA_CH UNISMA_CH + 1
		%assign CAPSMA_INX CAPSMA_INX + 1
		%endrep
%endmacro

; ------------- Macro - Area of invalid Unicode character type tables
; %1 = number of tables

%macro		INVCHTYPE 1
		%rep	%1
		dd	CHTypeInv
		%endrep
%endmacro

; -----------------------------------------------------------------------------
;                   Get Unicode character type descriptor
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	EBX = pointer to Unicode character type descriptor UNITYPE
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

GetUniType:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	GetUniType4	; invalid character

; ------------- Push registers

                push	eax		; push EAX

; ------------- Get pointer to descriptor (-> EBX)

		GETUNITYPE eax,al,ah,eax,ebx ; get pointer to descriptor

; ------------- Pop registers

		pop	eax		; pop EAX
		ret

; ------------- Invalid Unicode character (character is out of range)

GetUniType4:	mov	ebx,CHTypeInv	; EBX <- invalid character
		ret

; -----------------------------------------------------------------------------
;                      Get flags of Unicode character type
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	DL = flags of Unicode character type (see CHNONE, CHCTR, ...)
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

GetUniTypeFlags:cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	GetUniTypeFlgs4	; invalid character

; ------------- Push registers

                push	eax		; push EAX
		push	ebx		; push EBX

; ------------- Get flags (-> DL)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	eax,al		; EAX <- character LOW
		mov	dl,[ebx+eax*UNITYPE_size+UNI_Flags] ; DL <- flags

; ------------- Pop registers

		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Invalid Unicode character (character is out of range)

GetUniTypeFlgs4:mov	dl,CHNONE	; DL <- invalid character
		ret

; -----------------------------------------------------------------------------
;                      Check if Unicode character is valid
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	CY = character is not valid
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

CheckUniChar:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	CheckUniChar4	; invalid character

; ------------- Push registers

                push	eax		; push EAX
		push	ebx		; push EBX

; ------------- Check if character is valid (-> CY = character is not valid)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	eax,al		; EAX <- character LOW
		cmp	byte [ebx+eax*UNITYPE_size+UNI_Flags],1 ; check flags

; ------------- Pop registers

		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Invalid Unicode character (character is out of range)

CheckUniChar4:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;          Check if Unicode character is a letter (small or capital)
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	NZ = character is a letter
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

CheckUniLetter:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	CheckUniLetter4	; invalid character

; ------------- Push registers

                push	eax		; push EAX
		push	ebx		; push EBX

; ------------- Check if character is a letter

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	eax,al		; EAX <- character LOW
		test	byte [ebx+eax*UNITYPE_size+UNI_Flags],CHCAP+CHSMA

; ------------- Pop registers

		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Invalid Unicode character (character is out of range)

CheckUniLetter4:cmp	eax,eax		; set ZF (character is not a letter)
		ret

; -----------------------------------------------------------------------------
;                Convert Unicode character to ASCII alternative
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	EAX = ASCII alternative
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharToASCII:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharToASCII4	; invalid character

; ------------- Push registers

		push	ebx		; push EBX

; ------------- Get ASCII alternative (-> EAX)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	eax,al		; EAX <- character LOW
		mov	al,[ebx+eax*UNITYPE_size+UNI_ASCII] ; EAX <- ASCII

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; ------------- Invalid Unicode character (character is out of range)

UniCharToASCII4:mov	eax,UNINOASC	; EAX <- invalid character
		ret

; -----------------------------------------------------------------------------
;      Get alphabetic sorting value of Unicode character - case sensitive
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
;		EDI = pointer to nationality descriptor NATIONAL
; OUTPUT:	EAX = alphabetic sorting value
;			B0..B15 = unicode character
;			B16..B23 = ASCII alternative
;			B24..B31 = character flags (character group)
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharSort:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharSort8	; invalid character

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Prepare table address (-> EBX,ECX)

UniCharSort2:	movzx	ebx,ah		; EBX <- character HIGH
		movzx	ecx,al		; EAX <- character LOW
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable

; ------------- Get sorting value (-> EAX)

UniCharSort4:	shl	eax,16		; EAX <- shift Unicode character left
		mov	ax,[ebx+ecx*UNITYPE_size] ; AX <- Flags + ASCII
		xchg	al,ah		; AL <- ASCII, AH <- Flags
		ror	eax,16		; do correction of bit position

; ------------- Prepare sorting table

		movzx	ebx,ah		; EBX <- character HIGH
		shl	ebx,2		; EBX <- character HIGH * 4
		add	ebx,[edi+NAT_AlphaHash] ; EBX <- pointer to hash table
		mov	ebx,[ebx]	; EBX <- hash table

; ------------- Get sorting value

		cmp	byte [ebx+ecx],0 ; is sorting value valid?
		je	UniCharSort6	; sorting value is not valid
		ror	eax,16		; shift bits
		mov	al,[ebx+ecx]	; AL <- get sorting value
		ror	eax,16		; shift bits

; ------------- Pop registers

UniCharSort6:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; ------------- Invalid Unicode character (character is out of range)

UniCharSort8:	mov	eax,UNINOASC	; EAX <- invalid character
		ret

; -----------------------------------------------------------------------------
;       Get alphabetic sorting value of Unicode character - capital case
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
;		EDI = pointer to nationality descriptor NATIONAL
; OUTPUT:	EAX = alphabetic sorting value
;			B0..B15 = unicode character
;			B16..B23 = ASCII alternative
;			B24..B31 = character flags (character group)
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharSortCap:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharSort8	; invalid character

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Prepare table address (-> EBX,ECX)

		movzx	ebx,ah		; EBX <- character HIGH
		movzx	ecx,al		; EAX <- character LOW
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable

; ------------- Check if it has valid capital letter

		test	byte [ebx+ecx*UNITYPE_size+UNI_Flags],CHSMA ; small?
		jz	UniCharSort4	; not small letter
		cmp	word [ebx+ecx*UNITYPE_size+UNI_Change],0 ; change?
		je	UniCharSort4	; no change

; ------------- Change to capital letter

		mov	ax,[ebx+ecx*UNITYPE_size+UNI_Change] ; capital letter
		jmp	short UniCharSort2

; -----------------------------------------------------------------------------
;                 Convert Unicode character to small letter
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	EAX = small letter
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharToSmall:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharToSmall4	; invalid character

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Get type descriptor (-> ECX)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	ecx,al		; ECX <- character LOW
		mov	ecx,[ebx+ecx*UNITYPE_size] ; ECX <- descriptor

; ------------- Check if character is a capital letter

		test	cl,CHCAP	; is it a capital letter?
		jz	UniCharToSmall2	; it is not capital letter

; ------------- Convert to small letter

		shr	ecx,16		; ECX <- small letter
		jz	UniCharToSmall2	; there is no small letter
		xchg	eax,ecx		; EAX <- small letter

; ------------- Pop registers

UniCharToSmall2:pop	ecx		; pop ECX
		pop	ebx		; pop EBX
UniCharToSmall4:ret

; -----------------------------------------------------------------------------
;                 Convert Unicode character to capital letter
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	EAX = capital letter
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharToCap:	cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharToCap4	; invalid character

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Get type descriptor (-> ECX)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	ecx,al		; ECX <- character LOW
		mov	ecx,[ebx+ecx*UNITYPE_size] ; ECX <- descriptor

; ------------- Check if character is a small letter

		test	cl,CHSMA	; is it a small letter?
		jz	UniCharToCap2	; it is not small letter

; ------------- Convert to capital letter

		shr	ecx,16		; ECX <- capital letter
		jz	UniCharToCap2	; there is no capital letter
		xchg	eax,ecx		; EAX <- capital letter

; ------------- Pop registers

UniCharToCap2:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
UniCharToCap4:	ret

; -----------------------------------------------------------------------------
;          Convert Unicode character to small/capital letter (invert)
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
; OUTPUT:	EAX = small/capital letter
; -----------------------------------------------------------------------------

; ------------- Check range of Unicode character

UniCharToSmaCap:cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	UniCharToSmaCa4	; invalid character

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Get small/capital letter (-> ECX)

		movzx	ebx,ah		; EBX <- character HIGH
		mov	ebx,[CHTypeTab+ebx*4] ; EBX <- subtable
		movzx	ecx,al		; ECX <- character LOW
		movzx	ecx,word [ebx+ecx*UNITYPE_size+UNI_Change] ;ECX<-letter

; ------------- Convert to small/capital letter

		jecxz	UniCharToSmaCa2	; there is no small/capital letter
		xchg	eax,ecx		; EAX <- small/capital letter

; ------------- Pop registers

UniCharToSmaCa2:pop	ecx		; pop ECX
		pop	ebx		; pop EBX
UniCharToSmaCa4:ret

; -----------------------------------------------------------------------------
;                  Check table of Unicode character types
; -----------------------------------------------------------------------------

%ifdef DEBUG_UNICHECK

CheckUniType:	mov	esi,DebUniTypeTxt0
		call	DebOutText

; ------------- Loop through all characters

		xor	ebp,ebp		; EBP <- character

; ------------- Get pointer to character descriptor (-> EBX)

CheckUniType2:	mov	eax,ebp		; EAX <- character
		call	GetUniType	; EBX <- pointer to descriptor
		cmp	byte [ebx+UNI_Flags],CHNONE ; is character valid?
		je	CheckUniType4	; character is not valid

; ------------- Check validity of ASCII alternative character

		mov	esi,DebUniTypeTxtAs ; ESI <- text
		cmp	byte [ebx+UNI_ASCII],80h ; check ASCII character
		jae	CheckUniType7	; invalid ASCII character
		test	byte [ebx+UNI_Flags],CHCTR ; control character?
		jnz	CheckUniType3	; control character
		cmp	byte [ebx+UNI_ASCII],20h ; check ASCII character
		jb	CheckUniType7	; invalid ASCII character

; ------------- Check if it has change small/capital

CheckUniType3:	movzx	eax,word [ebx+UNI_Change] ; EAX <- change
		or	eax,eax		; change small/capital?
		je	CheckUniType4	; no change

; ------------- Check flags

		mov	esi,DebUniTypeTxtCh ; ESI <- text
		mov	dl,[ebx+UNI_Flags] ; DL <- flags
		and	dl,CHCAP+CHSMA	; DL <- flags
		jz	CheckUniType7	; invalid flags

; ------------- Get change character

		cmp	eax,CHTYPEMAX	; check maximal Unicode character
		ja	CheckUniType7	; invalid character
		call	GetUniType	; EBX <- pointer to descriptor

; ------------- Check change flags

		mov	dh,[ebx+UNI_Flags] ; DH <- flags
		and	dh,CHCAP+CHSMA	; DH <- flags
		xor	dh,CHCAP+CHSMA	; DH <- flip flags
		cmp	dl,dh		; check flags
		jne	CheckUniType7	; invalid flags

; ------------- Check character

		movzx	eax,word [ebx+UNI_Change] ; EAX <- change
		cmp	eax,ebp		; check character
		jne	CheckUniType7	; character is not OK

; ------------- Next character

CheckUniType4:	inc	ebp		; EBP <- next character
		cmp	ebp,CHTYPEMAX	; check character
		jbe	CheckUniType2	; next character

; ------------- Prepare to check character set tables

		mov	ebp,CharSetTab	; EBP <- character set tables
CheckUniType5:	test	byte [ebp+CHSET_Flags],CHSET_MBYTE ; multibyte?
		jnz	CheckUniType9	; multibyte character set

; ------------- Prepare to check one table

		mov	esi,[ebp+CHSET_ToUni] ; EDX <- table to Unicode
		mov	ecx,128		; ECX <- number of characters

; ------------- Get one character

CheckUniType6:	xor	eax,eax		; EAX <- 0
		lodsw			; EAX <- character
		or	eax,eax		; valid character?
		jz	CheckUniType8	; invalid character

; ------------- Check if character is valid

		call	GetUniType	; EBX <- descriptor
		cmp	byte [ebx+UNI_Flags],CHNONE ; valid character?
		jne	CheckUniType8	; character si valid OK

; ------------- ERROR (ESI = error text)

		mov	esi,DebUniTypeTxtFn ; ESI <- text
		xchg	eax,ebp		; EBP <- character
CheckUniType7:	call	DebOutText	; display error text
		call	DebOutSpc	; display space
		xchg	eax,ebp		; EAX <- character
		call	DebOutHexW	; display character
		jmp	$

; ------------- Next character in one table

CheckUniType8:	loop	CheckUniType6	; next character

; ------------- Next character set table

CheckUniType9:	add	ebp,CHARSET_size ; EBP <- next character set
		cmp	ebp,CharSetTab2	; end of table?
		jb	CheckUniType5	; next table	

; ------------- All OK

		mov	esi,DebUniTypeTxtOK
		call	DebOutText
		ret

%endif ; DEBUG_UNICHECK

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

%ifdef DEBUG_UNICHECK
DebUniTypeTxt0:	db	10,'Unicode character types: ',0
DebUniTypeTxtOK:db	'OK',10,0
DebUniTypeTxtAs:db	'Invalid ASCII',0
DebUniTypeTxtCh:db	'Invalid change small/capital',0
DebUniTypeTxtFn:db	'Not found',0
%endif ; DEBUG_UNICHECK

; ------------- Unicode character types - directory table
[list -]

		align	4, db 0
CHTypeTab:	dd	CHType0000	; 00xx
		dd	CHType0100	; 01xx
		dd	CHType0200	; 02xx
		dd	CHType0300	; 03xx
		dd	CHType0400	; 04xx
		dd	CHType0500	; 05xx
		dd	CHType0600	; 06xx
		INVCHTYPE 7		; 07xx .. 0Dxx
		dd	CHType0E00	; 0Exx
		INVCHTYPE 14		; 0Fxx .. 1Cxx
		dd	CHType1D00	; 1Dxx
		dd	CHType1E00	; 1Exx
		dd	CHType1F00	; 1Fxx
		dd	CHType2000	; 20xx
		dd	CHType2100	; 21xx
		dd	CHType2200	; 22xx
		dd	CHType2300	; 23xx
		dd	CHType2400	; 24xx
		dd	CHType2500	; 25xx
		dd	CHType2600	; 26xx
		dd	CHType2700	; 27xx
		dd	CHType2800	; 28xx
		dd	CHType2900	; 29xx
		dd	CHType2A00	; 2Axx
		dd	CHType2B00	; 2Bxx
		dd	CHType2C00	; 2Cxx

; ------------- Unicode character types

		align	4, db 0

CHType2B00:	UNISIGNUM 27		; 2B00..2B1A symbols
		UNIINV	5		; 2B1B..2B1F
		UNISIGNUM 4		; 2B20..2B23
;		UNIINV	220		; 2B24..2BFF
; CHTypeInv must follow.

CHTypeInv:	UNIINV	256		; invalid unicode type table

CHType0000:	UNICTRCH 0,32		; 0000..001F
		UNISPC	1		; 0020
		UNIMRK	"!"		; 0021 !
		UNIMRK	34		; 0022 "
		UNISIG	"#"		; 0023 #
		UNISIG	"$"		; 0024 $
		UNISIG	"%"		; 0025 %
		UNISIG	"&"		; 0026 &
		UNIMRK	39		; 0027 '
		UNIMRK	"("		; 0028 (
		UNIMRK	")"		; 0029 )
		UNISIG	"*"		; 002A *
		UNISIG	"+"		; 002B +
		UNIMRK	","		; 002C ,
		UNISIG	"-"		; 002D -
		UNIMRK	"."		; 002E .
		UNISIG	"/"		; 002F /
		UNIDIGCH "0",10		; 0030..0039 0..9
		UNIMRK	":"		; 003A :
		UNIMRK	";"		; 003B ;
		UNISIG	"<"		; 003C <
		UNISIG	"="		; 003D =
		UNISIG	">"		; 003E >
		UNIMRK	"?"		; 003F ?
		UNISIG	"@"		; 0040 @
		UNICAPINXCH "A","a",26	; 0041..005A A..Z
		UNIMRK	"["		; 005B [
		UNISIG	"\"		; 005C "\"
		UNIMRK	"]"		; 005D ]
		UNIMRK	"^"		; 005E ^
		UNISIG	"_"		; 005F _
		UNIMRK	"`"		; 0060 `
		UNISMAINXCH "a","A",26	; 0061..007A a..z
		UNIMRK	"{"		; 007B {
		UNISIG	"|"		; 007C |
		UNIMRK	"}"		; 007D }
		UNISIG	"~"		; 007E ~
		UNICTRCH 7fh,1		; 007F delete
		UNICTRCH 0,32		; 0080..009F
		UNISPC	1		; 00A0
		UNIMRK	"!"		; 00A1 inverted !
		UNISIG	"c"		; 00A2 cent sign
		UNISIG	"L"		; 00A3 pount
		UNISIG	"o"		; 00A4 currency
		UNISIG	"Y"		; 00A5 yen
		UNISIG	"|"		; 00A6 broken bar
		UNISIG	UNINOASC	; 00A7 section
		UNIMRK	UNINOASC	; 00A8 diaeresis
		UNISIG	"c"		; 00A9 copyright
		UNISIG	"a"		; 00AA feminine ordinal
		UNIMRK	"<"		; 00AB left double angle
		UNISIG	"-"		; 00AC not
		UNISIG	"-"		; 00AD soft hyphen
		UNISIG	"R"		; 00AE registered
		UNIMRK	"-"		; 00AF macron
		UNISIG	UNINOASC	; 00B0 degree
		UNISIG	UNINOASC	; 00B1 plus-minus
		UNISIG	"2"		; 00B2 superscript two
		UNISIG	"3"		; 00B3 superscript three
		UNIMRK	39		; 00B4 acute '
		UNISIG	"u"		; 00B5 micro
		UNISIG	"|"		; 00B6 pilcrow
		UNIMRK	"."		; 00B7 middle dot
		UNIMRK	","		; 00B8 cedilla
		UNISIG	"1"		; 00B9 superscript one
		UNISIG	"o"		; 00BA masculine ordinal
		UNIMRK	">"		; 00BB right double angle
		UNISIG	UNINOASC	; 00BC one quarter
		UNISIG	UNINOASC	; 00BD one half
		UNISIG	UNINOASC	; 00BE three quarters
		UNIMRK	"?"		; 00BF inverted question
		UNICAP	"A",0e0h	; 00C0 A grave
		UNICAP	"A",0e1h	; 00C1 A acute
		UNICAP	"A",0e2h	; 00C2 A circumflex
		UNICAP	"A",0e3h	; 00C3 A tilde
		UNICAP	"A",0e4h	; 00C4 A diaeresis
		UNICAP	"A",0e5h	; 00C5 A ring
		UNICAP	"A",0e6h	; 00C6 AE
		UNICAP	"C",0e7h	; 00C7 C cedilla
		UNICAP	"E",0e8h	; 00C8 E grave
		UNICAP	"E",0e9h	; 00C9 E acute
		UNICAP	"E",0eah	; 00CA E circumflex
		UNICAP	"E",0ebh	; 00CB E diaeresis
		UNICAP	"I",0ech	; 00CC I grave
		UNICAP	"I",0edh	; 00CD I acute
		UNICAP	"I",0eeh	; 00CE I circumflex
		UNICAP	"I",0efh	; 00CF I diaeresis
		UNICAP	"D",0f0h	; 00D0 D eth
		UNICAP	"N",0f1h	; 00D1 N tilde
		UNICAP	"O",0f2h	; 00D2 O grave
		UNICAP	"O",0f3h	; 00D3 O acute
		UNICAP	"O",0f4h	; 00D4 O circumflex
		UNICAP	"O",0f5h	; 00D5 O tilde
		UNICAP	"O",0f6h	; 00D6 O diaeresis
		UNISIG	"x"		; 00D7 multiplication
		UNICAP	"O",0f8h	; 00D8 O stroke
		UNICAP	"U",0f9h	; 00D9 U grave
		UNICAP	"U",0fah	; 00DA U acute
		UNICAP	"U",0fbh	; 00DB U circumflex
		UNICAP	"U",0fch	; 00DC U diaeresis
		UNICAP	"Y",0fdh	; 00DD Y acute
		UNICAP	"I",0feh	; 00DE I thorn
		UNISMA	"B",0		; 00DF small sharp s
		UNISMA	"a",0c0h	; 00E0 a grave
		UNISMA	"a",0c1h	; 00E1 a acute
		UNISMA	"a",0c2h	; 00E2 a circumflex
		UNISMA	"a",0c3h	; 00E3 a tilde
		UNISMA	"a",0c4h	; 00E4 a diaeresis
		UNISMA	"a",0c5h	; 00E5 a ring
		UNISMA	"a",0c6h	; 00E6 ae
		UNISMA	"c",0c7h	; 00E7 c cedilla
		UNISMA	"e",0c8h	; 00E8 e grave
		UNISMA	"e",0c9h	; 00E9 e acute
		UNISMA	"e",0cah	; 00EA e circumflex
		UNISMA	"e",0cbh	; 00EB e diaeresis
		UNISMA	"i",0cch	; 00EC i grave
		UNISMA	"i",0cdh	; 00ED i acute
		UNISMA	"i",0ceh	; 00EE i circumflex
		UNISMA	"i",0cfh	; 00EF i diaeresis
		UNISMA	"d",0d0h	; 00F0 d eth
		UNISMA	"n",0d1h	; 00F1 n tilde
		UNISMA	"o",0d2h	; 00F2 o grave
		UNISMA	"o",0d3h	; 00F3 o acute
		UNISMA	"o",0d4h	; 00F4 o circumflex
		UNISMA	"o",0d5h	; 00F5 o tilde
		UNISMA	"o",0d6h	; 00F6 o diaeresis
		UNISIG	":"		; 00F7 division
		UNISMA	"o",0d8h	; 00F8 o stroke
		UNISMA	"u",0d9h	; 00F9 u grave
		UNISMA	"u",0dah	; 00FA u acute
		UNISMA	"u",0dbh	; 00FB u circumflex
		UNISMA	"u",0dch	; 00FC u diaeresis
		UNISMA	"y",0ddh	; 00FD y acute
		UNISMA	"i",0deh	; 00FE i thorn
		UNISMA	"y",178h	; 00FF y diaeresis

CHType0100:	UNICAP	"A",101h	; 0100 A macron
		UNISMA	"a",100h	; 0101 a macron
		UNICAP	"A",103h	; 0102 A breve
		UNISMA	"a",102h	; 0103 a breve
		UNICAP	"A",105h	; 0104 A ogonek
		UNISMA	"a",104h	; 0105 a ogonek
		UNICAP	"C",107h	; 0106 C acute
		UNISMA	"c",106h	; 0107 c acite
		UNICAP	"C",109h	; 0108 C circumflex
		UNISMA	"c",108h	; 0109 c circumflex
		UNICAP	"C",10bh	; 010A C dot
		UNISMA	"c",10ah	; 010B c dot
		UNICAP	"C",10dh	; 010C C caron
		UNISMA	"c",10ch	; 010D c caron
		UNICAP	"D",10fh	; 010E D caron
		UNISMA	"d",10eh	; 010F d caron
		UNICAP	"D",111h	; 0110 D stroke
		UNISMA	"d",110h	; 0111 d stroke
		UNICAP	"E",113h	; 0112 E macron
		UNISMA	"e",112h	; 0113 e macron
		UNICAP	"E",115h	; 0114 E breve
		UNISMA	"e",114h	; 0115 e breve
		UNICAP	"E",117h	; 0116 E dot
		UNISMA	"e",116h	; 0117 e dot
		UNICAP	"E",119h	; 0118 E ogonek
		UNISMA	"e",118h	; 0119 e ogonek
		UNICAP	"E",11bh	; 011A E caron
		UNISMA	"e",11ah	; 011B e caron
		UNICAP	"G",11dh	; 011C G circumflex
		UNISMA	"g",11ch	; 011D g circumflex
		UNICAP	"G",11fh	; 011E G breve
		UNISMA	"g",11eh	; 011F g breve
		UNICAP	"G",121h	; 0120 G dot
		UNISMA	"g",120h	; 0121 g dot
		UNICAP	"G",123h	; 0122 G cedilla
		UNISMA	"g",122h	; 0123 g cedilla
		UNICAP	"H",125h	; 0124 H circumflex
		UNISMA	"h",124h	; 0125 h circumflex
		UNICAP	"H",127h	; 0126 H stroke
		UNISMA	"h",126h	; 0127 h stroke
		UNICAP	"I",129h	; 0128 I tilde
		UNISMA	"i",128h	; 0129 i tilde
		UNICAP	"I",12bh	; 012A I macron
		UNISMA	"i",12ah	; 012B i macron
		UNICAP	"I",12dh	; 012C I breve
		UNISMA	"i",12ch	; 012D i breve
		UNICAP	"I",12fh	; 012E I ogonek
		UNISMA	"i",12eh	; 012F i ogonek
; In Turkish+Azerbaijani it should be paired:
;   I dot (130h) + i dot (69h), I dotless (49h) + i dotless (131h)
		UNICAP	"I",131h	; 0130 I dot
		UNISMA	"i",130h	; 0131 i dotless
		UNICAP	"I",133h	; 0132 IJ
		UNISMA	"i",132h	; 0133 ij
		UNICAP	"J",135h	; 0134 J circumflex
		UNISMA	"j",134h	; 0135 j circumflex
		UNICAP	"K",137h	; 0136 K cedilla
		UNISMA	"k",136h	; 0137 k cedilla
		UNISMA	"k",0		; 0138 small kra
		UNICAP	"L",13ah	; 0139 L acute
		UNISMA	"l",139h	; 013A l acute
		UNICAP	"L",13ch	; 013B L cedilla
		UNISMA	"l",13bh	; 013C l cedilla
		UNICAP	"L",13eh	; 013D L caron
		UNISMA	"l",13dh	; 013E l caron
		UNICAP	"L",140h	; 013F L dot
		UNISMA	"l",13fh	; 0140 l dot
		UNICAP	"L",142h	; 0141 L stroke
		UNISMA	"l",141h	; 0142 l stroke
		UNICAP	"N",144h	; 0143 N acute
		UNISMA	"n",143h	; 0144 n acute
		UNICAP	"N",146h	; 0145 N cedilla
		UNISMA	"n",145h	; 0146 n cedilla
		UNICAP	"N",148h	; 0147 N caron
		UNISMA	"n",147h	; 0148 n caron
		UNISMA	"n",0		; 0149 n apostrophe
		UNICAP	"N",14bh	; 014A N eng
		UNISMA	"n",14ah	; 014B n eng
		UNICAP	"O",14dh	; 014C O macron
		UNISMA	"o",14ch	; 014D o macron
		UNICAP	"O",14fh	; 014E O breve
		UNISMA	"o",14eh	; 014F o breve
		UNICAP	"O",151h	; 0150 O double acute
		UNISMA	"o",150h	; 0151 o double acute
		UNICAP	"O",153h	; 0152 OE
		UNISMA	"o",152h	; 0153 oe
		UNICAP	"R",155h	; 0154 R acute
		UNISMA	"r",154h	; 0155 r acute
		UNICAP	"R",157h	; 0156 R cedilla
		UNISMA	"r",156h	; 0157 r cedilla
		UNICAP	"R",159h	; 0158 R caron
		UNISMA	"r",158h	; 0159 r caron
		UNICAP	"S",15bh	; 015A S acute
		UNISMA	"s",15ah	; 015B s acute
		UNICAP	"S",15dh	; 015C S circumflex
		UNISMA	"s",15ch	; 015D s circumflex
		UNICAP	"S",15fh	; 015E S cedilla
		UNISMA	"s",15eh	; 015F s cedilla
		UNICAP	"S",161h	; 0160 S caron
		UNISMA	"s",160h	; 0161 s caron
		UNICAP	"T",163h	; 0162 T cedilla
		UNISMA	"t",162h	; 0163 t cedilla
		UNICAP	"T",165h	; 0164 T caron
		UNISMA	"t",164h	; 0165 t caron
		UNICAP	"T",167h	; 0166 T stroke
		UNISMA	"t",166h	; 0167 t stroke
		UNICAP	"U",169h	; 0168 U tilde
		UNISMA	"u",168h	; 0169 u tilde
		UNICAP	"U",16bh	; 016A U macron
		UNISMA	"u",16ah	; 016B u macron
		UNICAP	"U",16dh	; 016C U breve
		UNISMA	"u",16ch	; 016D u breve
		UNICAP	"U",16fh	; 016E U ring
		UNISMA	"u",16eh	; 016F u ring
		UNICAP	"U",171h	; 0170 U double acute
		UNISMA	"u",170h	; 0171 u double acute
		UNICAP	"U",173h	; 0172 U ogonek
		UNISMA	"u",172h	; 0173 u ogonek
		UNICAP	"W",175h	; 0174 W circumflex
		UNISMA	"w",174h	; 0175 w circumflex
		UNICAP	"Y",177h	; 0176 Y circumflex
		UNISMA	"y",176h	; 0177 y circumflex
		UNICAP	"Y",0ffh	; 0178 Y diaeresis
		UNICAP	"Z",17ah	; 0179 Z acute
		UNISMA	"z",179h	; 017A z acute
		UNICAP	"Z",17ch	; 017B Z dot
		UNISMA	"z",17bh	; 017C z dot
		UNICAP	"Z",17eh	; 017D Z caron
		UNISMA	"z",17dh	; 017E z caron
		UNISMA	"f",0		; 017F long s
		UNISMA	"b",243h	; 0180 b stroke
		UNICAP	"B",253h	; 0181 B hook
		UNICAP	"B",183h	; 0182 B topbar
		UNISMA	"b",182h	; 0183 b topbar
		UNICAP	"B",185h	; 0184 B tone six
		UNISMA	"b",184h	; 0185 b tone six
		UNICAP	"C",254h	; 0186 open O
		UNICAP	"C",188h	; 0187 C hook
		UNISMA	"c",187h	; 0188 c hook
		UNICAP	"D",0		; 0189 african D
		UNICAP	"D",257h	; 018A D hook
		UNICAP	"D",18ch	; 018B D topbar
		UNISMA	"d",18bh	; 018C d topbar
		UNISMA	"o",0		; 018D small turned delta
		UNICAP	"E",1ddh	; 018E reversed E
		UNICAP	"E",259h	; 018F capital schwa
		UNICAP	"E",25bh	; 0190 open E
		UNICAP	"F",192h	; 0191 F hook
		UNISMA	"f",191h	; 0192 f hook
		UNICAP	"G",260h	; 0193 G hook
		UNICAP	"Y",263h	; 0194 capital gamma
		UNISMA	"h",1f6h	; 0195 small hv
		UNICAP	"I",269h	; 0196 capital iota
		UNICAP	"I",268h	; 0197 I stroke
		UNICAP	"K",199h	; 0198 K hook
		UNISMA	"k",198h	; 0199 k hook
		UNISMA	"l",23dh	; 019A l bar
		UNISMA	"l",0		; 019B small lambda stroke
		UNICAP	"M",26fh	; 019C turned M
		UNICAP	"N",272h	; 019D N hook
		UNISMA	"n",220h	; 019E n long leg
		UNICAP	"O",275h	; 019F O tilde
		UNICAP	"O",1a1h	; 01A0 O horn
		UNISMA	"o",1a0h	; 01A1 o horn
		UNICAP	"O",1a3h	; 01A2 OI
		UNISMA	"o",1a2h	; 01A3 oi
		UNICAP	"P",1a5h	; 01A4 P hook
		UNISMA	"p",1a4h	; 01A5 p hook
		UNICAP	"R",280h	; 01A6 capital yr
		UNICAP	UNINOASC,1a8h	; 01A7 capitel tone two
		UNISMA	UNINOASC,1a7h	; 01A8 small tone two
		UNICAP	UNINOASC,283h	; 01A9 capital esh
		UNICAP	UNINOASC,0	; 01AA esh loop
		UNISMA	"t",0		; 01AB t palatal hook
		UNICAP	"T",1adh	; 01AC T hook
		UNISMA	"t",1ach	; 01AD t hook
		UNICAP	"T",288h	; 01AE T retroflex hook
		UNICAP	"U",1b0h	; 01AF U horn
		UNISMA	"u",1afh	; 01B0 u horn
		UNICAP	"U",28ah	; 01B1 capital upsilon
		UNICAP	"V",28bh	; 01B2 V hook
		UNICAP	"Y",1b4h	; 01B3 Y hook
		UNISMA	"y",1b3h	; 01B4 y hook
		UNICAP	"Z",1b6h	; 01B5 Z stroke
		UNISMA	"z",1b5h	; 01B6 z stroke
		UNICAP	UNINOASC,292h	; 01B7 capital ezh
		UNICAP	UNINOASC,1b9h	; 01B8 capital ezh reversed
		UNISMA	UNINOASC,1b8h	; 01B9 small ezh reversed
		UNISMA	UNINOASC,0	; 01BA small ezh tail
		UNICAP	"2",0		; 01BB 2 stroke
		UNICAP	UNINOASC,1bdh	; 01BC capital tone five
		UNISMA	UNINOASC,1bch	; 01BD small tone five
		UNICAP	UNINOASC,0	; 01BE inverted glottal stop stroke
		UNISMA	"p",1f7h	; 01BF small wynn
		UNICAP	"|",0		; 01C0 dental click
		UNICAP	"|",0		; 01C1 lateral click
		UNICAP	"+",0		; 01C2 alveolar click
		UNICAP	"!",0		; 01C3 retroflex click
		UNICAP	"D",1c6h	; 01C4 DZ caron
		UNICAP	"D",0		; 01C5 Dz caron
		UNISMA	"d",1c4h	; 01C6 dz caron
		UNICAP	"L",1c9h	; 01C7 LJ
		UNICAP	"L",0		; 01C8 Lj
		UNISMA	"l",1c7h	; 01C9 lj
		UNICAP	"N",1cch	; 01CA NJ
		UNICAP	"N",0		; 01CB Nj
		UNISMA	"n",1cah	; 01CC nj
		UNICAP	"A",1ceh	; 01CD A caron
		UNISMA	"a",1cdh	; 01CE a caron
		UNICAP	"I",1d0h	; 01CF I caron
		UNISMA	"i",1cfh	; 01D0 i caron
		UNICAP	"O",1d2h	; 01D1 O caron
		UNISMA	"o",1d1h	; 01D2 o caron
		UNICAP	"U",1d4h	; 01D3 U caron
		UNISMA	"u",1d3h	; 01D4 u caron
		UNICAP	"U",1d6h	; 01D5 U diaeresis
		UNISMA	"u",1d5h	; 01D6 u diaeresis
		UNICAP	"U",1d8h	; 01D7 U diaeresis acute
		UNISMA	"u",1d7h	; 01D8 u diaeresis acute
		UNICAP	"U",1dah	; 01D9 U diaeresis caron
		UNISMA	"u",1d9h	; 01DA u diaeresis caron
		UNICAP	"U",1dch	; 01DB U diaeresis grave
		UNISMA	"u",1dbh	; 01DC u diaeresis grave
		UNISMA	"e",18eh	; 01DD turned e
		UNICAP	"A",1dfh	; 01DE A diaeresis macron
		UNISMA	"a",1deh	; 01DF a diaeresis macron
		UNICAP	"A",1e1h	; 01E0 A dot macron
		UNISMA	"a",1e0h	; 01E1 a dot macron
		UNICAP	"A",1e3h	; 01E2 AE macron
		UNISMA	"a",1e2h	; 01E3 ae macron
		UNICAP	"G",1e5h	; 01E4 G stroke
		UNISMA	"g",1e4h	; 01E5 g stroke
		UNICAP	"G",1e7h	; 01E6 G caron
		UNISMA	"g",1e6h	; 01E7 g caron
		UNICAP	"K",1e9h	; 01E8 K caron
		UNISMA	"k",1e8h	; 01E9 k caron
		UNICAP	"O",1ebh	; 01EA O ogonek
		UNISMA	"o",1eah	; 01EB o ogonek
		UNICAP	"O",1edh	; 01EC O ogonek macron
		UNISMA	"o",1ech	; 01ED o ogonek macron
		UNICAP	UNINOASC,1efh	; 01EE capital ezh caron
		UNISMA	UNINOASC,1eeh	; 01EF small ezh caron
		UNISMA	"j",0		; 01F0 j caron
		UNICAP	"D",1f3h	; 01F1 DZ
		UNICAP	"D",0		; 01F2 Dz
		UNISMA	"d",1f1h	; 01F3 dz
		UNICAP	"G",1f5h	; 01F4 G acute
		UNISMA	"g",1f4h	; 01F5 g acute
		UNICAP	"H",195h	; 01F6 capital hwair
		UNICAP	"P",1bfh	; 01F7 capital wynn
		UNICAP	"N",1f9h	; 01F8 N grave
		UNISMA	"n",1f8h	; 01F9 n grave
		UNICAP	"A",1fbh	; 01FA A ring
		UNISMA	"a",1fah	; 01FB a ring
		UNICAP	"A",1fdh	; 01FC AE acute
		UNISMA	"a",1fch	; 01FD ae acute
		UNICAP	"O",1ffh	; 01FE O stroke acute
		UNISMA	"o",1feh	; 01FF o stroke acute

CHType0200:	UNICAP	"A",201h	; 0200 A double grave
		UNISMA	"a",200h	; 0201 a double grave
		UNICAP	"A",203h	; 0202 A inverted breve
		UNISMA	"a",202h	; 0203 a inverted breve
		UNICAP	"E",205h	; 0204 E double grave
		UNISMA	"e",204h	; 0205 e double grave
		UNICAP	"E",207h	; 0206 E inverted breve
		UNISMA	"e",206h	; 0207 e inverted breve
		UNICAP	"I",209h	; 0208 I double grave
		UNISMA	"i",208h	; 0209 i double grave
		UNICAP	"I",20bh	; 020A I inverted breve
		UNISMA	"i",20ah	; 020B i inverted breve
		UNICAP	"O",20dh	; 020C O double grave
		UNISMA	"o",20ch	; 020D o double grave
		UNICAP	"O",20fh	; 020E O inverted breve
		UNISMA	"o",20eh	; 020F o inverted breve
		UNICAP	"R",211h	; 0210 R double grave
		UNISMA	"r",210h	; 0211 r double grave
		UNICAP	"R",213h	; 0212 R inverted breve
		UNISMA	"r",212h	; 0213 r inverted breve
		UNICAP	"U",215h	; 0214 U double grave
		UNISMA	"u",214h	; 0215 u double grave
		UNICAP	"U",217h	; 0216 U inverted breve
		UNISMA	"u",216h	; 0217 u inverted breve
		UNICAP	"S",219h	; 0218 S comma
		UNISMA	"s",218h	; 0219 s comma
		UNICAP	"T",21bh	; 021A T comma
		UNISMA	"t",21ah	; 021B t comma
		UNICAP	"3",21dh	; 021C capital yogh
		UNISMA	"3",21ch	; 021D small yogh
		UNICAP	"H",21fh	; 021E H caron
		UNISMA	"h",21eh	; 021F h caron
		UNICAP	"N",19eh	; 0220 N long leg
		UNISMA	"d",0		; 0221 d curl
		UNICAP	"8",223h	; 0222 capital ou
		UNISMA	"8",222h	; 0223 small ou
		UNICAP	"Z",225h	; 0224 Z hook
		UNISMA	"z",224h	; 0225 z hook
		UNICAP	"A",227h	; 0226 A dot
		UNISMA	"a",226h	; 0227 a dot
		UNICAP	"E",229h	; 0228 E cedilla
		UNISMA	"e",228h	; 0229 e cedilla
		UNICAP	"O",22bh	; 022A O diaeresis macron
		UNISMA	"o",22ah	; 022B o diaeresis macron
		UNICAP	"O",22dh	; 022C O tilde macron
		UNISMA	"o",22ch	; 022D o tilde macron
		UNICAP	"O",22fh	; 022E O dot
		UNISMA	"o",22eh	; 022F o dot
		UNICAP	"O",231h	; 0230 O dot macron
		UNISMA	"o",230h	; 0231 o dot macron
		UNICAP	"Y",233h	; 0232 Y macron
		UNISMA	"y",232h	; 0233 y macron
		UNISMA	"l",0		; 0234 l curl
		UNISMA	"n",0		; 0235 n curl
		UNISMA	"t",0		; 0236 t curl
		UNISMA	"j",0		; 0237 dotless j
		UNISMA	"d",0		; 0238 db
		UNISMA	"q",0		; 0239 qp
		UNICAP	"A",2c65h	; 023A A stroke
		UNICAP	"C",23ch	; 023B C stroke
		UNISMA	"c",23bh	; 023C c stroke
		UNICAP	"L",19ah	; 023D L bar
		UNICAP	"T",2c66h	; 023E T stroke
		UNISMA	"s",0		; 023F s tail
		UNISMA	"z",0		; 0240 z tail
		UNICAP	"?",242h	; 0241 capital glottal stop
		UNISMA	"?",241h	; 0242 small glottal stop
		UNICAP	"B",180h	; 0243 B stroke
		UNICAP	"U",289h	; 0244 U bar
		UNICAP	"V",28ch	; 0245 turned V
		UNICAP	"E",247h	; 0246 E stroke
		UNISMA	"e",246h	; 0247 e stroke
		UNICAP	"J",249h	; 0248 J stroke
		UNISMA	"j",248h	; 0249 j stroke
		UNICAP	"Q",24bh	; 024A Q hook
		UNISMA	"q",24ah	; 024B q hook
		UNICAP	"R",24dh	; 024C R stroke
		UNISMA	"r",24ch	; 024D r stroke
		UNICAP	"Y",24fh	; 024E Y stroke
		UNISMA	"y",24eh	; 024F y stroke
		UNISMA	"a",0		; 0250 turned a
		UNISMA	"a",0		; 0251 small alpha
		UNISMA	"a",0		; 0252 small turned alpha
		UNISMA	"b",181h	; 0253 b hook
		UNISMA	"c",186h	; 0254 open o
		UNISMA	"c",0		; 0255 c curl
		UNISMA	"d",0		; 0256 d tail
		UNISMA	"d",18ah	; 0257 d hook
		UNISMA	"e",0		; 0258 reversed e
		UNISMA	"e",18fh	; 0259 small schwa
		UNISMA	"e",0		; 025A small schwa hook
		UNISMA	"e",190h	; 025B open e
		UNISMA	"e",0		; 025C reversed open e
		UNISMA	"e",0		; 025D open e hook
		UNISMA	"e",0		; 025E closed reversed open e
		UNISMA	"j",0		; 025F dotless j stroke
		UNISMA	"g",193h	; 0260 g hook
		UNISMA	"g",0		; 0261 script g
		UNISMA	"g",0		; 0262 small capital G
		UNISMA	"y",194h	; 0263 small gamma
		UNISMA	"y",0		; 0264 small baby gamma
		UNISMA	"h",0		; 0265 turned h
		UNISMA	"h",0		; 0266 h hook
		UNISMA	"h",0		; 0267 small heng hook
		UNISMA	"i",197h	; 0268 i stroke
		UNISMA	"i",196h	; 0269 small iota
		UNISMA	"i",0		; 026A small capital I
		UNISMA	"l",2c62h	; 026B l tilde
		UNISMA	"l",0		; 026C l belt
		UNISMA	"l",0		; 026D l retroflex hook
		UNISMA	"l",0		; 026E small lezh
		UNISMA	"m",19ch	; 026F turned m
		UNISMA	"m",0		; 0270 turned m long leg
		UNISMA	"m",0		; 0271 m hook
		UNISMA	"n",19dh	; 0272 n hook
		UNISMA	"n",0		; 0273 n retroflex hook
		UNISMA	"n",0		; 0274 small capital N
		UNISMA	"o",19fh	; 0275 barred o
		UNISMA	"o",0		; 0276 small capital OE
		UNISMA	UNINOASC,0	; 0277 small closed omega
		UNISMA	UNINOASC,0	; 0278 small phi
		UNISMA	"r",0		; 0279 turned r
		UNISMA	"r",0		; 027A turned r long leg
		UNISMA	"r",0		; 027B turned r hook
		UNISMA	"r",0		; 027C r long leg
		UNISMA	"r",2c64h	; 027D r tail
		UNISMA	"r",0		; 027E r fishhook
		UNISMA	"r",0		; 027F reversed r fishhook
		UNISMA	"r",1a6h	; 0280 small yr
		UNISMA	"r",0		; 0281 small capital inverted R
		UNISMA	"s",0		; 0282 s hook
		UNISMA	UNINOASC,1a9h	; 0283 small esh
		UNISMA	"j",0		; 0284 dotless j stroke hook
		UNISMA	UNINOASC,0	; 0285 small squat reversed esh
		UNISMA	UNINOASC,0	; 0286 small esh curl
		UNISMA	"t",0		; 0287 small turned t
		UNISMA	"t",1aeh	; 0288 t retroflex hook
		UNISMA	"u",244h	; 0289 u bar
		UNISMA	"u",1b1h	; 028A small upsilon
		UNISMA	"v",1b2h	; 028B v hook
		UNISMA	"v",245h	; 028C turned v
		UNISMA	"w",0		; 028D turned w
		UNISMA	"y",0		; 028E turned y
		UNISMA	"y",0		; 028F small capital Y
		UNISMA	"z",0		; 0290 z retroflex hook
		UNISMA	"z",0		; 0291 z curl
		UNISMA	UNINOASC,1b7h	; 0292 small ezh
		UNISMA	UNINOASC,0	; 0293 small ezh curl
		UNICAP	"?",0		; 0294 glottal stop
		UNICAP	UNINOASC,0	; 0295 pharyngeal voiced fricative
		UNICAP	"?",0		; 0296 inverted glottal stop
		UNICAP	"C",0		; 0297 stretched C
		UNICAP	"O",0		; 0298 bilabial click
		UNISMA	"b",0		; 0299 small capital B
		UNISMA	"e",0		; 029A closed open e
		UNISMA	"g",0		; 029B small capital G hook
		UNISMA	"h",0		; 029C small capital H
		UNISMA	"j",0		; 029D j crossed-tail
		UNISMA	"k",0		; 029E turned k
		UNISMA	"l",0		; 029F small capital L
		UNISMA	"q",0		; 02A0 q hook
		UNICAP	"?",0		; 02A1 glottal stop stroke
		UNICAP	"?",0		; 02A2 reversed glottal stop stroke
		UNISMA	"d",0		; 02A3 small dz digraph
		UNISMA	"d",0		; 02A4 small dezh digraph
		UNISMA	"d",0		; 02A5 small dz digraph curl
		UNISMA	"t",0		; 02A6 small ts digraph
		UNISMA	"t",0		; 02A7 small tesh digraph
		UNISMA	"t",0		; 02A8 small tc digraph curl
		UNISMA	"f",0		; 02A9 small feng digraph
		UNISMA	"l",0		; 02AA small ls digraph
		UNISMA	"l",0		; 02AB small lz digraph
		UNICAP	"w",0		; 02AC bilabial percussive
		UNICAP	"m",0		; 02AD bidental percussive
		UNISMA	"h",0		; 02AE turned h fishhook
		UNISMA	"h",0		; 02AF turned h fishhook tail
		UNIMOD	"h"		; 02B0 modifier h
		UNIMOD	"h"		; 02B1 modifier h hook
		UNIMOD	"j"		; 02B2 modifier j
		UNIMOD	"r"		; 02B3 modifier r
		UNIMOD	"r"		; 02B4 modifier turned r
		UNIMOD	"r"		; 02B5 modifier turned r hook
		UNIMOD	"R"		; 02B6 modifier small cap. inverted R
		UNIMOD	"w"		; 02B7 modifier w
		UNIMOD	"y"		; 02B8 modifier y
		UNIMOD	39		; 02b9 modifier prime
		UNIMOD	34		; 02BA modifier double prime
		UNIMOD	39		; 02BB modifier turned comma
		UNIMOD	39		; 02BC modifier apostrophe
		UNIMOD	39		; 02BD modifier reversed comma
		UNIMOD	39		; 02BE modifier right half ring
		UNIMOD	39		; 02BF modifier left half ring
		UNIMOD	"?"		; 02C0 modifier glottal stop
		UNIMOD	"?"		; 02C1 modifier reversed glottal stop
		UNIMOD	"<"		; 02C2 modifier left arrowhead
		UNIMOD	">"		; 02C3 modifier right arrowhead
		UNIMOD	"^"		; 02C4 modifier up arrowhead
		UNIMOD	"v"		; 02C5 modifier down arrowhead
		UNIMOD	"^"		; 02C6 modifier circumflex accent
		UNIMRK	"^"		; 02C7 caron
		UNIMOD	39		; 02C8 modifier vertical line
		UNIMOD	"~"		; 02C9 modifier macron
		UNIMOD	39		; 02CA modifier acute accent
		UNIMOD	"`"		; 02CB modifier grave accent
		UNIMOD	","		; 02CC modifier low vertical line
		UNIMOD	"_"		; 02CD modifier low macron
		UNIMOD	","		; 02CE modifier low grave accent
		UNIMOD	","		; 02CF modifier low acute accent
		UNIMOD	":"		; 02D0 modifier triangular colon
		UNIMOD	"."		; 02D1 modifier half triangluar colon
		UNIMOD	","		; 02D2 modifier cent. right half ring
		UNIMOD	","		; 02D3 modifier cent. left half ring
		UNIMOD	"+"		; 02D4 modifier up tack
		UNIMOD	"+"		; 02D5 modifier down tack
		UNIMOD	"+"		; 02D6 modifier plus sign
		UNIMOD	"-"		; 02D7 modifier minus sign
		UNIMRK	"^"		; 02D8 breve
		UNIMRK	"."		; 02D9 dot above
		UNIMRK	"^"		; 02DA ring above
		UNIMRK	","		; 02DB ogonek
		UNIMRK	"~"		; 02DC small tilde
		UNIMRK	34		; 02DD double acute accent
		UNIMOD	"-"		; 02DE modifier rhotic hook
		UNIMOD	"x"		; 02DF modifier cross accent
		UNIMOD	"y"		; 02E0 modifier small gamma
		UNIMOD	"l"		; 02E1 modifier small l
		UNIMOD	"s"		; 02E2 modifier small s
		UNIMOD	"x"		; 02E3 modifier small x
		UNIMOD	"?"		; 02E4 modifier small rev. glottal stop
		UNIMOD	"~"		; 02E5 modifier extra-high tone
		UNIMOD	39		; 02E6 modifier high tone
		UNIMOD	"-"		; 02E7 modifier mid tone
		UNIMOD	","		; 02E8 modifier low tone
		UNIMOD	"_"		; 02E9 modifier extra-low tone
		UNIMOD	"^"		; 02EA modifier yin departing tone
		UNIMOD	"."		; 02EB modifier yang departing tone
		UNIMOD	"."		; 02EC modifier voicing
		UNIMOD	"="		; 02ED modifier unaspirated
		UNIMOD	34		; 02EE modifier double apostrophe
		UNIMOD	"v"		; 02EF modifier low down arrowhead
		UNIMOD	"^"		; 02F0 modifier lop up arrowhead
		UNIMOD	"<"		; 02F1 modifier low left arrowhead
		UNIMOD	">"		; 02F2 modifier low right arrowhead
		UNIMOD	"."		; 02F3 modifier low ring
		UNIMOD	"`"		; 02F4 modifier middle grave acent
		UNIMOD	34		; 02F5 modifier middle dbl grave accent
		UNIMOD	34		; 02F6 modifier middle dbl acute accent
		UNIMOD	"~"		; 02F7 modifier low tilde
		UNIMOD	":"		; 02F8 modifier raised colon
		UNIMOD	39		; 02F9 modifier begin high tone
		UNIMOD	"`"		; 02FA modifier eng high tone
		UNIMOD	","		; 02FB modifier begin low tone
		UNIMOD	"."		; 02FC modifier end low tone
		UNIMOD	"_"		; 02FD modifier shelf
		UNIMOD	"_"		; 02FE modifier open shelf
		UNIMOD	"<"		; 02FF modifier low left arrow

CHType0300:	UNIMOD	"`"		; 0300 combining grave accent
		UNIMOD	39		; 0301 combining acute accent
		UNIMOD	"^"		; 0302 combining circumflex accent
		UNIMOD	"~"		; 0303 combining tilde
		UNIMOD	"~"		; 0304 combining macron
		UNIMOD	"~"		; 0305 combining overline
		UNIMOD	"^"		; 0306 combining breve
		UNIMOD	39		; 0307 combining dot above
		UNIMOD	34		; 0308 combining diaeresis
		UNIMOD	39		; 0309 combining hook above
		UNIMOD	"^"		; 030A combining ring above
		UNIMOD	34		; 030B combining double acute accent
		UNIMOD	"^"		; 030C combining caron
		UNIMOD	39		; 030D combining vertical line
		UNIMOD	34		; 030E combining double vertical line
		UNIMOD	34		; 030F combining double grave accent
		UNIMOD	"^"		; 0310 combining candrabindu
		UNIMOD	"^"		; 0311 combining inverted breve
		UNIMOD	39		; 0312 combining turned comma above
		UNIMOD	"`"		; 0313 combining comma above
		UNIMOD	39		; 0314 combining reversed comma above
		UNIMOD	39		; 0315 combining comma above right
		UNIMOD	","		; 0316 combining grave accent below
		UNIMOD	","		; 0317 combining acute accent below
		UNIMOD	"+"		; 0318 combining left tack below
		UNIMOD	"+"		; 0319 combining right tack below
		UNIMOD	"~"		; 031A combining left angle above
		UNIMOD	"^"		; 031B combining horn
		UNIMOD	","		; 031C combining left half ring below
		UNIMOD	"."		; 031D combining up tack below
		UNIMOD	"."		; 031E combining down tack below
		UNIMOD	"+"		; 031F combining plus sign below
		UNIMOD	"-"		; 0320 combining minus sign below
		UNIMOD	","		; 0321 combining palatalized hook below
		UNIMOD	","		; 0322 combining retroflex hook below
		UNIMOD	"."		; 0323 combining dot below
		UNIMOD	"_"		; 0324 combining diaeresis below
		UNIMOD	"."		; 0325 combining ring below
		UNIMOD	","		; 0326 combining comma below
		UNIMOD	","		; 0327 combining cedilla
		UNIMOD	","		; 0328 combining ogonek
		UNIMOD	","		; 0329 combining vertical line below
		UNIMOD	"_"		; 032A combining bridge below
		UNIMOD	"_"		; 032B combining inv. double arch below
		UNIMOD	"_"		; 032C combining caron below
		UNIMOD	"_"		; 032D combining circumflex accent
		UNIMOD	"_"		; 032E combining breve below
		UNIMOD	"_"		; 032F combining inverted breve below
		UNIMOD	"_"		; 0330 combining tilde below
		UNIMOD	"_"		; 0331 combining macron below
		UNIMOD	"_"		; 0332 combining low line
		UNIMOD	"_"		; 0333 combining double low line
		UNIMOD	"~"		; 0334 combining tilde overlay
		UNIMOD	"-"		; 0335 combining short stroke overlay
		UNIMOD	"-"		; 0336 combining long stroke overlay
		UNIMOD	"/"		; 0337 combining short solidus overlay
		UNIMOD	"-"		; 0338 combining long solidus overlay
		UNIMOD	","		; 0339 combining right half ring below
		UNIMOD	"_"		; 033A combining inverted bridge below
		UNIMOD	"."		; 033B combining square below
		UNIMOD	"_"		; 033C combining seagull below
		UNIMOD	"x"		; 033D combining X above
		UNIMOD	"|"		; 033E combining vertical tilde
		UNIMOD	"~"		; 033F combining double overline
		UNIMOD	"`"		; 0340 combining grave tone mark
		UNIMOD	39		; 0341 combining acute tone mark
		UNIMOD	"~"		; 0342 combining greek perispomeni
		UNIMOD	39		; 0343 combining greek koronis
		UNIMOD	"^"		; 0344 combining greek dialytika tonos
		UNIMOD	","		; 0345 combining greek ypogegrammeni
		UNIMOD	"~"		; 0346 combining bridge above
		UNIMOD	"="		; 0347 combining equals sign below
		UNIMOD	"|"		; 0348 combining dbl. vert. line below
		UNIMOD	","		; 0349 combining left angle below
		UNIMOD	"~"		; 034A combining not tilde above
		UNIMOD	"~"		; 034B combining homothetic above
		UNIMOD	"~"		; 034C combining almost equal to above
		UNIMOD	"-"		; 034D combining left right arrow below
		UNIMOD	"^"		; 034E combining upwards arrow below
		UNICTR	1		; 034F combining grapheme joiner
		UNIMOD	">"		; 0350 combining right arrowhead above
		UNIMOD	39		; 0351 combining left half ring above
		UNIMOD	"^"		; 0352 combining fermata
		UNIMOD	"x"		; 0353 combining x below
		UNIMOD	"<"		; 0354 combining left arrowhead below
		UNIMOD	">"		; 0355 combining right arrowhead below
		UNIMOD	">"		; 0356 combining right+up arrow below
		UNIMOD	39		; 0357 combining right half ring above
		UNIMOD	39		; 0358 combining dot above right
		UNIMOD	"*"		; 0359 combining asterisk below
		UNIMOD	"_"		; 035A combining double ring below
		UNIMOD	"~"		; 035B combining zigzag above
		UNIMOD	"_"		; 035C combining double breve below
		UNIMOD	"^"		; 035D combining double breve
		UNIMOD	"~"		; 035E combining double macron
		UNIMOD	"_"		; 035F combining double macron below
		UNIMOD	"~"		; 0360 combining double tilde
		UNIMOD	"^"		; 0361 combining double inverted breve
		UNIMOD	"_"		; 0362 combining dbl rightwards arrow
		UNIMOD	"a"		; 0363 combining a
		UNIMOD	"e"		; 0364 combining e
		UNIMOD	"i"		; 0365 combining i
		UNIMOD	"o"		; 0366 combining o
		UNIMOD	"u"		; 0367 combining u
		UNIMOD	"c"		; 0368 combining c
		UNIMOD	"d"		; 0369 combining d
		UNIMOD	"h"		; 036A combining h
		UNIMOD	"m"		; 036B combining m
		UNIMOD	"r"		; 036C combining r
		UNIMOD	"t"		; 036D combining t
		UNIMOD	"v"		; 036E combining v
		UNIMOD	"x"		; 036F combining x
		UNIINV	4		; 0370..0373
		UNIMRK	39		; 0374 greek numeral sign
		UNIMRK	","		; 0375 greek lower numeral sign
		UNIINV	4		; 0376..0379
		UNIMRK	","		; 037A greek ypogegrammeni
		UNIMRK	UNINOASC	; 037B greek small rev. lunate sigma
		UNIMRK	"c"		; 037C greek small dotted lunate sigma
		UNIMRK	UNINOASC	; 037D greek small rev. dotted lunate
		UNIMRK	";"		; 037E greek question mark
		UNIINV	5		; 037F..0383
		UNIMRK	39		; 0384 greek tonos
		UNIMRK	"^"		; 0385 greek dialytika tonos
		UNICAP	"A",3ach	; 0386 greek capital alpha tonos
		UNIMRK	"."		; 0387 greek ano teleia
		UNICAP	"E",3adh	; 0388 greek capital epsilon tonos
		UNICAP	"H",3aeh	; 0389 greek capital eta tonos
		UNICAP	"I",3afh	; 038A greek capital iota tonos
		UNIINV	1		; 038B
		UNICAP	"O",3cch	; 038C greek capital omicron tonos
		UNIINV	1		; 038D
		UNICAP	"Y",3cdh	; 038E greek capital upsilon tonos
		UNICAP	UNINOASC,3ceh	; 038F greek capital omega tonos
		UNISMA	"i",0		; 0390 greek small iota dialytika tonos
		UNICAP	"A",3b1h	; 0391 greek capital alpha
		UNICAP	"B",3b2h	; 0392 greek capital beta
		UNICAP	UNINOASC,3b3h	; 0393 greek capital gamma
		UNICAP	UNINOASC,3b4h	; 0394 greek capital delta
		UNICAP	"E",3b5h	; 0395 greek capital epsilon
		UNICAP	"Z",3b6h	; 0396 greek capital zeta
		UNICAP	"H",3b7h	; 0397 greek capital eta
		UNICAP	UNINOASC,3b8h	; 0398 greek capital theta
		UNICAP	"I",3b9h	; 0399 greek capital iota
		UNICAP	"K",3bah	; 039A greek capital kappa
		UNICAP	UNINOASC,3bbh	; 039B greek capital lambda
		UNICAP	"M",3bch	; 039C greek capital mu
		UNICAP	"N",3bdh	; 039D greek capital nu
		UNICAP	UNINOASC,3beh	; 039E greek capital xi
		UNICAP	"O",3bfh	; 039F greek capital omicron
		UNICAP	UNINOASC,3c0h	; 03A0 greek capital pi
		UNICAP	"P",3c1h	; 03A1 greek capital rho
		UNIINV	1		; 03A2
		UNICAP	UNINOASC,3c3h	; 03A3 greek capital sigma
		UNICAP	"T",3c4h	; 03A4 greek capital tau
		UNICAP	"Y",3c5h	; 03A5 greek capital upsilon
		UNICAP	UNINOASC,3c6h	; 03A6 greek capital phi
		UNICAP	"X",3c7h	; 03A7 greek capital chi
		UNICAP	UNINOASC,3c8h	; 03A8 greek capital psi
		UNICAP	UNINOASC,3c9h	; 03A9 greek capital omega
		UNICAP	"I",3cah	; 03AA greek capital iota dialytika
		UNICAP	"Y",3cbh	; 03AB greek capital upsilon dialytika
		UNISMA	"a",386h	; 03AC greek small alpha tonos
		UNISMA	UNINOASC,388h	; 03AD greek small epsilon tonos
		UNISMA	"n",389h	; 03AE greek small eta tonos
		UNISMA	"i",38ah	; 03AF greek small iota tonos
		UNISMA	"u",0		; 03B0 greek small upsilon dial. tonos
		UNISMA	"a",391h	; 03B1 greek small alpha
		UNISMA	"B",392h	; 03B2 greek small beta
		UNISMA	"v",393h	; 03B3 greek small gamma
		UNISMA	"d",394h	; 03B4 greek small delta
		UNISMA	UNINOASC,395h	; 03B5 greek small epsilon
		UNISMA	UNINOASC,396h	; 03B6 greek small zeta
		UNISMA	"n",397h	; 03B7 greek small eta
		UNISMA	"0",398h	; 03B8 greek small theta
		UNISMA	"i",399h	; 03B9 greek small iota
		UNISMA	"k",39ah	; 03BA greek small kappa
		UNISMA	UNINOASC,39bh	; 03BB greek small lambda
		UNISMA	"u",39ch	; 03BC greek small mu
		UNISMA	"v",39dh	; 03BD greek small nu
		UNISMA	UNINOASC,39eh	; 03BE greek small xi
		UNISMA	"o",39fh	; 03BF greek small omikron
		UNISMA	UNINOASC,3a0h	; 03C0 greek small pi
		UNISMA	"p",3a1h	; 03C1 greek small rho
		UNISMA	UNINOASC,0	; 03C2 greek small stigma
		UNISMA	"o",3a3h	; 03C3 greek small sigma
		UNISMA	"t",3a4h	; 03C4 greek small tau
		UNISMA	"u",3a5h	; 03C5 greek small upsilon
		UNISMA	UNINOASC,3a6h	; 03C6 greek small phi
		UNISMA	"x",3a7h	; 03C7 greek small chi
		UNISMA	UNINOASC,3a8h	; 03C8 greek small psi
		UNISMA	UNINOASC,3a9h	; 03C9 greek small omega
		UNISMA	"i",3aah	; 03CA greek small iota dialytika
		UNISMA	"u",3abh	; 03CB greek small upsilon dialytika
		UNISMA	"o",38ch	; 03CC greek small omicron tonos
		UNISMA	"u",38eh	; 03CD greek small upsilon tonos
		UNISMA	UNINOASC,38fh	; 03CE greek small omega tonos
		UNIINV	1		; 03CF
		UNISIG	"b"		; 03D0 greek beta symbol
		UNISIG	UNINOASC	; 03D1 greek theta symbol
		UNISIG	"Y"		; 03D2 greek upsilon hook symbol
		UNISIG	"Y"		; 03D3 greek upsilon acute symbol
		UNISIG	"Y"		; 03D4 greek upsilon diaeresis hook
		UNISIG	UNINOASC	; 03D5 greek phi symbol
		UNISIG	UNINOASC	; 03D6 greek pi symbol
		UNISIG	"x"		; 03D7 greek kai symbol
		UNICAP	"O",3d9h	; 03D8 greek capital archaic koppa
		UNISMA	"o",3d8h	; 03D9 greek small archaic koppa
		UNICAP	"C",3dbh	; 03DA greek capital archaic stigma
		UNISMA	"c",3dah	; 03DB greek small archaic stigma
		UNICAP	"F",3ddh	; 03DC greek capital archaic digamma
		UNISMA	"f",3dch	; 03DD greek small archaic digamma
		UNICAP	UNINOASC,3dfh	; 03DE greek capital koppa
		UNISMA	UNINOASC,3deh	; 03DF greek small koppa
		UNICAP	UNINOASC,3e1h	; 03E0 greek capital koppa
		UNISMA	UNINOASC,3e0h	; 03E1 greek small koppa
		UNICAPSMA 3e2h,7	; 03E2..03EF coptic
		UNISIG	"x"		; 03F0 greek kappa symbol
		UNISIG	UNINOASC	; 03F1 greek rho symbol
		UNISIG	"c"		; 03F2 greek lunate symbol
		UNISMA	"j",0		; 03F3 greek yot
		UNISIG	UNINOASC	; 03F4 greek capital theta symbol
		UNISIG	UNINOASC	; 03F5 greek lunate epsilon symbol
		UNISIG	UNINOASC	; 03F6 greek reversed lunate symbol
		UNICAP	"I",3f8h	; 03F7 greek capital sho
		UNISMA	"l",3f7h	; 03F8 greek small sho
		UNISIG	"C"		; 03F9 greek capital lunate sigma
		UNICAP	"M",3fbh	; 03FA greek capital san
		UNISMA	"m",3fah	; 03FB greek small san
		UNISIG	UNINOASC	; 03FC greek rho stroke symbol
		UNISIG	UNINOASC	; 03FD greek capital rev. lunate sigma
		UNISIG	"C"		; 03FE greek cap. dotted lunate sigma
		UNISIG	UNINOASC	; 03FF greek cap. rev. dot lunate sigma

CHType0400:	UNICAP	"E",450h	; 0400 cyrillic capital IE grave
		UNICAP	"E",451h	; 0401 cyrillic capital IO
		UNICAP	UNINOASC,452h	; 0402 cyrillic capital DJE
		UNICAP	UNINOASC,453h	; 0403 cyrillic capital GJE
		UNICAP	"E",454h	; 0404 cyrillic capital ukrainian IE
		UNICAP	"S",455h	; 0405 cyrillic capital DZE
		UNICAP	"I",456h	; 0406 cyrillic capital byelorussian I
		UNICAP	"I",457h	; 0407 cyrillic capital YI
		UNICAP	"J",458h	; 0408 cyrillic capital JE
		UNICAP	UNINOASC,459h	; 0409 cyrillic capital LJE
		UNICAP	UNINOASC,45ah	; 040A cyrillic capital NJE
		UNICAP	UNINOASC,45bh	; 040B cyrillic capital TSHE
		UNICAP	"K",45ch	; 040C cyrillic capital KJE
		UNICAP	UNINOASC,45dh	; 040D cyrillic capital I grave
		UNICAP	"Y",45eh	; 040E cyrillic capital short U
		UNICAP	UNINOASC,45fh	; 040F cyrillic capital DZHE
		UNICAP	"A",430h	; 0410 cyrillic capital A
		UNICAP	UNINOASC,431h	; 0411 cyrillic capital BE
		UNICAP	"B",432h	; 0412 cyrillic capital VE
		UNICAP	UNINOASC,433h	; 0413 cyrillic capital GHE
		UNICAP	UNINOASC,434h	; 0414 cyrillic capital DE
		UNICAP	"E",435h	; 0415 cyrillic capital IE
		UNICAP	UNINOASC,436h	; 0416 cyrillic capital ZHE
		UNICAP	UNINOASC,437h	; 0417 cyrillic capital ZE
		UNICAP	UNINOASC,438h	; 0418 cyrillic capital I
		UNICAP	UNINOASC,439h	; 0419 cyrillic capital short I
		UNICAP	"K",43ah	; 041A cyrillic capital KA
		UNICAP	UNINOASC,43bh	; 041B cyrillic capital EL
		UNICAP	"M",43ch	; 041C cyrillic capital EM
		UNICAP	"H",43dh	; 041D cyrillic capital EN
		UNICAP	"O",43eh	; 041E cyrillic capital O
		UNICAP	UNINOASC,43fh	; 041F cyrillic capital PE
		UNICAP	"P",440h	; 0420 cyrillic capital ER
		UNICAP	"C",441h	; 0421 cyrillic capital ES
		UNICAP	"T",442h	; 0422 cyrillic capital TE
		UNICAP	"Y",443h	; 0423 cyrillic capital U
		UNICAP	UNINOASC,444h	; 0424 cyrillic capital EF
		UNICAP	"X",445h	; 0425 cyrillic capital HA
		UNICAP	UNINOASC,446h	; 0426 cyrillic capital TSE
		UNICAP	UNINOASC,447h	; 0427 cyrillic capital CHE
		UNICAP	UNINOASC,448h	; 0428 cyrillic capital SHA
		UNICAP	UNINOASC,449h	; 0429 cyrillic capital SHCHA
		UNICAP	UNINOASC,44ah	; 042A cyrillic capital hard sign
		UNICAP	UNINOASC,44bh	; 042B cyrillic capital YERU
		UNICAP	UNINOASC,44ch	; 042C cyrillic capital soft sign
		UNICAP	UNINOASC,44dh	; 042D cyrillic capital E
		UNICAP	UNINOASC,44eh	; 042E cyrillic capital YU
		UNICAP	UNINOASC,44fh	; 042F cyrillic capital YA
		UNISMA	"a",410h	; 0430 cyrillic small A
		UNISMA	UNINOASC,411h	; 0431 cyrillic small BE
		UNISMA	UNINOASC,412h	; 0432 cyrillic small VE
		UNISMA	"r",413h	; 0433 cyrillic small GHE
		UNISMA	UNINOASC,414h	; 0434 cyrillic small DE
		UNISMA	"e",415h	; 0435 cyrillic small IE
		UNISMA	UNINOASC,416h	; 0436 cyrillic small ZHE
		UNISMA	UNINOASC,417h	; 0437 cyrillic small ZE
		UNISMA	UNINOASC,418h	; 0438 cyrillic small I
		UNISMA	UNINOASC,419h	; 0439 cyrillic small short I
		UNISMA	UNINOASC,41ah	; 043A cyrillic small KA
		UNISMA	UNINOASC,41bh	; 043B cyrillic small EL
		UNISMA	UNINOASC,41ch	; 043C cyrillic small EM
		UNISMA	UNINOASC,41dh	; 043D cyrillic small EN
		UNISMA	"o",41eh	; 043E cyrillic small O
		UNISMA	UNINOASC,41fh	; 043F cyrillic small PE
		UNISMA	"p",420h	; 0440 cyrillic small ER
		UNISMA	"c",421h	; 0441 cyrillic small ES
		UNISMA	UNINOASC,422h	; 0442 cyrillic small TE
		UNISMA	"y",423h	; 0443 cyrillic small U
		UNISMA	UNINOASC,424h	; 0444 cyrillic small EF
		UNISMA	"x",425h	; 0445 cyrillic small HA
		UNISMA	UNINOASC,426h	; 0446 cyrillic small TSE
		UNISMA	UNINOASC,427h	; 0447 cyrillic small CHE
		UNISMA	UNINOASC,428h	; 0448 cyrillic small SHA
		UNISMA	UNINOASC,429h	; 0449 cyrillic small SHCHA
		UNISMA	UNINOASC,42ah	; 044A cyrillic small hard sign
		UNISMA	UNINOASC,42bh	; 044B cyrillic small YERU
		UNISMA	UNINOASC,42ch	; 044C cyrillic small soft sign
		UNISMA	UNINOASC,42dh	; 044D cyrillic small E
		UNISMA	UNINOASC,42eh	; 044E cyrillic small YU
		UNISMA	UNINOASC,42fh	; 044F cyrillic small YA
		UNISMA	"e",400h	; 0450 cyrillic small IE grave
		UNISMA	"e",401h	; 0451 cyrillic small IO
		UNISMA	UNINOASC,402h	; 0452 cyrillic small DJE
		UNISMA	"r",403h	; 0453 cyrillic small GJE
		UNISMA	UNINOASC,404h	; 0454 cyrillic small ukrainian IE
		UNISMA	"s",405h	; 0455 cyrillic small DZE
		UNISMA	"i",406h	; 0456 cyrillic small byelorussian I
		UNISMA	"i",407h	; 0457 cyrillic small YI
		UNISMA	"j",408h	; 0458 cyrillic small JE
		UNISMA	UNINOASC,409h	; 0459 cyrillic small LJE
		UNISMA	UNINOASC,40ah	; 045A cyrillic small NJE
		UNISMA	"h",40bh	; 045B cyrillic small TSHE
		UNISMA	UNINOASC,40ch	; 045C cyrillic small KJE
		UNISMA	UNINOASC,40dh	; 045D cyrillic small I grave
		UNISMA	"y",40eh	; 045E cyrillic small short U
		UNISMA	UNINOASC,40fh	; 045F cyrillic small DZHE
		UNICAPSMA 460h,17	; 0460..0481 historic letters
		UNISIG	UNINOASC	; 0482 cyrillic thousands
		UNIMOD	34		; 0483 combining cyrillic titlo
		UNIMOD	"^"		; 0484 combining cyril. palatalization
		UNIMOD	39		; 0485 combining cyril. dasia pneumata
		UNIMOD	"`"		; 0486 combining cyril. psili pneumata
		UNIINV	1		; 0487
		UNIMOD	UNINOASC	; 0488 combining cyr. hundred thousands
		UNIMOD	UNINOASC	; 0489 combining cyrillic millions
		UNICAP	UNINOASC,48bh	; 048A cyrillic capital short I tail
		UNISMA	UNINOASC,48ah	; 048B cyrillic small short I tail
		UNICAP	UNINOASC,48dh	; 048C cyrillic capital semisoft sign
		UNISMA	UNINOASC,48ch	; 048D cyrillic small semisoft sign
		UNICAP	"P",48fh	; 048E cyrillic capital ER tick
		UNISMA	"p",48eh	; 048F cyrillic small ER tick
		UNICAP	UNINOASC,491h	; 0490 cyrillic capital GHE upturn
		UNISMA	UNINOASC,490h	; 0491 cyrillic small GHE upturn
		UNICAP	UNINOASC,493h	; 0492 cyrillic capital GHE stroke
		UNISMA	UNINOASC,492h	; 0493 cyrillic small GHE stroke
		UNICAP	UNINOASC,495h	; 0494 cyrillic capital GHE middle hook
		UNISMA	UNINOASC,494h	; 0495 cyrillic small GHE middle hook
		UNICAP	UNINOASC,497h	; 0496 cyrillic capital ZHE descender
		UNISMA	UNINOASC,496h	; 0497 cyrillic small ZHE descender
		UNICAP	UNINOASC,499h	; 0498 cyrillic capital ZE descender
		UNISMA	UNINOASC,498h	; 0499 cyrillic small ZE descender
		UNICAP	"K",49bh	; 049A cyrillic capital KA descender
		UNISMA	UNINOASC,49ah	; 049B cyrillic small KA descender
		UNICAP	"K",49dh	; 049C cyrillic capital KA vert. stroke
		UNISMA	UNINOASC,49ch	; 049D cyrillic small KA vert. stroke
		UNICAP	"K",49fh	; 049E cyrillic capital KA stroke
		UNISMA	UNINOASC,49eh	; 049F cyrillic small KA stroke
		UNICAP	"K",4a1h	; 04A0 cyrillic capital bashkir KA
		UNISMA	UNINOASC,4a0h	; 04A1 cyrillic small bashkir KA
		UNICAP	"H",4a3h	; 04A2 cyrillic capital EN descender
		UNISMA	UNINOASC,4a2h	; 04A3 cyrillic small EN descender
		UNICAP	"H",4a5h	; 04A4 cyrillic capital EN GHE
		UNISMA	UNINOASC,4a4h	; 04A5 cyrillic small EN GHE
		UNICAP	UNINOASC,4a7h	; 04A6 cyrillic capital PE hook
		UNISMA	UNINOASC,4a6h	; 04A7 cyrillic small PE hook
		UNICAP	UNINOASC,4a9h	; 04A8 cyrillic capital abkhasian HA
		UNISMA	UNINOASC,4a8h	; 04A9 cyrillic small abkhasian HA
		UNICAP	"C",4abh	; 04AA cyrillic capital ES descender
		UNISMA	"c",4aah	; 04AB cyrillic small ES descender
		UNICAP	"T",4adh	; 04AC cyrillic capital TE descender
		UNISMA	UNINOASC,4ach	; 04AD cyrillic small TE descender
		UNICAP	"Y",4afh	; 04AE cyrillic capital straight U
		UNISMA	"y",4aeh	; 04AF cyrillic small straight U
		UNICAP	"Y",4b1h	; 04B0 cyril. capital straight U stroke
		UNISMA	"y",4b0h	; 04B1 cyril. small straight U stroke
		UNICAP	"X",4b3h	; 04B2 cyrillic capital HA descender
		UNISMA	"x",4b2h	; 04B3 cyrillic small HA descender
		UNICAP	UNINOASC,4b5h	; 04B4 cyrillic capital ligature TE TSE
		UNISMA	UNINOASC,4b4h	; 04B5 cyrillic small ligature TE TSE
		UNICAP	UNINOASC,4b7h	; 04B6 cyrillic capital CHE descender
		UNISMA	UNINOASC,4b6h	; 04B7 cyrillic small CHE descender
		UNICAP	UNINOASC,4b9h	; 04B8 cyril. capital CHE vert. stroke
		UNISMA	UNINOASC,4b8h	; 04B9 cyril. small CHE vert. stroke
		UNICAP	UNINOASC,4bbh	; 04BA cyrillic capital SHHA
		UNISMA	UNINOASC,4bah	; 04BB cyrillic small SHHA
		UNICAP	UNINOASC,4bdh	; 04BC cyrillic capital abkhasian CHE
		UNISMA	"e",4bch	; 04BD cyrillic small abkhasian CHE
		UNICAP	UNINOASC,4bfh	; 04BE cyr. cap. abkhas. CHE descender
		UNISMA	"e",4beh	; 04BF cyr. small abkhas. CHE descender
		UNICAP	"I",4cfh	; 04C0 cyrillic capital palochka
		UNICAP	UNINOASC,4c2h	; 04C1 cyrillic capital ZHE breve
		UNISMA	UNINOASC,4c1h	; 04C2 cyrillic small ZHE breve
		UNICAP	"K",4c4h	; 04C3 cyrillic capital KA hook
		UNISMA	UNINOASC,4c3h	; 04C4 cyrillic small KA hook
		UNICAP	UNINOASC,4c6h	; 04C5 cyrillic capital EL tail
		UNISMA	UNINOASC,4c5h	; 04C6 cyrillic small EL tail
		UNICAP	"H",4c8h	; 04C7 cyrillic capital EN hook
		UNISMA	UNINOASC,4c7h	; 04C8 cyrillic small EN hook
		UNICAP	"H",4cah	; 04C9 cyrillic capital EN tail
		UNISMA	UNINOASC,4c9h	; 04CA cyrillic small EN tail
		UNICAP	UNINOASC,4cch	; 04CB cyrillic capital khakassian CHE
		UNISMA	UNINOASC,4cbh	; 04CC cyrillic small khakassian CHE
		UNICAP	"M",4ceh	; 04CD cyrillic capital EM tail
		UNISMA	UNINOASC,4cdh	; 04CE cyrillic small EM tail
		UNISMA	"l",4c0h	; 04CF cyrillic small palochka
		UNICAP	"A",4d1h	; 04D0 cyrillic capital A breve
		UNISMA	"a",4d0h	; 04D1 cyrillic small A breve
		UNICAP	"A",4d3h	; 04D2 cyrillic capital A diaeresis
		UNISMA	"a",4d2h	; 04D3 cyrillic small A diaeresis
		UNICAP	"A",4d5h	; 04D4 cyrillic capital A IE
		UNISMA	"a",4d4h	; 04D5 cyrillic small A IE
		UNICAP	"E",4d7h	; 04D6 cyrillic capital IE breve
		UNISMA	"e",4d6h	; 04D7 cyrillic small IE breve
		UNICAP	UNINOASC,4d9h	; 04D8 cyrillic capital SCHWA
		UNISMA	"e",4d8h	; 04D9 cyrillic small SCHWA
		UNICAP	UNINOASC,4dbh	; 04DA cyrillic capital SCHWA diaeresis
		UNISMA	"e",4dah	; 04DB cyrillic small SCHWA diaeresis
		UNICAP	UNINOASC,4ddh	; 04DC cyrillic capital ZHE diaeresis
		UNISMA	UNINOASC,4dch	; 04DD cyrillic small ZHE diaeresis
		UNICAP	UNINOASC,4dfh	; 04DE cyrillic capital ZE diaeresis
		UNISMA	UNINOASC,4deh	; 04DF cyrillic small ZE diaeresis
		UNICAP	UNINOASC,4e1h	; 04E0 cyrillic capital abkhasian DZE
		UNISMA	UNINOASC,4e0h	; 04E1 cyrillic small abkhasian DZE
		UNICAP	UNINOASC,4e3h	; 04E2 cyrillic capital I macron
		UNISMA	UNINOASC,4e2h	; 04E3 cyrillic small I macron
		UNICAP	UNINOASC,4e5h	; 04E4 cyrillic capital I diaeresis
		UNISMA	UNINOASC,4e4h	; 04E5 cyrillic small I diaeresis
		UNICAP	"O",4e7h	; 04E6 cyrillic capital O diaeresis
		UNISMA	"o",4e6h	; 04E7 cyrillic small O diaeresis
		UNICAP	"O",4e9h	; 04E8 cyrillic capital barred O
		UNISMA	"o",4e8h	; 04E9 cyrillic small barred O
		UNICAP	"O",4ebh	; 04EA cyr. capital barred O diaeresis
		UNISMA	"o",4eah	; 04EB cyr. small barred O diaeresis
		UNICAP	UNINOASC,4edh	; 04EC cyrillic capital E diaeresis
		UNISMA	UNINOASC,4ech	; 04ED cyrillic small E diaeresis
		UNICAP	"Y",4efh	; 04EE cyrillic capital U macron
		UNISMA	"y",4eeh	; 04EF cyrillic small U macron
		UNICAP	"Y",4f1h	; 04F0 cyrillic capital U diaeresis
		UNISMA	"y",4f0h	; 04F1 cyrillic small U diaeresis
		UNICAP	"Y",4f3h	; 04F2 cyrillic capital U double acute
		UNISMA	"y",4f2h	; 04F3 cyrillic small U double acute
		UNICAP	UNINOASC,4f5h	; 04F4 cyrillic capital CHE diaeresis
		UNISMA	UNINOASC,4f4h	; 04F5 cyrillic small CHE diaeresis
		UNICAP	UNINOASC,4f7h	; 04F6 cyrillic capital GHE descender
		UNISMA	UNINOASC,4f6h	; 04F7 cyrillic small GHE descender
		UNICAP	UNINOASC,4f9h	; 04F8 cyrillic capital YERU diaeresis
		UNISMA	UNINOASC,4f8h	; 04F9 cyrillic small YERU diaeresis
		UNICAP	UNINOASC,4fbh	; 04FA cyrillic capital GHE stroke hook
		UNISMA	UNINOASC,4fah	; 04FB cyrillic small GHE stroke hook
		UNICAP	"X",4fdh	; 04FC cyrillic capital HA hook
		UNISMA	"x",4fch	; 04FD cyrillic small HA hook
		UNICAP	"X",4ffh	; 04FE cyrillic capital HA stroke
		UNISMA	"x",4feh	; 04FF cyrillic small HA stroke

CHType0500:	UNICAP	UNINOASC,501h	; 0500 cyrillic capital komi DE
		UNISMA	"d",500h	; 0501 cyrillic small komi DE
		UNICAP	UNINOASC,503h	; 0502 cyrillic capital komi DJE
		UNISMA	"d",502h	; 0503 cyrillic small komi DJE
		UNICAP	UNINOASC,505h	; 0504 cyrillic capital komi ZJE
		UNISMA	UNINOASC,504h	; 0505 cyrillic small komi ZJE
		UNICAP	UNINOASC,507h	; 0506 cyrillic capital komi DZJE
		UNISMA	UNINOASC,506h	; 0507 cyrillic small komi DZJE
		UNICAP	UNINOASC,509h	; 0508 cyrillic capital komi LJE
		UNISMA	UNINOASC,508h	; 0509 cyrillic small komi LJE
		UNICAP	"H",50bh	; 050A cyrillic capital komi NJE
		UNISMA	UNINOASC,50ah	; 050B cyrillic small komi NJE
		UNICAP	"G",50dh	; 050C cyrillic capital komi SJE
		UNISMA	UNINOASC,50ch	; 050D cyrillic small komi SJE
		UNICAP	"T",50fh	; 050E cyrillic capital komi TJE
		UNISMA	UNINOASC,50eh	; 050F cyrillic small komi TJE
		UNICAPSMA 510h,2	; 0510..0513 cyrillic extensions
		UNIINV	29		; 0514..0530
		UNICAPINX 561h,38	; 0531..0556 armenian uppercase
		UNIINV	2		; 0557..0558
		UNIMOD	39		; 0559 armenian modifier left half ring
		UNIMRK	39		; 055A armenian apostrophe
		UNIMRK	39		; 055B armenian emphasis
		UNIMRK	"~"		; 055C armenian exclamation
		UNIMRK	39		; 055D armenian comma
		UNIMRK	"^"		; 055E armenian question
		UNIMRK	"-"		; 055F armenian abbrevation
		UNIINV	1		; 0560
		UNISMAINX 531h,38	; 0561..0586 armenian lowercase
		UNISMA	UNINOASC,0	; 0587 armenian small ligature ECH YIWN
		UNIINV	1		; 0588
		UNIMRK	":"		; 0589 armenian full stop
		UNIMRK	"_"		; 058A armenian hyphen
		UNIINV	6		; 058B..0590
		UNIMODNUM 45		; 0591..05BD hebrew modifiers
		UNIMRK	"~"		; 05BE hebrew punctuation maqaf
		UNIMOD	UNINOASC	; 05BF hebrew point rafe
		UNIMRK	"|"		; 05C0 hebrew punctuation paseq
		UNIMODNUM 2		; 05C1..05C2 hebrew point
		UNIMRK	":"		; 05C3 hebrew punctuatuin sof pasuq
		UNIMODNUM 2		; 05C4..05C5 hebrew mark
		UNIMRK	UNINOASC	; 05C6 hebrew punctuation nun hafukha
		UNIMOD	UNINOASC	; 05C7 hebrew point qamats qatan
		UNIINV	8		; 05C8..05CF
		UNICAPNUM 27		; 05D0..05EA hebrew letter
		UNIINV	5		; 05EB..05EF
		UNIMRKNUM 5		; 05F0..05F4 hebrew ligature 
		UNIINV	11		; 05F5..05FF

CHType0600:	UNISIG	UNINOASC	; 0600 arabic number sign
		UNISIG	UNINOASC	; 0601 arabic sign sanah
		UNIMRK	UNINOASC	; 0602 arabic footnote marker
		UNISIG	UNINOASC	; 0603 arabic sign safha
		UNIINV	7		; 0604..060A
		UNISIGNUM 5		; 060B..060F arabic sign
		UNIMODNUM 6		; 0610..0615 arabic modifier
		UNIINV	5		; 0616..061A
		UNIMRK	";"		; 061B arabic semicolon
		UNIINV	2		; 061C..061D
		UNIMRK	"."		; 061E arabic triple dot punctuation
		UNIMRK	"?"		; 061F arabic question
		UNIINV	1		; 0620
		UNICAPNUM 26		; 0621..063A arabic letter
		UNIINV	5		; 063B..063F
		UNIMRK	"-"		; 0640 arabic tatweel
		UNICAPNUM 10		; 0641..064A arabic letter
		UNIMODNUM 20		; 064B..065E arabid modifier
		UNIINV	1		; 065F
		UNIDIGCH "0",10		; 0660..0669 arabic-indic digit
		UNISIGNUM 3		; 066A..066C arabic sign
		UNIMRK	"*"		; 066D arabic five pointed star
		UNICAPNUM 2		; 066E..066F arabic archaic letter
		UNIMOD	39		; 0670 arabic superscript alef
		UNICAPNUM 99		; 0671..06D3 arabic letter
		UNIMRK	"-"		; 06D4 arabic full stop
		UNICAPNUM 1		; 06D5 arabic letter AE
		UNIMODNUM 24		; 06D6..06ED arabic modifier
		UNICAPNUM 2		; 06EE..06EF arabic letter
		UNIDIGCH "0",10		; 06F0..06F9 extended arabic digit
		UNICAPNUM 3		; 06FA..06FC arabic letter
		UNISIGNUM 2		; 06FD..06FE arabic sign
		UNICAPNUM 1		; 06FF arabic letter HEH inv. V

CHType0E00:	UNIINV	1		; 0E00
		UNICAPNUM 46		; 0E01..0E2E thai character
		UNIMRK	UNINOASC	; 0E2F thai
		UNIMODNUM 11		; 0E30..0E3A thai symbol
		UNIINV	4		; 0E3B..0E3E
		UNISIG	UNINOASC	; 0E3F thai
		UNICAPNUM 6		; 0E40..0E45 thai
		UNIMODNUM 10		; 0E46..0E4F thai
		UNIDIGCH "0",10		; 0E50..0E59 thai digit
		UNISIGNUM 2		; 0E5A..0E5B
		UNIINV	37		; 0E5C..0E80
		UNICAPNUM 2		; 0E81..0E82 lao letter
		UNIINV	1		; 0E83
		UNICAPNUM 1		; 0E84 lao letter
		UNIINV	2		; 0E85..0E86
		UNICAPNUM 2		; 0E87..0E88 lao letter
		UNIINV	1		; 0E89
		UNICAPNUM 1		; 0E8A lao letter
		UNIINV	2		; 0E8B..0E8C
		UNICAPNUM 1		; 0E8D lao letter
		UNIINV	6		; 0E8E..0E93
		UNICAPNUM 4		; 0E94..0E97 lao letter
		UNIINV	1		; 0E98
		UNICAPNUM 7		; 0E99..0E9F lao letter
		UNIINV	1		; 0EA0
		UNICAPNUM 3		; 0EA1..0EA3 lao letter
		UNIINV	1		; 0EA4
		UNICAPNUM 1		; 0EA5 lao letter
		UNIINV	1		; 0EA6
		UNICAPNUM 1		; 0EA7 lao letter
		UNIINV	2		; 0EA8..0EA9
		UNICAPNUM 2		; 0EAA..0EAB lao letter
		UNIINV	1		; 0EAC
		UNICAPNUM 2		; 0EAD..0EAE lao letter
		UNISIG	UNINOASC	; 0EAF lao ellipsis
		UNIMODNUM 10		; 0EB0..0EB9 lao vowel
		UNIINV	1		; 0EBA
		UNIMODNUM 3		; 0EBB..0EBD lao vowel
		UNIINV	2		; 0EBE..0EBF
		UNISIGNUM 5		; 0EC0..0EC4 lao vowel
		UNIINV	1		; 0EC5
		UNISIG	UNINOASC	; 0EC6 lao kola
		UNIINV	1		; 0EC7
		UNIMODNUM 6		; 0EC8..0ECD lao sign
		UNIINV	2		; 0ECE..0ECF
		UNIDIGCH "0",10		; 0ED0..0ED9 lao digit
		UNIINV	2		; 0EDA..0EDB
		UNISIGNUM 2		; 0EDC..0EDD lao ho
		UNIINV	34		; 0EDE..0EFF

CHType1D00:	UNISMA	"a",0		; 1D00 small capital A
		UNISMA	"a",0		; 1D01 small capital AE
		UNISMA	"a",0		; 1D02 turned ae
		UNISMA	"b",0		; 1D03 small capital barred B
		UNISMA	"c",0		; 1D04 small capital C
		UNISMA	"d",0		; 1D05 small capital D
		UNISMA	"d",0		; 1D06 small capital ETH
		UNISMA	"e",0		; 1D07 small capital E
		UNISMA	"e",0		; 1D08 small capital turned open E
		UNISMA	"i",0		; 1D09 turned i
		UNISMA	"j",0		; 1D0A small capital J
		UNISMA	"k",0		; 1D0B small capital K
		UNISMA	"l",0		; 1D0C small capital L stroke
		UNISMA	"m",0		; 1D0D small capital M
		UNISMA	"n",0		; 1D0E small capital reversed N
		UNISMA	"o",0		; 1D0F small capital O
		UNISMA	"c",0		; 1D10 small capital open O
		UNISMA	"o",0		; 1D11 sideways o
		UNISMA	"c",0		; 1D12 sideways open o
		UNISMA	"o",0		; 1D13 sideways o stroke
		UNISMA	"o",0		; 1D14 sideways turned oe
		UNISMA	"o",0		; 1D15 small capital OU
		UNICAP	"o",0		; 1D16 top half o
		UNICAP	" ",0		; 1D17 bottom half o
		UNISMA	"p",0		; 1D18 small capital P
		UNISMA	"r",0		; 1D19 small capital reversed R
		UNISMA	"r",0		; 1D1A small capital turned R
		UNISMA	"t",0		; 1D1B small capital T
		UNISMA	"u",0		; 1D1C small capital U
		UNISMA	"u",0		; 1D1D small capital sideways U
		UNISMA	"u",0		; 1D1E small cap. sideways diaeresis U
		UNISMA	"m",0		; 1D1F sideways turned m
		UNISMA	"v",0		; 1D20 small capital V
		UNISMA	"w",0		; 1D21 small capital W
		UNISMA	"z",0		; 1D22 small capital Z
		UNISMA	UNINOASC,0	; 1D23 small capital EZH
		UNICAP	UNINOASC,0	; 1D24 voiced laryngeal spirant
		UNICAP	UNINOASC,0	; 1D25 letter ain
		UNISMA	UNINOASC,0	; 1D26 greek small capital gamma
		UNISMA	UNINOASC,0	; 1D27 greek small capital lambda
		UNISMA	UNINOASC,0	; 1D28 greek small capital pi
		UNISMA	"P",0		; 1D29 greek small capital rho
		UNISMA	UNINOASC,0	; 1D2A greek small capital psi
		UNISMA	UNINOASC,0	; 1D2B greek small capital el
		UNIMOD	"A"		; 1D2C modifier A
		UNIMOD	"A"		; 1D2D modifier AE
		UNIMOD	"B"		; 1D2E modifier B
		UNIMOD	"B"		; 1D2F modifier barred B
		UNIMOD	"D"		; 1D30 modifier D
		UNIMOD	"E"		; 1D31 modifier E
		UNIMOD	"E"		; 1D32 modifier reversed E
		UNIMOD	"G"		; 1D33 modifier G
		UNIMOD	"H"		; 1D34 modifier H
		UNIMOD	"I"		; 1D35 modifier I
		UNIMOD	"J"		; 1D36 modifier J
		UNIMOD	"K"		; 1D37 modifier K
		UNIMOD	"L"		; 1D38 modifier L
		UNIMOD	"M"		; 1D39 modifier M
		UNIMOD	"N"		; 1D3A modifier N
		UNIMOD	"N"		; 1D3B modifier reversed N
		UNIMOD	"O"		; 1D3C modifier O
		UNIMOD	"O"		; 1D3D modifier O
		UNIMOD	"P"		; 1D3E modifier P
		UNIMOD	"R"		; 1D3F modifier R
		UNIMOD	"T"		; 1D40 modifier T
		UNIMOD	"U"		; 1D41 modifier U
		UNIMOD	"W"		; 1D42 modifier W
		UNIMOD	"a"		; 1D43 modifier a
		UNIMOD	"a"		; 1D44 modifier turned a
		UNIMOD	"a"		; 1D45 modifier small alpha
		UNIMOD	"a"		; 1D46 modifier turned ae
		UNIMOD	"b"		; 1D47 modifier b
		UNIMOD	"d"		; 1D48 modifier d
		UNIMOD	"e"		; 1D49 modifier e
		UNIMOD	"e"		; 1D4A modifier small schwa
		UNIMOD	"e"		; 1D4B modifier open e
		UNIMOD	"e"		; 1D4C modifier turned open e
		UNIMOD	"g"		; 1D4D modifier g
		UNIMOD	"i"		; 1D4E modifier turned i
		UNIMOD	"k"		; 1D4F modifier k
		UNIMOD	"m"		; 1D50 modifier m
		UNIMOD	"n"		; 1D51 modifier small eng
		UNIMOD	"o"		; 1D52 modifier o
		UNIMOD	"c"		; 1D53 modifier open o
		UNIMOD	"o"		; 1D54 modifier top half o
		UNIMOD	" "		; 1D55 modifier bottom half o
		UNIMOD	"p"		; 1D56 modifier p
		UNIMOD	"t"		; 1D57 modifier t
		UNIMOD	"u"		; 1D58 modifier u
		UNIMOD	"u"		; 1D59 modifier sideways u
		UNIMOD	"m"		; 1D5A modifier turned m
		UNIMOD	"v"		; 1D5B modifier v
		UNIMOD	"a"		; 1D5C modifier small ain
		UNIMOD	UNINOASC	; 1D5D modifier small beta
		UNIMOD	UNINOASC	; 1D5E modifier small gamma
		UNIMOD	UNINOASC	; 1D5F modifier small delta
		UNIMOD	UNINOASC	; 1D60 modifier small phi
		UNIMOD	UNINOASC	; 1D61 modifier small chi
		UNISMA	"i",0		; 1D62 subscript i
		UNISMA	"r",0		; 1D63 subscript r
		UNISMA	"u",0		; 1D64 subscript u
		UNISMA	"v",0		; 1D65 subscript v
		UNISMA	UNINOASC,0	; 1D66 subscript small beta
		UNISMA	UNINOASC,0	; 1D67 subscript small gamma
		UNISMA	UNINOASC,0	; 1D68 subscript small rho
		UNISMA	UNINOASC,0	; 1D69 subscript small phi
		UNISMA	UNINOASC,0	; 1D6A subscript small chi
		UNISMA	"u",0		; 1D6B small ue
		UNISMA	"b",0		; 1D6C b tilde
		UNISMA	"d",0		; 1D6D d tilde
		UNISMA	"f",0		; 1D6E f tilde
		UNISMA	"m",0		; 1D6F m tilde
		UNISMA	"n",0		; 1D70 n tilde
		UNISMA	"p",0		; 1D71 p tilde
		UNISMA	"r",0		; 1D72 r tilde
		UNISMA	"r",0		; 1D73 b fishhook tilde
		UNISMA	"s",0		; 1D74 s tilde
		UNISMA	"t",0		; 1D75 t tilde
		UNISMA	"z",0		; 1D76 z tilde
		UNISMA	"g",0		; 1D77 turned g
		UNIMOD	"H"		; 1D78 modifier cyrillic EN
		UNISMA	"g",0		; 1D79 insular g
		UNISMA	"t",0		; 1D7A small th strikethrough
		UNISMA	"i",0		; 1D7B smal capital I stroke
		UNISMA	"i",0		; 1D7C small iota stroke
		UNISMA	"p",2c63h	; 1D7D p stroke
		UNISMA	"u",0		; 1D7E small capital U stroke
		UNISMA	"u",0		; 1D7F small upsilon stroke
		UNISMA	"b",0		; 1D80 b palatal hook
		UNISMA	"d",0		; 1D81 d palatal hook
		UNISMA	"f",0		; 1D82 f palatal hook
		UNISMA	"g",0		; 1D83 g palatal hook
		UNISMA	"k",0		; 1D84 k palatal hook
		UNISMA	"l",0		; 1D85 l palatal hook
		UNISMA	"m",0		; 1D86 m palatal hook
		UNISMA	"n",0		; 1D87 n palatal hook
		UNISMA	"p",0		; 1D88 p palatal hook
		UNISMA	"r",0		; 1D89 r palatal hook
		UNISMA	"s",0		; 1D8A s palatal hook
		UNISMA	UNINOASC,0	; 1D8B small esh palatal hook
		UNISMA	"v",0		; 1D8C v palatal hook
		UNISMA	"x",0		; 1D8D x palatal hook
		UNISMA	"z",0		; 1D8E z palatal hook
		UNISMA	"a",0		; 1D8F a retroflex hook
		UNISMA	"a",0		; 1D90 small alpha retroflex hook
		UNISMA	"d",0		; 1D91 d hook tail
		UNISMA	"e",0		; 1D92 e retroflex hook
		UNISMA	UNINOASC,0	; 1D93 open e retroflex hook
		UNISMA	UNINOASC,0	; 1D94 reversed open e retroflex hook
		UNISMA	"e",0		; 1D95 small schwa retroflex hook
		UNISMA	"i",0		; 1D96 i retroflex hook
		UNISMA	"c",0		; 1D97 open o retroflex hook
		UNISMA	UNINOASC,0	; 1D98 small esh retroflex hook
		UNISMA	"u",0		; 1D99 u retroflex hook
		UNISMA	UNINOASC,0	; 1D9A small ezh retroflex hook
		UNIMOD	"a"		; 1D9B modifier turned alpha
		UNIMOD	"c"		; 1D9C modifier c
		UNIMOD	"c"		; 1D9D modifier c curl
		UNIMOD	UNINOASC	; 1D9E modifier small eth
		UNIMOD	UNINOASC	; 1D9F modifier reversed open e
		UNIMOD	"f"		; 1DA0 modifier f
		UNIMOD	"j"		; 1DA1 modifier dotless j stroke
		UNIMOD	"g"		; 1DA2 modifier script g
		UNIMOD	"h"		; 1DA3 modifier turned h
		UNIMOD	"i"		; 1DA4 modifier i stroke
		UNIMOD	"i"		; 1DA5 modifier small iota
		UNIMOD	"i"		; 1DA6 modifier small capital I
		UNIMOD	"i"		; 1DA7 modifier small capital I stroke
		UNIMOD	"j"		; 1DA8 modifier j crossed-tail
		UNIMOD	"l"		; 1DA9 modifier l retroflex hook
		UNIMOD	"l"		; 1DAA modifier l palatal hook
		UNIMOD	"l"		; 1DAB modifier small capital L
		UNIMOD	"m"		; 1DAC modifier m hook
		UNIMOD	"m"		; 1DAD modifier turned m long leg
		UNIMOD	"n"		; 1DAE modifier n left hook
		UNIMOD	"n"		; 1DAF modifier n retroflex hook
		UNIMOD	"n"		; 1DB0 modifier small capital N
		UNIMOD	"o"		; 1DB1 modifier barred o
		UNIMOD	UNINOASC	; 1DB2 modifier small phi
		UNIMOD	"s"		; 1DB3 modifier s hook
		UNIMOD	UNINOASC	; 1DB4 modifier small esh
		UNIMOD	"t"		; 1DB5 modifier t palatal hook
		UNIMOD	"u"		; 1DB6 modifier u bar
		UNIMOD	"u"		; 1DB7 modifier small upsilon
		UNIMOD	"u"		; 1DB8 modifier small capital U
		UNIMOD	"v"		; 1DB9 modifier v hook
		UNIMOD	"v"		; 1DBA modifier turned v
		UNIMOD	"z"		; 1DBB modifier z
		UNIMOD	"z"		; 1DBC modifier z retroflex
		UNIMOD	"z"		; 1DBD modifier z curl
		UNIMOD	UNINOASC	; 1DBE modifier small ezh
		UNIMOD	UNINOASC	; 1DBF modifier small theta
		UNIMOD	"`"		; 1DC0 combining dotted grave accent
		UNIMOD	39		; 1DC1 combining dotted acute accent
		UNIMOD	","		; 1DC2 combining snake below
		UNIMOD	"~"		; 1DC3 combining suspension mark
		UNIMOD	"~"		; 1DC4 combining macron-acute
		UNIMOD	"~"		; 1DC5 combining grave-macron
		UNIMOD	"~"		; 1DC6 combining macron-grave
		UNIMOD	"~"		; 1DC7 combining acute-macron
		UNIMOD	"~"		; 1DC8 combining grave-acute-grave
		UNIMOD	"~"		; 1DC9 combining acute-grave-acute
		UNIMOD	"r"		; 1DCA combining r below
		UNIINV	51		; 1DCB..1DFD
		UNIMOD	"<"		; 1DFE combining left arrowhead above
		UNIMOD	">"		; 1DFF combining right arrowhead below

CHType1E00:	UNICAP	"A",1e01h	; 1E00 A ring below
		UNISMA	"a",1e00h	; 1E01 a ring below
		UNICAP	"B",1e03h	; 1E02 B dot above
		UNISMA	"b",1e02h	; 1E03 b dot above
		UNICAP	"B",1e05h	; 1E04 B dot below
		UNISMA	"b",1e04h	; 1E05 b dot below
		UNICAP	"B",1e07h	; 1E06 B line below
		UNISMA	"b",1e06h	; 1E07 b line below
		UNICAP	"C",1e09h	; 1E08 C cedilla acute
		UNISMA	"c",1e08h	; 1E09 c cedilla acute
		UNICAP	"D",1e0bh	; 1E0A D dot above
		UNISMA	"d",1e0ah	; 1E0B d dot above
		UNICAP	"D",1e0dh	; 1E0C D dot below
		UNISMA	"d",1e0ch	; 1E0D d dot below
		UNICAP	"D",1e0fh	; 1E0E D line below
		UNISMA	"d",1e0eh	; 1E0F d line below
		UNICAP	"D",1e11h	; 1E10 D cedilla
		UNISMA	"d",1e10h	; 1E11 d cedilla
		UNICAP	"D",1e13h	; 1E12 D circumflex
		UNISMA	"d",1e12h	; 1E13 d circumflex
		UNICAP	"E",1e15h	; 1E14 E macron grave
		UNISMA	"e",1e14h	; 1E15 e macron grave
		UNICAP	"E",1e17h	; 1E16 E macron acute
		UNISMA	"e",1e16h	; 1E17 e macron acute
		UNICAP	"E",1e19h	; 1E18 E circumflex below
		UNISMA	"e",1e18h	; 1E19 e circumflex below
		UNICAP	"E",1e1bh	; 1E1A E tilde below
		UNISMA	"e",1e1ah	; 1E1B e tilde below
		UNICAP	"E",1e1dh	; 1E1C E cedilla breve
		UNISMA	"e",1e1ch	; 1E1D e cedilla breve
		UNICAP	"F",1e1fh	; 1E1E F dot above
		UNISMA	"f",1e1eh	; 1E1F f dot above
		UNICAP	"G",1e21h	; 1E20 G macron
		UNISMA	"g",1e20h	; 1E21 g macron
		UNICAP	"H",1e23h	; 1E22 H dot above
		UNISMA	"h",1e22h	; 1E23 h dot above
		UNICAP	"H",1e25h	; 1E24 H dot below
		UNISMA	"h",1e24h	; 1E25 h dot below
		UNICAP	"H",1e27h	; 1E26 H diaeresis
		UNISMA	"h",1e26h	; 1E27 h diaeresis
		UNICAP	"H",1e29h	; 1E28 H cedilla
		UNISMA	"h",1e28h	; 1E29 h cedilla
		UNICAP	"H",1e2bh	; 1E2A H breve below
		UNISMA	"h",1e2ah	; 1E2B h breve below
		UNICAP	"I",1e2dh	; 1E2C I tilde below
		UNISMA	"i",1e2ch	; 1E2D i tilde below
		UNICAP	"I",1e2fh	; 1E2E I diaeresis acute
		UNISMA	"i",1e2eh	; 1E2F i diaeresis acute
		UNICAP	"K",1e31h	; 1E30 K acute
		UNISMA	"k",1e30h	; 1E31 k acute
		UNICAP	"K",1e33h	; 1E32 K dot below
		UNISMA	"k",1e32h	; 1E33 k dot below
		UNICAP	"K",1e35h	; 1E34 K line below
		UNISMA	"k",1e34h	; 1E35 k line below
		UNICAP	"K",1e37h	; 1E36 K dot below
		UNISMA	"k",1e36h	; 1E37 k dot below
		UNICAP	"L",1e39h	; 1E38 L dot below macron
		UNISMA	"l",1e38h	; 1E39 l dot below macron
		UNICAP	"L",1e3bh	; 1E3A L line below
		UNISMA	"l",1e3ah	; 1E3B l line below
		UNICAP	"L",1e3dh	; 1E3C L circumflex below
		UNISMA	"l",1e3ch	; 1E3D l circumflex below
		UNICAP	"M",1e3fh	; 1E3E M acute
		UNISMA	"m",1e3eh	; 1E3F m acute
		UNICAP	"M",1e41h	; 1E40 M dot above
		UNISMA	"m",1e40h	; 1E41 m dot above
		UNICAP	"M",1e43h	; 1E42 M dot below
		UNISMA	"m",1e42h	; 1E43 m dot below
		UNICAP	"N",1e45h	; 1E44 N dot above
		UNISMA	"n",1e44h	; 1E45 n dot above
		UNICAP	"N",1e47h	; 1E46 N dot below
		UNISMA	"n",1e46h	; 1E47 n dot below
		UNICAP	"N",1e49h	; 1E48 N line below
		UNISMA	"n",1e48h	; 1E49 n line below
		UNICAP	"N",1e4bh	; 1E4A N circumflex below
		UNISMA	"n",1e4ah	; 1E4B n circumflex below
		UNICAP	"O",1e4dh	; 1E4C O tilde acute
		UNISMA	"o",1e4ch	; 1E4D o tilde acute
		UNICAP	"O",1e4fh	; 1E4E O tilde diaeresis
		UNISMA	"o",1e4eh	; 1E4F o tilde diaeresis
		UNICAP	"O",1e51h	; 1E50 O macron grave
		UNISMA	"o",1e50h	; 1E51 o macron grave
		UNICAP	"O",1e53h	; 1E52 O macron acute
		UNISMA	"o",1e52h	; 1E53 o macron acute
		UNICAP	"P",1e55h	; 1E54 P acute
		UNISMA	"p",1e54h	; 1E55 p acute
		UNICAP	"P",1e57h	; 1E56 P dot above
		UNISMA	"p",1e56h	; 1E57 p dot above
		UNICAP	"R",1e59h	; 1E58 R dot above
		UNISMA	"r",1e58h	; 1E59 r dot above
		UNICAP	"R",1e5bh	; 1E5A R dot below
		UNISMA	"r",1e5ah	; 1E5B r dot below
		UNICAP	"R",1e5dh	; 1E5C R dot below macron
		UNISMA	"r",1e5ch	; 1E5D r dot below macron
		UNICAP	"R",1e5fh	; 1E5E R line below
		UNISMA	"r",1e5eh	; 1E5F r line below
		UNICAP	"S",1e61h	; 1E60 S dot above
		UNISMA	"s",1e60h	; 1E61 s dot above
		UNICAP	"S",1e63h	; 1E62 S dot below
		UNISMA	"s",1e62h	; 1E63 s dot below
		UNICAP	"S",1e65h	; 1E64 S acute dot above
		UNISMA	"s",1e64h	; 1E65 s acute dot above
		UNICAP	"S",1e67h	; 1E66 S caron dot above
		UNISMA	"s",1e66h	; 1E67 s caron dot above
		UNICAP	"S",1e69h	; 1E68 S dot below dot above
		UNISMA	"s",1e68h	; 1E69 s dot below dot above
		UNICAP	"T",1e6bh	; 1E6A T dot above
		UNISMA	"t",1e6ah	; 1E6B t dot above
		UNICAP	"T",1e6dh	; 1E6C T dot below
		UNISMA	"t",1e6ch	; 1E6D t dot below
		UNICAP	"T",1e6fh	; 1E6E T line below
		UNISMA	"t",1e6eh	; 1E6F t line below
		UNICAP	"T",1e71h	; 1E70 T circumflex below
		UNISMA	"t",1e70h	; 1E71 t circumflex below
		UNICAP	"U",1e73h	; 1E72 U diaeresis below
		UNISMA	"u",1e72h	; 1E73 u diaeresis below
		UNICAP	"U",1e75h	; 1E74 U tilde below
		UNISMA	"u",1e74h	; 1E75 u tilde below
		UNICAP	"U",1e77h	; 1E76 U circumflex below
		UNISMA	"u",1e76h	; 1E77 u circumflex below
		UNICAP	"U",1e79h	; 1E78 U tilde acute
		UNISMA	"u",1e78h	; 1E79 u tilde acute
		UNICAP	"U",1e7bh	; 1E7A U macron diaeresis
		UNISMA	"u",1e7ah	; 1E7B u macron diaeresis
		UNICAP	"V",1e7dh	; 1E7C V tilde
		UNISMA	"v",1e7ch	; 1E7D v tilde
		UNICAP	"V",1e7fh	; 1E7E V dot below
		UNISMA	"v",1e7eh	; 1E7F v dot below
		UNICAP	"W",1e81h	; 1E80 W grave
		UNISMA	"w",1e80h	; 1E81 w grave
		UNICAP	"W",1e83h	; 1E82 W acute
		UNISMA	"w",1e82h	; 1E83 w acute
		UNICAP	"W",1e85h	; 1E84 W diaeresis
		UNISMA	"w",1e84h	; 1E85 w diaeresis
		UNICAP	"W",1e87h	; 1E86 W dot above
		UNISMA	"w",1e86h	; 1E87 w dot above
		UNICAP	"W",1e89h	; 1E88 W dot below
		UNISMA	"w",1e88h	; 1E89 w dot below
		UNICAP	"X",1e8bh	; 1E8A X dot above
		UNISMA	"x",1e8ah	; 1E8B x dot above
		UNICAP	"X",1e8dh	; 1E8C X diaeresis
		UNISMA	"x",1e8ch	; 1E8D x diaeresis
		UNICAP	"Y",1e8fh	; 1E8E Y dot above
		UNISMA	"y",1e8eh	; 1E8F y dot above
		UNICAP	"Z",1e91h	; 1E90 Z circumflex
		UNISMA	"z",1e90h	; 1E91 z circumflex
		UNICAP	"Z",1e93h	; 1E92 Z dot below
		UNISMA	"z",1e92h	; 1E93 z dot below
		UNICAP	"Z",1e95h	; 1E94 Z line below
		UNISMA	"z",1e94h	; 1E95 z line below
		UNISMA	"h",0		; 1E96 h line below
		UNISMA	"t",0		; 1E97 t diaeresis
		UNISMA	"w",0		; 1E98 w ring above
		UNISMA	"y",0		; 1E99 y ring above
		UNISMA	"a",0		; 1E9A a right half ring
		UNISMA	UNINOASC,0	; 1E9B small long S dot above
		UNIINV	4		; 1E9C..1E9F
		UNICAP	"A",1ea1h	; 1EA0 A dot below
		UNISMA	"a",1ea0h	; 1EA1 a dot below
		UNICAP	"A",1ea3h	; 1EA2 A hook above
		UNISMA	"a",1ea2h	; 1EA3 a hook above
		UNICAP	"A",1ea5h	; 1EA4 A circumflex acute
		UNISMA	"a",1ea4h	; 1EA5 a circumflex acute
		UNICAP	"A",1ea7h	; 1EA6 A circumflex grave
		UNISMA	"a",1ea6h	; 1EA7 a circumflex grave
		UNICAP	"A",1ea9h	; 1EA8 A circumflex hook above
		UNISMA	"a",1ea8h	; 1EA9 a circumflex hook above
		UNICAP	"A",1eabh	; 1EAA A circumflex tilde
		UNISMA	"a",1eaah	; 1EAB a circumflex tilde
		UNICAP	"A",1eadh	; 1EAC A circumflex dot below
		UNISMA	"a",1each	; 1EAD a circumflex dot below
		UNICAP	"A",1eafh	; 1EAE A breve acute
		UNISMA	"a",1eaeh	; 1EAF a breve acute
		UNICAP	"A",1eb1h	; 1EB0 A breve grave
		UNISMA	"a",1eb0h	; 1EB1 a breve grave
		UNICAP	"A",1eb3h	; 1EB2 A breve hook above
		UNISMA	"a",1eb2h	; 1EB3 a breve hook above
		UNICAP	"A",1eb5h	; 1EB4 A breve tilde
		UNISMA	"a",1eb4h	; 1EB5 a breve tilde
		UNICAP	"A",1eb7h	; 1EB6 A breve dot below
		UNISMA	"a",1eb6h	; 1EB7 a breve dot below
		UNICAP	"E",1eb9h	; 1EB8 E dot below
		UNISMA	"e",1eb8h	; 1EB9 e dot below
		UNICAP	"E",1ebbh	; 1EBA E hook above
		UNISMA	"e",1ebah	; 1EBB e hook above
		UNICAP	"E",1ebdh	; 1EBC E tilde
		UNISMA	"e",1ebch	; 1EBD e tilde
		UNICAP	"E",1ebfh	; 1EBE E circumflex acute
		UNISMA	"e",1ebeh	; 1EBF e circumflex acute
		UNICAP	"E",1ec1h	; 1EC0 E circumflex grave
		UNISMA	"e",1ec0h	; 1EC1 e circumflex grave
		UNICAP	"E",1ec3h	; 1EC2 E circumflex hook above
		UNISMA	"e",1ec2h	; 1EC3 e circumflex hook above
		UNICAP	"E",1ec5h	; 1EC4 E circumflex tilde
		UNISMA	"e",1ec4h	; 1EC5 e circumflex tilde
		UNICAP	"E",1ec7h	; 1EC6 E circumflex dot below
		UNISMA	"e",1ec6h	; 1EC7 e circumflex dot below
		UNICAP	"I",1ec9h	; 1EC8 I hook above
		UNISMA	"i",1ec8h	; 1EC9 i hook above
		UNICAP	"I",1ecbh	; 1ECA I dot below
		UNISMA	"i",1ecah	; 1ECB i dot below
		UNICAP	"O",1ecdh	; 1ECC O dot below
		UNISMA	"o",1ecch	; 1ECD o dot below
		UNICAP	"O",1ecfh	; 1ECE O hook above
		UNISMA	"o",1eceh	; 1ECF o hook above
		UNICAP	"O",1ed1h	; 1ED0 O circumflex acute
		UNISMA	"o",1ed0h	; 1ED1 o circumflex acute
		UNICAP	"O",1ed3h	; 1ED2 O circumflex grave
		UNISMA	"o",1ed2h	; 1ED3 o circumflex grave
		UNICAP	"O",1ed5h	; 1ED4 O circumflex hook above
		UNISMA	"o",1ed4h	; 1ED5 o circumflex hook above
		UNICAP	"O",1ed7h	; 1ED6 O circumflex tilde
		UNISMA	"o",1ed6h	; 1ED7 o circumflex tilde
		UNICAP	"O",1ed9h	; 1ED8 O circumflex dot below
		UNISMA	"o",1ed8h	; 1ED9 o circumflex dot below
		UNICAP	"O",1edbh	; 1EDA O horn acute
		UNISMA	"o",1edah	; 1EDB o horn acute
		UNICAP	"O",1eddh	; 1EDC O horn grave
		UNISMA	"o",1edch	; 1EDD o horn grave
		UNICAP	"O",1edfh	; 1EDE O horn hook above
		UNISMA	"o",1edeh	; 1EDF o horn hook above
		UNICAP	"O",1ee1h	; 1EE0 O horn tilde
		UNISMA	"o",1ee0h	; 1EE1 o horn tilde
		UNICAP	"O",1ee3h	; 1EE2 O horn dot below
		UNISMA	"o",1ee2h	; 1EE3 o horn dot below
		UNICAP	"U",1ee5h	; 1EE4 U dot below
		UNISMA	"u",1ee4h	; 1EE5 u dot below
		UNICAP	"U",1ee7h	; 1EE6 U hook above
		UNISMA	"u",1ee6h	; 1EE7 u hook above
		UNICAP	"U",1ee9h	; 1EE8 U horn acute
		UNISMA	"u",1ee8h	; 1EE9 u horn acute
		UNICAP	"U",1eebh	; 1EEA U horn grave
		UNISMA	"u",1eeah	; 1EEB u horn grave
		UNICAP	"U",1eedh	; 1EEC U horn hook above
		UNISMA	"u",1eech	; 1EED u horn hook above
		UNICAP	"U",1eefh	; 1EEE U horn tilde
		UNISMA	"u",1eeeh	; 1EEF u horn tilde
		UNICAP	"U",1ef1h	; 1EF0 U horn dot below
		UNISMA	"u",1ef0h	; 1EF1 u horn dot below
		UNICAP	"Y",1ef3h	; 1EF2 Y grave
		UNISMA	"y",1ef2h	; 1EF3 y grave
		UNICAP	"Y",1ef5h	; 1EF4 Y dot below
		UNISMA	"y",1ef4h	; 1EF5 y dot below
		UNICAP	"Y",1ef7h	; 1EF6 Y hook above
		UNISMA	"y",1ef6h	; 1EF7 y hook above
		UNICAP	"Y",1ef9h	; 1EF8 Y tilde
		UNISMA	"y",1ef8h	; 1EF9 y tilde
		UNIINV	6		; 1EFA..1EFF

CHType1F00:	UNISMA	UNINOASC,1f08h	; 1F00 greek small alpha psili
		UNISMA	UNINOASC,1f09h	; 1F01 greek small alpha dasia
		UNISMA	UNINOASC,1f0ah	; 1F02 greek small alpha psili varia
		UNISMA	UNINOASC,1f0bh	; 1F03 greek small alpha dasia varia
		UNISMA	UNINOASC,1f0ch	; 1F04 greek small alpha psili oxia
		UNISMA	UNINOASC,1f0dh	; 1F05 greek small alpha dasia oxia
		UNISMA	UNINOASC,1f0eh	; 1F06 greek small alpha psili perisp.
		UNISMA	UNINOASC,1f0fh	; 1F07 greek small alpha dasia perisp.
		UNICAP	"A",1f00h	; 1F08 greek capital alpha psili
		UNICAP	"A",1f01h	; 1F09 greek capital alpha dasia
		UNICAP	"A",1f02h	; 1F0A greek capital alpha psili varia
		UNICAP	"A",1f03h	; 1F0B greek capital alpha dasia varia
		UNICAP	"A",1f04h	; 1F0C greek capital alpha psili oxia
		UNICAP	"A",1f05h	; 1F0D greek capital alpha dasia oxia
		UNICAP	"A",1f06h	; 1F0E greek cap. alpha psili perisp.
		UNICAP	"A",1f07h	; 1F0F greek cap. alpha dasia perisp.
		UNISMA	UNINOASC,1f18h	; 1F10 greek small epsilon psili
		UNISMA	UNINOASC,1f19h	; 1F11 greek small epsilon dasia
		UNISMA	UNINOASC,1f1ah	; 1F12 greek small epsilon psili varia
		UNISMA	UNINOASC,1f1bh	; 1F13 greek small epsilon dasia varia
		UNISMA	UNINOASC,1f1ch	; 1F14 greek small epsilon psili oxia
		UNISMA	UNINOASC,1f1dh	; 1F15 greek small epsilon dasia oxia
		UNIINV	2		; 1F16..1F17
		UNICAP	"E",1f10h	; 1F18 greek capital epsilon psili
		UNICAP	"E",1f11h	; 1F19 greek capital epsilon dasia
		UNICAP	"E",1f12h	; 1F1A greek cap. epsilon psili varia
		UNICAP	"E",1f13h	; 1F1B greek cap. epsilon dasia varia
		UNICAP	"E",1f14h	; 1F1C greek capital epsilon psili oxia
		UNICAP	"E",1f15h	; 1F1D greek capital epsilon dasia oxia
		UNIINV	2		; 1F1E..1F1F
		UNISMA	"n",1f28h	; 1F20 greek small eta psili
		UNISMA	"n",1f29h	; 1F21 greek small eta dasia
		UNISMA	"n",1f2ah	; 1F22 greek small eta psili varia
		UNISMA	"n",1f2bh	; 1F23 greek small eta dasia varia
		UNISMA	"n",1f2ch	; 1F24 greek small eta psili oxia
		UNISMA	"n",1f2dh	; 1F25 greek small eta dasia oxia
		UNISMA	"n",1f2eh	; 1F26 greek small eta psili perisp.
		UNISMA	"n",1f2fh	; 1F27 greek small eta dasia perisp.
		UNICAP	"H",1f20h	; 1F28 greek capital eta psili
		UNICAP	"H",1f21h	; 1F29 greek capital eta dasia
		UNICAP	"H",1f22h	; 1F2A greek capital eta psili varia
		UNICAP	"H",1f23h	; 1F2B greek capital eta dasia varia
		UNICAP	"H",1f24h	; 1F2C greek capital eta psili oxia
		UNICAP	"H",1f25h	; 1F2D greek capital eta dasia oxia
		UNICAP	"H",1f26h	; 1F2E greek capital eta psili perisp.
		UNICAP	"H",1f27h	; 1F2F greek capital eta dasia perisp.
		UNISMA	"i",1f38h	; 1F30 greek small iota psili
		UNISMA	"i",1f39h	; 1F31 greek small iota dasia
		UNISMA	"i",1f3ah	; 1F32 greek small iota psili varia
		UNISMA	"i",1f3bh	; 1F33 greek small iota dasia varia
		UNISMA	"i",1f3ch	; 1F34 greek small iota psili oxia
		UNISMA	"i",1f3dh	; 1F35 greek small iota dasia oxia
		UNISMA	"i",1f3eh	; 1F36 greek small iota psili perisp.
		UNISMA	"i",1f3fh	; 1F37 greek small iota dasia perisp.
		UNICAP	"I",1f30h	; 1F38 greek capital iota psili
		UNICAP	"I",1f31h	; 1F39 greek capital iota dasia
		UNICAP	"I",1f32h	; 1F3A greek capital iota psili varia
		UNICAP	"I",1f33h	; 1F3B greek capital iota dasia varia
		UNICAP	"I",1f34h	; 1F3C greek capital iota psili oxia
		UNICAP	"I",1f35h	; 1F3D greek capital iota dasia oxia
		UNICAP	"I",1f36h	; 1F3E greek capital iota psili perisp.
		UNICAP	"I",1f37h	; 1F3F greek capital iota dasia perisp.
		UNISMA	"o",1f48h	; 1F40 greek small epsilon psili
		UNISMA	"o",1f49h	; 1F41 greek small epsilon dasia
		UNISMA	"o",1f4ah	; 1F42 greek small epsilon psili varia
		UNISMA	"o",1f4bh	; 1F43 greek small epsilon dasia varia
		UNISMA	"o",1f4ch	; 1F44 greek small epsilon psili oxia
		UNISMA	"o",1f4dh	; 1F45 greek small epsilon dasia oxia
		UNIINV	2		; 1F46..1F47
		UNICAP	"O",1f40h	; 1F48 greek capital epsilon psili
		UNICAP	"O",1f41h	; 1F49 greek capital epsilon dasia
		UNICAP	"O",1f42h	; 1F4A greek cap. epsilon psili varia
		UNICAP	"O",1f43h	; 1F4B greek cap. epsilon dasia varia
		UNICAP	"O",1f44h	; 1F4C greek capital epsilon psili oxia
		UNICAP	"O",1f45h	; 1F4D greek capital epsilon dasia oxia
		UNIINV	2		; 1F4E..1F4F
		UNISMA	"u",0		; 1F50 greek small upsilon psili
		UNISMA	"u",1f59h	; 1F51 greek small upsilon dasia
		UNISMA	"u",0		; 1F52 greek small upsilon psili varia
		UNISMA	"u",1f5bh	; 1F53 greek small upsilon dasia varia
		UNISMA	"u",0		; 1F54 greek small upsilon psili oxia
		UNISMA	"u",1f5dh	; 1F55 greek small upsilon dasia oxia
		UNISMA	"u",0		; 1F56 greek small upsilon psili per.
		UNISMA	"u",1f5fh	; 1F57 greek small upsilon dasia per.
		UNIINV	1		; 1F58
		UNICAP	"Y",1f51h	; 1F59 greek capital upsilon dasia
		UNIINV	1		; 1F5A
		UNICAP	"Y",1f53h	; 1F5B greek capital upsilon dasia varia
		UNIINV	1		; 1F5C
		UNICAP	"Y",1f55h	; 1F5D greek capital upsilon dasia oxia
		UNIINV	1		; 1F5E
		UNICAP	"Y",1f57h	; 1F5F greek cap. upsilon dasia perisp.
		UNISMA	UNINOASC,1f68h	; 1F60 greek small omega psili
		UNISMA	UNINOASC,1f69h	; 1F61 greek small omega dasia
		UNISMA	UNINOASC,1f6ah	; 1F62 greek small omega psili varia
		UNISMA	UNINOASC,1f6bh	; 1F63 greek small omega dasia varia
		UNISMA	UNINOASC,1f6ch	; 1F64 greek small omega psili oxia
		UNISMA	UNINOASC,1f6dh	; 1F65 greek small omega dasia oxia
		UNISMA	UNINOASC,1f6eh	; 1F66 greek small omega psili perisp.
		UNISMA	UNINOASC,1f6fh	; 1F67 greek small omega dasia perisp.
		UNICAP	UNINOASC,1f60h	; 1F68 greek capital omega psili
		UNICAP	UNINOASC,1f61h	; 1F69 greek capital omega dasia
		UNICAP	UNINOASC,1f62h	; 1F6A greek capital omega psili varia
		UNICAP	UNINOASC,1f63h	; 1F6B greek capital omega dasia varia
		UNICAP	UNINOASC,1f64h	; 1F6C greek capital omega psili oxia
		UNICAP	UNINOASC,1f65h	; 1F6D greek capital omega dasia oxia
		UNICAP	UNINOASC,1f66h	; 1F6E greek cap. omega psili perisp.
		UNICAP	UNINOASC,1f67h	; 1F6F greek cap. omega dasia perisp.
		UNISMA	UNINOASC,1fbah	; 1F70 greek small alpha varia
		UNISMA	UNINOASC,1fbbh	; 1F71 greek small alpha oxia
		UNISMA	UNINOASC,1fc8h	; 1F72 greek small epsilon varia
		UNISMA	UNINOASC,1fc9h	; 1F73 greek small epsilon oxia
		UNISMA	"n",1fcah	; 1F74 greek small eta varia
		UNISMA	"n",1fcbh	; 1F75 greek small eta oxia
		UNISMA	"i",1fdah	; 1F76 greek small iota varia
		UNISMA	"i",1fdbh	; 1F77 greek small iota oxia
		UNISMA	"o",1ff8h	; 1F78 greek small omicron varia
		UNISMA	"o",1ff9h	; 1F79 greek small omicron oxia
		UNISMA	"u",1feah	; 1F7A greek small upsilon varia
		UNISMA	"u",1febh	; 1F7B greek small upsilon oxia
		UNISMA	UNINOASC,1ffah	; 1F7C greek small omega varia
		UNISMA	UNINOASC,1ffbh	; 1F7D greek small omega oxia
		UNIINV	2		; 1F7E..1F7F
		UNISMA	UNINOASC,1f88h	; 1F80 gr.sm.alpha psili ypogegrammeni
		UNISMA	UNINOASC,1f89h	; 1F81 gr.sm.alpha dasia ypogegrammeni
		UNISMA	UNINOASC,1f8ah	; 1F82 gr.sm.alpha psili varia ypogegr.
		UNISMA	UNINOASC,1f8bh	; 1F83 gr.sm.alpha dasia varia ypogegr.
		UNISMA	UNINOASC,1f8ch	; 1F84 gr.sm.alpha psili oxia ypogegr.
		UNISMA	UNINOASC,1f8dh	; 1F85 gr.sm.alpha dasia oxia ypogegr.
		UNISMA	UNINOASC,1f8eh	; 1F86 gr.sm.alpha psili perisp.ypog.
		UNISMA	UNINOASC,1f8fh	; 1F87 gr.sm.alpha dasia perisp.ypog.
		UNICAP	"A",1f80h	; 1F88 gr.ca.alpha psili ypogegr.
		UNICAP	"A",1f81h	; 1F89 gr.ca.alpha dasia ypogegr.
		UNICAP	"A",1f82h	; 1F8A gr.ca.alpha psili varia ypogegr.
		UNICAP	"A",1f83h	; 1F8B gr.ca.alpha dasia varia ypogegr.
		UNICAP	"A",1f84h	; 1F8C gr.ca.alpha psili oxia ypogegr.
		UNICAP	"A",1f85h	; 1F8D gr.ca.alpha dasia oxia ypogegr.
		UNICAP	"A",1f86h	; 1F8E gr.ca.alpha psili perisp.ypog.
		UNICAP	"A",1f87h	; 1F8F gr.ca.alpha dasia perisp.ypog.
		UNISMA	"n",1f98h	; 1F90 gr.sm. eta psili ypogegrammeni
		UNISMA	"n",1f99h	; 1F91 gr.sm. eta dasia ypogegrammeni
		UNISMA	"n",1f9ah	; 1F92 gr.sm. eta psili varia ypogegr.
		UNISMA	"n",1f9bh	; 1F93 gr.sm. eta dasia varia ypogegr.
		UNISMA	"n",1f9ch	; 1F94 gr.sm. eta psili oxia ypogegr.
		UNISMA	"n",1f9dh	; 1F95 gr.sm. eta dasia oxia ypogegr.
		UNISMA	"n",1f9eh	; 1F96 gr.sm. eta psili perisp.ypog.
		UNISMA	"n",1f9fh	; 1F97 gr.sm. eta dasia perisp.ypog.
		UNICAP	"H",1f90h	; 1F98 gr.ca. eta psili ypogegr.
		UNICAP	"H",1f91h	; 1F99 gr.ca. eta dasia ypogegr.
		UNICAP	"H",1f92h	; 1F9A gr.ca. eta psili varia ypogegr.
		UNICAP	"H",1f93h	; 1F9B gr.ca. eta dasia varia ypogegr.
		UNICAP	"H",1f94h	; 1F9C gr.ca. eta psili oxia ypogegr.
		UNICAP	"H",1f95h	; 1F9D gr.ca. eta dasia oxia ypogegr.
		UNICAP	"H",1f96h	; 1F9E gr.ca. eta psili perisp.ypog.
		UNICAP	"H",1f97h	; 1F9F gr.ca. eta dasia perisp.ypog.
		UNISMA	UNINOASC,1fa8h	; 1FA0 gr.sm.omega psili ypogegrammeni
		UNISMA	UNINOASC,1fa9h	; 1FA1 gr.sm.omega dasia ypogegrammeni
		UNISMA	UNINOASC,1faah	; 1FA2 gr.sm.omega psili varia ypogegr.
		UNISMA	UNINOASC,1fabh	; 1FA3 gr.sm.omega dasia varia ypogegr.
		UNISMA	UNINOASC,1fach	; 1FA4 gr.sm.omega psili oxia ypogegr.
		UNISMA	UNINOASC,1fadh	; 1FA5 gr.sm.omega dasia oxia ypogegr.
		UNISMA	UNINOASC,1faeh	; 1FA6 gr.sm.omega psili perisp.ypog.
		UNISMA	UNINOASC,1fafh	; 1FA7 gr.sm.omega dasia perisp.ypog.
		UNICAP	UNINOASC,1fa0h	; 1FA8 gr.ca.omega psili ypogegr.
		UNICAP	UNINOASC,1fa1h	; 1FA9 gr.ca.omega dasia ypogegr.
		UNICAP	UNINOASC,1fa2h	; 1FAA gr.ca.omega psili varia ypogegr.
		UNICAP	UNINOASC,1fa3h	; 1FAB gr.ca.omega dasia varia ypogegr.
		UNICAP	UNINOASC,1fa4h	; 1FAC gr.ca.omega psili oxia ypogegr.
		UNICAP	UNINOASC,1fa5h	; 1FAD gr.ca.omega dasia oxia ypogegr.
		UNICAP	UNINOASC,1fa6h	; 1FAE gr.ca.omega psili perisp.ypog.
		UNICAP	UNINOASC,1fa7h	; 1FAF gr.ca.omega dasia perisp.ypog.
		UNISMA	UNINOASC,1fb8h	; 1FB0 greek small alpha vrachy
		UNISMA	UNINOASC,1fb9h	; 1FB1 greek small alpha macron
		UNISMA	UNINOASC,0	; 1FB2 greek small alpha varia ypogegr.
		UNISMA	UNINOASC,0	; 1FB3 greek small alpha ypogegrammeni
		UNISMA	UNINOASC,0	; 1FB4 greek small alpha oxia ypogegr.
		UNIINV	1		; 1FB5
		UNISMA	UNINOASC,0	; 1FB6 greek small alpha perispomeni
		UNISMA	UNINOASC,0	; 1FB7 greek sm. alpha perisp. ypogegr.
		UNICAP	"A",1fb0h	; 1FB8 greek capital alpha vrachy
		UNICAP	"A",1fb1h	; 1FB9 greek capital alpha macron
		UNICAP	"A",1f70h	; 1FBA greek capital alpha varia
		UNICAP	"A",1f71h	; 1FBB greek capital alpha oxia
		UNICAP	"A",0		; 1FBC greek capital alpha prosgegram.
		UNISIG	"`"		; 1FBD greek koronis
		UNISIG	","		; 1FBE greek prosgegrammeni
		UNISIG	39		; 1FBF greek psili
		UNISIG	"~"		; 1FC0 greek perispomeni
		UNISIG	"~"		; 1FC1 greek dialytika perispomeni
		UNISMA	"n",0		; 1FC2 greek small eta varia ypogegr.
		UNISMA	"n",0		; 1FC3 greek small eta ypogegrammeni
		UNISMA	"n",0		; 1FC4 greek small eta oxia ypogegr.
		UNIINV	1		; 1FC5
		UNISMA	"n",0		; 1FC6 greek small eta perispomeni
		UNISMA	"n",0		; 1FC7 greek small eta perisp. ypogegr.
		UNICAP	"E",1f72h	; 1FC8 greek capital epsilon varia
		UNICAP	"E",1f73h	; 1FC9 greek capital epsilon oxia
		UNICAP	"H",1f74h	; 1FCA greek capital eta varia
		UNICAP	"H",1f75h	; 1FCB greek capital eta oxia
		UNICAP	"H",0		; 1FCC greek capital eta prosgegrammeni
		UNISIG	33		; 1FCD greek psili varia
		UNISIG	33		; 1FCE greek psili oxia
		UNISIG	"~"		; 1FCF greek psili perispomeni
		UNISMA	"i",1fd8h	; 1FD0 greek small iota vrachy
		UNISMA	"i",1fd9h	; 1FD1 greek small iota macron
		UNISMA	"i",0		; 1FD2 greek small iota dialytika varia
		UNISMA	"i",0		; 1FD3 greek small iota dialytika oxia
		UNIINV	2		; 1FD4..1FD5
		UNISMA	"i",0		; 1FD6 greek small iota perispomeni
		UNISMA	"i",0		; 1FD7 greek small iota dialyt. perisp.
		UNICAP	"I",1fd0h	; 1FD8 greek capital iota vrachy
		UNICAP	"I",1fd1h	; 1FD9 greek capital iota macron
		UNICAP	"I",1f76h	; 1FDA greek capital iota varia
		UNICAP	"I",1f77h	; 1FDB greek capital iota oxia
		UNIINV	1		; 1FDC
		UNISIG	33		; 1FDD greek dasia varia
		UNISIG	33		; 1FDE greek dasia oxia
		UNISIG	"~"		; 1FDF greek dasia perispomeni
		UNISMA	"u",1fe8h	; 1FE0 greek small upsilon vrachy
		UNISMA	"u",1fe9h	; 1FE1 greek small upsilon macron
		UNISMA	"u",0		; 1FE2 greek small upsilon dial. varia
		UNISMA	"u",0		; 1FE3 greek small upsilon dial. oxia
		UNISMA	UNINOASC,0	; 1FE4 greek small rho psili
		UNISMA	UNINOASC,1fech	; 1FE5 greek small rho dasia
		UNISMA	"u",0		; 1FE6 greek small upsilon perispomeni
		UNISMA	"u",0		; 1FE7 greek small upsilon dial.perisp.
		UNICAP	"Y",1fe0h	; 1FE8 greek capital upsilon vrachy
		UNICAP	"Y",1fe1h	; 1FE9 greek capital upsilon macron
		UNICAP	"Y",1f7ah	; 1FEA greek capital upsilon varia
		UNICAP	"Y",1f7bh	; 1FEB greek capital upsilon oxia
		UNICAP	"P",1fe5h	; 1FEC greek capital rho dasia
		UNISIG	"`"		; 1FED greek dialytika varia
		UNISIG	39		; 1FEE greek dialytika oxia
		UNISIG	"`"		; 1FEF greek varia
		UNIINV	2		; 1FF0..1FF1
		UNISMA	UNINOASC,0	; 1FF2 greek small omega varia ypogegr.
		UNISMA	UNINOASC,0	; 1FF3 greek small omega ypogegrammeni
		UNISMA	UNINOASC,0	; 1FF4 greek small omega oxia ypogegr.
		UNIINV	1		; 1FF5
		UNISMA	UNINOASC,0	; 1FF6 greek small omega perispomeni
		UNISMA	UNINOASC,0	; 1FF7 greek small omega perisp. ypog.
		UNICAP	"O",1f78h	; 1FF8 greek capital omicron varia
		UNICAP	"O",1f79h	; 1FF9 greek capital omicron oxia
		UNICAP	UNINOASC,1f7ch	; 1FFA greek capital omega varia
		UNICAP	UNINOASC,1f7dh	; 1FFB greek capital omega oxia
		UNICAP	UNINOASC,0	; 1FFC greek capital omega prosgegram.
		UNISIG	39		; 1FFD greek oxia
		UNISIG	"`"		; 1FFE greek dasia
		UNIINV	1		; 1FFF

CHType2000:	UNISPC	12		; 2000..200B space
		UNICTR	4		; 200C..200F format characters
		UNIMRK	"-"		; 2010 hyphen
		UNIMRK	"-"		; 2011 non-breaking hyphen
		UNIMRK	"-"		; 2012 figure dash
		UNIMRK	"-"		; 2013 en dash
		UNIMRK	"-"		; 2014 em dash
		UNISIG	"-"		; 2015 horizontal bar
		UNISIG	"|"		; 2016 double vertical line
		UNISIG	"_"		; 2017 double low line
		UNIMRK	39		; 2018 left single quotation mark
		UNIMRK	39		; 2019 right single quotation mark
		UNIMRK	39		; 201A single low-9 quotation mark
		UNIMRK	39		; 201B single high-reversed-9 quotation
		UNIMRK	33		; 201C left double quotation mark
		UNIMRK	33		; 201D right double quotation mark
		UNIMRK	33		; 201E double low-9 quotation mark
		UNIMRK	33		; 201F double high-reversed-9 quotation
		UNISIG	"+"		; 2020 dagger
		UNISIG	"+"		; 2021 double dagger
		UNIMRK	"."		; 2022 bullet
		UNIMRK	">"		; 2023 triangular bullet
		UNIMRK	"."		; 2024 one dot leader
		UNIMRK	"."		; 2025 two dot leader
		UNIMRK	"."		; 2026 horizontal ellipsis
		UNIMRK	"."		; 2027 hyphenation point
		UNICTR	8		; 2028..202F format characters
		UNISIG	"%"		; 2030 per mille sign
		UNISIG	"%"		; 2031 per ten thousand sign
		UNIMRK	39		; 2032 prime
		UNIMRK	33		; 2033 double prime
		UNIMRK	33		; 2034 triple prime
		UNIMRK	"`"		; 2035 reversed prime
		UNIMRK	33		; 2036 reversed double prime
		UNIMRK	33		; 2037 reversed triple prime
		UNIMRK	"^"		; 2038 caret
		UNIMRK	"<"		; 2039 single left angle quotation
		UNIMRK	">"		; 203A single right angle quotation
		UNIMRK	"X"		; 203B reference mark
		UNIMRK	"!"		; 203C double exclamation mark
		UNIMRK	"?"		; 203D interrobang
		UNIMRK	"~"		; 203E overline
		UNIMRK	"_"		; 203F undertie
		UNIMRK	"~"		; 2040 character tie
		UNIMRK	"/"		; 2041 caret insertion point
		UNISIG	"*"		; 2042 asterism
		UNIMRK	"-"		; 2043 hyphen bullet
		UNISIG	"/"		; 2044 fraction slash
		UNIMRK	"["		; 2045 left square bracket quill
		UNIMRK	"]"		; 2046 right square bracket quill
		UNIMRK	"?"		; 2047 double question mark
		UNIMRK	"?"		; 2048 question exclamation mark
		UNIMRK	"!"		; 2049 exclamation question mark
		UNISIG	UNINOASC	; 204A tironian sign et
		UNISIG	UNINOASC	; 204B reversed pilcrow sign
		UNISIG	UNINOASC	; 204C black leftwards bullet
		UNISIG	UNINOASC	; 204D black rightwards bullet
		UNISIG	"*"		; 204E low asterisk
		UNIMRK	";"		; 204F reversed semicolon
		UNIMRK	UNINOASC	; 2050 close up
		UNISIG	"*"		; 2051 two asterisks aligned vertically
		UNISIG	"%"		; 2052 commercial minus sign
		UNISIG	"~"		; 2053 swung dash
		UNIMRK	"_"		; 2054 inverted undertie
		UNISIG	"*"		; 2055 flower punctuation mark
		UNIMRK	":"		; 2056 three dot punctuation
		UNIMRK	33		; 2057 quadruple prime
		UNIMRK	":"		; 2058 four dot punctuation
		UNIMRK	":"		; 2059 five dot punctuation
		UNIMRK	":"		; 205A two dot punctuation
		UNIMRK	":"		; 205B four dot mark
		UNISIG	"+"		; 205C dotted cross
		UNIMRK	":"		; 205D tricolon
		UNIMRK	":"		; 205E vertical four dots
		UNISPC	1		; 205F medium mathematical space
		UNICTR	1		; 2060 word joiner
		UNICTR	3		; 2061..2063 invisible operators
		UNIINV	6		; 2064..2069
		UNICTR	6		; 206A..206F deprecated
		UNIDIG	"0"		; 2070 superscript 0
		UNISMA	"i",0		; 2071 superscript i
		UNIINV	2		; 2072..2073
		UNIDIGCH "4",6		; 2074..2079 superscript 4..9
		UNISIG	"+"		; 207A superscript plus
		UNISIG	"-"		; 207B superscript minus
		UNISIG	"="		; 207C superscript equals
		UNIMRK	"("		; 207D superscript left parenthesis
		UNIMRK	")"		; 207E superscript right parenthesis
		UNISMA	"n",0		; 207F superscript n
		UNIDIGCH "0",10		; 2080..2089 subscript 0..9
		UNISIG	"+"		; 208A subscript plus
		UNISIG	"-"		; 208B subscript minus
		UNISIG	"="		; 208C subscript equals
		UNIMRK	"("		; 208D subscript left parenthesis
		UNIMRK	")"		; 208E subscript right parenthesis
		UNIINV	1		; 208F
		UNISMA	"a",0		; 2090 subscript a
		UNISMA	"e",0		; 2091 subscript e
		UNISMA	"o",0		; 2092 subscript o
		UNISMA	"x",0		; 2093 subscript x
		UNISMA	"e",0		; 2094 subscript schwa
		UNIINV	11		; 2095..209F
		UNISIG	"e"		; 20A0 euro-currency
		UNISIG	"C"		; 20A1 colon
		UNISIG	"C"		; 20A2 cruzeiro
		UNISIG	"F"		; 20A3 french franc
		UNISIG	"L"		; 20A4 lira
		UNISIG	"m"		; 20A5 mill
		UNISIG	"N"		; 20A6 naira
		UNISIG	"P"		; 20A7 peseta
		UNISIG	"R"		; 20A8 rupee
		UNISIG	"W"		; 20A9 won
		UNISIG	UNINOASC	; 20AA new sheqel
		UNISIG	"d"		; 20AB dong
		UNISIG	"E"		; 20AC euro
		UNISIG	"K"		; 20AD kip
		UNISIG	"T"		; 20AE tugrik
		UNISIG	"D"		; 20AF drachma
		UNISIG	UNINOASC	; 20B0 german penny
		UNISIG	"P"		; 20B1 peso
		UNISIG	"G"		; 20B2 guarani
		UNISIG	"A"		; 20B3 austral
		UNISIG	UNINOASC	; 20B4 hryvnia
		UNISIG	"C"		; 20B5 cedi
		UNIINV	26		; 20B6..20CF
		UNIMODNUM 32		; 20D0..20EF Combining Diacrit. Marks
		UNIINV	16		; 20F0..20FF

CHType2200:	UNISIG	UNINOASC	; 2200 for all
		UNISIG	"C"		; 2201 complement
		UNISIG	UNINOASC	; 2202 partial differential
		UNISIG	"E"		; 2203 there exists
		UNISIG	UNINOASC	; 2204 there does not exist
		UNISIG	"0"		; 2205 empty set
		UNISIGNUM 12		; 2206..2211 mathematical symbols
		UNISIG	"-"		; 2212 minus
		UNISIG	UNINOASC	; 2213 minus-or-plus
		UNISIG	"+"		; 2214 dot plus
		UNISIG	"/"		; 2215 division slash
		UNISIG	"\"		; 2216 set minus
		UNISIG	"*"		; 2217 asterisk
		UNISIG	"."		; 2218 ring operator
		UNISIG	"."		; 2219 bullet operator
		UNISIG	"V"		; 221A square root
		UNISIGNUM 8		; 221B..2222
		UNISIG	"|"		; 2223 divides
		UNISIG	UNINOASC	; 2224 does not divide
		UNISIG	"|"		; 2225 parallel to
		UNISIG	UNINOASC	; 2226 not parallel to
		UNISIGNUM 15		; 2227..2235
		UNISIG	":"		; 2236
		UNISIG	UNINOASC	; 2237 proportion
		UNISIG	"-"		; 2238 dot minus
		UNISIG	"-"		; 2239 excess
		UNISIG	"-"		; 223A geometric proportion
		UNISIG	"~"		; 223B homothetic
		UNISIG	"~"		; 223C tilde operator
		UNISIG	"~"		; 223D reversed tilde
		UNISIG	"~"		; 223E inverted lazy S
		UNISIG	"~"		; 223F sine wave
		UNISIG	"|"		; 2240 wreath product
		UNISIG	"~"		; 2241 not tilde
		UNISIG	"="		; 2242 minus tilde
		UNISIG	"="		; 2243 asymptotically equal to
		UNISIG	UNINOASC	; 2244 not asymptotically equal to
		UNISIG	"="		; 2245 approximately equal to
		UNISIG	UNINOASC	; 2246 approx. but not actually equal
		UNISIG	UNINOASC	; 2247 neither approx. not act. equal
		UNISIG	"~"		; 2248 almost equal to
		UNISIG	UNINOASC	; 2249 not almost equal to
		UNISIG	"~"		; 224A almost equal or equal to
		UNISIG	"~"		; 224B triple tilde
		UNISIG	"~"		; 224C all equal to
		UNISIG	"="		; 224D equivalent to
		UNISIG	UNINOASC	; 224E geometrically equivalent to
		UNISIG	"="		; 224F difference between
		UNISIG	"="		; 2250 difference between
		UNISIG	"="		; 2251 geometrically equal to
		UNISIG	"="		; 2252 approx. equal to or the image of
		UNISIG	"="		; 2253 image of or approx. equal to
		UNISIG	"="		; 2254 colon equals
		UNISIG	"="		; 2255 equals colon
		UNISIG	"="		; 2256 ring in equal to
		UNISIG	"="		; 2257 ring equal to
		UNISIG	"="		; 2258 corresponds to
		UNISIG	"="		; 2259 estimates
		UNISIG	"="		; 225A equiangular to
		UNISIG	"="		; 225B star equals
		UNISIG	"="		; 225C delta equal to
		UNISIG	"="		; 225D equal to by definition
		UNISIG	"="		; 225E measured by
		UNISIG	"="		; 225F questioned equal to
		UNISIG	UNINOASC	; 2260 not equal to
		UNISIG	"="		; 2261 identical to
		UNISIG	UNINOASC	; 2262 not identical to
		UNISIG	"="		; 2263 strictly equivalent to
		UNISIGNUM 6		; 2264..2269 less equal
		UNISIG	"<"		; 226A much less-than
		UNISIG	">"		; 226B much greater-than
		UNISIGNUM 14		; 226C..2279
		UNISIG	"<"		; 227A precedes
		UNISIG	">"		; 227B succeeds
		UNISIGNUM 25		; 227C..2294 operators
		UNISIG	"+"		; 2295 circled plus
		UNISIG	"-"		; 2296 circled minus
		UNISIG	"x"		; 2297 circled times
		UNISIG	"/"		; 2298 circled division slash
		UNISIG	"."		; 2299 circled dot operator
		UNISIG	"."		; 229A circled ring operator
		UNISIG	"*"		; 229B circled asterisk operator
		UNISIG	"="		; 229C circled equals
		UNISIG	"-"		; 229D circled dash
		UNISIG	"+"		; 229E squared plus
		UNISIG	"-"		; 229F squared minus
		UNISIG	"x"		; 22A0 squared times
		UNISIG	"."		; 22A1 squared dot operator
		UNISIGNUM 34		; 22A2..22C3 operators
		UNISIG	"."		; 22C4 diamong operator
		UNISIG	"."		; 22C5 dot operator
		UNISIG	"*"		; 22C6 star operator
		UNISIG	"*"		; 22C7 division times
		UNISIGNUM 14		; 22C8..22D5 operators
		UNISIG	"<"		; 22D6 less-than with dot
		UNISIG	">"		; 22D7 greater-than with dot
		UNISIG	"<"		; 22D8 very much less-than
		UNISIG	">"		; 22D9 very much greater-than
		UNISIGNUM 20		; 22DA..22ED compare
		UNISIG	":"		; 22EE vertical ellipsis
		UNISIG	"."		; 22EF midline horizontal ellipsis
;		UNISIGNUM 16		; 22F0..22FF operators
; CHType2400 must follow.

CHType2400:	UNISIGNUM 38		; 2400..2425 control symbols
		UNISIG	"?"		; 2426 symbol for substitute form two
		UNIINV	25		; 2427..243F
		UNISIGNUM 11		; 2440..244A optical character recogn.
		UNIINV	21		; 244B..245F
		UNIDIGCH "1",9		; 2460..2468 circled digit 1..9
		UNIDIGNUM 11		; 2469..2473 circled number 11..20
		UNIDIGCH "1",9		; 2474..247C parenthesized digit 1..9
		UNIDIGNUM 11		; 247D..2487 parenthes. number 11..20
		UNIDIGCH "1",9		; 2488..2490 digit 1..9 full stop
		UNIDIGNUM 11		; 2491..249B number 10..20 full stop
		UNISMACH "a",26		; 249C..24B5 parenthesized a..z
		UNICAPINXCH "A",24d0h,26 ; 24B6..24CF circled A..Z
		UNISMAINXCH "a",24b6h,26 ; 24D0..24E9 circled a..z
		UNIDIG	"0"		; 24EA circled digit 0
		UNIDIGNUM 10		; 24EB..24F4 negative circled 11..20
		UNIDIGCH "1",9		; 24F5..24FD double circled 1..9
		UNIDIG	UNINOASC	; 24FE double circled 10
		UNIDIG	"0"		; 24FF negative circled 0

CHType2500:	UNISIGCH "-",2		; 2500..2501 box drawings horizontal
		UNISIGCH "|",2		; 2502..2503 box drawings vertical
		UNISIGCH "-",2		; 2504..2505 box drawings triple dash
		UNISIGCH "|",2		; 2506..2507 box draw.triple dash vert.
		UNISIGCH "-",2		; 2508..2509 box drawings quadr. dash
		UNISIGCH "|",2		; 250A..250B box draw.quadr. dash vert.
		UNISIGCH "+",64		; 250C..254B box drawings
		UNISIGCH "-",2		; 254C..254D box draw. double dash hor.
		UNISIGCH "|",2		; 254E..254F box draw.double dash vert.
		UNISIG	"-"		; 2550 box drawings double horizontal
		UNISIG	"|"		; 2551 box drawings double vertical
		UNISIGCH "+",31		; 2552..2570 box drawings
		UNISIG	"/"		; 2571 box drawings diagonal UR-DL
		UNISIG	"\"		; 2571 box drawings diagonal UL-DR
		UNISIG	"+"		; 2573 box drawings diagonal cross
		UNISIG	"-"		; 2574 box drawings light left
		UNISIG	"|"		; 2575 box drawings light up
		UNISIG	"-"		; 2576 box drawings light right
		UNISIG	"|"		; 2577 box drawings light down
		UNISIG	"-"		; 2578 box drawings heavy left
		UNISIG	"|"		; 2579 box drawings heavy up
		UNISIG	"-"		; 257A box drawings heavy right
		UNISIG	"|"		; 257B box drawings heavy down
		UNISIG	"-"		; 257C box drawings light-heavy left
		UNISIG	"|"		; 257D box drawings light-heavy up
		UNISIG	"-"		; 257E box drawings light-heavy right
		UNISIG	"|"		; 257F box drawings light-heavy down
		UNISIGCH " ",4		; 2580..2583 block
		UNISIGCH "#",8		; 2584..258B block
		UNISIGCH " ",4		; 258C..258F block
		UNISIG	"#"		; 2590 right half block
		UNISIG	" "		; 2591 shade
		UNISIGCH "#",2		; 2592..2593 shade
		UNISIGCH " ",5		; 2594..2598 graphics
		UNISIGCH "#",4		; 2599..259C graphics
		UNISIGCH " ",2		; 259D..259E graphics
		UNISIGCH "#",2		; 259F..25A0 graphics, black square
		UNISIGCH " ",2		; 25A1..25A2 white square
		UNISIG	"."		; 25A3 white square black square
		UNISIG	"="		; 25A4 horizontal fill
		UNISIG	"|"		; 25A5 vertical fill
		UNISIG	"#"		; 25A6 crosshatch fill
		UNISIG	"\"		; 25A7 diagonal fill
		UNISIG	"/"		; 25A8 diagonal fill
		UNISIG	"X"		; 25A9 diagonal crosshatch fill
		UNISIG	"."		; 25AA black small square
		UNISIG	"."		; 25AB white small square
		UNISIG	"#"		; 25AC black rectangle
		UNISIG	" "		; 25AD white rectangle
		UNISIG	"#"		; 25AE black vertical rectangle
		UNISIG	" "		; 25AF white vertical rectangle
		UNISIG	"#"		; 25B0 black parallelogram
		UNISIG	" "		; 25B1 white parallelogram
		UNISIGCH "^",4		; 25B2..25B5 up-pointing triangle
		UNISIGCH ">",6		; 25B6..25BB right-pointing triangle
		UNISIGCH "v",4		; 25BC..25BF down-pointing triangle
		UNISIGCH "<",6		; 25C0..25C5 left-pointing triangle
		UNISIGCH "O",16		; 25C6..25D5 circle
		UNISIG	"C"		; 25D6 left half circle
		UNISIG	"D"		; 25D7 right half circle
		UNISIG	"."		; 25D8 inverse bullet
		UNISIG	"#"		; 25D9 inverse circle
		UNISIG	"~"		; 25DA upper half circle
		UNISIG	"_"		; 25DB lower half circle
		UNISIGCH " ",32		; 25DC..25FB graphics
		UNISIG	"#"		; 25FC black medium square
		UNISIG	" "		; 25FD white medium square
		UNISIG	"."		; 25FE black medium square
		UNISIG	" "		; 25FF lower right triangle

CHType2100:	UNISIG	UNINOASC	; 2100 account of
		UNISIG	UNINOASC	; 2101 addressed to the subject
		UNICAP	"C",0		; 2102 double-struck C
		UNISIG	"C"		; 2103 degree Celsius
		UNISIG	"C"		; 2104 centre line symbol
		UNISIG	UNINOASC	; 2105 care of
		UNISIG	UNINOASC	; 2106 cada una
		UNISIG	UNINOASC	; 2107 Euler constant
		UNISIG	UNINOASC	; 2108 scruple
		UNISIG	UNINOASC	; 2109 degree Fahrenheit
		UNISMA	"g",0		; 210A script g
		UNICAP	"H",0		; 210B script H
		UNICAP	"H",0		; 210C black-letter H
		UNICAP	"H",0		; 210D double-struck H
		UNISIG	"h"		; 210E Planck constant
		UNISIG	"h"		; 210F Planck constant over two pi
		UNICAP	"I",0		; 2110 script I
		UNICAP	"I",0		; 2111 black-letter I
		UNICAP	"L",0		; 2112 script L
		UNISMA	"l",0		; 2113 script l
		UNISIG	"#"		; 2114 LB bas symbol
		UNICAP	"N",0		; 2115 double-struck N
		UNISIG	"N"		; 2116 numero sign
		UNISIG	"P"		; 2117 sound recording copyright
		UNICAP	"P",0		; 2118 script P
		UNICAP	"P",0		; 2119 double-struck P
		UNICAP	"Q",0		; 211A double-struck Q
		UNICAP	"R",0		; 211B script R
		UNICAP	"R",0		; 211C black-letter R
		UNICAP	"R",0		; 211D double-struck R
		UNISIG	"R"		; 211E prescription take
		UNISIG	"R"		; 211F response
		UNISIG	"S"		; 2120 service mark
		UNISIG	"T"		; 2121 telephone sign
		UNISIG	"T"		; 2122 trade mark sign
		UNISIG	"V"		; 2123 versicle
		UNICAP	"Z",0		; 2124 double-struck Z
		UNISIG	UNINOASC	; 2125 ounce sign
		UNISIG	UNINOASC	; 2126 ohm sign
		UNISIG	UNINOASC	; 2127 inverted ohm sign
		UNICAP	"Z",0		; 2128 black-letter Z
		UNISMA	"i",0		; 2129 turned greek small letter iota
		UNISIG	"K"		; 212A kelvin sign
		UNISIG	"A"		; 212B angstrom sign
		UNICAP	"B",0		; 211C black-letter R
		UNICAP	"C",0		; 211D black-letter C
		UNISIG	"e"		; 212E estimated symbol
		UNISMA	"e",2130h	; 212F script e
		UNICAP	"E",212fh	; 2130 script E
		UNICAP	"F",0		; 2131 script F
		UNICAP	"F",0		; 2132 turned F
		UNICAP	"M",0		; 2133 script M
		UNISMA	"o",0		; 2134 script o
		UNISIG	UNINOASC	; 2135 alef symbol
		UNISIG	UNINOASC	; 2136 bet symbol
		UNISIG	UNINOASC	; 2137 gimel symbol
		UNISIG	UNINOASC	; 2138 dalet symbol
		UNISIG	"i"		; 2139 information source
		UNICAP	"O",0		; 213A rotated O
		UNISIG	"F"		; 213B facsimile sign
		UNISMA	UNINOASC,0	; 213C double-struck small pi
		UNISMA	"y",0		; 213D double-struck small gamma
		UNICAP	UNINOASC,0	; 213E double-struck capital gamma
		UNICAP	UNINOASC,0	; 213F double-struck capital pi
		UNISIG	UNINOASC	; 2140 double-struck N-ary summation
		UNICAP	"G",0		; 2141 turned sans-serif G
		UNICAP	"L",0		; 2142 turned sans-serif L
		UNICAP	"L",0		; 2143 reversed sans-serif L
		UNICAP	"Y",0		; 2144 turned sans-serif Y
		UNICAP	"D",2146h	; 2145 double-struck italic D
		UNISMA	"d",2145h	; 2146 double-struck italic d
		UNISMA	"e",0		; 2147 double-struck italic e
		UNISMA	"i",0		; 2148 double-struck italic i
		UNISMA	"j",0		; 2149 double-struck italic j
		UNISIG	"P"		; 214A property line
		UNISIG	"&"		; 214B turned ampersand
		UNISIG	UNINOASC	; 214C per sign
		UNISIG	UNINOASC	; 214D aktieselskab
		UNISMA	"f",0		; 214E turned f
		UNIINV	4		; 214F..2152
		UNISIGNUM 13		; 2153..215F fractions
		UNIDIG	"I"		; 2160 roman numeral one
		UNIDIG	UNINOASC	; 2161 roman numeral two
		UNIDIG	UNINOASC	; 2162 roman numeral three
		UNIDIG	UNINOASC	; 2163 roman numeral four
		UNIDIG	"V"		; 2164 roman numeral five
		UNIDIG	UNINOASC	; 2165 roman numeral six
		UNIDIG	UNINOASC	; 2166 roman numeral seven
		UNIDIG	UNINOASC	; 2167 roman numeral eight
		UNIDIG	UNINOASC	; 2168 roman numeral nine
		UNIDIG	"X"		; 2169 roman numeral ten
		UNIDIG	UNINOASC	; 216A roman numeral eleven
		UNIDIG	UNINOASC	; 216B roman numeral twelve
		UNIDIG	"L"		; 216C roman numeral fifty
		UNIDIG	"C"		; 216D roman numeral one hundred
		UNIDIG	"D"		; 216E roman numeral five hundred
		UNIDIG	"M"		; 216F roman numeral one thousand
		UNIDIG	"i"		; 2170 small roman numeral one
		UNIDIG	UNINOASC	; 2171 small roman numeral two
		UNIDIG	UNINOASC	; 2172 small roman numeral three
		UNIDIG	UNINOASC	; 2173 small roman numeral four
		UNIDIG	"v"		; 2174 small roman numeral five
		UNIDIG	UNINOASC	; 2175 small roman numeral six
		UNIDIG	UNINOASC	; 2176 small roman numeral seven
		UNIDIG	UNINOASC	; 2177 small roman numeral eight
		UNIDIG	UNINOASC	; 2178 small roman numeral nine
		UNIDIG	"x"		; 2179 small roman numeral ten
		UNIDIG	UNINOASC	; 217A small roman numeral eleven
		UNIDIG	UNINOASC	; 217B small roman numeral twelve
		UNIDIG	"l"		; 217C small roman numeral fifty
		UNIDIG	"c"		; 217D small roman numeral one hundred
		UNIDIG	"d"		; 217E small roman numeral five hundred
		UNIDIG	"m"		; 217F small roman numeral one thousand
		UNIDIG	UNINOASC	; 2180 roman numeral one thousand CD
		UNIDIG	UNINOASC	; 2181 roman numeral five thousand
		UNIDIG	UNINOASC	; 2182 roman numeral ten thousand CD
		UNIDIG	UNINOASC	; 2183 roman num. reversed one thousand
		UNISMA	"c",0		; 2184 reversed c
		UNIINV	11		; 2185..218F
;		UNISIGNUM 112		; 2190..21FF
; CHType2600 must follow.

CHType2600:	UNISIGNUM 157		; 2600..269C symbols
		UNIINV	3		; 269D..269F
		UNISIGNUM 19		; 26A0..26B2 symbols
		UNIINV	77		; 26B3..26FF

CHType2700:	UNIINV	1		; 2700
		UNISIGNUM 4		; 2701..2704 symbols
		UNIINV	1		; 2705
		UNISIGNUM 4		; 2706..2709 symbols
		UNIINV	2		; 270A..270B
		UNISIGNUM 28		; 270C..2727
		UNIINV	1		; 2728
		UNISIGNUM 35		; 2729..274B symbols
		UNIINV	1		; 274C
		UNISIGNUM 1		; 274D symbols
		UNIINV	1		; 274E
		UNISIGNUM 4		; 274F..2752 symbols
		UNIINV	3		; 2753..2755
		UNISIGNUM 1		; 2756
		UNIINV	1		; 2757
		UNISIGCH "|",3		; 2758..275A vertical bar
		UNIMRK	39		; 275B heavy single turned comma quot.
		UNIMRK	39		; 275C heavy single comma quotation
		UNIMRK	33		; 275D heavy double turned comma quot.
		UNIMRK	33		; 275E heavy double comma quotation
		UNIINV	2		; 275F..2760
		UNISIGNUM 7		; 2761..2767 symbols
		UNIMRK	"("		; 2768 medium left parenthesis
		UNIMRK	")"		; 2769 medium right parenthesis
		UNIMRK	"("		; 276A medium flatt. left parenthesis
		UNIMRK	")"		; 276B medium flatt. right parenthesis
		UNIMRK	"<"		; 276C medium left angle bracket
		UNIMRK	">"		; 276D medium right angle bracket
		UNIMRK	"<"		; 276E heavy left angle quotation mark
		UNIMRK	">"		; 276F heavy right angle quotation mark
		UNIMRK	"<"		; 2770 heavy left angle bracket
		UNIMRK	">"		; 2771 heavy right angle bracket
		UNIMRK	"["		; 2772 light left tortoise bracket
		UNIMRK	"]"		; 2773 light right tortoise bracket
		UNIMRK	"{"		; 2774 medium left curcly bracket
		UNIMRK	"}"		; 2775 medium right curcly bracket
		UNIDIGCH "1",9		; 2776..277E negative circled digit
		UNIDIG	UNINOASC	; 277F dingbat negative circled number
		UNIDIGCH "1",9		; 2780..2788 sans-serif circled digit
		UNIDIG	UNINOASC	; 2789 sans-serif circled number
		UNIDIGCH "1",9		; 278A..2792 neg. sans-serif circ.digit
		UNIDIG	UNINOASC	; 2793 neg. sans-serif circled number
		UNISIG	">"		; 2794 heavy wide-headed arrow
		UNIINV	3		; 2795..2797
		UNISIGNUM 24		; 2798..27AF arrow
		UNIINV	1		; 27B0
		UNISIGNUM 14		; 27B1..27BE arrow
		UNIINV	1		; 27BF
		UNISIGNUM 11		; 27C0..27CA symbols
		UNIINV	5		; 27CB..27CF
		UNISIGNUM 28		; 27D0..27EB
		UNIINV	4		; 27EC..27EF
;		UNISIGNUM 16		; 27F0..27FF symbols
; CHType2C00 must follow.

CHType2C00:	UNISIGNUM 96		; 2C00..2C5F glagolitic
		UNICAP	"L",2c61h	; 2C60 L double bar
		UNISMA	"l",2c60h	; 2C61 l double bar
		UNICAP	"L",26bh	; 2C62 L middle tilde
		UNICAP	"P",1d7dh	; 2C63 P stroke
		UNICAP	"R",27dh	; 2C64 R tail
		UNISMA	"a",23ah	; 2C65 a stroke
		UNISMA	"t",23eh	; 2C66 t diagonal stroke
		UNICAP	"H",2c68h	; 2C67 H descender
		UNISMA	"h",2c67h	; 2C68 h descender
		UNICAP	"K",2c6ah	; 2C69 K descender
		UNISMA	"k",2c69h	; 2C6A k descender
		UNICAP	"Z",2c6ch	; 2C6B Z descender
		UNISMA	"z",2c6bh	; 2C6C z descender
		UNIINV	7		; 2C6D..2C73
		UNISMA	"v",0		; 2C74 v curl
		UNICAP	"H",2c76h	; 2C75 half H
		UNISMA	"h",2c75h	; 2C76 half h
		UNISMA	UNINOASC,0	; 2C77 small tailess phi
		UNIINV	8		; 2C78..2C7F
		UNICAPSMA 2c80h,50	; 2C80..2CE3 coptic
		UNICAPNUM 7		; 2CE4..2CEA coptic symbol
		UNIINV	14		; 2CEB..2CF8
		UNICAPNUM 7		; 2CF9..2CFF coptic symbol

CHType2800:	; Braille ASCII standard,
		; http://en.wikipedia.org/wiki/Braille_ASCII
		; http://www.baraha.com/html_help/baraha/braille.htm
		; http://www.baraha.com/html_help/baraha/braille.htm
		; bana braille unicode

		UNISPC	1		; 2800 Braille pattern space
		UNICAP	"A",0		; 2801 Braille pattern A
		UNIDIG	"1"		; 2802 Braille pattern 1
		UNICAP	"B",0		; 2803 Braille pattern B
		UNIMRK	39		; 2804 Braille pattern '
		UNICAP	"K",0		; 2805 Braille pattern K
		UNIDIG	"2"		; 2806 Braille pattern 2
		UNICAP	"L",0		; 2807 Braille pattern L
		UNISIG	"@"		; 2808 Braille pattern @
		UNICAP	"C",0		; 2809 Braille pattern C
		UNICAP	"I",0		; 280A Braille pattern I
		UNICAP	"F",0		; 280B Braille pattern F
		UNISIG	"/"		; 280C Braille pattern "/"
		UNICAP	"M",0		; 280D Braille pattern M
		UNICAP	"S",0		; 280E Braille pattern S
		UNICAP	"P",0		; 280F Braille pattern P
		UNIMRK	33		; 2810 Braille pattern "
		UNICAP	"E",0		; 2811 Braille pattern E
		UNIDIG	"3"		; 2812 Braille pattern 3
		UNICAP	"H",0		; 2813 Braille pattern H
		UNIDIG	"9"		; 2814 Braille pattern 9
		UNICAP	"O",0		; 2815 Braille pattern O
		UNIDIG	"6"		; 2816 Braille pattern 6
		UNICAP	"R",0		; 2817 Braille pattern R
		UNIMRK	"^"		; 2818 Braille pattern ^
		UNICAP	"D",0		; 2819 Braille pattern D
		UNICAP	"J",0		; 281A Braille pattern J
		UNICAP	"G",0		; 281B Braille pattern G
		UNISIG	">"		; 281C Braille pattern >
		UNICAP	"N",0		; 281D Braille pattern N
		UNICAP	"T",0		; 281E Braille pattern T
		UNICAP	"Q",0		; 281F Braille pattern Q
		UNIMRK	","		; 2820 Braille pattern ,
		UNISIG	"*"		; 2821 Braille pattern *
		UNIDIG	"5"		; 2822 Braille pattern 5
		UNISIG	"<"		; 2823 Braille pattern <
		UNISIG	"-"		; 2824 Braille pattern -
		UNICAP	"U",0		; 2825 Braille pattern U
		UNIDIG	"8"		; 2826 Braille pattern 8
		UNICAP	"V",0		; 2827 Braille pattern V
		UNIMRK	"."		; 2828 Braille pattern .
		UNISIG	"%"		; 2829 Braille pattern %
		UNIMRK	"["		; 282A Braille pattern [
		UNISIG	"$"		; 282B Braille pattern $
		UNISIG	"+"		; 282C Braille pattern +
		UNICAP	"X",0		; 282D Braille pattern X
		UNIMRK	"!"		; 282E Braille pattern !
		UNISIG	"&"		; 282F Braille pattern &
		UNIMRK	";"		; 2830 Braille pattern ;
		UNIMRK	":"		; 2831 Braille pattern :
		UNIDIG	"4"		; 2832 Braille pattern 4
		UNISIG	"\"		; 2833 Braille pattern "\"
		UNIDIG	"0"		; 2834 Braille pattern 0
		UNICAP	"Z",0		; 2835 Braille pattern Z
		UNIDIG	"7"		; 2836 Braille pattern 7
		UNIMRK	"("		; 2837 Braille pattern (
		UNISIG	"_"		; 2838 Braille pattern _
		UNIMRK	"?"		; 2839 Braille pattern ?
		UNICAP	"W",0		; 283A Braille pattern W
		UNIMRK	"]"		; 283B Braille pattern ]
		UNISIG	"#"		; 283C Braille pattern #
		UNICAP	"Y",0		; 283D Braille pattern Y
		UNIMRK	")"		; 283E Braille pattern )
		UNISIG	"="		; 283F Braille pattern =
;		UNISIGNUM 192		; 2840..28FF Braille pattern
; CHType2A00 must follow.

CHType2900:
CHType2A00:
		UNISIGNUM 256-175	; 2900..297F supplemental arrows-B
					; 2980..29FF mathematical symbols-B
					; 2A00..2AFF mathematical operators
; CHType2300 must follow.

CHType2300:	UNISIGNUM 175		; 2300..23AE symbols
		UNISIG	"-"		; 23AF horizontal line extension
		UNISIGNUM 8		; 23B0..23B7 symbols
		UNISIG	"|"		; 23B8 left vertical box line
		UNISIG	"|"		; 23B9 right vertical box line
		UNISIG	"-"		; 23BA horizontal scan line-1
		UNISIG	"-"		; 23BB horizontal scan line-3
		UNISIG	"-"		; 23BC horizontal scan line-7
		UNISIG	"-"		; 23BD horizontal scan line-9
		UNISIGNUM 18		; 23BE..23CF symbols
		UNISIG	"|"		; 23D0 vertical line extension
		UNISIGNUM 23		; 23D1..23E7 symbols
		UNIINV	24		; 23E8..23FF

[list +]

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION
