; =============================================================================
;
;                         Litos - Format text string
;
; -----------------------------------------------------------------------------
; Public functions:
;	HexToTextBufN	- Length of formated HEX number
;	HexSToTextBuf	- Format HEX number into buffer (small letters)
;	HexCToTextBuf	- Format HEX number into buffer (capital letters)
;	BinToTextBufN	- Length of formated BIN number
;	BinToTextBuf	- Format BIN number into buffer
;	OctToTextBufN	- Length of formated OCT number
;	OctToTextBuf	- Format OCT number into buffer
;	UIntToTextBufN	- Length of formated unsigned INT number
;	UIntToTextBuf	- Format unsigned INT number into buffer
;	IntToTextBufN	- Length of formated signed INT number
;	IntToTextBuf	- Format signed INT number into buffer
;	ExpToTextBufN	- Length of exponential float number
;	ExpSToTextBuf	- Format exponential float number into buffer (small e)
;	ExpCToTextBuf	- Format exponential float num. into buffer (capital E)
;	FltToTextBufN	- Length of floating point number
;	FltToTextBuf	- Format floating point number into buffer
;	MixToTextBufN	- Length of mixed float number
;	MixSToTextBuf	- Format mixed float number into buffer (small e)
;	MixCToTextBuf	- Format mixed float number into buffer (capital e)
;	FormGetArgLen	- Get argument length in argument stack
;	FormToTextBufN	- Length of formated text
;	FormToTextBuf	- Format text into text buffer
;	FormDateTimeN	- Length of date/time text
;	FormDateTime	- Format date/time into text buffer
;	FormAbsDateTimeN- Length of absolute time text
;	FormAbsDateTime	- Format absolute time into text buffer
; -----------------------------------------------------------------------------
; Format string specification:
;
; Format string may contain both ordinary characters (which are simply printed
; out) and conversion characters (beginning with a percent symbol %).
;
;   %[flags][width][.[.]precision][size]type
;
; flags:
;	- Left align the result within the given field width. Default is right
;	  align.
;
;	+ Prefix the output value with a sign (+ or -) even if it is positive
;	  number. Default sign appears only for negative signed values.
;
;   space Prefix the output value with a space if the output value is signed
;	  and positive.
;
;	0 Zeros are added to the left until the minimum width is reached.
;
;	# When used with the o, O, x, X, b or B format, the # flag prefixes
;	  any nonzero output value with 0, 0x or 0b respectively. Default
;	  no prefix appears.
;
;	  When used with the e, E, f or F format, the # flag forces the output
;	  value to contain a decimal point in all cases. Default decimal point
;	  appears only if digits follow it.
;
; 	  When used with the g or G format, the # flag forces the output value
;	  to contain a decimal point in all cases and prevents the truncation
;	  of trailing zeros. Default decimal point appears only if digits
;	  follow it and trailing zeros are truncated.
;
;      ## When used with the o, O, x, X, b or B format, the ## flag suffixes
;	  any nonzero output value with o, h or b respectively. If number in
;	  x or X format begins with non-digit value (e.g. A to F) it prefixes
;	  output value with 0. Default no suffix appears.
;
;	  When used with the e, E, f or F format, the ## flag forces the output
;	  value to truncate trailing zeros. Default trailing zeros are not
;	  truncated.
;
; 	  When used with the g or G format, the ## flag forces the output value
;	  to contain a decimal point in all cases and truncate trailing zeros.
;	  Default decimal point appears only if digits follow it.
;
;     ### When used with the o, O, x, X, b or B format, the ### acts as
;	  combination of # and ## flags.
;
;	  When used with the e, E, f or F format, the ### flag forces the
;	  output value to contain a decimal point in all cases and to truncate
;	  trailing zeros. Default decimal point appears only if digits follow
;	  it and trailing zeros are not truncated.
;
; 	  When used with the g or G format, the ### flag forces the output
;	  value to don't truncate trailing zeros. Default trailing zeros are
;	  truncated.
;
;	~ Use thousand separator.
;
;	@ Center the result within the given field width.
;
; type:
;	c,C	int	One Unicode character.
;	s,S	text 	Text string (argument is pointer to TEXT variable).
;	d,D,i	int	Signed decimal integer.
;	u,U	int	Unsigned decimal integer.
;	b,B	int	Unsigned binary integer.
;	o,O	int	Unsigned octal integer.
;	x	int	Unsigned hexadecimal integer, using "abcdef".
;	X	int	Unsigned hexadecimal integer, using "ABCDEF".
;	f	double	Signed value having the form [-]dddd.dddd, where
;			dddd is one or more decimal digits. The number of
;			digits before the decimal point depends on the
;			magnitude of the number, and the number of digits
;			after the decimal point depends on the requested
;			precision.
;	F	double	Identical to the f format except it does not round.
;	e	double	Signed value having the form [-]d.dddd e[sign]ddd[d]
;			where d is a single decimal digit, dddd is one or more
;			decimal digits, ddd[d] is three or four decimal digits
;			depending on the output format and size of the
;			exponent, and sign is + or -.
;	E	double	Identical to the e format except that E rather than e
;			introduces the exponent and it does not round.
;	g	double	Signed value printed in f or e format, whichever is
;			more compact for the given value and precision.
;			The e format is used only when the exponent of the
;			value is less than -4 or greater than or equal to the
;			precision argument. Trailing zeros are truncated, and
;			the decimal point appears only if one or more digits
;			follow it.
;	G	double	Identical to the g format except that E rather than e
;			introduces the exponent (where appropriate) and it
;			does not round.
;	%		Print % character.
;
; size:
;	Specifies size of argument.
;	I16 h H		Short size - short int (16 bits), single float.
;	I32		Default size - int (32 bits), double float.
;	I64 l L		Long size - long int (64 bits), extended double float.
;
; width:
;	The width argument is a nonnegative decimal integer controlling the
;	minimum number of characters printed. If the number of characters
;	in the output value is less than the specified width, spaces are added
;	to the left or the right of the values - depending on align mode. If
;	0 prefix is entered, zeros are added to the left until the minimum
;	width is reached.
;
;	The width specification never causes a value to be truncated. If the
;	number of characters in the output value is greater than the specified
;	width, or if width is not given, all characters of the value are
;	printed.
;
;	If the width specification is an asterisk *, an int argument from
;	the argument list supplies the value. The width argument must precede
;	the value being formatted in the argument list.
;
; precision:
;	Precision specifies a nonnegative decimal integer, preceded by
;	a period ".". The precision specification can cause either truncation
;	of the output value or rounding of a floating-point value.
;
;	If the precision specification is an asterisk *, an int argument
;	from the argument list supplies the value. The precision argument
;	must precede the value being formatted in the argument list.
;
;	c,C	The precision has no effect.
;
;	s,S	The precision specifies the maximum number of characters to be
;		printed. Default or 0: all characters are printed.
;
;	d,D,i,u,U,o,O,x,X,b,B
;		The precision specifies the minimum number of digits to be
;		printed. If the number of digits in the argument is less than
;		precision, the output value is padded on the left with
;		zeros. The value is not truncated when the number of digits
;		exceeds precision. Default precision is 1.
;
;	f,F	The precision value specifies the number of digits after the
;		decimal point. Default precision is 6. If precision is 0,
;		or if the period "." appears without a number following it,
;		no decimal point is printed (except if flag # is specified).
;
;	e,E	The precision specifies the number of digits to be printed
;		after the decimal point. The last printed digit is rounded.
;		Default precision is 6. If precision is 0, or if the period
;		"." appears without a number following it, no decimal point
;		is printed (except if flag # is specified).
;
;	g,G	The precision specifies the maximum number of significant
;		digits printed. Default 6 significant digits are printed,
;		with any trailing zeros truncated.
;
; alternate precision "..":
;	Alternate precision is used when two periods ".." are entered before
;	precision value. It affects only floating-point formats.
;
;	f,F,e,E	The precision specifies the maximum number of significant
;		digits printed. Default 6 significant digits are printed.
;
;	g,G	The precision value specifies the number of digits after the
;		decimal point. Default precision is 6.
;
; index$:
;	Explicit format parameter index "index$" can order that parameter
;	with specific index is used instead of normal order of arguments.
;	Index is specified as number in range 1 to 255 followed by a "$"
;	character. It does not affect normal order of arguments. When it is
;	used after asterisk *, it specifies corresponding int argument index
;	(width or precision). In other cases it specifies index of data
;	argument. Explicit parameter index refers to index od DWORD in 
;	the argument stack, not to a real argument ordinal number. It must
;	take care in case od float numbers, which can be 2 or 3 DWORDs long.
;
; If the argument corresponding to a floating-point specifier is infinite,
; indefinite, or NAN, function gives the following output.
;
;	infinity ..... -1.#INF, +1.#INF
;	Indefinite ... -1.#IND, +1.#IND
;	NAN .......... -1.#NAN, +1.#NAN
; -----------------------------------------------------------------------------
; Date/time formatting string parameters:
;
; a	absolute day in year 1 to 366, number of characters = number of digits
; d	day of month 1 to 31 without a leading zero
; dd	day of month 01 to 31 with a leading zero
; ddd	short name of day of week (3 characters)
; dddd	full name of day of week
; f	seconds fraction, number of characters = number of digits
; F	seconds fraction, number of characters = number of digits,
;		without trailing zeros
; g	period or era (CE, BCE), short name
; gg	period or era (CE, BCE), long name
; h	hour 1 to 12, number of characters = number of digits
; H	hour 0 to 23, number of characters = number of digits
; j	Juliand date and time, number of characters = number of fract.digits+1
; J	Juliand date and time, number of characters = number of fract.digits+1,
;		without trailing zeros and with thousand separators
; m	minute 0 to 59, number of characters = number of digits
; M	month 1 to 12 without a leading zero
; MM	month 01 to 12 with a leading zero
; MMM	short name of month (3 characters)
; MMMM	full name of month
; MMMMM	full name of month, 1st grammatical case
; s 	seconds, number of characters = minimal number of digits
; t	short form of AM/PM (1 character)
; tt	full form of AM/PM
; w	week number 1 to 53, number of characters = minimal number of digits
; y	year, number of characters = number of digits 
; Y	signed year, number of characters = minimal number of digits 
; :	time separator
; /	date separator
; .	decimal separator
; ,	thousand separator
; ;	data entry separator
; " '	quoted string (not interpreted), ""=valid " character, ''=valid ' char.
;
; Parameters f,F,j and J does not round but only truncate the value.
; =============================================================================

EXP1TABBEG	EQU	150		; starting exponent of Exp1Tab
EXP1TABNUM	EQU	300		; number of exponents in Exp1Tab

		CODE_SECTION

; -----------------------------------------------------------------------------
;     Convert unformatted signed DWORD into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EAX = signed number
;		ECX = minimal text length (0...)
; OUTPUT:	EAX = text length (0...)
; -----------------------------------------------------------------------------

; ------------- Check if number is negative

SDWToTextBufN:	or	eax,eax		; is number negative?
		jns	short UDWToTextBufN ; number is not negative

; ------------- Push registers

		push	ebx		; push EBX

; ------------- Get number of bits in number (-> EBX, number is not 0)

		neg	eax		; EAX <- absolute value
		bsr	ebx,eax		; EBX <- highest bit in number
		inc	ebx		; EBX <- number of bits in number

; ------------- Get number of digits (ln 10/ln 2=3.3219, 10000h/3.3219=19729)

		imul	ebx,ebx,19729	; EBX <- approx. number of digits << 16
		shr	ebx,16		; EBX <- approx. number of digits
		cmp	[IntMul10+ebx*4],eax ; check number (0:0, 1:9, 2:99,..)
		xchg	eax,ebx		; EAX <- text length
		adc	al,1		; EAX <- number of digits + sign char

; ------------- Minimal text length

		cmp	eax,ecx		; check minimal text length
		ja	SDWToTextBufN4	; text length is OK
		mov	eax,ecx		; EAX <- limit minimal text length

; ------------- Pop registers

SDWToTextBufN4:	pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;    Convert unformatted unsigned DWORD into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EAX = unsigned number
;		ECX = minimal number of digits (0...)
; OUTPUT:	EAX = text length (0...)
; -----------------------------------------------------------------------------

; ------------- Push registers

UDWToTextBufN:	push	ebx		; push EBX

; ------------- Get number of bits in number (-> EBX)

		bsr	ebx,eax		; EBX <- highest bit in number
		jz	UDWToTextBufN2	; number is zero, no digit
		inc	ebx		; EBX <- number of bits in number

; ------------- Get number of digits (ln 10/ln 2=3.3219, 10000h/3.3219=19729)

		imul	ebx,ebx,19729	; EBX <- approx. number of digits << 16
		shr	ebx,16		; EBX <- approx. number of digits
		cmp	[IntMul10+ebx*4],eax ; check number (0:0, 1:9, 2:99,..)
		xchg	eax,ebx		; EAX <- text length
		adc	al,0		; EAX <- number of digits

; ------------- Minimal text length

UDWToTextBufN2:	cmp	eax,ecx		; check number of digits
		ja	UDWToTextBufN4	; number of digits is OK
		mov	eax,ecx		; EAX <- limit minimal number of digits

; ------------- Pop registers

UDWToTextBufN4:	pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;              Convert unformatted signed DWORD into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = signed number
;		ECX = minimal number of characters
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; -----------------------------------------------------------------------------

; ------------- Check if number is negative

SDWToTextBuf:	or	eax,eax		; is number negative?
		jns	short UDWToTextBuf ; number is not negative

; ------------- Store sign character into buffer

		or	esi,esi		; check free space
		jle	SDWToTextBuf9	; no free space left
		mov	byte [edi],"-"	; store sign character into buffer
		dec	esi		; decrease free space
		inc	edi		; increase destination pointer

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX

; ------------- Convert number into buffer

		neg	eax		; EAX <- absolute value
		sub	ecx,byte 1	; ECX <- text length without sign
		adc	ecx,byte 0	; ECX <- limit text length to 0
		call	UDWToTextBuf	; convert number to text

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	eax		; pop EAX
SDWToTextBuf9:	ret

; -----------------------------------------------------------------------------
;             Convert unformatted unsigned DWORD into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = unsigned number
;		ECX = minimal number of digits
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; -----------------------------------------------------------------------------

; ------------- Push registers

UDWToTextBuf:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Prepare registers

		mov	ebx,eax		; EBX <- number to convert

; ------------- Get text length (-> ECX)

		call	UDWToTextBufN	; get text length (-> EAX)
		xchg	eax,ecx		; ECX <- text length
		jecxz	UDWToTextBuf3	; no digit

; ------------- Shift buffer pointer

		add	edi,ecx		; EDI <- behind end of text
		sub	esi,ecx		; ESI <- decrease free space

; ------------- Push new end of text

		push	esi		; push ESI
		push	edi		; push EDI
		js	UDWToTextBuf4	; buffer is full

; ------------- Buffer is OK, use fast conversion

		std			; set direction down
		dec	edi		; shift to last character
UDWToTextBuf2:	mov	eax,3435973837	; EAX <- 800000000h/10, rounded up
		mul	ebx		; EDX <- (number*8) / 10
		shr	edx,3		; EDX <- number / 10
		mov	al,10		; AL <- 10
		mul	dl		; AL <- number / 10 * 10 low byte
		xchg	eax,ebx		; EAX <- number, EBX <- number low byte
		sub	al,bl		; AL <- number % 10
		add	al,"0"		; AL <- convert to ASCII character
		stosb			; store one character
		mov	ebx,edx		; ESI <- number / 10
		loop	UDWToTextBuf2	; next character
		cld			; set direction up

; ------------- Pop new end of text

		pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Pop registers

UDWToTextBuf3:	pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Buffer is full, convert carefully

UDWToTextBuf4:	mov	eax,3435973837	; EAX <- 800000000h/10, rounded up
		mul	ebx		; EDX <- (number*8) / 10
		shr	edx,3		; EDX <- number / 10
		mov	al,10		; AL <- 10
		mul	dl		; AL <- number / 10 * 10 low byte
		xchg	eax,ebx		; EAX <- number, EBX <- number low byte
		sub	al,bl		; AL <- number % 10
		dec	edi		; shift to last character
		add	al,"0"		; AL <- convert to ASCII character
		inc	esi		; increase remaining space
		jle	UDWToTextBuf6	; pointer is behind the text
		mov	[edi],al	; store character into buffer
UDWToTextBuf6:	mov	ebx,edx		; ESI <- number / 10
		loop	UDWToTextBuf4	; next character

; ------------- Pop new end of text

		pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Correct end of text

		add	edi,esi		; EDI <- correct text pointer
		xor	esi,esi		; ESI <- correct remaining free space

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;     Convert HEX number into text buffer - get text length without spaces
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

Hex0ToTextBufN:	push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Push number (for testing HEX sufix) (-> EDI:ESI)

		xchg	eax,esi		; ESI <- number LOW
		mov	edi,edx		; EDI <- number HIGH

; ------------- Get number of bits in number LOW (-> AL)

		bsr	eax,esi		; AL <- highest bit in number LOW
		setz	ah		; AH <- 1 if EAX == 0, 0 if EAX != 0
		add	al,1		; AL <- number of bits in number LOW
		dec	ah		; AH <- 0 if EAX == 0, 0ffh if EAX != 0
		and	al,ah		; clear AL if number LOW is zero

; ------------- Get number of bits in number HIGH + 32 (-> DL)

		bsr	edx,edi		; DL <- highest bit in number HIGH
		setz	dh		; DH <- 1 if EDX == 0, 0 if EDX != 0
		add	dl,32+1		; DL <- add HIGH correction
		dec	dh		; DH <- 0 if EDX == 0, 0ffh if EDX != 0
		and	dl,dh		; clear DL if number HIGH is zero

; ------------- Put together HIGH and LOW bits (-> DL)

		not	dh		; DH <- 0ffh if EDX == 0, 0 if EDX != 0
		and	al,dh		; clear AL if number HIGH is not zero
		or	dl,al		; DL <- number of bits in the number

; ------------- Get number of digits in the number (-> EDX)

		movzx	edx,dl		; EDX <- number of bits in the number
		add	dl,3		; round up
		shr	edx,2		; EDX <- number of digits in the number

; ------------- Leading zero for HEX suffix

		bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Hex0ToTextBufN1	; not using suffix
		push	ecx		; push ECX
		mov	cl,dl		; CL <- number of digits
		sub	cl,1		; CL <- index of last digit
		jc	Hex0ToTextBfN03	; no digit, there must be a digit
		shl	cl,2		; CL <- bit offset of last digit
		cmp	cl,32		; QWORD?
		jb	Hex0ToTextBfN02	; only DWORD
		mov	esi,edi		; ESI <- number HIGH
		sub	cl,32		; CL <- offset in number HIGH
Hex0ToTextBfN02:shr	esi,cl		; ESI <- last digit
		cmp	esi,0ah		; is it HEX number or character?
		cmc			; CY <- it is HEX character, add zero
Hex0ToTextBfN03:adc	dl,dh		; add leading zero
Hex0ToTextBfN04:pop	ecx		; pop ECX

; ------------- Limit minimal number of digits (-> EDX)

Hex0ToTextBufN1:cmp	dl,bl		; check minimal number of digits
		ja	Hex0ToTextBufN2	; number of digits is OK
		mov	dl,bl		; EDX <- minimal numer of digits

; ------------- Check if use zeros instead of spaces

Hex0ToTextBufN2:bt	ebx,FORMFLAG_Zero_b ; add zeros instead of spaces?
		jnc	Hex0ToTextBufN6	; no zeros

; ------------- Prepare width of field (-> EAX)

		movzx	eax,bh		; EAX <- minimal width of field
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	Hex0ToTextBufN3	; not using prefix
		sub	al,2		; without prefix
		jbe	Hex0ToTextBufN6	; width is low
Hex0ToTextBufN3:bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Hex0ToTextBufN4	; not using suffix
		sub	al,1		; without suffix
		jbe	Hex0ToTextBufN6	; width is low

; ------------- Check if use thousand separator

Hex0ToTextBufN4:bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Hex0ToTextBufN5	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EAX)

		inc	eax		; EAX <- width + 1
		imul	eax,eax,52429	; multiply with 10000h*4/5, round up
		shr	eax,16		; EAX <- max. number of digits

; ------------- New minimal number of digits

Hex0ToTextBufN5:cmp	eax,edx		; check number of digits
		jb	Hex0ToTextBufN6	; number of digits is not greater
		xchg	eax,edx		; EDX <- new text length

; ------------- Add thousand separator

Hex0ToTextBufN6:bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Hex0ToTextBufN7	; not using thousand separator
		mov	eax,edx		; EAX <- number of digits
		dec	eax		; without 1 digit
		jle	Hex0ToTextBufN7	; no digit
		shr	eax,2		; EAX <- number of separators
		add	edx,eax		; EDX <- add thousand separators

; ------------- Add prefix "0x"

Hex0ToTextBufN7:bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	Hex0ToTextBufN8	; not using prefix
		cmp	edx,byte 1	; check minimal number of digits
		adc	edx,byte 2	; EDX <- add 2 or 3 characters

; ------------- Add suffix "h"

Hex0ToTextBufN8:bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		adc	edx,byte 0	; EDX <- add suffix "h"

; ------------- Pop registers

Hex0ToTextBufN9:xchg	eax,edx		; EAX <- text length
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;              Convert HEX number into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

HexToTextBufN:	call	Hex0ToTextBufN	; get text length without spaces
		push	ebx		; push EBX
		movzx	ebx,bh		; EBX <- minimal width of field
		cmp	ebx,eax		; check text length
		jb	HexToTextBufN8	; text length is OK
		xchg	eax,ebx		; EAX <- new text length
HexToTextBufN8:	pop	ebx		; pop EBX
HexToTextBufN9:	ret

; -----------------------------------------------------------------------------
;             Convert HEX number into text buffer - small letters
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "8*digits+50" nanoseconds.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		HEXNumL    ebp+16	; (4) number LOW
%define		HEXNat     ebp+8	; (4) pointer to nationality NATIONAL
%define		HEXNumH    ebp+4	; (4) number HIGH
					; (3)
%define		HEXCap	   ebp-4	; (1) capital (7) or small (27h) flag
%define		HEXForm    ebp-8	; (4) formatting parameters FORMPAR
%define		HEXLen	   ebp-12	; (4) text length

%define		HEXStack  12		; stack size

; ------------- Macro - store one HEX digit
; Uses: EDX:EBX=number, ESI=free space-1, EDI=end of destination, [HEXCap]
; Destroys: AL

%macro		HEXTOTEXTBUFDIG 0

		inc	esi		; shift free space counter
		mov	al,bl		; EAX <- number
		jle	%%L2		; position is behind end of buffer
		and	al,0fh		; mask low nibble
		cmp	al,9		; is it HEX character?
		jbe	%%L1		; it is digit
		add	al,[HEXCap]	; add letter flag
%%L1:		add	al,"0"		; convert to ASCII character
		mov	[edi],al	; store character
%%L2:		shrd	ebx,edx,4	; shift number LOW 4 bits right
		dec	edi		; shift destination pointer
		shr	edx,4		; shift number HIGH 4 bits right
%endmacro

; ------------- Macro - store thousand separator
; Uses: ESI=free space-1, EDI=end of destination, EAX.byte2=thousand separator

%macro		TOTEXTBUFSEP 0

		inc	esi		; shift free space counter
		jle	%%L1		; position is behind end of buffer
		ror	eax,16		; AL <- thousand separator
		mov	[edi],al	; store character
		rol	eax,16		; return EAX
%%L1:		dec	edi		; shift destination pointer

%endmacro

; ------------- Check remaining free space in buffer

HexSToTextBuf:	or	esi,esi		; check remaining space
		jle	short HexToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,HEXStack	; ESP <- create local variables

; ------------- Prepare "capital" flag

		mov	byte [HEXCap],27h ; prepare "small" flag
		jmp	short HexToTextBuf1

; -----------------------------------------------------------------------------
;            Convert HEX number into text buffer - capital letters
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "8*digits+50" nanoseconds.
; -----------------------------------------------------------------------------

; ------------- Check remaining free space in buffer

HexCToTextBuf:	or	esi,esi		; check remaining space
		jle	short HexToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,HEXStack	; ESP <- create local variables

; ------------- Prepare "capital" flag

		mov	byte [HEXCap],7	; prepare "capital" flag

; ------------- Prepare local variables

HexToTextBuf1:	mov	[HEXForm],ebx	; formatting parameters FORMPAR

; ------------- Get text length without spaces (-> EDX)

		call	Hex0ToTextBufN	; get text length without spaces
		mov	[HEXLen],eax	; store text length
		xchg	eax,edx		; EDX <- push text length

; ------------- Store leading spaces

		bt	ebx,FORMFLAG_Left_b ; left-justify?
		jc	HexToTextBuf22	; left-justify, no leading spaces
		movzx	ecx,bh		; ECX <- minimal width of field
		sub	ecx,edx		; ECX <- remaining spaces
		jle	HexToTextBuf22	; no spaces left
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	HexToTextBuf2	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	HexToTextBuf22	; no spaces left
HexToTextBuf2:	mov	al," "		; AL <- space
HexToTextBuf21:	dec	esi		; decrease space counter
		stosb			; store one space
		jz	short HexToTextBuf24 ; buffer is full
		loop	HexToTextBuf21	; next character

; ------------- Store prefix

HexToTextBuf22:	mov	ecx,edx		; ECX <- text length
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	HexToTextBuf3	; not using prefix
		mov	al,"0"		; AL <- character "0"
		dec	esi		; decrease counter of free space
		stosb			; store first character
		jz	short HexToTextBuf24 ; buffer is full
		dec	ecx		; ECX <- text length - 1
		mov	al,"x"		; AL <- character "x"
		dec	esi		; decrease counter of free space
		stosb			; store second character
HexToTextBuf24:	jz	HexToTextBuf9	; buffer is full
		dec	ecx		; ECX <- text length - 1

; ------------- Suffix correction

HexToTextBuf3:	bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		sbb	ecx,byte 0	; ECX <- decrease if suffix

; ------------- Prepare registers to convert number

		mov	ebx,[HEXNumL]	; EBX <- number LOW
		add	edi,ecx		; EDI <- last character + 1
		mov	edx,[HEXNumH]	; EDX <- number HIGH
		sub	esi,ecx		; ESI <- subtract length
		or	ecx,ecx		; check text length
		jle	HexToTextBuf81	; no characters

; ------------- Push registers 2

		push	esi		; push ESI
		push	edi		; push EDI
		dec	edi		; EDI <- set pointer to last character

; ------------- Convert number without thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars

		bt	dword [HEXForm],FORMFLAG_Thsn_b; thousand separator?
		jc	HexToTextBuf5	; use thousand separator
HexToTextBuf42:	HEXTOTEXTBUFDIG		; store one digit into buffer
		loop	HexToTextBuf42	; next digit
		jmp	HexToTextBuf8

; ------------- Convert number with thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars,
; AH=order counter, EAX.byte2=thousand separator

HexToTextBuf5:	mov	eax,[HEXNat]	; EAX <- pointer to nationality
		mov	al,[eax+NAT_ThsndSep] ; AL <- thousand separator
		shl	eax,16		; shift character
		mov	ah,5		; AH <- 5, init order counter
HexToTextBuf52:	dec	ah		; count order counter
		jnz	HexToTextBuf55	; no thousand separator
		TOTEXTBUFSEP		; store thousand separator
		dec	ecx		; decrease character counter
		mov	ah,4		; AH <- 4, init order counter
		jz	HexToTextBuf8	; no next character
HexToTextBuf55:	HEXTOTEXTBUFDIG		; store one digit into buffer
		loop	HexToTextBuf52	; next digit

; ------------- Pop registers 2

HexToTextBuf8:	pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Limit offset

HexToTextBuf81:	or	esi,esi		; buffer overflow?
		jg	HexToTextBuf82	; buffer is OK
		add	edi,esi		; EDI <- correct address
		xor	esi,esi		; limit free space to 0
		jmp	short HexToTextBuf9

; ------------- Store suffix

HexToTextBuf82:	test	byte [HEXForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; suffix?
		jz	HexToTextBuf83	; not using suffix
		mov	al,"h"		; AL <- suffix
		stosb			; store suffix
		dec	esi		; decrease space counter
		jz	HexToTextBuf9	; no free space

; ------------- Store trailing spaces

HexToTextBuf83:	mov	eax,[HEXLen]	; EAX <- text length
		movzx	ecx,byte [HEXForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	HexToTextBuf9	; no spaces remain
		test	byte [HEXForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	HexToTextBuf84	; left-justify
		test	byte [HEXForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	HexToTextBuf9	; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	HexToTextBuf9	; no spaces
HexToTextBuf84:	mov	al," "		; AL <- space
HexToTextBuf85:	stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	HexToTextBuf85	; next character

; ------------- Pop registers

HexToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;      Convert BIN number into text buffer - get text length without spaces
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

Bin0ToTextBufN:	push	edx		; push EDX
		push	edi		; push EDI

; ------------- Get number of bits in number LOW (-> AL)

		bsr	eax,eax		; AL <- highest bit in number LOW
		setz	ah		; AH <- 1 if EAX == 0, 0 if EAX != 0
		add	al,1		; AL <- number of bits in number LOW
		dec	ah		; AH <- 0 if EAX == 0, 0ffh if EAX != 0
		and	al,ah		; clear AL if number LOW is zero

; ------------- Get number of bits in number HIGH + 32 (-> DL)

		bsr	edx,edx		; DL <- highest bit in number HIGH
		setz	dh		; DH <- 1 if EDX == 0, 0 if EDX != 0
		add	dl,32+1		; DL <- add HIGH correction
		dec	dh		; DH <- 0 if EDX == 0, 0ffh if EDX != 0
		and	dl,dh		; clear DL if number HIGH is zero

; ------------- Put together HIGH and LOW bits (-> DL)

		not	dh		; DH <- 0ffh if EDX == 0, 0 if EDX != 0
		and	al,dh		; clear AL if number HIGH is not zero
		or	dl,al		; DL <- number of bits in the number

; ------------- Get number of digits in the number (-> EDX)

		movzx	edx,dl		; EDX <- number of digits in the number

; ------------- Limit minimal number of digits (-> EDX)

		cmp	dl,bl		; check minimal number of digits
		ja	Bin0ToTextBufN2	; number of digits is OK
		mov	dl,bl		; EDX <- minimal numer of digits

; ------------- Check if use zeros instead of spaces

Bin0ToTextBufN2:bt	ebx,FORMFLAG_Zero_b ; add zeros instead of spaces?
		jnc	Bin0ToTextBufN6	; no zeros

; ------------- Prepare width of field (-> EAX)

		movzx	eax,bh		; EAX <- minimal width of field
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	Bin0ToTextBufN3	; not using prefix
		sub	al,2		; without prefix
		jbe	Bin0ToTextBufN6	; width is low
Bin0ToTextBufN3:bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Bin0ToTextBufN4	; not using suffix
		sub	al,1		; without suffix
		jbe	Bin0ToTextBufN6	; width is low

; ------------- Check if use thousand separator

Bin0ToTextBufN4:bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Bin0ToTextBufN5	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EAX)

		inc	eax		; EAX <- width + 1
		imul	eax,eax,58255	; multiply with 10000h*8/9, round up
		shr	eax,16		; EAX <- max. number of digits

; ------------- New minimal number of digits

Bin0ToTextBufN5:cmp	eax,edx		; check number of digits
		jb	Bin0ToTextBufN6	; number of digits is not greater
		xchg	eax,edx		; EDX <- new text length

; ------------- Add thousand separator

Bin0ToTextBufN6:bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Bin0ToTextBufN7	; not using thousand separator
		mov	eax,edx		; EAX <- number of digits
		dec	eax		; without 1 digit
		jle	Bin0ToTextBufN7	; no digit
		shr	eax,3		; EAX <- number of separators
		add	edx,eax		; EDX <- add thousand separators

; ------------- Add prefix "0b"

Bin0ToTextBufN7:bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	Bin0ToTextBufN8	; not using prefix
		cmp	edx,byte 1	; check minimal number of digits
		adc	edx,byte 2	; EDX <- add 2 or 3 characters

; ------------- Add suffix "b"

Bin0ToTextBufN8:bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Bin0ToTextBufN9	; not using suffix
		cmp	edx,byte 1	; check minimal number of digits
		adc	edx,byte 1	; EDX <- add 1 or 2 characters

; ------------- Pop registers

Bin0ToTextBufN9:xchg	eax,edx		; EAX <- text length
		pop	edi		; pop EDI
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;             Convert BIN number into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

BinToTextBufN:	call	Bin0ToTextBufN	; get text length without spaces
		push	ebx		; push EBX
		movzx	ebx,bh		; EBX <- minimal width of field
		cmp	ebx,eax		; check text length
		jb	BinToTextBufN8	; text length is OK
		xchg	eax,ebx		; EAX <- new text length
BinToTextBufN8:	pop	ebx		; pop EBX
BinToTextBufN9:	ret

; -----------------------------------------------------------------------------
;                    Convert BIN number into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "8*digits+45" nanoseconds.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		BINNumL    ebp+16	; (4) number LOW
%define		BINNat     ebp+8	; (4) pointer to nationality NATIONAL
%define		BINNumH    ebp+4	; (4) number HIGH
%define		BINForm    ebp-4	; (4) formatting parameters FORMPAR
%define		BINLen     ebp-8	; (4) text length

%define		BINStack  8		; stack size

; ------------- Macro - store one BIN digit
; Uses: EDX:EBX=number, ESI=free space-1, EDI=end of destination
; Destroys: AL

%macro		BINTOTEXTBUFDIG 0

		inc	esi		; shift free space counter
		mov	al,bl		; EAX <- number
		jle	%%L1		; position is behind end of buffer
		and	al,1		; mask low bit
		add	al,"0"		; convert to ASCII character
		mov	[edi],al	; store character
%%L1:		shr	edx,1		; shift number HIGH 1 bit right
                rcr	ebx,1		; shift number LOW 1 bit right
		dec	edi		; shift destination pointer
%endmacro

; ------------- Check remaining free space in buffer

BinToTextBuf:	or	esi,esi		; check remaining space
		jle	short BinToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,BINStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[BINForm],ebx	; formatting parameters FORMPAR

; ------------- Get text length without spaces (-> EDX)

		call	Bin0ToTextBufN	; get text length without spaces
		mov	[BINLen],eax	; store text length
		xchg	eax,edx		; EDX <- push text length

; ------------- Store leading spaces

		bt	ebx,FORMFLAG_Left_b ; left-justify?
		jc	BinToTextBuf22	; left-justify, no leading spaces
		movzx	ecx,bh		; ECX <- minimal width of field
		sub	ecx,edx		; ECX <- remaining spaces
		jle	BinToTextBuf22	; no spaces left
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	BinToTextBuf2	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	BinToTextBuf22	; no spaces left
BinToTextBuf2:	mov	al," "		; AL <- space
BinToTextBuf21:	dec	esi		; decrease space counter
		stosb			; store one space
		jz	short BinToTextBuf24 ; buffer is full
		loop	BinToTextBuf21	; next character

; ------------- Store prefix

BinToTextBuf22:	mov	ecx,edx		; ECX <- text length
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	BinToTextBuf3	; not using prefix
		mov	al,"0"		; AL <- character "0"
		dec	esi		; decrease counter of free space
		stosb			; store first character
		jz	short BinToTextBuf24 ; buffer is full
		dec	ecx		; ECX <- text length - 1
		mov	al,"b"		; AL <- character "b"
		dec	esi		; decrease counter of free space
		stosb			; store second character
BinToTextBuf24:	jz	BinToTextBuf9	; buffer is full
		dec	ecx		; ECX <- text length - 1

; ------------- Suffix correction

BinToTextBuf3:	bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		sbb	ecx,byte 0	; ECX <- decrease if suffix

; ------------- Prepare registers to convert number

		mov	ebx,[BINNumL]	; EBX <- number LOW
		add	edi,ecx		; EDI <- last character + 1
		mov	edx,[BINNumH]	; EDX <- number HIGH
		sub	esi,ecx		; ESI <- subtract length
		or	ecx,ecx		; check text length
		jle	BinToTextBuf81	; no characters

; ------------- Push registers 2

		push	esi		; push ESI
		push	edi		; push EDI
		dec	edi		; EDI <- set pointer to last character

; ------------- Convert number without thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars

		bt	dword [BINForm],FORMFLAG_Thsn_b; thousand separator?
		jc	BinToTextBuf5	; use thousand separator
BinToTextBuf42:	BINTOTEXTBUFDIG		; store one digit into buffer
		loop	BinToTextBuf42	; next digit
		jmp	BinToTextBuf8

; ------------- Convert number with thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars,
; AH=order counter, EAX.byte2=thousand separator

BinToTextBuf5:	mov	eax,[BINNat]	; EAX <- pointer to nationality
		mov	al,[eax+NAT_ThsndSep] ; AL <- thousand separator
		shl	eax,16		; shift character
		mov	ah,9		; AH <- 9, init order counter
BinToTextBuf52:	dec	ah		; count order counter
		jnz	BinToTextBuf55	; no thousand separator
		TOTEXTBUFSEP		; store 1-byte thousand separator
		dec	ecx		; decrease character counter
		mov	ah,8		; AH <- 8, init order counter
		jz	BinToTextBuf8	; no next character
BinToTextBuf55:	BINTOTEXTBUFDIG		; store one digit into buffer
		loop	BinToTextBuf52	; next digit

; ------------- Pop registers 2

BinToTextBuf8:	pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Limit offset

BinToTextBuf81:	or	esi,esi		; buffer overflow?
		jg	BinToTextBuf82	; buffer is OK
		add	edi,esi		; EDI <- correct address
		xor	esi,esi		; limit free space to 0
		jmp	short BinToTextBuf9

; ------------- Store suffix

BinToTextBuf82:	test	byte [BINForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; suffix?
		jz	BinToTextBuf83	; not using suffix
		mov	al,"b"		; AL <- suffix
		stosb			; store suffix
		dec	esi		; decrease space counter
		jz	BinToTextBuf9	; no free space

; ------------- Store trailing spaces

BinToTextBuf83:	mov	eax,[BINLen]	; EAX <- text length
		movzx	ecx,byte [BINForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	BinToTextBuf9	; no spaces remain
		test	byte [BINForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	BinToTextBuf84	; left-justify
		test	byte [BINForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	BinToTextBuf9	; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	BinToTextBuf9	; no spaces
BinToTextBuf84:	mov	al," "		; AL <- space
BinToTextBuf85:	stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	BinToTextBuf85	; next character

; ------------- Pop registers

BinToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;     Convert OCT number into text buffer - get text length without spaces
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

Oct0ToTextBufN:	push	edx		; push EDX
		push	edi		; push EDI

; ------------- Get number of bits in number LOW (-> AL)

		bsr	eax,eax		; AL <- highest bit in number LOW
		setz	ah		; AH <- 1 if EAX == 0, 0 if EAX != 0
		add	al,1		; AL <- number of bits in number LOW
		dec	ah		; AH <- 0 if EAX == 0, 0ffh if EAX != 0
		and	al,ah		; clear AL if number LOW is zero

; ------------- Get number of bits in number HIGH + 32 (-> DL)

		bsr	edx,edx		; DL <- highest bit in number HIGH
		setz	dh		; DH <- 1 if EDX == 0, 0 if EDX != 0
		add	dl,32+1		; DL <- add HIGH correction
		dec	dh		; DH <- 0 if EDX == 0, 0ffh if EDX != 0
		and	dl,dh		; clear DL if number HIGH is zero

; ------------- Put together HIGH and LOW bits (-> EDX)

		not	dh		; DH <- 0ffh if EDX == 0, 0 if EDX != 0
		and	al,dh		; clear AL if number HIGH is not zero
		or	dl,al		; DL <- number of bits in the number

; ------------- Get number of digits in the number (-> EDX)

		add	dl,2		; DL <- round up
		movzx	edx,dl		; EDX <- number of bits
		imul	edx,edx,21846	; EDX <- number of bits * 10000h/3
		shr	edx,16		; EDX <- number of digits

; ------------- Limit minimal number of digits (-> EDX)

		cmp	dl,bl		; check minimal number of digits
		ja	Oct0ToTextBufN2	; number of digits is OK
		mov	dl,bl		; EDX <- minimal numer of digits

; ------------- Check if use zeros instead of spaces

Oct0ToTextBufN2:bt	ebx,FORMFLAG_Zero_b ; add zeros instead of spaces?
		jnc	Oct0ToTextBufN6	; no zeros

; ------------- Prepare width of field (-> EAX)

		movzx	eax,bh		; EAX <- minimal width of field
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	Oct0ToTextBufN3	; not using prefix
		sub	al,1		; without prefix
		jbe	Oct0ToTextBufN6	; width is low
Oct0ToTextBufN3:bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Oct0ToTextBufN4	; not using suffix
		sub	al,1		; without suffix
		jbe	Oct0ToTextBufN6	; width is low

; ------------- New minimal number of digits

Oct0ToTextBufN4:cmp	eax,edx		; check number of digits
		jb	Oct0ToTextBufN6	; number of digits is not greater
		xchg	eax,edx		; EDX <- new text length

; ------------- Add prefix "0"

Oct0ToTextBufN6:bt	ebx,FORMFLAG_Alt_b ; use prefix?
		adc	edx,byte 0	; EDX <- add prefix "0"

; ------------- Add suffix "o"

		bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		jnc	Oct0ToTextBufN9	; not using suffix
		cmp	edx,byte 1	; check minimal number of digits
		adc	edx,byte 1	; EDX <- add 1 or 2 characters

; ------------- Pop registers

Oct0ToTextBufN9:xchg	eax,edx		; EAX <- text size in characters
		pop	edi		; pop EDI
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;              Convert OCT number into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

OctToTextBufN:	call	Oct0ToTextBufN	; get text length without spaces
		push	ebx		; push EBX
		movzx	ebx,bh		; EBX <- minimal width of field
		cmp	ebx,eax		; check text length
		jb	OctToTextBufN8	; text length is OK
		xchg	eax,ebx		; EAX <- new text length
OctToTextBufN8:	pop	ebx		; pop EBX
OctToTextBufN9:	ret

; -----------------------------------------------------------------------------
;                    Convert OCT number into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "6*digits+45" nanoseconds.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		OCTNumL    ebp+16	; (4) number LOW
%define		OCTNumH    ebp+4	; (4) number HIGH
%define		OCTForm    ebp-4	; (4) formatting parameters FORMPAR
%define		OCTLen     ebp-8	; (4) text length

%define		OCTStack  8		; stack size

; ------------- Check remaining free space in buffer

OctToTextBuf:	or	esi,esi		; check remaining space
		jle	short OctToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,OCTStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[OCTForm],ebx	; formatting parameters FORMPAR

; ------------- Get text length without spaces (-> EDX)

		call	Oct0ToTextBufN	; get text length without spaces
		mov	[OCTLen],eax	; store text length
		xchg	eax,edx		; EDX <- push text length

; ------------- Store leading spaces

		bt	ebx,FORMFLAG_Left_b ; left-justify?
		jc	OctToTextBuf22	; left-justify, no leading spaces
		movzx	ecx,bh		; ECX <- minimal width of field
		sub	ecx,edx		; ECX <- remaining spaces
		jle	OctToTextBuf22	; no spaces left
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	OctToTextBuf2	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	OctToTextBuf22	; no spaces left
OctToTextBuf2:	mov	al," "		; AL <- space
OctToTextBuf21:	dec	esi		; decrease space counter
		stosb			; store one space
		jz	short OctToTextBuf9 ; buffer is full
		loop	OctToTextBuf21	; next character

; ------------- Store prefix

OctToTextBuf22:	mov	ecx,edx		; ECX <- length
		bt	ebx,FORMFLAG_Alt_b ; use prefix?
		jnc	OctToTextBuf3	; not using prefix
		mov	al,"0"		; AL <- character "0"
		dec	esi		; decrease counter of free space
		stosb			; store prefix character
		jz	short OctToTextBuf9 ; buffer is full
		dec	ecx		; ECX <- text length - 1

; ------------- Suffix correction

OctToTextBuf3:	bt	ebx,FORMFLAG_Alt2_b ; use suffix?
		sbb	ecx,byte 0	; ECX <- decrease if suffix

; ------------- Prepare registers to convert number

		mov	ebx,[OCTNumL]	; EBX <- number LOW
		add	edi,ecx		; EDI <- last character + 1
		mov	edx,[OCTNumH]	; EDX <- number HIGH
		sub	esi,ecx		; ESI <- subtract length
		or	ecx,ecx		; check text length
		jle	OctToTextBuf81	; no characters

; ------------- Push registers 2

		push	esi		; push ESI
		push	edi		; push EDI
		dec	edi		; EDI <- set pointer to last character

; ------------- Convert number
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars

OctToTextBuf42:	inc	esi		; shift free space counter
		mov	al,bl		; EAX <- number
		jle	OctToTextBuf44	; position is behind end of buffer
		and	al,7		; mask low bits
		add	al,"0"		; convert to ASCII character
		mov	[edi],al	; store character
OctToTextBuf44:	shrd	ebx,edx,3	; shift number LOW 3 bits right
		dec	edi		; shift destination pointer
		shr	edx,3		; shift number HIGH 3 bits right
		loop	OctToTextBuf42	; next digit

; ------------- Pop registers 2

OctToTextBuf8:	pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Limit offset

OctToTextBuf81:	or	esi,esi		; buffer overflow?
		jg	OctToTextBuf82	; buffer is OK
		add	edi,esi		; EDI <- correct address
		xor	esi,esi		; limit free space to 0
		jmp	short OctToTextBuf9

; ------------- Store suffix

OctToTextBuf82:	test	byte [OCTForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; suffix?
		jz	OctToTextBuf83	; not using suffix
		mov	al,"o"		; AL <- suffix
		stosb			; store suffix
		dec	esi		; decrease space counter
		jz	OctToTextBuf9	; no free space

; ------------- Store trailing spaces

OctToTextBuf83:	mov	eax,[OCTLen]	; EAX <- text length
		movzx	ecx,byte [OCTForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	OctToTextBuf9	; no spaces remain
		test	byte [OCTForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	OctToTextBuf84	; left-justify
		test	byte [OCTForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	OctToTextBuf9	; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	OctToTextBuf9	; no spaces
OctToTextBuf84:	mov	al," "		; AL <- space
OctToTextBuf85:	stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	OctToTextBuf85	; next character

; ------------- Pop registers

OctToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
; Convert unsigned INT number into text buffer - get text length without spaces
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

UInt0ToTextBufN:push	edx		; push EDX
		push	edi		; push EDI
		xchg	eax,edi		; EDI <- number LOW

; ------------- Check if number is QWORD

		or	edx,edx		; is number QWORD?
		jnz	UInt0ToTextBfN1	; number is QWORD

; ------------- Get number of bits in number LOW (-> AL)

		bsr	eax,edi		; AL <- highest bit in number LOW
		setz	ah		; AH <- 1 if EAX == 0, 0 if EAX != 0
		add	al,1		; AL <- number of bits in number LOW
		dec	ah		; AH <- 0 if EAX == 0, 0ffh if EAX != 0
		and	al,ah		; clear AL if number LOW is zero

; ------------- Get number of digits (-> EDX) (ln 10/ln 2 = 3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	edx,ah		; EDX <- aprox. number of digits
		cmp	[IntMul10+edx*4],edi ; check number LOW
		adc	dl,dh		; EDX <- number of digits
		jmp	UInt0ToTextBfN3

; ------------- Get number of bits in number HIGH (-> AL)

UInt0ToTextBfN1:bsr	eax,edx		; EAX <- highest bit in number HIGH
		add	al,32+1		; AL <- number of bits

; ------------- Get number of digits (-> EDX) (ln 10/ln 2 = 3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	eax,ah		; EAX <- aprox. number of digits
		sub	edi,[Int2Mul10+eax*8-10*8] ; check number LOW
		sbb	edx,[Int2Mul10+eax*8+4-10*8] ; check number HIGH
		cmc			; CY = number is not less
		adc	al,ah		; EAX <- number of digits
		xchg	eax,edx		; EDX <- number of digits

; ------------- Limit minimal number of digits (-> EDX)

UInt0ToTextBfN3:cmp	dl,bl		; check minimal number of digits
		ja	UInt0ToTextBfN4	; number of digits is OK
		mov	dl,bl		; EDX <- minimal numer of digits

; ------------- Check if use zeros instead of spaces

UInt0ToTextBfN4:bt	ebx,FORMFLAG_Zero_b ; add zeros instead of spaces?
		jnc	UInt0ToTextBfN8	; no zeros

; ------------- Prepare width of field (-> EAX)

		movzx	eax,bh		; EAX <- minimal width of field

; ------------- Check if use thousand separator

		bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	UInt0ToTextBfN7	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EAX)

		inc	eax		; EAX <- width + 1
		imul	eax,eax,3	; multiply with 3
		shr	eax,2		; EAX <- /4, max. number of digits

; ------------- New minimal number of digits

UInt0ToTextBfN7:cmp	eax,edx		; check number of digits
		jb	UInt0ToTextBfN8	; number of digits is not greater
		xchg	eax,edx		; EDX <- new text size

; ------------- Add thousand separator

UInt0ToTextBfN8:bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	UInt0ToTextBfN9	; not using thousand separator
		mov	eax,edx		; EAX <- number of digits
		dec	eax		; without 1 digit
		jle	UInt0ToTextBfN9	; no digit
		imul	eax,eax,21846	; EAX <- digits*10000h/3
		shr	eax,16		; EAX <- number of separators
		add	edx,eax		; EDX <- add separators in characters

; ------------- Pop registers

UInt0ToTextBfN9:xchg	eax,edx		; EAX <- text length
		pop	edi		; pop EDI
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;          Convert unsigned INT number into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

UIntToTextBufN:	call	UInt0ToTextBufN	; get text length without spaces
		push	ebx		; push EBX
		movzx	ebx,bh		; EBX <- minimal width of field
		cmp	ebx,eax		; check text length
		jb	UIntToTextBufN8	; text length is OK
		xchg	eax,ebx		; EAX <- new text length
UIntToTextBufN8:pop	ebx		; pop EBX
UIntToTextBufN9:ret

; -----------------------------------------------------------------------------
;                  Convert unsigned INT number into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "10*digits+40" nanosecs for
;		0 to 10	digits, "50*digits-360" nanosecs for 10 to 20 digits.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		UINTNumL   ebp+16	; (4) number LOW
%define		UINTNat    ebp+8	; (4) pointer to nationality NATIONAL
%define		UINTNumH   ebp+4	; (4) number HIGH
%define		UINTForm   ebp-4	; (4) formatting parameters FORMPAR
%define		UINTLen    ebp-8	; (4) text length
%define		UINT10Div  ebp-12	; (4) divider 10
					; (2)
%define		UINTThsnd  ebp-15	; (1) thousand separator
%define		UINTThsN   ebp-16	; (1) thousand separator counter

%define		UINTStack  16		; stack size

; ------------- Macro - decode and store one INT digit QWORD
; Uses: EDX:EBX=number, ESI=free space-1, EDI=end of destination, [UINT10Div]
; Destroys: EAX

%macro		INTTOTEXTBUFDIQ 0

		xor	eax,eax		; EAX <- 0
		xchg	eax,edx		; EAX <- number HIGH, EDX <- 0
		div	dword [UINT10Div] ; EAX <- number HIGH/10, EDX <- rest
		xchg	eax,ebx		; EBX <- number HIGH/10, EAX <- LOW
		div	dword [UINT10Div] ; EAX <- new LOW, EDX <- number % 10
		inc	esi		; shift free space counter
		xchg	eax,edx		; EDX <- new LOW, EAX <- number % 10
		jle	%%L1		; position is behind end of buffer
		add	al,"0"		; AL <- convert to ASCII character
		mov	[edi],al	; store character
%%L1:		xchg	edx,ebx		; EBX <- new LOW, EDX <- new HIGH
		dec	edi		; shift destination pointer
%endmacro

; ------------- Macro - decode and store one INT digit DWORD
; Uses: EBX=number, ESI=free space-1, EDI=end of destination
; Destroys: EAX, EDX

%macro		INTTOTEXTBUFDID 0

		mov	eax,3435973837	; EAX <- 800000000h/10, rounded up
		mul	ebx		; EDX <- (number*8) / 10
		shr	edx,3		; EDX <- number / 10
		mov	al,10		; AL <- 10
		mul	dl		; AL <- number / 10 * 10 low byte
		sub	bl,al		; BL <- number % 10
		inc	esi		; shift free space counter
		jle	%%L1		; position is behind end of buffer
		add	bl,"0"		; BL <- convert to ASCII character
		mov	[edi],bl	; store character
%%L1:		xchg	edx,ebx		; EBX <- number / 10
		dec	edi		; shift destination pointer
%endmacro

; ------------- Macro - store thousand separator
; Uses: ESI=free space-1, EDI=end of destination, ECX=number of characters,
; [UINTThsN], [UINTThsnd], Destroys: AL

%macro		INTTOTEXTBUFSEP 0

		dec	byte [UINTThsN]	; count order counter
		jnz	%%L2		; no thousand separator
		inc	esi		; shift free space counter
		jle	%%L1		; position is behind end of buffer
		mov	al,[UINTThsnd]	; AL <- thousand separator
		mov	[edi],al	; store character
%%L1:		dec	edi		; shift destination pointer
		dec	ecx		; decrease character counter
		mov	byte [UINTThsN],3 ; init order counter
		jz	UIntToTextBuf8	; no next character
%%L2:

%endmacro

; ------------- Check remaining free space in buffer

UIntToTextBuf:	or	esi,esi		; check remaining space
		jle	short UIntToTextBufN9 ; not enough free space

; ------------- Push registers (it must agree with IntToTextBuf)

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,UINTStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[UINTForm],ebx	; formatting parameters FORMPAR

; ------------- Get text length without spaces (-> EDX)

		call	UInt0ToTextBufN	; get text length without spaces
		mov	[UINTLen],eax	; store text length
		xchg	eax,edx		; EDX <- push text length

; ------------- Store leading spaces

		bt	ebx,FORMFLAG_Left_b ; left-justify?
		jc	UIntToTextBuf3	; left-justify, no leading spaces
		movzx	ecx,bh		; ECX <- minimal width of field
		sub	ecx,edx		; ECX <- remaining spaces
		jle	UIntToTextBuf3	; no spaces left
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	UIntToTextBuf2	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	UIntToTextBuf3	; no spaces left
UIntToTextBuf2:	mov	al," "		; AL <- space
UIntToTextBuf21:dec	esi		; decrease space counter
		stosb			; store one space
		jz	UIntToTextBuf9	; buffer is full
		loop	UIntToTextBuf21	; next character

; ------------- Prepare registers to convert number (here jumps IntToTextBuf)

UIntToTextBuf3:	mov	ecx,edx		; ECX <- length in characters
		mov	ebx,[UINTNumL]	; EBX <- number LOW
UIntToTextBuf32:add	edi,ecx		; EDI <- last character + 1
		mov	edx,[UINTNumH]	; EDX <- number HIGH
		sub	esi,ecx		; ESI <- subtract length
		or	ecx,ecx		; check text length
		jle	UIntToTextBuf81	; no characters

; ------------- Push registers 2

		push	esi		; push ESI
		push	edi		; push EDI
		dec	edi		; EDI <- set pointer to last character

; ------------- Convert number without thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars

		bt	dword [UINTForm],FORMFLAG_Thsn_b; thousand separator?
		jc	UIntToTextBuf5	; use thousand separator

; ------------- Prepare registers 2

UIntToTextBuf4:	push	ebp		; push EBP
		mov	ebp,10		; EBP <- 10, divider

; ------------- Check if number is DWORD

UIntToTextBuf42:or	edx,edx		; is number QWORD?
		jz	UIntToTextBuf44	; number is DWORD

; ------------- Decode digits as QWORD without separator

		xor	eax,eax		; EAX <- 0
		xchg	eax,edx		; EAX <- number HIGH, EDX <- 0
		div	ebp		; EAX <- number HIGH/10, EDX <- rest
		xchg	eax,ebx		; EBX <- number HIGH/10, EAX <- LOW
		div	ebp		; EAX <- new LOW, EDX <- number % 10
		inc	esi		; shift free space counter
		xchg	eax,edx		; EDX <- new LOW, EAX <- number % 10
		jle	UIntToTextBuf43	; position is behind end of buffer
		add	al,"0"		; AL <- convert to ASCII character
		mov	[edi],al	; store character
UIntToTextBuf43:xchg	edx,ebx		; EBX <- new LOW, EDX <- new HIGH
		dec	edi		; shift destination pointer
		loop	UIntToTextBuf42	; next digit
		pop	ebp		; pop EBP
		jmp	UIntToTextBuf8

; ------------- Decode digits as DWORD without separator

UIntToTextBuf44:INTTOTEXTBUFDID		; decode and store one INT digit DWORD
		loop	UIntToTextBuf44	; next digit
		pop	ebp		; pop EBP
		jmp	UIntToTextBuf8

; ------------- Prepare to convert number with thousand separator
; EDX:EBX=number, ESI=free space-1, EDI=end of destination, ECX=num. of chars

UIntToTextBuf5:	mov	eax,[UINTNat]	; EAX <- pointer to nationality
		mov	dword [UINT10Div],10 ; prepare divider
		mov	al,[eax+NAT_ThsndSep] ; AL <- thousand separator
		mov	byte [UINTThsN],4 ; prepare thousand separator counter
		mov	[UINTThsnd],al	; prepare thousand separator

; ------------- Check if number is DWORD

UIntToTextBuf51:or	edx,edx		; is number QWORD?
		jz	UIntToTextBuf55	; number is DWORD

; ------------- Separator

		INTTOTEXTBUFSEP		; store thousand separator

; ------------- Decode digits as QWORD

		INTTOTEXTBUFDIQ		; decode and store one INT digit QWORD
		loop	UIntToTextBuf51	; next digit
		jmp	UIntToTextBuf8

; ------------- Separator

UIntToTextBuf55:INTTOTEXTBUFSEP		; store thousand separator

; ------------- Decode digits as DWORD

		INTTOTEXTBUFDID		; decode and store one INT digit DWORD
		loop	UIntToTextBuf55	; next digit

; ------------- Pop registers 2

UIntToTextBuf8:	pop	edi		; pop EDI
		pop	esi		; pop ESI

; ------------- Limit offset

UIntToTextBuf81:or	esi,esi		; buffer overflow?
		jg	UIntToTextBuf82	; buffer is OK
		add	edi,esi		; EDI <- correct address
		xor	esi,esi		; limit free space to 0
		jmp	short UIntToTextBuf9

; ------------- Store trailing spaces

UIntToTextBuf82:mov	eax,[UINTLen]	; EAX <- text length
		movzx	ecx,byte [UINTForm+FORMPAR_Width] ; ECX <- min. width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	UIntToTextBuf9	; no spaces remain
		test	byte [UINTForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	UIntToTextBuf84	; left-justify
		test	byte [UINTForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	UIntToTextBuf9	; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	UIntToTextBuf9	; no spaces
UIntToTextBuf84:mov	al," "		; AL <- space
UIntToTextBuf85:stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	UIntToTextBuf85	; next character

; ------------- Pop registers

UIntToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;  Convert signed INT number into text buffer - get text length without spaces
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

Int0ToTextBufN:	push	edx		; push EDX
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Always use sign

		xor	ebp,ebp		; EBP <- 0, no sign characters
		bt	ebx,FORMFLAG_Spc_b ; add space? (CY = yes)
		jc	Int0ToTextBfN1	; add space
		bt	ebx,FORMFLAG_Sign_b ; always use sign? (CY = yes)
Int0ToTextBfN1:	adc	ebp,ebp		; EBP <- 1 if always use sign

; ------------- Negative number

		or	edx,edx		; negative number?
		jns	Int0ToTextBfN2	; not negative number
		neg	eax		; EAX <- negative value LOW
		adc	edx,byte 0	; carry
		or	ebp,1		; length = 1
		neg	edx		; EDX <- negative number HIGH

; ------------- Check if number is QWORD (here is ZY if EDX = 0)

Int0ToTextBfN2:	xchg	eax,edi		; EDI <- number LOW
		jnz	Int0ToTextBfN3	; number is QWORD

; ------------- Get number of bits in number LOW (-> AL)

		bsr	eax,edi		; AL <- highest bit in number LOW
		setz	ah		; AH <- 1 if EAX == 0, 0 if EAX != 0
		add	al,1		; AL <- number of bits in number LOW
		dec	ah		; AH <- 0 if EAX == 0, 0ffh if EAX != 0
		and	al,ah		; clear AL if number LOW is zero

; ------------- Get number of digits (-> EDX) (ln 10/ln 2=3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	edx,ah		; EDX <- aprox. number of digits
		cmp	[IntMul10+edx*4],edi ; check number LOW
		adc	dl,dh		; EDX <- number of digits
		jmp	Int0ToTextBfN4

; ------------- Get number of bits in number HIGH (-> AL)

Int0ToTextBfN3:	bsr	eax,edx		; EAX <- highest bit in number HIGH
		add	al,32+1		; AL <- number of bits

; ------------- Get number of digits (-> EDX) (ln 10/ln 2=3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	eax,ah		; EAX <- aprox. number of digits
		sub	edi,[Int2Mul10+eax*8-10*8] ; check number LOW
		sbb	edx,[Int2Mul10+eax*8+4-10*8] ; check number HIGH
		cmc			; CY = number is not less
		adc	al,ah		; EAX <- number of digits
		xchg	eax,edx		; EDX <- number of digits

; ------------- Limit minimal number of digits (-> EDX)

Int0ToTextBfN4:	cmp	dl,bl		; check minimal number of digits
		ja	Int0ToTextBfN5	; number of digits is OK
		mov	dl,bl		; EDX <- minimal numer of digits

; ------------- Check if use zeros instead of spaces

Int0ToTextBfN5:	bt	ebx,FORMFLAG_Zero_b ; add zeros instead of spaces?
		jnc	Int0ToTextBfN7	; no zeros

; ------------- Prepare width of field (-> EAX)

		movzx	eax,bh		; EAX <- minimal width of field
		sub	eax,ebp		; subtract sign
		jbe	Int0ToTextBfN7	; width is low

; ------------- Check if use thousand separator

		bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Int0ToTextBfN6	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EAX)

		inc	eax		; EAX <- width + 1
		imul	eax,eax,3	; multiply with 3
		shr	eax,2		; EAX <- /4, max. number of digits

; ------------- New minimal number of digits

Int0ToTextBfN6:	cmp	eax,edx		; check number of digits
		jb	Int0ToTextBfN7	; number of digits is not greater
		xchg	eax,edx		; EDX <- new text size

; ------------- Add thousand separator

Int0ToTextBfN7:	bt	ebx,FORMFLAG_Thsn_b ; use thousand separator?
		jnc	Int0ToTextBfN9	; not using thousand separator
		mov	eax,edx		; EAX <- number of digits
		dec	eax		; without 1 digit
		jle	Int0ToTextBfN9	; no digit
		imul	eax,eax,21846	; EAX <- digits*10000h/3
		shr	eax,16		; EAX <- number of separators
		add	edx,eax		; EDX <- add separators in characters

; ------------- Add sign

Int0ToTextBfN9:	add	edx,ebp		; EDX <- new number of bytes

; ------------- Pop registers

		xchg	eax,edx		; EAX <- text size in characters
		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;          Convert signed INT number into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; -----------------------------------------------------------------------------

IntToTextBufN:	call	Int0ToTextBufN	; get text length without spaces
		push	ebx		; push EBX
		movzx	ebx,bh		; EBX <- minimal width of field
		cmp	ebx,eax		; check text length
		jb	IntToTextBufN8	; text length is OK
		xchg	eax,ebx		; EAX <- new text length
IntToTextBufN8:	pop	ebx		; pop EBX
IntToTextBufN9:	ret

; -----------------------------------------------------------------------------
;                  Convert signed INT number into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = number
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	On AMD 1.6 GHz it takes approx. "10*digits+40" nanosecs for
;		0 to 10	digits, "50*digits-360" nanosecs for 10 to 20 digits.
; -----------------------------------------------------------------------------
; Local variables - see UIntToTextBuf

; ------------- Check remaining free space in buffer

IntToTextBuf:	or	esi,esi		; check remaining space
		jle	short IntToTextBufN9 ; not enough free space

; ------------- Push registers (it must agree with UIntToTextBuf)

		push	eax		; push EAX number LOW
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX number HIGH
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,UINTStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[UINTForm],ebx	; formatting parameters FORMPAR

; ------------- Get text length without spaces (-> EDX)

		call	Int0ToTextBufN	; get text length without spaces
		mov	[UINTLen],eax	; store text length
		xchg	eax,edx		; EDX <- push text length

; ------------- Store leading spaces

		bt	ebx,FORMFLAG_Left_b ; left-justify?
		jc	IntToTextBuf3	; left-justify, no leading spaces
		movzx	ecx,bh		; ECX <- minimal width of field
		sub	ecx,edx		; ECX <- remaining spaces
		jle	IntToTextBuf3	; no spaces left
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	IntToTextBuf2	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	IntToTextBuf3	; no spaces left
IntToTextBuf2:	mov	al," "		; AL <- space
IntToTextBuf21:	dec	esi		; decrease space counter
		stosb			; store one space
IntToTextBuf22:	jz	UIntToTextBuf9	; buffer is full
		loop	IntToTextBuf21	; next character

; ------------- Prepare registers to convert number

IntToTextBuf3:	mov	ecx,edx		; ECX <- length in characters
		mov	ebx,[UINTNumL]	; EBX <- number LOW
		mov	eax,[UINTNumH] 	; EAX <- number HIGH

; ------------- Number is negative

		or	eax,eax		; is number negative?
		jns	IntToTextBuf34	; number is not negative

; ------------- Negative value

		neg	ebx		; EBX <- negative value LOW
		adc	eax,0		; EAX <- carry
		neg	eax		; EAX <- negative value HIGH
		mov	[UINTNumH],eax	; store number HIGH

; ------------- Store negative sign

		mov	al,"-"		; AL <- negative sign
		dec	esi		; decrease space counter
		stosb			; store sign
		jz	short IntToTextBuf22 ; buffer is full
		dec	ecx		; ECX <- without sign character
		jmp	UIntToTextBuf32

; ------------- Store positive sign or space

IntToTextBuf34:	test	byte [UINTForm+FORMPAR_Flags1],FORMFLAG1_Sign ; sign?
		mov	al,"+"		; AL <- positive sign
		jnz	IntToTextBuf35	; use positive sign
		test	byte [UINTForm+FORMPAR_Flags1],FORMFLAG1_Space ; space?
		jz	UIntToTextBuf32	; no space
		mov	al," "		; AL <- space
IntToTextBuf35:	dec	esi		; decrease space counter
		stosb			; store sign
		jz	short IntToTextBuf22 ; buffer is full
		dec	ecx		; ECX <- without sign character
		jmp	UIntToTextBuf32

; -----------------------------------------------------------------------------
;                     Internal - Store infinity or non-number text
; -----------------------------------------------------------------------------
; INPUT:	AH = sign flag (bit 7)
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		EDX = pointer to extended double float number (in stack)
;		ESI = remaining free space in buffer (must be > 0)
;		EDI = destination buffer
; OUTPUT:	ZY = destination buffer is full (EAX is not valid)
;		EAX = text length (= 7, valid only if NZ)
;		ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	Infinity is +1.#INF or -1.#INF, non-number +1.#NAN or -1.#NAN.
;		It should continue with storing trailing spaces.
; -----------------------------------------------------------------------------

; ------------- Store leading spaces

FloatInfNan:	bt	ebx,FORMFLAG_Left_b ; left?
		jc	FloatInfNan4	; left-justify, no spaces
		cmp	bh,7		; check minimal width
		jbe	FloatInfNan4	; width is low
		push	ecx		; push ECX
		movzx	ecx,bh		; ECX <- minimal width
		sub	cl,7		; ECX <- remaining spaces
		bt	ebx,FORMFLAG_Cent_b ; center?
		jnc	FloatInfNan1	; no, right-justify
		cmp	cl,2		; check minimal width
		jb	FloatInfNan3	; width is low (it jumps with NZ)
		shr	ecx,1		; ECX <- spaces / 2, round down
FloatInfNan1:	mov	al," "		; AL <- space
FloatInfNan2:	dec	esi		; decrease space counter
		stosb			; store one space
		loopnz	FloatInfNan2	; next character
FloatInfNan3:	pop	ecx		; pop ECX
		jz	FloatInfNan8	; buffer is full

; ------------- Store sign

FloatInfNan4:	mov	al,"-"		; AL <- negative sign
		or	ah,ah		; negative number?
		js	FloatInfNan5	; negative number
		mov	al,"+"		; AL <- positive sign
FloatInfNan5:	dec	esi		; decrease space counter
		stosb			; store sign
		jz	FloatInfNan8	; buffer is full

; ------------- Store "1" digit

		mov	al,"1"		; AL <- "1" digit
		dec	esi		; decrease space counter
		stosb			; store sign
		jz	FloatInfNan8	; buffer is full

; ------------- Store decimal separator

		mov	al,[ecx+NAT_DecimalSep] ; AL <- decimal separator
		dec	esi		; decrease space counter
		stosb			; store decimal separator
		jz	FloatInfNan8	; buffer is full

; ------------- Store "#" character

		mov	al,"#"		; AL <- "#" symbol
		dec	esi		; decrease space counter
		stosb			; store "#" symbol
		jz	FloatInfNan8	; buffer is full

; ------------- Prepare text

		mov	eax,[edx]	; EAX <- mantissa LOW
		or	eax,[edx+4]	; check if mantissa is zero
		mov	eax,"NAN"	; non-number value
		jnz	FloatInfNan6	; not zero, it is non-number
		mov	eax,"INF"	; infinity value

; ------------- Store first character

FloatInfNan6:	dec	esi		; decrease space counter
		stosb			; store fist character
		jz	FloatInfNan8	; buffer is full

; ------------- Store second character

		shr	eax,8		; AL <- second character
		dec	esi		; decrease space counter
		stosb			; store second character
		jz	FloatInfNan8	; buffer is full

; ------------- Store third character

		shr	eax,8		; AL <- third character
		dec	esi		; decrease space counter
		stosb			; store third character

; ------------- New text length (here is ZY = buffer is full)

FloatInfNan8:	mov	al,7		; EAX <- text length
		ret

; -----------------------------------------------------------------------------
;           Internal - Split positive float to mantissa and exponent
; -----------------------------------------------------------------------------
; INPUT:	ECX = pointer to extended double float number in stack
;		EDX = binary exponent
;		ST0 = float number (positive number, not zero)
; OUTPUT:	EBX = decimal exponent
;		ST0 = mantissa (in range 1 including to 10 excluding)
; DESTROYS:	EAX, EDX
; NOTES:	It uses 1 more register from FPU stack.
; -----------------------------------------------------------------------------

; ------------- Convert binary exponent to decimal exponent, 100-step (-> EDX)
; 100 * ln 10 / ln 2 = 332.1928094887
; Binary exponent in range 0 to 32766 goes to range 0 to 98.

FloatSplit:	add	edx,60		; add shift correction
		mov	eax,12929140	; EAX <- 100000000h/332.1928094887
		mul	edx		; EDX <- decimal exponent

; ------------- Multiply number with 100-step exponent

		imul	eax,edx,byte 10	; EAX <- offset of exponent
		fld	tword [eax+Exp100Tab] ; load exponent
		fmulp	st1		; multiple with exponent

; ------------- Load exponent into exponent accumulator (-> EBX)

		imul	eax,edx,byte 100 ; EAX <- exponent correction
		lea	ebx,[eax-4900]	; EBX <- exponent correction

; ------------- Store new number into stack

		fld	st0		; duplicate number
		fstp	tword [ecx]	; store number into stack

; ------------- Load exponent (-> EDX)

		fwait			; synchronize FPU
		movzx	edx,word [ecx+8] ; EDX <- exponent

; ------------- Convert binary exponent to decimal exponent, 1-step (-> EDX)
; ln 10 / ln 2 = 3.321928094887
; Binary exponent in range 0 to 32766 goes to range 0 to 9863.

		mov	eax,1292913987	; EAX <- 100000000h/3.321928094887
		mul	edx		; EDX <- decimal exponent
		sub	edx,4932	; EDX <- correction of exponent base

; ------------- Multiply with 1-step exponent

		imul	eax,edx,byte 10	; EAX <- offset of exponent
		fld	tword [eax+Exp1Tab+EXP1TABBEG*10] ; load exponent
		fmulp	st1		; multiple with exponent

; ------------- Load exponent into exponent accumulator (-> EBX)

		add	ebx,edx		; EBX <- add exponent into accumulator

; ------------- Check if mantissa is less than 1

		fld1			; load 1
                fcomp	st1		; compare 1 with mantissa
		fstsw	ax		; AX <- FPU flags
		sahf			; get FPU flags
		ja	FloatSplit2	; mantissa is less than 1

; ------------- Check if mantissa is less than 10

		ficom	dword [Const10Num] ; compare mantissa with 10
		fstsw	ax		; AX <- FPU flags
		sahf			; get FPU flags
		jae	FloatSplit4	; mantissa is not less than 10
		ret

; ------------- Mantissa is less than 1

FloatSplit2:	fimul	dword [Const10Num] ; multiply mantissa with 10
		dec	ebx		; EBX <- exponent correction
		ret

; ------------- Mantissa is not less than 10

FloatSplit4:	fld	tword [Const01Num] ; load constant 0.1
		inc	ebx		; EBX <- exponent correction
		fmulp	st1,st0		; multiply mantissa with 0.1
		ret

; -----------------------------------------------------------------------------
;          Internal - Round, denormalize and convert mantissa to BCD
; -----------------------------------------------------------------------------
; INPUT:	EAX = precision of fractional part (it will be limited to 18)
;		EBX = decimal exponent
;		ECX = pointer to extended double float number in stack (10 B)
;		ST0 = normalized float number (in range 1 incl. to 10 excl.)
;		CY = don't round (e.g. round always to 19 digits)
; OUTPUT:	EBX = new decimal exponent
;		[ECX] = mantissa digits in BCD form (19 digits, 10 bytes),
;			more significant digits are in lower address and bits
; DESTROYS:	EAX,ECX,EDX
; NOTES:	It uses 1 more register from FPU stack.
;		It pops up input number ST0.
; -----------------------------------------------------------------------------

; ------------- Push registers

FloatMantBCD:	push	esi		; push ESI
		push	edi		; push EDI
		mov	edi,ecx		; EDI <- pointer to float

; ------------- Limit precision (here is CY = don't round)

		jc	short FloatMantBCD82 ; don't round
		cmp	eax,byte 18	; check precision
		ja	short FloatMantBCD8 ; limit precision
FloatMantBCD2:	mov	ch,al		; CH <- push precision

; ------------- Round mantissa

		imul	eax,eax,10	; EAX <- precision * 10
FloatMantBCD3:	fld	tword [RoundTab+eax] ; load rounding correction
		faddp	st1,st0		; add rounding correction

; ------------- Check if mantissa is less than 10

		ficom	dword [Const10Num] ; compare mantissa with 10
		fstsw	ax		; AX <- FPU flags
		sahf			; get FPU flags
		jae	short FloatMantBCD9 ; mantissa is not less than 10

; ------------- Store number into stack

FloatMantBCD4:	fstp	tword [edi]	; store number into stack

; ------------- Push registers 2

		push	ebx		; push EBX (exponent)

; ------------- Load and denormalize mantissa (-> EAX:ESI:EBX, CH=precision)

		fwait			; synchronize FPU
		xor	esi,esi		; ESI <- 0
		mov	cl,[edi+8]	; CL <- exponent
		xchg	esi,[edi+4]	; ESI <- mantissa HIGH, clear buffer
		add	cl,2		; bias correction (=subtract 3fffh) + 1
		xor	eax,eax		; EAX <- 0
		xor	ebx,ebx		; EBX <- 0
		mov	[edi+8],ax	; clear buffer
		xchg	ebx,[edi]	; EBX <- mantissa LOW, clear buffer
		shld	eax,esi,cl	; EAX <- shift bits from ESI
		shld	esi,ebx,cl	; ESI <- shift bits from EBX
		shl	ebx,cl		; EBX <- shift bits

; ------------- Prepare registers

		movzx	ecx,ch		; ECX <- required number of digits - 1
		inc	ecx		; ECX <- required number of digits
		jmp	FloatMantBCD6	; store first digit

; ------------- Convert first digit

FloatMantBCD5:	mov	al,10		; EAX <- 10
		mul	ebx		; EDX:EAX <- multiple mantissa LOW
		xchg	eax,ebx		; EBX <- store new mantissa LOW
		xchg	edx,esi		; EDX <- old mant.HIGH, ESI <- new HIGH
		xor	eax,eax		; EAX <- 0
		mov	al,10		; EAX <- 10
		mul	edx		; EDX:EAX <- new mantissa HIGH
		add	esi,eax		; ESI <- add new mantissa HIGH
		xchg	eax,edx		; AL <- one digit
		adc	al,0		; AL <- carry
FloatMantBCD6:	stosb			; store first digit
		dec	ecx		; digit counter
		jz	FloatMantBCD7	; no other digit

; ------------- Convert second digit

		mov	al,10		; EAX <- 10
		mul	ebx		; EDX:EAX <- multiple mantissa LOW
		xchg	eax,ebx		; EBX <- store new mantissa LOW
		xchg	edx,esi		; EDX <- old mant.HIGH, ESI <- new HIGH
		xor	eax,eax		; EAX <- 0
		mov	al,10		; EAX <- 10
		mul	edx		; EDX:EAX <- new mantissa HIGH
		add	esi,eax		; ESI <- add new mantissa HIGH
		xchg	eax,edx		; AL <- one digit
		adc	al,0		; AL <- carry
		shl	al,4		; shift digit 4 bits left
		or	[edi-1],al	; add digit
		loop	FloatMantBCD5	; next digit

; ------------- Pop registers

FloatMantBCD7:	pop	ebx		; pop EBX (decimal exponent)
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		ret

; ------------- Limit precision

FloatMantBCD8:	xor	eax,eax		; EAX <- 0
		mov	al,18		; EAX <- 18, limit precision
		jmp	short FloatMantBCD2

; ------------- Don't round

FloatMantBCD82:	cmp	eax,byte 18	; check precision
		jae	short FloatMantBCD8 ; limit precision
		mov	ch,al		; CH <- push precision
		mov	al,10*18	; EAX <- limit precision
		jmp	short FloatMantBCD3

; ------------- Mantissa is not less than 10

FloatMantBCD9:	fld	tword [Const01Num] ; load constant 0.1
		inc	ebx		; EBX <- exponent correction
		fmulp	st1,st0		; multiply mantissa with 0.1
		jmp	short FloatMantBCD4

; -----------------------------------------------------------------------------
;                    Internal - Truncate trailing zeros
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to last byte of mantissa in BCD form
;			more significant digits are in lower address and bits
; OUTPUT:	ECX = real precision (0 to 19 digits)
; DESTROYS:	AL, EDX
; -----------------------------------------------------------------------------

; ------------- Prepare digit counter (-> ECX)

FloatTrunc:	xor	ecx,ecx		; ECX <- 0
		mov	cl,20		; ECX <- 20, digit counter

; ------------- Check second digit

FloatTrunc2:	mov	al,[edx]	; AL <- get 2 digits
		test	al,0f0h		; text high digit
		jnz	FloatTrunc4	; digit is not zero
		dec	ecx		; decrease digit counter

; ------------- Check first digit

		test	al,0fh		; text low digit
		jnz	FloatTrunc4	; digit is not zero
		dec	edx		; decrease pointer
		loop	FloatTrunc2	; next digit

; ------------- Without first (integer) digit

		inc	ecx		; number is zero, limit to 0 precision
FloatTrunc4:	dec	ecx		; ECX <- without integer digit
		ret

; -----------------------------------------------------------------------------
;  Convert float number into text buffer in exponential form - get text length
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; NOTES:	It uses 2 more registers from FPU stack.
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------

; ------------- Push registers

ExpToTextBufN: 	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		sub	esp,12		; create local buffer

; ------------- Prepare precision (alternate: number of significant digits)

		bt	ebx,FORMFLAG_Prec_b ; alternate precision?
		sbb	bl,0		; without first digit
		adc	bl,0		; minimal precision 0 if borrow

; ------------- Store number into stack

		fld	st0		; duplicate number
		fstp	tword [esp]	; store number into stack
		
; ------------- Load number into registers (-> AX:EDX:ESI)

		fwait			; synchronize FPU
		mov	esi,[esp]	; ESI <- mantissa LOW
		mov	edx,[esp+4]	; EDX <- mantissa HIGH
		movzx	eax,word [esp+8] ; EAX <- exponent

; ------------- Special cases

		test	ax,7fffh	; clear sign flag
		jnz	ExpToTextBufN0	; not zero

; ------------- Zero

		mov	ecx,edx		; ECX <- mantissa HIGH
		or	ecx,esi		; ECX <- is mantissa zero?
		jnz	ExpToTextBufN2	; mantissa is not zero
		and	ah,B7		; get sign flag
		or	ax,3fffh	; EAX <- normalize exponent
		bt	ebx,FORMFLAG_Alt2_b ; truncate zeros?
		jnc	ExpToTextBufN28	; don't truncate trailing zeros
		jmp	ExpToTextBufN29	; truncate zeros (here is ECX = 0)

; ------------- Infinity (+1.#INF, -1.#INF) or non-number (+1.#NAN, -1.#NAN)

ExpToTextBufN0:	cmp	ax,0ffffh	; negative infinity or non-number?
		je	ExpToTextBufN1	; yes
		cmp	ax,7fffh	; infinity or non-number?
		jne	ExpToTextBufN2	; neither infinity nor non-number
ExpToTextBufN1:	xor	ecx,ecx		; ECX <- 0
		mov	cl,7		; ECX <- text length
		jmp	ExpToTextBufN8

; ------------- Check if truncate trailing zeros

ExpToTextBufN2:	bt	ebx,FORMFLAG_Alt2_b ; truncate zeros?
		jnc	ExpToTextBufN28	; don't truncate trailing zeros

; ------------- Push registers (here is AX=exponent, EBX=formatting param.)

		mov	ecx,esp		; ECX <- pointer to float in stack
		push	eax		; push EAX
		push	edx		; push EDX
		push	ebx		; push EBX (formatting parameters)

; ------------- Absolute value

		fld	st0		; duplicate number
		fabs			; absolute value

; ------------- Split number to mantissa and exponent (here is EDX=bin. exp.)

		and	ah,7fh		; mask sign flag
		xchg	eax,edx		; EDX <- exponent
		call	FloatSplit	; split number to mantissa and exponent

;-------------- Round, denormalize and convert mantissa to BCD (EBX = exponent)

		pop	eax		; EAX <- formatting parameters
		push	eax		; push EAX (formatting parameters)
		bt	eax,FORMTYPE_Cap_b ; round always down?
		movzx	eax,al		; EAX <- precision
		call	FloatMantBCD	; process mantissa

; ------------- Truncate trailing zeros

		lea	edx,[esp+3*4+9]	; EDX <- last digit
		call	FloatTrunc	; truncate trailing zeros

; ------------- Pop registers

		pop	ebx		; pop EBX (formatting parameters)
		pop	edx		; pop EDX
		pop	eax		; pop EAX
		cmp	cl,bl		; check precision
		jb	ExpToTextBufN29	; limit precision

; ------------- Prepare number of digits after decimal point (-> ECX)

ExpToTextBufN28:movzx	ecx,bl		; ECX <- number of digits

; ------------- Add decimal point and first digit

ExpToTextBufN29:cmp	cl,1		; 1 or more digits?
		cmc			; CY = 1 or more digits
		jc	ExpToTextBufN3	; 1 or more digits
		bt	ebx,FORMFLAG_Alt_b ; always use decimal point?
ExpToTextBufN3:	adc	ecx,byte 1	; add decimal point and first digit

; ------------- Add sign

		or	ax,ax		; negative number?
		js	ExpToTextBufN4	; negative number
		test	ebx,(FORMFLAG1_Sign + FORMFLAG1_Space)*256*256 ; sign?
		jz	ExpToTextBufN5	; no sign
ExpToTextBufN4:	inc	ecx		; add sign
ExpToTextBufN5:	and	ah,7fh		; clear sign flag

; ------------- Add exponent, 3 digits

		add	ecx,byte 1+1+3	; add exp. character, sign and 3 digits

; ------------- Add +exponent, 4 digits (1.0e+1000, 4CF8 F38DB1F9 DD3DAC05)

		cmp	ax,4cf8h	; exponent
		jb	ExpToTextBufN6	; number is less than 1.0e+1000
		ja	ExpToTextBufN7	; number if greater than 1.0e+1000
		cmp	edx,0f38db1f9h	; mantissa HIGH
		jb	ExpToTextBufN8	; number is less than 1.0e+1000
		ja	ExpToTextBufN7	; number if greater than 1.0e+1000
		cmp	esi,0dd3dac05h	; mantissa LOW
		jb	ExpToTextBufN8	; number is less than 1.0e+1000
		jmp	ExpToTextBufN7	; add 1 digit

; ------------- Add -exponent, 4 digits (1.0e-1000, 3305 868A9188 A89E1467)

ExpToTextBufN6:	cmp	ax,3305h	; exponent
		ja	ExpToTextBufN8	; number is greater than 1.0e-1000
		jb	ExpToTextBufN7	; number if less than 1.0e-1000
		cmp	edx,868a9188h	; mantissa HIGH
		ja	ExpToTextBufN8	; number is greater than 1.0e-1000
		jb	ExpToTextBufN7	; number if less than 1.0e-1000
		cmp	esi,0a89e1467h	; mantissa LOW
		ja	ExpToTextBufN8	; number is greater than 1.0e-1000
ExpToTextBufN7:	inc	ecx		; add 1 digit

; ------------- Check field width

ExpToTextBufN8:	movzx	eax,bh		; EAX <- minimal width
		cmp	eax,ecx		; check minimal width
		ja	ExpToTextBufN84	; use minimal width
		xchg	eax,ecx		; EAX <- text length

; ------------- Pop registers

ExpToTextBufN84:add	esp,12		; free local buffer
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
ExpToTextBufN9:	ret

; -----------------------------------------------------------------------------
;     Convert float number into text buffer in exponential form - small "e"
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	It uses 2 more registers from FPU stack.
;		FPU should be in default state - 64 bits, round to nearest.
;		Exponent is 3 or 4 digits.
;		To destroy FPU register you can use "ffreep st0" instruction.
;		On AMD 1.6 GHz it takes approx. "10*decimal places+200" nsec.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		EXPNat     ebp+8	; (4) pointer to nationality NATIONAL
%define		EXPForm    ebp-4	; (4) formatting parameters FORMPAR
%define		EXPLen     ebp-8	; (4) total length of text
%define		EXPChar    ebp-9	; (1) exponent character "E" or "e"
%define		EXPSign    ebp-10	; (1) stored flags, bit 7 = sign
					;	bit 6 = exponent sign
%define		EXPExp     ebp-12	; (2) float number, exponent
					;       (bit 0..14) and sign (bit 15)
%define		EXPMantH   ebp-16	; (4) float number, mantissa HIGH
%define		EXPMantL   ebp-20	; (4) float number, mantissa LOW
%define		EXPFloat   EXPMantL	; ...(10) float number

%define		EXPStack   20		; stack size

; ------------- Check remaining free space in buffer

ExpSToTextBuf:	or	esi,esi		; check remaining space
		jle	short ExpToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,EXPStack	; ESP <- create local variables

; ------------- Prepare exponent character

		mov	byte [EXPChar],"e" ; prepare exponent character
		jmp	short ExpToTextBuf1

; -----------------------------------------------------------------------------
;   Convert float number into text buffer in exponential form - capital "E"
; -----------------------------------------------------------------------------

; ------------- Check remaining free space in buffer

ExpCToTextBuf:	or	esi,esi		; check remaining space
		jle	short ExpToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,EXPStack	; ESP <- create local variables

; ------------- Prepare exponent character

		mov	byte [EXPChar],"E" ; prepare exponent character

; ------------- Prepare precision (alternate: number of significant digits)

ExpToTextBuf1:	bt	ebx,FORMFLAG_Prec_b ; alternate precision?
		sbb	bl,0		; without first digit
		adc	bl,0		; minimal precision 0 if borrow

; ------------- Prepare local variables

		mov	[EXPForm],ebx	; formatting parameters FORMPAR

; ------------- Store number into stack

		fld	st0		; duplicate number
		fstp	tword [EXPFloat] ; store number into stack

; ------------- Load exponent (-> EDX)

		fwait			; synchronize FPU
		movzx	edx,word [EXPExp] ; EDX <- exponent
		mov	ch,dh		; CH <- sign
		and	ch,B7		; left only sign flag
		mov	[EXPSign],ch	; push sign flag (bit 7)

; ------------- Absolute value (it uses copy of register)

		fld	st0		; duplicate number
		fabs			; absolute value
		and	dx,7fffh	; mask exponent (15 bits)
		jnz	ExpToTextBuf2	; neither zero nor denormalized number

; ============= Special cases

; ------------- Zero or denormalized number

		mov	eax,[EXPMantL]	; EAX <- mantissa LOW
		or	eax,[EXPMantH]	; check if mantissa is zero
		jnz	ExpToTextBuf2	; not zero, it is denormalized number

; ------------- Zero

		ffreep	st0		; pop register from the FPU stack
		xor	ebx,ebx		; EBX <- zero decimal exponent
		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; trunc.?
		jz	ExpToTextBuf6	; don't truncate
		mov	byte [EXPForm+FORMPAR_Prec],0 ; limit precision
		jmp	ExpToTextBuf6

; ------------- Infinity (+1.#INF, -1.#INF) or non-number (+1.#NAN, -1.#NAN)

ExpToTextBuf2:	cmp	dx,7fffh	; infinity or non-number?
		jne	ExpToTextBuf3	; neither infinity nor non-number
		ffreep	st0		; pop register from the FPU stack
		xchg	eax,ecx		; AH <- sign flag (bit 7)
		mov	ecx,[EXPNat]	; ECX <- pointer to nationality
		lea	edx,[EXPFloat]	; EDX <- pointer to float number
		call	FloatInfNan	; store text
		jz	ExpToTextBuf9	; buffer is full
		jmp	ExpToTextBuf86	; store trailing spaces

; ============= Convert number to digits

; ------------- Split number to mantissa and exponent (here is EDX=bin. exp.)

ExpToTextBuf3:	lea	ecx,[EXPFloat]	; ECX <- pointer to float in stack
		call	FloatSplit	; split number to mantissa and exponent

;-------------- Round, denormalize and convert mantissa to BCD (EBX = exponent)

		movzx	eax,byte [EXPForm+FORMPAR_Prec] ; EAX <- precision
		lea	ecx,[EXPFloat]	; ECX <- pointer to float in stack
		bt	dword [EXPForm],FORMTYPE_Cap_b ; round always down?
		call	FloatMantBCD	; process mantissa

; ------------- Truncate trailing zeros

		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; trunc.?
		jz	ExpToTextBuf6	; don't truncate
		lea	edx,[EXPFloat+9] ; EDX <- last digit
		call	FloatTrunc	; truncate trailing zeros
		cmp	cl,[EXPForm+FORMPAR_Prec] ; check precision
		jae	ExpToTextBuf6	; precision is OK
		mov	[EXPForm+FORMPAR_Prec],cl ; limit precision

; ============= Get text length (here is EBX = decimal exponent)

; ------------- Add decimal point and first digit

ExpToTextBuf6:	movzx	eax,byte [EXPForm+FORMPAR_Prec]	; EAX <- num. of digits
		or	eax,eax		; zero digits after decimal point?
		jz	ExpToTextBuf66	; no digits after decimal point
		bts	dword [EXPForm],FORMFLAG_Alt_b ; set dec. point flag
ExpToTextBuf66:	bt	dword [EXPForm],FORMFLAG_Alt_b ; use decimal point?
		adc	eax,byte 1	; add decimal point and first digit

; ------------- Add sign

		mov	dl,[EXPSign]	; DL <- sign (bit 7)
		mov	dh,[EXPForm+FORMPAR_Flags1] ; DH <- flags 1
		and	dx,B7 + (FORMFLAG1_Sign + FORMFLAG1_Space)*256
		setnz	dl		; DL <- 1 if sign, else 0
		movzx	edx,dl		; EDX <- 1 if sign, else 0
		add	eax,edx		; EAX <- add sign

; ------------- Absolute value of exponent (-> EBX)

		or	ebx,ebx		; is exponent negative?
		jns	ExpToTextBuf68	; exponent is not negative
		neg	ebx		; EBX <- absolute value of exponent
		or	byte [EXPSign],B6 ; set exponent sign flag

; ------------- Add exponent

ExpToTextBuf68:	add	eax,byte 1+1+3	; add exp. character, sign and 3 digits
		cmp	bx,1000		; exponent 1000 or greater?
		cmc			; CY = exponent is 1000 or greater
		adc	eax,byte 0	; EAX <- exponent is 4 digits
		mov	[EXPLen],eax	; set length of text
		xchg	eax,edx		; EDX <- length of text

; ============= Decode spaces and sign (here is EDX=text length, EBX=exponent)

; ------------- Store leading spaces

	  test word [EXPForm+FORMPAR_TypeFlg],FORMFLAG1_Left+FORMFLAG2_Zero*256
		jnz	ExpToTextBuf72	; left-justify or add zeros, no spaces
		movzx	ecx,byte [EXPForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,edx		; ECX <- remaining spaces
		jle	ExpToTextBuf72	; no spaces left
		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	ExpToTextBuf7	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	ExpToTextBuf72	; no spaces left
ExpToTextBuf7:	mov	al," "		; AL <- space
ExpToTextBuf71:	dec	esi		; decrease space counter
		stosb			; store one space
		loopnz	ExpToTextBuf71	; next character
		jz	short ExpToTextBuf76 ; buffer is full

; ------------- Store sign

ExpToTextBuf72:	mov	al,"-"		; AL <- negative sign
		test	byte [EXPSign],B7 ; negative number?
		jnz	ExpToTextBuf73	; negative number
		test	byte [EXPForm+FORMPAR_Flags1],FORMFLAG1_Sign ; sign?
		mov	al,"+"		; AL <- positive sign
		jnz	ExpToTextBuf73	; use positive sign
		test	byte [EXPForm+FORMPAR_Flags1],FORMFLAG1_Space ; space?
		jz	ExpToTextBuf74	; no space
		mov	al," "		; AL <- space
ExpToTextBuf73:	dec	esi		; decrease space counter
		stosb			; store sign
		jz	short ExpToTextBuf76 ; buffer is full

; ------------- Add leading zeros

ExpToTextBuf74:	test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Zero ; zeros?
		jz	ExpToTextBuf77	; no zeros
		movzx	ecx,byte [EXPForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,edx		; ECX <- remaining spaces
		jle	ExpToTextBuf77	; no spaces left
		mov	al,"0"		; AL <- zero
ExpToTextBuf75:	dec	esi		; decrease space counter
		stosb			; store zero
		loopnz	ExpToTextBuf75	; next character
ExpToTextBuf76:	jz	short ExpToTextBuf82 ; buffer is full

; ============= Decode mantissa

ExpToTextBuf77:	lea	edx,[EXPFloat]	; EDX <- pointer to mantissa

; ------------- Store integer part

		mov	al,[edx]	; AL <- first digit
		and	al,0fh		; AL <- mask digit
		add	al,"0"		; AL <- convert to ASCII
		dec	esi		; decrease space counter
		stosb			; store one digit
		jz	short ExpToTextBuf82 ; buffer is full

; ------------- Store decimal separator

		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Alt ; dot?
		jz	ExpToTextBuf8	; no decimal point
		mov	eax,[EXPNat]	; EAX<-pointer to nationality NATIONAL
		mov	al,[eax+NAT_DecimalSep] ; AL <- decimal separator
		dec	esi		; decrease space counter
		stosb			; store decimal separator
		jz	short ExpToTextBuf82 ; buffer is full

; ------------- Prepare number of digits after decimal point (-> ECX)

		movzx	ecx,byte [EXPForm+FORMPAR_Prec]	; ECX <- num. of digits
		jecxz	ExpToTextBuf8	; no digit
		cmp	cl,18		; check maximal precision
		jb	ExpToTextBuf78	; precision is OK
		mov	cl,18		; CL <- 18, limit precision

; ------------- Decode mantissa - second digit

ExpToTextBuf78:	mov	al,[edx]	; AL <- 2 digits
		shr	al,4		; AL <- mask digit
		add	al,"0"		; AL <- convert to ASCII
		dec	esi		; decrease space counter
		stosb			; store one digit
		jz	short ExpToTextBuf82 ; buffer is full
		dec	ecx		; character counter
		jz	ExpToTextBuf79	; no next character

; ------------- Decode mantissa - first digit

		inc	edx		; EDX <- shift pointer
		mov	al,[edx]	; AL <- 2 digits
		and	al,0fh		; AL <- mask digit
		add	al,"0"		; AL <- convert to ASCII
		dec	esi		; decrease space counter
		stosb			; store one digit
		loopnz	ExpToTextBuf78	; next character
		jz	short ExpToTextBuf82 ; buffer is full

; ------------- Store trailing zeros

ExpToTextBuf79:	movzx	ecx,byte [EXPForm+FORMPAR_Prec]	; ECX <- num. of digits
		sub	cl,18		; ECX <- remaining zeros
		jle	ExpToTextBuf8	; no zeros left
		mov	al,"0"		; AL <- trailing zero
ExpToTextBuf7A:	dec	esi		; decrease space counter
		stosb			; store one digit
		loopnz	ExpToTextBuf7A	; next digit
		jz	short ExpToTextBuf82 ; buffer is full

; ============= Exponent (here is EBX = decimal exponent)

; ------------- Exponent separator

ExpToTextBuf8:	mov	al,[EXPChar]	; AL <- exponent character
		dec	esi		; decrease space counter
		stosb			; store exponent separator
		jz	short ExpToTextBuf82 ; buffer is full

; ------------- Exponent sign

		mov	al,"-"		; AL <- minus sign
		test	byte [EXPSign],B6 ; negative exponent?
		jnz	ExpToTextBuf81	; negative exponent
		mov	al,"+"		; AL <- plus sign
ExpToTextBuf81:	dec	esi		; decrease space counter
		stosb			; store exponent separator
ExpToTextBuf82:	jz	short ExpToTextBuf9 ; buffer is full

; ------------- Decode exponent

		imul	eax,ebx,268436	; EAX <- multiple with 1000'0000h/1000
		shr	eax,28		; AL <- thousands
		jz	ExpToTextBuf83	; no thousand digit
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease space counter
		stosb			; store digit
		jz	short ExpToTextBuf9 ; buffer is full
		sub	al,"0"		; AL <- return thousands
		imul	eax,eax,1000	; EAX <- thousands
		sub	ebx,eax		; EBX <- exponent without thousands

ExpToTextBuf83:	imul	eax,ebx,2684355	; EAX <- multiple with 1000'0000h/100
		shr	eax,28		; AL <- hundreds
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease space counter
		stosb			; store digit
		jz	short ExpToTextBuf9 ; buffer is full
		sub	al,"0"		; AL <- return hundreds
		imul	eax,eax,100	; EAX <- hundreds
		sub	ebx,eax		; EBX <- exponent without hundreds

ExpToTextBuf85:	xchg	eax,ebx		; EAX <- exponent 0 to 99
		aam			; AL <- units, AH <- tens
		xchg	al,ah		; AL <- tens, AH <- units
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease space counter
		stosb			; store digit
		jz	short ExpToTextBuf9 ; buffer is full
		mov	al,ah		; AL <- units
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease space counter
		stosb			; store digit
		jz	short ExpToTextBuf9 ; buffer is full
		
; ------------- Store trailing spaces

		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Zero ; zeros?
		jnz	short ExpToTextBuf9 ; zeros, no spaces
		mov	eax,[EXPLen]	; EAX <- text length
ExpToTextBuf86:	movzx	ecx,byte [EXPForm+FORMPAR_Width] ; ECX <- min. width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	short ExpToTextBuf9 ; no spaces remain
		test	byte [EXPForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	ExpToTextBuf87	; left-justify
		test	byte [EXPForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	short ExpToTextBuf9 ; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	short ExpToTextBuf9 ; no spaces
ExpToTextBuf87:	mov	al," "		; AL <- space
ExpToTextBuf88:	stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	ExpToTextBuf88	; next character

; ------------- Pop registers

ExpToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;  Convert float number into text buffer in float point form - get text length
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; NOTES:	It uses 1 more register from FPU stack.
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------
; Local variables:
%define		FLTNForm   ebp-4	; (4) formatting parameters FORMPAR
					; (1)
%define		FLTNSign   ebp-6	; (1) stored flags, bit 7 = sign
%define		FLTNExp    ebp-8	; (2) float number, exponent
					;       (bit 0..14) and sign (bit 15)
%define		FLTNMantH  ebp-12	; (4) float number, mantissa HIGH
%define		FLTNMantL  ebp-16	; (4) float number, mantissa LOW
%define		FLTNFloat  FLTNMantL	; ...(10) float number

%define		FLTNStack   16		; stack size

; ------------- Push registers

FltToTextBufN: 	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,FLTNStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[FLTNForm],ebx	; formatting parameters FORMPAR

; ------------- Store number into stack

		fld	st0		; duplicate number
		fstp	tword [FLTNFloat] ; store number into stack

; ------------- Load exponent (-> EDX)

		fwait			; synchronize FPU
		movzx	edx,word [FLTNExp] ; EDX <- exponent
		mov	ch,dh		; CH <- sign
		and	ch,B7		; left only sign flag
		mov	[FLTNSign],ch	; push sign flag (bit 7)

; ------------- Absolute value (it uses copy of register)

		fld	st0		; duplicate number
		fabs			; absolute value
		and	dx,7fffh	; mask exponent (15 bits)
		jnz	FltToTextBufN2	; neither zero nor denormalized number

; ============= Special cases

; ------------- Zero or denormalized number

		mov	eax,[FLTNMantL]	; EAX <- mantissa LOW
		or	eax,[FLTNMantH]	; check if mantissa is zero
		jnz	FltToTextBufN2	; not zero, it is denormalized number

; ------------- Zero

		ffreep	st0		; pop register from the FPU stack
		xor	ebx,ebx		; EBX <- zero decimal exponent
	      test byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Alt2+FORMFLAG2_Prec
		jz	FltToTextBufN6	; don't truncate
FltToTextBufN1:	xor	ecx,ecx		; limit precision
		jmp	FltToTextBufN61

; ------------- Infinity (+1.#INF, -1.#INF) or non-number (+1.#NAN, -1.#NAN)

FltToTextBufN2:	cmp	dx,7fffh	; infinity or non-number?
		jne	FltToTextBufN3	; neither infinity nor non-number
		ffreep	st0		; pop register from the FPU stack
		xor	ecx,ecx		; ECX <- 0
		mov	cl,7		; ECX <- 7, text length
		jmp	FltToTextBufN8

; ------------- Split number to mantissa and exponent

FltToTextBufN3:	lea	ecx,[FLTNFloat]	; ECX <- pointer to float in stack
		call	FloatSplit	; split number to mantissa and exponent

;-------------- Round, denormalize and convert mantissa to BCD (EBX = exponent)

		movzx	eax,byte [FLTNForm+FORMPAR_Prec] ; EAX <- precision
		test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Prec; precis.?
		jz	FltToTextBufN4	; normal precision (=fractional digits)
		dec	eax		; without first digit
		jns	FltToTextBufN42	; digits are OK
		jmp	FltToTextBufN41	; limit to 0
FltToTextBufN4:	add	eax,ebx		; EAX <- position of last digit
		jg	FltToTextBufN42	; digits are OK
FltToTextBufN41:xor	eax,eax		; EAX <- 0, digits are out of display
FltToTextBufN42:lea	ecx,[FLTNFloat]	; ECX <- pointer to float in stack
		bt	dword [FLTNForm],FORMTYPE_Cap_b ; round always down?
		call	FloatMantBCD	; process mantissa

; ------------- Use precision digits (here is EBX = decimal exponent)

		lea	edx,[FLTNFloat+9] ; EDX <- last digit
		test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Prec; precis.?
		jz	FltToTextBufN43	; normal precision (=fractional digits)
		call	FloatTrunc	; truncate trailing zeros
		sub	ecx,ebx		; ECX <- maximal precision
		jg	FltToTextBufN61	; precision is positive
		xor	ecx,ecx		; ECX <- limit precision
		jmp	FltToTextBufN61

; ------------- Truncate trailing zeros (here is EBX = decimal exponent)

FltToTextBufN43:test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; trunc.?
		jz	FltToTextBufN6	; don't truncate
		movzx	eax,byte [FLTNForm+FORMPAR_Prec] ; EAX <- precision
		add	eax,ebx		; check if number is out of display
		jl	FltToTextBufN1	; number is not visible
		call	FloatTrunc	; truncate trailing zeros
		sub	ecx,ebx		; ECX <- maximal precision
		jg	FltToTextBufN44	; precision is positive
		xor	ecx,ecx		; ECX <- limit precision
FltToTextBufN44:movzx	eax,byte [FLTNForm+FORMPAR_Prec] ; EAX <- precision
		cmp	ecx,eax		; check precision
		jae	FltToTextBufN6	; precision is OK
		mov	[FLTNForm+FORMPAR_Prec],cl ; limit precision

; ============= Get text length (here is EBX = decimal exponent)

; ------------- Add sign (-> EAX)

FltToTextBufN6:	movzx	ecx,byte [FLTNForm+FORMPAR_Prec] ; ECX<-num. of digits
FltToTextBufN61:mov	al,[FLTNSign]	; AL <- sign (bit 7)
		mov	ah,[FLTNForm+FORMPAR_Flags1] ; AH <- flags 1
		and	ax,B7 + (FORMFLAG1_Sign + FORMFLAG1_Space)*256
		setnz	al		; AL <- 1 if sign, else 0
		movzx	eax,al		; EAX <- 1 if sign, else 0

; ------------- Add decimal point and precision (-> ECX)

		jecxz	FltToTextBufN62	; no digits after decimal point
		bts	dword [FLTNForm],FORMFLAG_Alt_b ; set dec. point flag
FltToTextBufN62:bt	dword [FLTNForm],FORMFLAG_Alt_b ; use decimal point?
		adc	ecx,eax		; ECX<-add decimal point and sign

; ------------- Prepare minimal integer part (-> EAX)

		xor	eax,eax		; EAX <- 0
		or	ebx,ebx		; is exponent negative?
		sets	al		; AL <- 1 if < 1.0, 0 if >= 1.0
		dec	eax		; EAX <- 0 if < 1.0, -1 if >= 1.0
		and	eax,ebx		; EAX <- 0 if < 1.0, exponent if >= 1.0
		inc	eax		; EAX <- add first digit

; ------------- Check if use zeros instead of spaces

		test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Zero ; zero?
		jz	FltToTextBufN64	; no zeros

; ------------- Prepare minimal integer part if use zeros (-> EDX)

		movzx	edx,byte [FLTNForm+FORMPAR_Width] ; EDX<-minimal width
		sub	edx,ecx		; EDX <- without sign and fractional
		jle	FltToTextBufN64	; not enough space

; ------------- Check if use thousand separator

		test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Thsnd ; thsnd?
		jz	FltToTextBufN63	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EDX)

		inc	edx		; EDX <- width + 1
		imul	edx,edx,3	; multiply with 3
		shr	edx,2		; EDX <- /4, max. number of digits

; ------------- New minimal number of integer digits (-> EAX)

FltToTextBufN63:cmp	eax,edx		; check minimal number of digits
		ja	FltToTextBufN64	; number of digits is OK
		xchg	eax,edx		; EAX <- new number of digits

; ------------- Add thousand separators (-> ECX)

FltToTextBufN64:test	byte [FLTNForm+FORMPAR_Flags2],FORMFLAG2_Thsnd ; thsnd?
		jz	FltToTextBufN66	; not using thousand separator
		lea	edx,[eax-1]	; EDX <- number of digits - 1
		imul	edx,edx,43691	; EDX <- (digits-1) * 20000h/3
		shr	edx,17		; EDX <- number of separators
		add	eax,edx		; EAX <- add thousand separators
FltToTextBufN66:add	ecx,eax		; EAX <- add sign and fractional part

; ------------- Check field width

FltToTextBufN8:	movzx	eax,byte [FLTNForm+FORMPAR_Width] ; EAX<-minimal width
		cmp	eax,ecx		; check minimal width
		ja	FltToTextBufN84	; use minimal width
		xchg	eax,ecx		; EAX <- text length

; ------------- Pop registers

FltToTextBufN84:mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
FltToTextBufN9:	ret

; -----------------------------------------------------------------------------
;          Convert float number into text buffer in float point form
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	It uses 2 more registers from FPU stack.
;		FPU should be in default state - 64 bits, round to nearest.
;		To destroy FPU register you can use "ffreep st0" instruction.
;		On AMD 1.6 GHz it takes approx. "10*decimal places+200" nsec.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
%define		FLTNat     ebp+8	; (4) pointer to nationality NATIONAL
%define		FLTForm    ebp-4	; (4) formatting parameters FORMPAR
%define		FLTLen     ebp-8	; (4) total length of text
%define		FLTThsnd   ebp-12	; (4) thousand separator counter
%define		FLTDec     ebp-16	; (4) decimal separator counter
%define		FLTIntDig  ebp-20	; (4) number of digits of integer part
%define		FLTDigits  ebp-24	; (4) total number of digits
%define		FLTDigNum  ebp-25	; (1) digit counter
%define		FLTSign    ebp-26	; (1) stored flags, bit 7 = sign
%define		FLTExp     ebp-28	; (2) float number, exponent
					;       (bit 0..14) and sign (bit 15)
%define		FLTMantH   ebp-32	; (4) float number, mantissa HIGH
%define		FLTMantL   ebp-36	; (4) float number, mantissa LOW
%define		FLTFloat   FLTMantL	; ...(10) float number

%define		FLTStack   36		; stack size

; ------------- Macro - store decimal separator (destroys EAX)
; %1 = jump label if buffer is full

%macro		FLTDECIM 1

		dec	dword [FLTDec]	; decrement decimal separator
		jnz	%%L2		; not decimal separator
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Alt ; point?
		jz	%%L1		; not using decimal separator
		mov	eax,[FLTNat]	; EAX <- pointer to nationality
		mov	al,[eax+NAT_DecimalSep] ; AL <- decimal separator
		dec	esi		; decrease space counter
		stosb			; store decimal separator
		jz	short %1	; buffer is full
%%L1:		and	dword [FLTThsnd],byte 0 ; no thousand separator
%%L2:

%endmacro

; ------------- Macro - store thousand separator (destroys EAX)
; %1 = jump label if buffer is full

%macro		FLTTHSND 1

		dec	dword [FLTThsnd] ; decrement thousand separator counter
		jnz	%%L1		; not thousand separator
		mov	eax,[FLTNat]	; EAX <- pointer to nationality
		mov	al,[eax+NAT_ThsndSep] ; AL <- thousand separator
		dec	esi		; decrease space counter
		stosb			; store one digit
		jz	short %1	; buffer is full
		mov	byte [FLTThsnd],3 ; new thousand separator
%%L1:

%endmacro

; ------------- Check remaining free space in buffer

FltToTextBuf:	or	esi,esi		; check remaining space
		jle	short FltToTextBufN9 ; not enough free space

; ------------- Push registers

		push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX pointer to nationality
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,FLTStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	[FLTForm],ebx	; formatting parameters FORMPAR

; ------------- Store number into stack

		fld	st0		; duplicate number
		fstp	tword [FLTFloat] ; store number into stack

; ------------- Load exponent (-> EDX)

		fwait			; synchronize FPU
		movzx	edx,word [FLTExp] ; EDX <- exponent
		mov	ch,dh		; CH <- sign
		and	ch,B7		; left only sign flag
		mov	[FLTSign],ch	; push sign flag (bit 7)

; ------------- Absolute value (it uses copy of register)

		fld	st0		; duplicate number
		fabs			; absolute value
		and	dx,7fffh	; mask exponent (15 bits)
		jnz	FltToTextBuf2	; neither zero nor denormalized number

; ============= Special cases

; ------------- Zero or denormalized number

		mov	eax,[FLTMantL]	; EAX <- mantissa LOW
		or	eax,[FLTMantH]	; check if mantissa is zero
		jnz	FltToTextBuf2	; not zero, it is denormalized number

; ------------- Zero

		ffreep	st0		; pop register from the FPU stack
		xor	ebx,ebx		; EBX <- zero decimal exponent
	       test byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Alt2+FORMFLAG2_Prec
		jz	FltToTextBuf5	; don't truncate
FltToTextBuf1:	mov	byte [FLTForm+FORMPAR_Prec],0 ; limit precision
		jmp	FltToTextBuf5

; ------------- Infinity (+1.#INF, -1.#INF) or non-number (+1.#NAN, -1.#NAN)

FltToTextBuf2:	cmp	dx,7fffh	; infinity or non-number?
		jne	FltToTextBuf3	; neither infinity nor non-number
		ffreep	st0		; pop register from the FPU stack
		xchg	eax,ecx		; AH <- sign flag (bit 7)
		mov	ecx,[FLTNat]	; ECX <- pointer to nationality
		lea	edx,[FLTFloat]	; EDX <- pointer to float number
		call	FloatInfNan	; store text
		jz	FltToTextBuf9	; buffer is full
		jmp	FltToTextBuf86	; store trailing spaces

; ============= Convert number to digits

; ------------- Split number to mantissa and exponent (here is EDX=bin. exp.)

FltToTextBuf3:	lea	ecx,[FLTFloat]	; ECX <- pointer to float in stack
		call	FloatSplit	; split number to mantissa and exponent

;-------------- Round, denormalize and convert mantissa to BCD (EBX = exponent)

		movzx	eax,byte [FLTForm+FORMPAR_Prec] ; EAX <- precision
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Prec ; precis.?
		jz	FltToTextBuf4	; normal precision (=fractional digits)
		dec	eax		; without first digit
		jns	FltToTextBuf42	; digits are OK
		jmp	FltToTextBuf41	; limit to 0
FltToTextBuf4:	add	eax,ebx		; EAX <- position of last digit
		jg	FltToTextBuf42	; digits are OK
FltToTextBuf41:	xor	eax,eax		; EAX <- 0, digits are out of display
FltToTextBuf42:	lea	ecx,[FLTFloat]	; ECX <- pointer to float in stack
		bt	dword [FLTForm],FORMTYPE_Cap_b ; round always down?
		call	FloatMantBCD	; process mantissa

; ------------- Use precision digits (here is EBX = decimal exponent)

		lea	edx,[FLTFloat+9] ; EDX <- last digit
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Prec ; precis.?
		jz	FltToTextBuf43	; normal precision (=fractional digits)
		call	FloatTrunc	; truncate trailing zeros
		mov	[FLTForm+FORMPAR_Prec],cl ; set new precision
		sub	ecx,ebx		; ECX <- maximal precision
		jg	FltToTextBuf52	; precision is positive
		xor	ecx,ecx		; ECX <- limit precision
		jmp	FltToTextBuf52

; ------------- Truncate trailing zeros (here is EBX = decimal exponent)

FltToTextBuf43:	test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Alt2 ; trunc.?
		jz	FltToTextBuf5	; don't truncate
		movzx	eax,byte [FLTForm+FORMPAR_Prec] ; EAX <- precision
		add	eax,ebx		; check if number is out of display
		jl	FltToTextBuf1	; number is not visible
		call	FloatTrunc	; truncate trailing zeros
		sub	ecx,ebx		; ECX <- maximal precision
		jg	FltToTextBuf44	; precision is positive
		xor	ecx,ecx		; ECX <- limit precision
FltToTextBuf44:	movzx	eax,byte [FLTForm+FORMPAR_Prec] ; EAX <- precision
		cmp	ecx,eax		; check precision
		jae	FltToTextBuf5	; precision is OK
		mov	[FLTForm+FORMPAR_Prec],cl ; limit precision

; ============= Get text length (here is EBX = decimal exponent)

; ------------- Add sign (-> EAX)

FltToTextBuf5:	movzx	ecx,byte [FLTForm+FORMPAR_Prec]	; ECX <- num. of digits
FltToTextBuf52:	mov	al,[FLTSign]	; AL <- sign (bit 7)
		mov	ah,[FLTForm+FORMPAR_Flags1] ; AH <- flags 1
		and	ax,B7 + (FORMFLAG1_Sign + FORMFLAG1_Space)*256
		setnz	al		; AL <- 1 if sign, else 0
		movzx	eax,al		; EAX <- 1 if sign, else 0

; ------------- Add decimal point and precision (-> ECX)

		mov	[FLTDigits],ecx	; prepare total number of digits
		jecxz	FltToTextBuf62	; no digits after decimal point
		bts	dword [FLTForm],FORMFLAG_Alt_b ; set dec. point flag
FltToTextBuf62:	bt	dword [FLTForm],FORMFLAG_Alt_b ; use decimal point?
		adc	ecx,eax		; ECX <- add decimal point and sign

; ------------- Prepare minimal integer part (-> EAX)

		xor	eax,eax		; EAX <- 0
		or	ebx,ebx		; is exponent negative?
		sets	al		; AL <- 1 if < 1.0, 0 if >= 1.0
		dec	eax		; EAX <- 0 if < 1.0, -1 if >= 1.0
		and	eax,ebx		; EAX <- 0 if < 1.0, exponent if >= 1.0
		inc	eax		; EAX <- add first digit

; ------------- Check if use zeros instead of spaces

		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Zero ; zero?
		jz	FltToTextBuf64	; no zeros

; ------------- Prepare minimal integer part if use zeros (-> EDX)

		movzx	edx,byte [FLTForm+FORMPAR_Width] ; EDX <- minimal width
		sub	edx,ecx		; EDX <- without sign and fractional
		jle	FltToTextBuf64	; not enough space

; ------------- Check if use thousand separator

		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Thsnd ; thsnd?
		jz	FltToTextBuf63	; not using thousand separator

; ------------- Number of digits without thousand separators (-> EDX)

		inc	edx		; EDX <- width + 1
		imul	edx,edx,3	; multiply with 3
		shr	edx,2		; EDX <- /4, max. number of digits

; ------------- New minimal number of integer digits (-> EAX)

FltToTextBuf63:	cmp	eax,edx		; check minimal number of digits
		ja	FltToTextBuf64	; number of digits is OK
		xchg	eax,edx		; EAX <- new number of digits
FltToTextBuf64:	mov	[FLTIntDig],eax	; store number of digits of integer
		add	[FLTDigits],eax	; set total number of digits
		mov	[FLTDec],eax	; init decimal separator counter

; ------------- Add thousand separators and prepare thousand counter (-> EDX)

		xor	edx,edx		; EDX <- 0, no thousand counter
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Thsnd ; thsnd?
		jz	FltToTextBuf66	; not using thousand separator
		lea	edx,[eax-1]	; EDX <- number of digits - 1
		imul	edx,edx,43691	; EDX <- (digits-1) * 20000h/3
		shr	edx,17		; EDX <- number of separators
		add	eax,edx		; EAX <- add thousand separators
		shl	edx,2		; EDX <- characters in whole groups
		neg	edx		; EDX <- -characters in whole groups
		add	edx,eax		; EDX <- digits in first group
FltToTextBuf66:	add	eax,ecx		; EAX <- add sign and fractional part
		mov	[FLTThsnd],edx	; set thousand separator counter
		mov	[FLTLen],eax	; set length of text
		xchg	eax,edx		; EDX <- length of text

; ============= Decode spaces, sign, zeros (EDX=text length, EBX=exponent)

; ------------- Store leading spaces

		test	byte [FLTForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	FltToTextBuf72	; left-justify, no spaces
		movzx	ecx,byte [FLTForm+FORMPAR_Width] ; ECX <- minimal width
		sub	ecx,edx		; ECX <- remaining spaces
		jle	FltToTextBuf72	; no spaces left
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	FltToTextBuf7	; no, right-justify
		shr	ecx,1		; ECX <- spaces / 2, round down
		jz	FltToTextBuf72	; no spaces left
FltToTextBuf7:	mov	al," "		; AL <- space
FltToTextBuf71:	dec	esi		; decrease space counter
		stosb			; store one space
		loopnz	FltToTextBuf71	; next character
		jz	short FltToTextBuf759 ; buffer is full

; ------------- Store sign

FltToTextBuf72:	mov	al,"-"		; AL <- negative sign
		test	byte [FLTSign],B7 ; negative number?
		jnz	FltToTextBuf73	; negative number
		test	byte [FLTForm+FORMPAR_Flags1],FORMFLAG1_Sign ; sign?
		mov	al,"+"		; AL <- positive sign
		jnz	FltToTextBuf73	; use positive sign
		test	byte [FLTForm+FORMPAR_Flags1],FORMFLAG1_Space ; space?
		jz	FltToTextBuf74	; no space
		mov	al," "		; AL <- space
FltToTextBuf73:	dec	esi		; decrease space counter
		stosb			; store sign
		jz	short FltToTextBuf759 ; buffer is full

; ------------- Store leading zeros

FltToTextBuf74:	mov	ecx,[FLTIntDig]	; ECX <- digits in integer part
		dec	ecx		; without first digit
		sub	ecx,ebx		; ECX <- rest of digits
		jle	FltToTextBuf76	; no zeros
		sub	[FLTDigits],ecx	; decrease total number of digits
		jns	FltToTextBuf75	; number of digits is OK
		xor	eax,eax		; EAX <- 0
		xchg	eax,[FLTDigits]	; no digits remain
		add	ecx,eax		; EXC <- decrease number of digits
		jle	FltToTextBuf76	; no zeros
FltToTextBuf75:	mov	al,"0"		; AL <- zero
		dec	esi		; decrease space counter
		stosb			; store one digit
FltToTextBuf759:jz	FltToTextBuf9	; buffer is full
		FLTDECIM FltToTextBuf759 ; store decimal separator
		FLTTHSND FltToTextBuf759 ; store thousand separator
		loop	FltToTextBuf75	; next character

; ============= Decode mantissa (here is EBX=decimal exponent)

FltToTextBuf76:	lea	edx,[FLTFloat]	; EDX <- pointer to mantissa

; ------------- Prepare number of digits of mantissa (-> CL)

		movzx	ecx,byte [FLTForm+FORMPAR_Prec] ; ECX <- precision
		inc	ecx		; add first digit
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Prec ; precis.?
		jnz	FltToTextBuf766	; use full precision
		add	ecx,ebx		; ECX <- add exponent
		jle	FltToTextBuf79	; no digits
FltToTextBuf766:cmp	ecx,19		; check maximal number of digits
		jb	FltToTextBuf77	; number of digits is OK
		xor	ecx,ecx		; ECX <- 0
		mov	cl,19		; ECX <- limit number of digits
FltToTextBuf77:	sub	[FLTDigits],ecx	; decrease total number of digits
		mov	[FLTDigNum],cl	; prepare number of digits

; ------------- Store mantissa - first digit

FltToTextBuf78:	mov	al,[edx]	; AL <- first digit
		and	al,0fh		; AL <- nask digit
		add	al,"0"		; AL <- convert to character
		dec	esi		; decrease space counter
		stosb			; store one digit
		jz	short FltToTextBuf7B ; buffer is full
		FLTDECIM FltToTextBuf7B ; store decimal separator
		FLTTHSND FltToTextBuf7B ; store thousand separator
		dec	byte [FLTDigNum] ; decrease digit counter
		jz	FltToTextBuf79	; no next character

; ------------- Store mantissa - second digit

		mov	al,[edx]	; AL <- 2 digits
		shr	al,4		; AL <- mask digit
		add	al,"0"		; AL <- convert to character + carry
		dec	esi		; decrease space counter
		stosb			; store one digit
		jz	short FltToTextBuf7B ; buffer is full
		FLTDECIM FltToTextBuf7B	; store decimal separator
		FLTTHSND FltToTextBuf7B	; store thousand separator
		inc	edx		; EDX <- shift pointer
		dec	byte [FLTDigNum] ; decrease digit counter
		jnz	FltToTextBuf78	; next character

; ------------- Store trailing zeros

FltToTextBuf79:	mov	ecx,[FLTDigits]	; ECX <- remaining digits
		jecxz	FltToTextBuf8	; no digits left
FltToTextBuf7A:	mov	al,"0"		; AL <- zero
		dec	esi		; decrease space counter
		stosb			; store one digit
FltToTextBuf7B:	jz	short FltToTextBuf9 ; buffer is full
		FLTDECIM FltToTextBuf9	; store decimal separator
		FLTTHSND FltToTextBuf9	; store thousand separator
		loop	FltToTextBuf7A	; next character

; ------------- Store trailing spaces

FltToTextBuf8:	test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Zero ; zeros?
		jnz	short FltToTextBuf9 ; zeros, no spaces
		mov	eax,[FLTLen]	; EAX <- text length
FltToTextBuf86:	movzx	ecx,byte [FLTForm+FORMPAR_Width] ; ECX <- min. width
		sub	ecx,eax		; ECX <- remaining spaces
		jle	short FltToTextBuf9 ; no spaces remain
		test	byte [FLTForm+FORMPAR_Flags1],FORMFLAG1_Left ; left?
		jnz	FltToTextBuf87	; left-justify
		test	byte [FLTForm+FORMPAR_Flags2],FORMFLAG2_Cent ; center?
		jz	short FltToTextBuf9 ; no center
		shr	ecx,1		; ECX <- spaces / 2
		adc	cl,ch		; ECX <- round up
		jz	short FltToTextBuf9 ; no spaces
FltToTextBuf87:	mov	al," "		; AL <- space
FltToTextBuf88:	stosb			; store one space
		dec	esi		; decrease space counter
		loopnz	FltToTextBuf88	; next character

; ------------- Pop registers

FltToTextBuf9:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Macro - convert float number in mixed form
; -----------------------------------------------------------------------------

%macro		MIXTOTEXTBUF 2
; parameter %1 = floating point function, %2 = exponential function

; ------------- Push registers

		push	ebx		; push EBX
		push	eax		; push EAX

; ------------- Convert flags

		btc	ebx,FORMFLAG_Prec_b ; invert precision flag
		mov	eax,ebx		; EAX <- flags
		shr	eax,FORMFLAG_Alt_b ; EAX <- alternate bits
		and	eax,B0+B1	; EAX <- alternate bits
		mov	eax,[MixFormFlag+eax] ; EAX <- new bits
		and	ebx,~((FORMFLAG2_Alt+FORMFLAG2_Alt2)*256*256) ; clear
		shl	eax,FORMFLAG_Alt_b ; EAX <- new bits
		or	ebx,eax		; set new flags

; ------------- Check minimal value 1.0e-4

		fld	st0		; duplicate number
		fabs			; absolute value
		fld	tword [MixFormMin] ; load minimal value 1.0e-4
		fcomp	st1		; check minimal value
		fstsw	ax		; AX <- FPU flags
		sahf			; get FPU flags
		ja	%%L4		; use exponential form

; ------------- Check maximal value 1.0e+precision

		xor	eax,eax		; EAX <- 0
		mov	al,10		; EAX <- default exponent 1e10
		bt	ebx,FORMFLAG_Prec_b ; alternate precision?
		jnc	%%L2		; use alternate precision (=fractional)
		cmp	bl,20		; maximal precision
		jae	%%L4		; use exponential form
		movzx	eax,bl		; EAX <- precision
%%L2:		ficomp	dword [MixFormMax+eax*4] ; check maximal value
		fstsw	ax		; AX <- FPU flags
		sahf			; get FPU flags
		jae	%%L6		; use exponential form
		
; ------------- Use float point form

		pop	eax		; pop EAX
		call	%1		; use float point form
		pop	ebx		; pop EBX
		ret

; ------------- Use exponential form

%%L4:		ffreep	st0		; free number
%%L6:		pop	eax		; pop EAX
		call	%2		; use exponential form
		pop	ebx		; pop EBX
		ret
%endmacro

; -----------------------------------------------------------------------------
;     Convert float number into text buffer in mixed form - get text length
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = text length
; NOTES:	It uses float point form in range 1e-4 to 1e+precision,
;			else uses exponential form.
;		It uses 2 more registers from FPU stack.
;		FPU should be in default state - 64 bits, round to nearest.
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------

MixToTextBufN:	MIXTOTEXTBUF FltToTextBufN, ExpToTextBufN ; get text length

; -----------------------------------------------------------------------------
;             Convert float number into text buffer in mixed form
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = formatting parameters FORMPAR
;		ECX = pointer to nationality descriptor NATIONAL
;		ESI = remaining free space in buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	It uses float point form in range 1e-4 to 1e+precision,
;			else uses exponential form.
;		It uses 2 more registers from FPU stack.
;		FPU should be in default state - 64 bits, round to nearest.
;		To destroy FPU register you can use "ffreep st0" instruction.
;		On AMD 1.6 GHz it takes approx. "10*decimal places+200" nsec.
; -----------------------------------------------------------------------------

MixSToTextBuf:	MIXTOTEXTBUF FltToTextBuf, ExpSToTextBuf ; use small "e"

MixCToTextBuf:	MIXTOTEXTBUF FltToTextBuf, ExpCToTextBuf ; use capital "E"

; -----------------------------------------------------------------------------
;                    Get argument length in argument stack
; -----------------------------------------------------------------------------
; INPUT:	EBX = formatting parameters FORMPAR
; OUTPUT:	EAX = argument length (in DWORDs: 1, 2 or 3)
; NOTES:	64-bit integer or double float takes 2 DWORDs, extended double
;		takes 3 DWORDs and other numbers take 1 DWORD.
; -----------------------------------------------------------------------------

; ------------- Push registers

FormGetArgLen:	push	ebx		; push EBX

; ------------- Prepare minimal argument length in DWORDs (-> EAX)

		xor	eax,eax		; EAX <- 0
		inc	eax		; EAX <- 1, default length 1 DWORD

; ------------- Prepare argument type (-> BL) and flags (-> BH)

		shr	ebx,FORMPAR_TypeF_b ; EBX <- type and flags
		and	bl,FORMTYPE_Mask0 ; BL <- argument type

; ------------- Argument type "always 1 DWORD"

		cmp	bl,FORMTYPE_ArgDW ; is it 1 DWORD argument type?
		jbe	FormGetArgLen4	; it is 1 DWORD argument type

; ------------- Argument length if "double float"

		cmp	bl,FORMTYPE_ArgInt ; integer argument?
		jbe	FormGetArgLen2	; integer argument
		test	bh,FORMFLAG2_Shrt ; short argument?
		jnz	FormGetArgLen2	; yes, short argument (single float)
		inc	eax		; AL <- 2, argument length for "double"

; ------------- Increase argument length if "long" flag is set

FormGetArgLen2:	test	bh,FORMFLAG2_Long ; long argument?
		jz	FormGetArgLen4	; not long argument
		inc	eax		; increase argument length for "long"

; ------------- Pop registers

FormGetArgLen4:	pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                        Format text into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = callback value
;		EBX = size of source text to be formatted
;		ECX = pointer to nationality descriptor NATIONAL
;		EDX = source text to be formatted
;		ESI = remaining free space in destination buffer
;		EDI = destination buffer
;		EBP = callback function to read argument
;			INPUT:	EAX = callback value
;				EBX = formatting parameters with argument type
;				EDX = DWORD index in argument stack (0 to 255)
;			OUTPUT: CY=error, invalid argument index
;				EAX = width or precision parameter
;				EDX:EAX integer
;				EAX character UNICODE
;				ST0 floating point (only if NC)
;				EAX pointer to text UTF-8, EDX length of text
;			DESTROYS: EBX, ECX, EDX:EAX (if not returning a value)
; OUTPUT:	CY = error, invalid argument index
;		ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	64-bit integer or double float takes 2 DWORDs, extended double
;		takes 3 DWORDs and other numbers take 1 DWORD. Argument index
;		refers to index of DWORD argument, not to real argument number.
; 		To get user default nationality use DEFAULT_NAT macro.
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
;
;  EAX = temporary register
;  EBX = formatting parameters
;  ECX = remaining size of source text
;  EDX = source text
;  ESI = remaining free space in destination buffer
;  EDI = destination buffer

%define		FORMVal		ebp+16	; (4) callback value
%define		FORMNat		ebp+8	; (4) pointer to nationality
%define		FORMCall	ebp+0	; (4) callback function
%define		FORMInx		ebp-1	; (1) current argument index
%define		FORMExpInx	ebp-2	; (1) explicit argument index
%define		FORMNum		ebp-3	; (1) read number
%define		FORMFlag	ebp-4	; (1) flags (see below)
%define		FORMFlagNum	FORMFlag; (2) flags and number
%define		FORMForm	ebp-8	; (4) formatting parameters

%define		FORMStack	8	; stack size

; Flags:
%define		FORMFLAG_NUM	B0	; start read number
%define		FORMFLAG_PREC	B1	; precision has been started
%define		FORMFLAG_EXP	B2	; explicit argument index valid
%define		FORMFLAG_INXW	B3	; get width from arguments
%define		FORMFLAG_INXP	B4	; get precision from arguments
%define		FORMFLAG_PAR	B5	; get parameter "*"
%define		FORMFLAG_PSET	B6	; precision has been set

; ------------- Push registers

FormToTextBuf:	push	eax		; push EAX (callback value)
		push	ebx		; push EBX
		push	ecx		; push ECX (pointer to nationality)
		push	edx		; push EDX
		push	ebp		; push EBP (callback function)
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,FORMStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	byte [FORMInx],0 ; clear current argument index
		mov	ecx,ebx		; ECX <- source text counter

; ------------- Check remaining free space in buffer

FormToTextBuf1:	or	esi,esi		; check remaining space
		jle	short FormToTextBuf8 ; not enough free space

; ------------- Read one character (-> AL)

FormToTextBuf2:	dec	ecx		; check next source character
		js	FormToTextBuf7	; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Check if it is switch character

		cmp	al,"%"		; is it switch character?
		je	FormToTextBuf4	; switch character

; ------------- Store one character into destination buffer

FormToTextBuf3:	dec	esi		; decrease free space
		stosb			; store character into buffer
		jnz	FormToTextBuf2	; next character

; ------------- Pop registers

FormToTextBuf7:	clc			; clear error flag
FormToTextBuf8:	mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Prepare variables to read formatting parameters

FormToTextBuf4:	xor	ebx,ebx		; EBX <- 0, formatting parameters
		mov	[FORMFlagNum],bx ; clear read number and flags

; ------------- Read one character (-> AL)

FormToTextBuf5:	dec	ecx		; check next source character
		js	FormToTextBuf7	; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Jump to character service

		cmp	al,32		; minimal character (space character)
		jb	FormToTextBuf3	; invalid character
		cmp	al,127		; maximal character
		jae	FormToTextBuf3	; invalid character
		movzx	eax,al		; EAX <- character
		jmp	[FormTextJumpTab+eax*4-32*4] ; jump

; ------------- Left-justify "-"

FormToTextLeft:	bts	ebx,FORMFLAG_Left_b ; set flag "left-justify"
		jmp	short FormToTextBuf5 ; next character

; ------------- Always use sign "+"

FormToTextSign:	bts	ebx,FORMFLAG_Sign_b ; set flag "sign"
		jmp	short FormToTextBuf5 ; next character

; ------------- Prefix space if number is positive " "

FormToTextSpace:bts	ebx,FORMFLAG_Spc_b ; set flag "space"
		jmp	short FormToTextBuf5 ; next character

; ------------- Thousand separator "~"

FormToTextThsnd:bts	ebx,FORMFLAG_Thsn_b ; set flag "thousand"
		jmp	short FormToTextBuf5 ; next character

; ------------- Center "@"

FormToTextCent:	bts	ebx,FORMFLAG_Cent_b ; set flag "center"
		jmp	short FormToTextBuf5 ; next character

; ------------- Alternate form "#" and "##"

FormToTextAlt:	btc	ebx,FORMFLAG_Alt_b ; alternate form
		jnc	short FormToTextBuf5 ; next character
		btc	ebx,FORMFLAG_Alt2_b ; flip alternate form 2
		jmp	short FormToTextBuf5 ; next character

; ------------- Precision "."

FormToTextPrec:	test	byte [FORMFlag],FORMFLAG_PREC ; precision?
		jnz	FormToTextPrec2	; precision has been started
		or	byte [FORMFlag],FORMFLAG_PREC ; start precision
		call	FormToTextStopW	; stop parsing width
		jmp	short FormToTextBuf5 ; next character

FormToTextPrec2:bts	ebx,FORMFLAG_Prec_b ; alternate precision
		jmp	short FormToTextBuf5 ; next character

; ------------- Integer size "I16", "I32", "I64"

FormToTextSize:	sub	ecx,byte 2	; check number of characters
		js	short FormToTextBuf7 ; end of source text
		mov	ax,[edx]	; AX <- get 2 characters
		inc	edx		; increase pointer
		inc	edx		; increase pointer
		cmp	ax,"64"		; long variant?
		je	short FormToTextLong ; long variant
		cmp	ax,"16"		; short variant?
		je	short FormToTextShrt ; short variant
		cmp	ax,"32"		; default variant?
		jne	FormToTextSize2	; no default variant
		btr	ebx,FORMFLAG_Long_b ; reset flag "long"
		btr	ebx,FORMFLAG_Shrt_b ; reset flag "short"
FormToTextBuf52:jmp	short FormToTextBuf5 ; next character

FormToTextSize2:inc	ecx		; increase number of characters
		dec	edx		; decrease pointer
		inc	ecx		; increase number of characters
		dec	edx		; decrease pointer
		jmp	short FormToTextBuf52 ; next character

; ------------- Digit "0" to "9"

FormToTextDig0:	test	byte [FORMFlag],FORMFLAG_NUM+FORMFLAG_PREC ; first 0?
		jz	short FormToTextZero ; first zero digit
FormToTextDig:	or	byte [FORMFlag],FORMFLAG_NUM ; start number
		mov	ah,[FORMNum]	; AH <- number
		sub	al,"0"		; AL <- digit
		cmp	ah,25		; check overflow
		jae	FormToTextDig4	; it will overflow or maybe it will not
		aad			; AL <- AL + AH*10
FormToTextDig2:	mov	[FORMNum],al	; store new number
		jmp	short FormToTextBuf52 ; next character

FormToTextDig4:	ja	FormToTextDig6	; it will overflow
		aad			; AL <- AL + AH*10
		cmp	al,250		; check overflow
		jae	FormToTextDig2	; parameter is OK
FormToTextDig6:	mov	al,255		; AL <- limit on overflow
		jmp	short FormToTextDig2 ; next character

; ------------- Add zeros instead of spaces "0"

FormToTextZero:	bts	ebx,FORMFLAG_Zero_b ; set flag "zero"
		jmp	short FormToTextBuf52 ; next character

; ------------- Get parameter "*"

FormToTextArg:	or	byte [FORMFlag],FORMFLAG_PAR ; set flag "parameter"
		jmp	short FormToTextBuf52 ; next character

; ------------- Long variant "l", "L"

FormToTextLong:	bts	ebx,FORMFLAG_Long_b ; set flag "long"
		btr	ebx,FORMFLAG_Shrt_b ; reset flag "short"
		jmp	short FormToTextBuf52 ; next character

; ------------- Short variant "h", "H"

FormToTextShrt:	btr	ebx,FORMFLAG_Long_b ; reset flag "long"
		bts	ebx,FORMFLAG_Shrt_b ; set flag "short"
		jmp	short FormToTextBuf52 ; next character

; ------------- Indexed argument "$"

FormToTextInx:	mov	al,0		; AL <- 0
		xchg	al,[FORMNum]	; AL <- number, clear number
		and	byte [FORMFlag],~FORMFLAG_NUM ; stop number
		or	al,al		; is index valid?
		jz	FormToTextBuf54	; index is not valid
		dec	eax		; AL <- correct index value
		test	byte [FORMFlag],FORMFLAG_PAR ; get "parameter"?
		jnz	FormToTextInx2	; yes, parameter is valid
		mov	[FORMExpInx],al	; store explicit argument index
		or	byte [FORMFlag],FORMFLAG_EXP ; explicit argument valid
FormToTextBuf54:jmp	short FormToTextBuf52 ; next character

FormToTextInx2:	and	byte [FORMFlag],~FORMFLAG_PAR ; clear "parameter"
		test	byte [FORMFlag],FORMFLAG_PREC ; precision?
		jnz	FormToTextInx4	; precision has been started
		mov	bh,al		; store width index
		or	byte [FORMFlag],FORMFLAG_INXW ; width from arguments
		jmp	short FormToTextBuf54 ; next character

FormToTextInx4:	mov	bl,al		; store precision index
		or	byte [FORMFlag],(FORMFLAG_INXP+FORMFLAG_PSET)
		jmp	short FormToTextBuf54 ; next character

; ------------- Signed decimal int "d", "D", "i"

FormToTextDC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextDS:
FormToTextIS:	or	ebx,(FORMTYPE_Int<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr2 ; invalid argument index
		jnz	FormToTextDS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextDS2:	push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	IntToTextBuf	; convert signed INT into buffer
		pop	ecx		; pop ECX
		pop	edx		; pop EDX
		jmp	short FormToTextOK2 ; read next character

; ------------- Unsigned decimal int "u", "U"

FormToTextUC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextUS:	or	ebx,(FORMTYPE_UInt<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr2 ; invalid argument index
		jnz	FormToTextUS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextUS2:	push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	UIntToTextBuf	; convert unsigned INT into buffer
		pop	ecx		; pop ECX
		clc
FormToTextErr2:	pop	edx		; pop EDX
		jc	FormToTextBuf8	; invalid argument index
FormToTextOK2:	jmp	FormToTextBuf1	; read next character

; ------------- Unsigned binary int "b", "B"

FormToTextBC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextBS:	or	ebx,(FORMTYPE_Bin<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr2 ; invalid argument index
		jnz	FormToTextBS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextBS2:	push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	BinToTextBuf	; convert BIN number into buffer
		pop	ecx		; pop ECX
		pop	edx		; pop EDX
		jmp	short FormToTextOK2 ; read next character

; ------------- Unsigned octal int "o", "O"

FormToTextOC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextOS:	or	ebx,(FORMTYPE_Oct<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr2 ; invalid argument index
		jnz	FormToTextOS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextOS2:	call	OctToTextBuf	; convert OCT number into buffer
		pop	edx		; op EDX
		jmp	short FormToTextOK2 ; read next character

; ------------- Unsigned hex int "x"

FormToTextXS:	or	ebx,(FORMTYPE_Hex<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr2 ; invalid argument index
		jnz	FormToTextXS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextXS2:	push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	HexSToTextBuf	; convert HEX small into buffer
		pop	ecx		; pop ECX
		pop	edx		; pop EDX
		jmp	short FormToTextOK ; read next character

; ------------- Unsigned hex int "X"

FormToTextXC:	or	ebx,((FORMTYPE_Hex+FORMTYPE_Cap)<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr ; invalid argument index
		jnz	FormToTextXC2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTextXC2:	push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	HexCToTextBuf	; convert HEX capital into buffer
		pop	ecx		; pop ECX
		pop	edx		; pop EDX
		jmp	short FormToTextOK ; read next character

; ------------- Character "C", "c"

FormToTextCC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextCS:	or	ebx,(FORMTYPE_Char<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr ; invalid argument index
		pop	edx		; pop EDX
		xchg	ebp,esi		; EBP <- remaining space
		call	CharUTF8Write	; store character into buffer
		xchg	ebp,esi		; ESI <- remaining space
		jmp	short FormToTextOK ; read next character

; ------------- String "S", "s"

FormToTextSC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextSS:	or	ebx,(FORMTYPE_String<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr ; invalid argument index
		call	FormToTextLim	; limit text length by precision
		push	ecx		; push ECX
		mov	ecx,edx		; ECX <- length of text
		cmp	ecx,esi		; check remaining space
		jl	FormToTextSS2	; text length is OK
		mov	ecx,esi		; ECX <- limit text length
FormToTextSS2:	sub	esi,ecx		; ESI <- new remaining text
		push	esi		; push ESI
		xchg	eax,esi		; ESI <- pointer to text
		mov	eax,ecx		; EAX <- text length
		shr	ecx,2		; ECX <- text length in DWORDs
		rep	movsd		; copy text in DWORDs
		and	eax,byte 3	; EAX <- size in last DWORD
		xchg	eax,ecx		; ECX <- size in last DWORD
		rep	movsb		; copy rest of text
		pop	esi		; pop ESI
		pop	ecx		; pop ECX (here is NC)
FormToTextErr:	pop	edx		; pop EDX
		jc	FormToTextBuf8	; invalid argument index
FormToTextOK:	jmp	FormToTextBuf1	; read next character

; ------------- Floating point "f", "F"

FormToTextFC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTextFS:	or	ebx,(FORMTYPE_Float<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr ; invalid argument index
		jnz	FormToTextFS2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTextFS2:	pop	edx		; pop EDX
		push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	FltToTextBuf	; convert float number into buffer
		pop	ecx		; pop ECX
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTextOK ; read next character

; ------------- Exponential "e"

FormToTextES:	or	ebx,(FORMTYPE_Exp<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr ; invalid argument index
		jnz	FormToTextES2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTextES2:	pop	edx		; pop EDX
		push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	ExpSToTextBuf	; convert exponent small into buffer
		pop	ecx		; pop ECX
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTextOK ; read next character

; ------------- Exponential "E"

FormToTextEC:	or	ebx,((FORMTYPE_Exp+FORMTYPE_Cap)<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
FormToTextErr3:	jc	short FormToTextErr ; invalid argument index
		jnz	FormToTextEC2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTextEC2:	pop	edx		; pop EDX
		push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	ExpCToTextBuf	; convert exponent capital into buffer
		pop	ecx		; pop ECX
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTextOK ; read next character

; ------------- Mixed "g"

FormToTextGS:	or	ebx,(FORMTYPE_Mix<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr3 ; invalid argument index
		jnz	FormToTextGS2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTextGS2:	pop	edx		; pop EDX
		push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	MixSToTextBuf	; convert mixed small into buffer
		pop	ecx		; pop ECX
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTextOK3 ; read next character

; ------------- Mixed "G"

FormToTextGC:	or	ebx,((FORMTYPE_Mix+FORMTYPE_Cap)<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTextErr3 ; invalid argument index
		jnz	FormToTextGC2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTextGC2:	pop	edx		; pop EDX
		push	ecx		; push ECX
		mov	ecx,[FORMNat]	; ECX <- pointer to nationality
		call	MixCToTextBuf	; convert mixed capital into buffer
		pop	ecx		; pop ECX
		ffreep	st0		; free number from FPU stack
FormToTextOK3:	jmp	FormToTextBuf1	; read next character

; -----------------------------------------------------------------------------
;           Local function - limit text length by precision parameter
; -----------------------------------------------------------------------------

FormToTextLim:	or	bl,bl		; limit text length?
		jz	FormToTextLim9	; length not limited

		push	ecx		; push ECX
		push	esi		; push ESI
		push	eax		; push EAX (start of text)

		mov	ecx,edx		; ECX <- text length
		xchg	eax,esi		; ESI <- source text

FormToTextLim2:	jecxz	FormToTextLim4	; no characters remain
		call	CharUTF8Read	; read one character
		dec	bl		; character counter
		jnz	FormToTextLim2	; get next character

FormToTextLim4:	mov	edx,esi		; EDX <- end of text
		pop	eax		; pop EAX (start of text)
		sub	edx,eax		; EDX <- new text length
		pop	esi		; pop ESI
		pop	ecx		; pop ECX

FormToTextLim9:	ret

; -----------------------------------------------------------------------------
;                     Local function - stop parsing width
; -----------------------------------------------------------------------------

FormToTextStopW:mov	al,0		; AL <- 0
		xchg	al,[FORMNum]	; AL <- number, clear number
		test	byte [FORMFlag],FORMFLAG_INXW ; width from arguments?
		jnz	FormToTextStpW4	; width is already set
		mov	bh,al		; store width (or index)
		test	byte [FORMFlag],FORMFLAG_PAR ; get parameter?
		jz	short FormToTextStpW4 ; no
		mov	bh,[FORMInx]	; BH <- current argument index
		inc	byte [FORMInx]	; increase argument index
		or	byte [FORMFlag],FORMFLAG_INXW ; width from arguments
FormToTextStpW4:and	byte [FORMFlag],~(FORMFLAG_PAR+FORMFLAG_NUM); clr flags
		ret

; -----------------------------------------------------------------------------
;                  Local function - stop parsing precision
; -----------------------------------------------------------------------------

FormToTextStopP:test	byte [FORMFlag],FORMFLAG_INXP ; prec. from arguments?
		jnz	FormToTextStpP4	; precision is already set
		test	byte [FORMFlag],FORMFLAG_NUM ; number entered?
		jz	FormToTextStpP2	; not valid
		mov	bl,[FORMNum]	; store precision (or index)
		or	byte [FORMFlag],FORMFLAG_PSET ; precision has been set
FormToTextStpP2:test	byte [FORMFlag],FORMFLAG_PAR ; get parameter?
		jz	short FormToTextStpP4 ; no
		mov	bl,[FORMInx]	; BL <- current argument index
		inc	byte [FORMInx]	; increase argument index
		or	byte [FORMFlag],(FORMFLAG_INXP+FORMFLAG_PSET)
FormToTextStpP4:ret

; -----------------------------------------------------------------------------
;            Local function - get parameters (output EAX, EDX, ST0)
; -----------------------------------------------------------------------------
; Output: CY=invalid argument index, NZ(and NC)=no default precision

; ------------- Push registers

FormToTextGet:	push	ecx		; push ECX

; ------------- Stop parsing

		test	byte [FORMFlag],FORMFLAG_PREC ; precision?
		jnz	FormToTextGet2	; precision has been started
		call	FormToTextStopW	; stop parsing width
		jmp	short FormToTextGet4
FormToTextGet2:	call	FormToTextStopP	; stop parsing precision

; ------------- Prepare width

FormToTextGet4:	test	byte [FORMFlag],FORMFLAG_INXW ; width from arguments?
		jz	FormToTextGet5	; no width from arguments
		push	ebx		; push EBX
		movzx	edx,bh		; EDX <- argument index
		and	ebx,~(FORMTYPE_Mask<<FORMPAR_TypeF_b) ; clear arg. type
		mov	eax,[FORMVal]	; EAX <- callback value
		call	dword [FORMCall] ; get argument
		pop	ebx		; pop EBX
		jc	short FormToTextGet9 ; invalid argument index
		mov	bh,255		; limit width
		cmp	eax,255		; check maximal width
		ja	FormToTextGet5	; overflow
		mov	bh,al		; BH <- width from argument

; ------------- Prepare precision

FormToTextGet5:	test	byte [FORMFlag],FORMFLAG_INXP ; prec. from arguments?
		jz	FormToTextGet6	; no precision from arguments
		push	ebx		; push EBX
		movzx	edx,bl		; EDX <- argument index
		and	ebx,~(FORMTYPE_Mask<<FORMPAR_TypeF_b) ; clear arg. type
		mov	eax,[FORMVal]	; EAX <- callback value
		call	dword [FORMCall] ; get argument
		pop	ebx		; pop EBX
		jc	short FormToTextGet9 ; invalid argument index
		mov	bl,255		; limit precision
		cmp	eax,255		; check maximal precision
		ja	FormToTextGet6	; overflow
		mov	bl,al		; BL <- precision from argument

; ------------- Prepare argument index

FormToTextGet6:	movzx	edx,byte [FORMExpInx] ; EDX <- explicit argument index
		test	byte [FORMFlag],FORMFLAG_EXP ; explicit argument?
		jnz	FormToTextGet7	; yes, use explicit argument
		movzx	edx,byte [FORMInx] ; EDX <- current argument index
		call	FormGetArgLen	; get argument length (-> EAX)
		add	[FORMInx],al	; increase argument index

; ------------- Get argument

FormToTextGet7:	mov	eax,[FORMVal]	; EAX <- callback value
		push	ebx		; push EBX
		call	dword [FORMCall] ; get argument
		pop	ebx		; pop EBX
		jc	short FormToTextGet9 ; invalid argument index
		test	byte [FORMFlag],FORMFLAG_PSET ; no default precision?

; ------------- Pop registers

FormToTextGet9: pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;               Format text into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EAX = callback value
;		EBX = size of source text to be formatted
;		EDX = source text to be formatted
;		EBP = callback function to read argument
;			INPUT:	EAX = callback value
;				EBX = formatting parameters with argument type
;				EDX = DWORD index in argument stack (0 to 255)
;			OUTPUT: CY=error, invalid argument index
;				EAX = width or precision parameter
;				EDX:EAX integer
;				EAX character UNICODE
;				ST0 floating point (only if NC)
;				EAX pointer to text UTF-8, EDX length of text
;			DESTROYS: EBX, ECX, EDX:EAX (if not returning a value)
; OUTPUT:	CY = error, invalid argument index
;		ESI = output text length
; NOTES:	64-bit integer or double float takes 2 DWORDs, extended double
;		takes 3 DWORDs and other numbers take 1 DWORD. Argument index
;		refers to index of DWORD argument, not to real argument number.
; -----------------------------------------------------------------------------

; ------------- Push registers

FormToTextBufN:	push	eax		; push EAX (callback value)
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	ebp		; push EBP (callback function)
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,FORMStack	; ESP <- create local variables

; ------------- Prepare local variables

		mov	byte [FORMInx],0 ; clear current argument index
		mov	ecx,ebx		; ECX <- source text counter
		xor	esi,esi		; ESI <- 0, text length

; ------------- Read one character (-> AL)

FormToTextBufN2:dec	ecx		; check next source character
		js	FormToTextBufN7	; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Check if it is switch character

		cmp	al,"%"		; is it switch character?
		je	FormToTextBufN4	; switch character

; ------------- Store one character into destination buffer

FormToTextBufN3:inc	esi		; increase text length
		jmp	short FormToTextBufN2 ; next character

; ------------- Prepare variables to read formatting parameters

FormToTextBufN4:xor	ebx,ebx		; EBX <- 0, formatting parameters
		mov	[FORMFlagNum],bx ; clear read number and flags

; ------------- Read one character (-> AL)

FormToTextBufN5:dec	ecx		; check next source character
		js	FormToTextBufN7	; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Jump to character service

		cmp	al,32		; minimal character (space character)
		jb	FormToTextBufN3	; invalid character
		cmp	al,127		; maximal character
		jae	FormToTextBufN3	; invalid character
		movzx	eax,al		; EAX <- character
		jmp	[FormTxtNJumpTab+eax*4-32*4] ; jump

; ------------- Pop registers

FormToTextBufN7:clc			; clear error flag
FormToTextBufN8:mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Left-justify "-"

FormToTxtNLeft:	bts	ebx,FORMFLAG_Left_b ; set flag "left-justify"
		jmp	short FormToTextBufN5 ; next character

; ------------- Always use sign "+"

FormToTxtNSign:	bts	ebx,FORMFLAG_Sign_b ; set flag "sign"
		jmp	short FormToTextBufN5 ; next character

; ------------- Prefix space if number is positive " "

FormToTxtNSpace:bts	ebx,FORMFLAG_Spc_b ; set flag "space"
		jmp	short FormToTextBufN5 ; next character

; ------------- Thousand separator "~"

FormToTxtNThsnd:bts	ebx,FORMFLAG_Thsn_b ; set flag "thousand"
		jmp	short FormToTextBufN5 ; next character

; ------------- Center "@"

FormToTxtNCent:	bts	ebx,FORMFLAG_Cent_b ; set flag "center"
		jmp	short FormToTextBufN5 ; next character

; ------------- Alternate form "#" and "##"

FormToTxtNAlt:	btc	ebx,FORMFLAG_Alt_b ; alternate form
		jnc	short FormToTextBufN5 ; next character
		btc	ebx,FORMFLAG_Alt2_b ; flip alternate form 2
		jmp	short FormToTextBufN5 ; next character

; ------------- Precision "."

FormToTxtNPrec:	test	byte [FORMFlag],FORMFLAG_PREC ; precision?
		jnz	FormToTxtNPrec2	; precision has been started
		or	byte [FORMFlag],FORMFLAG_PREC ; start precision
		call	FormToTextStopW	; stop parsing width
		jmp	short FormToTextBufN5 ; next character

FormToTxtNPrec2:bts	ebx,FORMFLAG_Prec_b ; alternate precision
FormToTxtNBuf52:jmp	short FormToTextBufN5 ; next character

; ------------- Digit "0" to "9"

FormToTxtNDig0:	test	byte [FORMFlag],FORMFLAG_NUM+FORMFLAG_PREC ; first 0?
		jz	short FormToTxtNZero ; first zero digit
FormToTxtNDig:	or	byte [FORMFlag],FORMFLAG_NUM ; start number
		mov	ah,[FORMNum]	; AH <- number
		sub	al,"0"		; AL <- digit
		cmp	ah,25		; check overflow
		jae	FormToTextDig4	; it will overflow or maybe it will not
		aad			; AL <- AL + AH*10
FormToTxtNDig2:	mov	[FORMNum],al	; store new number
		jmp	short FormToTxtNBuf52 ; next character

FormToTxtNDig4:	ja	FormToTxtNDig6	; it will overflow
		aad			; AL <- AL + AH*10
		cmp	al,250		; check overflow
		jae	FormToTxtNDig2	; parameter is OK
FormToTxtNDig6:	mov	al,255		; AL <- limit on overflow
		jmp	short FormToTxtNDig2 ; next character

; ------------- Integer size "I16", "I32", "I64"

FormToTxtNSize:	sub	ecx,byte 2	; check number of characters
		js	short FormToTextBufN7 ; end of source text
		mov	ax,[edx]	; AX <- get 2 characters
		inc	edx		; increase pointer
		inc	edx		; increase pointer
		cmp	ax,"64"		; long variant?
		je	short FormToTxtNLong ; long variant
		cmp	ax,"16"		; short variant?
		je	short FormToTxtNShrt ; short variant
		cmp	ax,"32"		; default variant?
		jne	FormToTxtNSize2	; no default variant
		btr	ebx,FORMFLAG_Long_b ; reset flag "long"
		btr	ebx,FORMFLAG_Shrt_b ; reset flag "short"
		jmp	short FormToTxtNBuf52 ; next character

FormToTxtNSize2:inc	ecx		; increase number of characters
		dec	edx		; decrease pointer
		inc	ecx		; increase number of characters
		dec	edx		; decrease pointer
		jmp	short FormToTxtNBuf52 ; next character

; ------------- Add zeros instead of spaces "0"

FormToTxtNZero:	bts	ebx,FORMFLAG_Zero_b ; set flag "zero"
		jmp	short FormToTxtNBuf52 ; next character

; ------------- Get parameter "*"

FormToTxtNArg:	or	byte [FORMFlag],FORMFLAG_PAR ; set flag "parameter"
		jmp	short FormToTxtNBuf52 ; next character

; ------------- Long variant "l", "L"

FormToTxtNLong:	bts	ebx,FORMFLAG_Long_b ; set flag "long"
		btr	ebx,FORMFLAG_Shrt_b ; reset flag "short"
		jmp	short FormToTxtNBuf52 ; next character

; ------------- Short variant "h", "H"

FormToTxtNShrt:	btr	ebx,FORMFLAG_Long_b ; reset flag "long"
		bts	ebx,FORMFLAG_Shrt_b ; set flag "short"
FormToTxtNBuf54:jmp	short FormToTxtNBuf52 ; next character

; ------------- Indexed argument "$"

FormToTxtNInx:	mov	al,0		; AL <- 0
		xchg	al,[FORMNum]	; AL <- number, clear number
		and	byte [FORMFlag],~FORMFLAG_NUM ; stop number
		or	al,al		; is index valid?
		jz	FormToTxtNBuf54	; index is not valid
		dec	eax		; AL <- correct index value
		test	byte [FORMFlag],FORMFLAG_PAR ; get "parameter"?
		jnz	FormToTxtNInx2	; yes, parameter is valid
		mov	[FORMExpInx],al	; store explicit argument index
		or	byte [FORMFlag],FORMFLAG_EXP ; explicit argument valid
		jmp	short FormToTxtNBuf54 ; next character

FormToTxtNInx2:	and	byte [FORMFlag],~FORMFLAG_PAR ; clear "parameter"
		test	byte [FORMFlag],FORMFLAG_PREC ; precision?
		jnz	FormToTxtNInx4	; precision has been started
		mov	bh,al		; store width index
		or	byte [FORMFlag],FORMFLAG_INXW ; width from arguments
		jmp	short FormToTxtNBuf54 ; next character

FormToTxtNInx4:	mov	bl,al		; store precision index
		or	byte [FORMFlag],(FORMFLAG_INXP+FORMFLAG_PSET)
		jmp	short FormToTxtNBuf54 ; next character

; ------------- Signed decimal int "d", "D", "i"

FormToTxtNDC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNDS:
FormToTxtNIS:	or	ebx,(FORMTYPE_Int<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr2 ; invalid argument index
		jnz	FormToTxtNDS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTxtNDS2:	call	IntToTextBufN	; convert signed INT - get length
		add	esi,eax		; increase text length
		pop	edx		; pop EDX
		jmp	short FormToTxtNOK2 ; read next character

; ------------- Unsigned decimal int "u", "U"

FormToTxtNUC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNUS:	or	ebx,(FORMTYPE_UInt<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr2 ; invalid argument index
		jnz	FormToTxtNUS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTxtNUS2:	call	UIntToTextBufN	; convert unsigned INT - get length
		add	esi,eax		; increase text length (and clear CF)
FormToTxtNErr2:	pop	edx		; pop EDX
		jc	FormToTextBufN8	; invalid argument index
FormToTxtNOK2:	jmp	FormToTextBufN2	; read next character

; ------------- Unsigned binary int "b", "B"

FormToTxtNBC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNBS:	or	ebx,(FORMTYPE_Bin<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr2 ; invalid argument index
		jnz	FormToTxtNBS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTxtNBS2:	call	BinToTextBufN	; convert BIN number - get length
		add	esi,eax		; increase text length
		pop	edx		; pop EDX
		jmp	short FormToTxtNOK2 ; read next character

; ------------- Unsigned octal int "o", "O"

FormToTxtNOC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNOS:	or	ebx,(FORMTYPE_Oct<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr2 ; invalid argument index
		jnz	FormToTxtNOS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTxtNOS2:	call	OctToTextBufN	; convert OCT number - get length
		add	esi,eax		; increase text length
		pop	edx		; op EDX
		jmp	short FormToTxtNOK2 ; read next character

; ------------- Unsigned hex int "x", "X"

FormToTxtNXC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNXS:	or	ebx,(FORMTYPE_Hex<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		jnz	FormToTxtNXS2	; no default precision
		mov	bl,1		; BL <- 1, default precision
FormToTxtNXS2:	call	HexToTextBufN	; convert HEX - get length
		add	esi,eax		; increase text length
		pop	edx		; pop EDX
		jmp	short FormToTxtNOK ; read next character

; ------------- Character "C", "c"

FormToTxtNCC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNCS:	or	ebx,(FORMTYPE_Char<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		pop	edx		; pop EDX
		xchg	esi,edi		; EDI <- data counter
		call	CharUTF8Size	; get size of UTF-8 character
		xchg	esi,edi		; ESI <- data counter
		jmp	short FormToTxtNOK ; read next character

; ------------- String "S", "s"

FormToTxtNSC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNSS:	or	ebx,(FORMTYPE_String<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		call	FormToTextLim	; limit text length by precision
		add	esi,edx		; ESI <- add text length
FormToTxtNErr:	pop	edx		; pop EDX
		jc	FormToTextBufN8	; invalid argument index
FormToTxtNOK:	jmp	FormToTextBufN2	; read next character

; ------------- Floating point "f", "F"

FormToTxtNFC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNFS:	or	ebx,(FORMTYPE_Float<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		jnz	FormToTxtNFS2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTxtNFS2:	pop	edx		; pop EDX
		call	FltToTextBufN	; convert float number - get length
		add	esi,eax		; increase text length
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTxtNOK ; read next character

; ------------- Exponential "e", "E"

FormToTxtNEC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNES:	or	ebx,(FORMTYPE_Exp<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		jnz	FormToTxtNES2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTxtNES2:	pop	edx		; pop EDX
		call	ExpToTextBufN	; convert exponent - get length
		add	esi,eax		; increase text length
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTxtNOK ; read next character

; ------------- Mixed "g", "G"

FormToTxtNGC:	bts	ebx,FORMTYPE_Cap_b ; set flag "capital"
FormToTxtNGS:	or	ebx,(FORMTYPE_Mix<<FORMPAR_TypeF_b)
		push	edx		; push EDX
		call	FormToTextGet	; get parameters
		jc	short FormToTxtNErr ; invalid argument index
		jnz	FormToTxtNGS2	; no default precision
		mov	bl,6		; BL <- 6, default precision
FormToTxtNGS2:	pop	edx		; pop EDX
		call	MixToTextBufN	; convert mixed - get length
		add	esi,eax		; increase text length
		ffreep	st0		; free number from FPU stack
		jmp	short FormToTxtNOK ; read next character

; -----------------------------------------------------------------------------
;                      Format date/time into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to date-time structure DATETIME
;		EBX = size of source text (with date-time format descriptor)
;		ECX = pointer to nationality descriptor NATIONAL
;		EDX = source text (with date-time format descriptor)
;		ESI = remaining free space in destination buffer
;		EDI = destination buffer
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; NOTES:	To get user default nationality use DEFAULT_NAT macro.
; -----------------------------------------------------------------------------

%define		FDTNat	esp+8		; (4) pointer to nationality

; ------------- Push registers

FormDateTime:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX (pointer to nationality)
		push	edx		; push EDX
		push	ebp		; push EBP 
		xchg	eax,ebp		; EBP <- pointer to DATETIME

; ------------- Check remaining free space in buffer

FormDateTime1:	or	esi,esi		; check remaining space
		jle	short FormDateTime9 ; not enough free space

; ------------- Read one character (-> AL)

FormDateTime2:	dec	ebx		; check next source character
		jl	short FormDateTime9 ; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Read repeated characters (-> ECX)

		xor	ecx,ecx		; ECX <- 0
		inc	ecx		; ECX <- 1, character counter
		or	ebx,ebx		; end of source text?
		jz	FormDateTime4	; no other characters
FormDateTime3:	cmp	al,[edx]	; identical character?
		jne	FormDateTime4	; character is not identical
		inc	edx		; EDX <- increase text pointer
		inc	ecx		; ECX <- increase repeat counter
		dec	ebx		; EBX <- decrease source counter
		jnz	FormDateTime3	; next character

; ------------- Jump to service

FormDateTime4:	cmp	al,34		; minimal character
		jb	FormDateTime8	; invalid character
		cmp	al,"z"		; maximal character
		ja	FormDateTime8	; invalid character
		movzx	eax,al		; EAX <- character
		jmp	[FormDateTimeJmp+eax*4-34*4] ; jump

; ------------- Return last character (quoted string)

FormDateTime7:	dec	edx		; EDX <- decrease text pointer
		inc	ebx		; EBX <- increase source counter

; ------------- Store one character into destination buffer (repeated)

FormDateTime8:	dec	esi		; decrease free space
		stosb			; store character into buffer
		loopnz	FormDateTime8	; next character
		jnz	short FormDateTime2 ; next character

; ------------- Pop registers

FormDateTime9:	pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Time separator ":"

FormDateTimeT:	mov	eax,[FDTNat]	; EAX <- nationality
		mov	al,[eax+NAT_TimeSep] ; AL <- time separator
		jmp	short FormDateTime8 ; store character

; ------------- Date separator "/"

FormDateTimeD:	mov	eax,[FDTNat]	; EAX <- nationality
		mov	al,[eax+NAT_DateSep] ; AL <- date separator
		jmp	short FormDateTime8 ; store character

; ------------- Decimal separator "."

FormDateTimeC:	mov	eax,[FDTNat]	; EAX <- nationality
		mov	al,[eax+NAT_DecimalSep] ; AL <- decimal separator
		jmp	short FormDateTime8 ; store character

; ------------- Thousand separator ","

FormDateTimeH:	mov	eax,[FDTNat]	; EAX <- nationality
		mov	al,[eax+NAT_ThsndSep] ; AL <- thousand separator
		jmp	short FormDateTime8 ; store character

; ------------- Data entry separator ";"

FormDateTimeE:	mov	eax,[FDTNat]	; EAX <- nationality
		mov	al,[eax+NAT_EntrySep] ; AL <- entry separator
		jmp	short FormDateTime8 ; store character

; ------------- Quoted string ' "

FormDateTimeQ:	mov	ah,al		; AH <- quoted character
		shr	ecx,1		; ECX <- number of real characters
		jnc	short FormDateTime8 ; store characters
		jnz	short FormDateTime7 ; we must store characters first
FormDateTimeQ2:	or	ebx,ebx		; end of source text?
		jz	short FormDateTime9 ; no other characters
		mov	al,[edx]	; AL <- read character
		dec	ebx		; EBX <- decrease source counter
		inc	edx		; EDX <- increase text pointer
		cmp	al,ah		; end of string?
		je	FormDateTimeQ4	; end of string
FormDateTimeQ3:	dec	esi		; decrease free space
		stosb			; store character into buffer
		jnz	FormDateTimeQ2	; next character
FormDateTime9A:	jmp	short FormDateTime9 ; end of text

FormDateTimeQ4:	or	ebx,ebx		; end of source text?
		jz	short FormDateTime9 ; end of text
		cmp	al,[edx]	; duplicated character?
		jne	short FormDateTime1A ; it is valid end of string
		dec	ebx		; EBX <- decrease source counter
		inc	edx		; EDX <- increase text pointer
		jmp	short FormDateTimeQ3

; ------------- Hour 12 hours "h"

FormDateTimeHS:	mov	al,[ebp+DATETIME_Hour] ; AL <- hour
		add	al,11		; correction, hour - 1
		aam	12		; AL <- hour in 12h cycle
		movzx	eax,al		; EAX <- hour
		inc	eax		; EAX <- hour
		jmp	short FormDateTimeAS2

; ------------- Hour 24 hours "H"

FormDateTimeHC:	movzx	eax,byte [ebp+DATETIME_Hour] ; EAX <- hour
		jmp	short FormDateTimeAS2

; ------------- Day in month "d", "dd"

FormDateTimeDS:	movzx	eax,byte [ebp+DATETIME_Day] ; EAX <- day in month
		cmp	ecx,byte 3	; 3 characters?
		jb	short FormDateTimeAS2 ; 1 or 2 characters

; ------------- Day of week "ddd", "dddd"

		mov	al,NAT_ShortWeek ; EAX <- table of short names
		je	FormDateTimeDS6	; use short names
		mov	al,NAT_LongWeek	; EAX <- long names
FormDateTimeDS6:movzx	ecx,byte [ebp+DATETIME_WDay] ; day of week
FormDateTimeDS7:add	eax,[FDTNat]	; EAX <- + nationality
		mov	ecx,[eax+ecx*4-4] ; ECX <- text data
		mov	ah,[ecx+TEXT_Length] ; AH <- text length
		add	ecx,TEXT_Text	; EBX <- start of text
		inc	ah		; prepare character counter
FormDateTimeDS8:dec	ah		; decrease character counter
		jz	short FormDateTime1A ; end of text, get next character
		mov	al,[ecx]	; AL <- get character
		inc	ecx		; increase text pointer
		dec	esi		; decrease free space
		stosb			; store character into buffer
		jnz	FormDateTimeDS8	; next character
		jmp	short FormDateTime9A ; buffer is full

; ------------- Absolute day in year "a"

FormDateTimeAS:	movzx	eax,word [ebp+DATETIME_YDay] ; EAX <- day in year
		inc	eax		; correction
FormDateTimeAS2:call	UDWToTextBuf	; convert absolute day in year
FormDateTime1A:	jmp	FormDateTime1	; next character

; ------------- Minute "m"

FormDateTimeMS:	movzx	eax,byte [ebp+DATETIME_Min] ; EAX <- minute
		jmp	short FormDateTimeAS2

; ------------- Month "M"

FormDateTimeMC:	movzx	eax,byte [ebp+DATETIME_Month] ; EAX <- month
		cmp	ecx,byte 3	; 3 characters?
		jb	short FormDateTimeAS2 ; 1 or 2 characters
		cmp	ecx,byte 4	; 4 characters?
		movzx	ecx,al		; ECX <- month
		mov	al,NAT_ShortMonth ; EAX <- table of short names
		jb	FormDateTimeMC2	; 3 characters, use short names
		mov	al,NAT_LongMonth ; EAX <- long names
		je	FormDateTimeMC2	; 4 characters, use long names
		mov	al,NAT_LongMonth1 ; EAX <- long names, 1st case
FormDateTimeMC2:jmp	short FormDateTimeDS7

; ------------- Period or era "g" (BCE=Before Common Era, CE=Common Era)

FormDateTimeGS:	mov	al,NAT_ShortBCECE ; EAX <- table of short names
		dec	ecx		; short form?
		jz	FormDateTimeGS2	; use short names
		mov	al,NAT_LongBCECE ; EAX <- long names
FormDateTimeGS2:cmp	word [ebp+DATETIME_Year],0 ; before common era?
		setns	cl		; CL <- 0 if BCE, 1 if CE
		movzx	ecx,cl		; ECX <- 0 if BCE, 1 if CE
		inc	ecx		; index correction
		jmp	short FormDateTimeDS7 ; write era name

; ------------- AM/PM "t" (AM=ante meridiem, PM=post meridiem)

FormDateTimeTS:	mov	al,NAT_ShortAMPM ; EAX <- table of short names
		dec	ecx		; short form?
		jz	FormDateTimeTS2	; use short names
		mov	al,NAT_LongAMPM	; EAX <- long names
FormDateTimeTS2:cmp	byte [ebp+DATETIME_Hour],12 ; afternoon?
		setae	cl		; CL <- 0 if AM, 1 if PM
		movzx	ecx,cl		; ECX <- 0 if BCE, 1 if CE
		inc	ecx		; index correction
		jmp	short FormDateTimeDS7 ; write era name

; ------------- Seconds "s"

FormDateTimeSS:	movzx	eax,byte [ebp+DATETIME_Sec] ; EAX <- seconds
		jmp	short FormDateTimeAS2

; ------------- Week number "w"

FormDateTimeWS:	movzx	eax,byte [ebp+DATETIME_Week] ; EAX <- week
		jmp	short FormDateTimeAS2

; ------------- Year "y"

FormDateTimeYS:	movsx	eax,word [ebp+DATETIME_Year] ; EAX <- year
		or	eax,eax		; negative?
		jns	FormDateTimeYS2	; not negative
		neg	eax		; EAX <- absolute value
		inc	eax		; correction
FormDateTimeYS2:cmp	ecx,3		; limit number of digits?
		ja	FormDateTimeYC2	; don't limit number of digits
		push	edx		; push EDX
		push	ecx		; push ECX
		cmp	cl,2		; 2 digits?
		mov	cl,10		; ECX <- limit number to 1 digit
		jb	FormDateTimeYS3	; 1 digit
		mov	cl,100		; ECX <- limit number to 2 digits
		je	FormDateTimeYS3	; 2 digits
		mov	cx,1000		; ECX <- limit number to 3 digits
FormDateTimeYS3:xor	edx,edx		; EDX <- 0
		div	ecx		; EDX <- limited number
		xchg	eax,edx		; EAX <- year
		pop	ecx		; pop ECX
		pop	edx		; pop EDX
		jmp	short FormDateTimeYC2

; ------------- Signed year "Y"

FormDateTimeYC:	movsx	eax,word [ebp+DATETIME_Year] ; EAX <- year
FormDateTimeYC2:call	SDWToTextBuf	; convert signed year
FormDateTime1B:	jmp	FormDateTime1	; next character

; ------------- Seconds fraction "f"

FormDateTimeFS:	call	FormDateSFrac	; decode seconds fraction
		jmp	short FormDateTime1B ; next character

; ------------- Seconds fraction, truncate trailing zeros "F"

FormDateTimeFC:	call	FormDateZFrac	; decode seconds fraction
		jmp	short FormDateTime1B ; next character

; ------------- Julian date and time "j"

FormDateTimeJS:	push	ebx		; push EBX
		push	edx		; push EDX
		mov	ebx,ebp		; EBX <- DATETIME structure
		call	DateTimeToAbs	; convert to absolulte time -> EDX:EAX
		mov	ebx,FORMTYPE_Cap<<FORMPAR_TypeF_b ; don't round
FormDateTimeJS2:dec	ecx		; ECX <- without one character
		mov	bl,cl		; BL <- precision
		call	AbsToJulian	; convert to Julian date -> ST0
		mov	ecx,[FDTNat+8]	; ECX <- nationality
		call	FltToTextBuf	; convert number into buffer
		ffreep	st0		; free number from FPU stack
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		jmp	short FormDateTime1B ; next character

; ------------- Julian date and time, truncate trailing zeros "J"

FormDateTimeJC:	push	ebx		; push EBX
		push	edx		; push EDX
		mov	ebx,ebp		; EBX <- DATETIME structure
		call	DateTimeToAbs	; convert to absolulte time -> EDX:EAX
               mov ebx,(((FORMFLAG2_Alt2+FORMFLAG2_Thsnd)<<8)+FORMTYPE_Cap)<<16
		jmp	short FormDateTimeJS2

; -----------------------------------------------------------------------------
;            Local function - convert seconds fraction into buffer
; -----------------------------------------------------------------------------
; INPUT: ECX=number of digits, EDI=buffer, ESI=size of buffer
; DESTROYS: EAX, ECX

; ------------- Push registers

FormDateSFrac:	push	ebx		; push EBX
		push	edx		; push EDX

; ------------- Multiply nanoseconds with 1 0000 0000 0000 0000h/1 000 000 000

		xor	eax,eax		; EAX <- 0
		mov	al,4		; EAX <- coefficient HIGH
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		xchg	eax,ebx		; EBX <- fractions
		mov	eax,4b82fa0ah	; EAX <- coefficient LOW
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		add	ebx,edx		; EBX <- fractions
		inc	ebx		; rounding correction

; ------------- Limit number of digits

		push	ecx		; push ECX
		cmp	ecx,9		; maximal number of digits
		jb	FormDateSFrac2	; number of digits is OK
		xor	ecx,ecx		; ECX <- 0
		mov	cl,9		; ECX <- 9, limit number of digits

; ------------- Decode digits

FormDateSFrac2:	mov	eax,10		; EAX <- multiplier
		mul	ebx		; multiple number by 10
		xchg	eax,ebx		; EBX <- new number * 10
		xchg	eax,edx		; EAX <- digit
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease free space
		stosb			; store character into buffer
		loopnz	FormDateSFrac2	; next character
		pop	ecx		; pop ECX
		jz	FormDateSFrac8	; buffer is full

; ------------- Store trailing zeros

		sub	ecx,9		; ECX <- rest of number
		jbe	FormDateSFrac8	; no digits left
		mov	al,"0"		; AL <- trailign zero character
FormDateSFrac6:	dec	esi		; decrease free space
		stosb			; store digit into buffer
		loopnz	FormDateSFrac6	; next digit

; ------------- Pop registers

FormDateSFrac8:	pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;      Local function - convert seconds fraction into buffer, truncate zeros
; -----------------------------------------------------------------------------
; INPUT: ECX=number of digits, EDI=buffer, ESI=size of buffer
; DESTROYS: EAX, ECX

; ------------- Push registers

FormDateZFrac:	push	ebx		; push EBX
		push	edx		; push EDX
		push	ebp		; push EBP
		
; ------------- Multiply nanoseconds with 1 0000 0000 0000 0000h/1 000 000 000

		xor	eax,eax		; EAX <- 0
		mov	al,4		; EAX <- coefficient HIGH
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		xchg	eax,ebx		; EBX <- fractions
		mov	eax,4b82fa0ah	; EAX <- coefficient LOW
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		add	ebx,edx		; EBX <- fractions
		inc	ebx		; rounding correction

; ------------- Limit number of digits

		cmp	ecx,9		; maximal number of digits
		jb	FormDateZFrac2	; number of digits is OK
		xor	ecx,ecx		; ECX <- 0
		mov	cl,9		; ECX <- 9, limit number of digits
FormDateZFrac2:	mov	ebp,ecx		; EBP <- remaining digits

; ------------- Decode digits

FormDateZFrac3:	mov	eax,10		; EAX <- multiplier
		mul	ebx		; multiple number by 10
		xchg	eax,ebx		; EBX <- number * 10
		xchg	eax,edx		; EAX <- digit
		cmp	al,1		; zero?
		jb	FormDateZFrac7	; skip zero
FormDateZFrac5:	cmp	ebp,ecx		; any zeros pending?
		je	FormDateZFrac6	; no zeros pending
		mov	byte [edi],"0"	; store pending zero
		inc	edi		; increase pointer
		dec	ebp		; decrease counter of pending zeros
		dec	esi		; decrease free space
		jnz	FormDateZFrac5	; next pending zero
		jmp	short FormDateZFrac8 ; buffer is full
FormDateZFrac6:	dec	ebp		; EBP <- remaining zeros
		add	al,"0"		; AL <- convert to ASCII character
		dec	esi		; decrease free space
		stosb			; store character into buffer
FormDateZFrac7:	loopnz	FormDateZFrac3	; next character

; ------------- Pop registers

FormDateZFrac8:	pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;              Format date/time into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to date-time structure DATETIME
;		EBX = size of source text (with date-time format descriptor)
;		ECX = pointer to nationality descriptor NATIONAL
;		EDX = source text (with date-time format descriptor)
; OUTPUT:	ESI = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

FormDateTimeN:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX (pointer to nationality)
		push	edx		; push EDX
		push	ebp		; push EBP 
		xchg	eax,ebp		; EBP <- pointer to DATETIME

; ------------- Clear text length

		xor	esi,esi		; ESI <- 0, text length

; ------------- Read one character (-> AL)

FormDateTimeN2:	dec	ebx		; check next source character
		jl	short FormDateTimeN9 ; end of source text
		mov	al,[edx]	; AL <- character from source text
		inc	edx		; increase source pointer

; ------------- Read repeated characters (-> ECX)

		xor	ecx,ecx		; ECX <- 0
		inc	ecx		; ECX <- 1, character counter
		or	ebx,ebx		; end of source text?
		jz	FormDateTimeN4	; no other characters
FormDateTimeN3:	cmp	al,[edx]	; identical character?
		jne	FormDateTimeN4	; character is not identical
		inc	edx		; EDX <- increase text pointer
		inc	ecx		; ECX <- increase repeat counter
		dec	ebx		; EBX <- decrease source counter
		jnz	FormDateTimeN3	; next character

; ------------- Quoted string ' "

FormDateTimeN4:	cmp	al,39		; string with '?
		je	short FormDateTimeNQ ; yes, string
		cmp	al,34		; string with "?
		je	short FormDateTimeNQ ; yes, string

; ------------- Jump to service

		cmp	al,"A"		; minimal character
		jb	FormDateTimeN8	; invalid character
		cmp	al,"z"		; maximal character
		ja	FormDateTimeN8	; invalid character
		movzx	eax,al		; EAX <- character
		jmp	[FormDateTimNJmp+eax*4-"A"*4] ; jump

; ------------- Return last character (quoted string)

FormDateTimeN7:	dec	edx		; EDX <- decrease text pointer
		inc	ebx		; EBX <- increase source counter

; ------------- Store one character into destination buffer (repeated)

FormDateTimeN8:	add	esi,ecx		; ESI <- add characters
		jmp	short FormDateTimeN2 ; next character

; ------------- Pop registers

FormDateTimeN9:	pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; ------------- Quoted string ' "

FormDateTimeNQ:	mov	ah,al		; AH <- quoted character
		shr	ecx,1		; ECX <- number of real characters
		jnc	short FormDateTimeN8 ; store characters
		jnz	short FormDateTimeN7 ; we must store characters first
FormDateTimeNQ2:or	ebx,ebx		; end of source text?
		jz	short FormDateTimeN9 ; no other characters
		mov	al,[edx]	; AL <- read character
		dec	ebx		; EBX <- decrease source counter
		inc	edx		; EDX <- increase text pointer
		cmp	al,ah		; end of string?
		je	FormDateTimeNQ4	; end of string
FormDateTimeNQ3:inc	esi		; increase text length
		jmp	short FormDateTimeNQ2 ; next character

FormDateTimeNQ4:or	ebx,ebx		; end of source text?
		jz	short FormDateTimeN9 ; end of text
		cmp	al,[edx]	; duplicated character?
		jne	short FormDateTimeN2 ; it is valid end of string
		dec	ebx		; EBX <- decrease source counter
		inc	edx		; EDX <- increase text pointer
		jmp	short FormDateTimeNQ3

; ------------- Hour 12 hours "h"

FormDateTimeNHS:mov	al,[ebp+DATETIME_Hour] ; AL <- hour
		add	al,11		; correction, hour - 1
		aam	12		; AL <- hour in 12h cycle
		movzx	eax,al		; EAX <- hour
		inc	eax		; EAX <- hour
		jmp	short FormDateTimNAS2

; ------------- Hour 24 hours "H"

FormDateTimeNHC:movzx	eax,byte [ebp+DATETIME_Hour] ; EAX <- hour
		jmp	short FormDateTimNAS2

; ------------- Day in month "d", "dd"

FormDateTimeNDS:movzx	eax,byte [ebp+DATETIME_Day] ; EAX <- day in month
		cmp	ecx,byte 3	; 3 characters?
		jb	short FormDateTimNAS2 ; 1 or 2 characters

; ------------- Day of week "ddd", "dddd"

		mov	al,NAT_ShortWeek ; EAX <- table of short names
		je	FormDateTimNDS6	; use short names
		mov	al,NAT_LongWeek	; EAX <- long names
FormDateTimNDS6:movzx	ecx,byte [ebp+DATETIME_WDay] ; day of week
FormDateTimNDS7:add	eax,[FDTNat]	; EAX <- + nationality
		mov	ecx,[eax+ecx*4-4] ; ECX <- text data
		movzx	eax,byte [ecx+TEXT_Length] ; EAX <- text length
		add	esi,eax		; ESI <- increase text length
		jmp	short FormDateTimeN2A ; end of text, get next character

; ------------- Absolute day in year "a"

FormDateTimeNAS:movzx	eax,word [ebp+DATETIME_YDay] ; EAX <- day in year
		inc	eax		; correction
FormDateTimNAS2:call	UDWToTextBufN	; get text length
		add	esi,eax		; increase text length
FormDateTimeN2A:jmp	FormDateTimeN2	; next character

; ------------- Minute "m"

FormDateTimeNMS:movzx	eax,byte [ebp+DATETIME_Min] ; EAX <- minute
		jmp	short FormDateTimNAS2

; ------------- Month "M"

FormDateTimeNMC:movzx	eax,byte [ebp+DATETIME_Month] ; EAX <- month
		cmp	ecx,byte 3	; 3 characters?
		jb	short FormDateTimNAS2 ; 1 or 2 characters
		cmp	ecx,byte 4	; 4 characters?
		movzx	ecx,al		; ECX <- month
		mov	al,NAT_ShortMonth ; EAX <- table of short names
		jb	FormDateTimNMC2	; 3 characters, use short names
		mov	al,NAT_LongMonth ; EAX <- long names
		je	FormDateTimNMC2	; 4 characters, use long names
		mov	al,NAT_LongMonth1 ; EAX <- long names, 1st case
FormDateTimNMC2:jmp	short FormDateTimNDS7

; ------------- Period or era "g" (BCE=Before Common Era, CE=Common Era)

FormDateTimeNGS:mov	al,NAT_ShortBCECE ; EAX <- table of short names
		dec	ecx		; short form?
		jz	FormDateTimNGS2	; use short names
		mov	al,NAT_LongBCECE ; EAX <- long names
FormDateTimNGS2:cmp	word [ebp+DATETIME_Year],0 ; before common era?
		setns	cl		; CL <- 0 if BCE, 1 if CE
		movzx	ecx,cl		; ECX <- 0 if BCE, 1 if CE
		inc	ecx		; index correction
		jmp	short FormDateTimNDS7 ; write era name

; ------------- AM/PM "t" (AM=ante meridiem, PM=post meridiem)

FormDateTimeNTS:mov	al,NAT_ShortAMPM ; EAX <- table of short names
		dec	ecx		; short form?
		jz	FormDateTimNTS2	; use short names
		mov	al,NAT_LongAMPM	; EAX <- long names
FormDateTimNTS2:cmp	byte [ebp+DATETIME_Hour],12 ; afternoon?
		setae	cl		; CL <- 0 if AM, 1 if PM
		movzx	ecx,cl		; ECX <- 0 if BCE, 1 if CE
		inc	ecx		; index correction
		jmp	short FormDateTimNDS7 ; write era name

; ------------- Seconds "s"

FormDateTimeNSS:movzx	eax,byte [ebp+DATETIME_Sec] ; EAX <- seconds
		jmp	short FormDateTimNAS2

; ------------- Week number "w"

FormDateTimeNWS:movzx	eax,byte [ebp+DATETIME_Week] ; EAX <- week
		jmp	short FormDateTimNAS2

; ------------- Year "y"

FormDateTimeNYS:movsx	eax,word [ebp+DATETIME_Year] ; EAX <- year
		or	eax,eax		; negative?
		jns	FormDateTimNYS2	; not negative
		neg	eax		; EAX <- absolute value
		inc	eax		; correction
FormDateTimNYS2:cmp	ecx,3		; limit number of digits?
		ja	FormDateTimNYC2	; don't limit number of digits
		add	esi,ecx		; increase text length
		jmp	short FormDateTimeN2A ; next character

; ------------- Signed year "Y"

FormDateTimeNYC:movsx	eax,word [ebp+DATETIME_Year] ; EAX <- year
FormDateTimNYC2:call	SDWToTextBufN	; get text length
		add	esi,eax		; increase text length
		jmp	short FormDateTimeN2A ; next character

; ------------- Seconds fraction "f"

FormDateTimeNFS:add	esi,ecx		; increase text length
FormDateTimeN1A:jmp	FormDateTimeN2	; next character

; ------------- Seconds fraction, truncate trailing zeros "F"

FormDateTimeNFC:call	FormDateZFracN	; decode seconds fraction
		jmp	short FormDateTimeN1A ; next character

; ------------- Julian date and time "j"

FormDateTimeNJS:push	ebx		; push EBX
		push	edx		; push EDX
		mov	ebx,ebp		; EBX <- DATETIME structure
		call	DateTimeToAbs	; convert to absolulte time -> EDX:EAX
		mov	ebx,FORMTYPE_Cap<<FORMPAR_TypeF_b ; don't round
FormDateTimNJS2:dec	ecx		; ECX <- without one character
		mov	bl,cl		; BL <- precision
		call	AbsToJulian	; convert to Julian date -> ST0
		call	FltToTextBufN	; get text length
		ffreep	st0		; free number from FPU stack
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		add	esi,eax		; icrease text length
		jmp	short FormDateTimeN1A ; next character

; ------------- Julian date and time, truncate trailing zeros "J"

FormDateTimeNJC:push	ebx		; push EBX
		push	edx		; push EDX
		mov	ebx,ebp		; EBX <- DATETIME structure
		call	DateTimeToAbs	; convert to absolulte time -> EDX:EAX
               mov ebx,(((FORMFLAG2_Alt2+FORMFLAG2_Thsnd)<<8)+FORMTYPE_Cap)<<16
		jmp	short FormDateTimNJS2

; -----------------------------------------------------------------------------
;   Local function - convert seconds fraction into buffer - get text length
; -----------------------------------------------------------------------------
; INPUT: ECX=number of digits, ESI=text length
; OUTPUT: ESI=new text length
; DESTROYS: EAX, ECX

; ------------- Push registers

FormDateZFracN:	push	ebx		; push EBX
		push	edx		; push EDX
		push	ebp		; push EBP
		
; ------------- Multiply nanoseconds with 1 0000 0000 0000 0000h/1 000 000 000

		xor	eax,eax		; EAX <- 0
		mov	al,4		; EAX <- coefficient HIGH
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		xchg	eax,ebx		; EBX <- fractions
		mov	eax,4b82fa0ah	; EAX <- coefficient LOW
		mul	dword [ebp+DATETIME_NSec]; multiply nanoseconds
		add	ebx,edx		; EBX <- fractions
		inc	ebx		; rounding correction

; ------------- Limit number of digits

		cmp	ecx,9		; maximal number of digits
		jb	FormDateZFracN2	; number of digits is OK
		xor	ecx,ecx		; ECX <- 0
		mov	cl,9		; ECX <- 9, limit number of digits
FormDateZFracN2:mov	ebp,ecx		; EBP <- remaining digits

; ------------- Decode digits

FormDateZFracN3:mov	eax,10		; EAX <- multiplier
		mul	ebx		; multiple number by 10
		xchg	eax,ebx		; EBX <- number * 10
		xchg	eax,edx		; EAX <- digit
		cmp	al,1		; zero?
		jb	FormDateZFracN7	; skip zero
		sub	ebp,ecx		; EBP <- remaining zeros
		add	esi,ebp		; ESI <- increase text length
		mov	ebp,ecx		; EBP <- synchronize digit counter
		dec	ebp		; EBP <- remaining zeros
		inc	esi		; increase text length
FormDateZFracN7:loop	FormDateZFracN3	; next character

; ------------- Pop registers

FormDateZFracN8:pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                    Format absolute time into text buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = size of source text (with date-time format descriptor)
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
;		ESI = remaining free space in destination buffer
;		EDI = destination buffer
;		EBP = source text (with date-time format descriptor)
; OUTPUT:	ESI = next remaining free space in buffer
;		EDI = next destination buffer
; -----------------------------------------------------------------------------

; ------------- Push registers

FormAbsDateTime:push	eax		; push EAX
		push	ebx		; push EBX (size of source text)
		push	ecx		; push ECX
		push	edx		; push EDX
		sub	esp,DATETIME_size ; ESP <- local buffer

; ------------- Decode absolute time into date-time structure

		mov	ebx,esp		; EBX <- local buffer
		call	AbsToDateTimeExt ; convert absolute time into DATETIME

; ------------- Default nationality

		or	ecx,ecx		; default nationality?
		jnz	FormAbsDateTim2	; not default nationality
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format date/time into text buffer

FormAbsDateTim2:xchg	eax,ebx		; EAX <- date-time structure
		mov	ebx,[esp+DATETIME_size+8] ; EBX <- size of source text
		mov	edx,ebp		; EDX <- source text
		call	FormDateTime	; format date/time

; ------------- Pop registers

		add	esp,DATETIME_size ; destroy local buffer
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;            Format absolute time into text buffer - get text length
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = size of source text (with date-time format descriptor)
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
;		EBP = source text (with date-time format descriptor)
; OUTPUT:	ESI = text length
; -----------------------------------------------------------------------------

; ------------- Push registers

FormAbsDateTimeN:
		push	eax		; push EAX
		push	ebx		; push EBX (size of source text)
		push	ecx		; push ECX
		push	edx		; push EDX
		sub	esp,DATETIME_size ; ESP <- local buffer

; ------------- Decode absolute time into date-time structure

		mov	ebx,esp		; EBX <- local buffer
		call	AbsToDateTimeExt ; convert absolute time into DATETIME

; ------------- Default nationality

		or	ecx,ecx		; default nationality?
		jnz	FormAbsDateTmN2	; not default nationality
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format date/time into text buffer

FormAbsDateTmN2:xchg	eax,ebx		; EAX <- date-time structure
		mov	ebx,[esp+DATETIME_size+8] ; EBX <- size of source text
		mov	edx,ebp		; EDX <- source text
		call	FormDateTimeN	; format date/time

; ------------- Pop registers

		add	esp,DATETIME_size ; destroy local buffer
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; Maximal valid value (compiles as ...FFF...):
;		dt	2.37946299071446353e4932
; NASM reports overflow if greater than:
;		dt	2.379462990714463530171518e4932
; Minimal full-range value (exponent = 1):
;		dt	3.362104e-4932
; Minimal non-zero value (exponent = 0, mantissa = 1):
;		dt	1.82260e-4951
; Exponents with full precision: 1.0e-4931 to 1.0e+4932

; ------------- Constants

		align	4, db 0

Const10Num:	dd	10		; constant 10
Const05Num:	dd	0.5		; constant 0.5
Const01Num:	dt	0.1		; constant 0.1
Const1E17Num:	dt	1.0e17		; constant 1e17
Const1E18Num:	dt	1.0e18		; constant 1e18

; ------------- Mixed float alternate flags

MixFormFlag:	db	B1		; 0 (default): no decimal, truncate 0
		db	B0		; Alt1: always decimal, don't truncate
		db	B0+B1		; Alt2: always decimal, truncate 0
		db	0		; Alt1+Alt2: no decimal, don't truncate

		align	4, db 0

MixFormMin:	dt	1.0e-4		; minimal value for mixed float

		align	4, db 0
MixFormMax:				; maximal value for mixed float
		%assign	INTMUL10 10
		%rep	20
		dd	INTMUL10
		%assign	INTMUL10 INTMUL10*10
		%endrep

; ------------- Jump table of format text

%macro		FORMJMPERR 1
		%rep %1
		dd	FormToTextBuf3	; invalid character
		%endrep
%endmacro

FormTextJumpTab:dd	FormToTextSpace	; space
		FORMJMPERR 2		; ! "
		dd	FormToTextAlt	; #
		dd	FormToTextInx	; $
		FORMJMPERR 5		; % & ' ( )
		dd	FormToTextArg	; *
		dd	FormToTextSign	; +
		FORMJMPERR 1		; ,
		dd	FormToTextLeft	; -
		dd	FormToTextPrec	; .
		FORMJMPERR 1		; "/"
		dd	FormToTextDig0	; 0
		dd	FormToTextDig	; 1
		dd	FormToTextDig	; 2
		dd	FormToTextDig	; 3
		dd	FormToTextDig	; 4
		dd	FormToTextDig	; 5
		dd	FormToTextDig	; 6
		dd	FormToTextDig	; 7
		dd	FormToTextDig	; 8
		dd	FormToTextDig	; 9
		FORMJMPERR 6		; : ; < = > ?
		dd	FormToTextCent	; @
		FORMJMPERR 1		; A
		dd	FormToTextBC	; B
		dd	FormToTextCC	; C
		dd	FormToTextDC	; D
		dd	FormToTextEC	; E
		dd	FormToTextFC	; F
		dd	FormToTextGC	; G
		dd	FormToTextShrt	; H
		dd	FormToTextSize	; I
		FORMJMPERR 2		; J K
		dd	FormToTextLong	; L
		FORMJMPERR 2		; M N
		dd	FormToTextOC	; O
		FORMJMPERR 3		; P Q R
		dd	FormToTextSC	; S
		FORMJMPERR 1		; T
		dd	FormToTextUC	; U
		FORMJMPERR 2		; V W
		dd	FormToTextXC	; X
		FORMJMPERR 9		; Y Z [ "\" ] ^ _ ` a
		dd	FormToTextBS	; b
		dd	FormToTextCS	; c
		dd	FormToTextDS	; d
		dd	FormToTextES	; e
		dd	FormToTextFS	; f
		dd	FormToTextGS	; g
		dd	FormToTextShrt	; h
		dd	FormToTextIS	; i
		FORMJMPERR 2		; j k
		dd	FormToTextLong	; l
		FORMJMPERR 2		; m n
		dd	FormToTextOS	; o
		FORMJMPERR 3		; p q r
		dd	FormToTextSS	; s
		FORMJMPERR 1		; t
		dd	FormToTextUS	; u
		FORMJMPERR 2		; v w
		dd	FormToTextXS	; x
		FORMJMPERR 5		; y z { | }
		dd	FormToTextThsnd	; ~

; ------------- Jump table of format text - get text length

%macro		FORMJMPNERR 1
		%rep %1
		dd	FormToTextBufN3	; invalid character
		%endrep
%endmacro

FormTxtNJumpTab:dd	FormToTxtNSpace	; space
		FORMJMPNERR 2		; ! "
		dd	FormToTxtNAlt	; #
		dd	FormToTxtNInx	; $
		FORMJMPNERR 5		; % & ' ( )
		dd	FormToTxtNArg	; *
		dd	FormToTxtNSign	; +
		FORMJMPNERR 1		; ,
		dd	FormToTxtNLeft	; -
		dd	FormToTxtNPrec	; .
		FORMJMPNERR 1		; "/"
		dd	FormToTxtNDig0	; 0
		dd	FormToTxtNDig	; 1
		dd	FormToTxtNDig	; 2
		dd	FormToTxtNDig	; 3
		dd	FormToTxtNDig	; 4
		dd	FormToTxtNDig	; 5
		dd	FormToTxtNDig	; 6
		dd	FormToTxtNDig	; 7
		dd	FormToTxtNDig	; 8
		dd	FormToTxtNDig	; 9
		FORMJMPNERR 6		; : ; < = > ?
		dd	FormToTxtNCent	; @
		FORMJMPNERR 1		; A
		dd	FormToTxtNBC	; B
		dd	FormToTxtNCC	; C
		dd	FormToTxtNDC	; D
		dd	FormToTxtNEC	; E
		dd	FormToTxtNFC	; F
		dd	FormToTxtNGC	; G
		dd	FormToTxtNShrt	; H
		dd	FormToTxtNSize	; I
		FORMJMPNERR 2		; J K
		dd	FormToTxtNLong	; L
		FORMJMPNERR 2		; M N
		dd	FormToTxtNOC	; O
		FORMJMPNERR 3		; P Q R
		dd	FormToTxtNSC	; S
		FORMJMPNERR 1		; T
		dd	FormToTxtNUC	; U
		FORMJMPNERR 2		; V W
		dd	FormToTxtNXC	; X
		FORMJMPNERR 9		; Y Z [ "\" ] ^ _ ` a
		dd	FormToTxtNBS	; b
		dd	FormToTxtNCS	; c
		dd	FormToTxtNDS	; d
		dd	FormToTxtNES	; e
		dd	FormToTxtNFS	; f
		dd	FormToTxtNGS	; g
		dd	FormToTxtNShrt	; h
		dd	FormToTxtNIS	; i
		FORMJMPNERR 2		; j k
		dd	FormToTxtNLong	; l
		FORMJMPNERR 2		; m n
		dd	FormToTxtNOS	; o
		FORMJMPNERR 3		; p q r
		dd	FormToTxtNSS	; s
		FORMJMPNERR 1		; t
		dd	FormToTxtNUS	; u
		FORMJMPNERR 2		; v w
		dd	FormToTxtNXS	; x
		FORMJMPNERR 5		; y z { | }
		dd	FormToTxtNThsnd	; ~

; ------------- Jump table of format date/time

%macro		FORMDTJMPERR 1
		%rep %1
		dd	FormDateTime8	; invalid character
		%endrep
%endmacro

FormDateTimeJmp:dd	FormDateTimeQ	; " (34)
		FORMDTJMPERR 4		; # $ % &
		dd	FormDateTimeQ	; ' (39)
		FORMDTJMPERR 4		; ( ) * +
		dd	FormDateTimeH	; ,
		FORMDTJMPERR 1		; -
		dd	FormDateTimeC	; .
		dd	FormDateTimeD	; "/"
		FORMDTJMPERR 10		; 0 1 2 3 4 5 6 7 8 9
		dd	FormDateTimeT	; :
		dd	FormDateTimeE	; ;
		FORMDTJMPERR 10		; < = > ? @ A B C D E
		dd	FormDateTimeFC	; F
		FORMDTJMPERR 1		; G
		dd	FormDateTimeHC	; H
		FORMDTJMPERR 1		; I
		dd	FormDateTimeJC	; J
		FORMDTJMPERR 2		; K L
		dd	FormDateTimeMC	; M
		FORMDTJMPERR 11		; N O P Q R S T U V W X
		dd	FormDateTimeYC	; Y
		FORMDTJMPERR 7		; Z [ "\" ] ^ _ `
		dd	FormDateTimeAS	; a
		FORMDTJMPERR 2		; b c
		dd	FormDateTimeDS	; d
		FORMDTJMPERR 1		; e
		dd	FormDateTimeFS	; f
		dd	FormDateTimeGS	; g
		dd	FormDateTimeHS	; h
		FORMDTJMPERR 1		; i
		dd	FormDateTimeJS	; j
		FORMDTJMPERR 2		; k l
		dd	FormDateTimeMS	; m
		FORMDTJMPERR 5		; n o p q r
		dd	FormDateTimeSS	; s
		dd	FormDateTimeTS	; t
		FORMDTJMPERR 2		; u v
		dd	FormDateTimeWS	; w
		FORMDTJMPERR 1		; x
		dd	FormDateTimeYS	; y
		FORMDTJMPERR 1		; z

; ------------- Jump table of format date/time - get text length

%macro		FORMDTJMPNERR 1
		%rep %1
		dd	FormDateTimeN8	; invalid character
		%endrep
%endmacro

FormDateTimNJmp:FORMDTJMPNERR 5		; A B C D E
		dd	FormDateTimeNFC	; F
		FORMDTJMPNERR 1		; G
		dd	FormDateTimeNHC	; H
		FORMDTJMPNERR 1		; I
		dd	FormDateTimeNJC	; J
		FORMDTJMPNERR 2		; K L
		dd	FormDateTimeNMC	; M
		FORMDTJMPNERR 11	; N O P Q R S T U V W X
		dd	FormDateTimeNYC	; Y
		FORMDTJMPNERR 7		; Z [ "\" ] ^ _ `
		dd	FormDateTimeNAS	; a
		FORMDTJMPNERR 2		; b c
		dd	FormDateTimeNDS	; d
		FORMDTJMPNERR 1		; e
		dd	FormDateTimeNFS	; f
		dd	FormDateTimeNGS	; g
		dd	FormDateTimeNHS	; h
		FORMDTJMPNERR 1		; i
		dd	FormDateTimeNJS	; j
		FORMDTJMPNERR 2		; k l
		dd	FormDateTimeNMS	; m
		FORMDTJMPNERR 5		; n o p q r
		dd	FormDateTimeNSS	; s
		dd	FormDateTimeNTS	; t
		FORMDTJMPNERR 2		; u v
		dd	FormDateTimeNWS	; w
		FORMDTJMPNERR 1		; x
		dd	FormDateTimeNYS	; y
		FORMDTJMPNERR 1		; z

; ------------- Table of exponents, step 128 (1.0e-4864 to 1.0e4864, size 1 KB)

[list -]

		align	8, db 0
Exp128Tab:
		%assign EXPN -4864
		%rep	77
		dt	1.0e %+ EXPN
		%assign EXPN EXPN + 128
		%endrep

; ------------- Table of exponents, step 100 (1.0e4900 to 1.0e-4900, size 1 KB)

		align	8, db 0
Exp100Tab:
		%assign EXPN 4900
		%rep	99
		dt	1.0e %+ EXPN
		%assign EXPN EXPN - 100
		%endrep

; ------------- Table of exponents, step 1 (1.0e150 to 1.0e-150, size 3 KB)

		align	8, db 0
Exp1Tab:
		%assign EXPN EXP1TABBEG
		%rep	EXP1TABNUM
		dt	1.0e %+ EXPN
		%assign EXPN EXPN - 1
		%endrep
[list +]

; ------------- Table of rounding corrections (0.5e0 to 0.5e-18, size 180 B)

		align	8, db 0
RoundTab:
		%assign EXPN 0
		%rep	19
		dt	0.5e %+ EXPN
		%assign EXPN EXPN - 1
		%endrep

; ------------- Table of multiple of 10 - 1 (1-1 to 1000000000-1)

		align	4, db 0
IntMul10:
		%assign	INTMUL10 1
		%rep	10
		dd	INTMUL10-1
		%assign	INTMUL10 INTMUL10*10
		%endrep

; ------------- Table of multiple of 10 (1e10 to 1e19)

		align	8, db 0
Int2Mul10:	dd	540BE400h,2		; 1e10
		dd	4876E800h,17h		; 1e11
		dd	0D4A51000h,0E8h		; 1e12
		dd	4E72A000h,918h		; 1e13
		dd	107A4000h,5AF3h		; 1e14
		dd	0A4C68000h,38D7Eh	; 1e15
		dd	6FC10000h,2386F2h	; 1e16
		dd	5D8A0000h,1634578h	; 1e17
		dd	0A7640000h,0DE0B6B3h	; 1e18
		dd	89E80000h,8AC72304h	; 1e19

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

