; =============================================================================
;
;                            Litos - Text string
;
; -----------------------------------------------------------------------------
; Public functions:
;
; Initialize:
;	TextInit	Initialize text string with empty text string
;	TextInitChar	Initialize text string with one character
;	TextInitBuf	Initialize text string with UTF-8 buffer
;	TextInitBufCP	Initialize text string with buffer in given code page
;	TextInitText	Initialize text string with other text
;	TextTerm	Destroy text string
;
; Position:
;	TextLength	Get text length in bytes
;	TextLengthChar	Get text length in characters
;	TextCheckOff	Check if byte offset is valid
;	TextPosToOff	Recalc character position to byte offset
;	TextOffToPos	Recalc byte offset to character position
;	TextFirst	Shift position to first character
;	TextNext	Shift position to next character
;	TextLast	Shift position to last character
;	TextPrev	Shift position to previous character
;
; Get:
;	TextGetByte	Get byte from text
;	TextGetChar	Get character from text
;	TextGetFirst	Get first character from text
;	TextGetLast	Get last character from text
;	TextLeft	Get left part of text
;	TextRight	Get right part of text
;	TextFrom	Get right part of text from given position
;	TextMid		Get middle part of text
;
; Add:
;	TextAddText	Add text to end of text
;	TextAddBuf	Add buffer (in UTF-8) to end of text
;	TextAddBufCP	Add buffer in given code page to end of text
;	TextAddChar	Add one character to end of text
;	TextAddByte	Add one-byte ASCII character to end of text
;	TextAddSpace	Add space character to end of text
;
; Add number:
;	TextAddDig	Add one digit character to end of text
;	TextAdd2Dig	Add two digit characters to end of text
;	TextAddInt	Add unformated unsigned integer number to end of text
;	TextAddIntSig	Add unformated signed integer number to end of text
;	TextAddFormHex	Add formated HEX number to end of text
;	TextAddFormBin	Add formated BIN number to end of text
;	TextAddFormOct	Add formated OCT number to end of text
;	TextAddFormUInt	Add formated unsigned INT number to end of text
;	TextAddFormInt	Add formated signed INT number to end of text
;	TextAddFormExp	Add formated exponencial number to end of text
;	TextAddFormFlt	Add formated floating point number to end of text
;	TextAddFormMix	Add formated mixed floating point number to end of text
;	TextAddFormat	Add formated text to end of text
;	TextAddFormatDW	Add formated text with one DWORD argument
;	TextAddFormatF	Add formated text with one float argument
;
; Add date/time:
;	TextAddDateTime	Add date/time to end of text
;	TextAddAbsTime	Add absolute time to end of text
;	TextAddShrtTime	Add short time format to end of text
;	TextAddLongTime	Add long time format to end of text
;	TextAddShrtDate	Add short date format to end of text
;	TextAddLongDate	Add long date format to end of text
;
; Delete:
;	TextEmpty	Empty text string (delete all)
;	TextDelete	Delete part of text
;	TextDelStart	Delete start of text
;	TextDelFrom	Delete rest of text from given position
;	TextDelEnd	Delete end of text
;	TextDelChar	Delete character from text
;	TextDelFirst	Delete first character of text string
;	TextDelLast	Delete last character of text string
;	TextTrimLeft	Trim text from the left
;	TextTrimRight	Trim text from the right
;	TextTrim	Trim text from the left and right
;	TextTrimMid	Trim spaces and control characters from the text
;	TextTrimList	Trim text using list of forbidden characters
;	TextTrimUnList	Trim text using list of allowed characters
;
; Find:
;	TextFind	Find text in forward direction
;	TextFindRev	Find text in reverse direction
;	TextFindFirst	Find first occurrence of the text
;	TextFindLast	Find last occurrence of the text
;	TextFindChar	Find character in forward direction
;	TextFindCharRev	Find character in reverse direction
;	TextFindCharFirst Find first occurrence of the character
;	TextFindCharLast Find last occurrence of the character
;	TextFindByte	Find 1 byte in text in forward direction
;	TextFindByteRev	Find 1 byte in text in reverse direction
;	TextFindByteFirst Find first occurrence of 1 byte
;	TextFindByteLast Find last occurrence of 1 byte
;	TextFindWord	Find 2 bytes in text in forward direction
;	TextFindWordRev	Find 2 bytes in text in reverse direction
;	TextFindWordFirst Find first occurrence of 2 bytes
;	TextFindWordLast Find last occurrence of 2 bytes
;
; Convert:
;	TextSmall	Convert text to small letters
;	TextCap		Convert text to capital letters
;	TextSmaCap	Convert text to small/capital letters (invert)
;	TextWords	Convert text to words (first letter of word is capital)
;
; Compare:
;	TextEqu		Compare text strings to equality
;	TextComp	Compare text strings alphabetically
;
; Write:
;	TextWrite	Write text UTF-8 into buffer
;	TextWriteCP	Write text into buffer with given code page
;
; Other:
;	CharHexToBin	Convert ASCII HEX character to a number
;	CharDecToBin	Convert ASCII decimal character to a number
; =============================================================================

		CODE_SECTION

; ------------- Macro - allocate text data buffer
; INPUT:	EAX = required size of text data buffer
; OUTPUT:	CY = memory error
;		EAX = address of memory block

%macro		TEXTBUFALLOC 0
		call	SysMemAlloc		; allocate system memory
%endmacro

; ------------- Macro - free text data buffer
; INPUT:	EAX = address of text data buffer

%macro		TEXTBUFFREE 0
		call	SysMemFree		; free system memory
%endmacro

; ------------- Macro - resize text data buffer
; INPUT:	EAX = required new size of text data buffer
;		EDX = old address of text data buffer
; OUTPUT:	CY = memory error (buffer not changed)
;		EDX = new address of text data buffer

%macro		TEXTBUFRESIZE 0
		call	SysMemRealloc		; reallocate system memory
%endmacro

; -----------------------------------------------------------------------------
;                   Convert ASCII HEX character to a number
; -----------------------------------------------------------------------------
; INPUT:	AL = HEX character ("0" to "9", "A" to "F" or "a" to "f")
; OUTPUT:	AL = number 0 to 15 (if NC) or AL not changed (if CY)
;		CY = invalid HEX character (AL not changed)
; -----------------------------------------------------------------------------

; ------------- Check digit

CharHexToBin:	cmp	al,"0"		; check number MIN
		jb	CharHexToBin9	; invalid character
		cmp	al,"9"		; check number MAX
		jbe	CharHexToBin7	; character is OK

; ------------- Check capital letter

		cmp	al,"A"		; check letter MIN
		jb	CharHexToBin9	; invalid character
		cmp	al,"F"		; check letter MAX
		jbe	CharHexToBin6	; character is OK

; ------------- Check small letter

		cmp	al,"a"		; check letter MIN
		jb	CharHexToBin9	; invalid character
		cmp	al,"f"		; check letter MAX
		ja	CharHexToBin8	; invalid character

; ------------- Convert small letter to BIN (it clears CF)

		sub	al,"0"+("a"-("9"+1)) ; convert ASCII to BIN
		ret			; here is NC

; ------------- Convert capital letter to BIN (it clears CF)

CharHexToBin6:	sub	al,"0"+("A"-("9"+1)) ; convert ASCII to BIN
		ret			; here is NC

; ------------- Convert ASCII digit to BIN (it clears CF)

CharHexToBin7:	sub	al,"0"		; convert ASCII to BIN
		ret			; here is NC

; ------------- Invalid HEX character

CharHexToBin8:	stc			; set error flag
CharHexToBin9:	ret

; -----------------------------------------------------------------------------
;                   Convert ASCII decimal character to a number
; -----------------------------------------------------------------------------
; INPUT:	AL = ASCII decimal character ("0" to "9")
; OUTPUT:	AL = number 0 to 9 (if NC) or AL not changed (if CY)
;		CY = invalid decimal character (AL not changed)
; -----------------------------------------------------------------------------

; ------------- Check digit

CharDecToBin:	cmp	al,"0"		; check number MIN
		jb	CharDecToBin9	; invalid character
		cmp	al,"9"		; check number MAX
		ja	CharDecToBin8	; invalid character

; ------------- Convert ASCII digit to BIN (it clears CF)

		sub	al,"0"		; convert ASCII to BIN
		ret			; here is NC

; ------------- Invalid HEX character

CharDecToBin8:	stc			; set error flag
CharDecToBin9:	ret

; -----------------------------------------------------------------------------
;                     Attach text string data to text string
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to TEXTDATA
;		EBX = pointer to TEXT
; NOTES:	It must be paired with TextDetach.
; -----------------------------------------------------------------------------

TextAttach:	mov	[ebx],eax	; save pointer to TEXTDATA
		cmp	byte [eax+TEXT_Ref+3],CTEXTREFHIGH ; read-only?
		je	TextAttach2	; text string is read only
		LOCKSMP			; CPU instruction lock
		inc	dword [eax+TEXT_Ref] ; increase reference counter
TextAttach2:	ret

; -----------------------------------------------------------------------------
;                  Detach text string data from text string
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; NOTES:	It must be paired with TextAttach.
; -----------------------------------------------------------------------------

TextTerm:				; destroy text string
TextDetach:	push	eax		; push EAX
		mov	eax,[ebx]	; EAX <- pointer to TEXTDATA
		cmp	byte [eax+TEXT_Ref+3],CTEXTREFHIGH ; read-only?
		je	TextDetach2	; text string is read only
		LOCKSMP			; CPU instruction lock
		dec	dword [eax+TEXT_Ref] ; decrease reference counter
		jnz	TextDetach2	; counter does not reached zero
		TEXTBUFFREE		; free TEXTDATA
TextDetach2:	pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                         Create new text data buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = required size of text data buffer in bytes (without head)
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (content of TEXT not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextNew:	push	ecx		; push ECX
		xchg	ecx,eax		; ECX <- save required size of text

; ------------- Allocate text data buffer

		lea	eax,[ecx+TEXTDATA_size] ; EAX <- size of buffer
		TEXTBUFALLOC		; allocate buffer
		jc	TextNew2	; memory error

; ------------- Initialize text data buffer (it exits with NC)

		mov	[ebx],eax	; store address of text data buffer
		mov	dword [eax+TEXT_Ref],1 ; initialize reference counter
		mov	[eax+TEXT_Length],ecx ; store length of text

; ------------- Pop registers

TextNew2:	xchg	eax,ecx		; EAX <- return required size of text
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                           Resize text data buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = required new size of text data buffer (without head)
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (content of TEXT not changed)
; NOTES:	Data buffer must be owned by this text string, i.e. reference
;			counter must be 1 (use TextCopyWrite).
; -----------------------------------------------------------------------------

; ------------- Push registers

TextResize:	push	edx		; push EDX
		push	eax		; push EAX

; ------------- Resize text data buffer

		mov	edx,[ebx]	; EDX <- address of data buffer
		add	eax,byte TEXTDATA_size ; EAX <- size of buffer
		TEXTBUFRESIZE		; resize text data buffer
		jc	TextResize2	; memory error

; ------------- Initialize data buffer and pop registers 1

		mov	[ebx],edx	; store new address of data buffer
		pop	eax		; pop EAX (new size of buffer)
		mov	[edx+TEXT_Length],eax ; new length of text
		pop	edx		; pop EDX
		ret			; (here is NC)

; ------------- Pop registers 2

TextResize2:	pop	eax		; pop EAX
		pop	edx		; pop EDX
		ret			; (here is CY)

; -----------------------------------------------------------------------------
;                     Copy text data buffer before write
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error (content of TEXT not changed)
; NOTES:	It duplicates text data if reference counter is > 1.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextCopyWrite:	push	esi		; push ESI

; ------------- Check reference counter

		mov	esi,[ebx]	; ESI <- address of data buffer
		cmp	dword [esi+TEXT_Ref],byte 1 ; check reference counter
		jne	TextCopyWrite2	; data buffer need to be duplicated

; ------------- Pop registers (here is NC)

		pop	esi		; pop ESI
		ret

; ------------- Push registers 2

TextCopyWrite2:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Create new data buffer

		mov	eax,[esi+TEXT_Length] ; EAX <- length of data buffer
		call	TextNew		; create new data buffer
		jc	TextCopyWrite6	; memory error

; ------------- Copy data

		push	esi		; push ESI (old data buffer)
		mov	edi,[ebx]	; EDI <- address of new data buffer
		add	edi,byte TEXTDATA_size ; EDI <- start of text
		add	esi,byte TEXTDATA_size ; ESI <- start of text
		mov	ecx,eax		; ECX <- size of data
		shr	ecx,2		; ECX <- size of data in DWORDs
		rep	movsd		; copy text in DWORDs
		and	eax,byte 3	; EAX <- size in last DWORD
		xchg	eax,ecx		; ECX <- size in last DWORD
		rep	movsb		; copy rest of text
		pop	eax		; pop EAX (old data buffer)

; ------------- Detach old data buffer

		cmp	byte [eax+TEXT_Ref+3],CTEXTREFHIGH ; read-only?
		je	TextCopyWrite6	; text string is read only
		LOCKSMP			; CPU instruction lock
		dec	dword [eax+TEXT_Ref] ; decrease reference counter
		jnz	TextCopyWrite4	; counter does not reached zero
		TEXTBUFFREE		; free TEXTDATA
TextCopyWrite4:	clc			; clear error flag

; ------------- Pop registers 2

TextCopyWrite6:	pop	edi		; pop EDI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		pop	esi		; pop ESI
		ret

; -----------------------------------------------------------------------------
;                             Empty text string
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; -----------------------------------------------------------------------------

TextEmpty:	call	TextDetach	; detach old text string

; TextInit must follow!

; -----------------------------------------------------------------------------
;               Initialize text string with empty text string
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; -----------------------------------------------------------------------------

TextInit:	mov	dword [ebx],EmptyTextData ; save pointer to empty text
		ret

; -----------------------------------------------------------------------------
;                 Initialize text string with one character
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text becomes empty)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextInitChar:	push	eax		; push EAX
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Prepare length of character (-> EAX, push character -> EBP)

		xor	edi,edi		; EDI <- 0, character length counter
		mov	ebp,eax		; EBP <- push EAX (character)
		call	CharUTF8Size	; get character size
		xchg	eax,edi		; EAX <- text length

; ------------- Create new text buffer

		mov	dword [ebx],EmptyTextData ; set pointer to empty text
		call	TextNew		; create new text buffer
		jc	TextInitChar2	; memory error

; ------------- Store character into buffer

		xchg	eax,ebp		; EBP <- size of buffer, EAX <- char
		mov	edi,[ebx]	; EDI <- new text data
		add	edi,TEXT_Text	; EDI <- start of buffer
		call	CharUTF8Write	; write character into buffer

; ------------- Pop registers

		clc			; clear error flag
TextInitChar2:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Initialize text string with UTF-8 buffer
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		ECX = length of text in buffer (in bytes)
;		EDX = pointer to buffer with text in UTF-8
; OUTPUT:	CY = memory error (text becomes empty)
; NOTES:	Detection bytes 0feh and 0ffh should be eliminated.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextInitBuf:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Create new text buffer

		mov	eax,ecx		; EAX <- size of text data
		mov	dword [ebx],EmptyTextData ; set pointer to empty text
		call	TextNew		; create new text buffer
		jc	TextInitBuf4	; memory error

; ------------- Copy text data (it exits with NC)

		mov	esi,edx		; ESI <- buffer with text data
		mov	edi,[ebx]	; EDI <- new text data buffer
		add	edi,byte TEXT_Text ; EDI <- start of text
		shr	ecx,2		; ECX <- size of data in DWORDs
		rep	movsd		; copy text in DWORDs
		and	eax,byte 3	; EAX <- size in last DWORD
		xchg	eax,ecx		; ECX <- size in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers (here is NC)

TextInitBuf4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;            Initialize text string with buffer in given code page
; -----------------------------------------------------------------------------
; INPUT:	AX = code page of text in buffer
;		EBX = pointer to TEXT
;		ECX = length of text in buffer (in bytes)
;		EDX = invalid character (0 = use default)
;		ESI = pointer to buffer with text
; OUTPUT:	CY = memory error (text becomes empty)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextInitBufCP:	push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Set empty text

		mov	dword [ebx],EmptyTextData ; set pointer to empty text

; ------------- Create new text buffer

		push	eax		; push EAX (source code page)
		push	ebx		; push EBX (pointer to TEXT)
		mov	bx,CP_UTF8	; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTrans	; get length of destination buffer
		pop	ebx		; pop EBX (pointer to TEXT)
		call	TextNew		; create new text buffer
		xchg	eax,ebp		; EBP <- size of buffer
		pop	eax		; pop EAX (source code page)
		jc	TextInitBufCP4	; memory error

; ------------- Convert text

		push	eax		; push EAX (source code page)
		push	ebx		; push EBX (pointer to TEXT)
		mov	edi,[ebx]	; EDI <- text string data
		add	edi,byte TEXT_Text ; EDI <- start of data buffer
		mov	bx,CP_UTF8	; BX <- destination codepage UTF-8
		call	CharTrans	; convert text
		pop	ebx		; pop EBX (pointer to TEXT)
		pop	eax		; pop EAX (source code page)

; ------------- Pop registers

		clc			; clear error flag
TextInitBufCP4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		ret

; -----------------------------------------------------------------------------
;                  Initialize text string with other text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to TEXT
; -----------------------------------------------------------------------------

TextInitText:	push	eax		; push EAX
		mov	eax,[eax]	; EAX <- source data buffer
		call	TextAttach	; attach source text
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                     Add one character to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = Unicode character
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddChar:	push	eax		; push EAX
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddChar2	; memory error

; ------------- Prepare length of character (-> EAX, push character -> EBP)

		xor	edi,edi		; EDI <- 0, character length counter
		mov	ebp,eax		; EBP <- push EAX (character)
		call	CharUTF8Size	; get character size
		xchg	eax,edi		; EAX <- text length

; ------------- Resize data buffer

		mov	edi,[ebx]	; EDI <- data buffer
		mov	edi,[edi+TEXT_Length] ; EDI <- old length of text
		add	eax,edi		; EAX <- new length of text
		call	TextResize	; resize data buffer
		jc	TextAddChar2	; memory error

; ------------- Store character into buffer

		xchg	eax,ebp		; EBP <- size of buffer, EAX <- char
		add	edi,[ebx]	; EDI <- new text data + old length
		add	edi,TEXT_Text	; EDI <- end of old data
		call	CharUTF8Write	; write character into buffer

; ------------- Pop registers

		clc			; clear error flag
TextAddChar2:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Add one-byte ASCII character to end of text
; -----------------------------------------------------------------------------
; INPUT:	AL = byte (it must be in range 0 to 7fh)
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddByte:	push	eax		; push EAX
		push	ecx		; push ECX
		xchg	eax,ecx		; CL <- character

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddByte2	; memory error

; ------------- Resize data buffer

		mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		inc	eax		; EAX <- new size of data buffer
		call	TextResize	; resize data buffer
		jc	TextAddDig2	; memory error

; ------------- Store character (it exits with NC)

		add	eax,[ebx]	; EAX <- end of data buffer
		mov	[eax+TEXT_Text-1],cl ; store character into buffer

; ------------- Pop registers

TextAddByte2:	pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                      Add space character to end of text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddSpace:	push	eax		; push EAX
		mov	al," "		; AL <- space character
		call	TextAddByte	; add space character to end of text
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Add one digit character to end of text
; -----------------------------------------------------------------------------
; INPUT:	AL = number (it must be in range 0 to 9)
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddDig:	push	eax		; push EAX

; ------------- Check valid range of number

		cmp	al,10		; check valid range
		cmc			; CY = error
		jc	TextAddDig2	; error, invalid number

; ------------- Add digit to end of text

		add	al,'0'		; AL <- digit character
		call	TextAddByte	; add digit to end of text

; ------------- Pop registers

TextAddDig2:	pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Add two digit characters to end of text
; -----------------------------------------------------------------------------
; INPUT:	AL = number (it must be in range 0 to 99)
;		EBX = pointer to TEXT variable
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAdd2Dig:	push	eax		; push EAX
		push	ecx		; push ECX

; ------------- Check valid range of number

		cmp	al,100		; check valid range
		cmc			; CY = error
		jc	TextAdd2Dig2	; error, invalid number

; ------------- Prepare digit characters (-> CX)

		aam			; AH <- first digit, AL <- second digit
		add	ax,'00'		; AX <- digit characters
		xchg	eax,ecx		; CX <- digit characters

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAdd2Dig2	; memory error

; ------------- Resize data buffer

		mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		inc	eax		; EAX <- length + 1
		inc	eax		; EAX <- new size of data buffer
		call	TextResize	; resize data buffer
		jc	TextAdd2Dig2	; memory error

; ------------- Store characters (it exits with NC)

		add	eax,[ebx]	; EAX <- end of data buffer
		mov	[eax+TEXT_Text-2],cx ; store characters into buffer

; ------------- Pop registers

TextAdd2Dig2:	pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;             Add unformated unsigned integer number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = unsigned number
;		EBX = pointer to TEXT variable
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddInt:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		xchg	eax,esi		; ESI <- number

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddInt8	; memory error

; ------------- Get number of bits in number (-> AL, minimal 1)

		xor	ecx,ecx		; ECX <- 0
		inc	ecx		; ECX <- 1, minimal 1 digit if zero
		bsr	eax,esi		; AL <- highest bit in number
		jz	TextAddInt2	; number is zero, use 1 digit
		inc	eax		; AL <- number of bits in number

; ------------- Get number of digits (-> ECX) (ln 10/ln 2 = 3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	ecx,ah		; ECX <- aprox. number of digits
		cmp	[IntMul10+ecx*4],esi ; check number
		adc	cl,0		; ECX <- number of digits

; ------------- Resize data buffer

TextAddInt2:	mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		add	eax,ecx		; EAX <- add text length
		call	TextResize	; resize data buffer
		jc	TextAddInt8	; memory error
		add	eax,[ebx]	; EAX <- end of data buffer
		add	eax,TEXT_Text-1	; EAX <- last character in buffer
		xchg	eax,edi		; EDI <- end of data buffer

; ------------- Convert number

		std			; set direction down
TextAddInt4:	mov	eax,3435973837	; EAX <- 800000000h/10, rounded up
		mul	esi		; EDX <- (number*8) / 10
		shr	edx,3		; EDX <- number / 10
		mov	al,10		; AL <- 10
		mul	dl		; AL <- number / 10 * 10 low byte
		xchg	eax,esi		; EAX <- number, ESI <- number low byte
		sub	eax,esi		; AL <- number % 10
		add	al,"0"		; AL <- convert to ASCII character
		stosb			; store one character
		mov	esi,edx		; ESI <- number / 10
		loop	TextAddInt4	; next character
		cld			; set direction up

; ------------- Pop registers

		clc			; clear error flag
TextAddInt8:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;              Add unformated signed integer number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = signed number
;		EBX = pointer to TEXT variable
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddIntSig:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		xchg	eax,esi		; ESI <- number

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddIntSig8	; memory error

; ------------- Absolute value

		xor	edx,edx		; EDX <- 0, number is positive
		or	esi,esi		; is number negative?
		jns	TextAddIntSig2	; number is not negative
		inc	edx		; EDX <- 1, number is negative
		neg	esi		; ESI <- absolute value

; ------------- Get number of bits in number (-> AL, minimal 1)

TextAddIntSig2:	xor	ecx,ecx		; ECX <- 0
		inc	ecx		; ECX <- 1, minimal 1 digit if zero
		bsr	eax,esi		; AL <- highest bit in number
		jz	TextAddIntSig4	; number is zero, use 1 digit
		inc	eax		; AL <- number of bits in number

; ------------- Get number of digits (-> ECX) (ln 10/ln 2 = 3.3219)

		mov	ah,78		; AH <- 100h / 3.3219 round up
		mul	ah		; AH <- aprox. number of digits
		movzx	ecx,ah		; ECX <- aprox. number of digits
		cmp	[IntMul10+ecx*4],esi ; check number
		adc	cl,0		; ECX <- number of digits

; ------------- Resize data buffer

TextAddIntSig4:	mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		add	eax,ecx		; EAX <- add text length
		add	eax,edx		; EAX <- add sign
		call	TextResize	; resize data buffer
		jc	TextAddIntSig8	; memory error
		add	eax,[ebx]	; EAX <- end of data buffer

; ------------- Store sign

		or	edx,edx		; is number negative?
		jz	TextAddIntSig5	; number is not negative
		mov	edx,eax		; EDX <- end of data buffer
		sub	edx,ecx		; EDX <- start of data in buffer
		mov	byte [edx+TEXT_Text-1],"-" ; store negative sign
TextAddIntSig5:	add	eax,TEXT_Text-1	; EAX <- last character in buffer
		xchg	eax,edi		; EDI <- end of data buffer

; ------------- Convert number

		std			; set direction down
TextAddIntSig6:	mov	eax,3435973837	; EAX <- 800000000h/10, rounded up
		mul	esi		; EDX <- (number*8) / 10
		shr	edx,3		; EDX <- number / 10
		mov	al,10		; AL <- 10
		mul	dl		; AL <- number / 10 * 10 low byte
		xchg	eax,esi		; EAX <- number, ESI <- number low byte
		sub	eax,esi		; AL <- number % 10
		add	al,"0"		; AL <- convert to ASCII character
		stosb			; store one character
		mov	esi,edx		; ESI <- number / 10
		loop	TextAddIntSig6	; next character
		cld			; set direction up

; ------------- Pop registers

		clc			; clear error flag
TextAddIntSig8:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Macro - add formated number to end of text
; -----------------------------------------------------------------------------
; %1 = function get text length, %2 = function convert to text

%macro		TEXTADDFORMNUM 2

; ------------- Push registers

		push	ebx		; push EBX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	%%L2		; memory error

; ------------- Get text length (-> ESI)

		mov	ebp,esi		; EBP <- pointer to nationality
		mov	esi,eax		; ESI <- push number LOW
		xchg	ebx,ecx		; EBX <- parameters, ECX <- TEXT
		call	%1		; get text length (-> EAX)
		xchg	ebx,ecx		; EBX <- TEXT, ECX <- parameters
		xchg	eax,esi		; EAX <- number LOW, ESI <- length

; ------------- Resize buffer

		xchg	eax,edi		; EDI <- push EAX
		mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		add	eax,esi		; EAX <- new text length
		call	TextResize	; resize data buffer
		xchg	eax,edi		; EAX <- pop EAX,EDI <- new text length
		jc	%%L2		; memory error

; ------------- Prepare registers

		sub	edi,esi		; EDI <- old text length
		add	edi,[ebx]	; EDI <- start of buffer
		add	edi,byte TEXT_Text ; EDI <- start of text

; ------------- Convert number to text

		mov	ebx,ecx		; EBX <- formatting parameters
		mov	ecx,ebp		; ECX <- pointer to nationality
		or	ecx,ecx		; use default nationality?
		jnz	%%L1		; no default nationality
		DEFAULT_NAT ecx		; ECX <- get default nationality
%%L1:		call	%2		; convert number to text
		mov	ecx,ebx		; ECX <- formatting parameters

; ------------- Pop registers

		clc			; clear error flag
%%L2:		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ebx		; pop EBX
		ret
%endmacro

; -----------------------------------------------------------------------------
;                    Add formated HEX number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = unsigned number
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = prefix 0x
;				FORMFLAG_Alt2_b = suffix h
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddFormHex:	TEXTADDFORMNUM HexToTextBufN, HexCToTextBuf

; -----------------------------------------------------------------------------
;                    Add formated BIN number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = unsigned number
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = prefix 0b
;				FORMFLAG_Alt2_b = suffix b
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddFormBin:	TEXTADDFORMNUM BinToTextBufN, BinToTextBuf

; -----------------------------------------------------------------------------
;                    Add formated OCT number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = unsigned number
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = prefix 0
;				FORMFLAG_Alt2_b = suffix o
;				FORMFLAG_Cent_b = center
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddFormOct:	TEXTADDFORMNUM OctToTextBufN, OctToTextBuf

; -----------------------------------------------------------------------------
;               Add formated unsigned INT number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = unsigned number
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddFormUInt:TEXTADDFORMNUM UIntToTextBufN, UIntToTextBuf

; -----------------------------------------------------------------------------
;                Add formated signed INT number to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = signed number
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Sign_b	= always use sign "+"
;				FORMFLAG_Spc_b  = prefix space if positive
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; -----------------------------------------------------------------------------

TextAddFormInt:	TEXTADDFORMNUM IntToTextBufN, IntToTextBuf

; -----------------------------------------------------------------------------
;                 Add formated exponencial number to end of text
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits after decimal point
;				or number of signific.digits if FORMFLAG_Prec_b
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Sign_b	= always use sign "+"
;				FORMFLAG_Spc_b  = prefix space if positive
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = always decimal point
;				FORMFLAG_Alt2_b = truncate trailing zeroes
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;				FORMFLAG_Prec_b = precision=significant digits
;							instead of fractional
;				FORMTYPE_Cap_b	= does not round
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; NOTES:	It uses 2 more registers from FPU stack.
;		Exponent is 3 or 4 digits and uses small "e".
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------

TextAddFormExp:	TEXTADDFORMNUM ExpToTextBufN, ExpSToTextBuf

; -----------------------------------------------------------------------------
;                Add formated floating point number to end of text
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = minimal number of digits after decimal point
;				or number of signific.digits if FORMFLAG_Prec_b
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Sign_b	= always use sign "+"
;				FORMFLAG_Spc_b  = prefix space if positive
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = always decimal point
;				FORMFLAG_Alt2_b = truncate trailing zeroes
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;				FORMFLAG_Prec_b = precision=significant digits
;							instead of fractional
;				FORMTYPE_Cap_b	= does not round
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; NOTES:	It uses 2 more registers from FPU stack.
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------

TextAddFormFlt:	TEXTADDFORMNUM FltToTextBufN, FltToTextBuf

; -----------------------------------------------------------------------------
;           Add formated mixed floating point number to end of text
; -----------------------------------------------------------------------------
; INPUT:	ST0 = float number (it does not pop it from the FPU stack)
;		EBX = pointer to TEXT variable
;		ECX = formatting parameters FORMPAR
;			CL = number of significant digits or minimal number
;			       of digits after decimal point if FORMFLAG_Prec_b
;			CH = minimal width of field
;			bits:	FORMFLAG_Left_b = left-justify
;				FORMFLAG_Sign_b	= always use sign "+"
;				FORMFLAG_Spc_b  = prefix space if positive
;				FORMFLAG_Zero_b	= add zeroes instead of spaces
;				FORMFLAG_Alt_b  = always point, don't truncate
;				FORMFLAG_Alt2_b = always point, truncate
;					neither Alt now Alt2 = no point, trunc.
;					Alt and Alt2 = no point, don't truncate
;				FORMFLAG_Thsn_b = use thousand separator
;				FORMFLAG_Cent_b = center
;				FORMFLAG_Prec_b = precision=fractional instead
;							of significant digits
;				FORMTYPE_Cap_b	= does not round
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid number (text not changed)
; NOTES:	It uses float point form in range 1e-4 to 1e+precision,
;			else uses exponential form.
;		It uses 2 more registers from FPU stack.
;		To destroy FPU register you can use "ffreep st0" instruction.
; -----------------------------------------------------------------------------

TextAddFormMix:	TEXTADDFORMNUM MixToTextBufN, MixSToTextBuf

; -----------------------------------------------------------------------------
;                     Add buffer (in UTF-8) to end of text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		ECX = length of text in buffer (in bytes)
;		EDX = pointer to buffer with text in UTF-8
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	Be careful to not copy text string to itself, it can be
;		  unsafe in multi-thread (source can be freed prematurely).
; 		Detection bytes 0feh and 0ffh should be eliminated.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddBuf:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddBuf4	; memory error

; ------------- Resize data buffer

		mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		lea	edi,[eax+TEXT_Text] ; EDI <- offset of old end of text
		add	eax,ecx		; EAX <- new size of data buffer
		call	TextResize	; resize data buffer
		jc	TextAddBuf4	; memory error

; ------------- Copy text (it exits with NC)

		add	edi,[ebx]	; EDI <- old end of text
		mov	esi,edx		; ESI <- source buffer
		mov	eax,ecx		; EAX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; copy text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextAddBuf4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                  Add buffer in given code page to end of text
; -----------------------------------------------------------------------------
; INPUT:	AX = code page of text in buffer
;		EBX = pointer to TEXT
;		ECX = length of text in buffer (in bytes)
;		EDX = invalid character (0 = use default)
;		ESI = pointer to buffer with text
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddBufCP:	push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddBufCP4	; memory error

; ------------- Get size of new data (-> EAX)

		push	eax		; push EAX (source code page)
		push	ebx		; push EBX (pointer to TEXT)
		mov	bx,CP_UTF8	; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTrans	; get length of destination buffer
		pop	ebx		; pop EBX (pointer to TEXT)

; ------------- Resize text buffer

		mov	edi,[ebx]	; EDI <- data buffer
		mov	edi,[edi+TEXT_Length] ; EDI <- old length of text
		add	eax,edi		; EAX <- new length of text
		call	TextResize	; resize data buffer
		xchg	eax,ebp		; EBP <- new length of text
		pop	eax		; pop EAX (source code page)
		jc	TextAddBufCP4	; memory error
		sub	ebp,edi		; EBP <- length of added text

; ------------- Convert text

		push	eax		; push EAX (source code page)
		push	ebx		; push EBX (pointer to TEXT)
		add	edi,[ebx]	; EDI <- text string data
		add	edi,byte TEXT_Text ; EDI <- end of old data
		mov	bx,CP_UTF8	; BX <- destination codepage UTF-8
		call	CharTrans	; convert text
		pop	ebx		; pop EBX (pointer to TEXT)
		pop	eax		; pop EAX (source code page)

; ------------- Pop registers

		clc			; clear error flag
TextAddBufCP4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		ret

; -----------------------------------------------------------------------------
;                          Add text to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	Source and destination text can be identical.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddText:	push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Safe variant of copy text to itself

		mov	edx,[eax]	; EDX <- data buffer of source text
		cmp	edx,[ebx]	; is it identical text buffer?
		je	TextAddText4	; it is identical text buffer

; ------------- Add text

TextAddText2:	mov	ecx,[edx+TEXT_Length] ; ECX <- length of source text
		add	edx,byte TEXT_Text ; EDX <- start of text
		call	TextAddBuf	; add text

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; ------------- Push registers 2

TextAddText4:	push	eax		; push EAX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextAddText6	; memory error

; ------------- Resize data buffer

		mov	eax,[ebx]	; EAX <- data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		mov	ecx,eax		; ECX <- text length
		shl	eax,1		; EAX <- new size of data buffer
		call	TextResize	; resize data buffer
		jc	TextAddText6	; memory error

; ------------- Copy text (it exits with NC)

		mov	esi,[ebx]	; ESI <- data buffer
		add	esi,TEXT_Text	; ESI <- start of text
		lea	edi,[esi+ecx]	; EDI <- end of old text
		mov	eax,ecx		; EAX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; copy text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextAddText6:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	eax		; pop EAX
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                    Shift position to first character
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EDX = position of first character (or end of text if CY)
;		CY = text is empty (EDX = end of text)
; -----------------------------------------------------------------------------

TextFirst:	xor	edx,edx		; EDX <- 0
		dec	edx		; EDX <- -1, minimal position

; TextNext must follow.

; -----------------------------------------------------------------------------
;                      Shift position to next character
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = character position (it can be out of range)
; OUTPUT:	EDX = position of next character (or end of text if CY)
;		CY = no next character (EDX = end of text)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextNext:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Get data and length of text

		mov	esi,[ebx]	; ESI <- data buffer
		mov	ecx,[esi+TEXT_Length] ; ECX <- length of text

; ------------- Check character position

		cmp	edx,ecx		; check character position
		jae	TextNext6	; position is not valid

; ------------- Increase character position

TextNext2:	inc	edx		; EDX <- increase character position

; ------------- Check character position

TextNext4:	cmp	edx,ecx		; check character position
		cmc			; CY = invalid position
		jc	TextNext5	; position is not valid

; ------------- Check character if it is start of next character

		mov	al,[esi+TEXT_Text+edx] ; AL <- current byte
		cmp	al,0feh		; detection bytes?
		jae	TextNext2	; skip detection bytes
		and	al,0c0h		; mask bits
		cmp	al,80h		; is it first byte of character?
		je	TextNext2	; it is not first byte of character

; ------------- Pop registers

		clc			; clear error flag
TextNext5:	pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Check minimal character position

TextNext6:	or	edx,edx		; is character position negative?
		js	TextNext8	; character position is negative

; ------------- Character position is behind end of text

		mov	edx,ecx		; EDX <- limit character position
		stc			; set error flag
		jmp	short TextNext5

; ------------- Character position is before start of text

TextNext8:	xor	edx,edx		; EDX <- shift to first position
		jmp	short TextNext4

; -----------------------------------------------------------------------------
;                    Shift position to last character
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EDX = position of last character (or start of text if CY)
;		CY = text is empty (EDX = start of text)
; -----------------------------------------------------------------------------

TextLast:	mov	edx,TEXTBIGPOS	; EDX <- maximal position

; TextPrev must follow.

; -----------------------------------------------------------------------------
;                    Shift position to previous character
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = character position (it can be out of range)
; OUTPUT:	EDX = position of previous character (or start of text if CY)
;		CY = no previous character (EDX = start of text)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextPrev:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Get data and length of text

		mov	esi,[ebx]	; ESI <- data buffer
		mov	ecx,[esi+TEXT_Length] ; ECX <- length of text

; ------------- Check character position

		cmp	edx,ecx		; check character position
		ja	TextPrev6	; position is not valid

; ------------- Check character position

TextPrev2:	or	edx,edx		; check character position
		stc			; CY = invalid position
		jz	TextPrev5	; position is not valid

; ------------- Decrease character position

		dec	edx		; EDX <- decrease character position

; ------------- Check character if it is start of previous character

		mov	al,[esi+TEXT_Text+edx] ; AL <- current byte
		cmp	al,0feh		; detection bytes?
		jae	TextPrev2	; skip detection bytes
		and	al,0c0h		; mask bits
		cmp	al,80h		; is it first byte of character?
		je	TextPrev2	; it is not first byte of character

; ------------- Pop registers

		clc			; clear error flag
TextPrev5:	pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Check minimal character position

TextPrev6:	or	edx,edx		; is character position negative?
		jns	TextPrev8	; character position is not negative

; ------------- Character position is before start of text

                xor	edx,edx		; EDX <- shift to first position
		stc			; set error flag
		jmp	short TextPrev5

; ------------- Character position is behind end of text

TextPrev8:	mov	edx,ecx		; EDX <- limit character position
		jmp	short TextPrev2

; -----------------------------------------------------------------------------
;                            Get byte from text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = byte offset (it may be out of range)
; OUTPUT:	AL = byte from the text
;		CY = error, position is out of range
; -----------------------------------------------------------------------------

TextGetByte:	push	ecx		; push ECX
		mov	ecx,[ebx]	; ECX <- pointer to TEXTDATA
		cmp	edx,[ecx+TEXT_Length] ; check byte offset
		jae	TextGetByte2	; invalid byte offset
		mov	al,[ecx+TEXT_Text+edx] ; AL <- byte
TextGetByte2:	cmc			; CY = invalid byte offset
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                           Get character from text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = character position (it may be out of range)
; OUTPUT:	EAX = Unicode character (or UNINOASC on error)
;		EDX = new character position
;		CY = error, position is out of range (EAX = UNINOASC)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextGetChar:	push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Get data and length of text

		mov	edi,[ebx]	; ESI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text

; ------------- Check character position

		cmp	edx,ecx		; check character position
		jae	TextGetChar6	; position is not valid

; ------------- Read character from buffer

TextGetChar2:	sub	ecx,edx		; ECX <- remaining bytes
		lea	esi,[edi+TEXT_Text+edx] ; ESI <- character position
		xor	edx,edx		; EDX <- default character
		call	CharUTF8Read	; read character from buffer

; ------------- New character position (it exits with NC)

		sub	esi,edi		; ESI <- new offset in buffer
		lea	edx,[esi-TEXT_Text] ; EDX <- new character position

; ------------- Pop registers

TextGetChar4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; ------------- Check if position is negative

TextGetChar6:	mov	eax,UNINOASC	; EAX <- invalid character
		or	edx,edx		; is position negative?
		js	TextGetChar8	; position is negative

; ------------- Position is behind end of text

		mov	edx,ecx		; EDX <- limit maximal position
		stc			; set error flag
		jmp	short TextGetChar4

; ------------- Position is before start of text

TextGetChar8:	xor	edx,edx		; EDX <- 0, minimal position
		stc			; set error flag
		jecxz	TextGetChar4	; no text
		jmp	short TextGetChar2		

; -----------------------------------------------------------------------------
;                        Get first character from text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EAX = Unicode character (or UNINOASC on error)
;		CY = error, no valid character (EAX = UNINOASC)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextGetFirst:	push	edx		; push EDX

; ------------- Prepare position of first character (-> EDX)

		call	TextFirst	; get position of first character

; ------------- Read first character

		call	TextGetChar	; read first character from text

; ------------- Pop registers

		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                        Get last character from text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EAX = Unicode character (or UNINOASC on error)
;		CY = error, no valid character (EAX = UNINOASC)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextGetLast:	push	edx		; push EDX

; ------------- Prepare position of last character (-> EDX)

		call	TextLast	; get position of last character

; ------------- Read last character

		call	TextGetChar	; read last character from text

; ------------- Pop registers

		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                         Delete part of text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
;		ECX = length of text to delete (in bytes, may be out of range)
;		EDX = offset of text to delete (it may be out of range)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It limits offset and length to a valid range.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextDelete:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextDelete6	; memory error

; ------------- Limit minimal start position

		or	edx,edx		; check minimal start position
		js	TextDelete7	; limit minimal start position

; ------------- Check minimal text length

TextDelete1:	or	ecx,ecx		; check text length
		jle	TextDelete6	; no text to delete (here is NC)

; ------------- Check maximal start position

		mov	edi,[ebx]	; EDI <- data buffer
		mov	eax,[edi+TEXT_Length] ; EAX <- text length
		sub	eax,edx		; EAX <- remaining characters
		jle	TextDelete5	; no data left

; ------------- Limit maximal text length

		sub	eax,ecx		; EAX <- rest of text
		js	TextDelete8	; length is too big
		
; ------------- Prepare buffer address

TextDelete3:	lea	edi,[edi+TEXT_Text+edx] ; EDI <- start of deleted text
		lea	esi,[edi+ecx]	; ESI <- end of deleted text

; ------------- New length of text (->EDX)

		add	edx,eax		; EDX <- new length of text

; ------------- Delete text

		mov	ecx,eax		; ECX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; shift text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Resize buffer

		xchg	eax,edx		; EAX <- new length of text
		call	TextResize	; resze buffer

; ------------- Pop registers

TextDelete5:	clc			; clear error flag
TextDelete6:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Limit minimal start position

TextDelete7:	add	ecx,edx		; ECX <- correct text length
		xor	edx,edx		; EDX <- limit start position
		jmp	short TextDelete1

; ------------- Limit maximal text length

TextDelete8:	add	ecx,eax		; ECX <- limit maximal text length
		xor	eax,eax		; EAX <- no text left
		jmp	short TextDelete3

; -----------------------------------------------------------------------------
;                         Delete start of text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
;		ECX = length of text to delete (in bytes, may be out of range)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It limits length to a valid range.
; -----------------------------------------------------------------------------

TextDelStart:	push	edx		; push EDX
		xor	edx,edx		; EDX <- 0, start position
		call	TextDelete	; delete characters from text
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                    Delete rest of text from given position
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
;		EDX = offset of start of deleted text (it may be out of range)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It limits offset to a valid range.
; -----------------------------------------------------------------------------

TextDelFrom:	push	ecx		; push ECX
		mov	ecx,TEXTBIGPOS	; ECX <- number of characters to delete
		call	TextDelete	; delete characters from text
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                             Delete end of text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
;		ECX = length of text to delete (in bytes, may be out of range)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It limits length to a valid range.
; -----------------------------------------------------------------------------

TextDelEnd:	push	edx		; push EDX
		mov	edx,[ebx]	; EDX <- data buffer
		mov	edx,[edx+TEXT_Length] ; EDX <- text length
		sub	edx,ecx		; EDX <- start position
		call	TextDelete	; delete characters from text
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                        Delete character from text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = character position (it may be out of range)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It checks validity of character position.
;		Character is in UTF-8 code and it can be multibyte.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextDelChar:	push	ecx		; push ECX

; ------------- Check minimal character position

		or	edx,edx		; check minimal character position
		js	TextDelChar4	; position is not valid

; ------------- Get length of character

		mov	ecx,edx		; ECX <- current position
		call	TextNext	; shift position to next character
		sub	edx,ecx		; EDX <- length of character

; ------------- Delete character

		xchg	edx,ecx		; ECX <- length, EDX <- old position
		call	TextDelete	; delete character

; ------------- Pop registers

TextDelChar4:	pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                     Delete first character of text string
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

TextDelFirst:	push	edx		; push EDX
		xor	edx,edx		; EDX <- 0, first position
		call	TextDelChar	; delete first character
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                      Delete last character of text string
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT variable
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextDelLast:	push	edx		; push EDX

; ------------- Delete last character

		call	TextLast	; get last character
		call	TextDelFrom	; delete text from given position

; ------------- Pop registers

		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                            Get left part of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
;		ECX = length of left part of text (it may be out of range)
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	It limits length to a valid range.
;		Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextLeft:	cmp	eax,ebx		; is source and destination identical?
		je	TextLeft8	; source is identical with destination

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Empty destination text

		call	TextEmpty	; empty text

; ------------- Check text length

		mov	esi,[eax]	; ESI <- source data buffer
		mov	eax,[esi+TEXT_Length] ; EAX <- source text length
		cmp	ecx,eax		; check text length
		ja	TextLeft6	; limit text length
		xchg	eax,ecx		; EAX <- text length

; ------------- Create destination buffer (it need not be detached)

TextLeft2:	or	eax,eax		; is text empty?
		jz	TextLeft4	; text is empty
		call	TextNew		; create data buffer
		jc	TextLeft4	; memory error

; ------------- Prepare buffers

		add	esi,byte TEXT_Text ; ESI <- source start of text
		mov	edi,[ebx]	; EDI <- destination data buffer
		add	edi,byte TEXT_Text ; EDI <- destination start of text

; ------------- Copy text (it exits with NC)

		mov	ecx,eax		; ECX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; shift text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextLeft4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Limit text length

TextLeft6:	or	ecx,ecx		; check minimal text length
		js	TextLeft4	; invalid text length
		jmp	short TextLeft2	; use text length in EAX

; ------------- Identical source and destination variable (here is EAX == EBX)

TextLeft8:	push	edx		; push EDX
		mov	edx,ecx		; EDX <- end of left part
		call	TextDelFrom	; delete rest of text
		pop	edx		; pop EDX
		jc	TextLeft9	; memory error
		ret

TextLeft9:	call	TextEmpty	; empty text
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                            Get right part of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
;		ECX = length of right part of text (it may be out of range)
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	It limits length to a valid range.
;		Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextRight:	cmp	eax,ebx		; is source and destination identical?
		je	TextRight8	; source is identical with destination

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Check text length

		mov	esi,[eax]	; ESI <- source data buffer
		mov	eax,[esi+TEXT_Length] ; EAX <- source text length
		cmp	ecx,eax		; check text length
		ja	TextRight6	; limit text length
		xchg	eax,ecx		; EAX <- dest length, ECX <- src length

; ------------- Create destination buffer (it need not be detached)

TextRight2:	or	eax,eax		; is text empty?
		jz	TextRight4	; text is empty
		call	TextNew		; create data buffer
		jc	TextRight4	; memory error

; ------------- Prepare offset of text (-> ECX)

		sub	ecx,eax		; ECX <- offset of text

; ------------- Prepare buffers

		lea	esi,[esi+ecx+TEXT_Text]	; ESI <- source start of text
		mov	edi,[ebx]	; EDI <- destination data buffer
		add	edi,byte TEXT_Text ; EDI <- dest. start of text

; ------------- Copy text (it exits with NC)

		mov	ecx,eax		; ECX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; shift text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextRight4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Limit text length

TextRight6:	or	ecx,ecx		; check minimal text length
		js	TextRight4	; invalid text length
		mov	ecx,eax		; ECX <- source length
		jmp	short TextRight2

; ------------- Identical source and destination variable (here is EAX == EBX)

TextRight8:	push	ecx		; push ECX
		push	edx		; push EDX
		mov	edx,[ebx]	; EDX <- data buffer
		sub	ecx,[edx+TEXT_Length] ; ECX <- -length of deleted data
		neg	ecx		; ECX = length of deleted data
		call	TextDelStart	; delete start of text
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		jc	TextLeft9	; memory error
		ret

; -----------------------------------------------------------------------------
;                  Get right part of text from given position
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
;		EDX = start position of right part (it may be out of range)
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	It limits position to a valid range.
;		Source and destination may be identical.
; -----------------------------------------------------------------------------

TextFrom:	push	ecx		; push ECX
		mov	ecx,[eax]	; ECX <- source data buffer
		mov	ecx,[ecx+TEXT_Length] ; ECX <- text length
		sub	ecx,edx		; ECX <- length of right part
		call	TextRight	; get right part of text
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                          Get middle part of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
;		ECX = length of part of text (it may be out of range)
;		EDX = start position of part of text (it may be out of range)
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	It limits length and position to a valid range.
;		Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextMid:	cmp	eax,ebx		; is source and destination identical?
		je	TextMid9	; source is identical with destination

; ------------- Push registers

TextMid1:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Limit minimal start position

		or	edx,edx		; check minimal start position
		js	TextMid7	; limit minimal start position

; ------------- Check minimal text length

TextMid2:	or	ecx,ecx		; check text length
		jle	TextMid6	; no text to copy (here is NC)

; ------------- Check maximal start position (rest of text -> EAX)

		mov	esi,[eax]	; EDI <- data buffer
		mov	eax,[esi+TEXT_Length] ; EAX <- text length
		sub	eax,edx		; EAX <- remaining characters
		jle	TextMid5	; no data to copy

; ------------- Limit maximal text length

		cmp	eax,ecx		; EAX <- rest of text
		jl	TextMid8	; length is too big

; ------------- Create destination buffer (it need not be detached)

TextMid4:	jecxz	TextMid5	; text is empty
		mov	eax,ecx		; EAX <- text length
		call	TextNew		; create data buffer
		jc	TextMid6	; memory error

; ------------- Prepare buffer address

		lea	esi,[esi+TEXT_Text+edx] ; EDI <- start of text
		mov	edi,[ebx]	; EDI <- destination buffer
		add	edi,byte TEXT_Text ; EDI <- destination text

; ------------- Copy text

		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; shift text in DWORDs
		and	eax,byte 3	; EAX <- rest in last DWORD
		xchg	eax,ecx		; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextMid5:	clc			; clear error flag
TextMid6:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Limit minimal start position

TextMid7:	add	ecx,edx		; ECX <- correct text length
		xor	edx,edx		; EDX <- 0, limit start position
		jmp	short TextMid2

; ------------- Limit maximal text length

TextMid8:	xchg	eax,ecx		; ECX <- limit maximal text length
		jmp	short TextMid4

; ------------- Identical source and destination variable (here is EAX == EBX)

TextMid9:	push	eax		; create temporary variable in stack

		mov	ebx,esp		; EBX <- temporary variable
		push	eax		; push EAX (destination variable)
		mov	eax,[eax]	; EAX <- TEXTDATA buffer
		call	TextAttach	; attach text to temporary variable
		xchg	eax,ebx		; EAX <- temporary variable
		pop	ebx		; EBX <- destination variable

		call	TextMid1	; get middle part of text

		pushf			; push flags
		xchg	eax,ebx		; EBX <- temporary, EAX <- destination
		call	TextDetach	; detach temporary variable
		popf			; pop flags

		pop	ebx		; delete temporary variable
		mov	ebx,eax		; EBX <- destination variable
		ret

; -----------------------------------------------------------------------------
;                       Write text UTF-8 into buffer
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		ECX = size of destination buffer (it is ignored if EDX = NULL)
;		EDX = pointer to destination buffer (NULL = get size of data)
; OUTPUT:	EAX = size of data in destination buffer
; -----------------------------------------------------------------------------

; ------------- Push registers

TextWrite:	push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare size of data (-> EAX)

		mov	esi,[ebx]	; ESI <- data buffer
		mov	eax,[esi+TEXT_Length] ; EAX <- text length
		or	edx,edx		; is destination buffer valid?
		jz	TextWrite4	; destination buffer is not valid
		cmp	eax,ecx		; check length of text
		ja	TextWrite8	; limit length of text

; ------------- Prepare pointers

TextWrite2:	add	esi,byte TEXT_Text ; ESI <- start of text
		mov	edi,edx		; EDI <- destination buffer

; ------------- Copy text

		mov	ecx,eax		; ECX <- length of text
		shr	ecx,2		; ECX <- length of text in DWORDs
		rep	movsd		; shift text in DWORDs
		mov	ecx,eax		; ECX <- length of text
		and	ecx,byte 3	; ECX <- rest in last DWORD
		rep	movsb		; copy rest of text

; ------------- Pop registers

TextWrite4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; ------------- Limit length of text

TextWrite8:	xchg	eax,ecx		; EAX <- limit length of text
		jmp	short TextWrite2

; -----------------------------------------------------------------------------
;                  Write text into buffer with given code page
; -----------------------------------------------------------------------------
; INPUT:	AX = code page of destination buffer
;		EBX = pointer to TEXT
;		ECX = size of destination buffer (it is ignored if EDI = NULL)
;		EDX = invalid character (0 = use similar ASCII or default char)
;		EDI = pointer to destination buffer (NULL = get size of data)
; OUTPUT:	EAX = size of data in destination buffer
; -----------------------------------------------------------------------------

; ------------- Push registers

TextWriteCP:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	ebp		; push EBP

; ------------- Convert text

		mov	ebp,ecx		; EBP <- size of destination buffer
		mov	esi,[ebx]	; ESI <- text data
		mov	ecx,[esi+TEXT_Length] ; ECX <- text length
		add	esi,TEXT_Text	; ESI <- start of text
		mov	ebx,eax		; BX <- destination code page
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		call	CharTrans	; convert text

; ------------- Pop registers

		pop	ebp		; pop EBP
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                        Convert text to small letters
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextSmall:	cmp	eax,ebx		; is source and destination identical?
		je	short TextSmall8 ; source is identical with destination

; ------------- Push registers

TextSmall1:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Prepare source pointers

		mov	esi,[eax]	; ESI <- source data
		mov	ecx,[esi+TEXT_Length] ; ECX <- size of source text
		add	esi,TEXT_Text	; ESI <- start of source text
		xor	edx,edx		; EDX <- 0, default invalid character

; ------------- Get required size of buffer

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTransSmall	; get size of destination buffer
		pop	ebx		; pop EBX

; ------------- Create destination buffer (it need not be detached)

		call	TextNew		; create data buffer
		jc	TextSmall4	; memory error

; ------------- Prepare destination pointers

		mov	edi,[ebx]	; EDI <- destination data
		mov	ebp,[edi+TEXT_Length] ; EBP <- size of destination
		add	edi,TEXT_Text	; EDI <- start of destination buffer

; ------------- Convert text

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		call	CharTransSmall	; convert text
		pop	ebx		; pop EBX

; ------------- Pop registers

		clc			; clear error flag
TextSmall4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Identical source and destination variable (here is EAX == EBX)

TextSmall8:	push	eax		; create temporary variable in stack

		mov	ebx,esp		; EBX <- temporary variable
		push	eax		; push EAX (destination variable)
		mov	eax,[eax]	; EAX <- TEXTDATA buffer
		call	TextAttach	; attach text to temporary variable
		xchg	eax,ebx		; EAX <- temporary variable
		pop	ebx		; EBX <- destination variable

		call	TextSmall1	; convert text

		pushf			; push flags
		xchg	eax,ebx		; EBX <- temporary, EAX <- destination
		call	TextDetach	; detach temporary variable
		popf			; pop flags

		pop	ebx		; delete temporary variable
		mov	ebx,eax		; EBX <- destination variable
		ret

; -----------------------------------------------------------------------------
;                        Convert text to capital letters
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextCap:	cmp	eax,ebx		; is source and destination identical?
		je	short TextCap8	; source is identical with destination

; ------------- Push registers

TextCap1:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Prepare source pointers

		mov	esi,[eax]	; ESI <- source data
		mov	ecx,[esi+TEXT_Length] ; ECX <- size of source text
		add	esi,TEXT_Text	; ESI <- start of source text
		xor	edx,edx		; EDX <- 0, default invalid character

; ------------- Get required size of buffer

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTransCap	; get size of destination buffer
		pop	ebx		; pop EBX

; ------------- Create destination buffer (it need not be detached)

		call	TextNew		; create data buffer
		jc	TextCap4	; memory error

; ------------- Prepare destination pointers

		mov	edi,[ebx]	; EDI <- destination data
		mov	ebp,[edi+TEXT_Length] ; EBP <- size of destination
		add	edi,TEXT_Text	; EDI <- start of destination buffer

; ------------- Convert text

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		call	CharTransCap	; convert text
		pop	ebx		; pop EBX

; ------------- Pop registers

		clc			; clear error flag
TextCap4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Identical source and destination variable (here is EAX == EBX)

TextCap8:	push	eax		; create temporary variable in stack

		mov	ebx,esp		; EBX <- temporary variable
		push	eax		; push EAX (destination variable)
		mov	eax,[eax]	; EAX <- TEXTDATA buffer
		call	TextAttach	; attach text to temporary variable
		xchg	eax,ebx		; EAX <- temporary variable
		pop	ebx		; EBX <- destination variable

		call	TextCap1	; convert text

		pushf			; push flags
		xchg	eax,ebx		; EBX <- temporary, EAX <- destination
		call	TextDetach	; detach temporary variable
		popf			; pop flags

		pop	ebx		; delete temporary variable
		mov	ebx,eax		; EBX <- destination variable
		ret

; -----------------------------------------------------------------------------
;                  Convert text to small/capital letters (invert)
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextSmaCap:	cmp	eax,ebx		; is source and destination identical?
		je	short TextSmaCap8; source is identical with destination

; ------------- Push registers

TextSmaCap1:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Prepare source pointers

		mov	esi,[eax]	; ESI <- source data
		mov	ecx,[esi+TEXT_Length] ; ECX <- size of source text
		add	esi,TEXT_Text	; ESI <- start of source text
		xor	edx,edx		; EDX <- 0, default invalid character

; ------------- Get required size of buffer

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTransSmaCap	; get size of destination buffer
		pop	ebx		; pop EBX

; ------------- Create destination buffer (it need not be detached)

		call	TextNew		; create data buffer
		jc	TextSmaCap4	; memory error

; ------------- Prepare destination pointers

		mov	edi,[ebx]	; EDI <- destination data
		mov	ebp,[edi+TEXT_Length] ; EBP <- size of destination
		add	edi,TEXT_Text	; EDI <- start of destination buffer

; ------------- Convert text

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		call	CharTransSmaCap	; convert text
		pop	ebx		; pop EBX

; ------------- Pop registers

		clc			; clear error flag
TextSmaCap4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Identical source and destination variable (here is EAX == EBX)

TextSmaCap8:	push	eax		; create temporary variable in stack

		mov	ebx,esp		; EBX <- temporary variable
		push	eax		; push EAX (destination variable)
		mov	eax,[eax]	; EAX <- TEXTDATA buffer
		call	TextAttach	; attach text to temporary variable
		xchg	eax,ebx		; EAX <- temporary variable
		pop	ebx		; EBX <- destination variable

		call	TextSmaCap1	; convert text

		pushf			; push flags
		xchg	eax,ebx		; EBX <- temporary, EAX <- destination
		call	TextDetach	; detach temporary variable
		popf			; pop flags

		pop	ebx		; delete temporary variable
		mov	ebx,eax		; EBX <- destination variable
		ret

; -----------------------------------------------------------------------------
;           Convert text to words (first letter of word is capital)
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT
;		EBX = pointer to destination TEXT
; OUTPUT:	CY = memory error, destination TEXT becomes empty
; NOTES:	Source and destination may be identical.
; -----------------------------------------------------------------------------

; ------------- Check if source and destination is identical

TextWords:	cmp	eax,ebx		; is source and destination identical?
		je	short TextWords8 ; source is identical with destination

; ------------- Push registers

TextWords1:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Empty destination text variable

		call	TextEmpty	; empty text

; ------------- Prepare source pointers

		mov	esi,[eax]	; ESI <- source data
		mov	ecx,[esi+TEXT_Length] ; ECX <- size of source text
		add	esi,TEXT_Text	; ESI <- start of source text
		xor	edx,edx		; EDX <- 0, default invalid character

; ------------- Get required size of buffer

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		xor	edi,edi		; EDI <- 0, no destination buffer
		call	CharTransWords	; get size of destination buffer
		pop	ebx		; pop EBX

; ------------- Create destination buffer (it need not be detached)

		call	TextNew		; create data buffer
		jc	TextWords4	; memory error

; ------------- Prepare destination pointers

		mov	edi,[ebx]	; EDI <- destination data
		mov	ebp,[edi+TEXT_Length] ; EBP <- size of destination
		add	edi,TEXT_Text	; EDI <- start of destination buffer

; ------------- Convert text

		push	ebx		; push EBX
		mov	ax,CP_UTF8	; AX <- source codepage UTF-8
		mov	ebx,eax		; BX <- destination codepage UTF-8
		call	CharTransWords	; convert text
		pop	ebx		; pop EBX

; ------------- Pop registers

		clc			; clear error flag
TextWords4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; ------------- Identical source and destination variable (here is EAX == EBX)

TextWords8:	push	eax		; create temporary variable in stack

		mov	ebx,esp		; EBX <- temporary variable
		push	eax		; push EAX (destination variable)
		mov	eax,[eax]	; EAX <- TEXTDATA buffer
		call	TextAttach	; attach text to temporary variable
		xchg	eax,ebx		; EAX <- temporary variable
		pop	ebx		; EBX <- destination variable

		call	TextWords1	; convert text

		pushf			; push flags
		xchg	eax,ebx		; EBX <- temporary, EAX <- destination
		call	TextDetach	; detach temporary variable
		popf			; pop flags

		pop	ebx		; delete temporary variable
		mov	ebx,eax		; EBX <- destination variable
		ret

; -----------------------------------------------------------------------------
;                     Compare text strings to equality
; -----------------------------------------------------------------------------
; INPUT:	EAX = first text string TEXT
;		EBX = second text string TEXT
; OUTPUT:	CY = string are not equal
; NOTES:	Comparison is case sensitive.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextEqu:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare pointers to text strings

		mov	esi,[eax]	; ESI <- data buffer of first text
		mov	edi,[ebx]	; EDI <- data buffer of second text
		cmp	esi,edi		; compare data buffers
		je	TextEqu8	; string are identical

; ------------- Compare length of strings

		mov	eax,[esi+TEXT_Length] ; EAX <- length of first text
		cmp	eax,[edi+TEXT_Length] ; compare length of strings
		jne	TextEqu6	; strings have different length

; ------------- Prepare pointers to compare strings

		add	esi,TEXT_Text	; ESI <- start of first text
		add	edi,TEXT_Text	; EDI <- start of second text

; ------------- Compare strings (here is NC)

		mov	ecx,eax		; ECX <- length of strings
		jecxz	TextEqu8	; empty strings are equal (here is NC)
		shr	ecx,2		; ECX <- length of strings in DWORD
		repe	cmpsd		; compare strings in DWORD
		jne	TextEqu6	; string are not equal
		and	eax,byte 3	; EAX <- length in last DWORD
		jz	TextEqu8	; strings are equal
		xchg	eax,ecx		; ECX <- length in last DWORD
		repe	cmpsb		; compare string in last DWORD
		je	TextEqu8	; strings are equal

; ------------- Pop registers

TextEqu6:	stc			; set error flag
TextEqu8:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                    Compare text strings alphabetically
; -----------------------------------------------------------------------------
; INPUT:	EAX = first text string TEXT
;		EBX = second text string TEXT
;		EDI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	EAX:	1 = first string is greater (later) than second string
;			0 = strings are equals
;			-1 = first string is smaller (earlier) than second one
; NOTES:	Comparison is not case sensitive.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextComp:	push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP

; ------------- Check if strings are identical

		mov	esi,[eax]	; ESI <- data buffer of first text
		cmp	esi,[ebx]	; compare data buffers
		je	TextComp9	; string are identical

; ------------- Prepare nationality descriptor (-> EDI)

		or	edi,edi		; use default nationality?
		jnz	TextComp1	; no default nationality
		DEFAULT_NAT edi		; EDI <- get default nationality

; ------------- Prepare registers

TextComp1:	xchg	eax,ecx		; ECX <- first text string
		xor	esi,esi		; ESI <- 0, pointer of first string
		xor	edx,edx		; EDX <- 0, pointer of second string

; ------------- Get sorting value of character of first string

TextComp2:	xchg	ecx,ebx		; EBX <- first string, ECX <- second
		xchg	edx,esi		; EDX <- pointer to first string
		call	TextGetChar	; get first character (-> EAX)
		xchg	ecx,ebx		; ECX <- first string, EBX <- second
		xchg	edx,esi		; ESI <- pointer to first string
		jc	TextComp6	; end of first string
		call	UniCharSortCap	; get sorting value (-> EAX)
		xchg	eax,ebp		; EBP <- save first character

; ------------- Get sorting value of character of second string

		call	TextGetChar	; get second character (-> EAX)
		jc	TextComp8	; end of second string
		call	UniCharSortCap	; get sorting value (-> EAX)

; ------------- Compare characters

		cmp	ebp,eax		; compare characters
		je	TextComp2	; characters are identical, try next

; ------------- Get result (here is CY if FIRST < SECOND)

		sbb	eax,eax		; EAX <- 0 if greater, -1 if smaller
		shl	eax,1		; EAX <- 0 if greater, -2 if smaller
TextComp3:	inc	eax		; EAX <- 1 if greater, -1 if smaller

; ------------- Pop registers

TextComp4:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret		

; ------------- End of first string (check end of second string)

TextComp6:	call	TextGetChar	; get second character
		sbb	eax,eax		; EAX <- 0 if smaller, -1 if equal
		neg	eax		; EAX <- 0 if smaller, 1 if equal
		dec	eax		; EAX <- -1 if smaller, 0 if equal
		jmp	short TextComp4

; ------------- End of second string (EAX <- 1, first string is greater)

TextComp8:	xor	eax,eax		; EAX <- 0
		jmp	short TextComp3

; ------------- String are identical

TextComp9:	xor	eax,eax		; EAX <- 0
		jmp	short TextComp4

; -----------------------------------------------------------------------------
;                       Add formated text to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT variable
;		EBX = pointer to destination TEXT variable
;		ECX = number of DWORDs in array of arguments (max. 256)
;		EDX = pointer to array of arguments
;		ESI = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error or invalid argument index (text not changed)
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):
;
;  ECX = local variables in stack

%define		TFRMSrc		ecx+24	; (4) source TEXT
%define		TFRMDst		ecx+20	; (4) destination TEXT
%define		TFRMArgN	ecx+16	; (4) number of arguments
%define		TFRMArg		ecx+12	; (4) pointer to arguments
%define		TFRMNat		ecx+8	; (4) pointer to nationality

; ------------- Push registers

TextAddFormat:	push	eax		; push EAX (source TEXT)
		push	ebx		; push EBX (destination TEXT)
		push	ecx		; push ECX (number of arguments)
		push	edx		; push EDX (pointer to arguments)
		push	esi		; push ESI (pointer to nationality)
		push	edi		; push EDI
		push	ebp		; push EBP
		mov	ecx,esp		; ECX <- local variables

; ------------- Get text length (-> ESI)

		mov	edx,[eax]	; EDX <- source text data
		mov	ebx,[edx+TEXT_Length] ; EBX <- size of source text
		add	edx,TEXT_Text	; EDX <- start of source text
		mov	ebp,TextAddFormCB ; EBP <- callback function
		xchg	eax,ecx		; EAX <- callback value,local variables
		call	FormToTextBufN	; get text length
		xchg	eax,ecx		; ECX <- local variables
		jc	TextAddFormat9	; invalid argument index

; ------------- Copy text on write

		mov	ebx,[TFRMDst]	; EBX <- destination text
		call	TextCopyWrite	; copy text on write
		jc	TextAddFormat9	; memory error

; ------------- Resize buffer

		mov	eax,[ebx]	; EAX <- destination data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		add	eax,esi		; EAX <- new text length
		call	TextResize	; resize data buffer
		jc	TextAddFormat9	; memory error

; ------------- Format text

		sub	eax,esi		; EAX <- old length of text
		add	eax,[ebx]	; EAX <- data buffer
		add	eax,TEXT_Text	; EAX <- start of new text
		xchg	eax,edi		; EDI <- start of new text
		mov	edx,[TFRMSrc]	; EDX <- pointer to source variable
		mov	edx,[edx]	; EDX <- source text data
		mov	ebx,[edx+TEXT_Length] ; EBX <- size of source text
		add	edx,TEXT_Text	; EDX <- start of source text
		mov	ebp,TextAddFormCB ; EBP <- callback function
		mov	eax,[TFRMNat]	; EAX <- pointer to nationality
		or	eax,eax		; default nationality?
		jnz	TextAddFormat6	; nationality is valid
		DEFAULT_NAT eax		; EAX <- get default nationality
TextAddFormat6:	xchg	eax,ecx		; EAX <- callback value,local variables
		call	FormToTextBuf	; format text into buffer

; ------------- Pop registers

TextAddFormat9:	pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;          Local function - Add formated text to end of text, callback
; -----------------------------------------------------------------------------
; INPUT:	EAX = callback value
;		EBX = formatting parameters with argument type
;		EDX = DWORD index in argument stack (0 to 255)
; OUTPUT:	CY=error, invalid argument index (no valid value returned)
;		EAX = width or precision parameter
;		EDX:EAX integer
;		EAX character UNICODE
;		ST0 floating point (only if NC)
;		EAX pointer to text UTF-8, EDX length of text
; DESTROYS:	EBX, ECX, EDX:EAX (if not returning a value)
; NOTES:	64-bit integer or double float takes 2 DWORDs, extended double
;		takes 3 DWORDs and other numbers take 1 DWORD. Argument index
;		refers to index of DWORD argument, not to real argument number.
; -----------------------------------------------------------------------------

; ------------- Prepare pointer to local variables

TextAddFormCB:	xchg	eax,ecx		; ECX <- local variables

; ------------- Check argument index and size

		call	FormGetArgLen	; get argument length -> EAX
		add	eax,edx		; EAX <- end of arguments
		cmp	[TFRMArgN],eax	; check index and size of argument
		jb	TextAddFormCB9	; invalid argument index or size

; ------------- Prepare pointer to argument (-> EDX)

		shl	edx,2		; EDX <- argument index * 4
		add	edx,[TFRMArg]	; EDX <- argument address

; ------------- Prepare argument type (-> AL) and flags (-> AH)

		xchg	eax,ebx		; EAX <- formatting parameters
		shr	eax,FORMPAR_TypeF_b ; EAX <- type and flags
		and	al,FORMTYPE_Mask0 ; AL <- argument type

; ------------- Read width/precision parameter

		jnz	TextAddFormCB2	; not parameter
TextAddFormCB1:	mov	eax,[edx]	; EAX <- get parameter
		ret			; return with NC

; ------------- Read single character

TextAddFormCB2:	cmp	al,FORMTYPE_Char ; single character "c","C"?
		je	TextAddFormCB1	; single character

; ------------- Read string

		cmp	al,FORMTYPE_String ; string "s", "S"?
		jne	TextAddFormCB3
		mov	eax,[edx]	; EAX <- pointer to text variable
		mov	eax,[eax]	; EAX <- pointer to text data
		mov	edx,[eax+TEXT_Length] ; EDX <- length of text
		add	eax,TEXT_Text	; EAX <- start of text
		ret			; return with NC

; ------------- Read integer

TextAddFormCB3:	cmp	al,FORMTYPE_ArgInt ; integer?
		ja	TextAddFormCB5	; not integer

; ------------- Integer 16-bit

		test	ah,FORMFLAG2_Shrt ; short argument?
		jz	TextAddFormCB43	; not short argument
		cmp	al,FORMTYPE_Int	; signed integer?
		jne	TextAddFormCB42	; not signed integer
		movsx	eax,word [edx]	; EAX <- signed short
TextAddFormCB41:cdq			; EDX:EAX <- signed integer
		ret			; return with NC

TextAddFormCB42:movzx	eax,word [edx]	; EAX <- unsigned short
		cdq			; EDX:EAX <- unsigned integer
		clc			; clear error flag
		ret			; return with NC

; ------------- Integer 32-bit

TextAddFormCB43:test	ah,FORMFLAG2_Long ; long argument?
		jnz	TextAddFormCB44	; long argument
		cmp	al,FORMTYPE_Int	; signed integer?
		mov	eax,[edx]	; EAX <- load integer
		je	TextAddFormCB41	; signed integer
		xor	edx,edx		; EDX:EAX <- unsigned integer
		ret			; return with NC

; ------------- Integer 64-bit

TextAddFormCB44:mov	eax,[edx]	; EAX <- integer LOW
		mov	edx,[edx+4]	; EDX <- integer HIGH
		ret			; return with NC

; ------------- Single float

TextAddFormCB5:	test	ah,FORMFLAG2_Shrt ; short argument?
		jz	TextAddFormCB52	; not short argument
		fld	dword [edx]	; ST0 <- load single float
		ret			; return with NC

; ------------- Double float

TextAddFormCB52:test	ah,FORMFLAG2_Long ; long argument?
		jnz	TextAddFormCB54	; long argument
		fld	qword [edx]	; ST0 <- load double float
		ret			; return with NC

; ------------- Extended double float

TextAddFormCB54:fld	tword [edx]	; ST0 <- load extended double float
TextAddFormCB9:	ret			; return with NC

; -----------------------------------------------------------------------------
;  Add formated text to end of text with DWORD argument and default nationality
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT variable
;		EBX = pointer to destination TEXT variable
;		EDX = DWORD argument (integer, character, pointer to TEXT)
; OUTPUT:	CY = memory error or invalid argument index (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddFormatDW:push	ecx		; push ECX
		push	edx		; push EDX (integer argument)
		push	esi		; push ESI

; ------------- Format text

		xor	ecx,ecx		; ECX <- 0
		lea	edx,[esp+4]	; EDX <- pointer to argument
		inc	ecx		; ECX <- 1, number of DWORD arguments		
		xor	esi,esi		; ESI <- 0, default nationality
		call	TextAddFormat	; format text

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;  Add formated text to end of text with float argument and default nationality
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT variable
;		EBX = pointer to destination TEXT variable
;		ST0 = float argument (number will be removed from FPU stack)
; OUTPUT:	CY = memory error or invalid argument index (text not changed)
; NOTES:	It uses double float size of argument (neither H nor L size).
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddFormatF:	push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Store number into stack

		push	ecx		; reserve 1 DWORD
		push	ecx		; reserve 1 DWORD
		mov	edx,esp		; EDX <- pointer to stack
		fstp	qword [edx]	; store number into stack

; ------------- Format text

		xor	ecx,ecx		; ECX <- 0
		mov	cl,2		; ECX <- 2, number of DWORD arguments		
		xor	esi,esi		; ESI <- 0, default nationality
		call	TextAddFormat	; format text

; ------------- Destroys local buffer

		pop	ecx		; destroy 1 DWORD
		pop	ecx		; destroy 1 DWORD

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                         Add date/time to end of text
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to source TEXT variable
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
;		EDX = pointer to date-time structure DATETIME
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):

%define		TFDTSrc		esp+20	; (4) source TEXT
%define		TFDTDst		esp+16	; (4) destination TEXT
%define		TFDTNat		esp+12	; (4) pointer to nationality
%define		TFDTDat		esp+8	; (4) pointer to DATETIME

; ------------- Push registers

TextAddDateTime:push	eax		; push EAX (source TEXT)
		push	ebx		; push EBX (destination TEXT)
		push	ecx		; push ECX (pointer to nationality)
		push	edx		; push EDX (pointer to DATETIME)
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Get text length (-> ESI)

		xchg	eax,edx		; EAX <- DATETIME, EDX <- source TEXT
		mov	edx,[edx]	; EDX <- source text data
		mov	ebx,[edx+TEXT_Length] ; EBX <- size of source text
		add	edx,TEXT_Text	; EDX <- start of source text
		call	FormDateTimeN	; get text length

; ------------- Copy text on write

		mov	ebx,[TFDTDst]	; EBX <- destination text
		call	TextCopyWrite	; copy text on write
		jc	TextAddDateTim9	; memory error

; ------------- Resize buffer

		mov	eax,[ebx]	; EAX <- destination data buffer
		mov	eax,[eax+TEXT_Length] ; EAX <- length of data buffer
		add	eax,esi		; EAX <- new text length
		call	TextResize	; resize data buffer
		jc	TextAddDateTim9	; memory error

; ------------- Default nationality

		mov	ecx,[TFDTNat]	; ECX <- pointer to nationality
		or	ecx,ecx		; default nationality?
		jnz	TextAddDateTim6	; nationality is valid
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format text

TextAddDateTim6:sub	eax,esi		; EAX <- old length of text
		add	eax,[ebx]	; EAX <- data buffer
		add	eax,TEXT_Text	; EAX <- start of new text
		xchg	eax,edi		; EDI <- start of new text
		mov	edx,[TFDTSrc]	; EDX <- pointer to source variable
		mov	edx,[edx]	; EDX <- source text data
		mov	ebx,[edx+TEXT_Length] ; EBX <- size of source text
		add	edx,TEXT_Text	; EDX <- start of source text
		mov	eax,[TFDTDat]	; EAX <- DATETIME structure
		call	FormDateTime	; format text into buffer

; ------------- Pop registers

		clc			; clear error flag
TextAddDateTim9:pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                       Add absolute time to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
;		ESI = pointer to source TEXT variable
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------
; Local variables (ebp+N are read-only variables):

%define		TFATDst		ebp+8	; (4) destination TEXT

; ------------- Push registers

TextAddAbsTime:	push	ebx		; push EBX (destination TEXT)
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- push ESP
		sub	esp,DATETIME_size ; ESP <- local buffer

; ------------- Decode absolute time into date-time structure

		mov	ebx,esp		; EBX <- local buffer
		call	AbsToDateTimeExt ; convert absolute time into DATETIME

; ------------- Convert date/time to text

		mov	edx,ebx		; EDX <- DATETIME structure
		mov	ebx,[TFATDst]	; EBX <- destination TEXT
		xchg	eax,esi		; EAX <- pointer to source TEXT
		call	TextAddDateTime	; convert date/time to text
		xchg	eax,esi		; return ESI

; ------------- Pop registers

		mov	esp,ebp		; pop ESP
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                     Add short time format to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It uses default nationality descriptor.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddShrtTime:push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Det default nationality (-> ECX)

		or	ecx,ecx		; default nationality?
		jnz	TextAddShrtTim2	; nationality is valid
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format text

TextAddShrtTim2:lea	esi,[ecx+NAT_ShortTime] ; ESI <- short time string
		call	TextAddAbsTime	; format text

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                     Add long time format to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It uses default nationality descriptor.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddLongTime:push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Det default nationality (-> ECX)

		or	ecx,ecx		; default nationality?
		jnz	TextAddLongTim2	; nationality is valid
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format text

TextAddLongTim2:lea	esi,[ecx+NAT_LongTime] ; ESI <- long time string
		call	TextAddAbsTime	; format text

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                     Add short date format to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It uses default nationality descriptor.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddShrtDate:push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Det default nationality (-> ECX)

		or	ecx,ecx		; default nationality?
		jnz	TextAddShrtDat2	; nationality is valid
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format text

TextAddShrtDat2:lea	esi,[ecx+NAT_ShortDate] ; ESI <- short date string
		call	TextAddAbsTime	; format text

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                     Add long date format to end of text
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = pointer to destination TEXT variable
;		ECX = pointer to nationality descriptor NATIONAL (NULL=default)
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	It uses default nationality descriptor.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextAddLongDate:push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Det default nationality (-> ECX)

		or	ecx,ecx		; default nationality?
		jnz	TextAddLongDat2	; nationality is valid
		DEFAULT_NAT ecx		; ECX <- get default nationality

; ------------- Format text

TextAddLongDat2:lea	esi,[ecx+NAT_LongDate] ; ESI <- long date string
		call	TextAddAbsTime	; format text

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                    Find 1 byte in text in forward direction
; -----------------------------------------------------------------------------
; INPUT:	AL = byte to find
;		EBX = pointer to TEXT
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found byte (or EDX = -1 if data not found)
;		CY = data not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the byte.
; -----------------------------------------------------------------------------

; ------------- Find first occurrence of 1 byte

TextFindByteFirst:
		xor	edx,edx		; EDX <- 0, find first occurrence

; ------------- Push registers

TextFindByte:	push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Limit minimal offset to 0 (-> EDX)

		or	edx,edx		; is start offset negative?
		sets	cl		; CL <- 0 if EDX >= 0, 1 if EDX < 0
		movzx	ecx,cl		; ECX <- 0 if EDX >= 0, 1 if EDX < 0
		dec	ecx		; ECX <- -1 if EDX >= 0, 0 if EDX < 0
		and	edx,ecx		; EDX <- 0 if EDX < 0

; ------------- Get text length (-> ECX) and text address (-> EDI)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer

; ------------- Find data

		sub	ecx,edx		; ECX <- remaining data
		jle	short TextFindWord6 ; data not found
		repne	scasb		; find data
		jne	short TextFindWord6 ; data not found

; ------------- Get offset of data

		inc	ecx		; ECX <- return found data
		mov	edx,[ebx]	; EDX <- data buffer
		mov	edx,[edx+TEXT_Length] ; EDX <- text length
		sub	edx,ecx		; EDX <- data offset (it sets NC)

; ------------- OK: Pop registers (here is NC)

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                  Find 2 bytes in text in forward direction
; -----------------------------------------------------------------------------
; INPUT:	AL = first byte of string to find
;		AH = second byte of string to find
;		EBX = pointer to TEXT
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found bytes (or EDX = -1 if data not found)
;		CY = data not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the bytes.
; -----------------------------------------------------------------------------

; ------------- Find first occurrence of 2 bytes

TextFindWordFirst:
		xor	edx,edx		; EDX <- 0, find first occurrence

; ------------- Push registers

TextFindWord:	push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Limit minimal offset to 0 (-> EDX)

		or	edx,edx		; is start offset negative?
		sets	cl		; CL <- 0 if EDX >= 0, 1 if EDX < 0
		movzx	ecx,cl		; ECX <- 0 if EDX >= 0, 1 if EDX < 0
		dec	ecx		; ECX <- -1 if EDX >= 0, 0 if EDX < 0
		and	edx,ecx		; EDX <- 0 if EDX < 0

; ------------- Get text length (-> ECX) and text address (-> EDI)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text
		dec	ecx		; ECX <- without last byte
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer

; ------------- Find data

		sub	ecx,edx		; ECX <- remaining data
		jle	short TextFindWord6 ; data not found
TextFindWord2:	repne	scasb		; find data
		jne	short TextFindWord6 ; data not found
		cmp	ah,[edi]	; check second byte
		jne	TextFindWord2	; continue searching

; ------------- Get offset of data

		inc	ecx		; ECX <- return found data
		inc	ecx		; ECX <- return found data
		mov	edx,[ebx]	; EDX <- data buffer
		mov	edx,[edx+TEXT_Length] ; EDX <- text length
		sub	edx,ecx		; EDX <- data offset (it sets NC)

; ------------- OK: Pop registers (here is NC)

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; ------------- Data not found (here jumps from other functions, too)

TextFindWord5:	cld			; set direction up
TextFindWord6:	xor	edx,edx		; EDX <- 0
		dec	edx		; EDX <- -1, data not found
		stc			; set error flag
		
; ------------- ERROR: Pop registers

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                     Find 1 byte in text in reverse direction
; -----------------------------------------------------------------------------
; INPUT:	AL = byte to find
;		EBX = pointer to TEXT
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found byte (or EDX = -1 if data not found)
;		CY = data not found (EDX = -1)
; NOTES:	Set EDX to TEXTBIGPOS to find last occurrence of the byte.
; -----------------------------------------------------------------------------

; ------------- Find last occurrence of 1 byte

TextFindByteLast:
		mov	edx,TEXTBIGPOS	; EDX <- find last occurrence

; ------------- Push registers

TextFindByteRev:push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Get number of comparisons (-> ECX)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text

; ------------- Limit maximal offset (-> EDX)

		cmp	edx,ecx		; check maximal offset
		jl	TextFindByteRe2	; offset is OK
		mov	edx,ecx		; EDX <- text length
		dec	edx		; EDX <- offset of last byte

; ------------- Get text address (-> EDI)

TextFindByteRe2:or	edx,edx		; is offset valid?
		js	short TextFindWord6 ; invalid offset
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer

; ------------- Find data
	
		mov	ecx,edx		; ECX <- offset
		std			; set direction down
		inc	ecx		; ECX <- number of bytes
		repne	scasb		; find data
		cld			; set direction up
		jne	short TextFindWord6 ; data not found

; ------------- Get offset of data (here is NC)

		mov	edx,ecx		; EDX <- remaining bytes

; ------------- OK: Pop registers (here is NC)

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                   Find 2 bytes in text in reverse direction
; -----------------------------------------------------------------------------
; INPUT:	AL = first byte of string to find
;		AH = second byte of string to find
;		EBX = pointer to TEXT
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found bytes (or EDX = -1 if data not found)
;		CY = data not found (EDX = -1)
; NOTES:	Set EDX to TEXTBIGPOS to find last occurrence of the bytes.
; -----------------------------------------------------------------------------

; ------------- Find last occurrence of 2 bytes

TextFindWordLast:
		mov	edx,TEXTBIGPOS	; EDX <- find last occurrence

; ------------- Push registers

TextFindWordRev:push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Get number of comparisons (-> ECX)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text
		dec	ecx		; ECX <- length - 1

; ------------- Limit maximal offset (-> EDX)

		cmp	edx,ecx		; check maximal offset
		jl	TextFindWordRe2	; offset is OK
		mov	edx,ecx		; EDX <- text length
		dec	edx		; EDX <- offset of last byte

; ------------- Get text address (-> EDI)

TextFindWordRe2:or	edx,edx		; is offset valid?
		js	short TextFindWord6 ; invalid offset
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer

; ------------- Find data
	
		mov	ecx,edx		; ECX <- offset
		std			; set direction down
		inc	ecx		; ECX <- number of bytes
TextFindWordRe4:repne	scasb		; find data
		jne	short TextFindWord5 ; data not found
		cmp	ah,[edi+2]	; check second byte
		jne	TextFindWordRe4	; continue searching
		cld			; set direction up

; ------------- Get offset of data (here is NC)

		mov	edx,ecx		; EDX <- remaining bytes

; ------------- OK: Pop registers (here is NC)

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                       Find text in forward direction
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to TEXT to find
;		EBX = pointer to TEXT where to search
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found text (or EDX = -1 if text not found)
;		CY = text not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the text.
; -----------------------------------------------------------------------------
; TEXT_Ref of text to find may be invalid (see TextFindChar, TextFindCharRev).

; ------------- Find first occurrence of the text

TextFindFirst:	xor	edx,edx		; EDX <- 0, find first occurrence

; ------------- Push registers 1

TextFind:	push	eax		; push EAX
		push	esi		; push ESI
		push	ebp		; push EBP

; ------------- Use faster method

		mov	esi,[eax]	; ESI <- source data buffer
		mov	ebp,[esi+TEXT_Length] ; EBP <- text length
		or	ebp,ebp		; zero length?
		jz	short TextFind9	; zero length, always return NC
		mov	al,[esi+TEXT_Text] ; AL <- first byte
		dec	ebp		; length 1?
		jz	TextFind2	; text is one byte long
		mov	ah,[esi+TEXT_Text+1] ; AH <- second byte
		dec	ebp		; length 2?
		jnz	TextFind3	; text is too long
		call	TextFindWord	; find two bytes
		jmp	short TextFind9
TextFind2:	call	TextFindByte	; find one byte
		jmp	short TextFind9

; ------------- Push registers 2

TextFind3:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Limit minimal offset to 0 (-> EDX)

		or	edx,edx		; is start offset negative?
		sets	cl		; CL <- 0 if EDX >= 0, 1 if EDX < 0
		movzx	ecx,cl		; ECX <- 0 if EDX >= 0, 1 if EDX < 0
		dec	ecx		; ECX <- -1 if EDX >= 0, 0 if EDX < 0
		and	edx,ecx		; EDX <- 0 if EDX < 0

; ------------- Get text length (-> ECX) and text address (-> EDI)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text
		sub	ecx,ebp		; ECX <- without text length - 2
		dec	ecx		; ECX <- without first byte
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer
		sub	ecx,edx		; ECX <- remaining data
		jle	short TextFind7	; data not found

; ------------- Find begin of data

		mov	edx,[ebx]	; EDX <- data buffer
		add	esi,TEXT_Text+2	; ESI <- start of text to find
TextFind4:	repne	scasb		; find data
		jne	short TextFind7	; data not found
		cmp	ah,[edi]	; check second byte
		jne	TextFind4	; text not equal

; ------------- Compare rest of text

		push	esi		; push ESI
		push	edi		; push EDI
		mov	ebx,ecx		; EBX <- push ECX

		inc	edi		; EDI <- start of text
		mov	ecx,ebp		; ECX <- text length - 2
		shr	ecx,2		; ECX <- text length in DWORDs
		repe	cmpsd		; compare texts in DWORDs
		jne	TextFind6	; texts not equal
		mov	ecx,ebp		; ECX <- text length
		and	ecx,byte 3	; ECX <- rest of text in last DWORD
		repe	cmpsb		; compare bytes in last DWORD

TextFind6:	mov	ecx,ebx		; pop ECX
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		jne	TextFind4	; text not equal

; ------------- Get offset of data

		lea	ecx,[ebp+ecx+2]	; ECX <- return found data
		mov	edx,[edx+TEXT_Length] ; EDX <- text length
		sub	edx,ecx		; EDX <- data offset (it sets NC)
		jmp	short TextFind8

; ------------- Text not found

TextFind7:	xor	edx,edx		; EDX <- 0
		dec	edx		; EDX <- -1, data not found
		stc			; set error flag

; ------------- Pop registers 2

TextFind8:	pop	edi		; pop EDI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX

; ------------- Pop registers 1

TextFind9:	pop	ebp		; pop EBP
		pop	esi		; pop ESI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                      Find text in reverse direction
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to TEXT to find
;		EBX = pointer to TEXT where to search
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found text (or EDX = -1 if text not found)
;		CY = text not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the text.
; -----------------------------------------------------------------------------

; ------------- Find last occurrence of the text

TextFindLast:	mov	edx,TEXTBIGPOS	; EDX <- find last occurrence

; ------------- Push registers 1

TextFindRev:	push	eax		; push EAX
		push	esi		; push ESI
		push	ebp		; push EBP

; ------------- Use faster method

		mov	esi,[eax]	; ESI <- source data buffer
		mov	ebp,[esi+TEXT_Length] ; EBP <- text length
		or	ebp,ebp		; zero length?
		jz	short TextFindRev9 ; zero length, always return NC
		mov	al,[esi+TEXT_Text] ; AL <- first byte
		dec	ebp		; length 1?
		jz	TextFindRev1	; text is one byte long
		mov	ah,[esi+TEXT_Text+1] ; AH <- second byte
		dec	ebp		; length 2?
		jnz	TextFindRev2	; text is too long
		call	TextFindWordRev	; find two bytes
		jmp	short TextFindRev9
TextFindRev1:	call	TextFindByteRev	; find one byte
		jmp	short TextFindRev9

; ------------- Push registers 2

TextFindRev2:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Get number of comparisons (-> ECX)

		mov	edi,[ebx]	; EDI <- data buffer
		mov	ecx,[edi+TEXT_Length] ; ECX <- length of text
		sub	ecx,ebp		; ECX <- without text length - 2
		dec	ecx		; ECX <- without first byte

; ------------- Limit maximal offset (-> EDX)

		cmp	edx,ecx		; check maximal offset
		jl	TextFindRev3	; offset is OK
		mov	edx,ecx		; EDX <- text length
		dec	edx		; EDX <- offset of last byte

; ------------- Get text address (-> EDI, ESI)

TextFindRev3:	or	edx,edx		; is offset valid?
		js	short TextFindRev7 ; invalid offset
		lea	edi,[edi+TEXT_Text+edx] ; EDI <- start pointer
		add	esi,TEXT_Text+2	; ESI <- start of text to find

; ------------- Find begin of data

		mov	ecx,edx		; ECX <- offset
		inc	ecx		; ECX <- number of bytes
TextFindRev32:	std			; set direction down
TextFindRev4:	repne	scasb		; find data
		jne	short TextFindRev6 ; data not found
		cmp	ah,[edi+2]	; check second byte
		jne	TextFindRev4	; text not equal
		cld			; set direction up

; ------------- Compare rest of text

		push	esi		; push ESI
		push	edi		; push EDI
		mov	ebx,ecx		; EBX <- push ECX

		add	edi,byte 3	; EDI <- start of text
		mov	ecx,ebp		; ECX <- text length - 2
		shr	ecx,2		; ECX <- text length in DWORDs
		repe	cmpsd		; compare texts in DWORDs
		jne	TextFindRev5	; texts not equal
		mov	ecx,ebp		; ECX <- text length
		and	ecx,byte 3	; ECX <- rest of text in last DWORD
		repe	cmpsb		; compare bytes in last DWORD

TextFindRev5:	mov	ecx,ebx		; pop ECX
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		jne	TextFindRev32	; text not equal

; ------------- Get offset of data (here is NC)

		mov	edx,ecx		; EDX <- remaining bytes
		jmp	short TextFindRev8

; ------------- Text not found

TextFindRev6:	cld			; set direction up
TextFindRev7:	xor	edx,edx		; EDX <- 0
		dec	edx		; EDX <- -1, data not found
		stc			; set error flag

; ------------- Pop registers 2

TextFindRev8:	pop	edi		; pop EDI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX

; ------------- Pop registers 1

TextFindRev9:	pop	ebp		; pop EBP
		pop	esi		; pop ESI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Find character in forward direction
; -----------------------------------------------------------------------------
; INPUT:	EAX = UNICODE character
;		EBX = pointer to TEXT where to search
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found text (or EDX = -1 if text not found)
;		CY = text not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the text.
; -----------------------------------------------------------------------------

; ------------- Find first occurrence of the character

TextFindCharFirst:
		xor	edx,edx		; EDX <- 0, find first occurrence

; ------------- Find 1-byte character

TextFindChar:	cmp	eax,7fh		; is it 1-byte character?
		jbe	TextFindByte	; find 1-byte character

; ------------- Find 2-byte character

		push	eax		; push EAX
		cmp	eax,7ffh	; is it 2-byte character?
		ja	TextFindChar2	; it is more than 2-byte character
		shl	eax,2		; save high 2 bits
		shr	al,2		; AL <- low 6 bits
		or	ax,0c080h	; AX <- add flags
		xchg	al,ah		; AL <- first byte, AH <- second byte
		call	TextFindWord	; find 2-byte character
		pop	eax		; pop EAX
		ret

; ------------- Push registers

TextFindChar2:	push	edi		; push EDI
		push	ebp		; push EBP		
		sub	esp,TEXT_size+8+4 ; ESP <- create local buffer

; ------------- Store character into local buffer (we may ignore TEXT_Ref)

		lea	edi,[esp+TEXT_Text] ; EDI <- local buffer
		mov	ebp,8		; EBP <- size of buffer
		call	CharUTF8Write	; write character into UTF-8 buffer
		sub	ebp,8		; EBP <- -text length
		neg	ebp		; EBP <- text length
		mov	[esp+TEXT_Length],ebp ; set text length

; ------------- Find character

		mov	ebp,esp		; EBP <- pointer to TEXTDATA to find
		lea	eax,[esp+TEXT_size+8] ; EAX <- pointer to TEXT variable
		mov	[eax],ebp	; store pointer to TEXTDATA
		call	TextFind	; find character

; ------------- Pop registers

		lea	esp,[esp+TEXT_size+8+4] ; pop ESP
		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                     Find character in reverse direction
; -----------------------------------------------------------------------------
; INPUT:	EAX = UNICODE character
;		EBX = pointer to TEXT where to search
;		EDX = start offset (it can be out of range)
; OUTPUT:	EDX = offset of found text (or EDX = -1 if text not found)
;		CY = text not found (EDX = -1)
; NOTES:	Set EDX to 0 to find first occurrence of the text.
; -----------------------------------------------------------------------------

; ------------- Find last occurrence of the character

TextFindCharLast:
		mov	edx,TEXTBIGPOS	; EDX <- find last occurrence

; ------------- Find 1-byte character

TextFindCharRev:cmp	eax,7fh		; is it 1-byte character?
		jbe	TextFindByteRev	; find 1-byte character

; ------------- Find 2-byte character

		push	eax		; push EAX
		cmp	eax,7ffh	; is it 2-byte character?
		ja	TextFindChrRev2	; it is more than 2-byte character
		shl	eax,2		; save high 2 bits
		shr	al,2		; AL <- low 6 bits
		or	ax,0c080h	; AX <- add flags
		xchg	al,ah		; AL <- first byte, AH <- second byte
		call	TextFindWordRev	; find 2-byte character
		pop	eax		; pop EAX
		ret

; ------------- Push registers

TextFindChrRev2:push	edi		; push EDI
		push	ebp		; push EBP		
		sub	esp,TEXT_size+8+4 ; ESP <- create local buffer

; ------------- Store character into local buffer (we may ignore TEXT_Ref)

		lea	edi,[esp+TEXT_Text] ; EDI <- local buffer
		mov	ebp,8		; EBP <- size of buffer
		call	CharUTF8Write	; write character into UTF-8 buffer
		sub	ebp,8		; EBP <- -text length
		neg	ebp		; EBP <- text length
		mov	[esp+TEXT_Length],ebp ; set text length

; ------------- Find character

		mov	ebp,esp		; EBP <- pointer to TEXTDATA to find
		lea	eax,[esp+TEXT_size+8] ; EAX <- pointer to TEXT variable
		mov	[eax],ebp	; store pointer to TEXTDATA
		call	TextFindRev	; find character

; ------------- Pop registers

		lea	esp,[esp+TEXT_size+8+4] ; pop ESP
		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                          Get text length in bytes
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EAX = text length in bytes
; -----------------------------------------------------------------------------

TextLength:	mov	eax,[ebx]	; EAX <- pointer to TEXTDATA
		mov	eax,[eax+TEXT_Length] ; EAX <- get text length
		ret

; -----------------------------------------------------------------------------
;                     Check if byte offset is valid
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
;		EDX = byte offset
; OUTPUT:	CY = byte offset is not valid (it is out of valid range)
; -----------------------------------------------------------------------------

TextCheckOff:	push	ebx		; push EBX
		mov	ebx,[ebx]	; EBX <- pointer to TEXTDATA
		cmp	edx,[ebx+TEXT_Length] ; check byte offset
		pop	ebx		; pop EBX
		cmc			; CY = invalid byte offset
		ret

; -----------------------------------------------------------------------------
;                        Get text length in characters
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	EAX = text length in characters
; -----------------------------------------------------------------------------

; ------------- Push registers

TextLengthChar:	push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- data buffer
		xor	edx,edx		; EDX <- 0, length accumulator
		mov	ecx,[esi+TEXT_Length] ; ECX <- length of text
		add	esi,TEXT_Text	; ESI <- start of text
		jecxz	TextLengthChar8	; no text

; ------------- Get text length

		xor	eax,eax		; EAX <- 0
TextLengthChar2:lodsb			; AL <- load character
		cmp	al,0feh		; detection bytes?
		jae	TextLengthChar4	; skip detection bytes
		and	al,0c0h		; mask bits
		cmp	al,80h		; is it first byte of character?
		setne	al		; AL <- 1 if it is first byte
		add	edx,eax		; increase text length
TextLengthChar4:loop	TextLengthChar2	; next character

; ------------- Pop registers

TextLengthChar8:xchg	eax,edx		; EAX <- text length
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                    Recalc character position to byte offset
; -----------------------------------------------------------------------------
; INPUT:	EAX = character position (it may be out of range)
;		EBX = pointer to TEXT
; OUTPUT:	EAX = byte offset
; -----------------------------------------------------------------------------

; ------------- Check minimal character position

TextPosToOff:	or	eax,eax		; check minimal character position
		jle	short TextPosToOff9 ; position is <= 0

; ------------- Push registers

		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare registers

		inc	eax		; EAX <- character index + 1
		mov	esi,[ebx]	; ESI <- data buffer
		xchg	eax,edx		; EDX <- required character position
		mov	ecx,[esi+TEXT_Length] ; ECX <- length of text
		add	esi,TEXT_Text	; ESI <- start of text
		jecxz	TextPosToOff6	; no text

; ------------- Convert position to offset

		xor	eax,eax		; EAX <- 0
TextPosToOff2:	lodsb			; AL <- load character
		cmp	al,0feh		; detection bytes?
		jae	short TextPosToOff4 ; skip detection bytes
		and	al,0c0h		; mask bits
		cmp	al,80h		; is it first byte of character?
		setne	al		; AL <- 1 if it is first byte
		sub	edx,eax		; decrease text length
		jz	TextPosToOff6	; found required position
TextPosToOff4:	loop	TextPosToOff2	; next character

; ------------- Get character position (-> EAX)

TextPosToOff6:	mov	esi,[ebx]	; ESI <- data buffer
		mov	eax,[esi+TEXT_Length] ; EAX <- length of text
		sub	eax,ecx		; EAX <- byte offset

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; ------------- Limit minimal character position

TextPosToOff9:	xor	eax,eax		; EAX <- 0, minimal byte offset
		ret

; -----------------------------------------------------------------------------
;                    Recalc byte offset to character position
; -----------------------------------------------------------------------------
; INPUT:	EAX = byte offset (it may be out of range)
;		EBX = pointer to TEXT
; OUTPUT:	EAX = character position
; -----------------------------------------------------------------------------

; ------------- Check minimal byte offset

TextOffToPos:	or	eax,eax		; check minimal character position
		jle	short TextPosToOff9 ; offset is <= 0

; ------------- Push registers

		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- data buffer
		xor	edx,edx		; EDX <- 0, position accumulator
		mov	ecx,[esi+TEXT_Length] ; ECX <- length of text
		cmp	eax,ecx		; check byte offset
		jae	TextOffToPos2	; byte offset is OK
		xchg	eax,ecx		; ECX <- limit byte offset
TextOffToPos2:	add	esi,TEXT_Text	; ESI <- start of text
		jecxz	TextOffToPos8	; no text

; ------------- Convert offset to position

		xor	eax,eax		; EAX <- 0
TextOffToPos4:	lodsb			; AL <- load character
		cmp	al,0feh		; detection bytes?
		jae	TextOffToPos6	; skip detection bytes
		and	al,0c0h		; mask bits
		cmp	al,80h		; is it first byte of character?
		setne	al		; AL <- 1 if it is first byte
		add	edx,eax		; increase character position
TextOffToPos6:	loop	TextOffToPos4	; next character

; ------------- Pop registers

TextOffToPos8:	xchg	eax,edx		; EAX <- text length
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;   Trim text from the left and right (delete spaces and control characters)
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text may be changed)
; -----------------------------------------------------------------------------

TextTrim:	call	TextTrimRight	; trim text from the right
		jc	short TextTrimLeft9 ; memory error

; TextTrimLeft must follow

; -----------------------------------------------------------------------------
;        Trim text from the left (delete spaces and control characters)
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextTrimLeft:	push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- pointer to TEXT
		mov	ecx,[esi+TEXT_Length] ; ECX <- text length
		add	esi,TEXT_Text	; ESI <- start of text
		jecxz	TextTrimLeft6	; nothing to delete (here is NC)

; ------------- Find valid character

		push	ecx		; push ECX
TextTrimLeft2:	cmp	byte [esi],32	; check one byte
		ja	TextTrimLeft4	; it is valid character
		inc	esi		; increase text pointer
		loop	TextTrimLeft2	; next byte
TextTrimLeft4:	xchg	ecx,esi		; ESI <- remaining bytes
		pop	ecx		; pop ECX

; ------------- Delete characters

		sub	ecx,esi		; ECX <- number of bytes
		jecxz	TextTrimLeft6	; nothing to delete
		call	TextDelStart	; delete start of text

; ------------- Pop registers

TextTrimLeft6:	pop	esi		; pop ESI
		pop	ecx		; pop ECX
TextTrimLeft9:	ret

; -----------------------------------------------------------------------------
;        Trim text from the right (delete spaces and control characters)
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextTrimRight:	push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- pointer to TEXT
		mov	ecx,[esi+TEXT_Length] ; ECX <- text length
		lea	esi,[esi+TEXT_Text+ecx-1] ; ESI <- end of text
		clc			; clear error flag
		jecxz	TextTrimRight6	; nothing to delete (here is NC)

; ------------- Find valid character

		push	ecx		; push ECX
TextTrimRight2:	cmp	byte [esi],32	; check one byte
		ja	TextTrimRight4	; it is valid character
		dec	esi		; decrease text pointer
		loop	TextTrimRight2	; next byte
TextTrimRight4:	xchg	ecx,esi		; ESI <- remaining bytes
		pop	ecx		; pop ECX

; ------------- Delete characters

		sub	ecx,esi		; ECX <- number of bytes
		jecxz	TextTrimRight6	; nothing to delete
		call	TextDelEnd	; delete end of text

; ------------- Pop registers

TextTrimRight6:	pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;               Trim spaces and control characters from the text
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextTrimMid:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Copy text on write

		call	TextCopyWrite	; copy text on write
		jc	TextTrimMid8	; memory error

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- pointer to TEXT
		mov	ecx,[esi+TEXT_Length] ; ECX <- text length
		add	esi,TEXT_Text	; ESI <- source pointer
		jecxz	TextTrimMid8	; nothing to delete (here is NC)
		mov	edi,esi		; EDI <- destination pointer

; ------------- Trim spaces and control characters

TextTrimMid2:	lodsb			; AL <- load one character
		cmp	al,33		; check one character
		stosb			; store character
		sbb	edi,byte 0	; return pointer if invalid character
		loop	TextTrimMid2	; next byte

; ------------- Resize data buffer

		cmp	esi,edi		; text changed?
		mov	eax,[ebx]	; EAX <- data buffer
		je	TextTrimMid8	; text not changed
		add	eax,TEXT_Text	; EAX <- start of text
		sub	edi,eax		; EDI <- new length
		xchg	eax,edi		; EAX <- new length
		call	TextResize	; resize data buffer

; ------------- Pop registers

TextTrimMid8:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                  Trim text using list of forbidden characters
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to TEXT with list of forbidden 1-byte characters
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	One-byte characters from the list will be deleted.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextTrimList:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP
		sub	esp,8*4		; ESP <- create local buffer
		mov	ebp,esp		; EBP <- local variables

; ------------- Clear local buffer (size: 8*4*8 = 256 bits)

		xor	ecx,ecx		; ECX <- 0
		mov	[ebp+0*4],ecx	; clear local buffer, DWORD 1
		mov	[ebp+1*4],ecx	; clear local buffer, DWORD 2
		mov	[ebp+2*4],ecx	; clear local buffer, DWORD 3
		mov	[ebp+3*4],ecx	; clear local buffer, DWORD 4
		mov	[ebp+4*4],ecx	; clear local buffer, DWORD 5
		mov	[ebp+5*4],ecx	; clear local buffer, DWORD 6
		mov	[ebp+6*4],ecx	; clear local buffer, DWORD 7
		mov	[ebp+7*4],ecx	; clear local buffer, DWORD 8

; ------------- Map list of characters

		mov	esi,[eax]	; ESI <- source TEXTDATA
		mov	ecx,[esi+TEXT_Length] ; ECX <- source text length
		xor	eax,eax		; EAX <- 0
		jecxz	TextTrimList8	; no data (here is NC)
		add	esi,TEXT_Text	; ESI <- start of text
TextTrimList2:	lodsb			; load 1 character
		bts	dword [ebp],eax	; set bit of character
		loop	TextTrimList2	; next character

; ------------- Copy text on write

TextTrimList4:	call	TextCopyWrite	; copy text on write
		jc	TextTrimList8	; memory error

; ------------- Prepare registers

		mov	esi,[ebx]	; ESI <- pointer to TEXT
		mov	ecx,[esi+TEXT_Length] ; ECX <- text length
		add	esi,TEXT_Text	; ESI <- source pointer
		jecxz	TextTrimList8	; nothing to delete (here is NC)
		mov	edi,esi		; EDI <- destination pointer

; ------------- Trim forbidden characters

TextTrimList6:	lodsb			; AL <- load one character
		bt	dword [ebp],eax	; check one character (CY=invalid)
		stosb			; store character
		sbb	edi,byte 0	; return pointer if invalid character
		loop	TextTrimList6	; next byte

; ------------- Resize data buffer

		cmp	esi,edi		; text changed?
		mov	eax,[ebx]	; EAX <- data buffer
		je	TextTrimList8	; text not changed
		add	eax,TEXT_Text	; EAX <- start of text
		sub	edi,eax		; EDI <- new length
		xchg	eax,edi		; EAX <- new length
		call	TextResize	; resize data buffer

; ------------- Pop registers

TextTrimList8:	lea	esp,[ebp+8*4]	; pop ESP
		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                  Trim text using list of allowed characters
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to TEXT with list of allowed 1-byte characters
;		EBX = pointer to TEXT
; OUTPUT:	CY = memory error (text not changed)
; NOTES:	One-byte characters from the list will be left.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextTrimUnList:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP
		sub	esp,8*4		; ESP <- create local buffer
		mov	ebp,esp		; EBP <- local variables

; ------------- Clear local buffer (size: 8*4*8 = 256 bits)

		xor	ecx,ecx		; ECX <- 0
		dec	ecx		; ECX <- -1
		mov	[ebp+0*4],ecx	; clear local buffer, DWORD 1
		mov	[ebp+1*4],ecx	; clear local buffer, DWORD 2
		mov	[ebp+2*4],ecx	; clear local buffer, DWORD 3
		mov	[ebp+3*4],ecx	; clear local buffer, DWORD 4
		mov	[ebp+4*4],ecx	; clear local buffer, DWORD 5
		mov	[ebp+5*4],ecx	; clear local buffer, DWORD 6
		mov	[ebp+6*4],ecx	; clear local buffer, DWORD 7
		mov	[ebp+7*4],ecx	; clear local buffer, DWORD 8

; ------------- Map list of characters

		mov	esi,[eax]	; ESI <- source TEXTDATA
		mov	ecx,[esi+TEXT_Length] ; ECX <- source text length
		xor	eax,eax		; EAX <- 0
		jecxz	TextTrimList4	; no data - trim all characters
		add	esi,TEXT_Text	; ESI <- start of text
TextTrimUnList2:lodsb			; load 1 character
		btr	dword [ebp],eax	; reset bit of character
		loop	TextTrimUnList2	; next character
		jmp	short TextTrimList4

; -----------------------------------------------------------------------------
;  Find text in multi-language text array in given language and sub-language
; -----------------------------------------------------------------------------
; INPUT:	AX = language identifier LANG
;		EBX = pointer to multi-language text array LANGTEXT
; OUTPUT:	CY = language not found (EBX not changed)
;		EBX = pointer to text variable TEXT (only if NC)
; -----------------------------------------------------------------------------

; ------------- Push registers

TextSubLangGet:	push	ecx		; push ECX
		push	ebx		; push EBX

; ------------- Prepare registers

		movzx	ecx,word [ebx+LANGTEXT_Num] ; ECX<-number of languages

; ------------- Find language

TextSubLangGet2:cmp	ax,[ebx+LANGTEXT_Lang] ; check language identifier
		je	TextSubLangGet4	; language found OK
		add	ebx,byte LANGTEXT_size ; EBX <- next string
		loop	TextSubLangGet2	; check next language

; ------------- Language not found

		stc			; set error flag
		pop	ebx		; pop EBX
		pop	ecx		; pop ECX
		ret

; ------------- Language has been found (here is NC)

TextSubLangGet4:pop	ecx		; destroy EBX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;         Find text in multi-language text array in given language
; -----------------------------------------------------------------------------
; INPUT:	AH = language main identifier
;		EBX = pointer to multi-language text array LANGTEXT
; OUTPUT:	CY = language not found (AL and EBX not changed)
;		AL = found sub-language identifier (only if NC)
;		EBX = pointer to text variable TEXT (only if NC)
; NOTES:	Function is not sub-language sensitive.
; -----------------------------------------------------------------------------

; ------------- Push registers

TextLangGet:	push	ecx		; push ECX
		push	ebx		; push EBX

; ------------- Prepare registers

		movzx	ecx,word [ebx+LANGTEXT_Num] ; ECX<-number of languages

; ------------- Find language

TextLangGet2:	cmp	ah,[ebx+LANGTEXT_Lang+LANG_ID] ; check language ident.
		je	TextLangGet4	; language found OK
		add	ebx,byte LANGTEXT_size ; EBX <- next string
		loop	TextLangGet2	; check next language

; ------------- Language not found

		stc			; set error flag
		pop	ebx		; pop EBX
		pop	ecx		; pop ECX
		ret

; ------------- Language has been found (here is NC)

TextLangGet4:	mov	al,[ebx+LANGTEXT_Lang+LANG_SubID] ; AL <- sub-language
		pop	ecx		; destroy EBX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;           Find text in multi-language text array in nearest language
; -----------------------------------------------------------------------------
; INPUT:	AX = required language identifier LANG
;		EBX = pointer to multi-language text array LANGTEXT
; OUTPUT:	AX = found language identifier LANG
;		EBX = pointer to text variable TEXT
; -----------------------------------------------------------------------------

; ------------- Try to find text with language and sub-language

TextLangNear:	call	TextSubLangGet	; find text
		jnc	TextLangNear8	; text has been found

; ------------- Try to find text with language (ignore sub-language)

		call	TextLangGet	; find text
		jnc	TextLangNear8	; text has been found

; ------------- push registers

		push	ecx		; push ECX

; ------------- Try to find similar language

		movzx	ecx,ah		; ECX <- language main identifier
		mov	ah,[LangNear+ecx] ; AH <- near language
		call	TextLangGet	; find text
		jnc	TextLangNear6	; text has been found

; ------------- Try to find english language

		mov	ah,LANG_ENGLISH	; AH <- english language
		call	TextLangGet	; find text
		jnc	TextLangNear6	; text has been found
		
; ------------- Use first language

		mov	ax,[ebx+LANGTEXT_Lang] ; AX <- language identifier

; ------------- pop registers

TextLangNear6:	pop	ecx		; pop ECX
TextLangNear8:	ret

; -----------------------------------------------------------------------------
;         Find text in multi-language text array in user default language
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to multi-language text array LANGTEXT
; OUTPUT:	EBX = pointer to text variable TEXT
; -----------------------------------------------------------------------------

TextLangDef:	push	eax		; push EAX
		DEFAULT_NAT eax		; EAX <- default nationality descriptor
		mov	ax,[eax+NAT_Language] ; AX <- language
		call	TextLangNear	; find text
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Empty text string

		align	4, db 0
EmptyText:	dd	EmptyTextData
EmptyTextData:	CTEXTDATA	''
