; =============================================================================
;
;                       Litos - Red-black balanced tree
;
; =============================================================================
; A red-black tree is a type of self-balancing binary search tree. Each node
; has a color attribute, the value of which is either red or black.
;
; Requirements:
;	1. Every node is either red or black
;	2. The root is alwys black
;	3. All leaves (dummy empty nodes NULL) are black
;	4. Both children of every red node are black
;	5. All paths from any given node to its leaf nodes contain the same
;	   number of black nodes
; =============================================================================

		CODE_SECTION

; ------------- Macro - set color of node to red (%1 = pointer to node)

%macro		SETRED	1
		mov	byte [%1+RB_Color],RB_RED
%endmacro

; ------------- Macro - set color of node to black (%1 = pointer to node)

%macro		SETBLK	1
		mov	byte [%1+RB_Color],RB_BLACK
%endmacro

; ------------- Macro - test color for red, ZY=is red (%1 = pointer to node)

%macro		CMPRED	1
		cmp	byte [%1+RB_Color],RB_RED
%endmacro

; ------------- Macro - test color for black, ZY=is black (%1=pointer to node)

%macro		CMPBLK	1
		cmp	byte [%1+RB_Color],RB_BLACK
%endmacro

; -----------------------------------------------------------------------------
;                   Initialize red-black balanced tree root
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to root (RBROOT)
; -----------------------------------------------------------------------------

RBTreeInit:	LISTINIT edx		; initialize list of entries
		mov	dword [edx+RB_ColorFlags],RB_BLACK+(RB_ROOT<<8)
		and	dword [edx+RBR_Node],byte 0 ; no first node
		and	dword [edx+RBR_Count],byte 0 ; no entry
		ret

; -----------------------------------------------------------------------------
;          Get first node (in sort order) of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to root (RBROOT)
; OUTPUT:	EBX = first node (NULL = none)
;		CY = no node found (EBX = NULL)
; -----------------------------------------------------------------------------

RBTreeFirst:	mov	ebx,[edx+LIST_Next]	; EBX <- first node
		test	byte [ebx+RB_Flags],RB_ROOT ; is it root?
		jz	RBTreeFirst2		; first node is valid
		xor	ebx,ebx			; EBX <- invalid node
		stc				; set error flag
RBTreeFirst2:	ret

; -----------------------------------------------------------------------------
;           Get last node (in sort order) of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to root (RBROOT)
; OUTPUT:	EBX = last node (NULL = none)
;		CY = no node found (EBX = NULL)
; -----------------------------------------------------------------------------

RBTreeLast:	mov	ebx,[edx+LIST_Prev]	; EBX <- last node
		test	byte [ebx+RB_Flags],RB_ROOT ; is it root?
		jz	RBTreeLast2		; last node is valid
		xor	ebx,ebx			; EBX <- invalid node
		stc				; set error flag
RBTreeLast2:	ret

; -----------------------------------------------------------------------------
;           Get next node (in sort order) of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to current node (RBNODE)
; OUTPUT:	EBX = next node (NULL = none)
;		CY = no other node (EBX = NULL)
; -----------------------------------------------------------------------------

RBTreeNext:	mov	ebx,[ebx+LIST_Next]	; EBX <- next node
		test	byte [ebx+RB_Flags],RB_ROOT ; is it root?
		jz	RBTreeNext2		; next node is valid
		xor	ebx,ebx			; EBX <- invalid node
		stc				; set error flag
RBTreeNext2:	ret

; -----------------------------------------------------------------------------
;          Get previous node (in sort order) of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to current node (RBNODE)
; OUTPUT:	EBX = previous node (NULL = none)
;		CY = no other node (EBX = NULL)
; -----------------------------------------------------------------------------

RBTreePrev:	mov	ebx,[ebx+LIST_Prev]	; EBX <- previous node
		test	byte [ebx+RB_Flags],RB_ROOT ; is it root?
		jz	RBTreePrev2		; previous node is valid
		xor	ebx,ebx			; EBX <- invalid node
		stc				; set error flag
RBTreePrev2:	ret

; -----------------------------------------------------------------------------
;             Get absolute indexed node of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EAX = absolute index of node
;		EDX = pointer to root (RBROOT)
; OUTPUT:	EBX = node (NULL = none, index is not valid)
;		CY = node not found, index is not valid (EBX = NULL)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeIndex:	push	ecx			; push ECX

; ------------- Check index validity

		mov	ecx,[edx+RBR_Count]	; ECX <- number of nodes
		xor	ebx,ebx			; EBX <- 0, invalid index
		cmp	eax,ecx			; is index valid?
		cmc				; CY = index is not valid
		jc	RBTreeIndex8		; index is not valid

; ------------- Search node from begin or end of the list?

		shr	ecx,1			; ECX <- number of nodes/2
		cmp	eax,ecx			; is it upper half?
		jae	RBTreeIndex4		; it is upper half

; ------------- Search node in bottom half of the list

		mov	ecx,eax			; ECX <- required index
		mov	ebx,[edx+LIST_Next]	; EBX <- get first node
		jecxz	RBTreeIndex7		; first node required
RBTreeIndex2:	mov	ebx,[ebx+LIST_Next]	; EBX <- get next node
		loop	RBTreeIndex2		; next node
		jmp	short RBTreeIndex7

; ------------- Search node in upper half of the list

RBTreeIndex4:	mov	ecx,[edx+RBR_Count]	; ECX <- number of nodes
		sub	ecx,eax			; ECX <- remaining nodes
		dec	ecx			; ECX <- reverse index
		mov	ebx,[edx+LIST_Prev]	; EBX <- get last node
		jecxz	RBTreeIndex7		; last node required
RBTreeIndex6:	mov	ebx,[ebx+LIST_Prev]	; EBX <- get previous node
		loop	RBTreeIndex6		; previous node

; ------------- Pop registers

RBTreeIndex7:	clc				; clear error flag
RBTreeIndex8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;              Get relative indexed node of red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EAX = relative index of node (positive or negative)
;		EBX = pointer to current node
; OUTPUT:	EBX = node (NULL = none)
;		CY = node not found (EBX = NULL)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeIndexRel:	push	ecx			; push ECX

; ------------- Check if index is egative or positive

		or	eax,eax			; is index positive?
		mov	ecx,eax			; ECX <- relative index
		jz	RBTreeIndexRel9		; index is 0 (=this node)
		js	RBTreeIndexRel4		; index is negative

; ------------- Index is positive, find next node

RBTreeIndexRel2:call	RBTreeNext		; get next node
		jc	RBTreeIndexRel9		; no next node found
		loop	RBTreeIndexRel2		; next node
		jmp	short RBTreeIndexRel9

; ------------- Index is negative, find previous node

RBTreeIndexRel4:neg	ecx			; ECX <- relative index
RBTreeIndexRel6:call	RBTreePrev		; get previous node
		jc	RBTreeIndexRel9		; no previous node found
		loop	RBTreeIndexRel6		; previous node

; ------------- Pop registers

RBTreeIndexRel9:pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;                    Verify node in red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to node (RBNODE)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	EAX = index of the node (or -1 = node not found)
;		CY = node not found (EAX = -1)
; NOTES: This function can be used to verify that tree contains the node.
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeVerify:	push	ebx			; push EBX
		push	ecx			; push ECX

; ------------- Get last node

		mov	ecx,ebx			; ECX <- node to find
		mov	eax,[edx+RBR_Count]	; EAX <- number of nodes
		dec	eax			; EAX <- index of last node
		call	RBTreeLast		; get last node
		jc	RBTreeVerify8		; no last node (and EAX = -1)

; ------------- Search node

RBTreeVerify2:	cmp	ecx,ebx			; is it searched node?
		je	RBTreeVerify8		; node found OK
		dec	eax			; EAX <- decrease node index
		call	RBTreePrev		; get previous node
		jnc	RBTreeVerify2		; previous node OK, continue

; ------------- Pop registers

RBTreeVerify8:	pop	ecx			; pop ECX
		pop	ebx			; pop EBX
		ret

; -----------------------------------------------------------------------------
;              Get index of the node in red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to node (RBNODE)
; OUTPUT:	EAX = index of the node
;		EDX = pointer to root (RBROOT)
; NOTES:	Node must be included in a valid red-black tree.
; -----------------------------------------------------------------------------

RBTreeGetInx:	xor	eax,eax			; EAX <- 0, node index
		mov	edx,ebx			; EDX <- current node
		dec	eax			; EAX <- -1, preset index
RBTreeGetInx2:	mov	edx,[edx+LIST_Prev]	; EBX <- get previous node
		inc	eax			; EAX <- increase node index
		test	byte [edx+RB_Flags],RB_ROOT ; is it root?
		jz	RBTreeGetInx2		; node is valid, try next one
		ret

; -----------------------------------------------------------------------------
;            Search node in red-black balanced tree with address
; -----------------------------------------------------------------------------
; INPUT:	EAX = address of node to find
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = node not found
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcAddr:	push	ebx			; push EBX
		push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcAddr6		; there is no node in tree

; ------------- Next node is valid

RBTreeSrcAddr2:	mov	ebx,ecx			; node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		cmp	eax,ebx			; test address
		je	RBTreeSrcAddr8		; node found OK

; ------------- Get next node

		seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcAddr2		; next node is valid

; ------------- Error, node not found

RBTreeSrcAddr6:	stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcAddr8:	pop	ecx			; pop ECX
		pop	ebx			; pop EBX
		ret

; -----------------------------------------------------------------------------
;        Search node in red-black balanced tree with WORD data unsigned
; -----------------------------------------------------------------------------
; INPUT:	AX = data to find (unsigned)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found node (NULL = node not found)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcW:	push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcW6		; there is no node in tree

; ------------- Next node is valid

RBTreeSrcW2:	mov	ebx,ecx			; node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		cmp	ax,[ebx+RB_DataW]	; test data
		je	RBTreeSrcW8		; node found OK

; ------------- Get next node

		seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcW2		; next node is valid

; ------------- Error, node not found

RBTreeSrcW6:	xor	ebx,ebx			; EBX <- 0, node not found
		stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcW8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;        Search node in red-black balanced tree with WORD data signed
; -----------------------------------------------------------------------------
; INPUT:	AX = data to find (signed)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found node (NULL = node not found)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcWS:	push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcWS6		; there is no node in tree

; ------------- Next node is valid

RBTreeSrcWS2:	mov	ebx,ecx			; node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		cmp	ax,[ebx+RB_DataW]	; test data
		je	RBTreeSrcWS8		; node found OK

; ------------- Get next node

		setg	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcWS2		; next node is valid

; ------------- Error, node not found

RBTreeSrcWS6:	xor	ebx,ebx			; EBX <- 0, node not found
		stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcWS8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;       Search node in red-black balanced tree with DWORD data unsigned
; -----------------------------------------------------------------------------
; INPUT:	EAX = data to find (unsigned)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found node (NULL = node not found)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcDW:	push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcDW6		; there is no node in the tree

; ------------- Next node is valid

RBTreeSrcDW2:	mov	ebx,ecx			; EBX <- node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		cmp	eax,[ebx+RBN_Data]	; test data
		je	RBTreeSrcDW8		; node find OK

; ------------- Get next node

		seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcDW2		; next node is valid

; ------------- Error, node not found

RBTreeSrcDW6:	xor	ebx,ebx			; EBX <- 0, node not found
		stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcDW8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;       Search node in red-black balanced tree with DWORD data signed
; -----------------------------------------------------------------------------
; INPUT:	EAX = data to find (signed)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found node (NULL = node not found)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcDWS:	push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcDWS6		; there is no node in the tree

; ------------- Next node is valid

RBTreeSrcDWS2:	mov	ebx,ecx			; EBX <- node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		cmp	eax,[ebx+RBN_Data]	; test data
		je	RBTreeSrcDWS8		; node find OK

; ------------- Get next node

		setg	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcDWS2		; next node is valid

; ------------- Error, node not found

RBTreeSrcDWS6:	xor	ebx,ebx			; EBX <- 0, node not found
		stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcDWS8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;     Search node in red-black balanced tree with callback compare function
; -----------------------------------------------------------------------------
; INPUT:	EAX = user data to find (or node with data)
;		EDX = pointer to root (RBROOT)
;		ESI = callback compare function
;			INPUT:	EAX = user data to find (or node with data)
;				EBX = node to test
;			OUTPUT:	ZY = equal, CY = EAX.Data < EBX.Data
;				(as CMP EAX.Data,EBX.Data)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found node (NULL = node not found)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcFnc:	push	ecx			; push ECX

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcFnc6		; there is no node in the tree

; ------------- Next node is valid

RBTreeSrcFnc2:	mov	ebx,ecx			; EBX <- node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		call	esi			; test data
		je	RBTreeSrcFnc8		; node find OK

; ------------- Get next node

		seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcFnc2		; next node is valid

; ------------- Error, node not found

RBTreeSrcFnc6:	xor	ebx,ebx			; EBX <- 0, node not found
		stc				; flag - node not found

; ------------- Pop registers

RBTreeSrcFnc8:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;  Search nearest higher node in red-black tree with callback compare function
; -----------------------------------------------------------------------------
; INPUT:	EAX = user data to find (or node with data)
;		EDX = pointer to root (RBROOT)
;		ESI = callback compare function
;			INPUT:	EAX = user data to find (or node with data)
;				EBX = node to test
;			OUTPUT:	ZY = equal, CY = EAX.Data < EBX.Data
;				(as CMP EAX.Data,EBX.Data)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found nearest higher or equal node (NULL=none)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcHigh:	push	ecx			; push ECX
		push	edi			; push EDI

; ------------- Get first node

		xor	edi,edi			; EDI <- 0, no nearest node
		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcHigh6		; there is no node in the tree

; ------------- Next node is valid

RBTreeSrcHigh2:	mov	ebx,ecx			; EBX <- node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		call	esi			; test data
		je	RBTreeSrcHigh8		; node find OK
		ja	RBTreeSrcHigh4		; data is above node
		mov	edi,ebx			; EDI <- save higher node

; ------------- Get next node

RBTreeSrcHigh4:	seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcHigh2		; next node is valid

; ------------- Equal node not found

RBTreeSrcHigh6:	mov	ebx,edi			; EBX <- nearest node
		cmp	ebx,byte 1		; set error flag CY if NULL

; ------------- Pop registers

RBTreeSrcHigh8:	pop	edi			; pop EDI
		pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;  Search nearest lower node in red-black tree with callback compare function
; -----------------------------------------------------------------------------
; INPUT:	EAX = user data to find (or node with data)
;		EDX = pointer to root (RBROOT)
;		ESI = callback compare function
;			INPUT:	EAX = user data to find (or node with data)
;				EBX = node to test
;			OUTPUT:	ZY = equal, CY = EAX.Data < EBX.Data
;				(as CMP EAX.Data,EBX.Data)
; OUTPUT:	CY = node not found (EBX = NULL)
;		EBX = pointer to found nearest lower or equal node (NULL=none)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeSrcLow:	push	ecx			; push ECX
		push	edi			; push EDI

; ------------- Get first node

		xor	edi,edi			; EDI <- 0, no nearest node
		mov	ecx,[edx+RBR_Node]	; get first node
		jecxz	RBTreeSrcLow6		; there is no node in the tree

; ------------- Next node is valid

RBTreeSrcLow2:	mov	ebx,ecx			; EBX <- node is valid
		xor	ecx,ecx			; ECX <- 0

; ------------- Compare data

		call	esi			; test data
		je	RBTreeSrcLow8		; node find OK
		jb	RBTreeSrcLow4		; data is below node
		mov	edi,ebx			; EDI <- save lower node

; ------------- Get next node

RBTreeSrcLow4:	seta	cl			; CL <- 1 (right), 0 (left)
		mov	ecx,[ebx+RBN_Left+ecx*(RBN_Right-RBN_Left)] ; next node
		or	ecx,ecx			; is next node valid?
		jnz	RBTreeSrcLow2		; next node is valid

; ------------- Equal node not found

RBTreeSrcLow6:	mov	ebx,edi			; EBX <- nearest node
		cmp	ebx,byte 1		; set error flag CY if NULL

; ------------- Pop registers

RBTreeSrcLow8:	pop	edi			; pop EDI
		pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;                    Rotate red-black balanced tree left
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to current node (RBNODE)
;		EDX = pointer to root (RBROOT)
; NOTES:	Right node must be valid.
; -----------------------------------------------------------------------------
;                  ri-le   ri-ri             left   ri-le
;                      \   /                    \   /
;               left   right                     node   ri-ri
;                  \   /           --->             \   /
;             ...   node                      ...   right
;               \   /                           \   /
;               parent                          parent
; -----------------------------------------------------------------------------
; Local variables:
%define		RBRi	EAX			; right node
%define		RBNo	EBX			; current node
%define		RBRL	ECX			; right-left node (hardcoded)
%define		RBRLL	CL			; right-left node LOW (temp.)
%define		RBRo	EDX			; root
%define		RBPa	ESI			; parent

; ------------- Push registers

RBTreeLeft:	push	RBRi			; push RBRi
		push	RBRL			; push RBRL
		push	RBPa			; push RBPa

; ------------- Get other nodes

		mov	RBRi,[RBNo+RBN_Right]	; get right node
		mov	RBRL,[RBRi+RBN_Left]	; get right-left node
		mov	RBPa,[RBNo+RBN_Parent]	; get parent

; ------------- Link right-left node as new right node of current node

		mov	[RBNo+RBN_Right],RBRL	; set as new right node
		jecxz	RBTreeLeft2		; no right-left node
		mov	[RBRL+RBN_Parent],RBNo	; current node is its parent

; ------------- Link current node as new left node of right node

RBTreeLeft2:	mov	[RBRi+RBN_Left],RBNo	; node is left for right one
		mov	[RBNo+RBN_Parent],RBRi	; new parent for current node
		
; ------------- Link right node to root

		mov	[RBRi+RBN_Parent],RBPa	; new parent for right node
		or	RBPa,RBPa		; is parent a root?
		jnz	RBTreeLeft4		; parent is not a root
		mov	[RBRo+RBR_Node],RBRi	; link right node to root
		jmp	short RBTreeLeft8

; ------------- Link right node to valid parent

RBTreeLeft4:    xor	RBRL,RBRL		; RBRL <- 0
		cmp	RBNo,[RBPa+RBN_Right]	; was it right or left node?
		sete	RBRLL			; RBRLL <- 1 (right), 0 (left)
		mov	[RBPa+RBN_Left+RBRL*(RBN_Right-RBN_Left)],RBRi

; ------------- Pop registers

RBTreeLeft8:	pop	RBPa			; pop RBPa
		pop	RBRL			; pop RBRL
		pop	RBRi			; pop RBRi
		ret

%undef		RBRi				; right node
%undef		RBNo				; current node
%undef		RBRL				; right-left node (hardcoded)
%undef		RBRLL				; right-left node LOW (temp.)
%undef		RBRo				; root
%undef		RBPa				; parent

; -----------------------------------------------------------------------------
;                    Rotate red-black balanced tree right
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to current node (RBNODE)
;		EDX = pointer to root (RBROOT)
; NOTES:	Left node must be valid.
; -----------------------------------------------------------------------------
;           le-le   le-ri                        le-ri   right
;               \   /                                \   /
;               left   right                  le-le   node
;                  \   /           --->           \   /
;             ...   node                    ...   left
;               \   /                         \   /
;               parent                        parent
; -----------------------------------------------------------------------------
; Local variables:
%define		RBLe	EAX			; left node
%define		RBNo	EBX			; current node
%define		RBLR	ECX			; left-right node (hardcoded)
%define		RBLRL	CL			; left-right node LOW (temp.)
%define		RBRo	EDX			; root
%define		RBPa	ESI			; parent

; ------------- Push registers

RBTreeRight:	push	RBLe			; push RBLe
		push	RBLR			; push RBLR
		push	RBPa			; push RBPa

; ------------- Get other nodes

		mov	RBLe,[RBNo+RBN_Left]	; get left node
		mov	RBLR,[RBLe+RBN_Right]	; get left-right node
		mov	RBPa,[RBNo+RBN_Parent]	; get parent

; ------------- Link left-right node as new left node of current node

		mov	[RBNo+RBN_Left],RBLR	; set as new left node
		jecxz	RBTreeRight2		; no left-right node
		mov	[RBLR+RBN_Parent],RBNo	; current node is its parent

; ------------- Link current node as new right node of left node

RBTreeRight2:	mov	[RBLe+RBN_Right],RBNo	; node is right for left one
		mov	[RBNo+RBN_Parent],RBLe	; new parent for current node
		
; ------------- Link left node to root

		mov	[RBLe+RBN_Parent],RBPa	; new parent for left node
		or	RBPa,RBPa		; is parent a root?
		jnz	RBTreeRight4		; parent is not a root
		mov	[RBRo+RBR_Node],RBLe	; link left node to root
		jmp	short RBTreeRight8

; ------------- Link left node to valid parent

RBTreeRight4:	xor	RBLR,RBLR		; RBLR <- 0
		cmp	RBNo,[RBPa+RBN_Right]	; was it right or left node?
		sete	RBLRL			; RBRLL <- 1 (right), 0 (left)
		mov	[RBPa+RBN_Left+RBLR*(RBN_Right-RBN_Left)],RBLe

; ------------- Pop registers

RBTreeRight8:	pop	RBPa			; pop RBPa
		pop	RBLR			; pop RBLR
		pop	RBLe			; pop RBLe
		ret

%undef		RBLe				; right node
%undef		RBNo				; current node
%undef		RBLR				; right-left node (hardcoded)
%undef		RBLRL				; right-left node LOW (temp.)
%undef		RBRo				; root
%undef		RBPa				; parent

; -----------------------------------------------------------------------------
;                Insert red color into red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to current node (RBNODE, its color is red)
;		EDX = pointer to root (RBROOT)
; NOTES:	Entry point is RBTreeInsCol.
; -----------------------------------------------------------------------------
; Local variables:
%define		RBGr	EAX			; grand parent
%define		RBNo	EBX			; current node
%define		RBUn	ECX			; uncle
%define		RBRo	EDX			; root
%define		RBPa	ESI			; parent

; ============= Macros

; ------------- Macro - insert color in one direction
; Parameters (direction word): %1 = Left (Right), %2 = Right (Left)

%macro		RBTreeInsC 2

; ------------- Get uncle as grand parent's right (left) node

		mov	RBUn,[RBGr+RBN_%2]	; get uncle
		or	RBUn,RBUn		; is uncle valid?
		jz	short %%L2		; uncle is invalid

; ------------- Check if uncle is red

		CMPRED	RBUn			; is uncle red?
		jne	short %%L2		; uncle is not red
		
; ------------- Uncle is red, set uncle and parent black, set grandpa red

		SETBLK	RBUn			; set uncle black
		SETBLK	RBPa			; set parent black
		SETRED	RBGr			; set grandpa red
		mov	RBNo,RBGr		; node <- grand parent
		jmp	short RBTreeInsCol2	; next node

; ------------- Uncle is black, check if node is right (left) node of parent 

%%L2:		cmp	RBNo,[RBPa+RBN_%2]	; is node right (left) node?
		jne	short %%L4		; node is not right (left) node

; ------------- Rotate left (right) to become node as new parent node

		push	RBNo			; push RBNo
		mov	RBNo,RBPa		; parent will be used
		call	RBTree%1		; rotate parent left (right)
		pop	RBPa			; parent <- node

; ------------- Set parent black, set grandpa red and rotate it right (left)

%%L4:		SETBLK	RBPa			; set parent black
		SETRED	RBGr			; set grandpa red
		push	RBNo			; push RBNo
		mov	RBNo,RBGr		; grandpa will be used
		call	RBTree%2		; rotate grandpa right (left)
		pop	RBNo			; pop RBNo
%endmacro

; ============= RBTreeInsCol function

; ------------- Push registers

RBTreeInsCol:	push	RBGr			; push RBGr
		push	RBNo			; push RBNo
		push	RBUn			; push RBUn
		push	RBPa			; push RBPa

; ------------- Get parent

RBTreeInsCol2:	mov	RBPa,[RBNo+RBN_Parent]	; get parent
		or	RBPa,RBPa		; is parent valid?
		jz	RBTreeInsCol8		; no other parent

; ------------- Check if parent is red

		CMPRED	RBPa			; parent is red?
		jne	RBTreeInsCol8		; parent is not red

; ------------- Get grand parent and check if parent is left of grand parent
; Parent is red and therefore grandpa is always valid (because root is black)

		mov	RBGr,[RBPa+RBN_Parent]	; get grand parent
		cmp	RBPa,[RBGr+RBN_Left]	; is parent left of grandpa?
		jne	short RBTreeInsCol4	; no, it is right of grandpa

; ------------- Parent is left of grand parent

		RBTreeInsC Left, Right		; insert color for left
		jmp	short RBTreeInsCol2	; next node

; ------------- Parent is right of grand parent

RBTreeInsCol4:	RBTreeInsC Right, Left		; insert color for right
		jmp	RBTreeInsCol2		; next node

; ------------- Set root node black

RBTreeInsCol8:	mov	RBNo,[RBRo+RBR_Node]	; get root node
		SETBLK	RBNo			; set root node black

; ------------- Pop registers

		pop	RBPa			; pop RBPa
		pop	RBUn			; pop RBUn
		pop	RBNo			; pop RBNo
		pop	RBGr			; pop RBGr
		ret

%undef		RBGr				; grand parent
%undef		RBNo				; current node
%undef		RBUn				; uncle
%undef		RBRo				; root
%undef		RBPa				; parent

; -----------------------------------------------------------------------------
;                 Insert new entry into red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE with filled user data)
;		EDX = pointer to root (RBROOT)
;		ESI = callback compare function
;			INPUT:	EAX = pointer to new entry (with user data)
;				EBX = node to test
;			OUTPUT:	ZY = equal, CY = EAX.Data < EBX.Data
;				(as CMP EAX.Data,EBX.Data)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

; ------------- Push registers

RBTreeInsert:	push	ecx			; push ECX

; ------------- Initialize new node

		xor	ecx,ecx			; ECX <- 0
		mov	[eax+RB_ColorFlags],cx	; set color to red, clear flags
		mov	[eax+RBN_Left],ecx	; no left node
		mov	[eax+RBN_Right],ecx	; no right node

; ------------- Get first node

		mov	ecx,[edx+RBR_Node]	; ECX <- get first node
		jecxz	RBTreeInsert3		; tree is empty

; ------------- Next node is valid

RBTreeInsert2:	mov	ebx,ecx			; EBX <- node is valid

; ------------- Compare data

		call	esi			; test data
		jb	RBTreeInsert4		; data is lower
		stc				; set error flag
		je	RBTreeInsert9		; node already exists

; ------------- Higher data, use right branch

		mov	ecx,[ebx+RBN_Right]	; ECX <- right node
		or	ecx,ecx			; is right node valid?
		jnz	RBTreeInsert2		; right node is valid

; ------------- Insert new node to the right

		mov	[ebx+RBN_Right],eax	; insert node to the right
		call	ListAfter		; add entry into list
		jmp	short RBTreeInsert6

; ------------- Store node into root

RBTreeInsert3:	mov	ebx,edx			; EBX <- root
		call	ListAdd			; add entry into list
		mov	[edx+RBR_Node],eax	; link root to this node
		mov	[eax+RBN_Parent],ecx	; set parent to NULL
		inc	byte [eax+RB_Color]	; set color to black for root
		mov	ebx,eax			; EBX <- new node
		jmp	short RBTreeInsert8

; ------------- Lower data, use left branch

RBTreeInsert4:	mov	ecx,[ebx+RBN_Left]	; ECX <- left node
		or	ecx,ecx			; is left node valid?
		jnz	RBTreeInsert2		; left node is valid

; ------------- Insert new node to the left

		mov	[ebx+RBN_Left],eax	; insert node to the left
		call	ListBefore		; add entry into list
RBTreeInsert6:	mov	[eax+RBN_Parent],ebx	; set node as parent of new one

; ------------- Insert red color into tree

		mov	ebx,eax			; EBX <- new node
		call	RBTreeInsCol		; insert color

; ------------- Increate number of entries

RBTreeInsert8:	inc	dword [edx+RBR_Count]	; increase number of entries

; ------------- Pop registers

		clc				; clear error flag
RBTreeInsert9:	pop	ecx			; pop ECX
		ret

; -----------------------------------------------------------------------------
;         Insert new entry into red-black balanced tree with address
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

RBTreeInsAddr:	push	esi			; push ESI
		mov	esi,RBTreeInsAddr4	; ESI <- callback function
		call	RBTreeInsert		; insert new entry
		pop	esi			; pop ESI
		ret

; ------------- Callback compare function for address

RBTreeInsAddr4:	cmp	eax,ebx			; compare nodes
		ret

; -----------------------------------------------------------------------------
;     Insert new entry into red-black balanced tree with WORD data unsigned
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE with filled WORD user data)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

RBTreeInsertW:	push	esi			; push ESI
		mov	esi,RBTreeInsertW4	; ESI <- callback function
		call	RBTreeInsert		; insert new entry
		pop	esi			; pop ESI
		ret

; ------------- Callback compare function for WORD data unsigned

RBTreeInsertW4:	push	eax			; push EAX
		mov	ax,[eax+RB_DataW]	; AX <- user data
		cmp	ax,[ebx+RB_DataW]	; test data
		pop	eax			; pop EAX
		ret

; -----------------------------------------------------------------------------
;      Insert new entry into red-black balanced tree with WORD data signed
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE with filled WORD user data)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

RBTreeInsertWS:	push	esi			; push ESI
		mov	esi,RBTreeInsertWS4	; ESI <- callback function
		call	RBTreeInsert		; insert new entry
		pop	esi			; pop ESI
		ret

; ------------- Callback compare function for WORD data signed

RBTreeInsertWS4:push	eax			; push EAX
		push	edx			; push EDX
		mov	ax,[eax+RB_DataW]	; AX <- user data
		add	ax,8000h		; AX <- convert to unsigned
		mov	dx,[ebx+RB_DataW]	; DX <- tested data
		add	dx,8000h		; DX <- convert to unsigned
		cmp	ax,dx			; compare data
		pop	edx			; pop EDX
		pop	eax			; pop EAX
		ret

; -----------------------------------------------------------------------------
;   Insert new entry into red-black balanced tree with DWORD data unsigned
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE with filled DWORD user data)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

RBTreeInsertDW:	push	esi			; push ESI
		mov	esi,RBTreeInsertDW4	; ESI <- callback function
		call	RBTreeInsert		; insert new entry
		pop	esi			; pop ESI
		ret

; ------------- Callback compare function for DWORD data unsigned

RBTreeInsertDW4:push	eax			; push EAX
		mov	eax,[eax+RBN_Data]	; EAX <- user data
		cmp	eax,[ebx+RBN_Data]	; test data
		pop	eax			; pop EAX
		ret

; -----------------------------------------------------------------------------
;    Insert new entry into red-black balanced tree with DWORD data signed
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to new entry (RBNODE with filled DWORD user data)
;		EDX = pointer to root (RBROOT)
; OUTPUT:	CY = entry already exists (EBX = pointer to existing node)
;		EBX = pointer to existing entry (if CY) or new entry (=EAX)
; -----------------------------------------------------------------------------

RBTreeInsertDWS:push	esi			; push ESI
		mov	esi,RBTreeInsertDS4	; ESI <- callback function
		call	RBTreeInsert		; insert new entry
		pop	esi			; pop ESI
		ret

; ------------- Callback compare function for DWORD data signed

RBTreeInsertDS4:push	eax			; push EAX
		push	edx			; push EDX
		mov	eax,[eax+RBN_Data]	; EAX <- user data
		add	eax,80000000h		; EAX <- convert to unsigned
		mov	edx,[ebx+RBN_Data]	; EDX <- tested data
		add	edx,80000000h		; EDX <- convert to unsigned
		cmp	eax,edx			; compare data
		pop	edx			; pop EDX
		pop	eax			; pop EAX
		ret

; -----------------------------------------------------------------------------
;              Delete color of node from red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to parent of node (RBNODE, NULL=root)
;		EBX = pointer to new current node
;		EDX = pointer to root (RBROOT)
; NOTES:	Entry point is RBTreeDelCol.
; -----------------------------------------------------------------------------
; Local variables:
%define		RBPa	EAX			; parent
%define		RBNo	EBX			; deleted node
%define		RBNe	ECX			; nephew (hardcoded)
%define		RBNeL	CL			; nephew LOW (temporary reg.)
%define		RBRo	EDX			; root
%define		RBBr	ESI			; brother

; ============= Macros

; ------------- Macro - check if left (right) nephew is black
; Parameters: %1 = Left (Right), %2 = jump label if nephew is not black

%macro		TESTNE	2

		mov	RBNe,[RBBr+RBN_%1]	; get nephew
		jecxz	%%black			; nephew is not valid
		CMPBLK	RBNe			; is nephew black?
		jne	%2			; nephew is not black
%%black:

%endmacro

; ------------- Macro - delete color in one direction
; Parameters (direction word): %1 = Left (Right), %2 = Right (Left)

%macro		RBTreeDelC 2

; ------------- Get brother node from right (left) and check if it is red
; Node is black or invalid and therefore its brother is always valid

		mov	RBBr,[RBPa+RBN_%2]	; get brother node
		CMPRED	RBBr			; is brother node red?
		jne	short %%L2		; brother node is not red

; ------------- Set brother node black, set parent red a rotate it left (right)

		SETBLK	RBBr			; set brother node black
		SETRED	RBPa			; set parent node red
		push	RBNo			; push RBNo
		mov	RBNo,RBPa		; parent node will be used
		call	RBTree%1		; rotate parent left (right)
		pop	RBNo			; pop RBNo
		mov	RBBr,[RBPa+RBN_%2]	; get new brother node

; ------------- Check if left (right) nephew is black

%%L2:		TESTNE	%1, %%L4		; check left (right) nephew

; ------------- Check if right (left) nephew is black

		TESTNE	%2, %%L4		; check right (left) nephew

; ------------- Set brother to red

		SETRED	RBBr			; set brother to red
		mov	RBNo,RBPa		; node <- parent
		mov	RBPa,[RBNo+RBN_Parent]	; get new parent
		jmp	RBTreeDelCol2		; next node
		
; ------------- Check if right (left) nephew is black

%%L4:		TESTNE	%2, %%L6		; check right (left) nephew

; ------------- Set left (right) nephew to black

		mov	RBNe,[RBBr+RBN_%1]	; get left (right) nephew
		jecxz	%%L5			; nephew is not valid
		SETBLK	RBNe			; set nephew black

; ------------- Set brother red and rotate it right (left)

%%L5:		SETRED	RBBr			; set brother red
		push	RBNo			; push RBNo
		mov	RBNo,RBBr		; node <- brother
		call	RBTree%2		; rotate brother right (left)
		pop	RBNo			; pop RBNo
		mov	RBBr,[RBPa+RBN_%2]	; get new brother node

; ------------- Copy parent's color to brother

%%L6:		mov	RBNeL,[RBPa+RB_Color]	; get parent's color
		mov	[RBBr+RB_Color],RBNeL	; set brother's color
		SETBLK	RBPa			; set parent black

; ------------- Set right (left) nephew to black

		mov	RBNe,[RBBr+RBN_%2]	; get right (left) nephew
		jecxz	%%L8			; nephew is not valid
		SETBLK	RBNe			; set nephew black

; ------------- Rotate parent left (right)

%%L8:		mov	RBNo,RBPa		; parent will be used
		call	RBTree%1		; rotate parent left (right)
		mov	RBNo,[RBRo+RBR_Node]	; get root node
%endmacro

; ============= RBTreeDelCol function

; ------------- Push registers

RBTreeDelCol:	push	RBPa			; push RBPa
		push	RBNo			; push RBNo
		push	RBNe			; push RBNe
		push	RBBr			; push RBBr

; ------------- Node must be black or invalid

RBTreeDelCol2:	or	RBNo,RBNo		; is node valid?
		jz	short RBTreeDelCol3	; node is not valid
		CMPBLK	RBNo			; is node black?
		jne	RBTreeDelCol6		; node is not black

; ------------- Node must not be a root node

RBTreeDelCol3:	cmp	RBNo,[RBRo+RBR_Node]	; is it root node?
		je	RBTreeDelCol6		; it is root node

; ------------- Check if current node is left node of parent

		cmp	RBNo,[RBPa+RBN_Left]	; is current node left node?
		jne	near RBTreeDelCol5	; it is not left node

; ------------- Node is left node

		RBTreeDelC Left, Right		; delete color for left
		jmp	RBTreeDelCol6		; quit

; ------------- Node is right node

RBTreeDelCol5:	RBTreeDelC Right, Left		; delete color for right

; ------------- Set node to black

RBTreeDelCol6:	or	RBNo,RBNo		; is node valid?
		jz	short RBTreeDelCol8	; node is invalid
		SETBLK	RBNo			; set node to black

; ------------- Pop registers

RBTreeDelCol8:	pop	RBBr			; pop RBBr
		pop	RBNe			; pop RBNe
		pop	RBNo			; pop RBNo
		pop	RBPa			; pop RBPa
		ret

%undef		RBPa				; parent
%undef		RBNo				; deleted node
%undef		RBNe				; nephew
%undef		RBNeL				; nephew LOW (temporary reg.)
%undef		RBRo				; root
%undef		RBBr				; brother

; -----------------------------------------------------------------------------
;                  Delete node from red-black balanced tree
; -----------------------------------------------------------------------------
; INPUT:	EBX = pointer to deleted node (RBNODE)
;		EDX = pointer to root (RBROOT)
; NOTES:	RBNODE memory block is not destroyed.
; -----------------------------------------------------------------------------
; Local variables:
%define		RBPa	EAX			; parent
%define		RBPaL	AL			; parent LOW
%define		RBNo	EBX			; current node
%define		RBCo	ECX			; color
%define		RBCoL	CL			; color LOW
%define		RBRo	EDX			; root
%define		RBOl	ESI			; old node
%define		RBCh	EDI			; child

; ------------- Push registers

RBTreeDelete:	push	RBPa			; push RBPa
		push	RBNo			; push RBNo
		push	RBCo			; push RBCo
		push	RBOl			; push RBOl
		push	RBCh			; push RBCh

; ------------- Decrease number of entries

		dec	dword [RBRo+RBR_Count]	; decrease number of entries

; ------------- Detach entry from the list

		call	ListDelEBX		; delete node

; ------------- Get child node from right node if left node is not valid

		mov	RBCh,[RBNo+RBN_Right]	; RBCh <- get right node
		mov	RBOl,[RBNo+RBN_Left]	; RBOl <- get left node
		or	RBOl,RBOl		; ir left node valid?
		jz	RBTreeDelete6		; left node is not valid

; ------------- Get child node from left node if right node is not valid

		xchg	RBOl,RBCh		; child <- left, old <- right
		or	RBOl,RBOl		; is right node valid?
		jz	RBTreeDelete6		; right node is not valid

; ------------- Find leftmost node of right node

		mov	RBCh,RBOl		; child <- save right node
		mov	RBOl,RBNo		; old <- save current node
RBTreeDelete1:	mov	RBNo,RBCh		; node <- right/left
		mov	RBCh,[RBNo+RBN_Left]	; child <- get left node
		or	RBCh,RBCh		; is left node valid?
		jnz	RBTreeDelete1		; next left node

; ------------- Get right child, parent and color of the node

		mov	RBCh,[RBNo+RBN_Right]	; child <- get right node
		mov	RBPa,[RBNo+RBN_Parent]	; parent <- get parent
		mov	RBCoL,[RBNo+RB_Color]	; color <- get color

; ------------- Link parent to right child

		or	RBCh,RBCh		; is child valid?	
		jz	RBTreeDelete2		; child is not valid
		mov	[RBCh+RBN_Parent],RBPa	; child <- set new parent

; ------------- If parent is old (deleted) node link it as its right node

RBTreeDelete2:	cmp	RBOl,RBPa		; is parent equal to old node?
		jne	RBTreeDelete3		; parent is not old node
		mov	[RBPa+RBN_Right],RBCh	; set child as right of parent
		mov	RBPa,RBNo		; parent <- node
		jmp	short RBTreeDelete4
RBTreeDelete3:	mov	[RBPa+RBN_Left],RBCh	; set child as left of parent

; ------------- Copy color, right and left node from old node to current one

RBTreeDelete4:	push	RBPa			; push RBPa (old parent)
		mov	RBPaL,[RBOl+RB_Color]	; color
		mov	[RBNo+RB_Color],RBPaL
		mov	RBPa,[RBOl+RBN_Left]	; left node
		mov	[RBNo+RBN_Left],RBPa
		mov	RBPa,[RBOl+RBN_Right]	; right node
		mov	[RBNo+RBN_Right],RBPa
		mov	RBPa,[RBOl+RBN_Parent]	; parent
		mov	[RBNo+RBN_Parent],RBPa

; ------------- Link current node to parent

		or	RBPa,RBPa		; is parent valid?
		jz	RBTreeDelete56		; parent is not valid
		cmp	[RBPa+RBN_Left],RBOl	; old node=left node of parent?
		jne	RBTreeDelete52		; no
		mov	[RBPa+RBN_Left],RBNo	; node -> left node of parent
		jmp	short RBTreeDelete58
RBTreeDelete52:	mov	[RBPa+RBN_Right],RBNo	; node -> right node of parent
		jmp	short RBTreeDelete58
RBTreeDelete56:	mov	[RBRo+RBR_Node],RBNo	; node -> root node

; ------------- Link current node as parent of old left node

RBTreeDelete58:	mov	RBPa,[RBOl+RBN_Left]	; RBPa <- get old left node
		mov	[RBPa+RBN_Parent],RBNo	; node -> parent of left node

; ------------- Link current node as parent of old right node

		mov	RBPa,[RBOl+RBN_Right]	; RBPa <- get old right node
		or	RBPa,RBPa		; is old right node valid?
		jz	RBTreeDelete59		; old right node is not valid
		mov	[RBPa+RBN_Parent],RBNo	; node -> parent of right node
RBTreeDelete59:	pop	RBPa			; pop RBPa (old parent)
		jmp	short RBTreeDelete7	; delete color

; ------------- Get parent and color of current node

RBTreeDelete6:	mov	RBPa,[RBNo+RBN_Parent]	; RBPa <- get current parent
		mov	RBCoL,[RBNo+RB_Color]	; RBCoL <- get current color

; ------------- Link parent as new parent of the child

		or	RBCh,RBCh		; is child valid?
		jz	RBTreeDelete62		; child is not valid
		mov	[RBCh+RBN_Parent],RBPa	; parent -> parent of the child

; ------------- Link child to the parent

RBTreeDelete62:	or	RBPa,RBPa		; is parent valid?
		jz	RBTreeDelete66		; parent is not valid
		cmp	RBNo,[RBPa+RBN_Left]	; is current node left child?
		jne	RBTreeDelete64		; is it not left child
		mov	[RBPa+RBN_Left],RBCh	; link child as left child
		jmp	short RBTreeDelete7
RBTreeDelete64:	mov	[RBPa+RBN_Right],RBCh	; link child as right child
		jmp	short RBTreeDelete7
RBTreeDelete66:	mov	[RBRo+RBR_Node],RBCh	; child -> root node

; ------------- Delete color

RBTreeDelete7:	cmp	RBCoL,RB_BLACK		; is it black color?
		jne	RBTreeDelete8		; it is not black color
		mov	ebx,RBCh		; EBX <- child node
		call	RBTreeDelCol		; delete color

; ------------- Pop registers

RBTreeDelete8:	pop	RBCh			; pop RBCh
		pop	RBOl			; pop RBOl
		pop	RBCo			; pop RBCo
		pop	RBNo			; pop RBNo
		pop	RBPa			; pop RBPa
		ret

; -----------------------------------------------------------------------------
;                          Debug test red-black tree
; -----------------------------------------------------------------------------

%ifdef DEBUG_RBTREE

TestRBTree:	call	DebOutClear
		mov	esi,TestTreeText1
		call	DebOutText

; ------------- Create random nodes

		mov	ecx,1000
		mov	edx,TestTree
TestRBTree1:	call	RandByte
		test	al,7
		jnz	TestRBTree2

; ------------- Delete one random node

		mov	eax,[TestTree+RBR_Count]
		dec	eax
		call	RandMaxDWord
		call	RBTreeIndex
		jc	TestRBTree1
		call	RBTreeDelete
		xchg	eax,ebx
		call	SysMemFree
		jmp	short TestRBTree1

; ------------- Create one random node

TestRBTree2:	mov	eax,RBNODE_size+4
		call	SysMemAlloc
		push	eax
		call	RandDWord
		xchg	eax,esi
		pop	eax
		mov	[eax+RBN_Data],esi
		call	RBTreeInsertDWS
		loop	TestRBTree1

; ------------- Limit number of nodes

TestRBTree3:	mov	eax,[TestTree+RBR_Count]
		cmp	eax,byte 100
		jbe	TestRBTree4
		dec	eax
		call	RandMaxDWord
		call	RBTreeIndex
		call	RBTreeDelete
		xchg	eax,ebx
		call	SysMemFree
		jmp	short TestRBTree3

; ------------- Display nodes

TestRBTree4:	call	RBTreeFirst
		jc	TestRBTree7
TestRBTree5:	mov	eax,[ebx+RBN_Data]
		call	DebOutNumSig
		add	byte [VideoPos],15
		and	byte [VideoPos],~0fh
		cmp	byte [VideoPos],70
		jb	TestRBTree6
		call	DebNewLine
TestRBTree6:   	call	RBTreeNext
		jnc	TestRBTree5

TestRBTree7:	jmp	$

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

TestTree:	RBTREE			; test tree

TestTreeText1:	db	'Test Red-Black Tree...',10,0

%endif ; DEBUG_RBTREE
