; =============================================================================
;
;                          Litos - Random generator
;
; =============================================================================
; General algorithm (new_seed = old_seed*214013 + 2531011) for the random
; generator is taken from interpreted BASIC and is described in Microsoft
; Knowledgebase article Q28150. Other possible coefficients are:
; 214013/2531011, 17405/10395331, 214013/13737667, 214013/10395331
;
; Access to seed of random generator is not SMP locked, it is a random
; number and therefore data distortion is unimportant.
;
; Random numbers in interval are usually calculated with modulo operation.
; Such random numbera are not evenly spreaded out, their density is lower at
; end of the interval. Therefore in this generator repeated generation is used 
; of random numbers until suitable number is found, such numbers are spreaded
; evenly in whole interval.
; =============================================================================

; Tip - alternative use BSD-style random generator ("Additive Congruential
; Random Number Generator" = ACORN):
;   - use table of DWORDs with 31 dimension
;   - fill it up with random numbers
;   - use pointers in table P1=3 and P2=0
;   - generate next number table[P1] = table[P1] + table[P2]
;	(formula x = x(-31) + x(-3))
;   - increment pointers modulo 31
; For 64-bits use recursion x = x(-63) + x(-1)


		CODE_SECTION

RNDCOEFF1	EQU	214013		; random coefficient 1 (multiple)
RNDCOEFF2	EQU	2531011		; random coefficient 2 (add)

; ------------- Macro - next random value (EAX:EDX=output)

%macro		RANDNEXT 0
		mov	eax,RNDCOEFF1		; EAX <- coefficient 1
		mul	dword [RandSeed]	; EAX <- multiple seed LOW
		add	eax,RNDCOEFF2		; add coefficient 2
		adc	edx,dword [RandSeed+4]	; add seed HIGH
		mov	dword [RandSeed],edx	; store new seed LOW
		mov	dword [RandSeed+4],eax	; store new seed HIGH
%endmacro 

; -----------------------------------------------------------------------------
;                            Get random byte
; -----------------------------------------------------------------------------
; OUTPUT:	AL = random byte (0 to 255)
; -----------------------------------------------------------------------------

RandByte:	push	edx		; push EDX
		push	eax		; push EAX
		RANDNEXT		; next global random seed
		shr	eax,16		; EAX <- random word
		xchg	eax,edx		; EDX <- push random word
		pop	eax		; pop EAX
		mov	al,dl		; AL <- random byte
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                            Get random word
; -----------------------------------------------------------------------------
; OUTPUT:	AX = random word (0 to 65535)
; -----------------------------------------------------------------------------

RandWord:	push	edx		; push EDX
		push	eax		; push EAX
		RANDNEXT		; next global random seed
		shr	eax,16		; EAX <- random word
		xchg	eax,edx		; EDX <- push random word
		pop	eax		; pop EAX
		xchg	ax,dx		; AX <- random word
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                       Get random quadruple word
; -----------------------------------------------------------------------------
; OUTPUT:	EDX:EAX = random quadruple word (0 to 0ffffffffffffffffh)
; NOTES:	RANDNEXT result is not used to avoid number predictions.
; -----------------------------------------------------------------------------

RandQWord:	push	edx		; push EDX
		RANDNEXT		; next global random seed
		pop	edx		; pop EDX
		xchg	eax,edx		; EDX <- first random double word

; RandDWord must follow!

; -----------------------------------------------------------------------------
;                       Get random double word
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = random double word (0 to 0ffffffffh)
; -----------------------------------------------------------------------------

RandDWord:	push	edx		; push EDX
		RANDNEXT		; next global random seed
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                  Get random float, precision 7 decimal digits
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = random float (0 to 1, including "0", excluding "1")
; NOTES:	Number in EAX is single-precision floating-point IEE 754.
;		This function does not use FPU.
;		To load result into FPU, you can use this sequence:
;			push	eax		; push result into stack
;			fld	dword [esp]	; load result into FPU
;			pop	eax		; pop result from the stack
;		Possible values (only 7 decimal digits are significant):
;			0.0000000
;			0.0000001
;			0.0000002
;			...
;			0.4999999
;			0.5000000
;			0.5000001
;			...
;			0.9999999
; -----------------------------------------------------------------------------

; ------------- Push registers

RandFloat:	push	ecx		; push ECX

; ------------- Generate random double word (-> EAX)

		push	edx		; push EDX
		RANDNEXT		; next global random seed (-> EAX)
		pop	edx		; pop EDX
		and	eax,~1ffh	; clear unused bits

; ------------- Find highest bit "1" (-> ECX)

		bsr	ecx,eax		; ECX <- find highest bit "1"
		jz	RandFloat2	; zero number

; ------------- Prepare mantissa (-> EAX)

		ror	eax,cl		; EAX <- rotate highest bit "1" out
		shr	eax,9		; EAX <- shift mantissa to its position

; ------------- Prepare exponent (-> EDX)

		mov	ch,127-32	; CH <- exponent for "1.0000" - 32
		add	ch,cl		; CH <- exponent
		movzx	ecx,ch		; ECX <- exponent
		shl	ecx,23		; ECX <- rotate exponent to position

; ------------- Pop registers

		or	eax,ecx		; EAX <- add exponent to result
RandFloat2:	pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;           Get random number using FPU, precision 10 decimal digits
; -----------------------------------------------------------------------------
; OUTPUT:	ST0 = random float (0 to 1, including "0", excluding "1")
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
;		Possible values (only 10 decimal digits are significant):
;			0.0000000000
;			0.0000000002
;			0.0000000005
;			...
;			0.4999999998
;			0.5000000000
;			0.5000000002
;			...
;			0.9999999998
; -----------------------------------------------------------------------------

; ------------- Push registers

RandFloatFPU:	push	eax		; push EAX

; ------------- Prepare HIGH dword of the number (= 0)

		xor	eax,eax		; EAX <- 0
		push	eax		; store HIGH dword of the number

; ------------- Generate LOW dword of the number

		push	edx		; push EDX
		RANDNEXT		; next global random seed (-> EAX)
		pop	edx		; pop EDX
		push	eax		; store number to the stack

; ------------- Recalc number to range 0 to 1

		fwait			; synchronize FPU
		fild	qword [esp]	; ST0 <- random number
		fmul	qword [RandFltCoef] ; recalc to range 0 to 1
		pop	eax		; destroy number from the stack
		pop	eax		; destroy HIGH dword from the stack

; ------------- Pop registers

		pop	eax		; pop EAX
		ret	

; -----------------------------------------------------------------------------
;                Get random double, precision 16 decimal digits
; -----------------------------------------------------------------------------
; OUTPUT:	EDX:EAX = random double (0 to 1, including "0", excluding "1")
; NOTES:	Number in EDX:EAX is double-precision floating-point IEE 754.
;		This function does not use FPU.
;		To load result into FPU, you can use this sequence:
;			push	edx		; push result HIGH into stack
;			push	eax		; push result LOW into stack
;			fld	qword [esp]	; load result into FPU
;			pop	eax		; pop result LOW from the stack
;			pop	edx		; pop result HIGH from stack
;		Possible values (only 16 decimal digits are significant):
;			0.0000000000000000
;			0.0000000000000001
;			0.0000000000000002
;			...
;			0.4999999999999999
;			0.5000000000000000
;			0.5000000000000001
;			...
;			0.9999999999999999
; -----------------------------------------------------------------------------

; ------------- Push registers

RandDouble:	push	ebx		; push EBX
		push	ecx		; push ECX
		xor	ecx,ecx		; ECX <- 0

; ------------- Generate random quadruple word (-> EDX:EAX)

		call	RandQWord	; generate random quadruple word
		and	edx,1fffffh	; mask mantissa HIGH (+ high "1" bit)
		jz	RandDouble4	; number is only DWORD

; ------------- Find highest bit "1" (-> EBX)

		bsr	ebx,edx		; EBX <- find highest bit "1"

; ------------- Prepare mantissa (-> EDX:EAX)

		mov	cl,20		; ECX <- maximal possible bit position
		sub	cl,bl		; ECX <- number of bits to rotate
		shld	edx,eax,cl	; EDX <- shift mantissa HIGH
		shl	eax,cl		; EAX <- shift mantissa LOW

; ------------- Prepare exponent (-> EDX)

		mov	bx,1022		; EBX <- exponent for "1.0000" - 1
		jmp	short RandDouble6

; ------------- Find highest bit "1" (-> EBX)

RandDouble4:	bsr	ebx,eax		; EBX <- find highest bit "1"
		jz	RandDouble8	; zero number

; ------------- Prepare mantissa (-> EDX:EAX)

		mov	cl,31		; ECX <- maximal possible bit position
		sub	cl,bl		; ECX <- number of bits to rotate
		shl	eax,cl		; EAX <- shift mantissa
		shld	edx,eax,21	; EDX <- mantissa HIGH
		shl	eax,21		; EAX <- mantissa LOW
		
; ------------- Prepare exponent (-> EDX)

		mov	bx,1022-21	; EBX <- exponent for "1.0000" - 1 - 21

; ------------- Composite result

RandDouble6:	sub	ebx,ecx		; EBX <- new exponent
		shl	ebx,52-32	; EBX <- rotate exponent to position
		and	edx,0fffffh	; mask mantissa HIGH
		or	edx,ebx		; add exponent to result

; ------------- Pop registers

RandDouble8:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;           Get random number using FPU, precision 16 decimal digits
; -----------------------------------------------------------------------------
; OUTPUT:	ST0 = random double (0 to 1)
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
;		Possible values (only 16 decimal digits are significant):
;			0.0000000000000000
;			0.0000000000000001
;			0.0000000000000002
;			...
;			0.4999999999999999
;			0.5000000000000000
;			0.5000000000000001
;			...
;			0.9999999999999999
; -----------------------------------------------------------------------------

; ------------- Push registers

RandDoubleFPU:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Generate HIGH random number

		RANDNEXT		; next global random seed (-> EAX)
		and	eax,1fffffh	; reset unused bits
		push	eax		; store HIGH number to the stack

; ------------- Generate LOW random number

		RANDNEXT		; next global random seed (-> EAX)
		push	eax		; store LOW number to the stack

; ------------- Recalc number to range 0 to 1

		fwait			; synchronize FPU
		fild	qword [esp]	; ST0 <- random number
		fmul	qword [RandDblCoef] ; recalc to range 0 to 1
		pop	eax		; destroy LOW number from the stack
		pop	eax		; destroy HIGH number from the stack

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret	

; -----------------------------------------------------------------------------
;                Get random unsigned byte with max value
; -----------------------------------------------------------------------------
; INPUT:	AL = maximal unsigned value (0 to 255)
; OUTPUT:	AL = random byte (0 to maximal value)
; -----------------------------------------------------------------------------

; ------------- Check maximal value 0

RandMaxByte:	or	al,al		; zero maximal value?
		jz	RandMaxByte8	; maximal value is 0

; ------------- Push registers

		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Prepare mask of maximal value

		movzx	edx,al		; EDX <- maximal value
		bsr	ecx,edx		; ECX <- bits of maximal value
		mov	ch,al		; CH <- maximal value
		mov	dl,B1		; DL <- maximal value bit
		shl	edx,cl		; EDX <- maximal value
		dec	edx		; DL <- mask of result

; ------------- Find required value

RandMaxByte2:	call	RandByte	; get random byte (-> AL)
		and	al,dl		; mask result
		cmp	al,ch		; is this value allowed?
		ja	RandMaxByte2	; invalid value, next attempt

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
RandMaxByte8:	ret

; -----------------------------------------------------------------------------
;                 Get random signed byte with max value
; -----------------------------------------------------------------------------
; INPUT:	AL = maximal signed value (-128 to +127)
; OUTPUT:	AL = random byte (0 to maximal value)
; -----------------------------------------------------------------------------

RandMaxChar:	or	al,al		; is it unsigned value?
		jns	RandMaxByte	; it is unsigned value or zero
		neg	al		; correct maximal value
		call	RandMaxByte	; generate random byte
		neg	al		; correct result
		ret

; -----------------------------------------------------------------------------
;               Get random unsigned word with max value
; -----------------------------------------------------------------------------
; INPUT:	AX = maximal unsigned value (0 to 65535)
; OUTPUT:	AX = random word (0 to maximal value)
; -----------------------------------------------------------------------------

; ------------- Check maximal value 0

RandMaxWord:	or	ax,ax		; zero maximal value?
		jz	RandMaxWord8	; maximal value is 0

; ------------- Push registers

		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Prepare mask of maximal value

		bsr	cx,ax		; ECX <- bits of maximal value
		xor	edx,edx		; EDX <- 0
		mov	dl,B1		; EDX <- B1, maximal value bit
		shl	edx,cl		; EDX <- maximal value
		dec	edx		; DX <- mask of result
		xchg	ax,cx		; CX <- maximal value

; ------------- Find required value

RandMaxWord2:	call	RandWord	; get random word (-> AX)
		and	ax,dx		; mask result
		cmp	ax,cx		; is this value allowed?
		ja	RandMaxWord2	; invalid value, next attempt

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
RandMaxWord8:	ret

; -----------------------------------------------------------------------------
;                Get random signed word with max value
; -----------------------------------------------------------------------------
; INPUT:	AX = maximal signed value (-32768 to +32767)
; OUTPUT:	AX = random word (0 to maximal value)
; -----------------------------------------------------------------------------

RandMaxShort:	or	ax,ax		; is it unsigned value?
		jns	RandMaxWord	; it is unsigned value or zero
		neg	ax		; correct maximal value
		call	RandMaxWord	; generate random word
		neg	ax		; correct result
		ret

; -----------------------------------------------------------------------------
;             Get random unsigned double word with max value
; -----------------------------------------------------------------------------
; INPUT:	EAX = maximal unsigned value (0 to 0ffffffffh)
; OUTPUT:	EAX = random double word (0 to maximal value)
; -----------------------------------------------------------------------------

; ------------- Check maximal value 0

RandMaxDWord:	or	eax,eax		; zero maximal value?
		jz	RandMaxDWord8	; maximal value is 0

; ------------- Push registers

		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Prepare mask of maximal value

		bsr	ecx,eax		; ECX <- bits of maximal value
		xor	edx,edx		; EDX <- 0
		mov	dl,B1		; EDX <- B1, maximal value bit
		shl	edx,cl		; EDX <- maximal value
		dec	edx		; EDX <- mask of result
		xchg	eax,ecx		; ECX <- maximal value

; ------------- Find required value

RandMaxDWord4:	call	RandDWord	; get random double word (-> EAX)
		and	eax,edx		; mask result
		cmp	eax,ecx		; is this value allowed?
		ja	RandMaxDWord4	; invalid value, next attempt

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
RandMaxDWord8:	ret

; -----------------------------------------------------------------------------
;             Get random signed double word with max value
; -----------------------------------------------------------------------------
; INPUT:	EAX = maximal signed value (-80000000h to +7fffffffh)
; OUTPUT:	EAX = random double word (0 to maximal value)
; -----------------------------------------------------------------------------

RandMaxLong:	or	eax,eax		; is it unsigned value?
		jns	RandMaxDWord	; it is unsigned value or zero
		neg	eax		; correct maximal value
		call	RandMaxDWord	; generate random double word
		neg	eax		; correct result
		ret

; -----------------------------------------------------------------------------
;            Get random unsigned quadruple word with max value
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = maximal unsigned value (0 to 0ffffffffffffffffh)
; OUTPUT:	EDX:EAX = random quadruple word (0 to maximal value)
; -----------------------------------------------------------------------------

; ------------- Check maximal value 0

RandMaxQWord:	or	edx,edx		; is maximal value QWORD?	
		jz	RandMaxDWord	; only DWORD required

; ------------- Push registers

		push	ebx		; push EBX
		push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Prepare mask of maximal value

		bsr	ecx,edx		; ECX <- bits of maximal value HIGH
		xor	ebx,ebx		; EBX <- 0
		mov	bl,B1		; EBX <- B1, maximal value bit
		shl	ebx,cl		; EBX <- maximal value HIGH
		dec	ebx		; EBX <- mask of result HIGH
		xchg	eax,ecx		; ECX <- maximal value LOW
		mov	esi,edx		; ESI <- maximal value HIGH

; ------------- Find required value

RandMaxQWord6:	call	RandQWord	; get random quadruple word (->EDX:EAX)
		and	edx,ebx		; mask result HIGH
		cmp	edx,esi		; check value HIGH
		jb	RandMaxQWord8	; value is OK
		ja	RandMaxQWord6	; invalid value, next attempt		
		cmp	eax,ecx		; check value LOW
		ja	RandMaxQWord6	; invalid value, next attempt

; ------------- Pop registers

RandMaxQWord8:	pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;            Get random signed quadruple word with max value
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = maximal signed value (-8000000000000000h 
;					         to +7fffffffffffffffh)
; OUTPUT:	EDX:EAX = random quadruple word (0 to maximal value)
; -----------------------------------------------------------------------------

RandMaxInt64:	or	edx,edx		; is it unsigned value?
		jns	RandMaxQWord	; it is unsigned value or zero
		neg	eax		; correct maximal value LOW
		adc	edx,byte 0	; carry
		neg	edx		; correct maximal value HIGH
		call	RandMaxQWord	; generate random double word
		neg	eax		; correct result LOW
		adc	edx,byte 0	; carry
		neg	edx		; correct result HIGH
		ret

; -----------------------------------------------------------------------------
;                 Get random float with max value (10 digits)
; -----------------------------------------------------------------------------
; INPUT:	ST0 = maximal value (can be signed)
; OUTPUT:	ST0 = random float (0 to max value)
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
; -----------------------------------------------------------------------------

RandMaxFloat:	call	RandFloatFPU	; generate random float
		fmulp	st1,st0		; multiple with max. value
		ret	

; -----------------------------------------------------------------------------
;          Get random double float with max value (16 digits)
; -----------------------------------------------------------------------------
; INPUT:	ST0 = maximal value (can be signed)
; OUTPUT:	ST0 = random double float (0 to max value)
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
; -----------------------------------------------------------------------------

RandMaxDouble:	call	RandDoubleFPU	; generate random double float
		fmulp	st1,st0		; multiple with max. value
		ret	

; -----------------------------------------------------------------------------
;                       Get random byte in interval
; -----------------------------------------------------------------------------
; INPUT:	AL = maximal value (signed or unsigned)
;		AH = minimal value (signed or unsigned)
; OUTPUT:	AL = random byte (min to max)
; -----------------------------------------------------------------------------

RandIntChar:
RandIntByte:	sub	al,ah		; AL <- new maximal value
		call	RandMaxByte	; generate random byte
		add	al,ah		; AL <- add minimal value
		ret

; -----------------------------------------------------------------------------
;                     Get random word in interval
; -----------------------------------------------------------------------------
; INPUT:	AX = maximal value (signed or unsigned)
;		DX = minimal value (signed or unsigned)
; OUTPUT:	AX = random word (min to max)
; -----------------------------------------------------------------------------

RandIntShort:
RandIntWord:	sub	ax,dx		; AX <- new maximal value
		call	RandMaxWord	; generate random word
		add	ax,dx		; AX <- add minimal value
		ret

; -----------------------------------------------------------------------------
;                   Get random double word in interval
; -----------------------------------------------------------------------------
; INPUT:	EAX = maximal value (signed or unsigned)
;		EDX = minimal value (signed or unsigned)
; OUTPUT:	EAX = random double word (min to max)
; -----------------------------------------------------------------------------

RandIntLong:
RandIntDWord:	sub	eax,edx		; EAX <- new maximal value
		call	RandMaxDWord	; generate random double word
		add	eax,edx		; EAX <- add minimal value
		ret

; -----------------------------------------------------------------------------
;                  Get random quadruple word in interval
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = maximal value (signed or unsigned)
;		EBX:ECX = minimal value (signed or unsigned)
; OUTPUT:	EDX:EAX = random quadruple word (min to max)
; -----------------------------------------------------------------------------

RandIntInt64:
RandIntQWord:	sub	eax,ecx		; EAX <- new maximal value LOW
		sbb	edx,ebx		; EDX <- new maximal value HIGH
		call	RandMaxQWord	; generate random quadruple word
		add	eax,ecx		; EAX <- add minimal value LOW
		adc	edx,ebx		; EDX <- add minimal value HIGH
		ret

; -----------------------------------------------------------------------------
;                  Get random float in interval (10 digits)
; -----------------------------------------------------------------------------
; INPUT:	ST0 = maximal value (can be signed)
;		ST1 = minimal value (can be signed)
; OUTPUT:	ST0 = random float (min to max)
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
; -----------------------------------------------------------------------------

RandIntFloat:	fsub	st0,st1		; subtract minimal value
		call	RandMaxFloat	; generate random float
		faddp	st1,st0		; random value
		ret	

; -----------------------------------------------------------------------------
;               Get random double float in interval (16 digits)
; -----------------------------------------------------------------------------
; INPUT:	ST0 = maximal value (can be signed)
;		ST1 = minimal value (can be signed)
; OUTPUT:	ST0 = random double float (min to max)
; NOTES:	Warning - this function uses FPU (push/pop state if in kernel).
; -----------------------------------------------------------------------------

RandIntDouble:	fsub	st0,st1		; subtract minimal value
		call	RandMaxDouble	; generate random double float
		faddp	st1,st0		; random value
		ret	

; -----------------------------------------------------------------------------
;                 Generate Pseudo-Gaussian random number
; -----------------------------------------------------------------------------
; INPUT:	AL = level (0 to 255, 0=linear)
; OUTPUT:	ST0 = random double float (0 to 1)
; NOTES:	This random generator is not true Gaussian generator. It
;		generates random numbers with distribution which is near to
;		the Gaussian distribution. This is very simple algorithm:
;		it simple adds given number of random numbers. In results
;		there is minimal probability of values near 0 and 1 and
;		maximal probability of values near 0.5. Level value determines
;		steepness of the probability curve.
;		Warning - this function uses FPU (push/pop state if in kernel).
; -----------------------------------------------------------------------------
; Level 0:	Level 1:	Level 2:		Level 10:  X
; |		|       	|			|         XXX
; |		|      x	|        xXXXx		|        xXXXx
; |		|    xXXXx	|      XXXXXXXXX	|        XXXXX
; |XXXXXXXXXXX	|  xXXXXXXXx	|     XXXXXXXXXXX	|       xXXXXXx
; |XXXXXXXXXXX	|xXXXXXXXXXXXx	| xxXXXXXXXXXXXXXXXxx	|    xxXXXXXXXXXxx
; +-----------	+-------------	+---------------------	+----------------------
; -----------------------------------------------------------------------------

; ------------- Push registers

RandGauss:	push	ecx		; push ECX

; ------------- Prepare level

		movzx	ecx,al		; ECX <- required level
		inc	ecx		; ECX <- number of random numbers
		push	ecx		; push ECX, count of numbers

; ------------- Add random numbers

		fldz			; load zero to ST0
RandGauss2:	call	RandDoubleFPU	; generate random double float number
		faddp	st1,st0		; add number to accumulator
		loop	RandGauss2	; next value

; ------------- Divide accumulator with count of numbers

		fidiv	dword [esp]	; divide with count of numbers
		pop	ecx		; destroy ECX from the stack

; ------------- Pop registers

		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Float coefficient 1/100000000h (= 3df0000000000000h)

		align	8, db 0
RandFltCoef:	dq	0.00000000023283064365386962890625

; ------------- Double coefficient 1/20000000000000h (= 3ca0000000000000h)

		align	8, db 0
RandDblCoef:	dq	1.1102230246251565404236316680908e-16

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Seed for random generator
; Seed is initialized in INIT\INIT.ASM and incremented with events.

		align	8, db 0
RandSeed:	dd	5623489,8234957	; seed for random generator
