; =============================================================================
;
;                              Litos - Calendar
;
; =============================================================================
; Julian date: days from 1/1/4713 BCE noon 12:00 (it was Monday)
; Julian date has period 7980 years, then starts again.
; Julian date range: 1/1/4713 BCE 12:00 to 1/1/3268 CE 12:00, e.g. 0 to 2914673
; days. Julian period is 2518277472000000000 100-nanoseconds (e.g. 7980 years).
;
; Julian calendar: till Thursday 10/4/1582 (begins julian date 2299159.5)
; Gregorian calendar: from Friday 10/15/1582 (begins julian date 2299160.5)
; 1/1/1 CE: JD=1721423.5, Saturday (there is no year 0!)
; 1/1/1 BCE: JD=1721057.5 Thursday
; Julian year: 365.25 days, Gregorian year: 365.2425 days.
;
; ISO-8601: date format YYYY-WW-DD (year-week-day_of_week). Week: 1 to 53,
; day of week: 1 to 7, 1=Monday. First week of year is such a week, which 
; contains first Thursday of Gregorian year, and week containing January 4th.
;
; Normal year, 365 days:
;    Mo  Tu  We  Th  Fr  Sa  Su
;                             1  no   1 + 51*7 + 7 yes, 52, 4.1. is Th
;	                  1   2  no   2 + 51*7 + 6 yes, 52, 4.1. is We
;                     1   2   3  no   3 + 51*7 + 5 yes, 52, 4.1. is Tu
;                 1   2   3   4  yes  4 + 51*7 + 4 yes, 53, 4.1. is Mo
;            1    2   3   4   5  yes  5 + 51*7 + 3 no,  52, 4.1. is Su
;        1   2    3   4   5   6  yes  6 + 51*7 + 2 no,  52, 4.1. is Sa
;    1   2   3    4   5   6   7  yes  7 + 51*7 + 1 no,  52, 4.1. is Fr
;
; Leap year, 366 days:
;    Mo  Tu  We  Th  Fr  Sa  Su
;                             1  no   1 + 52*7 + 1 no,  52, 4.1. is Fr
;	                  1   2  no   2 + 51*7 + 7 yes, 52, 4.1. is Th
;                     1   2   3  no   3 + 51*7 + 6 yes, 52, 4.1. is We
;                 1   2   3   4  yes  4 + 51*7 + 5 yes, 53, 4.1. is Tu
;            1    2   3   4   5  yes  5 + 51*7 + 4 yes, 53, 4.1. is Mo
;        1   2    3   4   5   6  yes  6 + 51*7 + 3 no,  52, 4.1. is Su
;    1   2   3    4   5   6   7  yes  7 + 51*7 + 2 no,  52, 4.1. is Fr
;
; Year 1582 has 355 days (it is not leap), 51 weeks.
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                    Convert absolute time to Julian date
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
; OUTPUT:	ST0 = Julian date (days from 1/1/4713 BCE noon 12:00)
; -----------------------------------------------------------------------------

; ------------- Push registers (save absolute time into stack)

AbsToJulian:	push	edx		; push EDX
		push	eax		; push EAX

; ------------- Convert absolute time to Julian date

		fild	qword [esp]	; load absolute time into ST0
		fld	tword [AbsToDayCoef] ; load recalc coefficient
		fmulp	st1		; recalc absolute time to days
		fadd	qword [JulianBase] ; add base of Julian date

; ------------- Pop registers

		pop	eax		; pop EAX
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                    Convert Julian date to absolute time
; -----------------------------------------------------------------------------
; INPUT:	ST0 = Julian date (days from 1/1/4713 BCE noon 12:00)
; OUTPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
; -----------------------------------------------------------------------------

; ------------- Push registers (prepare absolute time in stack)

JulianToAbs:	push	edx		; push EDX
		push	eax		; push EAX

; ------------- Convert Julian date to absolute time

		fsub	qword [JulianBase] ; subtract base of Julian date
		fmul	qword [DayToAbsCoef] ; recalc days to absolute time
		fistp	qword [esp]	; save absolute time from ST0

; ------------- Pop registers

		pop	eax		; pop EAX (absolute time LOW)
		pop	edx		; pop EDX (absolute time HIGH)
		ret

; -----------------------------------------------------------------------------
;                          Check if year is a leap year
; -----------------------------------------------------------------------------
; INPUT:	CX = year
; OUTPUT:	CY = leap year
; -----------------------------------------------------------------------------

; ------------- Check if it can be a leap year (year is divisible by 4)

CheckLeapYear:	test	cl,3		; is it divisible by 4?
		jz	CheckLeapYear4	; it can be a leap year
		ret			; here is NC = it is not leap year

; ------------- Set "leap year" flag if it is Julian calendar

CheckLeapYear2:	stc			; set "leap year" flag
		ret

; ------------- Check if it is Gregorian calendar (1582 and more)

CheckLeapYear4:	cmp	cx,SPLITYEAR	; is it Gregorian calendar?
		jle	CheckLeapYear2	; it is Julian calendar

; ------------- Push registers

		push	eax		; push EAX
		push	edx		; push EDX

; ------------- Get century (-> EDX)

		movzx	edx,cx		; EDX <- year
		mov	eax,42949673	; EAX <- 100000000h/100 (round up)
		mul	edx		; EDX <- century

; ------------- Get year in century

		imul	eax,edx,-100	; EAX <- convert century to years
		add	ax,cx		; AX <- year in century

; ------------- Check leap century (century is divisible by 4)

		jnz	CheckLeapYear6	; it is not century
		test	dl,3		; is century divisible by 4?
		jnz	CheckLeapYear8	; century is not divisible by 4

; ------------- Pop registers

CheckLeapYear6:	stc			; set "leap year" flag
CheckLeapYear8:	pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                           Get last day in month
; -----------------------------------------------------------------------------
; INPUT:	DL = month (1 to 12)
;		CX = year (MINYEAR to MAXYEAR)
; OUTPUT:	AL = last day in month (28, 29, 30, 31 or 0 on error)
;		CY = invalid month or year (AL = 0)
; -----------------------------------------------------------------------------

; ------------- Push registers

GetLastDayMonth:push	ebx		; push EBX

; ------------- Check month

		mov	al,dl		; AL <- month
		dec	al		; AL <- month relative (0 to 11)
		cmp	al,11		; check maximal month
		ja	GetLastDayMon8	; invalid month

; ------------- Check year

		cmp	cx,MINYEAR	; minimal year
		jl	GetLastDayMon8	; invalid year
		cmp	cx,MAXYEAR	; maximal year
		jg	GetLastDayMon8	; invalid year

; ------------- Get days in month (-> AL)

		xor	ebx,ebx		; EBX <- 0
		call	CheckLeapYear	; check leap year (CY = year)
		setnc	bl		; BL <- 0 leap year, 1 non-leap year
		dec	ebx		; EBX <- -1 leap year, 0 non-leap year
		and	ebx,DayMonthTabL-DayMonthTabN ; EBX <- table offset
		add	ebx,DayMonthTabN ; EBX <- leap/non-leap table pointer
		xlatb			; AL <- days in month

; ------------- OK: Pop registers

		clc			; clear error flag
		pop	ebx		; pop EBX
		ret

; ------------- ERROR: Pop registers

GetLastDayMon8:	mov	al,0		; AL <- 0, invalid month or year
		stc			; set error flag
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                            Get days in year
; -----------------------------------------------------------------------------
; INPUT:	CX = year (MINYEAR to MAXYEAR)
; OUTPUT:	AX = days in year (355, 365, 366 or 0 on error)
;		CY = invalid year (AX = 0)
; -----------------------------------------------------------------------------

; ------------- Check year

GetDaysInYear:	cmp	cx,MINYEAR	; minimal year
		jl	GetDaysInYear8	; invalid year
		cmp	cx,MAXYEAR	; maximal year
		jg	GetDaysInYear8	; invalid year

; ------------- Get days in year (-> EAX, it clears CF)

		mov	ax,355		; AX <- 355, days in year 1582
		cmp	cx,SPLITYEAR	; is it split year 1582?
		je	GetDaysInYear6	; it is split year 1582
		call	CheckLeapYear	; check leap year
		adc	al,10		; correction (it clears CF)
GetDaysInYear6:	ret			; here is NC

; ------------- ERROR

GetDaysInYear8:	xor	ax,ax		; AX <- 0, invalid year
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                           Check date-time structure
; -----------------------------------------------------------------------------
; INPUT:	EBX = date-time structure (see DATETIME)
; OUTPUT:	EAX = error code (ERR_OK or ERR_DT_* error codes)
;		CY = invalid date or time entry
; -----------------------------------------------------------------------------

; ------------- Push registers

DateTimeCheck:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Check "nanosecond" entry

		mov	edx,ERR_DT_NSEC	; EDX <- error code "invalid nsec"
		cmp	dword [ebx+DATETIME_NSec],999999999 ; check value
		ja	DateTimeCheck8	; invalid nanosecond

; ------------- Check "second" entry

		mov	eax,[ebx+DATETIME_Time] ; EAX <- time
		inc	edx		; EDX <- error code "invalid second"
		cmp	al,59		; check maximal value
		ja	DateTimeCheck8	; invalid second

; ------------- Check "minute" entry

		inc	edx		; EDX <- error code "invalid minute"
		cmp	ah,59		; check maximal value
		ja	DateTimeCheck8	; invalid minute

; ------------- Check "hour" entry

		shr	eax,16		; AL <- hour
		inc	edx		; EDX <- error code "invalid hour"
		cmp	al,23		; check maximal value
		ja	DateTimeCheck8	; invalid hour

; ------------- Prepare year (-> CX), month (-> AH) and day in month (-> AL)

		mov	eax,[ebx+DATETIME_Date] ; EAX <- date
		inc	edx		; EDX <- error code "invalid day"
		cmp	al,0		; check minimal day in month
		je	DateTimeCheck8	; invalid day in month
		shld	ecx,eax,16	; CX <- year
		cmp	cx,SPLITYEAR	; year 1582?
		je	DateTimeCheck4	; check year 1582

; ------------- Check day in month

DateTimeCheck2:	xchg	al,ah		; AL <- month, AH <- day in month
		xchg	eax,edx		; DL <- month, DH <- day, EAX <- error
		xchg	eax,ebx		; EBX <- error
		call	GetLastDayMonth	; get last day in month -> AL
		xchg	eax,ebx		; EAX <- error, BL <- last day in month
		xchg	eax,edx		; AL <- month, AH <- day, EDX <- error
		jc	DateTimeCheck6	; invalid month or year
		cmp	ah,bl		; check day in month
		ja	DateTimeCheck8	; invalid day in month

; ------------- OK: Pop registers

DateTimeCheck3:	xor	eax,eax		; EAX <- clear error
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; ------------- Check invalid date range in year 1582

DateTimeCheck4:	cmp	ah,10		; October 1582?
		jne	DateTimeCheck2	; not October 1582
		cmp	al,5		; start of invalid date interval
		jb	DateTimeCheck3	; date is OK
		cmp	al,14		; end of invalid date interval
		ja	DateTimeCheck2	; it is not invalid date interval
		jmp	short DateTimeCheck8 ; error

; ------------- Invalid month or year

DateTimeCheck6:	inc	edx		; EDX <- error code "invalid month"
		dec	eax		; AL <- month - 1
		cmp	al,12		; check maximal month (CY=invalid year)
		adc	edx,byte 0	; EDX <- error code "invalid year"

; ------------- ERROR: Pop registers

DateTimeCheck8:	xchg	eax,edx		; EAX <- error code
		stc			; set error flag
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                       Increase date-time by one day
; -----------------------------------------------------------------------------
; INPUT:	EBX = date-time structure (see DATETIME)
; OUTPUT:	CY = invalid date (structure has not been changed)
; NOTES:	Date entries are verified only partially.
; -----------------------------------------------------------------------------

; ------------- Push registers

DateTimeAddDay:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Increase day in month

		mov	eax,[ebx+DATETIME_Date] ; EAX <- date
		inc	eax		; increase day in month
		cmp	eax,(SPLITYEAR*256 + 10)*256 + 5 ; date 10/5/1582
		setne	dl		; DL <- 0=date 10/5/1582, 1=not
		dec	edx		; DL <- 0ffh=date 10/5/1582, 0=not
		and	dl,10		; DL <- 10=date 10/5/1582, 0=not
		add	al,dl		; date correction for 10/5/1582
		cmp	al,28		; maximal date we need not to check
		jbe	DateTimeAddDay6	; new date is OK

; ------------- Check day in month

		shld	ecx,eax,16	; CX <- year
		xchg	al,ah		; AL <- month, AH <- day in month
		xchg	eax,edx		; DL <- month, DH <- day in month
		call	GetLastDayMonth	; get last day in month -> AL
		xchg	eax,edx		; AL <- month, AH <- day, DL<-last day
		xchg	al,ah		; AL <- day in month, AH <- month
		jc	DateTimeAddDay8	; invalid month or year
		cmp	al,dl		; check day in month
		jbe	DateTimeAddDay6	; new day is OK

; ------------- Increase month

		mov	al,1		; AL <- 1, first day in next month
		inc	ah		; AH <- next month
		cmp	ah,12		; check month
		jbe	DateTimeAddDay6	; next month is OK

; ------------- Increase year

		mov	ah,1		; AH <- 1, first month in next year
		inc	ecx		; increase year
		cmp	cx,MAXYEAR	; maximal year
		stc			; set error flag
		jg	DateTimeAddDay8	; invalid year
		add	eax,10000h	; increase year
		
; ------------- Set new date

DateTimeAddDay6:mov	[ebx+DATETIME_Date],eax ; store new date
		clc			; clear error flag

; ------------- Pop registers

DateTimeAddDay8:pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                       Decrease date-time by one day
; -----------------------------------------------------------------------------
; INPUT:	EBX = date-time structure (see DATETIME)
; OUTPUT:	CY = invalid date (structure has not been changed)
; NOTES:	Date entries are verified only partially.
; -----------------------------------------------------------------------------

; ------------- Push registers

DateTimeSubDay:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Decrease day in month

		mov	eax,[ebx+DATETIME_Date] ; EAX <- date
		cmp	eax,(SPLITYEAR*256 + 10)*256 + 15 ; date 10/15/1582
		setne	dl		; DL <- 0=date 10/15/1582, 1=not
		dec	edx		; DL <- 0ffh=date 10/15/1582, 0=not
		and	dl,10		; DL <- 10=date 10/15/1582, 0=not
		inc	edx		; DL <- correction + 1 day
		sub	al,dl		; decrease day in month
		jnz	DateTimeSubDay6	; new date is OK

; ------------- Push registers 2

		push	ecx		; push ECX

; ------------- Decrease month

		shld	ecx,eax,16	; CX <- year
		dec	ah		; decrease month
		jnz	DateTimeSubDay2	; new month is OK

; ------------- Decrease year

		dec	ecx		; decrease year
		sub	eax,0F400h	; decrease year, set month to 12

; ------------- Get last day in previous month

DateTimeSubDay2:mov	dl,ah		; DL <- month
		call	GetLastDayMonth	; get last day in month -> AL

; ------------- Pop registers 2

		pop	ecx		; pop ECX
		jc	DateTimeSubDay8	; invalid month or year

; ------------- Set new date

DateTimeSubDay6:mov	[ebx+DATETIME_Date],eax ; store new date
		clc			; clear error flag

; ------------- Pop registers

DateTimeSubDay8:pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                       Convert date to absolute day
; -----------------------------------------------------------------------------
; INPUT:	AL = day (1..31, may be -128 to +127)
;		CX = year (MINYEAR to MAXYEAR)
;		DL = month (1..12, may be -128 to +127)
; OUTPUT:	EAX = absolute day (offset to 1/1/1 CE, MINDAY to MAXDAY)
; NOTES:	Input entries are signed and their overlaps are accounted.
; 		Takes 25 ns on 1.6 GHz CPU.
; -----------------------------------------------------------------------------
; Conversion from date to absolute day:
;	m = month - 1;
;	if (m >= 2) a = 0; else a = -1;
;	y = year + a + 40000;
;	m = m + (12 & a) - 2;
;	date = day + (153*m + 2)/5 + y*365;
;	y = y/4;
;	date = date + y - 14610307;
;	if (date >= SPLITDATE)
;	{
;		y = y / 25;
;		date = date - y + y/4 + 302;
;	}
; -----------------------------------------------------------------------------

; ------------- Push registers

DateToAbsDay:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Extend input entries (EAX <- month, EBX <- day, ECX <- year)

		movsx	ebx,al		; EBX <- day
		movsx	eax,dl		; EAX <- month
		movsx	ecx,cx		; ECX <- year

; ------------- Normalize month (EAX) and year (ECX)

		dec	eax		; EAX <- change month to zero-based
DateToAbsDay2:	cmp	eax,byte 12	; is it valid value?
		jae	DateToAbsDay6	; invalid value

; ------------- Prepare month flag (if month is 2 or more, -> EDX)

		cmp	al,2		; check (month-1) 0 or 1
		sbb	edx,edx		; EDX <- -1=(month-1)<2, 0=(month-1)>=2

; ------------- Year correction (-> ECX)

		lea	ecx,[ecx+edx+40000] ; ECX <- convert year to positive

; ------------- Month correction (-> EDX)
; Values: 10, 11, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9)

		and	edx,byte 12	; EDX <- 12=month 1 or 2, 0=month > 2
		add	edx,eax		; EDX <- add month - 1
		dec	edx		; EDX <- month correction
		dec	edx		; EDX <- month correction

; ------------- Recalc month to days (-> EBX)
; Values: 306, 337, 0, 31, 61, 92, 122, 153, 184, 214, 245, 275)

		imul	eax,edx,153	; EAX <- month coefficient * 153
		inc	eax		; EAX + 1
		inc	eax		; EAX + 2
		imul	edx,eax,13108	; EDX <- days * 10000h
		shr	edx,16		; EDX <- days

; ------------- Add day in month (-> EDX)

		add	edx,ebx		; EDX <- add day in month

; ------------- Recalc years to days (without leap correction, -> EAX)

		imul	eax,ecx,365	; EAX <- days of years
		add	eax,edx		; EAX <- add days in month and day

; ------------- Add leap year correction for Julian calendar

		shr	ecx,2		; ECX <- year / 4
		add	eax,ecx		; EAX <- add days in leap years
		sub	eax,14610307	; EAX <- start offset of calendar

; ------------- Check if it is Julian or Gregorian calendar

		cmp	eax,SPLITDATE	; check date 10/15/1582
		jl	DateToAbsDay4	; it is Julian calendar

; ------------- Do correction for Gregorian calendar

		imul	edx,ecx,167773	; EDX <- (year/100) * 400000h
		shr	edx,22		; EDX <- year/100
		sub	eax,edx		; EAX <- subtract century correction
		shr	edx,2		; EDX <- century / 4
		add	eax,edx		; EAX <- add 400-years correction
		add	eax,302		; EAX <- start offset of calendar

; ------------- Pop registers

DateToAbsDay4:	pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; ------------- Normalize month (-> EAX) and year (-> ECX)

DateToAbsDay6:	or	eax,eax		; is month negative?
                jns	DateToAbsDay8	; month is not negative
		add	eax,byte 12	; correct month
		dec	ecx		; correct year
		jmp	DateToAbsDay2	; next year	

DateToAbsDay8:	sub	eax,byte 12	; correct month
		inc	ecx		; correct year
		jmp	DateToAbsDay2	; next year

; -----------------------------------------------------------------------------
;                     Convert date-time to absolute time
; -----------------------------------------------------------------------------
; INPUT:	EBX = date-time structure (see DATETIME)
; OUTPUT:	EDX:EAX = absolute time (100-nanoseconds from 1/1/1 CE 0:00)
; NOTES:	Entries are taken as signed and their overlaps are accounted.
;		Only time and date entries are used. Flags, week, day in year
;			and day of week are ignored.
; 		Takes 70 ns on 1.6 GHz CPU.
; -----------------------------------------------------------------------------
; On whole function: ESI = date-time structure, EBP:EDI = accumulator.

; ------------- Push registers

DateTimeToAbs:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP
		mov	esi,ebx		; ESI <- date-time structure

; ------------- Convert second to 100-nanoseconds (-> EBP:EDI)

		movsx	eax,byte [esi+DATETIME_Sec] ; EAX <- second
		mov	ecx,10000000	; ECX <- 100-nsec per second
		imul	ecx		; EAX <- recalc seconds to 100-nanosec
		xchg	eax,edi		; EDI <- accumulator LOW
		mov	ebp,edx		; EBP <- accumulator HIGH

; ------------- Add nanosecond (-> EBP:EDI, ECX <- 0)

		mov	eax,1374389535	; EAX <- 2000000000h/100 (rounded up)
		imul	dword [esi+DATETIME_NSec] ; EDX:EAX <- (nsec/100) << 37
		sar	edx,5		; EDX <- nsec/100
		mov	eax,edx		; EAX <- nsec/100
		shr	eax,31		; EAX <- signum bit (0 or 1)
		add	eax,edx		; EAX <- nsec/100 + signum correction
		cdq			; EDX:EAX <- nsec/100
		add	edi,eax		; EDI <- add to accumulator LOW
		adc	ebp,edx		; EBP <- add to accumulator HIGH

; ------------- Add hour and minute (-> EBP:EDI)

		movsx	eax,byte [esi+DATETIME_Hour] ; EAX <- hour
		imul	eax,byte 60	; EAX <- minutes
		movsx	ecx,byte [esi+DATETIME_Min] ; ECX <- minute
		add	eax,ecx		; EAX <- hour + minute
		mov	ecx,60*10000000	; ECX <- 100-nsec per minute
		imul	ecx		; recalc minutes to 100-nanoseconds
		add	edi,eax		; EDI <- add to accumulator LOW
		adc	ebp,edx		; EBP <- add to accumulator HIGH

; ------------- Convert date to absolute day

		mov	ecx,[esi+DATETIME_Date]
		mov	al,cl		; AL <- day
		mov	dl,ch		; DL <- month
		sar	ecx,16		; CX <- year
		call	DateToAbsDay	; recalc date to absolute day

; ------------- Convert days to minutes (-> EDX:EAX)

		mov	ecx,24*60	; ECX <- minutes per day
		imul	ecx		; EDX:EAX <- recalc days to minutes

; ------------- Convert minutes to 100-nanoseconds (-> EDX:EAX)

		xchg	eax,ebx		; EBX <- store minutes LOW
		xchg	eax,edx		; EAX <- minutes HIGH
		mov	ecx,60*10000000 ; ECX <- 100-nanoseconds per minute
		imul	ecx		; recalc minutes HIGH to 100-nanosec
		add	ebp,eax		; add minutes HIGH to accumulator HIGH
		xchg	eax,ebx		; EAX <- minutes LOW
		mul	ecx		; recalc minutes LOW to 100-nanosec
		add	eax,edi		; add accumulator LOW to minutes LOW
		adc	edx,ebp		; add accumulator HIGH to minutes LOW

; ------------- Pop registers

		pop	ebp		; pop EBP
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                     Convert absolute time to date-time
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = date-time structure (see DATETIME)
; OUTPUT:	EBX structure filled up with date and time
; NOTES:	Only time and date entries (including day of week) are
;			returned. To get extended entries call DateTimeExt.
;		Takes 140 ns on 1.6 GHz CPU.
; -----------------------------------------------------------------------------
; Conversion from absolute day to date-time:
;	b = 0;					// b=0 century for Julian
;	c = date + 14610306;			// c=relative day to 3/1/-40000
;	if (c >= (SPLITDATE+14610306))		// check date 10/15/1582
;	{
;		c = c + (14610004-14610306);	// c=relative day for Gregorian
;		b = (c*4 + 3)/146097;		// b=century
;						// 146097=days per 400 years
;						//  = ((4*365+1)*25-1)*4+1
;		c = c - (b*146097)/4;		// c=day in this century
;	}
;
;	d = (c*4 + 3)/1461;			// d=year in this century
;						// 1461=days per 4 years
;						//  = 4*365+1
;	b = b*100 + d;				// b=year+40000
;	c = c - (1461*d)/4;			// c=day in year (from March)
;	d = (c*5 + 2)/153;			// d=month-3 (starting March)
;
;	day = c - (153*d + 2)/5 + 1;		// day in this month
;	if (d >= 10) e = 1; else e = 0;		// flag, month >= 10
;	year = b - 40000 + e;			// year
;	month = d + 3 + ((-12) & (-e));		// month
; -----------------------------------------------------------------------------
; On whole function: ESI = date-time structure

; ------------- Push registers

AbsToDateTime:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		mov	esi,ebx		; ESI <- date-time structure

; ------------- Convert time to positive value (-> EDX:EAX)

		sub	eax,MINTIMEL	; convert to positive value LOW
		sbb	edx,MINTIMEH	; convert to positive value HIGH

; ------------- Split time to minutes (-> EDI:EBX) and 100-nanoseconds (-> EDX)

		xchg	eax,ebx		; EBX <- save time LOW
		xor	eax,eax		; EAX <- 0
		mov	ecx,60*10000000 ; ECX <- 100-nanoseconds per minute
		xchg	eax,edx		; EAX <- time HIGH, EDX <- 0
		div	ecx		; split to minutes HIGH
		xchg	eax,edi		; EDI <- minutes HIGH
		xchg	eax,ebx		; EAX <- time LOW
		div	ecx		; split to minutes LOW
		xchg	eax,ebx		; EBX <- minutes LOW
		
; ------------- Split 100-nanoseconds to nanosecond and second

		mov	ecx,edx		; ECX <- 100-nanoseconds
		mov	eax,1801439851	; EAX <- 40 0000 0000 0000h/10000000
		mul	ecx		; EDX:EAX <- (100-nsec/10000000) << 54
		shr	edx,22		; EDX <- second
		mov	[esi+DATETIME_Sec],dl ; store second
		imul	edx,edx,10000000 ; EDX <- recalc second to 100-nsec
		sub	ecx,edx		; ECX <- 100-nanoseconds
		imul	eax,ecx,100	; EAX <- nanoseconds
		mov	[esi+DATETIME_NSec],eax ; store nanosecond

; ------------- Split minutes to minutes (-> EDX) and days (-> EAX)

		mov	edx,edi		; EDX <- minutes HIGH
		xchg	eax,ebx		; EAX <- minutes LOW
		mov	ecx,24*60	; ECX <- minutes per day
		div	ecx		; split to minutes and days

; ------------- Split minutes to minute and hour

		imul	ecx,edx,279621	; ECX <- (minutes/60) * 1000000h
		shr	ecx,24		; ECX <- hour
		mov	[esi+DATETIME_Hour],cl ; store hour
		imul	ecx,ecx,60	; ECX <- recalc hour to minutes
		sub	edx,ecx		; EDX <- minute
		mov	[esi+DATETIME_Min],dl ; store minute

; ------------- Get day of week (number of days -> ECX)

		lea	ebx,[eax+4]	; EBX <- number of days + correction
		xchg	eax,ecx		; ECX <- save number of days
		mov	eax,613566757	; EAX <- 100000000h/7 round up
		mul	ebx		; EDX <- number of weeks
		imul	eax,edx,7	; EAX <- number of weeks * 7
		sub	ebx,eax		; EBX <- day of week
		inc	ebx		; EBX <- day of week starting 1
		mov	[esi+DATETIME_WDay],bl ; store day of week

; ------------- Prepare Julian calendar (here is ECX = number of days)

		xor	ebx,ebx		; EBX <- b=0 century for Julian
		add	ecx,14610306+MINDAY ; ECX <- c=relative day 3/1/-40000

; ------------- Check if it is Julian or Gregorian calendar

		cmp	ecx,SPLITDATE+14610306 ; check date 10/15/1582
		jl	AbsToDateTime2	; Julian calendar

; ------------- Prepare Gregorian calendar

		add	ecx,14610004-14610306 ; ECX <- c=relative day Gregorian
		lea	ebx,[ecx*4+3]	; EBX <- c*4+3
		mov	eax,963315389	; EAX <- 8000 0000 0000h / 146097
		mul	ebx		; EDX:EAX <- ((c*4+3)/146097) << 47
		shr	edx,15		; EDX <- b=(c*4+3)/146097 century
		mov	ebx,edx		; EBX <- b=(c*4+3)/146097 century
		imul	eax,edx,146097	; EAX <- b*146097
		shr	eax,2		; EAX <- (b*146097)/4
		sub	ecx,eax		; ECX <- c=c-(b*146097)/4 day in cent.

; ------------- Prepare year in century (-> EDX = "d")

AbsToDateTime2:	lea	edx,[ecx*4+3]	; EDX <- c*4+3
		mov	eax,376287347	; EAX <- 80 0000 0000h / 1461
		mul	edx		; EDX:EAX <- ((c*4+3)/1461) << 39
		shr	edx,7		; EDX <- d=(c*4+3)/1461 year in century

; ------------- Prepare year+40000 (-> EBX = "b")

		imul	ebx,ebx,100	; EBX <- b*100
		add	ebx,edx		; EBX <- b=b*100+d, year+40000

; ------------- Prepare day in year (relative to March, -> ECX = "c")

		imul	edx,edx,1461	; EDX <- 1461*d
		shr	edx,2		; EDX <- (1461*d)/4
		sub	ecx,edx		; ECX <- c=c-(1461*d)/4 day in year

; ------------- Prepare month-3 (relative to March, -> EDX = "d")

		lea	edx,[ecx*4+ecx+2] ; EDX <- c*5+2
		mov	eax,3593175255	; EAX <- 80 0000 0000h / 153 
		mul	edx		; EDX:EAX <- ((c*5+2)/153) << 39
		shr	edx,7		; EDX <- d=(c*5+2)/153 month-3
		
; ------------- Calculate day in month (-> CL)

		mov	edi,edx		; EDI <- save "d"
		imul	edx,edx,153	; EDX <- 153*d
		inc	edx		; EDX <- 153*d+1
		inc	edx		; EDX <- 153*d+2
		mov	eax,858993460	; EAX <- 100000000h/5
		mul	edx		; EDX <- (153*d+2)/5
		sub	ecx,edx		; ECX <- c-(153*d+2)/5
		inc	ecx		; ECX <- day=c-(153*d+2)/5+1 day

; ------------- Calculate year (add to ECX)

		cmp	edi,10		; check if month >= 10
		sbb	eax,eax		; EAX <- 0 if month >= 10, -1 if < 10
		inc	eax		; EAX <- e, if(d>=10) e=1 else e=0
		lea	ebx,[ebx-40000+eax] ; EBX <- year
		shl	ebx,16		; EBX <- year << 16
		or	ecx,ebx		; ECX <- add year

; ------------- Calculate month

		neg	eax		; EAX <- -e
		and	eax,byte -12	; EAX <- (-12)&(-e)
		lea	eax,[edi+3+eax]	; EAX <- month=d+3-((-12)&(-e))
		mov	ch,al		; CH <- month
		mov	[esi+DATETIME_Date],ecx ; store date

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;             Convert absolute time to date-time with extended entries
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanosec from 1/1/1 BCE 0:00)
;		EBX = date-time structure (see DATETIME)
; OUTPUT:	EBX structure filled up with date and time + extended entries
; NOTES:	Takes 230 ns on 1.6 GHz CPU.
; -----------------------------------------------------------------------------

AbsToDateTimeExt:
		call	AbsToDateTime	; convert absolute time to date-time

; DateTimeExt must follow!

; -----------------------------------------------------------------------------
;                       Get calendar extended entries
; -----------------------------------------------------------------------------
; INPUT:	EBX = date-time structure with valid date (see DATETIME)
; OUTPUT:	Extended entries and day of week of EBX structure are filled.
; NOTES:	Date entries (excluding day of week) of EBX structure
;			should be valid or else result will be unpredictable.
;		Takes 90 ns on 1.6 GHz CPU.
; -----------------------------------------------------------------------------
; On whole function: ESI = date-time structure

; ------------- Push registers

DateTimeExt:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Clear flags

		mov	esi,ebx		; ESI <- date-time structure
		mov	byte [esi+DATETIME_Flags],0 ; clear flags

; ------------- Get current date (EAX <- day, EDX <- month, ECX <- year)

		mov	ecx,[esi+DATETIME_Date] ; ECX <- date
		mov	al,cl		; AL <- day
		mov	dl,ch		; DL <- month
		sar	ecx,16		; CX <- year

; ------------- Set leap year flag

		call	CheckLeapYear	; check if year is a leap year
		adc	byte [esi+DATETIME_Flags],0 ; set leap year flag

; ------------- Convert date to absolute day (-> EAX)

		call	DateToAbsDay	; recalc date to absolute day

; ------------- Set Gregorian calendar flag

		cmp	eax,SPLITDATE	; check split day 10/15/1582
		xchg	eax,ebx		; EBX <- absolute day
		setge	al		; AL <- 1 if date is greater or equal
		shl	al,1		; AL <- B1 if Gregorian calendar
		or	[esi+DATETIME_Flags],al ; set Gregorian calendar flag

; ------------- Day of week

		lea	edi,[ebx+4-MINDAY] ; EDI <- day + correction
		mov	eax,613566757	; EAX <- 100000000h/7 round up
		mul	edi		; EDX <- number of weeks
		imul	eax,edx,-7	; EAX <- number of weeks * -7
		add	eax,edi		; EAX <- day of week
		inc	eax		; EAX <- day of week starting 1
		mov	[esi+DATETIME_WDay],al ; store day of week
		xchg	eax,edi		; EDI <- day of week

; ------------- Get day of January 1st of this year (-> EAX)

		mov	al,1		; AL <- day 1
		mov	dl,1		; DL <- month 1
		call	DateToAbsDay	; get absolute day of start of year

; ------------- Day in year

		sub	ebx,eax		; EBX <- day in year
		mov	[esi+DATETIME_YDay],bx ; set day in year

; ------------- Prepare day of week - 1 of January 4th (-> EBX)

		sub	edi,ebx		; EDI <- day of week - day in year
		add	edi,2+53*7	; EDI <- correction
		mov	eax,613566757	; EAX <- 100000000h/7 round up
		mul	edi		; EDX <- number of weeks
		imul	eax,edx,-7	; EAX <- number of weeks * -7
		add	eax,edi		; EAX <- day of week of January 4th
		xchg	eax,ebx		; EBX <- day of week of January 4th

; ------------- Prepare number of weeks in this year

		mov	bh,52		; BH <- 52 weeks
		cmp	bl,7-1		; is it Sunday?
		je	DateTimeExt2	; Sunday - this year has 53 weeks
		cmp	bl,6-1		; is it Saturday?
		jne	DateTimeExt3	; this year has 52 weeks
		test	byte [esi+DATETIME_Flags],B0 ; is it leap year?
		jz	DateTimeExt3	; it is not leap year, it has 52 weeks
DateTimeExt2:	or	byte [esi+DATETIME_Flags],B2 ; set flag 53 weeks
		inc	bh		; BH <- 53 weeks

; ------------- Day offset (from begin of year) of first week (-> EAX)

DateTimeExt3:	mov	al,3		; AL <- day correction
		sub	al,bl		; AL <- day offset of first week
		movsx	eax,al		; EAX <- day offset of first week

; ------------- Get week (first week is week with January 4th, ISO-8601)

		movzx	edx,word [esi+DATETIME_YDay] ; EDX <- day in year
		sub	edx,eax		; EDX <- distance from first week
		add	edx,byte 7	; EDX <- add correction
		mov	eax,613566757	; EAX <- 100000000h/7 round up
		mul	edx		; EDX <- week

; ------------- Check last week of previous year (only for January 1,2,3)

		or	dl,dl		; is it last week of previous year?
		jnz	DateTimeExt5	; it is not last week
		or	byte [esi+DATETIME_Flags],B3 ; set flag of last week

; ------------- Last week of year 1582 (Saturday Januay 1 or Sunday January 2)

		mov	cx,[esi+DATETIME_Year] ; CX <- year
		mov	dl,51		; week 51 for 1582 year
		cmp	cx,1583		; is it 1583 year?
		je	DateTimeExt6	; previous year has 51 weeks

; ------------- Previous year is always 52 or 53 weeks

		inc	edx		; DL <- week 52
		cmp	bl,1-1		; is January 4th in Monday?
		je	DateTimeExt4	; previous year has always 53 weeks
		cmp	bl,3-1		; is January 4th in Wednesday?
		je	DateTimeExt6	; previous year has always 52 weeks

; ------------- Tuesday, check if previous year is a leap year

		dec	cx		; CX <- previous year
		call	CheckLeapYear	; check if previous year is a leap year
		jnc	DateTimeExt6	; previous year is not leap, 52 weeks
DateTimeExt4:	inc	edx		; DL <- week 53
		jmp	short DateTimeExt6 ; it is valid week

; ------------- First week of following year (1582 year is always OK)

DateTimeExt5:	cmp	dl,bh		; is it valid week?
		jbe	DateTimeExt6	; it is valid week
		mov	dl,1		; first week of next year
		or	byte [esi+DATETIME_Flags],B4 ; first week of next year

; ------------- Store week

DateTimeExt6:	mov	[esi+DATETIME_Week],dl ; store week

; ------------- Pop registers

DateTimeExt9:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                             Debug test calendar
; -----------------------------------------------------------------------------

%ifdef DEBUG_CALEND

; ------------- Init system time structure

DebTestCal:	mov	ebx,DebTestCalBuf1
		mov	dword [ebx+DATETIME_NSec],0
		mov	byte [ebx+DATETIME_Sec],0
		mov	byte [ebx+DATETIME_Min],0
		mov	byte [ebx+DATETIME_Hour],0
		mov	byte [ebx+DATETIME_WDay],1
		mov	byte [ebx+DATETIME_Day],1
		mov	byte [ebx+DATETIME_Month],1
		mov	word [ebx+DATETIME_Year],MINYEAR

; ------------- Prepare Julian start time

		call	DateTimeToAbs
		mov	[DebTestCalJul],eax
		mov	[DebTestCalJul+4],edx
		call	AbsToDateTime

; ------------- Display year

DebTestCal1:	cmp	byte [ebx+DATETIME_Day],1
		jne	DebTestCal2
		cmp	byte [ebx+DATETIME_Month],1
		jne	DebTestCal2
		call	DebOutCR
		movsx	eax,word [ebx+DATETIME_Year]
		call	DebOutNumSig
		call	DebOutSpc
		call	DebOutSpc
		call	DebOutSpc

; ------------- Check system time structure

DebTestCal2:	push	eax
		call	DateTimeCheck
		pop	eax
		jnc	DebTestCal4
		mov	esi,DebTestTxt1
DebTestCal3:	call	DebNewLine
		call	DebOutTextData
		jmp	$

; ------------- Convert to julian time

DebTestCal4:	call	DateTimeToAbs
		mov	esi,DebTestTxt2
		cmp	eax,[DebTestCalJul]
		jne	DebTestCal3
		cmp	edx,[DebTestCalJul+4]
		jne	DebTestCal3

; ------------- Convert to system time

		push	ebx
		mov	ebx,DebTestCalBuf2
		call	AbsToDateTime
		call	DateTimeExt
		pop	ebx

		mov	esi,DebTestTxt3
		mov	eax,[DebTestCalBuf2+DATETIME_NSec]
		cmp	eax,[ebx+DATETIME_NSec]
		jne	DebTestCal3
		mov	eax,[DebTestCalBuf2+DATETIME_Time]
		cmp	eax,[ebx+DATETIME_Time]
		jne	DebTestCal3
		mov	eax,[DebTestCalBuf2+DATETIME_Date]
		cmp	eax,[ebx+DATETIME_Date]
		jne	DebTestCal3

		mov	cx,[ebx+DATETIME_Year]
		call	CheckLeapYear
		rcl	al,1
		xor	al,[DebTestCalBuf2+DATETIME_Flags]
		and	al,1
		jnz	DebTestCal3

		mov	ax,[DebTestCalBuf2+DATETIME_YDay]
		cmp	ax,[ebx+DATETIME_YDay]
		jne	DebTestCal3

		mov	al,0
		cmp	word [ebx+DATETIME_Year],1582
		jl	DebTestCal42
		jg	DebTestCal41
		cmp	byte [ebx+DATETIME_Month],10
		jl	DebTestCal42
		jg	DebTestCal41
		cmp	byte [ebx+DATETIME_Day],4
		jle	DebTestCal42
DebTestCal41:	mov	al,2
DebTestCal42:	xor	al,[DebTestCalBuf2+DATETIME_Flags]
		and	al,2
		jnz	DebTestCal3

; ------------- Increase day of week

		mov	al,[ebx+DATETIME_WDay]
		inc	eax
		cmp	al,7
		jbe	DebTestCal5
		mov	al,1
DebTestCal5:	mov	[ebx+DATETIME_WDay],al

; ------------- Increase day in year

		mov	ax,[ebx+DATETIME_YDay]
		inc	eax
		cmp	byte [ebx+DATETIME_Day],31
		jne	DebTestCal6
		cmp	byte [ebx+DATETIME_Month],12
		jne	DebTestCal6
		xor	eax,eax
DebTestCal6:	mov	[ebx+DATETIME_YDay],ax

; ------------- Increase date

		add	dword [DebTestCalJul],2a69c000h
		adc	dword [DebTestCalJul+4],0c9h
		call	DateTimeAddDay
		jnc	DebTestCal1

; ------------- Test is OK

		mov	esi,DebTestTxt4
		call	DebOutTextData
		call	DebNewLine
		ret
%endif

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Days in month - non-leap year

DayMonthTabN:	db	31		; January
		db	28		; February
		db	31		; March
		db	30		; April
		db	31		; May
		db	30		; June
		db	31		; July
		db	31		; August
		db	30		; September
		db	31		; October
		db	30		; November
		db	31		; December

; ------------- Days in month - leap year

DayMonthTabL:	db	31		; January
		db	29		; February
		db	31		; March
		db	30		; April
		db	31		; May
		db	30		; June
		db	31		; July
		db	31		; August
		db	30		; September
		db	31		; October
		db	30		; November
		db	31		; December

; ------------- Date coefficients

		align	4, db 0
JulianBase:	dq	1721423.5	; base Julian day (1/1/1 CE, 0:00:00)
AbsToDayCoef:	dt	1.1574074074074074074074e-12 ; abs. time to days coeff.
		align	4, db 0
DayToAbsCoef:	dq	864000000000.0	; days to absolute time coefficient

; ------------- Debug test calendar
%ifdef DEBUG_CALEND
DebTestTxt1:	CTEXTDATA ' Error: DateTimeCheck '
DebTestTxt2:	CTEXTDATA ' Error: DateTimeToAbs '
DebTestTxt3:	CTEXTDATA ' Error: AbsToDateTime '
DebTestTxt4:	CTEXTDATA ' OK '
%endif

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- Debug test calendar
%ifdef DEBUG_CALEND
		align	4, resb 1
DebTestCalBuf1:	resb	DATETIME_size
DebTestCalBuf2:	resb	DATETIME_size
DebTestCalJul:	resd	2
%endif
