; =============================================================================
;
;                              Litos - Kernel
;
; =============================================================================

; The system is loaded from the second 4 KB page of memory (first 4 KB page
; is reserved for BIOS use), e.g. from address 0:1000h, starting with CS=0
; and IP=1000h. It enables easy share code to 16-bit and 32-bit instructions.
; Later it will be relocated to 2 GB position of memory (80000000h) and
; therefore whole kernel is compiled for this address. Real-mode instructions
; accept only low 16 bits of address and then it works mostly with this
; correctly. 

; [map all LITOS.MAP]

; ------------- Compiler configuration

%define		DEBUG			; config - display DEBUG info
;%define	MINI			; config - generate mini kernel
					;  - 1 MB minimal memory (else 2 MB)
					;  - only 1 processor (else 32)
					;  - max. 256 MB of RAM (else 2 GB)
					;  - max. 250 tasks (else 4090)
					;  - max. 3FFh I/O port (else 0FFFFh)
					;  - 4 KB task stack (else 8 KB)
					;  - no internal font 8x10
					;  - 16 KB cache buffer (else 32 KB)
					;  - 8 KB disk cache list (else 16 KB) 
					;  - 16 KB swap cache list (else 32 KB)
					;  - BYTE swap page counter (else WORD)
					;  - 32 KB max. memory (else 64 KB)
					;  - 16 KB disk read ahead (else 32 KB)

; ------------- System version

VER_MAJOR	EQU	0		; version - major
VER_MINOR	EQU	1		; version - minor (in hundredths)
VER_BUILD	EQU	1		; version - build
%ifdef MINI
VER_VARIANT	EQU	"M"		; version - variant MINI
%else
VER_VARIANT	EQU	"B"		; version - variant BIG
%endif

; ------------- Kernel setup

LINUX_INT	EQU	80h		; Linux system call vector
SYSTEM_INT	EQU	90h		; Litos system call vector

SYSTEM_ADDR	EQU	80000000h	; address of system area (at 2 GB)
					;   It must be multiple of 64KB due
					;   to portability between 32b/16b.
					;   It must be multiple of 128KB due
					;   to memory area blocks aligning.
					;   It must not be below 2 GB.
%if SYSTEM_ADDR < 80000000h
%error Invalid system address (hardcoded by PAGEDESC).
%endif

%ifdef MINI
SYSTEM_SIZE	EQU	10000000h	; size of system area (256 MB)
%else
SYSTEM_SIZE	EQU	0ffffffffh-SYSTEM_ADDR+1 ; size of system area (2 GB)
%endif

; SYSTEM_SIZE must not be greater than 2 GB due to hardcoded limitation
; in PAGEDESC and SWAPMAP (they uses 31-bit offset in system memory).

; ------------- Processor

%ifndef MINI
%define	SMP				; flag-enable symetric multiprocessors
%endif

%ifdef	SMP
CPU_MAX		EQU	32		; max. number of CPUs in SMP (max. 32)
%else
CPU_MAX		EQU	1		; maximal number of CPUs in UP
%endif

; -----------------------------------------------------------------------------
;                            Start of sections
; -----------------------------------------------------------------------------
; Kernel uses 3 main types of program sections:
;
; - initialized data
; - code section, initialization (16-bit part and 32-bit part)
; - code section, kernel (16-bit part and 32-bit part)
; - exception fixup sections
; - uninitialized data
;
; Initialized data must be accessible from both real mode and protected mode
; and therefore it must lie in first 64 KB memory area (together with 16-bit
; part of code section). Therefore it is the first section of the kernel, so
; "text" section must be used. Behind it lies code section (using "data") and
; then exception services ("fix", "exc") and uninitialized data "bss".
; -----------------------------------------------------------------------------

; ------------- 16-bit code section

%macro		CODE16_SECTION	0
SECTION		.text align=16 progbits
BITS		16
%endmacro

		CODE16_SECTION
		org	80001000h	; start address of the system
KernelStart:				; start of kernel (PAGE aligned)
Code16Start:				; start of 16-bit code section

; ------------- 16-bit data section

%macro		DATA16_SECTION	0
SECTION		.data align=16 progbits follows=.text
BITS		16
%endmacro

		DATA16_SECTION
Data16Start:				; start of 16-bit data section

; ------------- 32-bit code section

%macro		CODE_SECTION	0
SECTION		.text32 align=16 progbits follows=.data
BITS		32
%endmacro

		CODE_SECTION
CodeStart:				; start of 32-bit code section

; ------------- 32-bit data section

%macro		DATA_SECTION	0
SECTION		.data32 align=16 progbits follows=.text32
BITS		32
%endmacro

		DATA_SECTION
DataStart:				; start of 32-bit data section

; ------------- Constant data section

%macro		CONST_SECTION	0
SECTION		.const align=16 progbits follows=.data32
BITS		32
%endmacro

		CONST_SECTION
ConstStart:				; start of constant data section

; ------------- Exception code section - fixup code

%macro		FIX_SECTION	0
SECTION		.fix align=16 progbits follows=.const
BITS		32
%endmacro

		FIX_SECTION
FixStart:				; start of exception fixup section

; ------------- Exception table section - exception address

%macro		EXC_SECTION	0
SECTION		.exc align=16 progbits follows=.fix
BITS		32
%endmacro

		EXC_SECTION
ExcStart:				; start of exception table section

; ------------- Exception table section - fixup address

%macro		EXC2_SECTION	0
SECTION		.exc2 align=16 progbits follows=.exc
BITS		32
%endmacro

		EXC2_SECTION
Exc2Start:				; start of exception table section

; ------------- 16-bit uninitialized section (below 1 MB)
; It is zeroed after kernel start.

%macro		BSS16_SECTION	0
SECTION		.bss align=16 nobits follows=.exc2
BITS		32
%endmacro

		BSS16_SECTION
BSS16Start:				; start of uninitialized data

; ------------- 32-bit uninitialized section (above 1 MB)
; It is zeroed after 32-mode start.

%macro		BSS_SECTION	0
SECTION		.bss32 nobits vstart=80100000h
BITS		32
%endmacro

		BSS_SECTION
KernelHighStart:
BSSStart:

; -----------------------------------------------------------------------------
;         Start of system at 0000:1000  (jump here from BOOT\BOOT.ASM)
; -----------------------------------------------------------------------------

		CODE16_SECTION

Start:		jmp	near Start16	; init real mode (-> INIT\INIT.ASM)
		align	4

; ------------- Identification (offset 4 from the begin)

Ident:		dw	8A26h		; identification word
Variant:	db	VER_VARIANT	; version - variant
		db	0		; version - variant 2 (reserved)

; ------------- System version (in form: version X.XX)

Version:				; version as DWORD
VersionBuild:	dw	VER_BUILD	; version - build number
VersionMinor:	db	VER_MINOR	; version - minor (in hundredths)
VersionMajor:	db	VER_MAJOR	; version - major

; ------------- Configuration

ConfigCon:	db	B0		; console configuration
					;	B0: 1=output to display
					;	B1: 1=output to COM1
					;	B2: 1=output to COM2
					;	B3: 1=output to COM3
					;	B4: 1=output to COM4


; ------------- Checked A20 address (it must lay on first 1 MB of RAM)

		align	4, db 0
CheckA20Addr:	dd	0		; checked A20 address

; -----------------------------------------------------------------------------
;                        Includes - definitions
; -----------------------------------------------------------------------------

; ------------- Utilities

%include	"INCLUDE\UTIL\SPINLOCK.INC"	; spin-lock
%include	"INCLUDE\UTIL\LIST.INC"		; list
%include	"INCLUDE\UTIL\HASH.INC"		; hash list
%include	"INCLUDE\UTIL\TREE.INC"		; tree list
%include	"INCLUDE\UTIL\TEXT.INC"		; text string
%include	"INCLUDE\UTIL\TEXTFORM.INC"	; format text string
%include	"INCLUDE\UTIL\BUFFER.INC"	; buffer array
%include	"INCLUDE\UTIL\RBTREE.INC"	; red-black balanced tree
%include	"INCLUDE\UTIL\CALENDAR.INC"	; calendar, date and time
%include	"INCLUDE\UTIL\CHARSET.INC"	; character sets
%include	"INCLUDE\UTIL\RANDOM.INC"	; random generator
%include	"INCLUDE\UTIL\UNICHAR.INC"	; Unicode character type
%include	"INCLUDE\UTIL\LANG.INC"		; languages and nationality

%include	"INCLUDE\CONSTANT.INC"		; constants
%include	"INCLUDE\ERRORS.INC"		; error codes
%include	"INCLUDE\EXCEPT.INC"		; exceptions
%include	"INCLUDE\CPU.INC"		; processor
%include	"INCLUDE\TRAPS.INC"		; traps
%include	"INCLUDE\PAGE.INC"		; memory pages
%include	"INCLUDE\CROSS.INC"		; cross list
%include	"INCLUDE\ALARM.INC"		; alarm
%include	"INCLUDE\PGCACHE.INC"		; page cache
%include	"INCLUDE\VIRTMEM.INC"		; virtual memory

; ------------- Kernel

%include	"INCLUDE\KERNEL\TASKLOCK.INC"	; task lock
%include	"INCLUDE\KERNEL\SEMAPHOR.INC"	; semaphore and mutex
%include	"INCLUDE\KERNEL\USER.INC"	; user
%include	"INCLUDE\KERNEL\PROCESS.INC"	; process
%include	"INCLUDE\KERNEL\TASK.INC"	; task
%include	"INCLUDE\SWAP.INC"		; swap cache
%include	"INCLUDE\GLDT.INC"		; global/local descriptor
%include	"INCLUDE\IDT.INC"		; interrupt descriptor
%include	"INCLUDE\SIGNAL.INC"		; signals

; ------------- Drivers

%include	"INCLUDE\DRIVERS\DEVICE.INC"	; general device
%include	"INCLUDE\DRIVERS\SYSTEM\DMA.INC"; DMA driver
%include	"INCLUDE\DRIVERS\SYSTEM\IRQ.INC"; interrupt driver
%include	"INCLUDE\DRIVERS\INT.INC"	; device interrupt service
%include	"INCLUDE\DRIVERS\SYSTEM\SYS.INC"; system kernel device
%include	"INCLUDE\DRIVERS\SYSTEM\CMOS.INC"; CMOS memory
%include	"INCLUDE\DRIVERS\SYSTEM\TIMER.INC"; Programmable interval timer
%include	"INCLUDE\DRIVERS\SYSTEM\CLOCK.INC"; Wall clock

%include	"INCLUDE\DRIVERS\DISPLAY.INC"	; general display
%include	"INCLUDE\DRIVERS\KEYBOARD.INC"	; keyboard driver
%include	"INCLUDE\DRIVERS\CONSOLE.INC"	; console
%include	"INCLUDE\DRIVERS\VT102.INC"	; VT102 terminal emulator
%include	"INCLUDE\DRIVERS\DATADEV.INC"	; data device
%include	"INCLUDE\DRIVERS\CACHE.INC"	; disk cache
%include	"INCLUDE\DRIVERS\DISK.INC"	; block device
%include	"INCLUDE\DRIVERS\FLOPPY.INC"	; floppy disk

; ------------- Files

%include	"INCLUDE\FILES\FILESYS.INC"	; file system
%include	"INCLUDE\FILES\FILE.INC"	; file

; -----------------------------------------------------------------------------
;             16-bit uninitialized buffers (it must be PAGE aligned!)
; -----------------------------------------------------------------------------

		BSS16_SECTION

; ------------- Memory bit map of available 4 KB pages (size 64 KB or 8 KB)
; This bit map will be later (after initialization) freed as free DMA memory.

		align	PAGE_SIZE, resb 1; align to page size (already aligned)
PageMemMap:	resb	SYSTEM_SIZE/PAGE_SIZE/8 ; memory bit map of free pages
		align	PAGE_SIZE,resb 1 ; align to page size (req. SysMemInit)

; -----------------------------------------------------------------------------
;                Uninitialized buffers (it must be PAGE aligned!)
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- Kernel page directory (4 KB, must be 4 KB aligned)

		align	PAGE_SIZE,resb 1; align to page size (already aligned)
PageDir:	resb	PAGE_SIZE	; page directory (holds 1024 PDEs)

; ------------- Empty zero page (4 KB, must be 4 KB aligned)

		align	PAGE_SIZE,resb 1; align to page size (already aligned)
CharSetNone:				; invalid character set table
CP0ToUniTab:				; translation from ASCII to Unicode
AlphaHashNone:				; none alphabetic hash table
PageEmpty:	resb	PAGE_SIZE	; empty zero page (contains zeroes)

; ------------- System global descriptor table (size 64 KB or 4 KB)

		align	PAGE_SIZE,resb 1; align to page size (already aligned)
SystemGDTTab:	resb	SYSTEM_GDT_NUM*GLDT_size ; system GDT
SystemGDTTabEnd:
TaskLDTTab	EQU	SystemGDTTab+TASK_GDT_LDT*GLDT_size ; task LDT descr.
TaskTSSTab:	EQU	SystemGDTTab+TASK_GDT_TSS*GLDT_size ; task TSS descr.

; ------------- System interrupt descriptor table (6 KB or 2 KB)
; Correction of Intel Pentium F00F bug: see InitPDE in INIT\INIT_32.ASM.

		align	PAGE_SIZE,resb 1; align to page size (already aligned)
		resb	PAGE_SIZE - 7*8	; 56 bytes before end of page
SystemIDTTab:	resb	256*8		; system interrupt descriptor table

; -----------------------------------------------------------------------------
;                                Includes - code
; -----------------------------------------------------------------------------

; ------------- Initialization in real mode

%include	"INIT\INIT.ASM"		; initialization, real mode
%include	"INIT\INIT_DBG.ASM"	; initialization, debug
%include	"INIT\INIT_GRA.ASM"	; initialize graphics card
%include	"INIT\INIT_HD.ASM"	; initialize hard drives
%include	"INIT\INIT_MEM.ASM"	; initialize memory map
%include	"INIT\INIT_APM.ASM"	; initialize advanced power management
%include	"INIT\INIT_A20.ASM"	; initialize A20 address line

; ------------- Initialization in protected mode

%include	"INIT\INIT_32.ASM"	; initialization, protected mode
%include	"INIT\INIT_CPU.ASM"	; initialize CPU
%include	"INIT\INIT_SYS.ASM"	; initialize kernel

; ------------- Utilities

%include	"UTIL\LIST.ASM"		; list
%include	"UTIL\HASH.ASM"		; hash list
%include	"UTIL\TREE.ASM"		; tree list
%include	"UTIL\RBTREE.ASM"	; red-black balanced tree
%include	"UTIL\RANDOM.ASM"	; random generator
%include	"UTIL\BUFFER.ASM"	; buffer array
%include	"UTIL\TEXT.ASM"		; text string
%include	"UTIL\TEXTFORM.ASM"	; format text string
%include	"UTIL\CALENDAR.ASM"	; calendar
%include	"UTIL\CHARSET.ASM"	; character sets
%include	"UTIL\UNICHAR.ASM"	; Unicode character type
%include	"UTIL\LANG.ASM"		; languages and nationality

; ------------- Kernel

%include	"KERNEL\TASKLOCK.ASM"	; task lock
%include	"KERNEL\SEMAPHOR.ASM"	; semaphore and mutex
%include	"KERNEL\SYSMEM.ASM"	; system memory allocator
%include	"KERNEL\PAGE.ASM"	; memory pages
%include	"KERNEL\SWAP.ASM"	; swap cache
%include	"KERNEL\VIRTMEM.ASM"	; virtual memory
%include	"KERNEL\PROCESS.ASM"	; process
%include	"KERNEL\CPU.ASM"	; processor
%include	"KERNEL\USER.ASM"	; user
%include	"KERNEL\TASK.ASM"	; task
%include	"KERNEL\TRAPS.ASM"	; traps
%include	"KERNEL\SYSTEM.ASM"	; system calls
%include	"KERNEL\LINUX.ASM"	; Linux system calls
%include	"KERNEL\TIME.ASM"	; time
%include	"KERNEL\ALARM.ASM"	; alarm
%include	"KERNEL\SCHEDULE.ASM"	; scheduler
%include	"KERNEL\SIGNAL.ASM"	; signals
%include	"KERNEL\FILE.ASM"	; file

; ------------- Drivers

%include	"DRIVERS\DEVICE.ASM"		; general device
%include	"DRIVERS\SYSTEM\DMA.ASM"	; DMA controller 8237A
%include	"DRIVERS\SYSTEM\IRQ.ASM"	; Interrupt controller 8259A
%include	"DRIVERS\INT.ASM"		; device interrupt service
%include	"DRIVERS\SYSTEM\SYS.ASM"	; system kernel device
%include	"DRIVERS\SYSTEM\CMOS.ASM"	; CMOS memory
%include	"DRIVERS\SYSTEM\TIMER.ASM"	; Programmable interval timer
%include	"DRIVERS\SYSTEM\CLOCK.ASM"	; Wall clock


%include	"DRIVERS\FONT\FONT.ASM"		; fonts
%include	"DRIVERS\FONT\F08.ASM"		; font 8x8
%include	"DRIVERS\FONT\F10.ASM"		; font 8x10
%include	"DRIVERS\FONT\F14.ASM"		; font 8x14

%include	"DRIVERS\VIDEO\DISPLAY.ASM"	; general display
%include	"DRIVERS\VIDEO\TXT.ASM"		; TXT display
%include	"DRIVERS\VIDEO\HGC.ASM"		; HGC display (Hercules)
%include	"DRIVERS\VIDEO\CGA.ASM"		; CGA display
%include	"DRIVERS\VIDEO\EGA.ASM"		; EGA display
%include	"DRIVERS\VIDEO\VGA.ASM"		; VGA display
%include	"DRIVERS\VIDEO\SVGA.ASM"	; SVGA display
%include	"DRIVERS\KEYBOARD.ASM"		; keyboard driver
%include	"DRIVERS\CONSOLE.ASM"		; console
%include	"DRIVERS\VT102.ASM"		; VT102 terminal emulator
%include	"DRIVERS\DATADEV.ASM"		; data device
%include	"DRIVERS\CACHE.ASM"		; disk cache
%include	"DRIVERS\DISK.ASM"		; block device
%include	"DRIVERS\FLOPPY.ASM"		; floppy disk

; -----------------------------------------------------------------------------
;                            End of sections
; -----------------------------------------------------------------------------

		CODE16_SECTION
		align	16, db 0
Code16End:				; end of 16-bit code section

		DATA16_SECTION
		align	16, db 0
Data16End:				; end of 16-bit data section

		CODE_SECTION
		align	16, db 0
CodeEnd:				; end of 32-bit code section

		DATA_SECTION
		align	16, db 0
DataEnd:				; end of 32-bit data section

		CONST_SECTION
		align	16, db 0
ConstEnd:				; end of constant data section

		FIX_SECTION
		align	16, db 0
FixEnd:					; end of exception fixup section

		EXC_SECTION
		align	16, db 0
ExcEnd:					; end of exception fixup section

		EXC2_SECTION
		align	16, db 0
Exc2End:				; end of exception table section

		BSS16_SECTION
		align	PAGE_SIZE, resb 1
BSS16End:				; end of uninitial. data (PAGE align.)
KernelEnd:				; end of kernel (PAGE aligned)

		BSS_SECTION
		align	PAGE_SIZE, resb 1
BSSEnd:					; end of uninitial. data (PAGE align.)
KernelHighEnd:				; high end of kernel (PAGE aligned)
