; =============================================================================
;
;                               Litos - Swap cache
;
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                          Lock/unlock swap area
; -----------------------------------------------------------------------------
; NOTES:	Use macro SWAPLOCK to lock, SWAPUNLOCK to unlock.
; -----------------------------------------------------------------------------

; ------------- Macro - lock swap area

%macro		SWAPLOCK 0
		LOCK_Lock SwapAreaLock	; lock swap area
%endmacro

; ------------- Macro - unlock swap area

%macro		SWAPUNLOCK 0
		LOCK_Unlock SwapAreaLock ; unlock swap area
%endmacro

; -----------------------------------------------------------------------------
;                       Initialize swap area tables
; -----------------------------------------------------------------------------

; ------------- Prepare registers

SwapCacheInit:	mov	eax,SwapArea	; EBX <- first swap area
		mov	ebx,SwapAreaUnList ; EBX <- list of inactive swap areas
		mov	ecx,SWAPAREAS	; ECX <- number of swap areas
		xor	edx,edx		; EDX <- area index
		mov	esi,SwapAreaAddr ; ESI <- pointer to first area

; ------------- Initialize one swap area

SwapCacheInit2:	mov	[eax+SWA_Index],dl ; store area index
		mov	[esi],eax	; store pointer to area
		call	ListLast	; add swap area into inactive list

; ------------- Next swap area

		inc	edx		; EDX <- increase area index
		add	esi,byte 4	; ESI <- pointer to next area
		add	eax,byte SWAPAREA_size ; EAX <- next area
		loop	SwapCacheInit2	; next area
		ret

; -----------------------------------------------------------------------------
;                          Read from swap device
; -----------------------------------------------------------------------------
; INPUT:	EAX = page index (0=swap header)
; 		EBX = swap device private data
; 		ECX = number of pages
; 		EDX = list of pages (pointer to list of page addresses)
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

;SwapDevRead:
;		ret

; -----------------------------------------------------------------------------
;                            Write to swap device
; -----------------------------------------------------------------------------
; INPUT:	EAX = page index (0=swap header)
; 		EBX = swap device private data
; 		ECX = number of pages
; 		EDX = list of pages (pointer to list of page addresses)
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

;SwapDevWrite:
;		ret

; -----------------------------------------------------------------------------
;                           Resort swap areas
; -----------------------------------------------------------------------------

; ------------- Push registers

SwapAreaSort:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Lock swap area

		pushf			; push flags
		cli			; disable interrupts
		SWAPLOCK		; lock swap area

; ------------- Prepare last area (-> EBX)

		mov	edx,SwapAreaList ; EDX <- swap area list head
		mov	ebx,[edx+LIST_Prev] ; EBX <- last area
		cmp	ebx,edx		; is swap area list empty?
		je	SwapAreaSort8	; swap area list is empty

; ------------- Prepare current area (-> ECX) and previous area (-> EBX)

SwapAreaSort2:	mov	ecx,ebx		; ECX <- current area
		mov	ebx,[ecx+LIST_Prev] ; EBX <- previous area
		cmp	ebx,edx		; start of list?
		je	SwapAreaSort8	; start of list, stop sorting

; ------------- Compare priority of current (ECX) and previous (EBX) area

SwapAreaSort4:	mov	ax,[ecx+SWA_Priority] ; AX <- priority of current area
		cmp	ax,[ebx+SWA_Priority] ; compare with previous area
		jbe	SwapAreaSort2	; priority is OK, get next entry

; ------------- Exchange areas

		call	ListXchgNext	; exchange area EBX with next area

; ------------- Move to next area

		cmp	edx,[ecx+LIST_Next] ; any next area?
		je	SwapAreaSort2	; no next area, continue to previous
		mov	ebx,[ecx+LIST_Next] ; EBX <- next area
		xchg	ebx,ecx		; correct area order
		jmp	short SwapAreaSort4 ; sort next area

; ------------- Unlock swap area

SwapAreaSort8:	SWAPUNLOCK		; unlock swap area
		popf			; pop flags (and enable interrupts)

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                             Activate swap area
; -----------------------------------------------------------------------------
; INPUT:	CX = priority (signed, -32K to +32K, 0 = normal priority)
;		EDX = pathname to file or data device
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; TODO: create private filename buffer
; TODO: set flags in ECX for FileOpen.
; TODO: File must be opened private, check other instances.

; ------------- Check if current user has admin rights

SwapAreaOn:	call	CheckAdmin	; has current user admin rights?
		jz	short SwapAreaOn1 ; user has not admin rights

; ------------- Fast check if another swap area is available

		LISTTEST SwapAreaUnList	; next area?
SwapAreaOn1:	jz	near SwapAreaOn99 ; no next area available

; ------------- Push registers

		pusha			; push all registers
		mov	edi,ecx		; EDI <- priority

; ------------- Open file or data device (-> EBX, EBP)

; TODO !!!!!!! (copy from user to system buffer, check if it is not opened)

		xor	ecx,ecx		; ECX <- flags
		call	FileOpen	; open file or data device
		jc	near SwapAreaOn98 ; file or data device opened OK
		mov	ebp,eax		; EBP <- file handle
		xchg	eax,ebx		; EBX <- file handle

; ------------- Allocate memory block to read first swap page (-> ESI)

		mov	eax,PAGE_SIZE	; EAX <- size of memory block
		mov	ecx,eax		; ECX <- size of first page
		call	SysMemAllocWait	; allocate memory block
		jc	near SwapAreaOn97 ; memory error
		xchg	eax,esi		; ESI <- memory block address

; ------------- Read first page slot (into ESI memory block)

		xor	eax,eax		; EAX <- 0, data offset LOW
		xor	edx,edx		; EDX <- 0, data offset HIGH
		call	FileRead	; read first page slot
		jc	short SwapAreaOn2 ; data read error

; ------------- Check number of bytes

		cmp	eax,ecx		; check number of bytes
		jb	short SwapAreaOn2 ; read error

; ------------- Get swap file size in pages (-> EDX)

		call	FileGetSize	; get file size (-> EDX:EAX)
SwapAreaOn2:	jc	near SwapAreaOn96 ; error
		shrd	eax,edx,PAGE_SHIFT ; EAX <- number of pages
		shr	edx,PAGE_SHIFT	; test overflow
		jz	SwapAreaOn3	; no overflow
		xor	eax,eax		; EAX <- 0
		dec	eax		; EAX <- maximum number of pages
SwapAreaOn3:	xchg	eax,edx		; EDX <- number of pages

; ------------- Lock swap area

		pushf			; push flags
		cli			; disable interrupts
		SWAPLOCK		; lock swap area

; ------------- Get unused swap area (-> EBX)

		mov	ebx,[SwapAreaUnList] ; EBX <- first unused swap area
		LISTTEST ebx		; is this entry valid?
		je	near SwapAreaOn95 ; no free swap area
		call	ListDelEBX	; delete swap area from the list

; ------------- Unlock swap area

		SWAPUNLOCK		; unlock swap area
		popf			; pop flags (and enable interrupts)

; ------------- Initialize some entries of swap area descriptor

		mov	ecx,edi		; ECX <- priority
		mov	[ebx+SWA_Priority],cx ; set priority
		mov	[ebx+SWA_FileHandle],ebp ; set swap device handle

; ------------- Compare swap header magic version 1

		push	esi		; push ESI
		add	esi,SWH_Magic	; ESI <- magic
		mov	edi,SwapHeadMagic1 ; EDI <- swap header magic version 1
		xor	ecx,ecx		; ECX <- 0
		mov	cl,SWAPMAGIC_SIZE ; ECX <- size of magic text
		repe	cmpsb		; compare magic text
		pop	esi		; pop ESI
		jne	short SwapAreaOn4 ; no header version 1

; ------------- Initialize swap header version 1

		and	dword [esi+SWH_BadPageNum],0 ; no bad pages
		jmp	short SwapAreaOn5

; ------------- Compare swap header magic version 2

SwapAreaOn4:	push	esi		; push ESI
		add	esi,SWH_Magic	; ESI <- magic text
		mov	edi,SwapHeadMagic2 ; EDI <- swap header magic version 2
		mov	cl,SWAPMAGIC_SIZE ; ECX <- size of magic text
		repe	cmpsb		; compare magic text
		pop	esi		; pop ESI
		jne	short SwapAreaOn42 ; not header version 2, error

; ------------- Check swap header version

		cmp	dword [esi+SWH_Version],byte 1 ; must be version 1
SwapAreaOn42:	jne	near SwapAreaOn94 ; unsupported version

; ------------- Limit number of page slots (-> EDX)

		mov	eax,[esi+SWH_MaxPages] ; EAX <- number of page slots
		cmp	edx,eax		; check number of page slots
		jbe	SwapAreaOn5	; number of pages is OK
		xchg	eax,edx		; EDX <- limit number of page slots

; ------------- Check (and limit) number of page slots

SwapAreaOn5:	cmp	edx,byte 2	; check minimal number of page slots
		jb	short SwapAreaOn53 ; invalid number of page slots
		mov	eax,SWE_SLOT_MAX ; EAX <- naximal number of page slots
		cmp	edx,eax		; check number of page slots
		jbe	SwapAreaOn52	; number of page slots is OK
		xchg	eax,edx		; EDX <- limit number of page slots
SwapAreaOn52:	mov	[ebx+SWA_TotalPages],edx ; set total page slots
		mov	[ebx+SWA_FreePages],edx ; set number of free pages
		cmp	edx,[esi+SWH_BadPageNum] ; check number of bad pages
		jbe	near SwapAreaOn94 ; invalid swap header

; ------------- Allocate swap page slot map (-> EDI, max. 128 KB)

		xchg	eax,edx		; EAX <- total number of page slots
		add	eax,SWM_CLUST_NUM-1 ; round up to whole cluster
		shr	eax,SWM_CLUST_SHIFT ; EAX <- number of clusters
		mov	[ebx+SWA_ClustNum],eax ; store number of clusters
		lea	ecx,[eax*(SWAPMAP_size/4)] ; ECX <- size in DWORDs
		shl	eax,SWAPMAP_SIZEBIT ; EAX <- size of page slot map
		jz	near SwapAreaOn94 ; invalid map size
		call	SysMemAllocWait	; allocate swap page slot map
SwapAreaOn53:	jc	near SwapAreaOn94 ; memory error
		mov	[ebx+SWA_Map],eax ; store pointer to page slot map
		xchg	eax,edi		; EDI <- page slot map

; ------------- Clear page slot map (to enable to release tables on error)

		push	edi		; push EDI
		xor	eax,eax		; EAX <- 0
		rep	stosd		; clear slot map
		pop	edi		; pop EDI
		mov	[ebx+SWA_LastClust],eax ; clear last searched cluster
		mov	[ebx+SWA_NextPage],eax ; clear next page index

; ------------- Create cluster tables

		mov	edx,[ebx+SWA_ClustNum] ; EDX <- number of clusters
SwapAreaOn54:	mov	eax,SWM_CLUST_SIZE ; EAX <- cluster size
		call	SysMemAllocWait ; allocate cluster table
		jc	near SwapAreaOn92 ; memory error
		stosd			; store pointer to swap cluster

; ------------- Clear cluster table

		push	edi		; push EDI
		xchg	eax,edi		; EDI <- cluster table
		xor	eax,eax		; EAX <- 0
		mov	ecx,SWM_CLUST_SIZE/4 ; ECX <- cluster size in DWORDs
		rep	stosd		; clear cluster table
		pop	edi		; pop EDI
		mov	eax,SWM_CLUST_NUM ; EAX <- number of slots
		stosd			; set number of free slots in clusters
		dec	edx		; number of clusters
		jnz	SwapAreaOn54	; prepare next cluster table

; ------------- Initialize unused slots in last cluster (mark as bad slots)

		xchg	eax,ecx		; ECX <- number of slots
		dec	ecx		; ECX <- number of slots - 1
		mov	eax,[ebx+SWA_TotalPages] ; EAX <- number of total pages
		and	eax,ecx		; mask slots in last cluster
		jz	SwapAreaOn6	; no remaining slots
		mov	[edi-4],eax	; set slots in last cluster
		inc	ecx		; ECX <- number of slots
		sub	ecx,eax		; ECX <- remainging slots in table
		mov	edi,[edi-8]	; EDI <- last cluster table
		lea	edi,[edi+eax*4]	; EDI <- end of valid data
		xor	eax,eax		; AX <- 0
		inc	eax		; AX <- 1, bad page flag
		rep	stosd		; set "bad page" flags

; ------------- Mark first slot as reserved (=swap header)

SwapAreaOn6:	mov	edx,[ebx+SWA_Map] ; EDX <- swap page slot map
		dec	dword [edx+SWM_Free] ; decrease free slots
		mov	edi,[edx+SWM_Cluster] ; EDI <- first cluster
		xor	eax,eax		; EAX <- 0
		inc	eax		; EAX <- 1, bad page flag
		stosd			; mark first page slot as reserved

; ------------- Prepare number of bad pages (-> ECX)

		mov	ecx,[esi+SWH_BadPageNum] ; ECX <- number of pad pages
		cmp	ecx,SWAPBADPAGES; check number of bad pages
		jbe	SwapAreaOn61	; number of bad pages is OK
		mov	ecx,SWAPBADPAGES ; ECX <- limit number of bad pages
SwapAreaOn61:	jecxz	SwapAreaOn7	; no bad pages

; ------------- Get index of next bad page (-> EAX)

		push	esi		; push ESI
		add	esi,SWH_BadPages ; ESI <- table od bad pages
SwapAreaOn62:	lodsd			; EAX <- index of bad page
		cmp	eax,[ebx+SWA_TotalPages] ; check index of bad page
		jae	SwapAreaOn66	; invalid bad page

; ------------- Pointer to swap page slot map (-> EDX)

		mov	edx,eax		; EDX <- index of bad page
		shr	edx,SWM_CLUST_SHIFT ; EDX <- cluster index
		shl	edx,SWAPMAP_SIZEBIT ; EDX <- offset in cluster map
		add	edx,[ebx+SWA_Map] ; EDX <- pointer to swap map

; ------------- Pointer to slot counter

		and	eax,SWM_CLUST_MASK ; EAX <- mask slot index
		mov	edi,[edx+SWM_Cluster] ; EDI <- pointer to slot map

; ------------- Mark bad page (if it is still free)

		cmp	dword [edi+eax*4],byte SWAPMAP_FREE ; is slot free?
		jne	SwapAreaOn66	; slot is not free
		inc	dword [edi+eax*4] ; mark slot as bad (used)
		dec	dword [edx+SWM_Free] ; decrease counter of free slots

; ------------- Next bad page

SwapAreaOn66:	loop	SwapAreaOn62	; next bad page
		pop	esi		; pop ESI

; ------------- Check number of free pages (here is ECX = 0)

SwapAreaOn7:	cmp	[ebx+SWA_FreePages],ecx ; free page slots?
		je	SwapAreaOn92	; no free page slot

; ------------- Lock swap area

		pushf			; push flags
		cli			; disable interrupts
		SWAPLOCK		; lock swap area

; ------------- Add area into used area list

		xchg	eax,ebx		; EAX <- area
		mov	ebx,SwapAreaList ; EBX <- list of used areas
		call	ListLast	; add entry into list of areas
		or	byte [eax+SWA_Flags],SWA_WRITE ; write is enabled

; ------------- Increase free slots

		mov	eax,[eax+SWA_FreePages] ; EAX <- number of free slots
		add	[SwapPagesFree],eax ; increase number of free slots

; ------------- Unlock swap area

		SWAPUNLOCK		; unlock swap area
		popf			; pop flags (and enable interrupts)

; ------------- Free memory block ESI

		xchg	eax,esi		; EAX <- memory block
		call	SysMemFree	; free memory block

; ------------- Pop registers

		popa			; pop all registers

; ------------- Resort swap areas

		call	SwapAreaSort	; resort swap areas
		clc			; clear error flag
		ret

; ------------- ERROR: Free cluster map

SwapAreaOn92:	push	esi		; push ESI
		mov	esi,[ebx+SWA_Map] ; ESI <- page slot map
		push	esi		; push ESI (page slot map)
		mov	ecx,[ebx+SWA_ClustNum] ; ECX <- number of clusters
SwapAreaOn93:	lodsd			; EAX <- cluster address
		call	SysMemFree	; free memory block
		lodsd			; skip number of slots
		loop	SwapAreaOn93	; free next cluster map
		pop	eax		; pop EAX (page slot map)
		pop	esi		; pop ESI
		call	SysMemFree	; free cluster map		

; ------------- ERROR: Lock swap area

SwapAreaOn94:	pushf			; push flags
		cli			; disable interrupts
		SWAPLOCK		; lock swap area

; ------------- ERROR: Free unused area EBX

		xchg	eax,ebx		; EAX <- area
		mov	ebx,SwapAreaUnList ; EBX <- list of unused areas
		call	ListAdd		; add entry into list of areas

; ------------- ERROR: Unlock swap area

SwapAreaOn95:	SWAPUNLOCK		; unlock swap area
		popf			; pop flags (and enable interrupts)

; ------------- ERROR: Free memory block ESI

SwapAreaOn96:	xchg	eax,esi		; EAX <- memory block
		call	SysMemFree	; free memory block

; ------------- ERROR: Close file or data device EBP

		mov	ebx,ebp		; EBX <- file handle
SwapAreaOn97:	call	FileClose	; close file

; ------------- ERROR: Pop registers

SwapAreaOn98:	popa			; pop all registers
SwapAreaOn99:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                             Deactivate swap area
; -----------------------------------------------------------------------------
; INPUT:	EDX = pathname to partition or file
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Check if current user has admin rights

SwapAreaOff:	call	CheckAdmin	; has current user admin rights?
		jz	near SwapAreaOff99 ; user has not admin rights

; ------------- Push registers

		pusha			; push all registers

; ------------- Lock swap area

		pushf			; push flags
		cli			; disable interrupts
		SWAPLOCK		; lock swap area

; ------------- 

; TODO !!!! (copy string from user space to kernel buffer and find area)




		mov	ebx,[SwapAreaList+LIST_Next] ; EBX <- first area
		cmp	ebx,SwapAreaList ; valid area ?
		je	SwapAreaOff98

; ------------- Disable write to swap area

		btr	dword [ebx+SWA_Flags],SWA_WRITE_BIT ; disable write
		jnc	SwapAreaOff98	; area is already unmounting

; ------------- Decrease number of free slots

		mov	eax,[eax+SWA_FreePages] ; EAX <- number of free slots
		sub	[SwapPagesFree],eax ; decrease number of free slots




; TODO Load all pages into memory


; TODO Close file and move area into UnList


; TODO on error undo changes




; ------------- ERROR: Unlock swap area

SwapAreaOff98:	SWAPUNLOCK		; unlock swap area
		popf			; pop flags (and enable interrupts)

; ------------- ERROR: Pop registers

		popa			; pop all registers
SwapAreaOff99:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                  Allocate swap page slot in one swap area
; -----------------------------------------------------------------------------
; INPUT:	EBX = swap area
;		EBP = flag: 0=must be minimal cluster size, 1=all cluster sizes
; OUTPUT:	EAX = page slot index (if NC)
;		CY = error, no free page slot
; NOTES:	Swap area must be locked.
;		It marks page slot as engaged (set it to 1).
; -----------------------------------------------------------------------------

; ------------- Check if this area has some free page slot

SwapAreaAlloc:	cmp	dword [ebx+SWA_FreePages],byte 0 ; any free page slots?
		je	SwapAreaAlloc1	; this area has no free page slot

; ------------- Check if this area can be written

		test	byte [ebx+SWA_Flags],SWA_WRITE ; is write enabled?
		jnz	SwapAreaAlloc2	; write is enabled

; ------------- Error, no free page slot (or invaid swap area)

SwapAreaAlloc1:	stc			; set error flag
		ret

; ------------- Push registers

SwapAreaAlloc2:	push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare pointer to swap page slot map (-> EDX)

		mov	edx,[ebx+SWA_LastClust] ; EDX <- last cluster index
		mov	edi,[ebx+SWA_Map] ; EDI <- swap page map
		lea	edx,[edi+edx*SWAPMAP_size] ; EDX <- swap page map

; ------------- Find free page slot from last position

		mov	esi,[ebx+SWA_NextPage] ; ESI <- next page index
		mov	edi,[edx+SWM_Cluster] ; EDI <- swap cluster address
		lea	edi,[edi+esi*4]	; EDI <- next page slot address
		xor	eax,eax		; EAX <- 0
		mov	ecx,SWM_CLUST_NUM ; ECX <- number of slots
		sub	ecx,esi		; ECX <- remaining pages
		jz	SwapAreaAlloc3	; end of cluster
		repne	scasd		; find free page slot
		je	SwapAreaAlloc8	; found free page slot

; ------------- Push registers 2

SwapAreaAlloc3:	push	ebx		; push EBX

; ------------- Prepare to find next cluster with enough free slots

		mov	ecx,[ebx+SWA_ClustNum] ; ECX <- number of clusters
		xor	ebx,ebx		; EBX <- index of tested cluster
		xor	esi,esi		; ESI <- free pages of best cluster
		mov	edi,[ebx+SWA_Map] ; EDI <- swap page map

; ------------- Find best cluster

SwapAreaAlloc4:	mov	eax,[edi+ebx*SWAPMAP_size+SWM_Free] ; EAX <- free slots
		or	ebp,ebp		; use all cluster sizes?
		jnz	SwapAreaAlloc5	; use all cluster sizes
		cmp	eax,SWM_CLUST_NUM/4 ; check minimal cluster size
		jb	SwapAreaAlloc6	; cluster is not free enough
SwapAreaAlloc5:	cmp	esi,eax		; check free slots
		jae	SwapAreaAlloc6	; it is not better cluster
		xchg	eax,esi		; ESI <- new best free slots
		mov	edx,ebx		; EDX <- index of new best cluster
		cmp	esi,SWM_CLUST_NUM*4/5 ; is this cluster free enough?
		jae	SwapAreaAlloc7	; cluster is free enough (up to 80%)
SwapAreaAlloc6:	inc	ebx		; increase index of tested cluster
		loop	SwapAreaAlloc4	; test next cluster

; ------------- Pop registers 2

SwapAreaAlloc7:	pop	ebx		; pop EBX

; ------------- Check if cluster has enough free space

		or	esi,esi		; suitable cluster has been found?
		stc			; set error flag
		jz	short SwapAreaAlloc9 ; cluster not found

; ------------- Save new last searched cluster

		mov	[ebx+SWA_LastClust],edx ; index of last cluster
		lea	edx,[edi+edx*SWAPMAP_size] ; EDX <- best cluster

; ------------- Find first free page slot in cluster

		xor	eax,eax		; EAX <- 0
		mov	edi,[edx+SWM_Cluster] ; EDI <- swap cluster
		mov	ecx,SWM_CLUST_NUM ; ECX <- number of slots
		repne	scasd		; find free page slot

; ------------- Decrease counters of free slots

SwapAreaAlloc8:	inc	dword [edi-4]	; mark page slot as engaged
		dec	dword [edx+SWM_Free] ; decrease free swap pages
		dec	dword [ebx+SWA_FreePages] ; decrease free pages
		dec	dword [SwapPagesFree] ; decrease free pages

; ------------- Prepare page slot index (-> EAX)

		mov	eax,edi		; EAX <- swap counter address
		sub	eax,[edx+SWM_Cluster] ; EAX <- counter offset
		shr	eax,2		; EAX <- counter index
		mov	[ebx+SWA_NextPage],eax ; save new next page to search
		dec	eax		; EAX <- found page

; ------------- Add cluster offset

		sub	edx,[ebx+SWA_Map] ; EDX <- cluster offset
		shl	edx,SWM_CLUST_SHIFT-SWAPMAP_SIZEBIT ; cluster index
		add	eax,edx		; EAX <- page index

; ------------- Pop registers (here is NC)

SwapAreaAlloc9:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                          Allocate new swap page
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = swap entry (in PTE format)
;		CY = memory error
; NOTES:	It locks swap area. Slot pointer remains empty (=no PTE).
; -----------------------------------------------------------------------------

; ------------- Push registers

SwapPageAlloc:	push	ebx		; push EBX
		push	ebp		; push EBP
		pushf			; push flags
		cli			; disable interrupts

; ------------- Lock swap area

		SWAPLOCK		; lock swap area

; ------------- Check if some free page is available

		cmp	dword [SwapPagesFree],byte 0 ; check free pages
		je	short SwapPageAlloc9 ; error, no free page

; ------------- Try to search minimal cluster

		xor	ebp,ebp		; EBP <- 0, flag: use minimal size
		mov	ebx,SwapAreaList ; EBX <- swap area list
SwapPageAlloc2:	mov	ebx,[ebx+LIST_Next] ; EBX <- next area
		call	SwapAreaAlloc	; try to allocate new swap page
		jnc	SwapPageAlloc6	; page allocated OK
		cmp	dword [ebx+LIST_Next],SwapAreaList ; is it last area?
		jne	SwapPageAlloc2	; try next area

; ------------- Search whichever cluster

		inc	ebp		; EBP <- 1, flag: use all clusters
SwapPageAlloc4:	mov	ebx,[ebx+LIST_Next] ; EBX <- next area
		call	SwapAreaAlloc	; try to allocate new swap page
		jnc	SwapPageAlloc6	; page allocated OK
		cmp	dword [ebx+LIST_Next],SwapAreaList ; is it last area?
		jne	SwapPageAlloc4	; try next area
		jmp	short SwapPageAlloc9 ; error (no usable area)

; ------------- Prepare swap entry (in PTE format)

SwapPageAlloc6:	movzx	ebx,byte [ebx+SWA_Index] ; EBX <- area index
		shl	ebx,SWE_AREA_SHIFT ; EBX <- area index in PTE format
		shl	eax,SWE_SLOT_SHIFT ; EAX <- shift slot index
		or	eax,ebx		; EAX <- swap entry

; ------------- OK: Unlock swap area

SwapPageAlloc8:	SWAPUNLOCK		; unlock swap area

; ------------- OK: Pop registers

		popf			; pop flags (and enable interrupts)
		pop	ebp		; pop EBP
		pop	ebx		; pop EBX
		clc			; clear error flag
		ret

; ------------- ERROR: Unlock swap area

SwapPageAlloc9:	SWAPUNLOCK		; unlock swap area

; ------------- ERROR: Pop registers

		popf			; pop flags (and enable interrupts)
		pop	ebp		; pop EBP
		pop	ebx		; pop EBX
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                             Free swap page
; -----------------------------------------------------------------------------
; INPUT:	EAX = swap entry (in PTE format, it must be valid swap entry)
; NOTES:	It locks swap cache.
;		If it contains PTE array, it must be freed elsewhere.
; -----------------------------------------------------------------------------

; ------------- Push registers

SwapPageFree:	push	eax		; push EAX
		push	ebx		; push EBX
		push	edx		; push EDX
		pushf			; push flags
		cli			; disable interrupts

; ------------- Lock swap area

		SWAPLOCK		; lock swap area

; ------------- Prepare swap area (-> EBX)

		movzx	ebx,al		; EBX <- swap area index * 2
		mov	ebx,[SwapAreaAddr+ebx*(4/SWE_AREA_MUL)] ; EBX <- area

; ------------- Prepare page slot index (-> EAX)

		shr	eax,SWE_SLOT_SHIFT ; EAX <- page slot index

; ------------- Get page slot map (-> EDX)

		mov	edx,eax		; EDX <- page slot index
		shr	edx,SWM_CLUST_SHIFT ; EDX <- page slot map index
		shl	edx,SWAPMAP_SIZEBIT ; EDX <- page slot map offset
		add	edx,[ebx+SWA_Map] ; EDX <- page slot map

; ------------- Get slot pointer (-> EAX)

		and	eax,SWM_CLUST_MASK ; mask entry index
		shl	eax,2		; EAX <- offset in cluster
		add	eax,[edx+SWM_Cluster] ; EAX <- counter address

; ------------- Mark page slot as free

		and	dword [eax],byte 0 ; mark page slot as free

; ------------- Increase counters of free slots

		inc	dword [edx+SWM_Free] ; increase free swap pages
		inc	dword [ebx+SWA_FreePages] ; increase free pages
		inc	dword [SwapPagesFree] ; increase free pages

; ------------- Unlock swap area

		SWAPUNLOCK		; unlock swap area

; ------------- Pop registers

		popf			; pop flags (and enable interrupts)
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                         Set swap page slot entry
; -----------------------------------------------------------------------------
; INPUT:	EAX = swap entry (in PTE format, it must be valid swap entry)
;		ECX = new PTE (single PTE or PTE array, it must not be 0)
; NOTES:	It locks swap cache.
; -----------------------------------------------------------------------------

; ------------- Push registers

SwapPageSet:	push	eax		; push EAX
		push	ebx		; push EBX
		push	edx		; push EDX

; ------------- Prepare swap area (-> EBX)

		movzx	ebx,al		; EBX <- swap area index * 2
		mov	ebx,[SwapAreaAddr+ebx*(4/SWE_AREA_MUL)] ; EBX <- area

; ------------- Prepare page slot index (-> EAX)

		shr	eax,SWE_SLOT_SHIFT ; EAX <- page slot index

; ------------- Get page slot map (-> EDX)

		mov	edx,eax		; EDX <- page slot index
		shr	edx,SWM_CLUST_SHIFT ; EDX <- page slot map index
		shl	edx,SWAPMAP_SIZEBIT ; EDX <- page slot map offset
		add	edx,[ebx+SWA_Map] ; EDX <- page slot map

; ------------- Get slot pointer (-> EAX)

		and	eax,SWM_CLUST_MASK ; mask entry index
		shl	eax,2		; EAX <- offset in cluster
		add	eax,[edx+SWM_Cluster] ; EAX <- counter address

; ------------- Set page slot entry

		mov	dword [eax],ecx	; set new page slot entry

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                       Write page to swap cache
; -----------------------------------------------------------------------------
; INPUT:	EBX = page descriptor PAGEDESC
; OUTPUT:	CY = write error or page is written with another task
; -----------------------------------------------------------------------------

; ------------- Push registers

SwapPageWrite:	push	eax		; push EAX
		push	ebx		; push EBX
		push	edx		; push EDX




; ------------- Prepare swap area (-> EBX)

		movzx	ebx,al		; EBX <- swap area index * 2
		mov	ebx,[SwapAreaAddr+ebx*(4/SWE_AREA_MUL)] ; EBX <- area

; ------------- Prepare page slot index (-> EAX)

		shr	eax,SWE_SLOT_SHIFT ; EAX <- page slot index

; ------------- Get page slot map (-> EDX)

		mov	edx,eax		; EDX <- page slot index
		shr	edx,SWM_CLUST_SHIFT ; EDX <- page slot map index
		shl	edx,SWAPMAP_SIZEBIT ; EDX <- page slot map offset
		add	edx,[ebx+SWA_Map] ; EDX <- page slot map

; ------------- Get slot pointer (-> EAX)

		and	eax,SWM_CLUST_MASK ; mask entry index
		shl	eax,2		; EAX <- offset in cluster
		add	eax,[edx+SWM_Cluster] ; EAX <- counter address




; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Swap header magic

SwapHeadMagic1:	db	'SWAP-SPACE'	; swap header version 1 magic
SwapHeadMagic2:	db	'SWAPSPACE2'	; swap header version 2 magic

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Swap pages

		align	4, db 0
SwapPagesTotal:	dd	0		; total number of page slots
SwapPagesFree:	dd	0		; free page slots

; ------------- Swap area lock

		align	4, db 0
SwapAreaLock:	SPINLOCK		; swap area lock

; ------------- List of active and inactive swap areas

		align	8, db 0
SwapAreaList:	LISTHEAD		; list of active swap areas
SwapAreaUnList:	LISTHEAD		; list of inactive swap areas
SwapAreaNum:	dd	0		; number of active swap areas

; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- Swap areas (size aprox. 2 KB)

		align	8, resb 1
SwapArea:	resb	SWAPAREA_size*SWAPAREAS	; swap areas

; ------------- Table od addresses of swap areas (size 128 B)

		align	4, resb 1
SwapAreaAddr:	resd	SWAPAREA_size
