; =============================================================================
;
;                               Litos - Process
;
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                           Lock/unlock process list
; -----------------------------------------------------------------------------
; NOTES:	Use macro PROCLOCK to lock, PROCUNLOCK to unlock.
; -----------------------------------------------------------------------------

; ------------- Macro - lock process list

%macro		PROCLOCK 0
		LOCK_Lock ProcListLock	; lock process list
%endmacro

; ------------- Macro - unlock process list

%macro		PROCUNLOCK 0
		LOCK_Unlock ProcListLock ; unlock process list lock
%endmacro

; -----------------------------------------------------------------------------
;                           Link task to process
; -----------------------------------------------------------------------------
; INPUT:	EAX = task
;		EBX = process
; NOTES:	It locks process list.
; -----------------------------------------------------------------------------

; ------------- Push registers

ProcTaskLink:	push	eax		; push EAX
		push	ebx		; push EBX
		pushf			; push flags
		cli			; disable interrupts

; ------------- Lock process list

		PROCLOCK		; lock process list

; ------------- Link task with process

		inc	dword [ebx+PROC_TaskNum] ; increase number of tasks
		add	eax,TASK_ProcList ; EAX <- list of tasks in process
		mov	[eax-TASK_ProcList+TASK_Process],ebx ; save pointer
		add	ebx,byte PROC_TaskList ; list of tasks in process
		call	ListLast	; link task to task list of process

; ------------- Unlock process list

		PROCUNLOCK		; unlock process list

; ------------- Pop registers

		popf			; pop flags (and enable interrupts)
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                         Link task to root process
; -----------------------------------------------------------------------------
; INPUT:	EBX = task
; NOTES:	It locks process list.
; -----------------------------------------------------------------------------

ProcRootLink:	push	eax		; push EAX
		xchg	eax,ebx		; EAX <- task
		mov	ebx,RootProc	; EBX <- root process
		call	ProcTaskLink	; link task to process
		xchg	eax,ebx		; EBX <- task
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                          Clone current process
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = new child process (if NC, or EAX=0 if CY)
;		CY = memory error (EAX = 0)
; -----------------------------------------------------------------------------

; ------------- Push registers

ProcClone:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI
		pushf			; push flags
		cli			; disable interrupts

; ------------- Lock process list

		PROCLOCK		; lock process list

; ------------- Create new process descriptor (-> EBX)

		mov	eax,PROCESS_size ; EAX <- size of process descriptor
		call	SysMemAlloc	; allocate process descriptor
		jc	ProcClone9	; memory error (EAX = 0)
		xchg	eax,ebx		; EBX <- new process descriptor

; ------------- Copy process descriptor (-> ECX=0, EAX=parent process)

		CURRENT	esi		; ESI <- get current task
		mov	eax,[esi+TASK_Process] ; EAX <- current process
		mov	esi,eax		; ESI <- current process
		mov	edi,ebx		; EDI <- new process descriptor
		xor	ecx,ecx		; ECX <- 0
		mov	cl,(PROCESS_size+3)/4 ; ECX <- descriptor size
		rep	movsd		; copy process descriptor

; ------------- Initialize red-black tree head (list of memory regions)

		call	RBTreeInit	; initialize red-black tree head

; ------------- Push registers

		push	eax		; push EAX (current process)
		push	ebx		; push EBX (new process)

; ------------- Initialize list of tasks in process

		add	ebx,byte PROC_TaskList ; EBX <- list of tasks
		call	ListInit	; initialize list head

; ------------- Initialize mutex-lock

		add	ebx,byte PROC_VMMutex-PROC_TaskList ; EBX <- mutex lock
		call	MutexInit	; initialize mutex

; ------------- Initialize spin-lock

		LOCK_Init (ebx+PROC_VMLock-PROC_VMMutex); initialize spin-lock

; ------------- Initialize tree link to other processes

		add	ebx,byte PROC_ProcLink-PROC_VMMutex ; new tree link
		call	TreeInit	; initialize tree list entry

; ------------- Link process to parent process

		add	eax,byte PROC_ProcLink ; EAX <- current tree link
		xchg	eax,ebx		; EAX <- new link, EBX <- current link
		call	TreeAdd		; add new process to parent process

; ------------- Pop registers

		pop	ebx		; pop EBX (new process)
		pop	eax		; pop EAX (current process)

; ------------- Initialize other variables

		mov	[ebx+PROC_TaskNum],ecx ; clear number of tasks
		mov	[ebx+PROC_VMCache],ecx ; clear region cache
		mov	[ebx+PROC_Pages],ecx ; no page allocated

; ------------- Clone memory regions
; !!!! TODO



; ------------- Clone global page directory
; !!!! TODO



; ------------- Increase page list reference counters
; !!!! TODO



; ------------- OK: Unlock process list (here is EBX = new process)

ProcClone8:	xchg	eax,ebx		; EAX <- process descriptor
		PROCUNLOCK		; unlock process list

; ------------- ERROR: Pop registers

		popf			; pop flags (and enable interrupts)
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		clc			; clear error flag
		ret

; ------------- ERROR: Unlock process list (here is EAX = 0)

ProcClone9:	PROCUNLOCK		; unlock process list

; ------------- ERROR: Pop registers

		popf			; pop flags (and enable interrupts)
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Root process, it contains process list

		align	4, db 0
RootProc:	PROCESSINI UserRoot

;		RBTREE			; RB-tree root of memory regions
;		dd	0		; number of tasks
;		LISTHEAD		; list of tasks in process
;		TREEHEAD		; tree link to other processes
;		MUTEX			; virtual memory mutex-lock
;		SPINLOCK		; virtual memory spin-lock
;		LISTHEAD		; user's process list
;		dd	NULL		; page global directory (NULL=kernel)
;		dd	NULL		; cache, last used region (NULL=none)
;		dd	0		; counter of allocated pages
;		dd	NULL		; start address of the heap
;		dd	NULL		; address behind end of the heap
;		dd	UserRoot	; pointer to current user
;		dd	0		; current user's ID
;		dd	0		; process flags

; ------------- Process list lock

		align	4, db 0
ProcListLock:	SPINLOCK		; process list lock
