; =============================================================================
;
;                           Litos - Memory pages
;
; =============================================================================

; TODO: Use INVLPG instruction

		CODE_SECTION

; ------------- Macro - test if PTE (page table entry) is present
; %1 = pointer to PTE, output: NZ = page is present, ZY = page is not present

;%macro		PTE_PRESENT 1
;
;		test	byte [%1],PE_PRESENT|PE_NOPROT ; present or no protect
;%endmacro

; ------------- Macro - initialized page list (%1 = page list index)

%macro		PAGELIST_HEAD 1

		LISTHEAD		; list of pages
		dd	0		; number of pages
		db	%1		; page list index
		db	0,0,0
%endmacro

; -----------------------------------------------------------------------------
;                           Lock/unlock page list
; -----------------------------------------------------------------------------
; NOTES:	Use macro PGCLOCK to lock, PGCUNLOCK to unlock.
; -----------------------------------------------------------------------------

; ------------- Macro - lock page list

%macro		PGCLOCK 0
		LOCK_Lock PageListLock	; lock page list lock
%endmacro

; ------------- Macro - unlock page list

%macro		PGCUNLOCK 0
		LOCK_Unlock PageListLock ; unlock page list lock
%endmacro

; -----------------------------------------------------------------------------
;                          Initialize memory page map
; -----------------------------------------------------------------------------
; DESTROYS:	All registers
; NOTES:	It must be called before SysMemInit and after InitPDE.
;		If no continuous space found (very unlikely), it halts.
; -----------------------------------------------------------------------------

; ------------- Prepare number of physical pages

PageMapInit:	mov	eax,[MemoryMax]	; EAX <- end of physical memory
		shr	eax,PAGE_SHIFT	; EAX <- number of pages
		mov	[PageMapNum],eax ; number of page descriptors

; ------------- Prepare size of page map

		xor	edx,edx		; EDX <- 0
		mov	dl,PAGEDESC_size ; EDX <- page descriptor size
		mul	edx		; EAX <- size of page map
		mov	[PageMapSize],eax ; size of page map

; ------------- Prepare size of page map in memory bitmap

		add	eax,PAGE_SIZE-1	; round up to page size
		shr	eax,PAGE_SHIFT	; EAX <- number of pages
		mov	ebx,eax		; EBX <- save number of pages
		add	eax,byte 7	; round up to byte
		shr	eax,3		; EAX <- size of page map in bytes

; ------------- Prepare to find continuous memory for page map

		mov	esi,PageMemMap+SYSTEM_SIZE/PAGE_SIZE/8 ; end of map
		sub	esi,eax		; ESI <- start of possible page map
		xchg	eax,edx		; EDX <- size of page map in bytes
		xor	eax,eax		; EAX <- 0
		dec	eax		; EAX <- -1, available flags

; ------------- Check one possible space

PageMapInit2:	mov	edi,esi		; EDI <- address of the space
		mov	ecx,edx		; ECX <- size of the space
		repe	scasb		; check one space
		je	PageMapInit4	; suitable space found

; ------------- Try another space (it will be located below PDE table)

		dec	esi		; ESI <- lower address
		cmp	esi,PageMemMap	; is address OK?
		jae	PageMapInit2	; it is OK
		jmp	short $		; emergency halt

; ------------- Initialize found space

PageMapInit4:	mov	edi,esi		; EDI <- address of the space
		xor	eax,eax		; EAX <- 0, used flags
		mov	ecx,edx		; ECX <- size of the space
		rep	stosb		; mark space as used

; ------------- Correct last byte of the map

		and	bl,7		; pages in last byte
		jz	PageMapInit5	; no pages remain
		mov	cl,bl		; CL <- number of pages
		mov	bh,-1		; BH <- mask, pages are free
		shl	bh,cl		; BH <- new mask of used pages
		mov	[edi-1],bh	; set new mask of used pages

; ------------- Address of page map

PageMapInit5:	xchg	eax,esi		; EAX <- address in memory map
		sub	eax,PageMemMap	; EAX <- offset in memory map
		shl	eax,PAGE_SHIFT+3 ; EAX <- offset in system memory
		add	eax,SYSTEM_ADDR	; EAX <- address in memory
		mov	[PageMap],eax	; address of page map

; ------------- Initialize page map

		mov	ecx,[PageMapNum] ; ECX <- number of pages
		xor	edx,edx		; EDX <- 0
		mov	esi,SYSTEM_ADDR	; ESI <- system address
PageMapInit6:	mov	[eax+PG_FlagsDW],edx ; initialize flags
		mov	[eax+PG_Address],esi ; address in system memory
		mov	[eax+PG_Index],edx ; clear index
		mov	[eax+PG_PTEList],edx ; no PTE list
		add	eax,byte PAGEDESC_size ; EAX <- next descriptor
		loop	PageMapInit6	; next descriptor

; ------------- Initialize zero page descriptor

		mov	eax,PageEmpty	; EAX <- zero page
		call	GetPageDesc	; get page descriptor
		mov	byte [ebx+PG_Flags],PG_RESERVE+PG_COPYW ; flags
		mov	[PageZero],ebx	; store zero page descriptor
		ret

; -----------------------------------------------------------------------------
;                           Copy page on a write
; -----------------------------------------------------------------------------
; INPUT:	EAX = source page descriptor PAGEDESC
;		EBX = destination page descriptor PAGEDESC
; -----------------------------------------------------------------------------

; ------------- Push registers

PageCopyWrite:	push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare destination address (-> EDI) and page size (-> ECX)

		mov	edi,[ebx+PG_Address] ; EDI <- destination address
		xor	ecx,ecx		; ECX <- 0
		mov	ch,PAGE_SIZE/4/256 ; ECX <- page size/4

; ------------- Check if source page is zero page

		mov	esi,[eax+PG_Address] ; ESI <- source address
		cmp	esi,PageEmpty	; is it zero page?
		je	PageCopyWrite4	; it is zero page

; ------------- Copy page

		rep	movsd		; copy page

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; ------------- Clear page

PageCopyWrite4:	push	eax		; push EAX
		xor	eax,eax		; EAX <- 0
		rep	stosd		; clear page
		pop	eax		; pop EAX
	
; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                             Allocate zeroed page
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = page address (if NC, or EAX = 0 if CY)
;		CY = memory error (EAX = 0)
; NOTES:	Page is filled up with zero bytes.
; -----------------------------------------------------------------------------

; ------------- Allocate new page (-> EAX)

PageAllocZero:	xor	eax,eax		; EAX <- 0
		mov	ah,PAGE_SIZE/256 ; EAX <- page size
		call	SysMemAlloc	; allocate new page
		jc	PageAllocZero8	; memory error

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Clear the page (and clear CF)

		xchg	eax,edi		; EDI <- page address
		xor	eax,eax		; EAX <- 0
		xor	ecx,ecx		; ECX <- 0 (it clears CF)
		mov	ch,PAGE_SIZE/4/256 ; ECX <- page size in DWORDs
		rep	stosd		; clear page

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
PageAllocZero8:	ret

; -----------------------------------------------------------------------------
;                            Get page descriptor
; -----------------------------------------------------------------------------
; INPUT:	EAX = address in system memory (address must be valid) or PTE
; OUTPUT:	EBX = page descriptor PAGEDESC
; NOTES:  Bits 0 to 11 of the address are ignored.
; -----------------------------------------------------------------------------

; ------------- Push registers

GetPageDesc:	mov	ebx,eax		; EBX <- push EAX

%if (PAGEDESC_size == 32)		; optimized for 32 bytes

; ------------- Calculate address of page descriptor (-> EAX)

		sub	eax,SYSTEM_ADDR	; EAX <- offset in system memory
		shr	eax,PAGE_SHIFT	; EAX <- page number
		shl	eax,5		; EAX <- page number * descriptor size
		add	eax,[PageMap]	; EAX <- address of page descriptor

%else ; (PAGEDESC_size == 32)

; ------------- Push registers

		push	edx		; push EDX
		
; ------------- Calculate address of page descriptor

		sub	eax,SYSTEM_ADDR	; EAX <- offset in system memory
		shr	eax,PAGE_SHIFT	; EAX <- page number
		xor	edx,edx		; EDX <- 0
		mov	dl,PAGEDESC_size ; EDX <- size of page descriptor
		mul	edx		; EAX <- offset of page descriptor
		add	eax,[PageMap]	; EAX <- address of page descriptor

; ------------- Pop registers

		pop	edx		; pop EDX

%endif ; (PAGEDESC_size == 32)

; ------------- Pop registers

		xchg	eax,ebx		; pop EAX, EBX <- page descriptor
PTEFree99:	ret

; -----------------------------------------------------------------------------
;                       Free one PTE (page table entry)
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to PTE (page table entry)
; NOTES:	Page list must NOT be locked.
; -----------------------------------------------------------------------------

; ------------- Fast check if PTE is valid

PTEFree:	cmp	dword [eax],byte 0 ; is PTE valid?
		je	short PTEFree99	; PTE is not valid

; ------------- Push registers

		push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Lock page list

		pushf			; push flags
		cli			; disable interrupts
		PGCLOCK			; lock page list

; ------------- Get PTE (-> EAX)

		xchg	eax,edx		; EDX <- address of page table entry
		xor	eax,eax		; EAX <- 0, invalid PTE
		xchg	eax,[edx]	; EAX <- page table entry
		or	eax,eax		; is PTE valid?
		jz	short PTEFree0	; PTE is not valid

; ------------- Check if page is present (else it is swap page slot)

		test	al,PE_PRESENT	; is page present?
		jnz	short PTEFree1	; page is present

; ------------- Free swap page slot

		test	al,PE_FILE	; is it file page?
		jnz	PTEFree0	; it is file page
		call	SwapPageFree	; free swap page slot
PTEFree0:	jmp	near PTEFree8

; ------------- Get page descriptor (-> EBX)

PTEFree1:	call	GetPageDesc	; get page descriptor (-> EBX)

; ------------- Check if page is reserved for system (e.g. zero page)

		test	byte [ebx+PG_Flags],PG_RESERVE ; page reserved?
		jnz	short PTEFree0	; page is reserved for system

; ------------- Check if it is last PTE

		mov	ecx,[ebx+PG_PTEList] ; ECX <- list of PTE pointers
		or	ecx,ecx		; is it last PTE entry?
		js	PTEFree3	; it is array of PTE pointers

; ------------- Remove page from the current list

		call	ListDelEBX	; remove page
		movzx	ecx,byte [ebx+PG_PageList] ; ECX <- current page list
		mov	ecx,[PageListAddr+ecx*4] ; ECX <- pointer to page list
		dec	dword [ecx+PGL_Num] ; decrease number of pages

; ------------- Unlock page list

		PGCUNLOCK		; unlock page list
		popf			; pop flags (and enable interrupts)

; ------------- Free swap page slot

		test	byte [ebx+PG_Flags],PG_SWAPPED ; is page swapped?
		jz	PTEFree2	; page is not swapped
		mov	eax,[ebx+PG_Index] ; EAX <- page index
		call	SwapPageFree	; free swap page slot

; ------------- Free memory block

PTEFree2:	mov	eax,[ebx+PG_Address] ; EAX <- page address
		call	SysMemFree	; free memory block
		jmp	near PTEFree9

; ------------- Find PTE entry in PTE array

PTEFree3:       xchg	eax,edx		; EAX <- pointer to PTE, EDX <- PTE
		push	ecx		; push ECX
PTEFree4:	add	ecx,byte 4	; ECX <- next PTE entry
		cmp	eax,[ecx-4]	; check PTE entry
		jne	PTEFree4	; find PTE entry

; ------------- Remove PTE entry

PTEFree5:	mov	eax,[ecx]	; EAX <- next PTE entry
		mov	[ecx-4],eax	; shift PTE entry
		add	ecx,byte 4	; ECX <- next PTE entry
		or	eax,eax		; was it last entry?
		jnz	PTEFree5	; shift next entry
		pop	ecx		; pop ECX

; ------------- Check if last PTE entry remains

		cmp	dword [ecx+4],byte 0 ; is it last PTE entry?
		jne	PTEFree7	; it is not last PTE entry

; ------------- Change to one PTE entry

		mov	eax,[ecx]	; EAX <- last PTE entry
		sub	eax,SYSTEM_ADDR ; EAX <- offset of PTE entry
		mov	[ebx+PG_PTEList],eax ; set single PTE entry
		xchg	eax,ecx		; EAX <- array of pointers,ECX <- entry
		call	SysMemFree	; free array of PTE pointers

; ------------- Update PTE (ECX) in swap entry (EAX)

		test	byte [ebx+PG_Flags],PG_SWAPPED ; is page swapped?
		jz	PTEFree7	; page is not swapped
		mov	eax,[ebx+PG_Index] ; EAX <- page index
		call	SwapPageSet	; set swap page slot

; ------------- Test dirty flag (EDX = PTE entry)

PTEFree7:	test	dl,PE_DIRTY	; is page dirty?
		jz	short PTEFree8	; page is not dirty

; ------------- Test and set dirty flag (EBX = page descriptor)

		bts	dword [ebx+PG_Flags],PG_DIRTY_BIT ; test and set flag
		jc	short PTEFree8	; flag was already set

; ------------- Remove page from the current list

		call	ListDelEBX	; remove page
		movzx	ecx,byte [ebx+PG_PageList] ; ECX <- current page list
		mov	ecx,[PageListAddr+ecx*4] ; ECX <- pointer to page list
		dec	dword [ecx+PGL_Num] ; decrease number of pages

; ------------- Set new last access time

		GETSYSTIMELOW ecx	; EAX <- get system time LOW
		mov	[ebx+PG_LastTime],ecx ; set new last access time

; ------------- Add page into dirty list

		mov	eax,PagesDirty	; EAX <- dirty page list
		xchg	eax,ebx		; EAX <- page, EBX <- page list
		call	ListLast	; add page into end of list
		inc	dword [ebx+PGL_Num] ; increase number of pages
		mov	byte [eax+PG_PageList],PGLIST_DIRTY ; set new page list

; ------------- Unlock page list

PTEFree8:	PGCUNLOCK		; unlock page list
		popf			; pop flags (and enable interrupts)

; ------------- Pop registers

PTEFree9:	pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
PDEFree99:	ret

; -----------------------------------------------------------------------------
;                      Free one PDE (page directory entry)
; -----------------------------------------------------------------------------
; INPUT:	EAX = pointer to PDE (page directory entry)
; NOTES:	Page list must NOT be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

PDEFree:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Get PDE (-> EAX)

		xchg	eax,edx		; EDX <- address of page dir. entry
		xor	eax,eax		; EAX <- 0, invalid PDE
		xchg	eax,[edx]	; EAX <- page directory entry
		or	eax,eax		; is PDE valid?
		jz	short PDEFree8	; PDE is not valid

; ------------- Check if page directory is present

		test	al,PE_PRESENT	; is page directory present?
		jz	short PTEFree8	; page directory is not present
		and	ax,PDE_MASK	; EAX <- address of page directory

; ------------- Free PTE entries in page directory

		mov	ecx,PAGEDIR_NUM	; ECX <- number of entries
		push	eax		; push EAX
PDEFree2:	call	PTEFree		; free one PTE entry
		add	eax,byte 4	; EAX <- address of next PTE entry
		loop	PDEFree2	; free next PTE entry
		pop	eax		; pop EAX

; ------------- Free page directory

		call	SysMemFree	; free page directory

; ------------- Pop registers

PDEFree8:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                      Remove pages from the user space    
; -----------------------------------------------------------------------------
; INPUT:	EAX = start address of the interval (= first byte)
;		ECX = end address of the interval (= last byte + 1)
;		EDX = process descriptor PROCESS
; NOTES:	Process descriptor must be locked.
; 		Page list must NOT be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

PageRemove:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare start page (-> EAX) and number of pages (-> EDX)

		add	eax,PAGE_SIZE-1 ; EAX <- round up
		shr	eax,PAGE_SHIFT	; EAX <- start page (round up)
		shr	ecx,PAGE_SHIFT	; ECX <- end page + 1 (round down)
		sub	ecx,eax		; ECX <- number of pages
		jbe	short PageRemove9 ; invalid interval length
		mov	edx,ecx		; EDX <- number of pages

; ------------- Prepare first page directory entry (-> ESI)

		mov	esi,eax		; ESI <- start page of the interval
		shr	esi,PAGEDIR_SHIFT-PAGE_SHIFT ; ESI <- index of PDE
		shl	esi,2		; ESI <- offset of PDE
		mov	ebx,[edx+PROC_PGDir] ; EBX <- page global directory
		or	ebx,ebx		; is page global directory valid?
		jz	short PageRemove9 ; invalid page global directory
		add	esi,ebx		; ESI <- first page directory entry

; ------------- Prepare offset of first PTE (->EAX) and number of pages (->ECX)

		and	eax,PAGEDIR_NUM-1 ; EAX <- page index in one directory
PageRemove2:	xor	ecx,ecx		; ECX <- 0
		mov	ch,PAGEDIR_NUM/256 ; ECX <- entries per page directory
		sub	ecx,eax		; ECX <- remaining pages
		shl	eax,2		; EAX <- offset of first PTE

; ------------- Limit number of pages (-> ECX)

		cmp	ecx,edx		; check number of pages
		jbe	PageRemove3	; number of pages is OK
		mov	ecx,edx		; ECX <- limit number of pages
PageRemove3:	sub	edx,ecx		; EDX <- decrease number of pages

; ------------- Check if page directory entry is valid

		mov	ebx,[esi]	; EAX <- page directory entry
		or	ebx,ebx		; is page directory entry valid?
		jz	PageRemove6	; page directory is not valid

; ------------- Free whole directory entry

		cmp	ch,PAGEDIR_NUM/256 ; whole directory entry?
		jne	PageRemove4	; not whole directory entry
		mov	eax,esi		; EAX <- address of PDE
		call	PDEFree		; free page directory entry
		jmp	short PageRemove6

; ------------- Free PTE entries in one directory entry

PageRemove4:	add	eax,ebx		; EAX <- address of first entry
PageRemove5:	call	PTEFree		; free one PTE entry
		add	eax,byte 4	; EAX <- next PTE entry
		loop	PageRemove5	; free next PTE entry

; ------------- Next page directory entry

PageRemove6:	add	esi,byte 4	; ESI <- next page directory entry
		xor	eax,eax		; EAX <- 0, offset of next first PTE
		or	edx,edx		; any pages remain?
		jnz	PageRemove2	; next page directory

; ------------- Pop registers

PageRemove9:    pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                     Remove all pages from the user space    
; -----------------------------------------------------------------------------
; INPUT:	EDX = process descriptor PROCESS
; NOTES:	Process descriptor must be locked.
; 		Page list must NOT be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

PageRemoveAll:	push	eax		; push EAX
		push	ecx		; push ECX

; ------------- Remove pages

		xor	eax,eax		; EAX <- 0, start address
		mov	ecx,SYSTEM_ADDR ; ECX <- end address
		call	PageRemove	; remove pages

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                Copy virtual memory region from one task to another
; -----------------------------------------------------------------------------
; INPUT:	EDX = source virtual memory region VMREG
;		ESI = source process descriptor PROCESS
;		EDI = destination process descriptor PROCESS
; -----------------------------------------------------------------------------

CopyVMReg:


; ------------- Prepare 




		ret

; -----------------------------------------------------------------------------
;                Int 14: #PF Page fault service (interrupt gate)
; -----------------------------------------------------------------------------
; Page fault was caused by:
;	- Accessed page is not present in memory (P present flag is clear).
;	- Corresponding table entry is null.
;	- The procedure does not have sufficient privilege to access the page.
;	- Code running in user mode attempts to write to a read-only page.
;
; Page fault error code:
;	B0: (P) 0=The fault was caused by a nonpresent page.
;		1=The fault was caused by a page-level protection violation.
;	B1: (W/R) 0=The access causing the fault was a read.
;		  1=The access causing the fault was a write.
;	B2: (U/S) 0=Processor was executing in supervisor mode.
;		  1=Processor was executing in user mode.
;	B3: (RSVD) 0=The fault was not caused by a reserved bit violation.
;		   1=The page fault occured because a 1 was detected in one
;		     of the reserved bit positions of a page table entry or
;		     directory entry that was marked present.
; -----------------------------------------------------------------------------

; ------------- Push registers (it must correspond with the TRAPSTACK)

PageFault:	pusha			; push all registers
		push	ds		; push DS
		push	es		; push ES
		cld			; direction up

; ------------- Initialize kernel segments

		xor	eax,eax		; EAX <- 0
		mov	al,SYSTEM_DS	; EAX <- system DS
		mov	ds,eax		; DS <- system DS
		mov	es,eax		; ES <- system DS
		mov	ebp,esp		; EBP <- trap stack frame TRAPSTACK

; ------------- Get fault address (-> ESI)

		mov	esi,cr2		; ESI <- get fault address

; ------------- Check fault address if it is not in system memory

		cmp	esi,SYSTEM_ADDR	; is fault address in system memory?
		jae	PageFault8	; yes, system error

; ------------- Handle system exception

		test	byte [ebp+TRAP_Int+REGI_CS],3 ; privilege level?
		jnz	PageFault1	; user trap
		call	DoException	; handle system exception
		jmp	PageFault9

; ------------- Re-enable interrupts

PageFault1:	test	byte [ebp+TRAP_Int+REGI_Flags+1],EFLAGS_IF>>8; enabled?
		jz	PageFault2	; interrupts were not enabled
		sti			; re-enable interrupts

; ------------- Get current task (-> EDI)

PageFault2:	CURRENT	edi		; EDI <- get current task

; ------------- Find nearest higher region in virtual memory (-> EBX)

		mov	edx,[edi+TASK_Process] ; EDX <- parent process
		mov	eax,esi		; EAX <- fault address
		call	VirtMemFindAddr	; find region in virtual memory
		jc	PageFault5	; region not found

; ------------- Check if region is OK

		cmp	eax,[ebx+VMR_Start] ; is region OK?
		jae	PageFault4	; region is OK

; ------------- Address is below region, check if it can grow down

		test	byte [ebx+VMR_Flags+1],VMR_GROWSDOWN>>8 ; grows down?
		jz	PageFault5	; invalid address

; ------------- Region can grow down, check access distance of user stack

		test	byte [ebp+TRAP_Code],B2 ; user mode?
		jz	PageFault3	; no, it came from system mode
		add	eax,10000h+32*4	; reserve for "enter 65535,31"
		cmp	eax,[ebp+TRAP_ESP] ; check distance of stack
		jb	PageFault5	; invalid distance

; ------------- Expand stack

PageFault3:




; good area

PageFault4:





; Bad area

PageFault5:





; ------------- TODO!!!!! System error

PageFault8:

%ifdef DEBUG
		mov	al,[VideoRows]
		sub	al,2
		mov	[VideoRow],al
		mov	byte [VideoPos],0

		push	esi		; push fault address
		mov	esi,PageFaultTxt
		call	DebOutText	; display text

; ------------- Task index

		mov	eax,[edi+TASK_TSSIndex] ; ESI <- error code
		call	DebOutNum
		mov	esi,PageFaultTxt1
		call	DebOutText	; display text

; ------------- Get error code (-> EAX)

		mov	eax,[ebp+TRAP_Code] ; ESI <- error code
		call	DebOutNum
		mov	esi,PageFaultTxt2
		call	DebOutText	; display text

; ------------- Get fault address

		pop	eax		; EAX <- fault address
		call	DebOutHexD
		mov	esi,PageFaultTxt3
		call	DebOutText	; display text

; ------------- Get instruction address

		mov	eax,[ebp+TRAP_Int+REGI_EIP]
		call	DebOutHexD

		mov	esi,PageFaultTxt4
		call	DebOutText	; display text

%endif ; DEBUG


; ------------- Pop registers

PageFault9:	pop	es		; pop ES
		pop	ds		; pop DS
		popa			; pop all registers
		add	esp,byte 4	; skip error code
		iret

; -----------------------------------------------------------------------------
;                      Write modified page on disk
; -----------------------------------------------------------------------------
; INPUT:	EBX = page descriptor PAGEDESC
; NOTES:	Page need not be in a page lists.
; -----------------------------------------------------------------------------

PageWrite:

; !!!!!!!!!!! TODO


		ret

; -----------------------------------------------------------------------------
;                        Get flags of page activity
; -----------------------------------------------------------------------------
; INPUT:	EBX = page descriptor PAGEDESC
; OUTPUT:	DL = flags
;			PE_DIRTY: page is dirty (modified)
;			PE_ACCESSED: page was accessed
; NOTES:	It clears "dirty" and "accessed" flags of PTEs.
;		Page list must be locked.
;		It flushes TLB page caches.
; -----------------------------------------------------------------------------

; ------------- Push registers

GetPTEFlags:	push	eax		; push EAX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare flag accumulator (-> EDX)

		xor	edx,edx		; EDX <- 0, clear flag accumulator

; ------------- Prepare PTE list (-> ESI)

		mov	esi,[ebx+PG_PTEList] ; ESI <- list of PTE pointers
		or	esi,esi		; is it single PTE or array of PTEs?
		jz	short GetPTEFlags8 ; no PTE entry
		js	short GetPTEFlags7 ; array of PTE pointers

; ------------- Check if single PTE is dirty

		add	esi,SYSTEM_ADDR	; ESI <- pointer to PTE entry
		LOCKSMP			; CPU instruction lock
		btr	dword [esi],PE_DIRTY_BIT ; check if page is dirty
		jnc	short GetPTEFlags2 ; page is not dirty
		mov	dl,PE_DIRTY	; DL <- mark page as dirty
		INVTLB	esi,GetPTEFlags3 ; flush page table

; ------------- Check if single PTE was accessed

GetPTEFlags2:	LOCKSMP			; CPU instruction lock
		btr	dword [esi],PE_ACCESSED_BIT ; check if page is accessed
		jnc	short GetPTEFlags8 ; page was not accessed
		or	dl,PE_ACCESSED	; DL <- mark page as accessed
		INVTLB	esi,GetPTEFlags4 ; flush page table
		jmp	short GetPTEFlags8

; ------------- Flush TLB for 386 CPU

GetPTEFlags3:	call	FlushPageCache	; flush TLB cache for 386 CPU
		jmp	short GetPTEFlags2

GetPTEFlags4:	call	FlushPageCache	; flush TLB cache for 386 CPU
		jmp	short GetPTEFlags8

; ------------- Check if multiple PTE is dirty

GetPTEFlags5:	LOCKSMP			; CPU instruction lock
		btr	dword [eax],PE_DIRTY_BIT ; check if page is dirty
		jnc	short GetPTEFlags6 ; page is not dirty
		or	dl,PE_DIRTY	; DL <- mark page as dirty
		INVTLB	eax,GetPTEFlags9 ; flush page table

; ------------- Check if multiple PTE was accessed

GetPTEFlags6:	LOCKSMP			; CPU instruction lock
		btr	dword [eax],PE_ACCESSED_BIT ; check if page is accessed
		jnc	short GetPTEFlags7 ; page was not accessed
		or	dl,PE_ACCESSED	; DL <- mark page as accessed
		INVTLB	eax,GetPTEFlags92 ; flush page table

; ------------- Next PTE entry

GetPTEFlags7:	lodsd			; EAX <- load next PTE pointer
		or	eax,eax		; end of table?
		jnz	short GetPTEFlags5 ; next PTE entry

; ------------- Pop registers

GetPTEFlags8:	pop	esi		; pop ESI
		xchg	eax,edx		; AL <- flags
		pop	edx		; pop EDX
		mov	dl,al		; DL <- flags
		pop	eax		; pop EAX
SwapDaemonAct9:	ret

; ------------- Flush TLB for 386 CPU

GetPTEFlags9:	call	FlushPageCache	; flush TLB cache for 386 CPU
		jmp	short GetPTEFlags6

GetPTEFlags92:	call	FlushPageCache	; flush TLB cache for 386 CPU
		jmp	short GetPTEFlags7

; -----------------------------------------------------------------------------
;                   Check page activity in active page list
; -----------------------------------------------------------------------------
; DESTROYS:	EAX, EBX, ECX, EDX, ESI, EDI
; NOTES:	Page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Get first page from active page list (-> EBX)

SwapDaemonAct:	mov	edx,PagesActive	; EDX <- active page list
		mov	ebx,[edx+LIST_Next] ; EBX <- first page
		LISTTEST ebx		; check if list is empty
		jz	short SwapDaemonAct9 ; no page

; ------------- Check if page should be tested for activity

		GETSYSTIMELOW eax	; EAX <- get system time LOW
		mov	esi,eax		; ESI <- system time LOW
		sub	eax,[ebx+PG_LastTime] ; EAX <- elapsed time
		cmp	eax,PAGETIME_ACT ; time to test page activity?
		jb	short SwapDaemonAct9 ; no test for activity

; ------------- Remove page from active page list

		LISTDEL	ebx, eax, ecx	; remove page
		dec	dword [edx+PGL_Num] ; decrease number of pages

; ------------- Check page activity

		call	GetPTEFlags	; DL <- get flags of page activity
		xchg	eax,ebx		; EAX <- page
		test	dl,PE_DIRTY	; check if page is dirty
		jnz	short SwapDaemonAct8 ; page is dirty

; ------------- Page is not accessed, add page into inactive page list

		mov	ebx,PagesInact	; EBX <- inactive page list
		test	dl,PE_ACCESSED	; check if page is accessed
		jz	short SwapDaemonAct6 ; page is not accessed

; ------------- Add page into active page list

SwapDaemonAct3:	mov	ebx,PagesActive	; EBX <- active page list
SwapDaemonAct4:	mov	[eax+PG_LastTime],esi ; set new last access time

; ------------- Add page into new page list

SwapDaemonAct6:	LISTLAST ebx,eax,ecx	; add page into end of list
		inc	dword [ebx+PGL_Num] ; increase number of pages
		mov	dh,[ebx+PGL_PageList] ; DH <- page list index
		mov	[eax+PG_PageList],dh ; set new page list index
		jmp	short SwapDaemonAct ; next page

; ------------- Mark page dirty

SwapDaemonAct8:	LOCKSMP			; CPU instruction lock
		bts	dword [eax+PG_Flags],PG_DIRTY_BIT ; set dirty flag

; ------------- Check if page should be saved (only memory mapped file)

		test	byte [eax+PG_Flags],PG_FILE ; memory mapped file?
		jz	short SwapDaemonAct3 ; not memory mapped file

; ------------- Add page into dirty page list

		mov	byte [eax+PG_WriteMax],PAGETIME_WRITEM ; max. writes
		mov	ebx,PagesDirty	; EBX <- dirty page list
		jmp	short SwapDaemonAct4

; -----------------------------------------------------------------------------
;                     Save pages in dirty page list
; -----------------------------------------------------------------------------
; DESTROYS:	EAX, EBX, ECX, EDX, ESI
; NOTES:	Page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Get first page from dirty page list (-> EBX)

SwapDaemonWrite:mov	edx,PagesDirty	; EDX <- dirty page list
		mov	ebx,[edx+LIST_Next] ; EBX <- first page
		LISTTEST ebx		; check if list is empty
		jz	short SwapDaemonWri3 ; no page

; ------------- Check if page should be tested for activity

		GETSYSTIMELOW eax	; EAX <- get system time LOW
		sub	eax,[ebx+PG_LastTime] ; EAX <- elapsed time
		cmp	eax,PAGETIME_WRITE ; time to test page activity?
		jb	short SwapDaemonWri3 ; no test for activity

; ------------- Set new last access time

		add	[ebx+PG_LastTime],eax ; set new last access time

; ------------- Remove page from dirty page list

		LISTDEL	ebx, eax, ecx	; remove page
		dec	dword [edx+PGL_Num] ; decrease number of pages

; ------------- Check page activity

		call	GetPTEFlags	; DL <- get flags of page activity
		test	dl,PE_DIRTY	; check if page is dirty
		jz	short SwapDaemonWri6 ; page is not dirty

; ------------- Maximum write counter

		dec	byte [ebx+PG_WriteMax] ; check write counter
		jz	SwapDaemonWri5	; maximum write time reached

; ------------- Add page back into dirty page list

SwapDaemonWri1:	mov	eax,PagesDirty	; EBX <- dirty page list
SwapDaemonWri2:	xchg	eax,ebx		; EAX <- page, EBX <- page list
		LISTLAST ebx,eax,ecx	; add page into end of list
		inc	dword [ebx+PGL_Num] ; increase number of pages
		mov	dh,[ebx+PGL_PageList] ; DH <- page list index
		mov	[eax+PG_PageList],dh ; set new page list index
		jmp	short SwapDaemonWrite ; next page

SwapDaemonWri3:	ret

; ------------- Set locked flag

SwapDaemonWri5:	inc	byte [ebx+PG_WriteMax] ; undo write counter
SwapDaemonWri6:	LOCKSMP			; CPU instruction lock
		bts	dword [ebx+PG_Flags],PG_LOCKED_BIT ; set locked flag
		jc	SwapDaemonWri1	; someone else have caught the lock
		mov	byte [ebx+PG_PageList],PGLIST_NONE ; no page list

; ------------- Unlock page list

		PGCUNLOCK		; unlock page list
		sti			; enable interrupts

; ------------- Write page (EBX) on disk

		call	PageWrite	; write page on disk

; ------------- Lock page list

		cli			; disable interrupts
		PGCLOCK			; lock page list

; ------------- Set new time of last access

		GETSYSTIMELOW eax	; EAX <- get system time LOW
		mov	[ebx+PG_LastTime],eax ; set new last access time

; ------------- Clear locked and dirty flags

		LOCKSMP			; CPU instruction lock
		and	byte [ebx+PG_Flags],~(PG_LOCKED+PG_DIRTY) ; clear flags

; ------------- Flush flags of page activity

		call	GetPTEFlags	; get flags of page activity

; ------------- Add page into active page list

		mov	eax,PagesActive	; EAX <- active page list
		jmp	short SwapDaemonWri2 ; add page into active page list

; -----------------------------------------------------------------------------
;                           Free one page
; -----------------------------------------------------------------------------
; INPUT:	EBX = page descriptor PAGEDESC
; OUTPUT:	CY = cannot free pages (it was modified)
; NOTES:	Page must be free and page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

PageFree:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Prepare PTE list (-> ESI)

		mov	esi,[ebx+PG_PTEList] ; ESI <- list of PTE pointers
		or	esi,esi		; is it single PTE or array of PTEs?
;		jz	short GetPTEFlags8 ; no PTE entry
;		js	short GetPTEFlags6 ; array of PTE pointers

; ------------- Get page flags from one PTE entry

		add	esi,SYSTEM_ADDR	; ESI <- pointer to PTE entry










; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                  Free some pages from one page list
; -----------------------------------------------------------------------------
; INPUT:	EDI = page list
;		DL = 1: use active page, 0: don't use active page
;		DH = 1: use dirty page, 0: don't use dirty page
; OUTPUT:	CY = continue with next attempt, NC = memory freed OK
; DESTROYS:	EAX, EBX, ECX
; NOTES:	Page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Prepare number of pages (-> ECX)

SwapListFree:	mov	ecx,[edi+PGL_Num] ; ECX <- number of pages
		jecxz	SwapListFree8	; no page in the list		

; ------------- Check if there is enough free memory

SwapListFree2:	mov	eax,[TotalMemory] ; EAX <- total memory
		shr	eax,3		; EAX <- required free memory
		cmp	[TotalMemFree],eax ; check free memory
		jae	SwapListFree9	; there is enough free memory

; ------------- Get next page from the list (-> EBX)

		mov	ebx,[edi+LIST_Next] ; EBX <- first page
		LISTTEST ebx		; check if list is empty
		jz	short SwapListFree8 ; no page

; ------------- Remove page from the list

		call	ListDelEBX	; remove page
		dec	dword [edi+PGL_Num] ; decrease number of pages

; ------------- Check page activity

		call	GetPTEFlags	; DL <- get flags of page activity







SwapListFree8:	stc			; set flag - continue with next attempt
SwapListFree9:	ret

; -----------------------------------------------------------------------------
;            Free some pages if there is not enough free memory space
; -----------------------------------------------------------------------------
; DESTROYS:	EAX, EBX, ECX, EDX, ESI
; NOTES:	Page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Free pages from inactive list - use inactive pages

SwapDaemonFree:	xor	edx,edx		; EDX <- 0, don't use active pages
		mov	edi,PagesInact	; EDI <- inactive page list
		call	SwapListFree	; free pages from the list
		jnc	SwapDaemonFree9	; memory is OK

; ------------- Free pages from active list - use inactive pages

		mov	edi,PagesActive	; EDI <- active page list
		call	SwapListFree	; free pages from the list
		jnc	SwapDaemonFree9	; memory is OK

; ------------- Free pages from inactive list - use active pages

		inc	edx		; DL <- 1, use active pages
		mov	edi,PagesInact	; EDI <- inactive page list
		call	SwapListFree	; free pages from the list
		jnc	SwapDaemonFree9	; memory is OK

; ------------- Free pages from active list - use active pages

		mov	edi,PagesActive	; EDI <- active page list
		call	SwapListFree	; free pages from the list
		jnc	SwapDaemonFree9	; memory is OK

; ------------- Free pages from dirty list - use active and dirty pages

		mov	dh,1		; DH <- 1, active and dirty pages
		mov	edi,PagesInact	; EDI <- inactive page list
		call	SwapListFree	; free pages from the list
SwapDaemonFree9:ret

; -----------------------------------------------------------------------------
;                      Fast free pages from one page list
; -----------------------------------------------------------------------------
; INPUT:	ECX = number of pages to free
;		EDX = page list
; OUTPUT:	ECX = remaining pages which were not freed
; NOTES:	Page list must be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

FastFreeList:	push	esi		; push ESI

; ------------- Prepare number of pages (-> ESI)

		mov	esi,[edx+PGL_Num] ; ESI <- number of pages

; ------------- Get next page (-> EBX)

		mov	ebx,edx		; EBX <- page list head
FastFreeList2:	mov	ebx,[ebx+LIST_Next] ; EBX <- next page
		cmp	ebx,edx		; is it end of list?
		je	FastFreeList9	; end of list

; ------------- 





; ------------- Pop registers

FastFreeList9:	pop	esi		; pop ESI
		ret

; -----------------------------------------------------------------------------
;                         Fast free some pages
; -----------------------------------------------------------------------------
; OUTPUT:	CY = no pages freed
; NOTES:	It tries to free some unused pages without waiting
;		It locks page list.
; -----------------------------------------------------------------------------

%define		FREEPAGENUM 8		; required number of pages to free

; ------------- Push registers

FastFreePages:	push	ecx		; push ECX
		push	edx		; push EDX
		pushf			; push flags
		cli			; disable interrups

; ------------- Lock page list

		PGCLOCK			; lock page list

; ------------- Prepare number of pages to free

		xor	ecx,ecx		; ECX <- 0
		mov	cl,FREEPAGENUM	; ECX <- required number of pages

; ------------- Free pages from inactive page list

		mov	edx,PagesInact	; EDX <- inactive page list
		call	FastFreeList	; free pages from one page list
		jecxz	FastFreePages8	; no other page to free

; ------------- Free pages from active page list

		mov	edx,PagesActive	; EDX <- active page list
		call	FastFreeList	; free pages from one page list

; ------------- Unlock page list

FastFreePages8:	PGCUNLOCK		; unlock page list

; ------------- Pop registers

		popf			; pop flags (and enable interrupts)
		pop	edx		; pop EDX
		cmp	cl,FREEPAGENUM	; check number of pages
		cmc			; CY = no page freed
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                              Swap daemon
; -----------------------------------------------------------------------------
; NOTES:	It does not return from this function (=background daemon).
; -----------------------------------------------------------------------------

; ------------- Lock page list

SwapDaemon:	cli			; disable interrupts
		PGCLOCK			; lock page list

; ------------- Check page activity in active page list

		call	SwapDaemonAct	; check page activity

; ------------- Save pages in dirty page list

		call	SwapDaemonWrite	; save pages in dirty page list

; ------------- Free some pages if there is not enough free memory space

		call	SwapDaemonFree	; free some pages

; ------------- Unlock page list

		PGCUNLOCK		; unlock page list
		sti			; enable interrupts

; ------------- Sleep for a while

		mov	eax,PAGETIME_DAEMON ; EAX <- sleep time (in [ms])
		call	Sleep		; sleep
		jmp	near SwapDaemon

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

PageFaultTxt:	db	">>>>> Task ",0
PageFaultTxt1:	db	", Page fault ",0
PageFaultTxt2:	db	" at ",0
PageFaultTxt3:	db	" called from ",0
PageFaultTxt4:	db	" <<<<<",10,0

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Memory page map of physical pages

		align	4, db 0
PageMap:	dd	0		; array of page descriptors
PageMapNum:	dd	0		; number of page descriptors
PageMapSize:	dd	0		; size of array of page descriptors
PageDestroyable:dd	0		; number of fast destroyable pages

PageZero:	dd	0		; pointer to zero page descriptor

; ------------- Page list lock

		align	4, db 0
PageListLock:	SPINLOCK		; page list lock

; ------------- Page list

		align	8, db 0
PagesActive:	PAGELIST_HEAD PGLIST_ACT ; active page list (last accessed)
PagesInact:	PAGELIST_HEAD PGLIST_INACT ; inactive page list
PagesDirty:	PAGELIST_HEAD PGLIST_DIRTY ; dirty page list

; ------------- Page list addresses

		align	8, db 0
PageListAddr:	dd	NULL		; no page list
		dd	PagesActive	; active page list
		dd	PagesInact	; inactive page list
		dd	PagesDirty	; dirty page list (waiting for write)
