; ============================================================================
;
;                       Install Litos System on Floppy
;
; ============================================================================

; ------------- Setup parameters

DISK		EQU	0		; booting disk (0=diskette A:)
HEADS		EQU	2		; number of heads
SECTORS		EQU	18		; number of sectors per track

; ------------- Start of program

SECTION		.text
BITS		16

		org	100h		; start of COM program

; ------------- Prepare address of disk buffer (cannot overlap 64 KB boundary)

Start:		mov	ax,ds		; AX <- program segment
		shl	ax,1
		shl	ax,1
		shl	ax,1
		shl	ax,1		; AX * 16
		add	ax,[Sector]	; add current buffer offset
		add	ax,200h		; test buffer overlapping
		jnc	BufferOK	; buffer is OK
		mov	ax,[Sector]	; AX <- current buffer address
		add	ax,200h		; move over boundary
		mov	[Sector],ax	; new buffer address

; ------------- Open file with boot sector

BufferOK:	mov	ax,3d00h	; AX <- function (open for reading)
		mov	dx,Filename	; DX <- name of file with boot sector
		int	21h		; open file for reading
		jnc	OpenOK		; file open OK

; ------------- Error - file not found

		mov	dx,FoundErr	; DX <- error message
Error:		mov	ah,9		; AH <- function code
		int	21h		; display error message
		int	20h		; end of program

; ------------- Read file into memory (AX=file identifier)

OpenOK: 	mov	bx,ax		; BX <- file identifier
		mov	cx,512		; CX <- length of boot sector
		mov	dx,[Sector]	; DX <- buffer for reading sector
		mov	ah,3fh		; AH <- function code
		int	21h		; read sector into memory

; ------------- Close file

		mov	ah,3eh		; AH <- function code
		int	21h		; close file

; ------------- Displays prompt text for user

		mov	ah,9		; AH <- function code
		mov	dx,Message	; DX <- message to display
		int	21h		; display message

; ------------- Waiting for a key

		mov	ax,0c08h	; AX <- function code
		int	21h		; waiting for a key
		cmp	al,27		; is it Esc key?
		jne	InitDisk	; it is not Esc, start write
		int	20h		; end of program

; ------------- Display message - Writing sector

InitDisk:	mov	ah,9		; AH <- function code
		mov	dx,StartText	; DX <- text to display
		int	21h		; display message

; ------------- Open file with program

		mov	ax,3d00h	; AX <- function (open for reading)
		mov	dx,Filename2	; DX <- name of file with boot sector
		int	21h		; open file for reading
		mov	dx,FoundErr2	; DX <- error message
		jc	Error		; error - file not found
		mov	bx,ax		; BX <- file identifier

; ------------- Determine file size
		
		mov	ax,4202h	; AX <- function (seek to end of file)
		xor	cx,cx		; CX <- 0 offset HIGH
		xor	dx,dx		; DX <- 0 offset LOW
		int	21h		; determine file size -> DX:AX

; ------------- Convert file size to number of sectors (multiple of 512 Bytes)

		add	ax,511		; round up to nearest sector
		adc	dx,0		; overflow
		mov	al,ah		; AL <- file size / 256 (LOW)
		mov	ah,dl		; AH <- file size / 256 (HIGH)
		shr	ax,1		; AX = file size in sectors
		mov	si,[Sector]	; SI <- buffer

; ------------- Filling up setup parameters

		mov	[si+3],ax	; number of sectors to read
		mov	byte [si+5],DISK ; booting disk
		mov	byte [si+6],HEADS ; number of heads
		mov     byte [si+7],SECTORS ; number of sectors per track

; ------------- Write boot sector

		call	Write		; write boot sector

; ------------- Reset file pointer

		push	ax		; push AX
		mov	ax,4200h	; AH <- function (seek to begin of file)
		xor	cx,cx		; CX <- 0 offset HIGH
		xor	dx,dx		; DX <- 0 offset LOW
		int	21h		; reset file pointer
		pop	ax		; pop AX

; ------------- Read next sector of program

CopyProg:	push	ax		; push AX (number of sectors)
		mov	cx,512		; CX <- length of boot sector
		mov	dx,[Sector]	; DX <- buffer for reading sector
		mov	ah,3fh		; AH <- function code
		int	21h		; read sector into memory
		pop	ax		; pop AX (number of sectors)

; ------------- Write sector of program

		call	Write		; write sector of program
		dec	ax		; number of sectors
		jnz	CopyProg	; copy next sector of program

; ------------- Close file

		mov	ah,3eh		; AH <- function code
		int	21h		; close file

; ------------- All OK

		mov	ah,9		; AH <- function code
		mov	dx,AllOKText	; DX <- message "All OK"
		int	21h		; display message
		int	20h		; end of program

; ----------------------------------------------------------------------------
;                   Procedure - write 1 sector
; ----------------------------------------------------------------------------

; ------------- Push registers

Write:		push	ax		; push AX
		push	bx		; push BX
		push	cx		; push CX
		push	dx		; push DX
		push	si		; push SI
		
; ------------- Prepare registers

		mov	si,3		; number of attempts
WriteAgain:	mov	cl,[WriteSector] ; CL <- write sector
		mov	dh,[WriteHead]	; DH <- write head
		mov	ch,[WriteTrack]	; CH <- write track
		mov	dl,DISK		; DL <- disk
		mov	bx,[Sector]	; BX <- buffer with sector
		mov	ax,301h		; AH <- function, AL <- num. of sectors
		push	ds		; push DS to stack
		pop	es		; ES <- DS data segment
		push	si		; push attempt counter
		int	13h		; write sector
		pop	si		; pop attempt counter
		jnc	Write2		; operation OK

; ------------- Error - reset disk and try again (maybe disk change)

		mov	ah,0		; AH <- function code
		int	13h		; reset disk system
		dec	si		; attempt counter
		jnz	WriteAgain	; next attempt

; ------------- Error - cannot write to a floppy

		mov	ah,9		; AH <- function code
		mov	dx,WriteErr	; DX <- error message
		int	21h		; display error message
		add	sp,6*2		; delete registers and return address
		int	20h		; end of program

; ------------- Shift write pointers

Write2:		mov	al,[WriteSector] ; AL <- write sector
		inc	ax		; increase sector number
		cmp	al,SECTORS	; is it valid sector?
		jbe	Write4		; it is valid sector
		mov	al,1		; AL <- 1 new sector number
		mov	ah,[WriteHead]	; AH <- write head
		inc	ah		; increase head number
		cmp	ah,HEADS	; is it valid head?
		jb	Write3		; it is valid head
		mov	ah,0		; AH <- 0 new head number
		inc	byte [WriteTrack] ; new track number
Write3:		mov	[WriteHead],ah	; new write head
Write4:		mov	[WriteSector],al ; new write sector

; ------------- Pop registers

		pop	si		; pop SI
		pop	dx		; pop DX
		pop	cx		; pop CX
		pop	bx		; pop BX
		pop	ax		; pop AX
		ret

; ------------- Messages

FoundErr	db	"ERROR: File BOOT\BOOT.BIN not found!",13,10,"$"
FoundErr2	db	"ERROR: File LITOS.SYS not found!",13,10,"$"
WriteErr	db	"ERROR: Cannot write to disk!",13,10,"$"

Message		db	"Insert formated disk into A: and press",13,10
		db	"any key to continue or Esc to abort...",13,10,"$"

StartText	db	"Writing sectors...",13,10,"$"

AllOKText	db	"All OK",13,10,"$"

; ------------- Write pointers

WriteSector	db	1		; write sector
WriteHead	db	0		; write head
WriteTrack	db	0		; write track

; ------------- Files

Filename	db	"BOOT\BOOT.BIN",0
Filename2	db	"LITOS.SYS",0

; ------------- Buffer of boot sector

Sector		dw	Buffer
Buffer:
