; =============================================================================
;
;                           Litos - Memory pages
;
; =============================================================================

; Page cache (virtual memory):
; - active_list pages (clean or dirty, age > 0), may be mapped by process
; - inactive_dirty_list pages (clean or dirty, age = 0), not mapped
; - inactive_clean_list  (clean, age = 0), not mapped
;
; Page fault - move fault page to active_list
;
; User page:
;   - Load page:
;      	- attempt to access-added to page cache
;	- read page during swap-ahead
;  	- read page during memory map cluster read-ahead
;   - Page becomes dirty when written by the process (still in active_list)
;   - Page is not used, age reduced by kswapd() daemon
;   - page_launder()

; ------------- Memory page frame

PAGE_SIZE	EQU	4096		; memory page size
PAGE_SHIFT	EQU	12		; bits of memory page size
PAGE_MASK	EQU	~(PAGE_SIZE-1)	; mask page bits, round to page frame
PAGE_MAX	EQU	(SYSTEM_SIZE >> PAGE_SHIFT)-1 ; maximal page number

PAGEDIR_SHIFT	EQU	22		; bits to get page directory index
PAGEDIR_NUM	EQU	1024		; number of entries per page directory

; ------------- System memory allocator

DMAMAXMEM	EQU	1000000h	; maximal DMA memory size (16 MB)
DMAMINMEM	EQU	2*FDBUFFSIZE	; minimal DMA memory size (32 KB)

; Memory area size: minimal 4 KB (=12), optimized 32 KB (=15) to 128 KB (=17)

%ifdef MINI
MEMAREASHIFT	EQU	15		; bit shifts of memory area (32 KB)
%else
MEMAREASHIFT	EQU	16		; bit shifts of memory area (64 KB)
%endif

MEMAREAMINSHIFT	EQU	3		; minimal order of memory block
MEMMINBLOCK	EQU	1<<MEMAREAMINSHIFT ; minimal memory block

MEMAREASIZE	EQU	1<<MEMAREASHIFT	; size of memory area (64 KB, 32 KB)
MEMAREAMASK	EQU	~(MEMAREASIZE-1) ; mask memory area
MEMAREAPAGES	EQU	MEMAREASIZE/PAGE_SIZE ; number of pages per area (16,8)
MEMAREANUM	EQU	SYSTEM_SIZE/MEMAREASIZE; number of memory areas(32K,8K)
MEMNUMCORR	EQU	SYSTEM_ADDR >> MEMAREASHIFT ; area number correction

MAXMEMBITS	EQU	MEMAREASHIFT	; memory block bits (16=64 KB,...)
MAXMEM		EQU	MEMAREASIZE	; maximal memory block (64 KB, 32 KB)

MEMAHEADBITS	EQU	MAXMEMBITS-1	; memory ahead buffer - bits (14, 15)
MEMAHEADSIZE	EQU	1<<MEMAHEADBITS ; memory ahead buffer (32 KB, 16 KB)
MEMAHEADPAGES	EQU	MEMAHEADSIZE/PAGE_SIZE ; number of pages per buffer
MEMAHEADBLOCKS	EQU	MAXMEM/MEMAHEADSIZE ; number of buffers per block
MEMAHEADBLKPAGE	EQU	MEMAHEADBLOCKS*MEMAHEADPAGES; number of pages per block

; ------------- Page Directory Entry, PDE

struc		PDE

		resd	1		; B0: P: 1=present in memory
					; B1: R/W: Read/Write protection
					;	0=page can only be read
					;	1=page can be read and write
					; B2: U/S: User/Supervisor protection
					;	0=only CPL < 3 can access page
					;	(e.g. CPU is in kernel mode)
					;	1=page can always be accessed
					; B3: PWT: page write-through control
					;	(only 486 and later)
					;	0=hw cache uses write-through
					;	1=hw cache uses write-back
					; B4: PCD: page cache disable
					;	(only 486 and later)
					;	0=IVP can be enabled (see CR0)
					;	1=IVP is disabled
					; B5: A: 1=page frame was accessed
					; B7: Page size, 1=2(4) MB pages, 0=4KB
					; B8: Global page (not flushed)
					; B9-B11: available for system
					; B12-B31: page table address (B12-B31)

endstruc				; size 4 bytes

PDE_MASK	EQU	PAGE_MASK

; ------------- Page Table Entry, PTE

struc		PTE

		resd	1		; B0: P: 1=present in memory
					; B1: R/W: Read/Write protection
					;	0=page can only be read
					;	1=page can be read and write
					; B2: U/S: User/Supervisor protection
					;	0=only CPL < 3 can access page
					;	(e.g. CPU is in kernel mode)
					;	1=page can always be accessed
					; B3: PWT: Page write-through control
					;	(only 486 and later)
					;	0=hw cache uses write-back
					;	1=hw cache uses write-through
					; B4: PCD: page cache disable
					;	(only 486 and later)
					;	0=IVP can be enabled (see CR0)
					;	1=IVP is disabled
					; B5: A: 1=page frame was accessed
					; B6: D: 1=dirty (modified)
					; B7: Page size, 1=2(4) MB pages, 0=4KB
					; B8: Global page (not flushed)
					; B9-B11: available for system
					; B12-B31: page frame address (B12-B31)

endstruc				; size 4 bytes

PTE_MASK	EQU	PAGE_MASK

; ------------- Page entry flags

PE_PRESENT_BIT	EQU	0		; page present in memory
PE_PRESENT	EQU	1<<PE_PRESENT_BIT

PE_RW_BIT	EQU	1		; page can be written
PE_RW		EQU	1<<PE_RW_BIT

PE_USER_BIT	EQU	2		; page can be accessed by user
PE_USER		EQU	1<<PE_USER_BIT

PE_PWT_BIT	EQU	3		; hw cache uses write-back
PE_PWT		EQU	1<<PE_PWT_BIT

PE_PCD_BIT	EQU	4		; page cache disabled
PE_PCD		EQU	1<<PE_PCD_BIT

PE_ACCESSED_BIT	EQU	5		; page frame was accessed
PE_ACCESSED	EQU	1<<PE_ACCESSED_BIT

PE_DIRTY_BIT	EQU	6		; page is modified (dirty)
PE_DIRTY	EQU	1<<PE_DIRTY_BIT

PE_PSE_BIT	EQU	7		; large pages 4 MB (or 2 MB)
PE_PSE		EQU	1<<PE_PSE_BIT

PE_GLOBAL_BIT	EQU	8		; global page (not flushed)
PE_GLOBAL	EQU	1<<PE_GLOBAL_BIT

; ------------- Special flags if PE_PRESENT flag is clear

PE_FILE_BIT	EQU	PE_DIRTY_BIT	; nonlinear file mapping (else swap)
PE_FILE		EQU	1<<PE_FILE_BIT

;PE_NOPROT_BIT	EQU	PE_PSE_BIT	; mapped with no protection
;PE_NOPROT	EQU	1<<PE_NOPROT_BIT

; ------------- Page types

PE_INIT		EQU	PE_PRESENT+PE_RW			; system
PE_NONE		EQU	PE_ACCESSED ;+PE_NOPROT			; reserved
PE_SHARED	EQU	PE_PRESENT+PE_RW+PE_USER+PE_ACCESSED	; shared
PE_COPY		EQU	PE_PRESENT+PE_USER+PE_ACCESSED		; copy on write
PE_READONLY	EQU	PE_PRESENT+PE_USER+PE_ACCESSED		; read only

; Page table entry variants:
;   0 = page is not used
;   B0 (PE_PRESENT) is set = page is present in memory, B12-B31 = page address
;   B0 (PE_PRESENT) is clear:
;	B6 (PE_FILE) is clear = page swaped out, B1-B6 = swap area index,
;				 B8-B31 = slot index
;  	B6 (PE_FILE) is set = page is file mapped

; ------------- Page descriptor

struc		PAGEDESC

		resb	LIST_size	; 0: page list
					;	- PagesActive
					;	- PagesInact
					;	- PagesInact2
					;	- PagesDirty
PG_FlagsDW:
PG_Flags:	resb	1		; 8: page flags (see below)
		resb	1
PG_PageList:	resb	1		; 0Ah: current page list (see below)
PG_WriteMax:	resb	1		; 0Bh: maximum write counter
PG_Address:	resd	1		; 0Ch: address in system memory
PG_Index:	resd	1		; 10h: page index in swap or file
PG_Handle:	resd	1		; 14h: file or disk handle
PG_LastTime:	resd	1		; 18h: time of last access LOW
PG_PTEList:	resd	1		; 1Ch: list of PTE pointers/PTE entry
					;     B31: 1=array of PTE pointers+NULL
					;	   0=single PTE entry (=offset)
endstruc				; size 20h = 32 bytes (see GetPageDesc)

PAGETIME_DAEMON	EQU	250		; daemon interval (milliseconds)
PAGETIME_ACT	EQU	(TIME_1MS*1900)	; interval of test page activity
PAGETIME_WRITE	EQU	(TIME_1MS*450)	; interval of writting dirty page
PAGETIME_WRITEM	EQU	5		; max. count of writting dirty page

; ------------- Page flags

PG_RESERVE_BIT	EQU	0		; reserved for system (do not delete)
PG_RESERVE	EQU	1<<PG_RESERVE_BIT

PG_LOCKED_BIT	EQU	1		; page is locked in I/O operation
PG_LOCKED	EQU	1<<PG_LOCKED_BIT

PG_DIRTY_BIT	EQU	2		; page is dirty (modified)
PG_DIRTY	EQU	1<<PG_DIRTY_BIT

PG_ERROR_BIT	EQU	3		; I/O error occured while transfering
PG_ERROR	EQU	1<<PG_ERROR_BIT

PG_COPYW_BIT	EQU	4		; copy on write
PG_COPYW	EQU	1<<PG_COPYW_BIT

PG_SWAPPED_BIT	EQU	5		; page is swapped-out
PG_SWAPPED	EQU	1<<PG_SWAPPED_BIT

PG_FILE_BIT	EQU	6		; page is memory mapped file
PG_FILE		EQU	1<<PG_FILE_BIT

; ------------- Current page list

PGLIST_NONE	EQU	0		; no page list
PGLIST_ACT	EQU	1		; active page list
PGLIST_INACT	EQU	2		; inactive page list
PGLIST_DIRTY	EQU	3		; dirty page list

; ------------- Page list

struc		PAGELIST

		resb	LIST_size	; 0: list of pages
PGL_Num:	resd	1		; 8: number of pages
PGL_PageList:	resb	1		; 0Ch: page list index
		resb	3

endstruc				; size 16 bytes
