; =============================================================================
;
;                             Litos - CMOS memory
;
; =============================================================================

CMOS_IRQ	EQU	8		; CMOS IRQ number

; ------------- CMOS device descriptor

struc		DEVCMOS

		resb	DEV_size	; general device

DEVCMOS_GetByte:resd	1		; read byte from CMOS memory
					;	INPUT:	EBX=device descriptor
					;		EDX=index of data
					;	OUTPUT:	EAX=data
					;		CY=invalid index

DEVCMOS_SetByte:resd	1		; write byte to CMOS memory
					;	INPUT:	AL=data
					;		EBX=device descriptor
					;		EDX=index of data
					;	OUTPUT:	CY=invalid index

DEVCMOS_GetData:resd	1		; read data from CMOS memory
					;	INPUT:	EAX=data buffer
					;		EBX=device descriptor
					;		ECX=number of bytes
					;		EDX=start index
					;	OUTPUT:	CY=invalid parameters

DEVCMOS_SetData:resd	1		; write data to CMOS memory
					;	INPUT:	EAX=data buffer
					;		EBX=device descriptor
					;		ECX=number of bytes
					;		EDX=start index
					;	OUTPUT:	CY=invalid parameters

DEVCMOS_GetTime:resd	1		; read date and time from CMOS memory
					;	INPUT:	EBX=device descriptor
					;		EDX=DATETIME buffer
					;	OUTPUT:	CY=time update is in
					;		   progress, try again

DEVCMOS_SetTime:resd	1		; write date and time to CMOS memory
					;	INPUT:	EBX=device descriptor
					;		EDX=DATETIME buffer
					;	OUTPUT:	CY=time update is in
					;		   progress, try again

DEVCMOS_GetAlrm:resd	1		; read alarm from CMOS memory
					;	INPUT:	EBX=device descriptor
					;		EDX=DATETIME buffer
					;	OUTPUT: EAX=flags (see below)
					;		CY=error

DEVCMOS_SetAlrm:resd	1		; write alarm to CMOS memory
					;	INPUT:	EAX=flags (see below)
					;		EBX=device descriptor
					;		EDX=DATETIME buffer
					;	OUTPUT:	CY=error

DEVCMOS_GetRate:resd	1		; get periodic interrupt rate (IRQ 8)
					;	INPUT:	EBX=device descriptor
					;	OUTPUT:	CY=error
					;		EAX=rate (see below)

DEVCMOS_SetRate:resd	1		; set periodic interrupt rate (IRQ 8)
					;	INPUT:	EAX=rate (see below)
					;		EBX=device descriptor
					;	OUTPUT:	CY=error

DEVCMOS_AckInt:	resd	1		; acknowledge interrupt
					;	INTPUT:	EBX=device descriptor
					;	OUTPUT:	CY=error

DEVCMOS_NextNMI:resd	1		; Reassert NMI

endstruc

%define DEV_CMOS_ID "CMS1"		; CMOS interface identifier

; ------------- Alarm flags (entries not "valid" are "don't care")

ALARM_ENABLED	EQU	B0		; alarm is enabled
ALARM_USE_SEC	EQU	B1		; second is valid
ALARM_USE_MIN	EQU	B2		; minute is valid
ALARM_USE_HOUR	EQU	B3		; hour is valid

; Currently unused and reserved flags:
ALARM_USE_DAYW	EQU	B4		; day of week is valid
ALARM_USE_DAY	EQU	B5		; day in month is valid
ALARM_USE_MONTH	EQU	B6		; month is valid
ALARM_USE_YEAR	EQU	B7		; year is valid
ALARM_OK_DAYW	EQU	B8		; day of week is supported
ALARM_OK_DAY	EQU	B9		; day in month is supported
ALARM_OK_MONTH	EQU	B10		; month is supported
ALARM_OK_YEAR	EQU	B11		; year is supported

; ------------- Periodic interrupt rate

CMOS_RATE_NONE	EQU	0		; periodic interrupt is OFF
CMOS_RATE_8K	EQU	3		; 8192 Hz, 122.0703125 us
CMOS_RATE_4K	EQU	4		; 4096 Hz, 244.140625 us
CMOS_RATE_2K	EQU	5		; 2048 Hz, 488.28125 us
CMOS_RATE_1K	EQU	6		; 1024 Hz, 976.5625 us
CMOS_RATE_512	EQU	7		; 512 Hz, 1.953125 ms
CMOS_RATE_256	EQU	8		; 256 Hz, 3.90625 ms
CMOS_RATE_128	EQU	9		; 128 Hz, 7.8125 ms
CMOS_RATE_64	EQU	10		; 64 Hz, 15.625 ms
CMOS_RATE_32	EQU	11		; 32 Hz, 31.25 ms
CMOS_RATE_16	EQU	12		; 16 Hz, 62.5 ms
CMOS_RATE_8	EQU	13		; 8 Hz, 125 ms
CMOS_RATE_4	EQU	14		; 4 Hz, 250 ms
CMOS_RATE_2	EQU	15		; 2 Hz, 500 ms

CMOS_RATE_MAX	EQU	15		; maximal value

; ------------- CMOS structure
; BCD/BIN selection depends on bit 2 of register 0bh
; 12/24-hour selection depends on bit 1 of register 0bh
; Alarm will trigger when contents of all three alarm byte registers
;	math their companions

CMOS_SEC	EQU	0		; RTC seconds in BCD or BIN mode
					;	BCD: 0 - 59h, BIN: 0 - 59
					;	bit 7 is read only

CMOS_ALARM_SEC	EQU	1		; RTC alarm seconds in BCD or BIN mode
					;	BCD: 0 - 59h, BIN: 0 - 59
					;	don't care if 0c0h - 0ffh

CMOS_MIN	EQU	2		; RTC minutes in BCD or BIN mode
					;	BCD: 0 - 59h, BIN: 0 - 59

CMOS_ALARM_MIN	EQU	3		; RTC alarm minutes in BCD or BIN mode
					;	BCD: 0 - 59h, BIN: 0 - 59
					;	don't care if 0c0h - 0ffh

CMOS_HOUR	EQU	4		; RTC hours in BCD or BIN mode
					;	24-hour: BCD: 0 - 23h
					;		 BIN: 0 - 23
					;	12-hour: BCD: 01h - 12h
					;		 BIN: 1 - 12
					;		 bit 7: 0=AM, 1=PM

CMOS_ALARM_HOUR	EQU	5		; RTC alarm hours in BCD or BIN mode
					;	24-hour: BCD: 0 - 23h
					;		 BIN: 0 - 23
					;	12-hour: BCD: 01h - 12h
					;		 BIN: 1 - 12
					;		 bit 7: 0=AM, 1=PM
					;	don't care if 0c0h - 0ffh

CMOS_DAYWEEK	EQU	6		; RTC day of week (1=Sunday-7=Saturday)

CMOS_DAY	EQU	7		; RTC day in month in BCD or BIN mode
					;	BCD: 01h - 31h, BIN: 1 - 31

CMOS_MONTH	EQU	8		; RTC month in BCD or BIN mode
					;	BCD: 01h - 12h, BIN: 1 - 12

CMOS_YEAR	EQU	9		; RTC year in BCD or BIN mode
					;	BCD: 0 - 99h, BIN: 0 - 99


CMOS_STATUS_A	EQU	0ah		; RTC status register A
					;	bit 0-3: rate selection for
					;		  interrupt (default 6)
					;		0=none
					;		1=30.517 us, 32768 Hz
					;		    for base > 32768 Hz
					;		 or 3.90625 ms, 256 Hz
					;		    for base = 32768 Hz
					;		2=61.035 us, 16384 Hz
					;		    for base > 32768 Hz
					;		 or 7.8125 ms, 128 Hz
					;		    for base = 32768 Hz
					;		3=122.070 us, 8192 Hz
					;		4=244.141 us, 4096 Hz
					;		5=488.281 us, 2048 Hz
					;	      * 6=976.562 us, 1024 Hz
					;		7=1.953125 ms, 512 Hz
					;		8=3.90625 ms, 256 Hz
					;		9=7.8125 ms, 128 Hz
					;		10=15.625 ms, 64 Hz
					;		11=31.25 ms, 32 Hz
					;		12=62.5 ms, 16 Hz
					;		13=125 ms, 8 Hz
					;		14=250 ms, 4 Hz
					;		15=500 ms, 2 Hz
					;	bit 4-6: 22 stage divider
					;		0=4.194304 MHz base
					;		1=1.048576 MHz base
					;		2=32768 Hz time base
					;		3-5=internal use, test
					;		6,7=none,reset divider
					;	bit 7: 1=time update cycle is
					;		in progress (read-only)
					;		(takes 248 or 1984 us)

RTC_UIP		EQU	B7		; RTC update cycle flag - status reg. A
                                                                  
CMOS_STATUS_B	EQU	0bh		; RTC status register B
					;	bit 0: daylight savings enable
					;		in April and October
					;	bit 1: 1=24 hour/0=12 hour mode
					;	bit 2: 0=BCD, 1=BIN
					;	bit 3: enab. square wave output
					;	bit 4: enable update-ended int.
					;	bit 5: enable alarm interrupt
					;	bit 6: enable periodic inter.
					;	bit 7: enable clock setting
					;		by freezing updates

RTC_BIN		EQU	B2		; BIN mode flag - status reg. B
RTC_AIE		EQU	B5		; alarm interrupt enable
RTC_PIE		EQU	B6		; periodic interrupt enabled
RTC_SET		EQU	B7		; RTC time set flag - status reg. B

CMOS_STATUS_C	EQU	0ch		; RTC status register C (read only)
					;	bit 0..3: unused
					;	bit 4: update-ended int. flag
					;	bit 5: alarm interrupt flag
					;	bit 6: periodic interrupt flag
					;	bit 7: interrupt request flag
					;	 (any int. request and enabled)

RTC_UF		EQU	B4		; update-ended interrupt flag
RTC_AF		EQU	B5		; alarm interrupt flag
RTC_PF		EQU	B6		; periodic interrupt flag
RTC_IRQF	EQU	B7		; interrupt request flag
					;    and interrupt is enabled

CMOS_STATUS_D	EQU	0dh		; RTC status register D (read only)
					;	bit 0..6: unused
					;	bit 7: 1=batery power good

CMOS_DIAG	EQU	0eh		; diagnostic status byte
					;	bit 0: time-out while reading
					;		adaptor ID
					;	bit 1: installed adaptors do
					;		not match configuration
					;	bit 2: time is invalid
					;	bit 3: controller or disk drive
					;		failed initialization
					;	bit 4: error in memory size
					;	bit 5: equipm. conf. incorrect
					;	bit 6: incorrect CMOS checksum
					;	bit 7: clock has lost power

CMOS_RESET	EQU	0fh		; reset code (shutdown status byte)
					;	0=software or unexpected reset
					;	1=reset after memory size
					;	  check (or real mode reentry)
					;	2=after successful memory test
					;	3=after failed memory test
					;	4=INT 19h reboot
					;	5=flush keyboard (EOI) and
					;	  jump via 40h:0067h
					;	6=reset (success. test) or jump
					;	  via 40h:0067h (without EOI)
					;	7=reset after failed test
					;	8=return to POST during
					;	   protected-mode RAM test
					;	9=block move for INT 15/87h
					;	10=resume by jump via 40h:0067h
					;	11=resume by IRET via 40h:0067h
					;	12=resume by RETF via 40h:0067h

CMOS_FLOPPY	EQU	10h		; floppy drive type
					;	bit 0-3: second floppy drive
					;	bit 4-7: first floppy drive
					;	  0=no drive
					;	  1=360 KB 5.25"
					;	  2=1.2 MB 5.25"
					;	  3=720 KB 3.5"
					;	  4=1.44 MB 3.5"
					;	  5=2.88 MB 3.5"

CMOS_REG_DEF	EQU	CMOS_STATUS_D	; default CMOS register

; ------------- Initialized CMOS device descriptor
; %1 = device index, %2 = flags, %3 = major version, %4 = minor version,
; %5 = build version, %6 = name prefix ###
;	###GetByte = read byte from CMOS memory
;	###SetByte = write byte to CMOS memory
;	###GetData = read data from CMOS memory
;	###SetData = write data to CMOS memory
;	###GetTime = read date and time from CMOS memory
;	###SetTime = write date and time to CMOS memory
;	###GetAlrm = read alarm from CMOS memory
;	###SetAlrm = write alarm to CMOS memory
;	###GetRate = get periodic interrupt rate (IRQ 8)
;	###SetRate = set periodic interrupt rate (IRQ 8)
;	###AckInt = acknowledge interrupt
;	###NextNMI = reassert NMI

%macro		DEVICECMOS 6

		DEVICE	DEVCLASS_SYS,DEV_SYS_CMOS,%1,%2,%3,%4,%5,%6

		dd	%6 %+ GetByte	; read byte from CMOS memory
		dd	%6 %+ SetByte	; write byte to CMOS memory
		dd	%6 %+ GetData	; read data from CMOS memory
		dd	%6 %+ SetData	; write data to CMOS memory
		dd	%6 %+ GetTime	; read date and time from CMOS memory
		dd	%6 %+ SetTime	; write date and time to CMOS memory
		dd	%6 %+ GetAlrm	; read alarm from CMOS memory
		dd	%6 %+ SetAlrm	; write alarm to CMOS memory
		dd	%6 %+ GetRate	; get periodic interrupt rate (IRQ 8)
		dd	%6 %+ SetRate	; set periodic interrupt rate (IRQ 8)
		dd	%6 %+ AckInt	; acknowledge interrupt
		dd	%6 %+ NextNMI	; reassert NMI

%endmacro
