; =============================================================================
;
;                            Litos - Keyboard driver
;
; =============================================================================

KEYBBUF_SIZE	EQU	64		; keyb. buffer size (DWORDs, hardcoded)

; ------------- Keyboard buffer

struc		KEYBBUF

		resb	SPINLOCK_size	; 0: keyboard buffer lock
KBUF_Write:	resb	1		; 4: write offset into input buffer
KBUF_ReadKey:	resb	1		; 5: key code read offset
KBUF_ReadChar:	resb	1		; 6: character read offset
KBUF_Cache:	resb	1		; 7: cache unread character
KBUF_Flags:	resb	1		; 8: flags (see below)
KBUF_Index:	resb	1		; 9: console index
		resb	2		; 0Ah: ...padding
KBUF_WKeyData:	resd	1		; 0Ch: private data for write function
KBUF_WKeyFnc:	resd	1		; 10h: virtual function to write
					;	key code into buffer
					;	INPUT:	EAX = key code
					;		EBX = private data
					;  It can be overwritten by other func.
KBUF_Buff:	resd	KEYBBUF_SIZE	; 14h: key buffer

endstruc				; size 114h = 276 bytes

; ------------- Keyboard buffer flags

KBUF_CACHE_BIT:	EQU	0		; cache contains unread character
KBUF_CACHE:	EQU	1 << KBUF_CACHE_BIT

; ------------- Keyboard driver flags

KEYB_SCROLL	EQU	B0		; Scroll Lock is on (LED hardcoded)
KEYB_NUM	EQU	B1		; Num Lock is on (LED hardcoded)
KEYB_CAPS	EQU	B2		; Caps Lock is on (LED hardcoded)
KEYB_INSERT	EQU	B3		; Insert is on
KEYB_SHIFT	EQU	B4		; Left or Right Shift is on
KEYB_CTRL	EQU	B5		; Left or Right Ctrl is on
KEYB_ALT	EQU	B6		; Left or Right Alt is on
KEYB_PAUSE	EQU	B7		; Pause is on
KEYB_EXT_BIT	EQU	8		; extended prefix E0
KEYB_EXT	EQU	1<<KEYB_EXT_BIT
KEYB_EXT2_BIT	EQU	9		; extended prefix E1
KEYB_EXT2	EQU	1<<KEYB_EXT2_BIT
KEYB_SEND_BIT	EQU	10		; sending to keyboard is running
KEYB_SENDING	EQU	1<<KEYB_SEND_BIT
KEYB_SENDSET	EQU	B11		; send keyboard setup request
KEYB_SENDLED	EQU	B12		; send LED state request
KEYB_SENDRATE	EQU	B13		; send typematic rate requset
KEYB_PRESENT	EQU	B15		; keyboard is present

KEYB_REQMASK	EQU	KEYB_SENDSET+KEYB_SENDLED+KEYB_SENDRATE ; request mask
KEYB_DEFAULT	EQU	KEYB_NUM+KEYB_PRESENT+KEYB_REQMASK ; default flags

; ------------- Key code

struc		KEYCODE

KEYCODE_Char:	resw	1		; 0: character or virtual code
KEYCODE_Scan:	resb	1		; 2: scan code (see below)
KEYCODE_Flags:	resb	1		; 3: flags (see below)

endstruc				; size 4 bytes (hardcoded)

; ------------- Key code flags

KEYCODE_SHIFT_BIT EQU	0		; Left or Right Shift is on
KEYCODE_CTRL_BIT EQU	1		; Left or Right Ctrl is on
KEYCODE_ALT_BIT	 EQU	2		; Left or Right Alt is on
KEYCODE_CH16_BIT EQU	5		; character is 16-bit (else 8-bit)
KEYCODE_CHAR_BIT EQU	6		; character code (else virtual)
KEYCODE_UP_BIT	 EQU	7		; key is released (flag is hardcoded)

KEYCODE_SHIFT	EQU	1<<KEYCODE_SHIFT_BIT; Left or Right Shift is on
KEYCODE_CTRL	EQU	1<<KEYCODE_CTRL_BIT ; Left or Right Ctrl is on
KEYCODE_ALT	EQU	1<<KEYCODE_ALT_BIT  ; Left or Right Alt is on
KEYCODE_CHAR16	EQU	1<<KEYCODE_CH16_BIT ; character is 16-bit (else 8-bit)
KEYCODE_CHAR	EQU	1<<KEYCODE_CHAR_BIT ; character code (else virtual)
KEYCODE_UP	EQU	1<<KEYCODE_UP_BIT   ; key is released

; ------------- Virtual key (key code as DWORD)

VIRTKEY_SHIFT_BIT EQU	KEYCODE_SHIFT_BIT+24 ; Left or Right Shift is on
VIRTKEY_CTRL_BIT EQU	KEYCODE_CTRL_BIT+24  ; Left or Right Ctrl is on
VIRTKEY_ALT_BIT	 EQU	KEYCODE_ALT_BIT+24   ; Left or Right Alt is on
VIRTKEY_CH16_BIT EQU	KEYCODE_CH16_BIT+24  ; character is 16-bit (else 8-bit)
VIRTKEY_CHAR_BIT EQU	KEYCODE_CHAR_BIT+24  ; character code (else virtual)
VIRTKEY_UP_BIT	 EQU	KEYCODE_UP_BIT+24    ; key is released

VIRTKEY_SHIFT	EQU	1<<VIRTKEY_SHIFT_BIT; Left or Right Shift is on
VIRTKEY_CTRL	EQU	1<<VIRTKEY_CTRL_BIT ; Left or Right Ctrl is on
VIRTKEY_ALT	EQU	1<<VIRTKEY_ALT_BIT  ; Left or Right Alt is on
VIRTKEY_CHAR16	EQU	1<<VIRTKEY_CH16_BIT ; character is 16-bit (else 8-bit)
VIRTKEY_CHAR	EQU	1<<VIRTKEY_CHAR_BIT ; character code (else virtual)
VIRTKEY_UP	EQU	1<<VIRTKEY_UP_BIT   ; key is released

VIRTKEY_MASK	EQU	0ff00ffffh	; virtual key mask (without scan code)

; ------------- Keyboard scan and virtual codes

KEY_NONE	EQU	0		; no key (or virtual character)
KEY_ESC		EQU	1		; Esc		(01 / 81)
KEY_1		EQU	2		; 1 !		(02 / 82)
KEY_2		EQU	3		; 2 @		(03 / 83)
KEY_3		EQU	4		; 3 #		(04 / 84)
KEY_4		EQU	5		; 4 $		(05 / 85)
KEY_5		EQU	6		; 5 %		(06 / 86)
KEY_6		EQU	7		; 6 ^		(07 / 87)
KEY_7		EQU	8		; 7 &		(08 / 88)
KEY_8		EQU	9		; 8 *		(09 / 89)
KEY_9		EQU	0ah		; 9 (		(0A / 8A)
KEY_0		EQU	0bh		; 0 )		(0B / 8B)
KEY_HYPHEN	EQU	0ch		; - _		(0C / 8C)
KEY_EQU		EQU	0dh		; = +		(0D / 8D)
KEY_BS		EQU	0eh		; Backspace	(0E / 8E)
KEY_TAB		EQU	0fh		; Tab		(0F / 8F)
KEY_Q		EQU	10h		; Q		(10 / 90)
KEY_W		EQU	11h		; W		(11 / 91)
KEY_E		EQU	12h		; E		(12 / 92)
KEY_R		EQU	13h		; R		(13 / 93)
KEY_T		EQU	14h		; T		(14 / 94)
KEY_Y		EQU	15h		; Y		(15 / 95)
					;	or virtual KEY_Z
KEY_U		EQU	16h		; U		(16 / 96)
KEY_I		EQU	17h		; I		(17 / 97)
KEY_O		EQU	18h		; O		(18 / 98)
KEY_P		EQU	19h		; P		(19 / 99)
KEY_LBRACKET	EQU	1ah		; [ {		(1A / 9A)
KEY_RBRACKET	EQU	1bh		; ] }		(1B / 9B)
KEY_ENTER	EQU	1ch		; Enter		(1C / 9C)
KEY_CTRL	EQU	1dh		; Left Ctrl	(1D / 9D)
KEY_A		EQU	1eh		; A		(1E / 9E)
KEY_S		EQU	1fh		; S		(1F / 9F)
KEY_D		EQU	20h		; D		(20 / A0)
KEY_F		EQU	21h		; F		(21 / A1)
KEY_G		EQU	22h		; G		(22 / A2)
KEY_H		EQU	23h		; H		(23 / A3)
KEY_J		EQU	24h		; J		(24 / A4)
KEY_K		EQU	25h		; K		(25 / A5)
KEY_L		EQU	26h		; L		(26 / A6)
KEY_SEMICOLON	EQU	27h		; ; :		(27 / A7)
KEY_SQUOTE	EQU	28h		; ' "		(28 / A8)
KEY_BACKQUOTE	EQU	29h		; ` ~		(29 / A9)
KEY_SHIFT	EQU	2ah		; Left Shift	(2A / AA)
KEY_BACKSLASH	EQU	2bh		; \ |		(2B / AB)
KEY_Z		EQU	2ch		; Z		(2C / AC)
					; 	or virtual KEY_Y
KEY_X		EQU	2dh		; X		(2D / AD)
KEY_C		EQU	2eh		; C		(2E / AE)
KEY_V		EQU	2fh		; V		(2F / AF)
KEY_B		EQU	30h		; B		(30 / B0)
KEY_N		EQU	31h		; N		(31 / B1)
KEY_M		EQU	32h		; M		(32 / B2)
KEY_COMMA	EQU	33h		; , <		(33 / B3)
KEY_PERIOD	EQU	34h		; . >		(34 / B4)
KEY_SLASH	EQU	35h		; / ?		(35 / B5)
KEY_RSHIFT	EQU	36h		; Right Shift	(36 / B6)
					;	virtual KEY_SHIFT
KEY_GREYSTAR	EQU	37h		; Gray [*]	(37 / B7)
KEY_ALT		EQU	38h		; Left Alt	(38 / B8)
KEY_SPACE	EQU	39h		; SpaceBar	(39 / B9)
KEY_CAPSLOCK	EQU	3ah		; CapsLock	(3A / BA)
KEY_F1		EQU	3bh		; F1		(3B / BB)
KEY_F2		EQU	3ch		; F2		(3C / BC)
KEY_F3		EQU	3dh		; F3		(3D / BD)
KEY_F4		EQU	3eh		; F4		(3E / BE)
KEY_F5		EQU	3fh		; F5		(3F / BF)
KEY_F6		EQU	40h		; F6		(40 / C0)
KEY_F7		EQU	41h		; F7		(41 / C1)
KEY_F8		EQU	42h		; F8		(42 / C2)
KEY_F9		EQU	43h		; F9		(43 / C3)
KEY_F10		EQU	44h		; F10		(44 / C4)
KEY_NUMLOCK	EQU	45h		; NumLock	(45 / C5)
KEY_SCROLL	EQU	46h		; ScrollLock	(46 / C6)
KEY_NUM7	EQU	47h		; [7 Home]	(47 / C7)
					;	virtual KEY_HOME
KEY_NUM8	EQU	48h		; [8 Up]	(48 / C8)
					;	virtual KEY_UP
KEY_NUM9	EQU	49h		; [9 PgUp]	(49 / C9)
					;	virtual KEY_PAGEUP
KEY_GREYMINUS	EQU	4ah		; Gray [-]	(4A / CA)
KEY_NUM4	EQU	4bh		; [4 Left]	(4B / CB)
					;	virtual KEY_LEFT
KEY_NUM5	EQU	4ch		; [5]		(4C / CC)
KEY_NUM6	EQU	4dh		; [6 Right]	(4D / CD)
					;	virtual KEY_RIGHT
KEY_GREYPLUS	EQU	4eh		; Gray [+]	(4E / CE)
KEY_NUM1	EQU	4fh		; [1 End]	(4F / CF)
					;	virtual KEY_END
KEY_NUM2	EQU	50h		; [2 Down]	(50 / D0)
					;	virtual KEY_DOWN
KEY_NUM3	EQU	51h		; [3 PgDn]	(51 / D1)
					;	virtual KEY_PAGEDOWN
KEY_NUM0	EQU	52h		; [0 Ins]	(52 / D2)
					;	virtual KEY_INSERT
KEY_DECIMAL	EQU	53h		; [. Del]	(53 / D3)
					;	virtual KEY_DELETE
KEY_SYSRQ	EQU	54h		; SysRq	    (PrtScr with Alt) (54 / D4)
KEY_BACKSLASH2	EQU	56h		; \ | alternative (56 / D6)
					;	virtual KEY_BACKSLASH
KEY_F11		EQU	57h		; F11		(57 / D7)
KEY_F12		EQU	58h		; F12		(58 / D8)
KEY_MESSENGER	EQU	5+B7		; Messenger	(E0 05 / E0 85)
KEY_EDITREDO	EQU	7+B7		; Edit Redo	(E0 07 / E0 87)
KEY_EDITUNDO	EQU	8+B7		; Edit Undo	(E0 08 / E0 88)
KEY_APPLEFT	EQU	9+B7		; Application Left (E0 09 / E0 89)
KEY_EDITPASTE	EQU	0ah+B7		; Edit Paste	(E0 0A / E0 8A)
KEY_SCRLNORM	EQU	0bh+B7		; Scroll Normal	(E0 0B / E0 8B)
					;  press=scroll up, release=scrool down
KEY_MEDPREV	EQU	10h+B7		; Media Prev	(E0 10 / E0 90)
KEY_SCRLFAST	EQU	11h+B7		; Scroll Fast	(E0 11 / E0 91)
					;  press=scroll up, release=scrool down
KEY_SCRLFASTER	EQU	12h+B7		; Scroll Faster	(E0 12 / E0 92)
					;  press=scroll up, release=scrool down
KEY_WORD	EQU	13h+B7		; Word		(E0 13 / E0 93)
KEY_EXCEL	EQU	14h+B7		; Excel		(E0 14 / E0 94)
KEY_CALENDAR	EQU	15h+B7		; Calendar	(E0 15 / E0 95)
KEY_LOGOFF	EQU	16h+B7		; Log Off	(E0 16 / E0 96)
KEY_EDITCUT	EQU	17h+B7		; Edit Cut	(E0 17 / E0 97)
KEY_EDITCOPY	EQU	18h+B7		; Edit Copy	(E0 18 / E0 98)
KEY_MEDNEXT	EQU	19h+B7		; Media Next	(E0 19 / E0 99)
KEY_NUMENTER	EQU	1ch+B7		; Gray [Enter]	(E0 1C / E0 9C)
KEY_RCTRL	EQU	1dh+B7		; Right Ctrl	(E0 1D / E0 9D)
					;	virtual KEY_CTRL
KEY_APPRIGHT	EQU	1eh+B7		; Application Right (E0 1E / E0 9E)
KEY_SCRLFASTEST	EQU	1fh+B7		; Scroll Fastest (E0 1F / E0 9F)
					;  press=scroll up, release=scrool down
KEY_VOLMUTE	EQU	20h+B7		; Volume Mute	(E0 20 / E0 A0)
KEY_CALCUL	EQU	21h+B7		; Calculator	(E0 21 / E0 A1)
KEY_MEDPLAY	EQU	22h+B7		; Media Play	(E0 22 / E0 A2)
KEY_SPELL	EQU	23h+B7		; Spell		(E0 23 / E0 A3)
KEY_MEDSTOP	EQU	24h+B7		; Media Stop	(E0 24 / E0 A4)
KEY_SHIFT2	EQU	2ah+B7		; fake Left Shift (E0 2A / E0 AA)
					;	virtual KEY_SHIFT
KEY_VOLDOWN	EQU	2eh+B7		; Volume-	(E0 2E / E0 AE)
KEY_VOLUP	EQU	30h+B7		; Volume+	(E0 30 / E0 B0)
KEY_WWWHOME	EQU	32h+B7		; WWW Home	(E0 32 / E0 B2)
KEY_DIVIDE	EQU	35h+B7		; Gray [/]	(E0 35 / E0 B5)
KEY_RSHIFT2	EQU	36h+B7		; fake Right Shift (E0 36 / E0 B6)
					;	virtual KEY_SHIFT
KEY_PRINT	EQU	37h+B7		; PrintScr  (E0 2A E0 37 / E0 B7 E0 AA)
					; or Ctrl+PrintScr (E0 37 / E0 B7)
KEY_RALT	EQU	38h+B7		; Right Alt	(E0 38 / E0 B8)
					;	virtual KEY_ALT
KEY_HELP	EQU	3bh+B7		; Help		(E0 3B / E0 BB)
KEY_MYMUSIC	EQU	3ch+B7		; My Music	(E0 3C / E0 BC)
KEY_TASK	EQU	3dh+B7		; Task Pane	(E0 3D / E0 BD)
KEY_FILENEW	EQU	3eh+B7		; File New	(E0 3E / E0 BE)
KEY_FILEOPEN	EQU	3fh+B7		; File Open	(E0 3F / E0 BF)
KEY_FILECLOSE	EQU	40h+B7		; File Close	(E0 40 / E0 C0)
KEY_EMAILREPLY	EQU	41h+B7		; Email Reply	(E0 41 / E0 C1)
KEY_EMAILFRWRD	EQU	42h+B7		; Email Forward	(E0 42 / E0 C2)
KEY_EMAILSEND	EQU	43h+B7		; Email Send	(E0 43 / E0 C3)
KEY_PAUSE	EQU	45h+B7		; Pause		(E1 1D 45 / E1 9D C5)
KEY_BREAK	EQU	46h+B7		; Break (Pause with Ctrl) (E0 46/E0 C6)
KEY_HOME	EQU	47h+B7		; Home		(E0 47 / E0 C7)
KEY_UP		EQU	48h+B7		; Up		(E0 48 / E0 C8)
KEY_PAGEUP	EQU	49h+B7		; Page Up	(E0 49 / E0 C9)
KEY_LEFT	EQU	4bh+B7		; Left		(E0 4B / E0 CB)
KEY_RIGHT	EQU	4dh+B7		; Right		(E0 4D / E0 CD)
KEY_END		EQU	4fh+B7		; End		(E0 4F / E0 CF)
KEY_DOWN	EQU	50h+B7		; Down		(E0 50 / E0 D0)
KEY_PAGEDOWN	EQU	51h+B7		; Page Down	(E0 51 / E0 D1)
KEY_INSERT	EQU	52h+B7		; Insert	(E0 52 / E0 D2)
KEY_DELETE	EQU	53h+B7		; Delete	(E0 53 / E0 D3)
KEY_FILESAVE	EQU	57h+B7		; File Save	(E0 57 / E0 D7)
KEY_FILEPRINT	EQU	58h+B7		; File Print	(E0 58 / E0 D8)
KEY_WIN		EQU	5bh+B7		; Left Win	(E0 5B / E0 DB)
KEY_RWIN	EQU	5ch+B7		; Right Win	(E0 5C / E0 DC)
					;	virtual KEY_WIN
KEY_WINMENU	EQU	5dh+B7		; Win Menu	(E0 5D / E0 DD)
KEY_POWER	EQU	5eh+B7		; Power		(E0 5E / E0 DE)
KEY_SLEEP	EQU	5fh+B7		; Sleep		(E0 5F / E0 DF)
KEY_WAKEUP	EQU	63h+B7		; Wake Up	(E0 63 / E0 E3)
KEY_MYPIC	EQU	64h+B7		; My Picture	(E0 64 / E0 E4)
KEY_FILESEARCH	EQU	65h+B7		; File Search	(E0 65 / E0 E5)
KEY_WWWFAVOR	EQU	66h+B7		; WWW Favorites	(E0 66 / E0 E6)
KEY_WWWREF	EQU	67h+B7		; WWW Refresh	(E0 67 / E0 E7)
KEY_WWWSTOP	EQU	68h+B7		; WWW Stop	(E0 68 / E0 E8)
KEY_WWWFWRD	EQU	69h+B7		; WWW Forward	(E0 69 / E0 E9)
KEY_WWWBACK	EQU	6ah+B7		; WWW Back	(E0 6A / E0 EA)
KEY_MYCOMP	EQU	6bh+B7		; My Computer	(E0 6B / E0 EB)
KEY_EMAIL	EQU	6ch+B7		; E-mail	(E0 6C / E0 EC)
KEY_MEDSEL	EQU	6dh+B7		; Media	Select	(E0 6D / E0 ED)

; ------------- Keyboard state scan codes (read from port 60h)

KEYSTATE_NONE	EQU	0		; no key code or keyboard error
KEYSTATE_TEST	EQU	0aah		; self-test complete OK
KEYSTATE_PREF	EQU	0e0h		; prefix
KEYSTATE_PREF2	EQU	0e1h		; prefix 2 (Pause/Break, no release)
KEYSTATE_ECHO	EQU	0eeh		; result of echo command
KEYSTATE_ACK	EQU	0fah		; acknowledge
KEYSTATE_FAILMF	EQU	0fch		; diagnostic failure (MF keyboard)
KEYSTATE_FAILAT	EQU	0fdh		; diagnostic failure (AT keyboard)
KEYSTATE_NACK	EQU	0feh		; send error, resend last code
KEYSTATE_ERROR	EQU	0ffh		; keyboard error/buffer full

; ------------- Keyboard commands (write it to port 60h)

KEYCMD_SETLED	EQU	0edh		; set LED, 1 byte follows:
					;	B0 = Scroll Lock is on
					;	B1 = Num Lock is on
					;	B2 = Caps Lock is on
KEYCMD_ECHO	EQU	0eeh		; diagnostic echo, returns 0eeh
KEYCMD_SETRATE	EQU	0f3h		; set typematic rate, 1 byte follows:
					;	B0..B4: typematic rate
					;	   0=30.0, 1=26.7,..31=2.0
					;	B5..B6: delay
					;	   0=250 ms,...3=1000 ms
KEYCMD_SCANMODE	EQU	0f0h		; set scan mode, 1 byte follows:
					;	0 = get keyboard scan mode
					;	1..3 = set scan mode 1..3
KEYCMD_ENABLE	EQU	0f4h		; keyboard enable
KEYCMD_DISABLE	EQU	0f5h		; keyboard disable
KEYCMD_RESET	EQU	0ffh		; keyboard reset
