; =============================================================================
;
;                           Litos - General device
;
; =============================================================================
; Driver can be installed via module "*Install" function and uninstalled with
; "*Uninstall" function. Driver module "*Entry" function detects and creates
; devices and initializes them using Drv_Init. Terminate with Drv_Deinit.
;
; Locking order:
;  1) driver list lock
;  2) device mutex lock
;  3) device spin-lock
; =============================================================================

; ------------- Device resource descriptor

struc		DEVRES

		resb	LIST_size	; 0: resource list
DEVRES_Start:	resd	1		; 8: start of resource
DEVRES_End:	resd	1		; 0Ch: end of resource
DEVRES_TypeFlagDW:			;   type and flags as DWORD
DEVRES_Type:	resb	1		; 10h: type of resource (see below)
DEVRES_Flags:	resb	1		; 11h: flags of resource (see below)
		resw	1		;   ...padding		
DEVRES_Name:	resb	TEXT_size	; 14h: name of resource

endstruc				; size 18h (=24) bytes

; ------------- Device resource type

DEVRES_IRQ	EQU	0		; resource type - IRQ (interrupt)
DEVRES_DMA	EQU	1		; resource type - DMA
DEVRES_PORT	EQU	2		; resource type - port
DEVRES_MEM	EQU	3		; resource type - memory

; ------------- Device resource flags

DEVRES_STATIC	EQU	B0		; static descriptor, don't delete it

; ------------- Macro - Initialized resource descriptor
; %1 = next link, %2 = previous link, %3 = start of resource,
; %4 = end of resource, %5 = resource type, %6 = resource flags
; %7 = pointer to name of resource CTEXTDATA

%macro		DEVRESOURCE 7
        	
		LINKEDLIST %1,%2	; resource list
		dd	%3		; start of resource
		dd	%4		; end of resource
		db	%5		; resource type
		db	%6		; resource flags
		dw	0		;  ...padding
		TEXTSTR	%7		; name of resource
%endmacro

; ------------- General device descriptor

struc		DEV

		resb	RBNODE_size	; 0: device address list
					;    (sorted by address)

DEV_Hash:	resb	HASHE_size	; 18h: hash list of this class

DEV_DevList:	resb	TREE_size	; 20h: device hierarchy list

DEV_Mutex:	resb	MUT_size	; 30h: device mutex lock

DEV_Lock:	resb	SPINLOCK_size	; 4Ch: device spin-lock

DEV_Resource:	resb	LIST_size	; 50h: device resource list DEVRES

DEV_Ref:	resd	1		; 58h: usage reference counter

DEV_IntList:	resd	1		; 5Ch: pointer to device interface list

DEV_ClassFlags:				;    class and flags as DWORD
DEV_Flags:	resb	1		; 60h: device flags (see below)
DEV_InxSubW:				;    index and sub-class as WORD
DEV_Index:	resb	1		; 61h: device index
DEV_SubClsW:				;    sub-class and class as WORD
DEV_SubClass:	resb	1		; 62h: device sub-class
DEV_Class:	resb	1		; 63h: device class (see below)

DEV_Version:				;    driver version (as DWORD)
DEV_Build:	resw	1		; 64h: build number of driver version
DEV_VerMin:	resb	1		; 66h: minor driver version
DEV_VerMaj:	resb	1		; 67h: major driver version

DEV_Name:	resb	TEXT_size	; 68h: device system name (empty=none)
DEV_Vendor:	resb	TEXT_size	; 6Ch: vendor name
DEV_Short:	resb	TEXT_size	; 70h: device short name
DEV_Full:	resd	1		; 74h: pointer to multi-language string
					;      array of full device name

DEV_Init:	resd	1		; 78h: initialize device
					;	INPUT:	EBX=device descriptor
					;	OUTPUT:	CY=error
					;	Use DevInit function (no lock)

DEV_Deinit:	resd	1		; 7Ch: deinitialize device
					;	INPUT:	EBX=device descriptor
					;	OUTPUT:	CY=error
					;	Use DevDeinit funct. (no lock)

endstruc				; size 80h = 128 bytes

; Device hash: Class AND mask
DEVHASH_SIZE	EQU	256		; size of device hash (hardcoded)
DEVHASH_MASK	EQU	DEVHASH_SIZE-1	; mask of device hash

; ------------- Supported device interface list

; Each entry is 4-byte DWORD (it may be text identifier or random number) which
; identifies supported interfaces. List is terminated with DEV_NUL_ID.

%define DEV_NUL_ID 0			; stop mark of device interface list
%define DEV_GEN_ID "DEV1"		; general device interface identifier

; ------------- Driver flags

DEV_STATIC	EQU	B0		; static descriptor, don't delete it
DEV_INIT_BIT	EQU	1		; bit of DEV_INIT flag
DEV_INIT	EQU	(1 << DEV_INIT_BIT) ; device has been initialized
DEV_ROOT	EQU	B2		; root device, system is its parent

; ------------- Device class

DEVCLASS_NONE	EQU	0		; unspecified class (reserved)
DEVCLASS_DRV	EQU	1		; general driver (without device)
DEVCLASS_SYS	EQU	2		; system
;DEVCLASS_KEYB	EQU	3		; keyboard
;DEVCLASS_MOUSE	EQU	4		; mouse
;DEVCLASS_JOY	EQU	5		; joystick
;DEVCLASS_DISP	EQU	6		; display
;DEVCLASS_PRINT	EQU	7		; printer
;DEVCLASS_PORT	EQU	8		; port
;DEVCLASS_MODEM	EQU	9		; modem
;DEVCLASS_MEDIA	EQU	10		; media card
;DEVCLASS_CTRL	EQU	11		; controller
;DEVCLASS_DISK	EQU	12		; disk
;DEVCLASS_TAPE	EQU	13		; tape
;DEVCLASS_NET	EQU	14		; network

; ------------- Device sub-class

DEVSCLASS_NONE	EQU	0		; unspecified sub-class (reserved)
DEVSCLASS_DEF	EQU	1		; default sub-class

; System DEVCLASS_SYS
DEV_SYS_SYS	EQU	2		; system kernel
DEV_SYS_DMA	EQU	3		; DMA controller
DEV_SYS_IRQ	EQU	4		; IRQ controller
DEV_SYS_CMOS	EQU	5		; CMOS memory
DEV_SYS_TIMER	EQU	6		; timer
DEV_SYS_CLOCK	EQU	7		; wall clock
;DEV_SYS_PCI	EQU	8		; PCI bus
;DEV_SYS_AGP	EQU	9		; AGP bus

; Keyboard DEVCLASS_KEYB
;DEV_KEYB_XT	EQU	2		; XT keyboard
;DEV_KEYB_AT	EQU	3		; AT keyboard
;DEV_KEYB_PS2	EQU	4		; keyboard PS/2

; Mouse DEVCLASS_MOUSE
;DEV_MOUSE_2	EQU	2		; 2-button mouse
;DEV_MOUSE_3	EQU	3		; 3-button mouse
;DEV_MOUSE_W	EQU	4		; wheel mouse

; Joystick DEVCLASS_JOY
;DEV_JOY_JOY	EQU	2		; standard joystick

; Display DEVCLASS_DISP
;DEV_DISP_TEXT	EQU	2		; text display
;DEV_DISP_MDA	EQU	3		; MDA monochrome card
;DEV_DISP_HGC	EQU	4		; Hercules graphic card
;DEV_DISP_CGA	EQU	5		; CGA graphic card
;DEV_DISP_EGA	EQU	6		; EGA graphic card
;DEV_DISP_MCGA	EQU	7		; MCGA graphic card
;DEV_DISP_VGA	EQU	8		; VGA graphic card
;DEV_DISP_SVGA	EQU	9		; SVGA graphic card
;DEV_DISP_3D	EQU	10		; 3D graphic card

; Printer DEVCLASS_PRINT
;DEV_PRINT_ASCII	EQU	2		; ASCII printer
;DEV_PRINT_ESC	EQU	3		; ESC sequence text printer
;DEV_PRINT_GRAPH	EQU	4		; graphic printer

; Port DEVCLASS_PORT
;DEV_PORT_COM	EQU	2		; COM port
;DEV_PORT_LPT	EQU	3		; LPT port
;DEV_PORT_USB	EQU	4		; USB port

; Modem DEVCLASS_MODEM
;DEV_MODEM_ASCII	EQU	2		; ASCII modem

; Media DEVCLASS_MEDIA
;DEV_MEDIA_SOUND	EQU	2		; sound card
;DEV_MEDIA_CAP	EQU	3		; capture card
;DEV_MEDIA_TUNER	EQU	4		; TV/FM tuner card
;DEV_MEDIA_CROSS	EQU	5		; crossbar
;DEV_MEDIA_CODEC	EQU	6		; codec

; Controller DEVCLASS_CTRL
;DEV_CTRL_FLOPPY	EQU	2		; floppy controller
;DEV_CTRL_IDE	EQU	3		; IDE controller
;DEV_CTRL_SCSI	EQU	4		; SCSI controller
;DEV_CTRL_SATA	EQU	5		; SATA controller

; Disk DEVCLASS_DISK
;DEV_DISK_FDD	EQU	2		; floppy disk
;DEV_DISK_HD	EQU	3		; hard disk
;DEV_DISK_RAM	EQU	4		; RAM disk
;DEV_DISK_CDROM	EQU	5		; CDROM or DVDROM

; Tape DEVCLASS_TAPE
;DEV_TAPE_TAPE	EQU	2		; tape

; Network DEVCLASS_NET
;DEV_NET_LOOP	EQU	2		; loop
;DEV_NET_PHONE	EQU	3		; phone adapter
;DEV_NET_NET	EQU	4		; NET card

; ------------- Initialized general device descriptor
; %1 = device class, %2 = device sub-class, %3 = device index, %4 = flags,
; %5 = major version, %6 = minor version, %7 = build version,
; %8 = name prefix ###
;	###Int = interface identifier list
;	###Res0 = head of resource list
;	###Res1 = first resource
;	###ResN = last resource
;	###Name = pointer to device system name CTEXTDATA
;	###Vendor = pointer to vendor name CTEXTDATA
;	###Short = pointer to device short name CTEXTDATA                   
;	###Full = pointer to multi-language string array of full device name
;	###Init = initialize device
;	###Deinit = deinitialize device

%macro		DEVICE	8

		RBTREENODE		; device address list
		HASHENTRY		; hash list
		TREEHEAD		; device hierarchy list
		MUTEX			; device mutex lock
		SPINLOCK		; device spin-lock
%8 %+ Res0:	LINKEDLIST %8 %+ Res1, %8 %+ ResN ; resource list

		dd	0		; usage reference counter

		dd	%8 %+ Int	; interface identifier list

		db	%4		; device flags
		db	%3		; device index
		db	%2		; device sub-class
		db	%1		; device class

		dw	%7		; build number
		db	%6		; minor version
		db	%5		; major version

		TEXTSTR %8 %+ Name	; device system name
		TEXTSTR	%8 %+ Vendor	; vendor name
		TEXTSTR	%8 %+ Short	; device short name
		dd	%8 %+ Full	; pointer to multi-language string
					;    array of full device name

		dd	%8 %+ Init	; initialize device
		dd	%8 %+ Deinit	; deinitialize device
%endmacro

; ------------- Macro - get total number of devices
; %1 = destination register, device list should be locked + int disabled

%macro		DEVNUMBER 1

		mov	%1,[DevAdrList+RBR_Count] ; get number of devices
%endmacro

; ------------- Macro - get parent of device (with jump)
; %1 = device descriptor, %2 = output register, %3 = jump if it has no parent
; Input and output registers may be identical.

%macro		DEVPARENT 3

		test	byte [%1+DEV_Flags],DEV_ROOT ; is it root device?
		jnz	%3		; it has no parent
		mov	%2,[%1+DEV_DevList+TREE_Parent] ; %2 <- parent
		sub	%2,DEV_DevList	; %2 <- driver head (it clears CF)
%endmacro

; ------------- Macro - find first device of given class (NULL = none)
; %1 = input BYTE register with device class, %2 = output register DEV
; %3 = jump if none device. Input and output registers may be identical.

%macro		DEVFINDCLASS 3

		movzx	%2,%1		; %2 <- device class
		mov	%2,[DevHashList+%2*4] ; %2 <- first entry
		or	%2,%2		; check if it is valid entry
		jz	%3		; invalid entry
		sub	%2,DEV_Hash	; %2 <- entry (it clears CF)
%endmacro

; ------------- Macro - get next device of this class (NULL = none)
; %1 = input register DEV, %2 = output register DEV
; %3 = jump if none device. Input and output registers may be identical.

%macro		DEVNEXTCLASS 3

		mov	%2,[%1+DEV_Hash+HASHE_Next] ; %2 <- next entry
		or	%2,%2		; check if it is valid entry
		jz	%3		; invalid entry
		sub	%2,DEV_Hash	; %2 <- entry (it clears CF)
%endmacro

; ------------- Macro - lock device list

%macro		DEVLSTLOCK 0

		LOCK_Lock DevListLock	; lock device list
%endmacro

; ------------- Macro - unlock device list (it saves flags)

%macro		DEVLSTUNLOCK 0

		LOCK_Unlock DevListLock ; unlock device list
%endmacro

; ------------- Macro - fast lock device (using spin-lock)
; EBX = pointer to device descriptor DEV

%macro		DEVLOCK 0

		LOCK_Lock ebx+DEV_Lock	; lock device
%endmacro

; ------------- Macro - fast try to lock device (using spin-lock)
; EBX = pointer to device descriptor DEV, returns CY = cannot lock

%macro		DEVTRYLOCK 0

		LOCK_TryLockC ebx+DEV_Lock ; try to lock device
%endmacro

; ------------- Macro - fast unlock device (using spin-lock, it saves flags)
; EBX = pointer to device descriptor DEV

%macro		DEVUNLOCK 0

		LOCK_Unlock ebx+DEV_Lock ; unlock device
%endmacro

; ------------- Macro - slow lock device (using mutex)
; EBX = pointer to device descriptor DEV

%macro		DEVMUTLOCK 0

		add	ebx,DEV_Mutex	; EBX <- mutex
		call	MutexLock	; lock mutex
		sub	ebx,DEV_Mutex	; return pointer
%endmacro

; ------------- Macro - slow try to lock device (using mutex)
; EBX = pointer to driver descriptor DRV, returns CY = already locked

%macro		DEVMUTTRYLOCK 0

		push	ebx		; push EBX
		add	ebx,DEV_Mutex	; EBX <- mutex
		call	MutexTryLock	; try to lock mutex
		pop	ebx		; pop EBX
%endmacro

; ------------- Macro - slow unlock device (using mutex,it does NOT save flags)
; EBX = pointer to device descriptor DEV

%macro		DEVMUTUNLOCK 0

		push	ebx		; push EBX
		add	ebx,DEV_Mutex	; EBX <- mutex
		call	MutexUnlock	; unlock mutex
		pop	ebx		; pop EBX
%endmacro

; ------------- Macro - slow unlock device (using mutex, it saves flags)
; EBX = pointer to device descriptor DEV

%macro		DEVMUTUNLOCKF 0

		pushf			; push flags
		push	ebx		; push EBX
		add	ebx,DEV_Mutex	; EBX <- mutex
		call	MutexUnlock	; unlock mutex
		pop	ebx		; pop EBX
		popf			; pop flags
%endmacro

; ------------- Macro - call device interface function without lock
; EBX = pointer to device DEV, %1 = function offset

%macro		DEVFNC	1

		call	dword [ebx+%1]	; call device function
%endmacro

; ------------- Macro - call device interface function with lock
; EBX = pointer to device DEV, %1 = function offset

%macro		DEVFNCLOCK 1

		pushf			; push flags
		cli			; disable interrupts
		DEVLOCK			; lock device
		call	dword [ebx+%1]	; call device function
		DEVUNLOCK		; unlock device
		popf			; pop flags
%endmacro

; ------------- Macro - call device function with lock and carry
; EBX = pointer to device DEV, %1 = function offset,

%macro		DEVFNCLOCKC 1

		clc			; clear error flag
		pushf			; push flags
		cli			; disable interrupts
		DEVLOCK			; lock device
		call	dword [ebx+%1]	; call device function
		adc	byte [esp],0	; set error flag (bit 0)
		DEVUNLOCK		; unlock device
		popf			; pop flags
%endmacro

; ------------- Macro - call device interface function with mutex lock
; EBX = pointer to device DEV, %1 = function offset

%macro		DEVFNCMUTLOCK 1

		DEVMUTLOCK		; mutex lock device
		call	dword [ebx+%1]	; call device function
		DEVMUTUNLOCK		; mutex unlock device
%endmacro

; ------------- Macro - call device function with mutex lock, carry
; EBX = pointer to device DEV, %1 = function offset

%macro		DEVFNCMUTLOCKC 1

		DEVMUTLOCK		; mutex lock device
		call	dword [ebx+%1]	; call device function
		DEVMUTUNLOCKF		; mutex unlock device
%endmacro
