; =============================================================================
;
;                             Litos - VGA display
;
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                         Initialize VGA driver
; -----------------------------------------------------------------------------

; ------------- Prepare to generate standard VGA palettes

VGAInit:	mov	edi,VGAStdPal	; EDI <- VGA standard palette table
		mov	ah,0		; AH <- 0, palette index
		xor	ecx,ecx		; ECX <- 0

; ------------- Prepare to generate one color (entries: red, green, blue)

VGAInit2:	mov	cl,3		; CL <- 3, number of color components

; ------------- 2/3 intensity (red 2/3 bit)

VGAInit3:	mov	al,0		; AL <- 0, accumulator
		test	ah,B2		; 2/3 intensity?
		jz	VGAInit4	; no
		mov	al,42		; AL <- 2/3 intensity

; ------------- 1/3 intensity (red 1/2 bit)

VGAInit4:	test	ah,B5		; 1/3 intensity?
		jz	VGAInit5	; no
		add	al,21		; AL <- 1/3 intensity

; ------------- Next color component

VGAInit5:	stosb			; store color component
		rol	ah,1		; AH <- rotate palette index
		loop	VGAInit3	; generate next color component

; ------------- Next color

		ror	ah,3		; AH <- return palette index
		inc	ah		; increase palette index
		jnz	VGAInit2	; generate next color

; ------------- Initialize VGA driver parameter block

		mov	ebx,VGADevDDPB	; EBX <- driver parameter block
		call	EGAInitDev	; initialize driver parameter block
		ret

; -----------------------------------------------------------------------------
;                              Set VGA palette
; -----------------------------------------------------------------------------
; INPUT:	AL = start index (0..255)
;		EBX = display driver
;		CL = stop index (0..255)
;		EDX = pointer to VGA palette (palette entry with start index)
; NOTES: VGA palettes are array of byte triples: red, green and blue color
;	 components with range 0 to 63.
; -----------------------------------------------------------------------------

; ------------- Push registers

VGASetPal:	push	eax		; push EAX
		push	edx		; push EDX
		push	esi		; push ESI
		mov	esi,edx		; ESI <- palettes

; ------------- Select starting color index

		mov	dx,3c8h		; DX <- palette index port
		out	dx,al		; select palette index
		dec	eax		; AL <- preset palette index - 1
		mov	ah,al		; AH <- palette index - 1

; ------------- Set color palette

		inc	edx		; DX <- palette data port
VGASetPal2:	lodsb			; AL <- palette value Red
		out	dx,al		; set palette value Red
		lodsb			; AL <- palette value Green
		out	dx,al		; set palette value Green
		lodsb			; AL <- palette value Blue
		out	dx,al		; set palette value Blue
		inc	ah		; increase register number
		cmp	ah,cl		; check register number
		jne	VGASetPal2	; set next register

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                              Set indexed palette
; -----------------------------------------------------------------------------
; INPUT:	EAX = index table (array of bytes with value 0 to 255)
;		EBX = display driver
;		ECX = number of color entries (1 to 255)
;		EDX = pointer to VGA palette
; NOTES: VGA palettes are array of byte triples: red, green and blue color
;	 components with range 0 to 63.
; -----------------------------------------------------------------------------

; ------------- Push registers

VGASetPalInx:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI	
		push	edi		; push EDI

; ------------- Prepare registers

		mov	esi,edx		; ESI <- palettes
		xchg	eax,edi		; EDI <- indexes
		mov	dx,3c8h		; DX <- palette index port

; ------------- Select color index

VGASetPalInx2:	mov	al,[edi]	; AL <- color index
		inc	edi		; increase index pointer
		out	dx,al		; select palette index
		inc	edx		; DX <- palette data port

; ------------- Set color palette

		lodsb			; AL <- palette value Red
		out	dx,al		; set palette value Red
		lodsb			; AL <- palette value Green
		out	dx,al		; set palette value Green
		lodsb			; AL <- palette value Blue
		out	dx,al		; set palette value Blue
		dec	edx		; EDX <- palette index port
		loop	VGASetPalInx2	; set next register

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

VGADevName:	CTEXTDATA 'VGA display'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- VGA display driver

		align	8, db 0
VGADevDDPB:	RBTREENODE		; red-black tree node
		SPINLOCK		; driver lock
		db	0;,DEV_OUT_GRAPH	; index, class and subclass
		db	DEV_STATIC,0	; flags and class flags
		db	0,0,0,1		; driver version
;		dd	DevVendorName	; pointer to vendor name
		dd	VGADevName	; pointer to driver name
		EMPTYTEXT		; pointer to model name
		EMPTYTEXT		; modul path
		dd	VGADevDDFBText	; pointer to function table
VGADevDDPB1:	LINKEDLIST VGADevDDPB2,VGADevDDPB2 ; resource list

		dd	DDPB_CURON+DDPB_CANTEXT ; flags
		dd	8000h		; memory size
		dd	0,0		; cursor position
		dd	14,15		; cursor size
		dd	0,0		; current display offset
		dd	SYSTEM_ADDR+0b8000h ; memory address
		dd	EGADevUniMap	; mapping from Unicode
		dd	EGAFontBuff	; font buffer (128*32 = 4 KB)
		dd	0		; ...padding

		dd	80,25*4		; virtual dimension
		dd	80,25		; display dimension
		dd	8,16		; font dimension
		dd	60		; vertical frequency
		db	VMODE_TEXTCOLOR	; memory mode
		db	16		; total bits per point
		db	16		; bits in one plane
		db	1		; color planes
		dd	SYSTEM_ADDR+0b8000h ; memory base address
		dd	80*25*2		; size of display page
		dd	80*25*2*3	; maximal offset
		dd	80*2		; bytes per scan line
		dd	0		; size of one plane
		dd	3d4h		; control port
		times	DISPMAXREG db -1 ; cache of display registers

		align	4, db 0
VGADevDDPB2:	LINKEDLIST VGADevDDPB1,VGADevDDPB1 ; resource list
		dd	SYSTEM_ADDR+0a0000h ; start of resource
; !!!!!!!!!!!!!!!!!!!!!!!!
		dw	20000h/PAGE_SIZE-1 ; size of resource-1
		db	DEVRES_MEM		; resource type
		db	DEVRES_STATIC;+RES_AUTO+RES_PAGES ; flags

; ------------- VGA display interface DDFB - text mode

		align	4, db 0
VGADevDDFBText:;	dd	DevStdFuncOK	; device detection
	;	dd	DevStdFuncOK	; device initialization
	;	dd	DevStdFuncERR	; device deinitialization
	;	dd	DevStdFuncERR	; enable device
	;	dd	DevStdFuncERR	; disable device

		dd	TXTEnumMode	; enumerate videomodes
		dd	TXTTestMode	; test videomode
		dd	TXTSetMode	; set videomode
		dd	TXTClear	; clear screen
		dd	TXTSetOffset	; set memory offset
		dd	TXTSetCursor	; set cursor position
		dd	TXTSetCurSize	; set cursor size
		dd	TXTSetVisible	; set cursor visible
		dd	EGALoadFont	; load text font
		dd	EGASetFont	; set font bank
		dd	TXTSetBorder	; set border color
		dd	TXTSetPalCGA	; set CGA palette
		dd	EGASetPal	; set EGA palette
		dd	VGASetPal	; set VGA palette
		dd	VGASetPalInx	; set indexed palette
		dd	TXTFillUp	; fill-up region
		dd	TXTMove		; move region
		dd	TXTBufferSize	; get buffer size
		dd	TXTGetRegion	; get region
		dd	TXTSetRegion	; set region
		dd	TXTSetRegCol	; set text region with color
		dd	TXTSetColMask	; set region with color mask
		dd	TXTSetMasked	; set region with mask
		dd	TXTSetAlpha	; set region with alpha

; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- Standard VGA palettes (red, green, blue, range 0 to 63)

		align	4, resb 1
VGAStdPal:	resd	3*256		; standard VGA palettes
