; =============================================================================
;
;                Litos - Programmable interval timer Intel 8254
;
; =============================================================================

		CODE_SECTION

TIMER_BASE	EQU	40h		; TIMER base register
TIMER_CTRL	EQU	TIMER_BASE+3	; TIMER control register

; *****************************************************************************
;
;                           Driver interface functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Install TIMER device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

TIMERInstall:	push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Register TIMER device

		mov	ebx,TIMERDev	; EBX <- device descriptor
		xor	ecx,ecx		; ECX <- 0, no parent device
		call	DevRegister	; register device

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                           Uninstall TIMER device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

TIMERUninstall:	push	ebx		; push EBX

; ------------- Unregister IRQ device

		mov	ebx,TIMERDev	; EBX <- device descriptor
		call	DevUnregister	; unregister device

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize TIMER device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVTIMER
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

TIMERDevInit:	clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Deinitialize TIMER device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVTIMER
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

TIMERDevDeinit:	clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize counter
; -----------------------------------------------------------------------------
; INPUT:	EAX = timer mode (DEVTIM_MODE_CNT, ...)
;		EBX = device descriptor DEVTIMER
;		ECX = counter initial value (1 to 65536, 0 stands for 65536)
;		EDX = counter index (0 to 2)
; OUTPUT:	CY = error, invalid arguments
; NOTES:	On mode DEVTIM_MODE_GEN value of 1 is illegal (not checked).
;		It takes aprox. 7 us.
; -----------------------------------------------------------------------------

; ------------- Check arguments

TIMERDevSetMode:cmp	eax,DEVTIM_MODE_MAX ; check mode
		ja	TIMERDevSetMod9	; invalid mode
		cmp	ecx,65536	; check maximal initial value
		ja	TIMERDevSetMod9	; invalid initial value
		cmp	edx,2		; check counter index
		ja	TIMERDevSetMod9	; invalid counter index

; ------------- Push registers

		push	eax		; push EAX
		push	edx		; push EDX

; ------------- Prepare control word -> AL

		shl	eax,1		; AL = timer mode << 1, BIN mode
		or	al,B5+B4	; set 2-byte mode
		mov	ah,dl		; AH <- counter index
		shl	ah,6		; AH <- shift to bits 6-7
		or	al,ah		; AH <- add counter index
		
; ------------- Set counter mode

		out	TIMER_CTRL,al	; set counter mode
		SHORT_DELAY		; short delay

; ------------- Prepare counter port -> EDX (it sets NC)

TIMERDevSetMod4:add	dl,TIMER_BASE	; EDX <- counter port

; ------------- Set initial value LOW

		mov	al,cl		; AL <- initial value LOW
		out	dx,al		; set initial value LOW
		SHORT_DELAY		; short delay
		
; ------------- Set initial value HIGH

		mov	al,ch		; AL <- initial value HIGH
		out	dx,al		; set initial value HIGH

; ------------- Pop registers (here is NC)

		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; ------------- Error

TIMERDevSetMod9:stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                 Driver function: Set new counter initial value
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVTIMER
;		ECX = new counter init. value (1 to 65536, 0 stands for 65536)
;		EDX = counter index (0 to 2)
; OUTPUT:	CY = error, invalid arguments
; NOTES:	On mode DEVTIM_MODE_GEN value of 1 is illegal (not checked).
;		It takes aprox. 4 us.
; -----------------------------------------------------------------------------

; ------------- Check arguments

TIMERDevSetVal:	cmp	ecx,65536	; check maximal initial value
		ja	TIMERDevSetMod9	; invalid initial value
		cmp	edx,2		; check counter index
		ja	TIMERDevSetMod9	; invalid counter index

; ------------- Push registers

		push	eax		; push EAX
		push	edx		; push EDX
		jmp	short TIMERDevSetMod4

; -----------------------------------------------------------------------------
;                 Driver function: Get counter current value
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVTIMER
;		EDX = counter index (0 to 2)
; OUTPUT:	EAX = current counter value (0 on error)
;		CY = error, invalid arguments (EAX=0)
; NOTES:	It takes aprox. 7 us.
; -----------------------------------------------------------------------------

; ------------- Check arguments

TIMERDevGetVal:	cmp	edx,2		; check counter index
		ja	TIMERDevGetVal9	; invalid counter index

; ------------- Push registers

		push	edx		; push EDX

; ------------- Prepare control word -> EAX

		mov	eax,edx		; EAX <- counter index
		shl	eax,6		; EAX <- shift to bits 6-7
		
; ------------- Set counter latch command

		out	TIMER_CTRL,al	; set counter latch command
		SHORT_DELAY		; short delay

; ------------- Prepare counter port -> EDX (it sets NC)

		add	dl,TIMER_BASE	; EDX <- counter port

; ------------- Get current value LOW

		in	al,dx		; AL <- get current value LOW
		SHORT_DELAY		; short delay
		mov	ah,al		; AH <- current value LOW
		
; ------------- Get current value HIGH

		in	al,dx		; AL <- get current value HIGH
		xchg	al,ah		; AL <- value LOW, AH <- value HIGH

; ------------- Pop registers (here is NC)

		pop	edx		; pop EDX
		ret

; ------------- Error

TIMERDevGetVal9:xor	eax,eax		; EAX <- 0
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                    Driver function: Get counter output
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVTIMER
;		EDX = counter index (0 to 2)
; OUTPUT:	EAX = current counter output (0 or 1)
;		CY = error, invalid arguments (EAX=0)
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

; ------------- Check arguments

TIMERDevGetOut:	cmp	edx,2		; check counter index
		ja	TIMERDevGetVal9	; invalid counter index

; ------------- Push registers

		push	edx		; push EDX

; ------------- Set counter LATCH STATUS command

		mov	al,B1		; AL <- prepare bit of COUNTER 0
		xchg	ecx,edx		; CL <- counter index
		shl	al,cl		; AL <- counter bit
		xchg	ecx,edx		; ECX, EDX <- return old value
		or	al,B7+B6+B5	; prepare LATCH STATUS command
		out	TIMER_CTRL,al	; set LATCH STATUS command
		SHORT_DELAY		; short delay

; ------------- Get current status

		xor	eax,eax		; EAX <- 0
		add	dl,TIMER_BASE	; EDX <- counter port
		in	al,dx		; AL <- get current status
		or	al,al		; check bit 7 (= OUT pin value)
		sets	al		; EAX <- 1 if OUT is "1", else 0

; ------------- Pop registers (here is NC)

		pop	edx		; pop EDX
		ret

; *****************************************************************************
;
;                            Timer global functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                   TIMER global function: Initialize counter
; -----------------------------------------------------------------------------
; INPUT:	EAX = timer mode (DEVTIM_MODE_CNT, ...)
;		ECX = counter initial value (1 to 65536, 0 stands for 65536)
;		EDX = counter index (0 to 2)
; OUTPUT:	CY = error, invalid arguments
; NOTES:	On mode DEVTIM_MODE_GEN value of 1 is illegal (not checked).
;		It takes aprox. 7 us.
; -----------------------------------------------------------------------------

TIMERSetMode:  	push	ebx		; push EBX
		mov	ebx,[TIMERDevice] ; EBX <- current TIMER device
		DEVFNCLOCKC DEVTIM_SetMode ; initialize counter
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;              TIMER global function: Set new counter initial value
; -----------------------------------------------------------------------------
; INPUT:	ECX = new counter init. value (1 to 65536, 0 stands for 65536)
;		EDX = counter index (0 to 2)
; OUTPUT:	CY = error, invalid arguments
; NOTES:	On mode DEVTIM_MODE_GEN value of 1 is illegal (not checked).
;		It takes aprox. 4 us.
; -----------------------------------------------------------------------------

TIMERSetVal:  	push	ebx		; push EBX
		mov	ebx,[TIMERDevice] ; EBX <- current TIMER device
		DEVFNCLOCKC DEVTIM_SetVal ; set new counter initial value
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               TIMER global function: Get counter current value
; -----------------------------------------------------------------------------
; INPUT:	EDX = counter index (0 to 2)
; OUTPUT:	EAX = current counter value (0 on error)
;		CY = error, invalid arguments (EAX=0)
; NOTES:	It takes aprox. 7 us.
; -----------------------------------------------------------------------------

TIMERGetVal: 	push	ebx		; push EBX
		mov	ebx,[TIMERDevice] ; EBX <- current TIMER device
		DEVFNCLOCKC DEVTIM_GetVal ; get counter current value
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                 TIMER global function: Get counter output
; -----------------------------------------------------------------------------
; INPUT:	EDX = counter index (0 to 2)
; OUTPUT:	EAX = current counter output (0 or 1)
;		CY = error, invalid arguments (EAX=0)
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

TIMERGetOut: 	push	ebx		; push EBX
		mov	ebx,[TIMERDevice] ; EBX <- current TIMER device
		DEVFNCLOCKC DEVTIM_GetOut ; get counter current value
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Text strings

		align	4, db 0
TIMERDevName:	CTEXTDATA 'timer'
TIMERDevShort:	CTEXTDATA 'Timer 8254'

TIMERDevFull:	LANGTEXTSTR TIMERDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
 		LANGTEXTSTR TIMERDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

TIMERDevFullEN:	CTEXTDATA 'Programmable interval timer Intel 8254'
TIMERDevFullCZ:	CTEXTDATA 'Programovateln',0c3h,0bdh,' ',0c4h,8dh,'asova', \
				0c4h,8dh,' Intel 8254'

TIMERDevInt:	dd	DEV_GEN_ID
		dd	DEV_TIMER_ID
		dd	DEV_NUL_ID

TIMERDevRes1Name:CTEXTDATA 'ctrl'
TIMERDevRes2Name:CTEXTDATA 'irq'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- TIMER device descriptor

		align	4, db 0
TIMERDevice:	dd	TIMERDev		; current TIMER device descr.

%define		TIMERDevVendor  DefaultVendor

		align	8, db 0
TIMERDev:	DEVICETIMER DEV_STATIC,1,0,0,TIMERDev

TIMERDevRes1:	DEVRESOURCE TIMERDevResN,TIMERDevRes0,TIMER_BASE,TIMER_BASE+3, \
		 DEVRES_PORT,DEVRES_STATIC,TIMERDevRes1Name

TIMERDevResN:	DEVRESOURCE TIMERDevRes0,TIMERDevRes1,TIMER_IRQ,TIMER_IRQ, \
		 DEVRES_IRQ,DEVRES_STATIC,TIMERDevRes2Name
