; =============================================================================
;
;                      Litos - Interrupt controller 8259A
;
; =============================================================================
; Default IRQ assignments on PC/AT:
;  Master 8259:
;	IRQ 0: programmable interval timer (PIT)
;	IRQ 1: keyboard, mouse, RTC interrupt
;	IRQ 2: ....cascade to slave 8259 INT line (+video interrupt)
;	IRQ 3: serial port COM2 and COM4
;	IRQ 4: serial port COM1 and COM3
;	IRQ 5: LPT2, sound card, fixed disk controller
;	IRQ 6: floppy disk controller
;	IRQ 7: LPT1 (or sound card, older usage)
;  Slave 8259:
;	IRQ 8: real-time clock (RTC)
;	IRQ 9: MPU-401 MID port, ACPI SCI, redirect cascade
;	IRQ 10:
;	IRQ 11:
;	IRQ 12: PS/2 mouse
;	IRQ 13: math coprocessor
;	IRQ 14: hard disk controller 1
;	IRQ 15: hard disk controller 2
; =============================================================================

		CODE_SECTION

IRQ_CHANNELS	EQU	16		; number of IRQ channels
IRQ_CHANNEL2	EQU	8		; first channel of slave controller
IRQ_CASCADE	EQU	2		; cascade IRQ channel (on controller 1)

; ------------- Ports - controller 1 (master)

IRQ1_BASE	EQU	20h		; base port of controller 1
IRQ1_CMD	EQU	IRQ1_BASE+0	; command register WRITE (controller 1)
IRQ1_REQ	EQU	IRQ1_BASE+0	; request register READ (controller 1)
IRQ1_IMR	EQU	IRQ1_BASE+1	; interrupt mask register (contr. 1)

; ------------- Ports - controller 2 (slave)

IRQ2_BASE	EQU	0a0h		; base port of controller 2
IRQ2_CMD	EQU	IRQ2_BASE+0	; command register WRITE (controller 2)
IRQ2_REQ	EQU	IRQ2_BASE+0	; request register READ (controller 2)
IRQ2_IMR	EQU	IRQ2_BASE+1	; interrupt mask register (contr. 2)

; -----------------------------------------------------------------------------
;                    Internal function: Enable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number (0 to 15)
;		EBX = device descriptor DEVIRQ
; NOTES:	EAX not checked for validity
;		IRQ device must be locked.
; -----------------------------------------------------------------------------

; ------------- Clear IRQ bit

IRQDev0Enable:	btr	dword [ebx+DEVIRQ_Cache],eax ; reset IRQ bit (=enable)
		jnc	IRQDev0Enable8	; IRQ is already enabled

; ------------- Push registers

IRQDev0Enable2:	push	eax		; push EAX

; ------------- Check if use controller 2

		cmp	al,IRQ_CHANNEL2	; use controller 2?
		mov	eax,[ebx+DEVIRQ_Cache] ; EAX <- current mask
		jae	IRQDev0Enable4	; use controller 2

; ------------- Send mask to controller 1 (MASTER)

		out	IRQ1_IMR,al	; send interrupt mask 1

; ------------- Pop registers

		pop	eax		; pop EAX
		ret

; ------------- Send mask to controller 2 (SLAVE)

IRQDev0Enable4:	mov	al,ah		; AL <- interrupt mask 2
		out	IRQ2_IMR,al	; send interrupt mask 2

; ------------- Pop registers

		pop	eax		; pop EAX
IRQDev0Enable8:	ret

; -----------------------------------------------------------------------------
;                   Internal function: Disable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number (0 to 15)
;		EBX = device descriptor DEVIRQ
; NOTES:	EAX not checked for validity
;		IRQ device must be locked.
; -----------------------------------------------------------------------------

; ------------- Set IRQ bit

IRQDev0Disable:	bts	dword [ebx+DEVIRQ_Cache],eax ; set IRQ bit (=disable)
		jnc	short IRQDev0Enable2 ; disable IRQ
		ret

; -----------------------------------------------------------------------------
;                Internal function: Check if interrupt is pending
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number (0 to 15)
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = interrupt is pending
; NOTES:	EAX not checked for validity
;		IRQ device must be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDev0Pending:	push	ecx		; push ECX

; ------------- Check if use controller 2

		cmp	al,IRQ_CHANNEL2	; use controller 2?
		xchg	eax,ecx		; ECX <- push IRQ number
		mov	al,B3+2		; OCW3: read request register
		jae	IRQDev0Pending5	; use controller 2

; ------------- Get request mask from controller 1 (MASTER)

		btr	dword [ebx+DEVIRQ_Flags],DEVIRQ_InSrv1_b ; reset flag
		jc	IRQDev0Pending3	; register is in-service mode
IRQDev0Pending2:in	al,IRQ1_REQ	; get request mask 1
		jmp	short IRQDev0Pending8

; ------------- Set pending mode

IRQDev0Pending3:out	IRQ1_CMD,al	; switch to request register
		jmp	short IRQDev0Pending2 ; short delay

IRQDev0Pending4:out	IRQ2_CMD,al	; switch to request register
		jmp	short IRQDev0Pending6 ; short delay

; ------------- Get request mask from controller 2 (SLAVE)

IRQDev0Pending5:btr	dword [ebx+DEVIRQ_Flags],DEVIRQ_InSrv2_b ; reset flag
		jc	IRQDev0Pending4	; register is in-service mode
IRQDev0Pending6:in	al,IRQ2_REQ	; get request mask 2
		mov	ah,al		; AH <- request mask 2

; ------------- Check if IRQ is pending

IRQDev0Pending8:bt	eax,ecx		; check if IRQ is pending

; ------------- Pop registers

		xchg	eax,ecx		; pop EAX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;             Internal function: Check if interrupt is in-service
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number (0 to 15)
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = interrupt is in-service
; NOTES:	EAX not checked for validity
;		IRQ device must be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDev0InServ:	push	ecx		; push ECX

; ------------- Check if use controller 2

		cmp	al,IRQ_CHANNEL2	; use controller 2?
		xchg	eax,ecx		; ECX <- push IRQ number
		mov	al,B3+3		; OCW3: read in-service register
		jae	IRQDev0InServ5	; use controller 2

; ------------- Get in-service mask from controller 1 (MASTER)

		bts	dword [ebx+DEVIRQ_Flags],DEVIRQ_InSrv1_b ; set flag
		jnc	IRQDev0InServ3	; register is not in-service mode
IRQDev0InServ2:	in	al,IRQ1_REQ	; get in-service mask 1
		jmp	short IRQDev0InServ8

; ------------- Set in-service mode

IRQDev0InServ3:	out	IRQ1_CMD,al	; switch to in-service register
		jmp	short IRQDev0InServ2 ; short delay

IRQDev0InServ4:	out	IRQ2_CMD,al	; switch to in-service register
		jmp	short IRQDev0InServ6 ; short delay

; ------------- Get in-service mask from controller 2 (SLAVE)

IRQDev0InServ5:	bts	dword [ebx+DEVIRQ_Flags],DEVIRQ_InSrv2_b ; set flag
		jnc	IRQDev0InServ4	; register is not in-service mode
IRQDev0InServ6:	in	al,IRQ2_REQ	; get in-service mask 2
		mov	ah,al		; AH <- in-service mask 2

; ------------- Check if IRQ is in-service

IRQDev0InServ8:	bt	eax,ecx		; check if IRQ is in-service

; ------------- Pop registers

		xchg	eax,ecx		; pop EAX
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                 Internal function: Acknowledge interrupt
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number (0 to 15)
;		EBX = device descriptor DEVIRQ (currently not used)
; NOTES:	EAX not checked for validity
;		Interrupt should be disabled first.
;		IRQ device must be locked.
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDev0Ack:	push	eax		; push EAX

; ------------- Check if use controller 2 (SLAVE)

		cmp	al,IRQ_CHANNEL2	; use controller 2?
		jb	IRQDev0Ack4	; use controller 1 (MASTER)

; ------------- Acknowledge interrupt to controller 2 (using specific EOI)

		add	al,(3 << 5) - IRQ_CHANNEL2 ; AL <- OCW2 command 60h
		out	IRQ2_CMD,al	; acknowledge interrupt to SLAVE

; ------------- Acknowledge interrupt to controller 1 (using specific EOI)

		mov	al,IRQ_CASCADE	; AL <- IRQ 2 (cascade)
IRQDev0Ack4:	add	al,(3 << 5)	; AL <- OCW2 command 60h
		out	IRQ1_CMD,al	; acknowledge interrupt to MASTER

; ------------- Pop registers

		pop	eax		; pop EAX
		ret

; *****************************************************************************
;
;                           Driver interface functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Install IRQ device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQInstall:	push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Register IRQ device

		mov	ebx,IRQDev	; EBX <- device descriptor
		xor	ecx,ecx		; ECX <- 0, no parent device
		call	DevRegister	; register device

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                           Uninstall IRQ device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQUninstall:	push	ebx		; push EBX

; ------------- Unregister IRQ device

		mov	ebx,IRQDev	; EBX <- device descriptor
		call	DevUnregister	; unregister device

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               Driver function: Remap base interrupt address
; -----------------------------------------------------------------------------
; INPUT:	EAX = new base interrupt (0 to 248, must be multiple of 8)
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = error (invalid base interrupt)
; NOTES:	All interrupts should be disabled first.
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDevRemap:	push	eax		; push EAX

; ------------- Check base interrupt (0 to 255, must be multiple of 8)

		test	eax,~0f8h	; check valid bits
		stc			; set error flag
		jnz	IRQDevRemap9	; invalid base interrupt
		mov	[ebx+DEVIRQ_Base],eax ; set new base interrupt
		mov	ah,al		; AH <- push base interrupt

; ------------- Initialize flags

		mov	byte [ebx+DEVIRQ_Flags],0 ; not in-service

; ------------- Disable all interrupts

		mov	al,0ffh		; AL <- interrupt mask
		out	IRQ2_IMR,al	; disable all interrupts on SLAVE
		out	IRQ1_IMR,al	; disable all interrupts on MASTER

; ------------- Set MASTER control word ICW1
; ICW4 needed, cascade mode, interval of 8, edge triggered, A5..A7=0

		mov	al,B0+B4	; ICW1: ICW1 and ICW4 flags
		out	IRQ1_CMD,al	; set ICW1

; ------------- Set MASTER control word ICW2 (base interrupt address)

		mov	al,ah		; ICW2: base interrupt address
		out	IRQ1_IMR,al	; set ICW2

; ------------- Set MASTER control word ICW3 (cascade mask)

		mov	al,(1 << IRQ_CASCADE) ; ICW3: slave attached to line 2
		out	IRQ1_IMR,al	; set ICW3

; ------------- Set MASTER control word ICW4
; 8086 mode, normal EOI (not auto), non-buffered mode, not fully nested mode

		mov	al,B0		; ICW4: x86 mode, normal EOI
		out	IRQ1_IMR,al	; set ICW4

; ------------- Set MASTER control word OCW3
; Read interrupt request register, no pool command, no special mask

		mov	al,B3+2		; switch to interrupt request register
		out	IRQ1_CMD,al	; set OCW3 (IRR register)

; ------------- Set SLAVE control word ICW1
; ICW4 needed, cascade mode, interval of 8, edge triggered, A5..A7=0

		mov	al,B0+B4	; ICW1: ICW1 and ICW4 flags
		out	IRQ2_CMD,al	; set ICW1

; ------------- Set SLAVE control word ICW2 (base interrupt address)

		mov	al,ah		; ICW2: base interrupt address
		add	al,IRQ_CHANNEL2	; ICW2: base interrupt address
		out	IRQ2_IMR,al	; set ICW2

; ------------- Set SLAVE control word ICW3 (cascade number)

		mov	al,IRQ_CASCADE	; ICW3: it is slave on line 2
		out	IRQ2_IMR,al	; set ICW3

; ------------- Set SLAVE control word ICW4
; 8086 mode, normal EOI (not auto), non-buffered mode, not fully nested mode

		mov	al,B0		; ICW4: x86 mode, normal EOI
		out	IRQ2_IMR,al	; set ICW4

; ------------- Set SLAVE control word OCW3
; Read interrupt request register, no pool command, no special mask

		mov	al,B3+2		; switch to interrupt request register
		out	IRQ2_CMD,al	; set OCW3 (IRR register)

; ------------- Restore current interrupt mask

		mov	eax,[ebx+DEVIRQ_Cache] ; AX <- current master mask
		out	IRQ1_IMR,al	; set master mask
		mov	al,ah		; AL <- slave mask
		out	IRQ2_IMR,al	; set slave mask

; ------------- Acknowledge all interrupts (set OCW2)

		mov	al,B5		; AL <- non-specific EOI
		out	IRQ1_CMD,al	; acknowledge interrupts on slave
		out	IRQ2_CMD,al	; acknowledge interrupts on master

; ------------- Pop registers

		clc			; clear error flag
IRQDevRemap9:	pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize IRQ device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVIRQ
; OUTPUT:	CY = error
; NOTES:	Base interrupt is validated into valid value.
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDevInit:	push	eax		; push EAX
		push	ecx		; push ECX

; ------------- Clear mask

		mov	dword [ebx+DEVIRQ_Cache],~(1 << IRQ_CASCADE)

; ------------- Initialize controllers (and remap base interrupt)

		mov	eax,[ebx+DEVIRQ_Base] ; EAX <- current base interrupt
		and	eax,0f8h	; validate base inteerrupt
		call	IRQDevRemap	; remap base interrupt

; ------------- Pop registers (here should be NC)

		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Deinitialize IRQ device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVIRQ
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

IRQDevDeinit:	push	eax		; push EAX

; ------------- Disable all interrupts

		mov	al,0ffh		; AL <- interrupt mask
		out	IRQ2_IMR,al	; disable all interrupts on SLAVE
		out	IRQ1_IMR,al	; disable all interrupts on MASTER

; ------------- Acknowledge all interrupts (set OCW2)

		mov	al,B5		; AL <- non-specific EOI
		out	IRQ1_CMD,al	; acknowledge interrupts on slave
		out	IRQ2_CMD,al	; acknowledge interrupts on master

; ------------- Pop registers

		clc			; clear error flag
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                    Driver function: Get IRQ channel info
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	ECX = number of channels
;		EDX = flags (see Channel info flags), 0=invalid channel
;		ESI = base interrupt
;		CY = invalid channel number (ECX and ESI are valid)
; -----------------------------------------------------------------------------

; ------------- Prepare default parameters

IRQDevInfo:	mov	ecx,IRQ_CHANNELS ; ECX <- number of IRQ channels
		xor	edx,edx		; EDX <- 0, invalid channel
		mov	esi,[ebx+DEVIRQ_Base] ; ESI <- base interrupt

; ------------- Check IRQ channel number

		cmp	eax,ecx		; check IRQ channel number
		cmc			; set error flag
		jc	IRQDevInfo8	; invalid channel number

; ------------- Disable cascade IRQ

		cmp	al,IRQ_CASCADE	; cascade IRQ ?
		stc			; set error flag
		je	IRQDevInfo8	; cascade IRQ is invalid
		mov	dl,IRQINFO_VALID ; EDX <- channel is valid

; ------------- Check if channel is enabled

		bt	dword [ebx+DEVIRQ_Cache],eax ; is channel enabled?
		jc	IRQDevInfo4	; channel is disabled
		or	dl,IRQINFO_ENABLED ; set enabled flag
IRQDevInfo4:	clc			; clear error flag
IRQDevInfo8:	ret

; -----------------------------------------------------------------------------
;                    Driver function: Enable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = invalid channel number or channel is already enabled
; -----------------------------------------------------------------------------

; ------------- Check channel number

IRQDevEnable:	cmp	eax,IRQ_CHANNELS ; check IRQ channel number
		jae	short IRQDevPending8 ; invalid channel number

; ------------- Check if channel is already enabled

		bt	dword [ebx+DEVIRQ_Cache],eax ; is channel enabled?
		jnc	short IRQDevPending8 ; channel is already enabled

; ------------- Enable IRQ channel

		call	IRQDev0Enable	; enable IRQ channel
		clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                    Driver function: Disable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = invalid channel number or channel is already disabled
; -----------------------------------------------------------------------------

; ------------- Check channel number

IRQDevDisable:	cmp	eax,IRQ_CHANNELS ; check IRQ channel number
		jae	short IRQDevPending8 ; invalid channel number

; ------------- Cannot disable reserved cascade channel

		cmp	al,IRQ_CASCADE	; cascade IRQ channel ?
		je	short IRQDevPending8 ; cascade IRQ is reserved

; ------------- Check if channel is already disabled

		bt	dword [ebx+DEVIRQ_Cache],eax ; is channel disabled?
		jc	IRQDevDisable9	; channel is already disabled

; ------------- Disable IRQ channel

		call	IRQDev0Disable	; disable IRQ channel
		clc			; clear error flag
IRQDevDisable9:	ret

; -----------------------------------------------------------------------------
;                 Driver function: Check if interrupt is pending
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = interrupt is pending or invalid channel number
; -----------------------------------------------------------------------------

IRQDevPending:	cmp	eax,IRQ_CHANNELS ; check IRQ channel number
		jb	IRQDev0Pending	; channel is OK, check pending
IRQDevPending8:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;               Driver function: Check if interrupt is in-service
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = interrupt is in-service or invalid channel number
; -----------------------------------------------------------------------------

IRQDevInServ:	cmp	eax,IRQ_CHANNELS ; check IRQ channel number
		jb	IRQDev0InServ	; channel is OK, check in-service
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Acknowledge interrupt
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
;		EBX = device descriptor DEVIRQ
; OUTPUT:	CY = invalid channel number
; -----------------------------------------------------------------------------

; ------------- Check channel number

IRQDevAck:	cmp	eax,IRQ_CHANNELS ; check IRQ channel number
		jae	short IRQDevPending8 ; invalid channel number

; ------------- Acknowledge interrupt

		call	IRQDev0Ack	; acknowledge interrupt
		clc			; clear error flag
		ret

; *****************************************************************************
;
;                            IRQ global functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                   IRQ global function: Get IRQ channel info
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	ECX = number of channels
;		EDX = flags (see Channel info flags), 0=invalid channel
;		ESI = base interrupt
;		CY = invalid channel number (ECX and ESI are valid)
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQInfo:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Info	; get IRQ channel info
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;              IRQ global function: Remap base interrupt address
; -----------------------------------------------------------------------------
; INPUT:	EAX = new base interrupt (0 to 248, must be multiple of 8)
; OUTPUT:	CY = error (invalid base interrupt)
; NOTES:	All interrupts should be disabled first.
; 		It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQRemap:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Remap ; remap base interrupt address
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   IRQ global function: Enable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	CY = invalid channel number or channel is already enabled
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQEnable:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Enable ; enable IRQ channel
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   IRQ global function: Disable IRQ channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	CY = invalid channel number or channel is already disabled
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQDisable:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Disable ; disable IRQ channel
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;             IRQ global function: Check if interrupt is pending
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	CY = interrupt is pending or invalid channel number
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQPending:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Pending ; check if interrupt is pending
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;             IRQ global function: Check if interrupt is in-service
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	CY = interrupt is in-service or invalid channel number
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQInServ:	push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_InServ ; check if interrupt is in-service
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;              IRQ global function: Acknowledge interrupt
; -----------------------------------------------------------------------------
; INPUT:	EAX = IRQ channel number
; OUTPUT:	CY = invalid channel number
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

IRQAck:		push	ebx		; push EBX
		mov	ebx,[IRQDevice]	; EBX <- current IRQ device
		DEVFNCLOCKC DEVIRQ_Ack	; acknowledge interrupt
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Text strings

		align	4, db 0
IRQDevName:	CTEXTDATA 'irq'
IRQDevShort:	CTEXTDATA 'Interrupt 8259A'

IRQDevFull:	LANGTEXTSTR IRQDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
 		LANGTEXTSTR IRQDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

IRQDevFullEN:	CTEXTDATA 'Programmable interrupt controller 8259A'
IRQDevFullCZ:	CTEXTDATA 'Programovateln',0c3h,0bdh,' ',0c5h,99h,'adi',\
			0c4h,8dh,' p',0c5h,99h,'eru',0c5h,0a1h,'en',0c3h,0adh,\
			' 8259A'

IRQDevInt:	dd	DEV_GEN_ID
		dd	DEV_IRQ_ID
		dd	DEV_NUL_ID

IRQDevRes1Name:	CTEXTDATA 'cascade'
IRQDevRes2Name:	CTEXTDATA 'ctrl#1'
IRQDevResNName:	CTEXTDATA 'ctrl#2'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- IRQ device descriptor

		align	4, db 0
IRQDevice:	dd	IRQDev			; current IRQ device descriptor

; ------------- Interrupt controller 8259A

%define		IRQDevVendor  DefaultVendor

		align	8, db 0
IRQDev:		DEVICEIRQ 0,DEV_STATIC,1,0,0,IRQ_CHANNELS,IRQDev

IRQDevRes1:	DEVRESOURCE IRQDevRes2,IRQDevRes0,IRQ_CASCADE,IRQ_CASCADE, \
			DEVRES_IRQ,DEVRES_STATIC,IRQDevRes1Name
IRQDevRes2:	DEVRESOURCE IRQDevResN,IRQDevRes1,IRQ1_BASE,IRQ1_BASE+1, \
			DEVRES_PORT,DEVRES_STATIC,IRQDevRes2Name
IRQDevResN:	DEVRESOURCE IRQDevRes0,IRQDevRes2,IRQ2_BASE,IRQ2_BASE+1, \
			DEVRES_PORT,DEVRES_STATIC,IRQDevResNName
