; =============================================================================
;
;                        Litos - DMA controller 8237A
;
; =============================================================================
; NOTES:
; - DMA0-3 is 8-bit controller 1, max. 64 KB, must not cross 64K boundary
; - DMA4 channel is used as cascade (cannot be used)
; - DMA5-7 is 16=bit controller 2, max. 128 KB, must not cross 128K boundary
; - transfers are limited to lower 16 MB of physical memory (use DMAMemGet)
;
; DMA transfer:
; - Allocate memory block using DMAMemAlloc (max. 64 KB)
; - Allocate DMA channel using DMAAlloc
; - Start transfer using DMAStart
; - Repeatedly test transfer using DMARemain until it becomes 0
;	(or use interrupt from the device)
; - Stop transfer using DMAStop
; - Free DMA channel using DMAFree
; - Free memory block using DMAMemFree
;
; DMA ports:
; 0000..000F - DMA #1 controller 8237A
; 0080       - on PC/AT diagnostic point
; 0081..008F - DMA page register 74612
; 00C0..00DF - DMA #2 controller 8237A
;
; Default DMA channels assignments on PC/AT:
;	DMA 2: floppy disk controller
;	DMA 3: ECP printer port (LPT1) or IDE controller
;	DMA 4: cascade to slave DMA controller
; =============================================================================

		CODE_SECTION

DMA_CHANNELS	EQU	8		; number of DMA channels
DMA_CASCADE	EQU	4		; channel 4 is reserved (cascade)

; ------------- DMA modes

DMAMODE_VERIFY	EQU	B6		; DMA verify (verify,increment,single)
DMAMODE_READ	EQU	B2+B6		; DMA read (read, increment, single)
DMAMODE_WRITE	EQU	B3+B6		; DMA write (write, increment, single)
DMAMODE_CASCADE	EQU	B6+B7		; DMA cascade
DMAMODE_INIT	EQU	B4		; DMA autoinit

; ------------- Ports - controller 1

DMA1_BASE	EQU	0		; base port of controller 1 (8-bit)

DMA1_CMD	EQU	DMA1_BASE+8	; command register WRITE (controller 1)
DMA1_STAT	EQU	DMA1_BASE+8	; status register READ (controller 1)
DMA1_REQ	EQU	DMA1_BASE+9	; request register WRITE (controller 1)
DMA1_MASK	EQU	DMA1_BASE+10	; single-channel mask WRITE (ctrl 1)
DMA1_MODE	EQU	DMA1_BASE+11	; mode register WRITE (controller 1)
DMA1_CLRFF	EQU	DMA1_BASE+12	; clear flip-flop WRITE (controller 1)
DMA1_RESET	EQU	DMA1_BASE+13	; master clear WRITE (controller 1)
DMA1_TEMP	EQU	DMA1_BASE+13	; temporary register READ (ctrl 1)
DMA1_CMASK	EQU	DMA1_BASE+14	; clear mask (controller 1)
DMA1_AMASK	EQU	DMA1_BASE+15	; all-channels mask WRITE (ctrl 1)

; ------------- Ports - controller 2

DMA2_BASE	EQU	0c0h		; base port of controller 2 (16-bit)

DMA2_CMD	EQU	DMA2_BASE+2*8	; command register WRITE (controller 2)
DMA2_STAT	EQU	DMA2_BASE+2*8	; status register READ (controller 2)
DMA2_REQ	EQU	DMA2_BASE+2*9	; request register WRITE (controller 2)
DMA2_MASK	EQU	DMA2_BASE+2*10	; single-channel mask WRITE (ctrl 2)
DMA2_MODE	EQU	DMA2_BASE+2*11	; mode register WRITE (controller 2)
DMA2_CLRFF	EQU	DMA2_BASE+2*12	; clear flip-flop WRITE (controller 2)
DMA2_RESET	EQU	DMA2_BASE+2*13	; master clear WRITE (controller 2)
DMA2_TEMP	EQU	DMA2_BASE+2*13	; temporary register READ (ctrl 2)
DMA2_CMASK	EQU	DMA2_BASE+2*14	; clear mask (controller 2)
DMA2_AMASK	EQU	DMA2_BASE+2*15	; all-channels mask WRITE (ctrl 2)

; ------------- Ports - address registers

DMA_ADDR0	EQU	DMA1_BASE+0	; channel 0 address
DMA_ADDR1	EQU	DMA1_BASE+2	; channel 1 address
DMA_ADDR2	EQU	DMA1_BASE+4	; channel 2 address
DMA_ADDR3	EQU	DMA1_BASE+6	; channel 3 address
DMA_ADDR4	EQU	DMA2_BASE+2*0	; channel 4 address
DMA_ADDR5	EQU	DMA2_BASE+2*2	; channel 5 address
DMA_ADDR6	EQU	DMA2_BASE+2*4	; channel 6 address
DMA_ADDR7	EQU	DMA2_BASE+2*6	; channel 7 address

; ------------- Ports - address registers

DMA_COUNT0	EQU	DMA1_BASE+1	; channel 0 count
DMA_COUNT1	EQU	DMA1_BASE+3	; channel 1 count
DMA_COUNT2	EQU	DMA1_BASE+5	; channel 2 count
DMA_COUNT3	EQU	DMA1_BASE+7	; channel 3 count
DMA_COUNT4	EQU	DMA2_BASE+2*1	; channel 4 count
DMA_COUNT5	EQU	DMA2_BASE+2*3	; channel 5 count
DMA_COUNT6	EQU	DMA2_BASE+2*5	; channel 6 count
DMA_COUNT7	EQU	DMA2_BASE+2*7	; channel 7 count

; ------------- Ports - page registers

DMA_PAGE_BASE	EQU	80h		; base port of page registers

DMA_PAGE0	EQU	DMA_PAGE_BASE+7	; page 0
DMA_PAGE1	EQU	DMA_PAGE_BASE+3	; page 1
DMA_PAGE2	EQU	DMA_PAGE_BASE+1	; page 2
DMA_PAGE3	EQU	DMA_PAGE_BASE+2	; page 3
DMA_PAGE4	EQU	DMA_PAGE_BASE+15; page 4
DMA_PAGE5	EQU	DMA_PAGE_BASE+11; page 5
DMA_PAGE6	EQU	DMA_PAGE_BASE+9	; page 6
DMA_PAGE7	EQU	DMA_PAGE_BASE+10; page 7

; -----------------------------------------------------------------------------
;                   Internal function: Enable DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
; NOTES:	EAX not checked for validity.
; -----------------------------------------------------------------------------

; ------------- Check if it is controller 2

DMADevEnable:	cmp	al,4		; is it controller 2?
		jae	DMADevEnable2	; it is controller 2

; ------------- Enable DMA channel on controller 1

		out	DMA1_MASK,al	; enable DMA channel on controller 1
		ret

; ------------- Enable DMA channel on controller 2

DMADevEnable2:	sub	al,4		; Al <- relative DMA channel number
		out	DMA2_MASK,al	; enable DMA channel on controller 2
		add	al,4		; AL <- return DMA channel number
		ret

; -----------------------------------------------------------------------------
;                   Internal function: Disable DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
; NOTES:	EAX not checked for validity.
; -----------------------------------------------------------------------------

; ------------- Check if it is controller 2

DMADevDisable:	cmp	al,4		; is it controller 2?
		jae	DMADevDisable2	; it is controller 2

; ------------- Disable DMA channel on controller 1

		add	al,B2		; set disable flag
		out	DMA1_MASK,al	; disable DMA channel on controller 1
		sub	al,B2		; return DMA channel number
		ret

; ------------- Disable DMA channel on controller 2 (bit 2 is set)

DMADevDisable2:	out	DMA2_MASK,al	; disable DMA channel on controller 2
		ret

; -----------------------------------------------------------------------------
;                Internal function: Clear DMA channel flip-flop
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
; NOTES:	EAX not checked for validity.
; -----------------------------------------------------------------------------

; ------------- Check if it is controller 2

DMADevClearFF:	cmp	al,4		; is it controller 2?
		jae	DMADevClearFF2	; controller 2

; ------------- Clear DMA channel flip-flop on controller 1

		out	DMA1_CLRFF,al	; clear DMA channel flip-flop on ctrl 1
		ret

; ------------- Clear DMA channel flip-flop on controller 2

DMADevClearFF2:	out	DMA2_CLRFF,al	; clear DMA channel flip-flop on ctrl 2
		ret

; -----------------------------------------------------------------------------
;                    Internal function: Set DMA channel mode
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
;		BL = DMA channel mode
;			B0-B1: must be 0
;			B2-B3: transfer mode
;				00=verify
;				01=read from device (write to memory)
;				10=write to device (read from memory)
;			B4: 1=autoinitialisation enabled
;			B5: 0=increment address, 1=decrement address
;			B6-B7:	00=demand mode
;				01=single mode
;				10=block transfer
;				11=cascade mode
;			preset modes:	DMAMODE_VERIFY - verify
;					DMAMODE_READ - DMA read from device
;					DMAMODE_WRITE - DMA write to device
;					DMAMODE_CASCADE - DMA cascade
;					DMAMODE_INIT - DMA autoinit
; NOTES:	EAX not checked for validity.
; -----------------------------------------------------------------------------

; ------------- Check if it is controller 2

DMADevSetMode:	push	eax		; push EAX
		cmp	al,4		; is it controller 2?
		jae	DMADevSetMode2	; controller 2

; ------------- Set DMA channel mode on controller 1

		or	al,bl		; AL <- channel + mode
		out	DMA1_MODE,al	; set DMA mode on controller 1
		pop	eax		; pop EAX
		ret

; ------------- Set DMA channel mode on controller 2

DMADevSetMode2:	sub	al,4		; AL <- relative DMA channel number
		or	al,bl		; AL <- channel + mode
		out	DMA2_MODE,al	; set DMA mode on controller 2
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                      Internal function: Set DMA channel page
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
;		EDX = DMA address (physical address)
; NOTES:	EAX not checked for validity.
;		Only address bits 16-23 (DMA0-3) or 17-23 (DMA4-7) are used.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevSetPage:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Prepare DMA address
	
		shr	edx,16		; DL <- page number
		cmp	al,4		; is it controller 1?
		jb	DMADevSetPage2	; it is controller 1
		and	dl,~B0		; clear bit 16 for controller 2

; ------------- Prepare port address

DMADevSetPage2:	mov	al,[eax+DMAPagePort] ; AL <- address (here AH = 0)

; ------------- Set DMA channel page

		xchg	eax,edx		; DX <- port, AL <- page
		out	dx,al		; set DMA channel page

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;          Internal function: Set DMA transfer address (offset and page)
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
;		EDX = DMA address (physical address, only bits 0-23 are used)
; NOTES:	EAX not checked for validity.
;		DMA transfer cannot cross 64K (or 128K for DMA4-7) boundary.
;		DMA5 to DMA7 address must be word aligned.
;		Transfers are limited to lower 16 MB of physical memory.
;		DMADevClearFF should be caled first.
; -----------------------------------------------------------------------------

; ------------- Set DMA channel page

DMADevSetAddr:	call	DMADevSetPage	; set DMA channel page

; ------------- Push registers

		push	eax		; push EAX
		push	edx		; push EDX

; ------------- Prepare port for controller 1 (00h, 02h, 04h, 06h)

		shl	eax,1		; AX <- channel number*2
		cmp	al,DMA1_BASE+4*2 ; is it DMA0..DMA3?
		jb	DMADevSetAddr2	; it is DMA0..DMA3

; ------------- Prepare port for controller 2 (0C0h, 0C4h, 0C8h, 0CCh)

		shl	eax,1		; AX <- channel number*4
                add	al,DMA2_BASE-DMA1_BASE-4*4 ; AL <- port
		shr	edx,1		; convert address to words

; ------------- Set transfer address

DMADevSetAddr2:	xchg	eax,edx		; DX <- port, EAX <- address
		out	dx,al		; set transfer address LOW
		mov	al,ah		; AL <- transfer address HIGH
		jmp	short DMADevSetAddr4 ; short delay
DMADevSetAddr4:	out	dx,al		; set transfer address HIGH

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                     Internal function: Set DMA transfer size
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
;		ECX = DMA transfer size (cannot be 0; B0 is ignored for DMA4-7)
; NOTES:	EAX not checked for validity.
;		DMA transfer cannot cross 64K (or 128K for DMA4-7) boundary.
;		Maximal transfer size is 64K for DMA0-3 and 128K for DMA4-7.
;		DMADevClearFF should be caled first.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevSetSize:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Prepare port for controller 1 (01h, 03h, 05h, 07h)

		shl	eax,1		; AX <- channel number*2
		inc	eax		; EAX <- channel number*2 + 1
		cmp	al,DMA1_BASE+4*2+1 ; is it DMA0..DMA3?
		jb	DMADevSetSize2	; it is DMA0..DMA3

; ------------- Prepare port for controller 2 (0C2h, 0C6h, 0CAh, 0CEh)

		shl	eax,1		; AX <- channel number*4 + 2
                add	al,DMA2_BASE-DMA1_BASE-4*4 ; AL <- port
		shr	ecx,1		; convert size to words

; ------------- Set transfer size

DMADevSetSize2:	dec	ecx		; correct size (=last transfered item)
		xchg	eax,edx		; DX <- port
		xchg	eax,ecx		; AX <- size
		out	dx,al		; set transfer size LOW
		mov	al,ah		; AL <- transfer size HIGH
		jmp	short DMADevSetSize4 ; short delay
DMADevSetSize4:	out	dx,al		; set transfer size HIGH

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;               Internal function: Get DMA remaining transfer size
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
; OUTPUT:	ECX = remaining bytes to transfer (0 if transfer finished)
; NOTES:	EAX not checked for validity.
;		If called before the channel has been used, it may return 1
;		(or 2 on DMA4..DMA7).
;		DMADevClearFF should be caled first.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevGetSize:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Prepare port for controller 1 (01h, 03h, 05h, 07h)

		shl	eax,1		; EAX <- channel number*2
		inc	eax		; EAX <- channel number*2 + 1
		cmp	al,DMA1_BASE+4*2+1 ; is it DMA0..DMA3?
		jb	DMADevGetSize2	; it is DMA0..DMA3

; ------------- Prepare port for controller 2 (0C2h, 0C6h, 0CAh, 0CEh)

		shl	eax,1		; EAX <- channel number*4 + 2
                add	al,DMA2_BASE-DMA1_BASE-4*4 ; AL <- port

; ------------- Get transfer size

DMADevGetSize2:	xchg	eax,edx		; DX <- port
		xor	eax,eax		; EAX <- 0
		in	al,dx		; get transfer size LOW
		xchg	eax,ecx		; ECX <- transfer size LOW
		jmp	short DMADevGetSize4 ; short delay
DMADevGetSize4:	in	al,dx		; get transfer size HIGH
		mov	ch,al		; CH <- transfer size HIGH
		inc	ecx		; correction

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX

; ------------- Convert to words for DMA4 - DMA7

		cmp	al,4		; is it DMA4..DMA7?
		jb	DMADevGetSize8	; it is DMA0..DMA3
		shl	ecx,1		; convert to words
DMADevGetSize8:	ret

; -----------------------------------------------------------------------------
;               Internal function: Check if transfer is still running
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number (0 to 7)
; OUTPUT:	CY = transfer is not running
; NOTES:	EAX not checked for validity.
;		If called before the channel has been used, it may return NC.
;		It may be faster than DMADevGetSize.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevRunning:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Prepare port for controller 1 (01h, 03h, 05h, 07h)

		shl	eax,1		; EAX <- channel number*2
		inc	eax		; EAX <- channel number*2 + 1
		cmp	al,DMA1_BASE+4*2+1 ; is it DMA0..DMA3?
		jb	DMADevRunning2	; it is DMA0..DMA3

; ------------- Prepare port for controller 2 (0C2h, 0C6h, 0CAh, 0CEh)

		shl	eax,1		; EAX <- channel number*4 + 2
                add	al,DMA2_BASE-DMA1_BASE-4*4 ; AL <- port

; ------------- Get transfer size

DMADevRunning2:	xchg	eax,edx		; DX <- port
		in	al,dx		; get transfer size 1
		cmp	al,0ffh		; is transfer still running?
		jne	DMADevRunning6	; transfer is still running
		jmp	short DMADevRunning4 ; short delay
DMADevRunning4:	in	al,dx		; get transfer size 2
		cmp	al,0ffh		; is transfer still running?
DMADevRunning6:	cmc			; NC = transfer is still running

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; *****************************************************************************
;
;                           Driver interface functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Install DMA device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

DMAInstall:	push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Register DMA device

		mov	ebx,DMADev	; EBX <- device descriptor
		xor	ecx,ecx		; ECX <- 0, no parent device
		call	DevRegister	; register device

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                           Uninstall DMA device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

DMAUninstall:	push	ebx		; push EBX

; ------------- Unregister DMA device

		mov	ebx,DMADev	; EBX <- device descriptor
		call	DevUnregister	; unregister device

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize DMA device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVDMA
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevInit:	push	eax		; push EAX
		push	edx		; push EDX

; ------------- Clear mask

		mov	dword [ebx+DEVDMA_Used],~(B0+B1+B2+B3+B5+B6+B7) ; mask
		mov	dword [ebx+DEVDMA_Running],0 ; running channels

; ------------- Send master clear to DMA controllers

		out	DMA1_RESET,al	; send master clear to controller 1
		out	DMA2_RESET,al	; send master clear to controller 2

; ------------- Clear DMA channel flip-flop

		out	DMA1_CLRFF,al	; clear DMA channel flip-flop on ctrl 1
		out	DMA2_CLRFF,al	; clear DMA channel flip-flop on ctrl 2

; ------------- Initialize channel address and count registers of controller 1

		xor	edx,edx		; DX <- 0, base port od controller 1
		mov	al,0ffh		; EAX <- 0ffh
DMADevInit2:	out	dx,al		; initialize one register LOW
		jmp	short DMADevInit3 ; short delay
DMADevInit3:	out	dx,al		; initialize one register HIGH		
		inc	edx		; increase port
		cmp	dl,DMA1_BASE+8	; all registers done?
		jne	DMADevInit2	; initialize next register

; ------------- Initialize channel address and count registers of controller 2

		mov	dl,DMA2_BASE	; DX <- base port of controller 2
DMADevInit4:	out	dx,al		; initialize one register LOW
		jmp	short DMADevInit5 ; short delay
DMADevInit5:	out	dx,al		; initialize one register HIGH		
		inc	edx		; increase port
		inc	edx		; increase port
		cmp	dl,DMA2_BASE+2*8 ; all registers done?
		jne	DMADevInit4	; initialize next register

; ------------- Set DMA command
; DACK sense low, DREQ sense high, late write, fixed priority, normal timing,
; controller enable, channel 0 address hold disable, memory-to-memory disable

		inc	eax		; AL <- 0
		out	DMA1_CMD,al	; set command to controller 1
		out	DMA2_CMD,al	; set command to controller 2

; ------------- Set mode of all DMA channels
; single mode, address increment, autoinitialization disable, verify transfer

		mov	al,DMAMODE_CASCADE ; AL <- cascade mode of channel 4
		out	DMA2_MODE,al	; set mode of channel 4
		mov	al,DMAMODE_VERIFY ; AL <- mode of channel 0
		out	DMA1_MODE,al	; set mode of channel 0
		inc	eax		; AL <- mode of channel 1 and 5
		out	DMA1_MODE,al	; set mode of channel 1
		out	DMA2_MODE,al	; set mode of channel 5
		inc	eax		; AL <- mode of channel 2 and 6
		out	DMA1_MODE,al	; set mode of channel 2
		out	DMA2_MODE,al	; set mode of channel 6
		inc	eax		; AL <- mode of channel 3 and 7
		out	DMA1_MODE,al	; set mode of channel 3
		out	DMA2_MODE,al	; set mode of channel 7

; ------------- Set mask of channels (disable all channels except channel 4)

		mov	al,B0+B1+B2+B3	; set mask of channels 0 to 3
		out	DMA1_AMASK,al	; disable all channels
		mov	al,B1+B2+B3	; set mask of channels 5 to 7
		out	DMA2_AMASK,al	; disable all channels except channel 4

; ------------- Initialize page registers (except DMA refresh page register)

		mov	al,0		; AL <- 0
		mov	dl,DMA_PAGE_BASE+1 ; DL <- start DMA page register
DMADevInit6:	out	dx,al		; clear one register
		inc	edx		; increase register address
		cmp	dl,DMA_PAGE_BASE+15 ; last DMA page register?
		jne	DMADevInit6	; initialize next register

; ------------- Pop registers

		clc			; clear error flag
		pop	edx		; pop EDX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Deinitialize DMA device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVDMA
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevDeinit:	push	eax		; push EAX

; ------------- Stop DMA transfers

		xor	eax,eax		; EAX <- 0
DMADevDeInit2:	call	DMADevStop	; stop DMA transfer
		inc	eax		; increase channel number
		cmp	al,DMA_CHANNELS	; is it valid DMA channel?
		jb	DMADevDeInit2	; stop next DMA channel

; ------------- Pop registers

		clc			; clear error flag
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                    Driver function: Get DMA channel info
; -----------------------------------------------------------------------------
; INPUT:	EAX = DMA channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	ECX = number of channels
;		EDX = flags (see Channel info flags), 0=invalid channel
;		CY = invalid channel number (ECX is valid)
; -----------------------------------------------------------------------------

; ------------- Check DMA channel number

DMADevInfo:	mov	ecx,DMA_CHANNELS ; ECX <- number of DMA channels
		xor	edx,edx		; EDX <- 0, invalid channel
		cmp	eax,ecx		; check DMA channel number
		cmc			; set error flag
		jc	DMADevInfo8	; invalid channel number

; ------------- Prepare channel flags

		mov	dl,DMAINFO_CH4_7 ; channels 4..7
		cmp	al,4		; channel 4..7 ?
		jae	DMADevInfo2	; channel 4..7
		mov	dl,DMAINFO_CH0_3 ; otherwise channel 0..3

; ------------- Check if channel is in use

DMADevInfo2:	bt	dword [ebx+DEVDMA_Used],eax ; check if channel is used
		jnc	DMADevInfo4	; channel is not used
		or	dl,DMAINFO_USED	; set used flag

; ------------- Check if channel is running (it clears error flag NC)

DMADevInfo4:	bt	dword [ebx+DEVDMA_Running],eax ; is channel running?
		jnc	DMADevInfo8	; channel is not running
		or	dl,DMAINFO_RUNNING ; set running flag
DMADevInfo8:	ret

; -----------------------------------------------------------------------------
;                     Driver function: Allocate DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	CY = cannot allocate, channel already used
; -----------------------------------------------------------------------------

; ------------- Check channel number

DMADevAlloc:	cmp	eax,DMA_CHANNELS ; check channel number
		cmc			; set error flag if invalid channel
		jc	DMADevAlloc2	; invalid channel number

; ------------- Allocate channel

		bts	[ebx+DEVDMA_Used],eax ; set channel flag (CY = used)
DMADevAlloc2:	ret

; -----------------------------------------------------------------------------
;                     Driver function: Free DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	CY = invalid channel number or it is already free or reserved
;			or it is still running (call DMADevStop first)
; -----------------------------------------------------------------------------

; ------------- Check channel number

DMADevFree:	cmp	eax,DMA_CHANNELS ; check channel number
		jae	DMADevFree8	; invalid channel number

; ------------- Reserved channel

		cmp	al,DMA_CASCADE	; cascade channel is reserved
		je	DMADevFree8	; channel is reserved

; ------------- Check if channel is running

		bt	dword [ebx+DEVDMA_Running],eax ; is channel running?
		jc	DMADevFree9	; channel is still running

; ------------- Free DMA channel (and check if it is used)

		btr	dword [ebx+DEVDMA_Used],eax ; reset usage bit
DMADevFree8:	cmc			; set error flag if it was not used
DMADevFree9:	ret

; -----------------------------------------------------------------------------
;                      Driver function: Start DMA transfer
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
;		ECX = number of bytes (must be in range 2 to 64K, word aligned)
;		EDX = system address (word aligned)
;		ESI = DMA direction (DMADIR_READ, DMADIR_WRITE, DMADIR_VERIFY)
; OUTPUT:	CY = invalid argument (or channel is already running)
; NOTES:	Function translates system address to physical address.
;		Block cannot cross 64K boundary, size and address must be word
;			aligned, size can be max. 64 KB and transfered block 
;			must be in lower 16MB of physical RAM.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMADevStart:	push	edx		; push EDX

; ------------- Check channel number

		cmp	eax,DMA_CHANNELS ; check channel number
		jae	DMADevStart8	; invalid channel number
		cmp	al,DMA_CASCADE	; cascade channel is reserved
		je	DMADevStart8	; channel is reserved

; ------------- Check if channel is used

		bt	dword [ebx+DEVDMA_Used],eax ; is channel used?
		jnc	DMADevStart8	; channel is not used

; ------------- Check number of bytes

		cmp	ecx,10000h	; maximal number of bytes
		ja	DMADevStart8	; size is out of range
		test	cl,1		; is size aligned?
		jnz	DMADevStart8	; size is not aligned
		jecxz	DMADevStart8	; size is 0

; ------------- Check DMA direction

		cmp	esi,DMADIR_NUM	; check maximal DMA direction
		jae	DMADevStart8	; invalid DMA direction

; ------------- Check address (and translate to physical memory)

		sub	edx,SYSTEM_ADDR ; translate to physical address
		jc	DMADevStart8	; invalid DMA address
		cmp	edx,DMAMAXMEM	; maximal DMA address
		jae	DMADevStart8	; invalid DMA address
		test	dl,1		; check if address is aligned
		jnz	DMADevStart8	; address is not word aligned

; ------------- Check 64K boundary

		push	ecx		; push ECX
		push	edx		; push EDX
		dec	ecx		; ECX <- last byte of size
		add	dx,cx		; check boundary
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		jc	DMADevStart8	; invalid boundary

; ------------- Check if channel is running and set running flag

		bts	dword [ebx+DEVDMA_Running],eax ; set running flag
		jc	DMADevStart8	; channel is already running

; ------------- Set DMA mode (in EDX)

		push	ebx		; push EBX
		mov	bl,[DMAModeTab+esi] ; BL <- DMA mode
		call	DMADevSetMode	; set DMA mode
		pop	ebx		; pop EBX

; ------------- Clear DMA channel flip-flop

		call	DMADevClearFF	; clear DMA channel flip-flop

; ------------- Set transfer address (in EDX)

		call	DMADevSetAddr	; set DMA address

; ------------- Set transfer size (in ECX)

		call	DMADevSetSize	; set transfer size

; ------------- Start DMA transfer

		call	DMADevEnable	; start DMA transfer

; ------------- OK

		pop	edx		; pop EDX
		clc			; clear error flag
		ret

; ------------- ERROR

DMADevStart8:	pop	edx		; pop EDX
DMADevStart9:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                     Driver function: Stop DMA transfer
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	CY = invalid channel number or channel is not running
; -----------------------------------------------------------------------------

; ------------- Check channel number

DMADevStop:	cmp	eax,DMA_CHANNELS ; check channel number
		jae	short DMADevStart9 ; invalid channel number
		cmp	al,DMA_CASCADE	; cascade channel is reserved
		je	short DMADevStart9 ; channel is reserved

; ------------- Check if channel is used

		bt	dword [ebx+DEVDMA_Used],eax ; is channel used?
		jnc	short DMADevStart9 ; channel is not used

; ------------- Check if channel is running

		btr	dword [ebx+DEVDMA_Running],eax ; is channel running?
		jnc	short DMADevStart9 ; channel is not running

; ------------- Stop DMA transfer

		call	DMADevDisable	; stop DMA transfer
		clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;               Driver function: Get remaining transfer size
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	CY = invalid channel number or channel is not running
;		ECX = remaining bytes (0 on error)
;		If called before the channel has been used, it may return 1
;		(or 2 on DMA5..DMA7).
; -----------------------------------------------------------------------------

; ------------- Check channel number

DMADevRemain:	cmp	eax,DMA_CHANNELS ; check channel number
		jae	DMADevRemain8	; invalid channel number
		cmp	al,DMA_CASCADE	; cascade channel is reserved
		je	DMADevRemain8	; channel is reserved

; ------------- Check if channel is used

		bt	dword [ebx+DEVDMA_Used],eax ; is channel used?
		jnc	DMADevRemain8	; channel is not used

; ------------- Check if channel is running

		bt	dword [ebx+DEVDMA_Running],eax ; is channel running?
		jnc	DMADevRemain8	; channel is not running

; ------------- Clear DMA channel flip-flop

		call	DMADevClearFF	; clear DMA channel flip-flop

; ------------- Get remaining transfer size

		call	DMADevGetSize	; get remaining size
		clc			; clear error flag
		ret

; ------------- ERROR

DMADevRemain8:	xor	ecx,ecx		; ECX <- 0 no data left
DMADevRemain9:	stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;               Driver function: Check if transfer is still running
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = device descriptor DEVDMA
; OUTPUT:	CY = invalid channel number or channel is not running
;		If called before the channel has been used, it may return NC.
;		It is faster than DMADevRemain.
; -----------------------------------------------------------------------------

; ------------- Check channel number

DMADevCheck:	cmp	eax,DMA_CHANNELS ; check channel number
		jae	DMADevRemain9	; invalid channel number
		cmp	al,DMA_CASCADE	; cascade channel is reserved
		je	DMADevRemain9	; channel is reserved

; ------------- Check if channel is used

		bt	dword [ebx+DEVDMA_Used],eax ; is channel used?
		jnc	DMADevRemain9	; channel is not used

; ------------- Check if channel is running

		bt	dword [ebx+DEVDMA_Running],eax ; is channel running?
		jnc	DMADevRemain9	; channel is not running

; ------------- Check if channel is still running

		jmp	DMADevRunning	; check if channel is still running

; *****************************************************************************
;
;                            DMA global functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                  DMA global function: Get DMA channel info
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	ECX = number of channels
;		EDX = flags (see Channel info flags), 0=invalid channel
;		CY = invalid channel number (ECX is valid, EDX = 0)
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMAInfo:	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Info	; get DMA channel info
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                 DMA global function: Allocate DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	CY = cannot allocate, channel allready used
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMAAlloc:	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Alloc ; allocate DMA channel
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   DMA global function: Free DMA channel
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	CY = invalid channel number or it is already free or it is
;			still running
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMAFree:	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Free	; free DMA channel
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   DMA global function: Start DMA transfer
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		ECX = number of bytes (cannot be 0, max. 64K, word aligned)
;		EDX = system address (word aligned)
;		ESI = DMA direction (DMADIR_READ, DMADIR_WRITE, DMADIR_VERIFY)
; OUTPUT:	CY = invalid argument or channel is already running
; NOTES:	Function translates system address to physical address.
;		Block cannot cross 64K boundary, size and address must be word
;			aligned, size can be max. 64 KB and transfered block 
;			must be in lower 16MB of physical RAM.
; 		It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMAStart:	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Start ; start DMA transfer
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   DMA global function: Stop DMA transfer
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	CY = invalid channel number or channel is not running
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMAStop: 	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Stop	; stop DMA transfer
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               DMA global function: Get remaining transfer size
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	CY = invalid channel number or channel is not running
;		ECX = remaining bytes (0 on error)
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMARemain: 	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Remain ; get remaining transfer size
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               DMA global function: Check if channel is still running
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
; OUTPUT:	CY = invalid channel number or channel is not running
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

DMACheck: 	push	ebx		; push EBX
		mov	ebx,[DMADevice]	; EBX <- current DMA device
		DEVFNCLOCKC DEVDMA_Check ; check if channel is still running
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               DMA global function: Wait until transfer stops
; -----------------------------------------------------------------------------
; INPUT:	EAX = channel number
;		EBX = minimal time in ms (0=immediately)
;		ECX = maximal time in ms (time-out, 0=immediately)
;		EDX = delta time in ms of tests until stops (1 minimal)
; OUTPUT:	CY = invalid channel number or it is not running or time-out
; NOTES:	It locks fast spin-lock of device interface.
; -----------------------------------------------------------------------------

; ------------- Push registers

DMAWait:	push	ebx		; push EBX
		push	ecx		; push ECX (it must be last)

; ------------- Check if channel is valid, used and running

		push	edx		; push EDX
		call	DMAInfo		; get DMA channel info
		jc	DMAWait2	; invalid channel
		and	dl,DMAINFO_RUNNING+DMAINFO_USED	; flags
		cmp	dl,DMAINFO_RUNNING+DMAINFO_USED	; is used + running?
DMAWait2:	pop	edx		; pop EDX
		jc	DMAWait8	; invalid channel or not running
		pop	ecx		; pop ECX
		push	ecx		; push ECX

; ------------- Wait for a while

DMAWait3:	push	eax		; push EAX
		mov	eax,ebx		; EAX <- time in ms
		cmp	eax,ecx		; check time-out
		jb	DMAWait4	; time-out is OK
		mov	eax,ecx		; EAX <- limit time-out
DMAWait4:	or	eax,eax		; any time?
		jz	DMAWait6	; no time
		call	Sleep		; sleep for a while
DMAWait6:	sub	ecx,ebx		; check time-out
		pop	eax		; pop EAX
		jb	DMAWait8	; time-out
		mov	ebx,edx		; EBX <- delta time

; ------------- Check if channel is running

		call	DMACheck	; check if channel is running
		jc	DMAWait7	; transfer stops

; ------------- Time-out is 0

		or	ecx,ecx		; is time-out 0 ?
		jnz	DMAWait3	; continue waiting
DMAWait7:	cmc			; CY = error flag

; ------------- Pop registers

DMAWait8:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- DMA page ports

DMAPagePort:	db	DMA_PAGE0	; DMA 0
		db	DMA_PAGE1	; DMA 1
		db	DMA_PAGE2	; DMA 2
		db	DMA_PAGE3	; DMA 3
		db	DMA_PAGE4	; DMA 4
		db	DMA_PAGE5	; DMA 5
		db	DMA_PAGE6	; DMA 6
		db	DMA_PAGE7	; DMA 7

; ------------- DMA modes

DMAModeTab:	db	DMAMODE_READ	; 0: DMADIR_READ
		db	DMAMODE_WRITE	; 1: DMADIR_WRITE
		db	DMAMODE_VERIFY	; 2: DMADIR_VERIFY

; ------------- Text strings

		align	4, db 0
DMADevName:	CTEXTDATA 'dma'
DMADevShort:	CTEXTDATA 'DMA 8237A'

DMADevFull:	LANGTEXTSTR DMADevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
 		LANGTEXTSTR DMADevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

DMADevFullEN:	CTEXTDATA 'Direct memory access (DMA) controller 8237A'
DMADevFullCZ:	CTEXTDATA 0c5h,98h,'adi',0c4h,8dh,' p',0c5h,99h,0c3h,0adh,\
			'm',0c3h,0a9h,'ho p',0c5h,99h,0c3h,0adh,'stupu do pam',\
			0c4h,9bh,'ti (DMA) 8237A'

DMADevInt:	dd	DEV_GEN_ID
		dd	DEV_DMA_ID
		dd	DEV_NUL_ID

DMADevRes1Name:	CTEXTDATA 'cascade'
DMADevRes2Name:	CTEXTDATA 'ctrl#1'
DMADevRes3Name:	CTEXTDATA 'pageregs'
DMADevResNName:	CTEXTDATA 'ctrl#2'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- DMA device descriptor

		align	4, db 0
DMADevice:	dd	DMADev			; current DMA device descriptor

%define		DMADevVendor  DefaultVendor

		align	8, db 0
DMADev:		DEVICEDMA 0,DEV_STATIC,1,0,0,DMA_CHANNELS,DMADev

DMADevRes1:	DEVRESOURCE DMADevRes2,DMADevRes0,DMA_CASCADE,DMA_CASCADE, \
		 DEVRES_DMA,DEVRES_STATIC,DMADevRes1Name
DMADevRes2:	DEVRESOURCE DMADevRes3,DMADevRes1,DMA1_BASE,DMA1_BASE+15, \
		 DEVRES_PORT,DEVRES_STATIC,DMADevRes2Name
DMADevRes3:	DEVRESOURCE DMADevResN,DMADevRes2,DMA_PAGE_BASE, \
		 DMA_PAGE_BASE+15,DEVRES_PORT,DEVRES_STATIC,DMADevRes3Name
DMADevResN:	DEVRESOURCE DMADevRes0,DMADevRes3,DMA2_BASE,DMA2_BASE+31, \
		 DEVRES_PORT,DEVRES_STATIC,DMADevResNName
