; =============================================================================
;
;                      Litos - CMOS memory Motorola MC146818
;
; =============================================================================

		CODE_SECTION

CMOS_BASE	EQU	70h		; CMOS base register
CMOS_INDEX	EQU	CMOS_BASE+0	; CMOS index register
CMOS_DATA 	EQU	CMOS_BASE+1	; CMOS data register

; ------------- Macro - read byte from CMOS (INPUT: %1=index, OUTPUT: AL=value)
; (It takes aprox. 4 us)

%macro		GET_CMOS 1

		mov	al,%1		; AL <- index of the byte
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		in	al,CMOS_DATA	; read byte from the CMOS
%endmacro

; ------------- Macro - pack binary data in AL to BCD format (destroys AH)

%macro		BINBCD	0

		aam	10		; unpack digits modulo 10
		aad	16		; pack digits modulo 16
%endmacro

; ------------- Macro - unpack BCD format in AL to binary data (destroys AH)

%macro		BCDBIN	0

		aam	16		; unpack digits modulo 16
		aad	10		; pack digits modulo 10
%endmacro

; -----------------------------------------------------------------------------
;       Internal function: Read byte from CMOS in BCD or BIN mode into buffer
; -----------------------------------------------------------------------------
; INPUT:	ESI = index of the byte in CMOS (in BCD or BIN mode)
;		EDI = buffer to store byte (in BIN mode)
;		EAX = bit 31 "BIN mode" flag
; OUTPUT:	AL = byte in BIN mode (was stored into [EDI])
;		ESI = ESI + 1
;		EDI = EDI + 1
; DESTROYS:	AH
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

CMOSReadBCD:	xchg	eax,esi		; AL <- index of the byte
		out	CMOS_INDEX,al	; set index of the byte
		inc	eax		; increase index of the byte
		xchg	eax,esi		; ESI <- save new index of the byte
		SHORT_DELAY		; short delay
		in	al,CMOS_DATA	; read byte from the CMOS
		or	eax,eax		; BIN mode flag?
		js	CMOSReadBCD2	; byte is in BIN mode
		BCDBIN			; convert from BCD to BIN
CMOSReadBCD2:	stosb			; store byte
		ret

; -----------------------------------------------------------------------------
;       Internal function: Write byte to CMOS in BCD or BIN mode from buffer
; -----------------------------------------------------------------------------
; INPUT:	AL = byte in BIN mode
;		EDI = index of the byte in CMOS (in BCD or BIN mode)
;		DL = bit 2 "BIN mode" flag
; OUTPUT:	EDI = EDI + 1
; DESTROYS:	AL, AH
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

CMOSWriteBCD:	xchg	eax,edi		; AL <- index of the byte
		out	CMOS_INDEX,al	; set index of the byte
		inc	eax		; increase index of the byte
		xchg	eax,edi		; EDI <- save new index of the byte
		SHORT_DELAY		; short delay
		test	dl,RTC_BIN	; BIN mode?
		jnz	CMOSWriteBCD2	; BIN mode
		BINBCD			; convert BIN to BCD
CMOSWriteBCD2:	out	CMOS_DATA,al	; write byte to CMOS
		ret

; -----------------------------------------------------------------------------
;     Internal function: Read alarm from CMOS in BCD or BIN mode into buffer
; -----------------------------------------------------------------------------
; INPUT:	AL = index of the byte in CMOS (in BCD or BIN mode)
;		AH = flags to set into CL if byte is valid
;		EAX = bit 31 "BIN mode" flag
;		EDI = buffer to store byte (in BIN mode)
;		CL = flag accumulator
; OUTPUT:	EDI = EDI + 1
; DESTROYS:	AL, AH
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

CMOSReadAlarm:	out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		in	al,CMOS_DATA	; read byte from the CMOS
		cmp	al,0c0h		; is byte valid?
		jae	CMOSReadAlarm4	; byte is not valid
		or	cl,ah		; set byte valid flag
		or	eax,eax		; BIN mode flag?
		js	CMOSReadAlarm2	; byte is in BIN mode
		BCDBIN			; convert from BCD to BIN
CMOSReadAlarm2:	stosb			; store byte
		dec	edi		; decrease pointer
CMOSReadAlarm4:	inc	edi		; increase pointer
		ret

; -----------------------------------------------------------------------------
;     Internal function: Write alarm to CMOS in BCD or BIN mode from buffer
; -----------------------------------------------------------------------------
; INPUT:	AL = index of the byte in CMOS (in BCD or BIN mode)
;		AH = flags to test in CL
;		EAX = bit 31 "BIN mode" flag
;		ESI = buffer to load byte (in BIN mode)
;		CL = flags
; OUTPUT:	ESI = ESI + 1
; DESTROYS:	AL, AH
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

CMOSWriteAlarm:	out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		inc	esi		; skip data
		mov	al,0ffh		; AL <- invalid value
		test	cl,ah		; is byte valid?
		jz	CMOSWriteAlarm2	; byte is not valid
		dec	esi		; return pointer
		lodsb			; load seconds
		or	eax,eax		; BIN mode flag?
		js	CMOSWriteAlarm2	; byte is in BIN mode
		BINBCD			; convert BIN to BCD
CMOSWriteAlarm2:out	CMOS_DATA,al	; write byte to CMOS
		ret

; *****************************************************************************
;
;                           Driver interface functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Install CMOS device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSInstall:	push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Register CMOS device

		mov	ebx,CMOSDev	; EBX <- device descriptor
		xor	ecx,ecx		; ECX <- 0, no parent device
		call	DevRegister	; register device

; ------------- Pop registers

		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                           Uninstall CMOS device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSUninstall:	push	ebx		; push EBX

; ------------- Unregister CMOS device

		mov	ebx,CMOSDev	; EBX <- device descriptor
		call	DevUnregister	; unregister device

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize CMOS device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

CMOSDevInit:	clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Deinitialize CMOS device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

CMOSDevDeinit:	clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                 Driver function: Read byte from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
;		EDX = index of byte (0 to 127)
; OUTPUT:	EAX = read byte (0 to 255, 0 on error)
;		CY = invalid index (EAX = 0)
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

; ------------- Check index of the byte

CMOSDevGetByte:	xor	eax,eax		; EAX <- 0
		cmp	edx,128		; check index of byte
		cmc			; CY = invalid index
		jc	CMOSDevGetByte4	; invalid index

; ------------- Read data

		GET_CMOS dl		; AL <- read CMOS byte
		clc			; clear error flag
CMOSDevGetByte4:ret

; -----------------------------------------------------------------------------
;                 Driver function: Write byte to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	AL = data
;		EBX = device descriptor DEVCMOS
;		EDX = index of data (0 to 127)
;		CY = invalid index
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

; ------------- Check index of the byte

CMOSDevSetByte:	cmp	edx,128		; check index of byte
		cmc			; CY = invalid index
		jc	CMOSDevSetByte4	; invalid index

; ------------- Write byte

		xchg	eax,edx		; AL <- index of data
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		xchg	eax,edx		; AL <- byte to write to CMOS
		out	CMOS_DATA,al	; write byte into CMOS
CMOSDevSetByte4:ret

; -----------------------------------------------------------------------------
;                 Driver function: Read data from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = data buffer
;		EBX = device descriptor DEVCMOS
;		ECX = number of bytes (0 to 128)
;		EDX = start index (0 to 127)
; OUTPUT:	CY = invalid parameters (no data read)
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevGetData:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	edi		; push EDI
		xchg	eax,edi		; EDI <- data buffer

; ------------- Check start index

		cmp	edx,128		; check start index
		cmc			; CY = invalid index
		jc	CMOSDevGetData8	; invalid index

; ------------- Check number of bytes

		mov	eax,edx		; EAX <- start index
		add	eax,ecx		; EAX <- end index
		jc	CMOSDevGetData8	; invalid number of bytes
		cmp	eax,129		; check end index
		cmc			; CY = invalid number of bytes
		jc	CMOSDevGetData8	; invalid number of bytes
		
; ------------- Read data from CMOS

		jecxz	CMOSDevGetData8	; no data to read (here is NC)
		cld			; set direction UP
CMOSDevGetData4:GET_CMOS dl		; AL <- read CMOS byte
		inc	edx		; increase index of byte
		stosb			; store byte
		loop	CMOSDevGetData4	; get next byte
		clc			; clear erro flag

; ------------- Pop registers

CMOSDevGetData8:pop	edi		; pop EDI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Driver function: Write data to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = data buffer
;		EBX = device descriptor DEVCMOS
;		ECX = number of bytes (0 to 128)
;		EDX = start index (0 to 127)
; OUTPUT:	CY = invalid parameters (no data writen)
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevSetData:	push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		xchg	eax,esi		; ESI <- data buffer

; ------------- Check start index

		cmp	edx,128		; check start index
		cmc			; CY = invalid index
		jc	CMOSDevSetData8	; invalid index

; ------------- Check number of bytes

		mov	eax,edx		; EAX <- start index
		add	eax,ecx		; EAX <- end index
		jc	CMOSDevSetData8	; invalid number of bytes
		cmp	eax,129		; check end index
		cmc			; CY = invalid number of bytes
		jc	CMOSDevSetData8	; invalid number of bytes
		
; ------------- Write data to CMOS

		jecxz	CMOSDevSetData8	; no data to write (here is NC)
		cld			; set direction UP
CMOSDevSetData4:mov	al,dl		; AL <- index of data
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		lodsb			; AL <- load byte
		inc	edx		; increase index of byte
		out	CMOS_DATA,al	; write byte into CMOS
		loop	CMOSDevSetData4	; get next byte
		clc			; clear erro flag

; ------------- Pop registers

CMOSDevSetData8:pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;              Driver function: Read date and time from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
;		EDX = pointer to DATETIME buffer
; OUTPUT:	CY = time update is in progress (try again later) or DATETIME
;			entries are not	in valid range
; NOTES:	Time update flag can be set up to 2 ms.
;		Values are loaded into buffer even on error state.
;		It takes aprox. 40 us.
;		On some PC's day of week may be invalid (replaced with Sunday).
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevGetTime:	push	eax		; push EAX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare registers

		mov	edi,edx		; EDI <- DATETIME buffer
		cld			; set direction UP

; ------------- Clear nanoseconds

		xor	eax,eax		; EAX <- 0
		stosd			; clear nanoseconds
		xchg	eax,esi		; ESI <- 0, index of register
		push	edi		; push buffer address

; ------------- Get "time update" flag on start of transmission

		GET_CMOS CMOS_STATUS_A	; read status register A
		shl	eax,16+3	; save "time update" flag

; ------------- Get "BIN mode" flag (-> bit 31)

		GET_CMOS CMOS_STATUS_B	; read status register B
		ror	eax,3		; roll BIN flag into bit 31

; ------------- Read seconds (index CMOS_SEC = 0)

		call	CMOSReadBCD	; read data

; ------------- Read minutes (index CMOS_MIN = 2)

		inc	esi		; icrease index of register
		call	CMOSReadBCD	; read data

; ------------- Read hours (index CMOS_HOUR = 4)

		inc	esi		; icrease index of register
		call	CMOSReadBCD	; read data

; ------------- Read day of week (index CMOS_DAYWEEK = 6)

		inc	esi		; icrease index of register
		call	CMOSReadBCD	; read data
		dec	al		; correction
		jz	CMOSDevGetTime1	; it is Sunday
		cmp	al,7		; check day of week
		jb	CMOSDevGetTime2	; day of week is OK
CMOSDevGetTime1:mov	al,7		; set to Sunday
CMOSDevGetTime2:dec	edi		; return buffer pointer
		stosb			; store day of week

; ------------- Read day in month (index CMOS_DAY = 7)

		call	CMOSReadBCD	; read data

; ------------- Read month (index CMOS_MONTH = 8)

		call	CMOSReadBCD	; read data

; ------------- Read year (index CMOS_YEAR = 9)

		call	CMOSReadBCD	; read data
		mov	ah,0		; AX = year
		cmp	al,80		; year 1980 or more
		jae	CMOSDevGetTime4	; it is year 1980 or more
		add	al,100		; century correction
CMOSDevGetTime4:add	ax,1900		; add century
		dec	edi		; return buffer pointer
		stosw			; store year

; ------------- Check seconds

		pop	esi		; ESI <- start of time
		lodsb			; load seconds
		cmp	al,59		; check seconds
		ja	CMOSDevGetTime6	; invalid entry

; ------------- Check minutes

		lodsb			; load minutes
		cmp	al,59		; check minutes
		ja	CMOSDevGetTime6	; invalid entry

; ------------- Check hours

		lodsb			; load hours
		cmp	al,23		; check hours
		ja	CMOSDevGetTime6	; invalid entry

; ------------- Check day in month

		inc	esi		; skip day of week
		lodsb			; load day in month
		cmp	al,1		; check day in month minimal
		jb	CMOSDevGetTime6	; invalid entry
		cmp	al,31		; check day in month maximal
		ja	CMOSDevGetTime6	; invalid entry

; ------------- Check month

		lodsb			; load month
		cmp	al,1		; check month minimal
		jb	CMOSDevGetTime6	; invalid entry
		cmp	al,12		; check month maximal
		ja	CMOSDevGetTime6	; invalid entry

; ------------- Get and check "time update" flag on end of transmission
; This second check is NOT necessary - after setting UIP flag we have 244 us.

		GET_CMOS CMOS_STATUS_A	; read status register A
		and	eax,(RTC_UIP << 16) + RTC_UIP ; check time update flag
		jz	CMOSDevGetTime8	; time update is not in progress

; ------------- Pop registers

CMOSDevGetTime6:stc			; set error flag
CMOSDevGetTime8:pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;              Driver function: Write date and time to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
;		EDX = pointer to DATETIME buffer
; OUTPUT:	CY = error (currently not used, allways NC)
; NOTES:	It takes aprox. 50 us.
;		Next time increment begins 0.5 sec later.
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevSetTime:	push	eax		; push EAX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare registers

		mov	esi,edx		; ESI <- DATETIME buffer
		cld			; set direction UP
		lodsd			; skip nanoseconds
		xor	edi,edi		; EDI <- 0, index of data

; ------------- Set RTC_SET flag (=abort update cycle)

		GET_CMOS CMOS_STATUS_B	; read status register B
		mov	dl,al		; DL <- save register B
		or	al,RTC_SET	; set RTC_SET flag
		out	CMOS_DATA,al	; write byte into CMOS

; ------------- Reset stage divider

		GET_CMOS CMOS_STATUS_A	; read status register A
		mov	dh,al		; DH <- save register A
		or	al,70h		; set RESET divider flag
		out	CMOS_DATA,al	; write byte into CMOS

; ------------- Write seconds (index CMOS_SEC = 0)

        	lodsb			; load seconds
		call	CMOSWriteBCD	; write data

; ------------- Write minutes

        	lodsb			; load minutes
		inc	edi		; icrease index of register
		call	CMOSWriteBCD	; write data

; ------------- Write hours

        	lodsb			; load hours
		inc	edi		; icrease index of register
		call	CMOSWriteBCD	; write data

; ------------- Write day of week

        	lodsb			; load day of week
		inc	eax		; correction
		cmp	al,8		; Sunday?
		jb	CMOSDevSetTime2	; no
		mov	al,1		; AL <- 1, Sunday
CMOSDevSetTime2:inc	edi		; icrease index of register
		call	CMOSWriteBCD	; write data

; ------------- Write day in month

        	lodsb			; load day in month
		call	CMOSWriteBCD	; write data

; ------------- Write month

        	lodsb			; load month
		call	CMOSWriteBCD	; write data

; ------------- Write year

        	lodsw			; load year
		sub	ax,1900		; year correction
		cmp	al,99		; maximal value
		jbe	CMOSDevSetTime4	; value is OK
		sub	al,100		; century correction
CMOSDevSetTime4:call	CMOSWriteBCD	; write data

; ------------- Return status register B

		mov	al,CMOS_STATUS_B ; AL <- index of status reg. B
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		xchg	eax,edx		; AL <- byte to write to CMOS
		out	CMOS_DATA,al	; write byte into CMOS

; ------------- Return status register A

		mov	al,CMOS_STATUS_A ; AL <- index of status reg. A
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		xchg	al,ah		; AL <- byte to write to CMOS
		out	CMOS_DATA,al	; write byte into CMOS

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	eax		; pop EAX
		clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Read alarm from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
;		EDX = pointer to DATETIME buffer
; OUTPUT:	EAX = flags (ALARM_ENABLED, ...)
;		CY = error (currently always NC)
; NOTES:	It takes aprox. 20 us.
;		Unused entries are initialized to 0:00:00 1/1/1
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevGetAlrm:	push	ecx		; push ECX
		push	edi		; push EDI

; ------------- Initialize DATETIME buffer (-> ECX = 0)

		cld			; set direction UP
		mov	edi,edx		; EDI <- DATETIME buffer
		xor	eax,eax		; EAX <- 0
		mov	ecx,DATETIME_size/4 ; ECX <- size of the buffer
		rep	stosd		; clear DATETIME buffer
		mov	edi,edx		; EDI <- DATETIME buffer
		stosd			; skip nanoseconds
		inc	eax		; EAX <- 1
		mov	byte [edi-4+DATETIME_WDay],6 ; Saturday
		mov	[edi-4+DATETIME_Day],al ; day in month
		mov	[edi-4+DATETIME_Month],al ; month
		mov	[edi-4+DATETIME_Year],al ; year

; ------------- Get "BIN mode" flag (-> bit 31), check "alarm enabled" flag

		GET_CMOS CMOS_STATUS_B	; read status register B
		test	al,RTC_AIE	; alarm interrupt enabled?
		jz	CMOSDevGetAlrm2	; alarm not enabled
		or	cl,ALARM_ENABLED ; set "alarm enabled" flag
CMOSDevGetAlrm2:ror	eax,3		; roll BIN flag into bit 31

; ------------- Read seconds

		mov	ax,ALARM_USE_SEC*256 + CMOS_ALARM_SEC ; index+flag
		call	CMOSReadAlarm	; read data

; ------------- Read minutes

		mov	ax,ALARM_USE_MIN*256 + CMOS_ALARM_MIN ; index+flag
		call	CMOSReadAlarm	; read data

; ------------- Read hours

		mov	ax,ALARM_USE_HOUR*256 + CMOS_ALARM_HOUR ; index+flag
		call	CMOSReadAlarm	; read data

; ------------- Pop registers

		xchg	eax,ecx		; EAX <- flags
		clc			; clear error flag
		pop	edi		; pop EDI
		pop	ecx		; pop ECX
		ret

; -----------------------------------------------------------------------------
;                Driver function: Write alarm to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = flags (ALARM_ENABLED, ...)
;		EBX = device descriptor DEVCMOS
;		EDX = pointer to DATETIME buffer
; OUTPUT:	CY = error (currently not used, allways NC)
; NOTES:	It takes aprox. 20 us.
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevSetAlrm:	push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI

; ------------- Prepare registers

		xchg	eax,ecx		; ECX <- flags
		mov	esi,edx		; ESI <- DATETIME buffer
		cld			; set direction UP
		lodsd			; skip nanoseconds

; ------------- Get "BIN mode" flag (-> bit 31), clear "alarm enabled" flag

		GET_CMOS CMOS_STATUS_B	; read status register B
		mov	ch,al		; CH <- save status byte
		test	al,RTC_AIE	; was interrupt enabled?
		jz	CMOSDevSetAlrm2	; interrupt was not enabled
		and	al,~RTC_AIE	; clear alarm enabled flag
		out	CMOS_DATA,al	; set new alarm enabled flag
CMOSDevSetAlrm2:ror	eax,3		; roll BIN flag into bit 31

; ------------- Write seconds

		mov	ax,ALARM_USE_SEC*256 + CMOS_ALARM_SEC ; index + flag
		call	CMOSWriteAlarm	; write data

; ------------- Write minutes

		mov	ax,ALARM_USE_MIN*256 + CMOS_ALARM_MIN ; index + flag
		call	CMOSWriteAlarm	; write data

; ------------- Write hours

		mov	ax,ALARM_USE_HOUR*256 + CMOS_ALARM_HOUR ; index + flag
		call	CMOSWriteAlarm	; write data

; ------------- Alarm enable

		test	cl,ALARM_ENABLED ; check flag
		jz	CMOSDevSetAlrm4	; alarm is not enabled
		mov	al,CMOS_STATUS_B ; AL <- index of status register B
		out	CMOS_INDEX,al	; set index of the byte
		SHORT_DELAY		; short delay
		mov	al,ch		; AL <- old value
		or	al,RTC_AIE	; set alarm enabled flag
		out	CMOS_DATA,al	; set new alarm enabled flag

; ------------- Pop registers

CMOSDevSetAlrm4:clc			; clear error flag
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;             Driver function: Get periodic interrupt rate (IRQ 8)
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
; OUTPUT:	CY = error (currently not used, allways NC)
;		EAX = periodic interrupt rate (CMOS_RATE_NONE,...)
; NOTES:	It takes aprox. 8 us.
; -----------------------------------------------------------------------------

; ------------- Check if periodic interrupt is enabled

CMOSDevGetRate:	GET_CMOS CMOS_STATUS_B	; read status register B
		and	eax,RTC_PIE	; is periodic interrupt enabled?
		jz	CMOSDevGetRate9	; periodic interrupt is disabled

; ------------- Get periodic interrupt rate

		GET_CMOS CMOS_STATUS_A	; read status register A
		and	al,0fh		; mask rate selection bits
CMOSDevGetRate9:ret

; -----------------------------------------------------------------------------
;             Driver function: Set periodic interrupt rate (IRQ 8)
; -----------------------------------------------------------------------------
; INPUT:	EAX = periodic interrupt rate (CMOS_RATE_NONE,...)
;		EBX = device descriptor DEVCMOS
; OUTPUT:	CY = error, invalid rate selection
; NOTES:	It takes aprox. 12 us.
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevSetRate:	push	eax		; push EAX

; ------------- Check rate selection value

		cmp	eax,CMOS_RATE_MAX+1 ; check rate selection
		cmc			; CY = error flag
		jc	CMOSDevSetRate9	; invalid rate selection

; ------------- Check if periodic interrupt is enabled

		or	eax,eax		; should be disabled?
		jz	CMOSDevSetRate6	; disable periodic interrupt

; ------------- Set periodic interrupt rate

		mov	ah,al		; AH <- save rate selection
		GET_CMOS CMOS_STATUS_A	; read status register A
		and	al,~0fh		; clear rate selection bits
		or	al,ah		; set rate selection bits
		SHORT_DELAY		; short delay
		out	CMOS_DATA,al	; write byte to CMOS

; ------------- Enable periodic interrupt (it returns NC)

		mov	ah,RTC_PIE	; enable periodic interrupt

; ------------- Set periodic interrupt (AH=PIE bit, it returns NC)

CMOSDevSetRate6:GET_CMOS CMOS_STATUS_B	; read status register B
		and	al,~RTC_PIE	; disable periodic interrupt
		or	al,ah		; set periodic interrupt bit
		SHORT_DELAY		; short delay
		out	CMOS_DATA,al	; write byte to CMOS

; ------------- Pop registers

CMOSDevSetRate9:pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Acknowledge interrupt
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
; OUTPUT:	CY = error (currently not used, allways NC)
; NOTES:	It takes aprox. 4 us.
;		It reads status register C.
; -----------------------------------------------------------------------------

CMOSDevAckInt:	push	eax		; push EAX
		GET_CMOS CMOS_STATUS_C	; read status register C
		clc			; clear error flag
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                     Driver function: Reassert NMI
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCMOS
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSDevNextNMI:	push	eax		; push EAX

; ------------- Disable NMI

		mov	al,B7 + CMOS_REG_DEF ; AL <- NMI disabled
		out	CMOS_INDEX,al	; disable NMI
		in	al,CMOS_DATA	; ...dummy

; ------------- Enable NMI

		mov	al,CMOS_REG_DEF	; AL <- NMI enabled
		out	CMOS_INDEX,al	; enable NMI
		in	al,CMOS_DATA	; ...dummy

; ------------- Pop registers

		pop	eax		; pop EAX
		ret

; *****************************************************************************
;
;                            CMOS global functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;               CMOS global function: Read byte from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EDX = index of byte (0 to 127)
; OUTPUT:	EAX = read byte (0 to 255, 0 on error)
;		CY = invalid index (EAX = 0)
; NOTES:	It locks fast spin-lock of device interface.
; NOTES:	It takes aprox. 4 us.
; -----------------------------------------------------------------------------

CMOSGetByte:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_GetByte ; read byte
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                CMOS global function: Write byte to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	AL = data
;		EDX = index of data (0 to 127)
; OUTPUT:	CY = invalid index
; -----------------------------------------------------------------------------

CMOSSetByte:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_SetByte ; write byte
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               CMOS global function: Read data from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = data buffer
;		ECX = number of bytes (0 to 128)
;		EDX = start index (0 to 127)
; OUTPUT:	CY = invalid parameters (no data read)
; -----------------------------------------------------------------------------

CMOSGetData:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_GetData ; read data
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                CMOS global function: Write data to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = data buffer
;		ECX = number of bytes (0 to 128)
;		EDX = start index (0 to 127)
; OUTPUT:	CY = invalid parameters (no data writen)
; -----------------------------------------------------------------------------

CMOSSetData:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_SetData ; write data
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;            CMOS global function: Read date and time from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to DATETIME buffer
; OUTPUT:	CY = time update is in progress (try again later) or DATETIME
;			entries are not	in valid range
; NOTES:	Time update flag can be set up to 2 ms.
;		Values are loaded into buffer even on error state.
;		It takes aprox. 40 us.
;		On some PC's day of week may be invalid (-> Sunday).
; -----------------------------------------------------------------------------

CMOSGetTime:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_GetTime ; read date and time
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;            CMOS global function: Write date and time to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to DATETIME buffer
; OUTPUT:	CY = error (currently not used, allways NC)
; NOTES:	It takes aprox. 50 us.
;		Next time increment begins 0.5 sec later.
; -----------------------------------------------------------------------------

CMOSSetTime:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_SetTime ; write date and time
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               CMOS global function: Read alarm from CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EDX = pointer to DATETIME buffer
; OUTPUT:	EAX = flags (ALARM_ENABLED, ...)
;		CY = error
; NOTES:	It takes aprox. 20 us.
;		Unused entries (or on error) are initialized to 0:00:00 1/1/1
; -----------------------------------------------------------------------------

CMOSGetAlarm:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_GetAlrm ; read alarm
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               CMOS global function: Write alarm to CMOS memory
; -----------------------------------------------------------------------------
; INPUT:	EAX = flags (ALARM_ENABLED, ...)
;		EDX = pointer to DATETIME buffer
; OUTPUT:	CY = error
; NOTES:	It takes aprox. 20 us.
; -----------------------------------------------------------------------------

CMOSSetAlarm:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_SetAlrm ; write alarm
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;           CMOS global function: Get periodic interrupt rate (IRQ 8)
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
;		EAX = periodic interrupt rate (CMOS_RATE_NONE,...)
; NOTES:	It takes aprox. 8 us.
; -----------------------------------------------------------------------------

CMOSGetRate:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_GetRate ; get interrupt rate
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;           CMOS global function: Set periodic interrupt rate (IRQ 8)
; -----------------------------------------------------------------------------
; INPUT:	EAX = periodic interrupt rate (CMOS_RATE_NONE,...)
; OUTPUT:	CY = error, invalid rate selection
; NOTES:	It takes aprox. 12 us.
; -----------------------------------------------------------------------------

CMOSSetRate:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_SetRate ; set interrupt rate
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                 CMOS global function: Acknowledge interrupt
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error (currently not used, allways NC)
; NOTES:	It takes aprox. 4 us.
;		It reads status register C.
; -----------------------------------------------------------------------------

CMOSAckInt:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_AckInt ; acknowledge interrupt
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                      CMOS global function: Reassert NMI
; -----------------------------------------------------------------------------

CMOSNextNMI:	push	ebx		; push EBX
		mov	ebx,[CMOSDevice] ; EBX <- current CMOS device
		DEVFNCLOCKC DEVCMOS_NextNMI ; get IRQ channel info
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                    CMOS global function: Read CMOS time
; -----------------------------------------------------------------------------
; OUTPUT:	EDX:EAX = absolute time (100-nanoseconds from 1/1/1 CE 0:00)
;		CY = error (EDX:EAX = 0)
; NOTES:	Function takes normaly 40 us but it can take up to 10 ms.
; -----------------------------------------------------------------------------

; ------------- Push registers

GetCMOSTime:	push	ebx		; push EBX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- local variables
		sub	esp,DATETIME_size ; buffer for date-time structure

; ------------- Read date and time from CMOS (wait 256 * 40 = 10240 us)

		xor	ecx,ecx		; ECX <- 0
		mov	ch,1		; ECX <- 256, max. attempts (= 10 ms)
		xor	esi,esi		; ESI <- 0, old value - time
		xor	ebx,ebx		; EBX <- 0, old value - date
GetCMOSTime2:	mov	edx,esp		; EDX <- DATETIME buffer in stack
		call	CMOSGetTime	; read date and time
		jc	GetCMOSTime6	; error

; ------------- Check old value if it has been changed

		cmp	esi,[ebp-DATETIME_size+DATETIME_Time] ; check time
		jne	GetCMOSTime4	; change, try again
		cmp	ebx,[ebp-DATETIME_size+DATETIME_Date] ; check date
		jne	GetCMOSTime4	; change, try again

; ------------- Convert system time to absolute time
		
		mov	ebx,esp		; EBX <- DATETIME buffer in stack
		call	DateTimeToAbs	; convert date-time to absolute time
		clc			; clear error flag
		jmp	short GetCMOSTime8

; ------------- Next attempt

GetCMOSTime4:	mov	esi,[ebp-DATETIME_size+DATETIME_Time] ; ESI <- get time
		mov	ebx,[ebp-DATETIME_size+DATETIME_Date] ; EBX <- get date
GetCMOSTime6:	loop	GetCMOSTime2	; next attempt

; ------------- Error

		xor	eax,eax		; EAX <- 0
		xor	edx,edx		; EDX <- 0
		stc			; set error flag

; ------------- Pop registers

GetCMOSTime8:	mov	esp,ebp		; EPS <- return stack
		pop	ebp		; pop EBP
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                    CMOS global function: Write CMOS time
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanoseconds from 1/1/1 CE 0:00)
; OUTPUT:	CY = error
; NOTES:	Function takes aprox. 50 us.
;		Next time increment begins 0.5 sec later.
; -----------------------------------------------------------------------------

; ------------- Push registers

SetCMOSTime:	push	ebx		; push EBX
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- local variables
		sub	esp,DATETIME_size ; buffer for date-time structure

; ------------- Convert absolute time to DATETIME

		mov	ebx,esp		; EBX <- DATETIME buffer in stack
		call	AbsToDateTime	; convert absolute time to date-time

; ------------- Write date and time to CMOS memory

		mov	edx,esp		; EDX <- DATETIME buffer in stack
		call	CMOSSetTime	; write date and time to CMOS memory

; ------------- Pop registers

		mov	esp,ebp		; EPS <- return stack
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                    CMOS global function: Read CMOS alarm
; -----------------------------------------------------------------------------
; OUTPUT:	EDX:EAX = absolute time (100-nanoseconds from 1/1/1 CE 0:00)
; 		ECX = flags (ALARM_ENABLED, ...)
;		CY = error (EDX:EAX = 0)
; NOTES:	It takes aprox. 20 us.
; -----------------------------------------------------------------------------

; ------------- Push registers

GetCMOSAlarm:	push	ebx		; push EBX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- local variables
		sub	esp,DATETIME_size ; buffer for date-time structure

; ------------- Read alarm from CMOS

		mov	edx,esp		; EDX <- DATETIME buffer in stack
		call	CMOSGetAlarm	; read alarm
		xchg	eax,ecx		; ECX <- flags

; ------------- Convert system time to absolute time

		mov	ebx,esp		; EBX <- DATETIME buffer in stack
		pushf			; push flags (CY = error)
		call	DateTimeToAbs	; convert date-time to absolute time
		popf			; pop flags (CY = error)

; ------------- Pop registers

		mov	esp,ebp		; EPS <- return stack
		pop	ebp		; pop EBP
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                    CMOS global function: Write CMOS alarm
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = absolute time (100-nanoseconds from 1/1/1 CE 0:00)
; 		ECX = flags (ALARM_ENABLED, ...)
; OUTPUT:	CY = error
; NOTES:	Function takes aprox. 20 us.
; -----------------------------------------------------------------------------

; ------------- Push registers

SetCMOSAlarm:	push	ebx		; push EBX
		push	edx		; push EDX
		push	ebp		; push EBP
		mov	ebp,esp		; EBP <- local variables
		sub	esp,DATETIME_size ; buffer for date-time structure

; ------------- Convert absolute time to DATETIME

		mov	ebx,esp		; EBX <- DATETIME buffer in stack
		call	AbsToDateTime	; convert absolute time to date-time

; ------------- Write alarm to CMOS memory

		xchg	eax,ecx		; EAX <- flags, ECX <- time LOW
		mov	edx,esp		; EDX <- DATETIME buffer in stack
		call	CMOSSetAlarm	; write alarm to CMOS memory
		xchg	eax,ecx		; return EAX and ECX

; ------------- Pop registers

		mov	esp,ebp		; EPS <- return stack
		pop	ebp		; pop EBP
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Text strings

		align	4, db 0
CMOSDevName:	CTEXTDATA 'cmos'
CMOSDevShort:	CTEXTDATA 'CMOS MC146818'

CMOSDevFull:	LANGTEXTSTR CMOSDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
 		LANGTEXTSTR CMOSDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

CMOSDevFullEN:	CTEXTDATA 'CMOS memory MC146818'
CMOSDevFullCZ:	CTEXTDATA 'Pam',0c4h,9bh,0c5h,0a5h,' CMOS MC146818'

CMOSDevInt:	dd	DEV_GEN_ID
		dd	DEV_CMOS_ID
		dd	DEV_NUL_ID

CMOSDevRes1Name:CTEXTDATA 'ctrl'
CMOSDevRes2Name:CTEXTDATA 'irq'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- CMOS device descriptor

		align	4, db 0
CMOSDevice:	dd	CMOSDev			; current CMOS device descr.

%define		CMOSDevVendor  DefaultVendor

		align	8, db 0
CMOSDev:	DEVICECMOS 0,DEV_STATIC,1,0,0,CMOSDev

CMOSDevRes1:	DEVRESOURCE CMOSDevResN,CMOSDevRes0,CMOS_BASE,CMOS_BASE+1, \
		 DEVRES_PORT,DEVRES_STATIC,CMOSDevRes1Name

CMOSDevResN:	DEVRESOURCE CMOSDevRes0,CMOSDevRes1,CMOS_IRQ,CMOS_IRQ, \
		 DEVRES_IRQ,DEVRES_STATIC,CMOSDevRes2Name
