; =============================================================================
;
;                             Litos - Wall clock
;
; =============================================================================
; CLOCK driver uses TIMER device.

; PC Real-time clock - interrupted every 1 ms:
;	Hardware clock generator: 1193182 Hz
; 	Timer divisor: 1193
;       Interrupt frequency: 1000.1525566 Hz (0.9998474667 ms)
;	Increment in 100-nanosecs: 9998.474666899
;	Low DWORD increment (decimal part): 2038678808 (7983C518h)
;	High DWORD increment: 9998
;	Clock ticks per day: 86413180.89
; =============================================================================
; TODO: Synchronize time-stamp counter on all CPUs (or use add-on correction)
; TODO: Adjust time precission correcting time delta in run-time.
; TODO: Synchronize CMOS access with NMI interrupt (it cannot be disabled).

;TIMER_IRQ	EQU	0		; IRQ for system timer

; ------------- Clock constants

TIME_DIVISOR	EQU	1193		; real-time clock divisor (1000.153 Hz)
TIME_DELTA_DEC	EQU	2038678808	; time delta in 100-ns, decimal part
TIME_DELTA_INT	EQU	9998		; time delta in 100-ns, integer part
TIME_1MS	EQU	10000		; time 1 ms in 100-ns
TIME_1MS_REAL	EQU	9998		; real precision of time 1 ms in 100-ns

		CODE_SECTION

; *****************************************************************************
;
;                           Driver interface functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Install CLOCK device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

CLOCKInstall:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX
		push	edx		; push EDX

; ------------- Register CLOCK device

		mov	ebx,CLOCKDev	; EBX <- device descriptor
		xor	ecx,ecx		; ECX <- 0, no parent device
		call	DevRegister	; register device
		jc	CLOCKInstall9	; error

; ------------- Initialize device

		call	DevInit		; initialize CLOCK device
		jc	CLOCKInstall9	; error

; ------------- Install CLOCK interrupt handler

		mov	ebx,CLOCKHandler ; EBX <- CLOCK handler
		call	IntInstall	; install interrupt handler
		jc	CLOCKInstall9	; error

; ------------- Initialize TIMER 0 channel (here is EAX = 0)

		mov	al,DEVTIM_MODE_GEN ; EAX <- mode 2, rate generator
		mov	ecx,TIME_DIVISOR ; ECX <- counter initial value
		xor	edx,edx		; EDX <- counter 0
		call	TIMERSetMode	; set timer mode
		jc	CLOCKInstall9	; error


; ------------- Install CLOCK interrupt handler

;		mov	ebx,CMOSHandler ; EBX <- CLOCK handler
;		call	IntInstall	; install interrupt handler
;		jc	CMOSInstall9	; error

; ------------- Initialize TIMER 0 channel (here is EAX = 0)

;		mov	eax,CMOS_RATE_4
;		call	CMOSSetRate


; ------------- Pop registers

CLOCKInstall9:	pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                           Uninstall CLOCK device
; -----------------------------------------------------------------------------
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

CLOCKUninstall:	push	ebx		; push EBX

; ------------- Unregister CLOCK device

		mov	ebx,CLOCKDev	; EBX <- device descriptor
		call	DevUnregister	; unregister device

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Initialize CLOCK device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCLOCK
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

CLOCKDevInit:	clc			; clear error flag
		ret

; -----------------------------------------------------------------------------
;                   Driver function: Deinitialize CLOCK device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEVCLOCK
; OUTPUT:	CY = error
; -----------------------------------------------------------------------------

CLOCKDevDeinit:	clc			; clear error flag
		ret









; -----------------------------------------------------------------------------
;                    Timer interrupt (called every 1 ms)
; -----------------------------------------------------------------------------

; ------------- Push registers

TimerInt:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ds		; push DS
		push	es		; push ES

; ------------- Initialize registers

		mov	eax,SYSTEM_DS	; EAX <- system data segment
		mov	ds,eax		; DS <- system data segment
		mov	es,eax		; ES <- system data segment
		cld			; direction up


		; !!!!!!!!!!!!!!!
		inc	dword [SYSTEM_ADDR+0b8000h]



; ------------- Acknowledge interrupt IRQ 0

		xor	eax,eax
		call	IRQAck		; acknowledge interrupt		

; ------------- Get current task (-> EBX)

		CURRENT	ebx		; EBX <- get current task

; ------------- Increase current time (only for CPU 0)
%ifdef	SMP
		cmp	byte [ebx+TASK_CPU],0 ; is it CPU 0 ?
		jne	TimerInt4	; it is not CPU 0
%endif
;		TIMELOCK		; lock current time
		add	dword [CurrentTimeDec],TIME_DELTA_DEC
		adc	dword [CurrentTime],TIME_DELTA_INT
		adc	dword [CurrentTime+4],byte 0
;		TIMEUNLOCK		; unlock current time

; ------------- Next alarm (only CPU 0)

		sub	dword [AlarmListNext],TIME_1MS_REAL ; next service
		jg	TimerInt4	; no service
		call	AlarmTime	; alarm service

; ------------- Next scheduler

TimerInt4:	mov	ebx,[ebx+TASK_RunQueue] ; EBX <- current run-queue
		add	dword [ebx+RUNQ_Elapsed],TIME_1MS_REAL ; elapsed time
		sub	dword [ebx+RUNQ_NextSched],TIME_1MS_REAL; next schedule
		jg	TimerInt9	; no next scheduling yet

; ------------- Call scheduler

		sti			; enable interrupts
		call	Schedule	; scheduler

; ------------- Pop registers

TimerInt9:	pop	es		; pop ES
		pop	ds		; pop DS
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		iret






; -----------------------------------------------------------------------------
;                    Timer interrupt (called every 1 ms)
; -----------------------------------------------------------------------------

; ------------- Push registers

CMOSInt:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ds		; push DS
		push	es		; push ES

; ------------- Initialize registers

		mov	eax,SYSTEM_DS	; EAX <- system data segment
		mov	ds,eax		; DS <- system data segment
		mov	es,eax		; ES <- system data segment
		cld			; direction up


		; !!!!!!!!!!!!!!!
		inc	dword [SYSTEM_ADDR+0b8010h]



; ------------- Acknowledge interrupt IRQ 8

		mov	eax,8
		call	IRQAck		; acknowledge interrupt		

		call	CMOSAckInt	; acknowledge interrupt

;		GET_CMOS CMOS_STATUS_C


; ------------- Pop registers

		pop	es		; pop ES
		pop	ds		; pop DS
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		iret






; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Text strings

		align	4, db 0
CLOCKDevName:	CTEXTDATA 'clock'
CLOCKDevShort:	CTEXTDATA 'Wall Clock'

CLOCKDevFull:	LANGTEXTSTR CLOCKDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
 		LANGTEXTSTR CLOCKDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

CLOCKDevFullEN:	CTEXTDATA 'Wall Clock'
CLOCKDevFullCZ:	CTEXTDATA 'Nastenne hodiny'

CLOCKDevInt:	dd	DEV_GEN_ID
		dd	DEV_CLOCK_ID
		dd	DEV_NUL_ID

CLOCKDevRes1Name:CTEXTDATA 'ctrl'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- CLOCK device descriptor

		align	4, db 0
CLOCKDevice:	dd	CLOCKDev		; current CLOCK device descr.

%define		CLOCKDevVendor  DefaultVendor

%define		CLOCKDevRes1 CLOCKDevRes0
%define		CLOCKDevResN CLOCKDevRes0

		align	8, db 0
CLOCKDev:	DEVICECLOCK DEV_STATIC,1,0,0,CLOCKDev

; ------------- CLOCK interrupt handler

		align	8, db 0
CLOCKHandler:	INTHANDLER INT_ACTIVE+INT_PRIVATE,TIMER_IRQ,(1<<TIMER_IRQ), \
			TimerInt





; ------------- CLOCK interrupt handler

		align	8, db 0
CMOSHandler:	INTHANDLER INT_ACTIVE+INT_PRIVATE,CMOS_IRQ,(1<<CMOS_IRQ), \
			CMOSInt





; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION
