; =============================================================================
;
;                           Litos - General device
;
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                        Get total number of devices
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = number of devices
; -----------------------------------------------------------------------------

DevNumber:	DEVNUMBER eax		; EAX <- number of devices
		ret

; -----------------------------------------------------------------------------
;  Check device descriptor address validity (test if it is in the device list)
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; OUTPUT:	CY = entry not found
; NOTES:	Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevCheck:	push	edx		; push EDX
		mov	edx,DevAdrList	; EDX <- device address list
		xchg	eax,ebx		; EAX <- entry
		call	RBTreeSrcAddr	; search entry
		xchg	eax,ebx		; EBX <- entry
		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                              Get parent of device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV (must be valid)
; OUTPUT:	EAX = parent device descriptor DEV (or NULL = it has no parent)
;		CY = device has no parent (EAX = NULL)
; -----------------------------------------------------------------------------

DevParent:	DEVPARENT ebx,eax,short DevParent2 ; get parent of device
		ret

DevParent2:	xor	eax,eax		; EAX <- no parent
DevParent4:	stc			; set error flag, no parent
		ret

; -----------------------------------------------------------------------------
;                       Find first device of given class
; -----------------------------------------------------------------------------
; INPUT:	AL = device class
; OUTPUT:	EBX = device descriptor DEV (or NULL if no such device)
;		CY = no device found (EBX = NULL)
; NOTES:	Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevFindClass:	DEVFINDCLASS al,ebx,short DevParent4 ; find first device
		ret

; -----------------------------------------------------------------------------
;                       Find next device of this class
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; OUTPUT:	EBX = device descriptor DEV (or NULL if no next device)
;		CY = no next device found (EBX = NULL)
; NOTES:	Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevNextClass:	DEVNEXTCLASS ebx,ebx,short DevParent4 ; find next device
		ret

; -----------------------------------------------------------------------------
;                  Find first device of given class and sub-class
; -----------------------------------------------------------------------------
; INPUT:	AH = device class
;		AL = device sub-class
; OUTPUT:	EBX = device descriptor DEV (or NULL if no such device)
;		CY = no device found (EBX = NULL)
; NOTES:	Input can be EAX = class*256 + subclass.
; 		Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevFindSubCls:	movzx	ebx,ah		; EBX <- device class
		lea	ebx,[DevHashList+ebx*4] ; EBX <- hash list head
DevFindSubCls2:	mov	ebx,[ebx+HASHE_Next] ; EBX <- next entry
		or	ebx,ebx		; check if it is valid entry
		jz	short DevParent4 ; invalid entry
		cmp	byte [ebx-DEV_Hash+DEV_SubClass],al ; check sub-class
		jne	DevFindSubCls2	; get next device
		sub	ebx,DEV_Hash	; EBX <- entry (it clears CF)
		ret

; -----------------------------------------------------------------------------
;                Find next device of this class and sub-class
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; OUTPUT:	EBX = device descriptor DEV (or NULL if no other device)
;		CY = no other device found (EBX = NULL)
; NOTES:	Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevNextSubCls:	push	eax		; push EAX
		add	ebx,DEV_Hash	; EBX <- hash entry
		mov	al,[ebx-DEV_Hash+DEV_SubClass] ; AL <- get sub-class
DevNextSubCls2:	mov	ebx,[ebx+HASHE_Next] ; EBX <- next entry
		or	ebx,ebx		; check if it is valid entry
		jz	short DevNextSubCls4 ; invalid entry
		cmp	byte [ebx-DEV_Hash+DEV_SubClass],al ; check sub-class
		jne	DevNextSubCls2	; get next device
		sub	ebx,DEV_Hash	; EBX <- entry (it clears CF)
		pop	eax		; pop EAX
		ret

DevNextSubCls4:	stc			; set error flag
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;              Find device of given class, sub-class and index
; -----------------------------------------------------------------------------
; INPUT:	AH = device class
;		AL = device sub-class
;		CL = device index (0 to 255)
; OUTPUT:	EBX = device descriptor DEV (or NULL if no such device)
;		CY = no device found (EBX = NULL)
; NOTES:	Input can be EAX = class*256 + subclass.
; 		Device list should be locked and interrupts should be disabled.
; -----------------------------------------------------------------------------

DevFindIndex:	push	eax		; push EAX
		movzx	ebx,ah		; EBX <- device class
		mov	ah,al		; AH <- sub-class
		lea	ebx,[DevHashList+ebx*4] ; EBX <- hash list head
		mov	al,cl		; AL <- index
DevFindIndex2:	mov	ebx,[ebx+HASHE_Next] ; EBX <- next entry
		or	ebx,ebx		; check if it is valid entry
		jz	short DevNextSubCls4 ; invalid entry
		cmp	word [ebx-DEV_Hash+DEV_InxSubW],ax ; check inx/subcls
		jne	DevFindIndex2	; get next device
		sub	ebx,DEV_Hash	; EBX <- entry (it clears CF)
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Register device (add device into device list)
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV (must be valid)
;		ECX = parent device descriptor (must be valid or NULL = system)
; OUTPUT:	CY = error, entry already exists in the list
; NOTES:	It locks device list lock.
;		It sets-up DEV_ROOT flag if needed.
; -----------------------------------------------------------------------------

; ------------- Push registers

DevRegister:	clc			; clear error flag
		pushf			; push flags
		cli			; disable interrupts
		push	eax		; push EAX
		push	edx		; push EDX

; ------------- Lock device list

		DEVLSTLOCK		; lock device list

; ------------- Insert entry into driver list

		mov	edx,DevAdrList	; EDX <- list of al drivers
		xchg	eax,ebx		; EAX <- new entry
		call	RBTreeInsAddr	; insert entry into list
		jc	DevRegister7	; entry already exists

; ------------- Add entry into hash list

		movzx	ebx,byte [eax+DEV_Class] ; EBX <- device class
		lea	ebx,[DevHashList+ebx*4] ; EBX <- hash list head
		add	eax,DEV_Hash	; EAX <- hash list entry
		call	HashAdd		; add new entry into hash list

; ------------- Add entry into hierarchy list

		mov	ebx,ecx		; EBX <- parent
		and	byte [eax-DEV_Hash+DEV_Flags],~DEV_ROOT;clear root flag
		or	ecx,ecx		; is it root device?
		jnz	DevRegister2	; it is not root device
		mov	ebx,DevDevList-DEV_DevList ; ECX <- list head
		or	byte [eax-DEV_Hash+DEV_Flags],DEV_ROOT ; set root flag
DevRegister2:	add	ebx,DEV_DevList	; EBX <- device hierarchy list entry
		add	eax,DEV_DevList-DEV_Hash ; EAX <- device list entry
		call	TreeAdd		; add entry into list
		sub	eax,DEV_DevList	; EAX <- new entry

; ------------- Register standard system devices

		mov	bx,[eax+DEV_SubClsW] ; BX <- class and sub-class
		cmp	bh,DEVCLASS_SYS	; system device?
		jne	DevRegister6	; not system device
		cmp	byte [eax+DEV_Index],0 ; first device?
		jne	DevRegister6	; not first device

; ------------- SYS device

		cmp	bl,DEV_SYS_SYS	; SYS device?
		jne	DevRegister4	; not SYS device
		mov	[SYSDevice],eax	; register SYS device

; ------------- DMA device

DevRegister4:	cmp	bl,DEV_SYS_DMA	; DMA device?
		jne	DevRegister5	; not DMA device
		mov	[DMADevice],eax	; register DMA device

; ------------- IRQ device

DevRegister5:	cmp	bl,DEV_SYS_IRQ	; IRQ device?
		jne	DevRegister52	; not IRQ device
		mov	[IRQDevice],eax	; register IRQ device

; ------------- CMOS device

DevRegister52:	cmp	bl,DEV_SYS_CMOS	; CMOS device?
		jne	DevRegister53	; not CMOS device
		mov	[CMOSDevice],eax ; register CMOS device

; ------------- TIMER device

DevRegister53:	cmp	bl,DEV_SYS_TIMER ; TIMER device?
		jne	DevRegister54	; not TIMER device
		mov	[TIMERDevice],eax ; register TIMER device

; ------------- CLOCK device

DevRegister54:	cmp	bl,DEV_SYS_CLOCK ; CLOCK device?
		jne	DevRegister6	; not CLOCK device
		mov	[CLOCKDevice],eax ; register CLOCK device

; ------------- Unlock device list (it saves flags, CY=error)

DevRegister6:	clc			; clear error flag
DevRegister7:	xchg	eax,ebx		; EBX <- new entry
		DEVLSTUNLOCK		; unlock device list

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	eax		; pop EAX

; ------------- Pop flags

		adc	byte [esp],0	; set error flag (bit 0)
		popf			; enable interrupts
		ret

; -----------------------------------------------------------------------------
;               Unregister device (remove device from device list)
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV (it must be valid entry)
; OUTPUT:	CY = cannot unregister, it is used
; NOTES:	Device descriptor DEV is not destroyed, it is only detached.
;		Reference counter must be zero and it must have no children.
; -----------------------------------------------------------------------------

; ------------- Disable interrupts

DevUnregister:	pushf			; push flags
		cli			; disable interrupts

; ------------- Lock device list

		DEVLSTLOCK		; lock device list

;-------------- Check reference counter and children

		cmp	dword [ebx+DEV_Ref],0 ; check reference counter
		jne	DevUnregister8	; cannot unregister
		cmp	dword [ebx+DEV_DevList+TREE_Child],0 ; children?
		jne	DevUnregister8	; cannot unregister

; ------------- Push registers

		push	eax		; push EAX
		push	ebx		; push EBX
		push	edx		; push EDX

; ------------- Delete entry from driver list

		mov	edx,DevAdrList	; EDX <- device address list
		call	RBTreeDelete	; delete entry from the list

; ------------- Delete entry from hash list

		lea	eax,[ebx+DEV_Hash] ; EAX <- hash entry
		call	HashDel		; delete entry from hash list

; ------------- Delete entry from hierarchy list

		lea	eax,[ebx+DEV_DevList] ; EAX <- tree entry
		mov	ebx,[eax+TREE_Parent] ; EBX <- parent
		call	TreeDel		; delete entry

; ------------- Pop registers

		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX

; ------------- Unlock device list

		DEVLSTUNLOCK		; unlock device list

; ------------- OK

		popf			; enable interrupts
		clc			; clear error flag
		ret

; ------------- Unlock device list

DevUnregister8:	DEVLSTUNLOCK		; unlock device list

; ------------- ERROR

		popf			; enable interrupts
		stc			; set error flag
		ret

; -----------------------------------------------------------------------------
;                      Add new resource to device descriptor
; -----------------------------------------------------------------------------
; INPUT:	AL = type of resource (DEVRES_IRQ, DEVRES_DMA,...)
;		EBX = device descriptor DEV
;		ECX = start of resource
;		EDX = end of resource
;		ESI = pointer to name of resource TEXTDATA
; OUTPUT:	CY = memory error
; NOTES:	It does not check if such resource already exists.
; -----------------------------------------------------------------------------

; ------------- Push registers

DevResAdd:	push	eax		; push EAX
		push	edi		; push EDI

; ------------- Prepare flags and type (-> EDI)

		movzx	edi,al		; EDI <- type of resource

; ------------- Create resource descriptor

		mov	eax,DEVRES_size ; EAX <- size of resource descriptor
		call	SysMemAlloc	; get system memory block
		jc	DevResAdd8	; memory error

; ------------- Initialize resource descriptor

		mov	[eax+DEVRES_Start],ecx ; start of resource
		mov     [eax+DEVRES_End],edx ; end of resource
		mov	[eax+DEVRES_TypeFlagDW],edi ; type and flags

; ------------- Name of resource

		mov	edi,ebx		; EDI <- save device descriptor
		lea	ebx,[eax+DEVRES_Name] ; EBX <- name of resource
		xchg	eax,esi		; EAX <- name, ESI <- resource
		call	TextAttach	; attach text string
		xchg	eax,esi		; EAX <- resource, ESI <- name

; ------------- Add resource descriptor into driver parameter block

		lea	ebx,[edi+DEV_Resource] ; EBX <- resource list
		call	ListLast	; add resource descriptor into list
		sub	ebx,DEV_Resource ; EBX <- device DEV (it clears CF)

; ------------- Pop registers

DevResAdd8:	pop	edi		; pop EDI
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                 Add resource descriptor to driver descriptor
; -----------------------------------------------------------------------------
; INPUT:	EAX = resource descriptor DEVRES
;		EBX = device descriptor DEV
; NOTES:	Use DEVRES_STATIC flag to specify static descriptor.
; -----------------------------------------------------------------------------

DevResDescAdd:	push	ebx		; push EBX
		add	ebx,DEV_Resource ; EBX <- resource list
		call	ListLast	; add resource descriptor into list
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;               Delete one resource from the device descriptor
; -----------------------------------------------------------------------------
; INPUT:	AL = type of resource (DEVRES_IRQ, DEVRES_DMA,...)
;		EBX = device descriptor DEV
;		ECX = start of resource
; OUTPUT:	CY = no such resource found
; NOTES:	Resource flags and resource size are not compared.
;		Resource descriptor is destroyed if it is not static.
; -----------------------------------------------------------------------------

; ------------- Push registers

DevResDel:	push	ebx		; push EBX
		push	esi		; push ESI

; ------------- Get next resource descriptor (-> ESI)

		add	ebx,DEV_Resource ; EBX <- resource list head
		mov	esi,ebx		; ESI <- resource list head
DevResDel2:	mov	esi,[esi+LIST_Next] ; ESI <- next resource descriptor
		cmp	esi,ebx		; any other resource descriptor?
		stc			; set error flag, no such resource
		je	DevResDel8	; no other resource descriptor

; ------------- Check if it is required resource

		cmp	al,[esi+DEVRES_Type] ; check resource type
		jne	DevResDel2	; resource type is not valid
		cmp	ecx,[esi+DEVRES_Start] ; check start address
		jne	DevResDel2	; start address is not valid

; ------------- Delete resource descriptor from the list

		xchg	eax,esi		; EAX <- descriptor, ESI <- type
		call	ListDel		; delete resource descriptor from list

; ------------- Check if resource descriptor may be destroyed

		test	byte [eax+DEVRES_Flags],DEVRES_STATIC ; static?
		jnz	DevResDel8	; cannot delete static descriptor

; ------------- Detach resource name

		lea	ebx,[eax+DEVRES_Name] ; EBX <- name of resource
		call	TextDetach	; detach text of resource name

; ------------- Destroy block of memory

                call	SysMemFree	; destroy resource descriptor
		xchg	eax,esi		; EAX <- type
		clc			; clear error flag

; ------------- Pop registers

DevResDel8:	pop	esi		; pop ESI
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                       Delete all resources of the driver
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; -----------------------------------------------------------------------------

; ------------- Push registers

DevResDelAll:	push	eax		; push EAX
		push	ebx		; push EBX
		push	ecx		; push ECX

; ------------- Get next resource descriptor (-> EAX)

		lea	ecx,[ebx+DEV_Resource] ; ECX <- resource list head
DevResDelAll4:	mov	eax,[ecx+LIST_Next] ; EAX <- next resource descriptor
		cmp	eax,ecx		; any other resource descriptor?
		je	DevResDelAll8	; no other resource descriptor

; ------------- Detach resource descriptor from the list

		call	ListDel		; detach resource descriptor from list

; ------------- Check if resource descriptor may be destroyed

		test	byte [eax+DEVRES_Flags],DEVRES_STATIC ; static?
		jnz	DevResDelAll4	; cannot delete static descriptor

; ------------- Detach resource name

		lea	ebx,[eax+DEVRES_Name] ; EBX <- name of resource
		call	TextDetach	; detach text of resource name

; ------------- Destroy block of memory

                call	SysMemFree	; destroy resource descriptor
		jmp	DevResDelAll4	; next descriptor

; ------------- Pop registers

DevResDelAll8:	pop	ecx		; pop ECX
		pop	ebx		; pop EBX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                  Initialize all not-initialized devices
; -----------------------------------------------------------------------------

; ------------- Push registers

DevInitAll:	push	ebx		; push EBX

; ------------- Disable interrupts

DevInitAll2:	pushf			; push flags
		cli			; disable interrupts

; ------------- Lock device list

		DEVLSTLOCK		; lock device list

; ------------- Get first device in the list (-> EBX)

		push	edx		; push EDX
		mov	edx,DevAdrList	; EDX <- list of all drivers
		call	RBTreeFirst	; get first device in the list
		pop	edx		; pop EDX
		jc	DevInitAll8	; no device found

; ------------- Check if this device is initialized

DevInitAll4:	test	byte [ebx+DEV_Flags],DEV_INIT ; is device initialized?
		jnz	DevInitAll6	; device is already initialized

; ------------- Unlock device list

		DEVLSTUNLOCK		; unlock device list

; ------------- Enable interrupts

		popf			; enable interrupts

; ------------- Initialize device

		call	DevInit		; initialize device
		jmp	short DevInitAll2 ; try next device

; ------------- Get next device

DevInitAll6:	call	RBTreeNext	; get next device
		jnc	DevInitAll4	; check next device

; ------------- Unlock device list

DevInitAll8:	DEVLSTUNLOCK		; unlock device list

; ------------- Enable interrupts

		popf			; enable interrupts

; ------------- Pop registers

		pop	ebx		; pop EBX
		ret


; -----------------------------------------------------------------------------
;                  Initialize basic system devices
; -----------------------------------------------------------------------------

; ------------- Install system kernel device (DRIVERS\SYSTEM\SYS.ASM)

DevBaseInit:	call	SYSInstall	; install SYS driver

; ------------- Install IRQ driver (DRIVERS\SYSTEM\IRQ.ASM)

		call	IRQInstall	; install IRQ driver
		mov	ebx,[IRQDevice]	; EBX <- IRQ device
		call	DevInit		; initialize IRQ device

; ------------- Initialize interrupt service (DRIVERS\INT.ASM)

		call	IntInit		; initialize interrupt service

; ------------- Install DMA driver (DRIVERS\SYSTEM\DMA.ASM)

		call	DMAInstall	; install DMA driver

; ------------- Install CMOS driver (DRIVERS\SYSTEM\CMOS.ASM)

		call	CMOSInstall	; install CMOS driver

; ------------- Install TIMER driver (DRIVERS\SYSTEM\TIMER.ASM)

		call	TIMERInstall	; install TIMER driver
		mov	ebx,[TIMERDevice] ; EBX <- TIMER device
		call	DevInit		; initialize TIMER device

; ------------- Install CLOCK driver (DRIVERS\SYSTEM\CLOCK.ASM)

		call	CLOCKInstall	; install CLOCK driver




; ------------- Initialize system timer (KERNEL\TIME.ASM)

;		call	TimerInit	; initialize system timer

; ------------- Install timer handler (IRQ 0)

;		mov	ebx,TimeIRQHandler ; EBX <- timer handler
;		call	IntInstall	; install timer handler

; ------------- Initialize all not-initialized devices

		call	DevInitAll	; initialize all devices
		ret

; -----------------------------------------------------------------------------
;                       Initialize system devices
; -----------------------------------------------------------------------------

DevStdInit:




;		mov	ebx,20h
;		mov	edx,

;		call	SetIntGate	; set interrupt gate
;		call	IntIntAct	; activate IRQ






; ------------- Install keyboard handler (IRQ 1)

;		mov	ebx,KeybIRQHandler ; EBX <- keyboard handler
;		call	IRQInstall	; install keyboard handler

; ------------- Install floppy disk handler (IRQ 6)

;		mov	ebx,FDIRQHandler ; EBX <- IRQ handler
;		call	IRQInstall	; install IRQ handler

; ------------- Install CMOS timer handler (IRQ 8)

;		mov	ebx,CMOSIRQHandler ; EBX <- CMOS timer handler
;		call	IRQInstall	; install CMOS timer handler

; ------------- Install FPU handler (IRQ 13)

;		mov	ebx,FPUIRQHandler ; EBX <- FPU handler
;		call	IRQInstall	; install FPU handler











;		xor	ecx,ecx


; ------------- Register system timer

;		mov	ebx,TimeDev	; EBX <- driver descriptor
;		call	DevRegister	; insert driver into list





; ------------- Keyboard controller

;		mov	ebx,KeybDev	; EBX <- driver descriptor
;		call	DevRegister	; insert driver into list

; ------------- CMOS real-time clock

;		mov	ebx,CMOSDev	; EBX <- driver descriptor
;		call	DevRegister	; insert driver into list

; ------------- Mathematic coprocessor

;		mov	ebx,FPUDev	; EBX <- driver descriptor
;		call	DevRegister	; insert driver into list

; ------------- Kernel data

;		mov	ebx,DataDev	; EBX <- driver descriptor
;		xor	ecx,ecx		; ECX <- 0, no parent device
;		call	DevRegister	; insert driver into list

; ------------- Kernel code

;		mov	ebx,CodeDev	; EBX <- driver descriptor
;		xor	ecx,ecx		; ECX <- 0, no parent device
;		call	DevRegister	; insert driver into list

; ------------- ROM BIOS

;		mov	ebx,BIOSDev	; EBX <- driver descriptor
;		xor	ecx,ecx		; ECX <- 0, no parent device
;		call	DevRegister	; insert driver into list




; ------------- Initialize all not-initialized devices

		call	DevInitAll	; initialize all devices



		ret

; -----------------------------------------------------------------------------
;                         Display list of devices
; -----------------------------------------------------------------------------

%ifdef	DEBUG_DEVLIST

; ------------- Lock list of drivers

DevLstDisp:	DEVLSTLOCK		; lock list of devices

; ------------- Display number of devices

		mov	esi,DebDevDevNum ; ESI <- text
		call	DebOutTextData	; display text
		DEVNUMBER eax		; EAX <- get number of devices
		call	DebOutNum	; display number of devices
		call	DebNewLine	; display new line

; ------------- Get first device (-> EBX)

		call	DebNewLine	; display new line
		mov	edx,DevAdrList	; EDX <- device address list
		call	RBTreeFirst	; get first device
		jc	DevLstDisp9	; no device

; ------------- Display device long name

DevLstDisp2:	push	ebx		; push EBX
		mov	ebx,[ebx+DEV_Full] ; EBX <- multi-language array
		call	TextLangDef	; find default text string
		mov	esi,[ebx]	; ESI <- text data
		call	DebOutTextData	; display device name
		pop	ebx		; pop EBX

; ------------- Get next resource

DevLstDisp3:   	lea	ecx,[ebx+DEV_Resource] ; ECX <- resource list
		mov	edx,ecx		; EDX <- list head
DevLstDisp4:	mov	edx,[edx+LIST_Next] ; EDX <- next resource
		cmp	edx,ecx		; last resource?
		je	DevLstDisp8	; no other resource

; ------------- Type of resource

		movzx	eax,byte [edx+DEVRES_Type] ; EAX <- type of resource
		mov	esi,[DebDevLstTab+eax*4] ; ESI <- type string
		call	DebOutTextData	; display resource type

; ------------- Memory

		cmp	al,DEVRES_MEM	; memory?
		jne	DevLstDisp5	; not memory
		mov	eax,[edx+DEVRES_Start] ; EAX <- start address
		call	DebOutHexD0	; display start address
		mov	al,"-"		; AL <- separator
		call	DebOutChar	; display separator
		mov	eax,[edx+DEVRES_End] ; EAX <- end address
		call	DebOutHexD0	; display end address
		jmp	DevLstDisp7	; next resource

; ------------- Port

DevLstDisp5:	cmp	al,DEVRES_PORT	; port?
		jne	DevLstDisp6	; not port
		mov	eax,[edx+DEVRES_Start] ; EAX <- start address
		call	DebOutHexW	; display start address
		cmp	eax,[edx+DEVRES_End] ; size = 1 ?
		je	DevLstDisp7	; next resource
		mov	al,"-"		; AL <- separator
		call	DebOutChar	; display separator
		mov	eax,[edx+DEVRES_End] ; EAX <- end address
		call	DebOutHexW	; display end address
		jmp	DevLstDisp7	; next resource

; ------------- IRQ and DMA

DevLstDisp6:	mov	eax,[edx+DEVRES_Start] ; EAX <- start address
		call	DebOutNum	; display start address
		cmp	eax,[edx+DEVRES_End] ; size = 1 ?
		je	DevLstDisp7	; next resource
		mov	al,"-"		; AL <- separator
		call	DebOutChar	; display separator
		mov	eax,[edx+DEVRES_End] ; EAX <- end address
		call	DebOutNum	; display end address

; ------------- Resource name

DevLstDisp7:	call	DebOutSpc	; display space
		mov	esi,[edx+DEVRES_Name] ; ESI <- resource name
		call	DebOutTextData	; display resource name
		jmp	DevLstDisp4	; next resource

; ------------- Display new line

DevLstDisp8:	call	DebNewLine	; display new line

; ------------- Get next driver (-> EBX)

		call	RBTreeNext	; get next device
		jnc	DevLstDisp2	; next driver is OK

; ------------- Unlock list of drivers

DevLstDisp9:	DEVLSTUNLOCK		; unlock list of drivers
		ret

%endif


;TimeDevInit:
;KeybDevInit:
;CMOSDevInit:
;FPUDevInit:
;DataDevInit:
;CodeDevInit:
;BIOSDevInit:
		clc

;TimeDevDeinit:
;KeybDevDeinit:
;CMOSDevDeinit:
;FPUDevDeinit:
;DataDevDeinit:
;CodeDevDeinit:
;BIOSDevDeinit:
		ret

; *****************************************************************************
;
;                         Device global functions
;
; *****************************************************************************

; -----------------------------------------------------------------------------
;                           Initialize device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; OUTPUT:	CY = error or already initialized
; NOTES:	Function does not lock device.
; -----------------------------------------------------------------------------

DevInit:	bts	dword [ebx+DEV_Flags],DEV_INIT_BIT ; initialized?
		jc	DevInit8	; already initialized
		DEVFNC	DEV_Init	; initialize device
DevInit8:	ret

; -----------------------------------------------------------------------------
;                           Deinitialize device
; -----------------------------------------------------------------------------
; INPUT:	EBX = device descriptor DEV
; OUTPUT:	CY = error or not initialized
; NOTES:	Function does not lock device.
; -----------------------------------------------------------------------------

DevDeinit:	btr	dword [ebx+DEV_Flags],DEV_INIT_BIT ; initialized?
		cmc			; invert error flag
		jc	DevDeinit8	; not initialized
		DEVFNC	DEV_Deinit	; deinitialize device
DevDeinit8:	ret

; -----------------------------------------------------------------------------
;                              Constant Data
; -----------------------------------------------------------------------------

		CONST_SECTION

; ------------- Debug display device list

%ifdef	DEBUG_DEVLIST

		align	4, db 0
DebDevDevNum:	CTEXTDATA 'Number of devices: '

		align	4, db 0
DebDevLstIRQ:	CTEXTDATA ',IRQ '
DebDevLstDMA:	CTEXTDATA ',DMA '
DebDevLstPort:	CTEXTDATA ',I/O '
DebDevLstMem:	CTEXTDATA ',MEM '

		align	4, db 0
DebDevLstTab:	dd	DebDevLstIRQ
		dd	DebDevLstDMA
		dd	DebDevLstPort
		dd	DebDevLstMem
%endif

; ------------- System vendor name

		align	4, db 0
DefaultVendor:	CTEXTDATA 'Litos Kernel'

; ------------- System timer 8254

;%define		TimeDevVendor DefaultVendor
;
;TimeDevName:	CTEXTDATA 'time'
;TimeDevShort:	CTEXTDATA 'Timer 8254'
;
;TimeDevFull:	LANGTEXTSTR TimeDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
; 		LANGTEXTSTR TimeDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0
;
;TimeDevFullEN:	CTEXTDATA 'System timer 8254'
;TimeDevFullCZ:	CTEXTDATA 'Syst',0c3h,0a9h,'mov',0c3h,0bdh,' ',0c4h,8dh,\
;			'asova',0c4h,8dh,' 8254'

; ------------- Keyboard controller 8042

;%define		KeybDevVendor DefaultVendor

;KeybDevName:	CTEXTDATA 'keyb'
;KeybDevShort:	CTEXTDATA 'Keyboard 8254'
;KeybDevFull:	LANGTEXTSTR KeybDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
; 		LANGTEXTSTR KeybDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0

;KeybDevFullEN:	CTEXTDATA 'Keyboard controller 8042'
;KeybDevFullCZ:	CTEXTDATA 0c5h,98h,'adi',0c4h,8dh,' kl',0c3h,0a1h,\
;			'vesnice 8042'

; ------------- CMOS real-time clock MC146818

;%define		CMOSDevVendor DefaultVendor
;
;CMOSDevName:	CTEXTDATA 'rtc'
;
;CMOSDevShort:	CTEXTDATA 'CMOS MC146818'
;
;CMOSDevFull:	LANGTEXTSTR CMOSDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
; 		LANGTEXTSTR CMOSDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0
;
;CMOSDevFullEN:	CTEXTDATA 'CMOS and real-time clock MC146818'
;CMOSDevFullCZ:	CTEXTDATA 'Hodiny re',0c3h,0a1h,'ln',0c3h,0a9h,'ho ',\
;			0c4h,8dh,'asu a CMOS MC146818'

; ------------- Mathematic coprocessor 80387

;%define		FPUDevVendor DefaultVendor
;
;FPUDevName:	CTEXTDATA 'fpu'
;
;FPUDevShort:	CTEXTDATA 'FPU 80387'
;
;FPUDevFull:	LANGTEXTSTR FPUDevFullEN,LANG_ENGLISH,SUBLANG_DEFAULT,2
; 		LANGTEXTSTR FPUDevFullCZ,LANG_CZECH,  SUBLANG_DEFAULT,0
;
;FPUDevFullEN:	CTEXTDATA 'Mathematic coprocessor 80387'
;FPUDevFullCZ:	CTEXTDATA 'Matematick',0c3h,0bdh,' koprocesor 80387'

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- List of all installed devices

		align	8, db 0
DevAdrList:	RBTREE			; device address list
DevListLock:	SPINLOCK		; device list lock
DevDevList:	TREEHEAD		; device hierarchy list

; ------------- System timer 8254

		align	8, db 0
;TimeDev:	DEVICE	DEVCLASS_SYS,0,0,DEV_STATIC,1,0,0,TimeDev
;
;TimeDevRes1: DEVRESOURCE TimeDevResN,TimeDevRes0,0,1,DEVRES_IRQ,DEVRES_STATIC
;TimeDevResN: DEVRESOURCE TimeDevRes0,TimeDevRes1,40h,4,\
;				DEVRES_PORT,DEVRES_STATIC
;
;TimeDevInt:	dd	DEV_GEN_ID
;		dd	DEV_NUL_ID

; ------------- Keyboard controller 8042

		align	8, db 0
;KeybDev:	DEVICE	DEVCLASS_SYS,0,0,DEV_STATIC,1,0,0,KeybDev
;
;KeybDevRes1: DEVRESOURCE KeybDevRes2,KeybDevRes0,1,1,DEVRES_IRQ,DEVRES_STATIC
;KeybDevRes2: DEVRESOURCE KeybDevResN,KeybDevRes1,60h,1,\
;			DEVRES_PORT,DEVRES_STATIC
;KeybDevResN: DEVRESOURCE KeybDevRes0,KeybDevRes2,64h,1,\
;			DEVRES_PORT,DEVRES_STATIC
;
;KeybDevInt:	dd	DEV_GEN_ID
;		dd	DEV_NUL_ID

; ------------- CMOS real-time clock MC146818

		align	8, db 0
;CMOSDev:	DEVICE	DEVCLASS_SYS,0,0,DEV_STATIC,1,0,0,CMOSDev
;
;CMOSDevRes1: DEVRESOURCE CMOSDevResN,CMOSDevRes0,8,1,DEVRES_IRQ,DEVRES_STATIC
;CMOSDevResN: DEVRESOURCE CMOSDevRes0,CMOSDevRes1,70h,2,\
;			DEVRES_PORT,DEVRES_STATIC
;
;CMOSDevInt:	dd	DEV_GEN_ID
;		dd	DEV_NUL_ID

; ------------- Mathematic coprocessor 80387

		align	8, db 0
;FPUDev:		DEVICE	DEVCLASS_SYS,0,0,DEV_STATIC,1,0,0,FPUDev
;
;FPUDevRes1: DEVRESOURCE FPUDevResN,FPUDevRes0,13,1,DEVRES_IRQ,DEVRES_STATIC
;FPUDevResN: DEVRESOURCE FPUDevRes0,FPUDevRes1,0f0h,16,DEVRES_PORT,DEVRES_STATIC
;
;FPUDevInt:	dd	DEV_GEN_ID
;		dd	DEV_NUL_ID

; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- hash list of devices (base index = device class) (size 1 KB)

DevHashList:	resd	DEVHASH_SIZE
