; =============================================================================
;
;                             Litos - Disk cache
;
; =============================================================================

		CODE_SECTION

; -----------------------------------------------------------------------------
;                       Lock/unlock list of cache buffers
; -----------------------------------------------------------------------------
; NOTES:	Use macro CACHELOCK to lock, CACHEUNLOCK to unlock.
; -----------------------------------------------------------------------------

; ------------- Macro - lock cache

%macro		CACHELOCK 0
		LOCK_Lock LockCache	; lock cache
%endmacro

; ------------- Macro - unlock cache

%macro		CACHEUNLOCK 0
		LOCK_Unlock LockCache	; unlock cache
%endmacro

; -----------------------------------------------------------------------------
;                             Get hash list head
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
; OUTPUT:	EDI = hash list head
; -----------------------------------------------------------------------------

CacheGetHash:	mov	edi,eax		; EDI <- data offset LOW
		shrd	edi,edx,CACHE_BUFBITS ; EDI <- offset
		add	edi,ebx		; add block device
		and	edi,CACHE_HASHMASK ; EDI <- hash index
		lea	edi,[CacheHash+edi*HASHH_size] ; EDI <- hash address
		ret

; -----------------------------------------------------------------------------
;                             Find cache buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
; OUTPUT:	EDI = cache buffer head (if NC)
;		CY = buffer not found (EDI is not valid)
; -----------------------------------------------------------------------------

; ------------- Get hash list head (-> EDI)

CacheFindBuf:	call	CacheGetHash	; get hash list head

; ------------- Push registers

		push	eax		; push EAX

; ------------- Round data offset to buffer size (-> EDX:EAX)

		and	eax,CACHE_BUFINVMSK ; round data offset to buffer size

; ------------- Get next list entry (-> EDI)

CacheFindBuf2:	mov	edi,[edi+HASHE_Next] ; EDI <- next entry
		cmp	edi,byte 1	; is next entry valid?
		jb	CacheFindBuf8	; buffer not found

; ------------- Compare one buffer

		cmp	eax,[edi+CACHE_Offset] ; compare offset LOW
		jne	CacheFindBuf2	; buffer does not correspond
		cmp	ebx,[edi+CACHE_Driver] ; compare data device
		jne	CacheFindBuf2	; buffer does not correspond
		cmp	edx,[edi+CACHE_Offset+4] ; compare offset HIGH
		jne	CacheFindBuf2	; buffer does not correspond

; ------------- Pop registers (here is CY=error, buffer not found)

CacheFindBuf8:	pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                            Create cache buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
; OUTPUT:	EDI = cache buffer head (if NC)
;		CY = memory error (EDI is not valid)
; NOTES: Cache list must be locked.
; -----------------------------------------------------------------------------

; ------------- Get hash list head (-> EDI)

CacheCreateBuf:	call	CacheGetHash	; get hash list head

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX

; ------------- Round data offset to buffer size (-> EDX:ECX)

		and	eax,CACHE_BUFINVMSK ; round data offset to buffer size
		xchg	eax,ecx		; ECX <- data offset LOW

; ------------- Allocate new cache buffer (-> EAX)

		call	CacheBuffAlloc	; allocate cache disk buffer
		jc	CacheCreateBuf8	; memory error

; ------------- Link cache buffer to cache hash list head

		xchg	ebx,edi		; EBX <- hash list head,EDI <- push EBX
		call	HashAdd		; add cache buffer to cache hash list

; ------------- Link cache buffer to other cache list

		push	eax		; push cache buffer head
		add	eax,byte CACHE_List ; EAX <- list of buffers
		mov	ebx,OtherCacheList ; EBX <- other cache list
		call	ListLast	; add buffer into end of other list
		mov	ebx,edi		; EBX <- pop EBX
		pop	edi		; EDI <- cache buffer head

; ------------- Initialize cache buffer head

		mov	[edi+CACHE_Offset],ecx ; set data offset LOW
		mov	[edi+CACHE_Offset+4],edx ; set data offset HIGH
		mov	[edi+CACHE_Driver],ebx ; set data device

; ------------- Initialize size of data in buffer

		mov	eax,[ebx+DDEV_Size] ; EAX <- media size LOW
		sub	eax,ecx		; EAX <- remaining data LOW
		mov	ecx,[ebx+DDEV_Size+4] ; ECX <- media size HIGH
		sbb	ecx,edx		; ECX <- remaining data HIGH
		mov	ecx,CACHE_BUFSIZE ; ECX <- buffer size
		ja	CacheCreateBuf4	; remaining data is bigger than DWORD
		je	CacheCreateBuf2	; remaining data is DWORD
		xor	ecx,ecx		; ECX <- 0, no remaining data
CacheCreateBuf2:cmp	eax,ecx		; check buffer size
		jae	CacheCreateBuf4	; buffer size is OK
		xchg	eax,ecx		; ECX <- limit buffer size
CacheCreateBuf4:mov	[edi+CACHE_Size],ecx ; set buffer size

; ------------- Pop registers

		clc			; clear error flag
CacheCreateBuf8:pop	ecx		; pop ECX
		pop	eax		; pop EAX
		ret

; -----------------------------------------------------------------------------
;                           Prepare cache operation
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
;		ECX = data length (bytes)
; OUTPUT:	ECX = limited data length (0 if data offset overflow)
; -----------------------------------------------------------------------------

; ------------- Push registers

CacheRWPrep:	push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare media size (-> EDI:ESI)

		mov	esi,[ebx+DDEV_Size] ; ESI <- media size LOW
		mov	edi,[ebx+DDEV_Size+4] ; EDI <- media size HIGH

; ------------- Remaining data (-> EDI:ESI)

		sub	esi,eax		; ESI <- remaining data LOW
		sbb	edi,edx		; EDI <- remaining data HIGH
		ja	CacheRWPrep4	; there is enough data
		je	CacheRWPrep2	; there is less than DWORD
		xor	esi,esi		; ESI <- 0 on data offset overflow

; ------------- Limit data length

CacheRWPrep2:	cmp	esi,ecx		; check remaining data LOW
		ja	CacheRWPrep4	; remaining data is OK
		mov	ecx,esi		; ECX <- limit data length

; ------------- Pop registers

CacheRWPrep4:	pop	edi		; pop EDI
		pop	esi		; pop ESI
		ret

; -----------------------------------------------------------------------------
;                      Load cache buffer if it is not loaded
; -----------------------------------------------------------------------------
; INPUT:	EBX = data device DATADEV
;		EDI = cache buffer head
; OUTPUT:	CY = read error
; NOTES: Cache buffer must be locked (using mutex).
; -----------------------------------------------------------------------------

; ------------- Check (and reset) if buffer should be loaded

CacheReadBuf:	btr	dword [edi+CACHE_Flags],CACHE_READ_BIT ; check flag
		jnc	CacheReadBuf8	; should not be loaded

; ------------- Push registers

		push	eax		; push EAX
		push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI

; ------------- Read buffer from device

		mov	eax,[edi+CACHE_Offset] ; EAX <- data offset LOW
		mov	edx,[edi+CACHE_Offset+4] ; EDX <- data offset HIGH
		mov	ecx,[edi+CACHE_Size] ; ECX <- data size
		mov	esi,[edi+CACHE_Data] ; ESI <- cache data block
		call	DataDevRead	; read data from device

; ------------- Pop registers

		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		pop	eax		; pop EAX
CacheReadBuf8:	ret

; -----------------------------------------------------------------------------
;                            Read data from cache buffer
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
;		ECX = data length (bytes)
;		ESI = data buffer
;		EDI = cache buffer head
;		EBP = bytes OK read
; OUTPUT:	EDX:EAX = new data offset
;		ECX = new data length
;		EBP = new bytes OK read
; -----------------------------------------------------------------------------

; ------------- Push registers

CacheGetByte:	push	ebx		; push EBX
		push	eax		; push EAX
		push	ecx		; push ECX
		push	esi		; push ESI
		push	edi		; push EDI

; ------------- Prepare data size (-> EBX, ECX)

		and	eax,CACHE_BUFMASK ; EAX <- offset in buffer
		mov	ebx,[edi+CACHE_Size] ; EBX <- size of data in buffer
		sub	ebx,eax		; EBX <- remaining data
		jnc	CacheGetByte2	; data is OK
		xor	ebx,ebx		; EBX <- 0, limit data size
CacheGetByte2:	cmp	ebx,ecx		; check remaining data
		jae	CacheGetByte4	; data is OK
		mov	ecx,ebx		; ECX <- limit data size
CacheGetByte4:	mov	ebx,ecx		; EBX <- push data size

; ------------- Prepare pointers

		add	eax,[edi+CACHE_Data] ; ESI <- cache data block
		mov	edi,esi		; EDI <- data buffer
		xchg	eax,esi		; ESI <- source data

; ------------- Transfer data from cache buffer

		shr	ecx,2		; ECX <- data size in DWORD
		rep	movsd		; transfer data in DWORD
		mov	ecx,ebx		; ECX <- data size
		and	ecx,byte 3	; last data in DWORD
		rep	movsb		; transfer rest of data

; ------------- Pop registers

		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	ecx		; pop ECX
		pop	eax		; pop EAX

; ------------- Shift data counter

		sub	ecx,ebx		; decrease data size
		add	ebp,ebx		; increase data counter
		add	eax,ebx		; increase data offset
		adc	edx,byte 0	; overflow
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                            Read cached data from device
; -----------------------------------------------------------------------------
; INPUT:	EDX:EAX = data offset
;		EBX = data device DATADEV
;		ECX = data length (bytes)
;		ESI = data buffer
; OUTPUT:	EAX = bytes OK read
;		CY = error
; -----------------------------------------------------------------------------

; ------------- Push registers

CacheRead:	push	ecx		; push ECX
		push	edx		; push EDX
		push	esi		; push ESI
		push	edi		; push EDI
		push	ebp		; push EBP
		xor	ebp,ebp		; EBP <- 0, data counter
		
; ------------- Prepare cache operation

		call	CacheRWPrep	; prepare cache operation

; ------------- Check remaining data length

CacheRead2:	clc			; clear error flag
		jecxz	CacheRead9	; no data

; ------------- Lock list of cache buffers

CacheRead3:	pushf			; push flags
		cli			; disable interrupts
		CACHELOCK		; lock list of cache buffers

; ------------- Find cache buffer (-> EDI)

		call	CacheFindBuf	; find cache buffer
		jnc	CacheRead4	; buffer found OK

; ------------- Create cache buffer

		call	CacheCreateBuf	; create cache buffer
		jnc	CacheRead4	; buffer created OK

; ------------- Unlock list of cache buffers

		CACHEUNLOCK		; unlock cache
		popf			; pop flags

; ------------- Free some memory

		call	FreeMemWait	; free some memory
		jmp	short CacheRead3 ; next try to get cache buffer

; ------------- Increase used counter

CacheRead4:	inc	dword [edi+CACHE_Used] ; increase used counter

; ------------- Unlock list of cache buffers

		CACHEUNLOCK		; unlock cache
		popf			; pop flags

; ------------- Lock buffer

		push	ebx		; push EBX
		lea	ebx,[edi+CACHE_Lock] ; EBX <- mutex
		call	MutexLock	; lock mutex
		pop	ebx		; pop EBX

; ------------- Load buffer if it is not loaded

		call	CacheReadBuf	; load buffer
		jc	CacheRead8	; read error

; ------------- Read data from buffer

		call	CacheGetByte	; get data from buffer

; ------------- Unlock buffer

	 	push	ebx		; push EBX
		lea	ebx,[edi+CACHE_Lock] ; EBX <- mutex
		call	MutexUnlock	; unlock mutex
		pop	ebx		; pop EBX

; ------------- Decrease used counter

		LOCKSMP			; CPU instruction lock
		dec	dword [edi+CACHE_Used] ; decrease used counter
		jmp	short CacheRead2 ; next buffer










; ------------- Read error

CacheRead8: 	push	ebx		; push EBX
		lea	ebx,[edi+CACHE_Lock] ; EBX <- mutex
		call	MutexUnlock	; unlock mutex
		pop	ebx		; pop EBX






		jmp	short CacheRead3 ; next attempt










; ------------- Unlock list of cache buffers

		CACHEUNLOCK		; unlock cache
		popf			; pop flags







; ------------- Pop registers

CacheRead9:     xchg	eax,ebp		; EAX <- bytes OK
		pop	ebp		; pop EBP	
		pop	edi		; pop EDI
		pop	esi		; pop ESI
		pop	edx		; pop EDX
		pop	ecx		; pop ECX
		ret


CacheWrite:

		ret

CacheFlush:


		ret


;DDEV_WriteCache:resd	1		; 40h: write cached data to device
					;	INPUT:	EDX:EAX=data offset
					;		EBX=data device
					;		ECX=data length
					;		ESI=data buffer
					;	OUTPUT:	EAX=bytes OK written
					;		CY=error

;DDEV_FlushCache:resd	1		; 44h: flush write cache
					;	INPUT:	EDX:EAX=data offset
					;		EBX=data device
					;		ESI:ECX=data length
					;	OUTPUT:	CY=error

; -----------------------------------------------------------------------------
;                        Allocate cache disk buffer
; -----------------------------------------------------------------------------
; OUTPUT:	EAX = new cache disk buffer head with data block (if NC)
;		CY = memory error
; LOCKS:	BUFH_Lock, SysMemLock
; NOTES:	It initializes mutex, flags and used counter.
; -----------------------------------------------------------------------------

; ------------- Push registers

CacheBuffAlloc:	push	ebx		; push EBX
		push	edx		; push EDX

; ------------- Allocate cache disk data block (-> EBX)

		mov	eax,CACHE_BUFSIZE ; EAX <- cache buffer size
		call	SysMemAlloc	; allocate cache disk data block
		jc	CacheBuffAlloc8	; memory error
		xchg	eax,ebx		; EBX <- cache disk data block

; ------------- Allocate cache disk buffer head (-> EAX)

		mov	edx,CacheDiskBuff ; EDX <- buffer of cache buffer heads
		call	BuffAlloc	; allocate cache buffer head
		jc	CacheBuffAlloc6	; memory error

; ------------- Initialize cache disk buffer head

		mov	[eax+CACHE_Data],ebx ; set pointer to cache data block
		mov	byte [eax+CACHE_Flags],CACHE_READ ; set flags
		and	dword [eax+CACHE_Used],byte 0 ; clear used counter

; ------------- Initialize mutex

		lea	ebx,[eax+CACHE_Lock] ; EBX <- mutex
		call	MutexInit	; initialize mutex

; ------------- OK: Pop registers

		clc			; clear error flag
		pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; ------------- Error, free cache disk data block

CacheBuffAlloc6:xchg	eax,ebx		; EAX <- cache disk data block
		call	SysMemFree	; free cache disk data block

; ------------- ERROR: Pop registers

		stc			; set error flag
CacheBuffAlloc8:pop	edx		; pop EDX
		pop	ebx		; pop EBX
		ret

; -----------------------------------------------------------------------------
;                            Free cache disk buffer
; -----------------------------------------------------------------------------
; INPUT:	EAX = cache disk buffer head with data block
; LOCKS:	BUFH_Lock, SysMemLock
; -----------------------------------------------------------------------------

; ------------- Push registers

CacheBuffFree:	push	edx		; push EDX

; ------------- Free cache disk data block

		xchg	eax,edx		; EDX <- cache disk buffer head
		mov	eax,[edx+CACHE_Data] ; EAX <- cache data block
		call	SysMemFree	; free cache disk data block

; ------------- Free cache disk buffer head

		xchg	eax,edx		; EAX <- cache disk buffer head		
		mov	edx,CacheDiskBuff ; EDX <- buffer of cache buffer heads
		call	BuffFree	; free cache disk buffer head

; ------------- Pop registers

		pop	edx		; pop EDX
		ret

; -----------------------------------------------------------------------------
;                                   Data
; -----------------------------------------------------------------------------

		DATA_SECTION

; ------------- Buffer of cache buffer heads (block 2 KB, aprox.23 entries)

		align	8, db 0
CacheDiskBuff:	BUFFER	CACHE_size,800h

; ------------- Lock of list of cache buffers

		align	4, db  0
LockCache:	SPINLOCK		; cache lock

; ------------- Dirty cache list

DirtyCacheList:	LISTHEAD		; dirty cache list

; ------------- Other cache list

OtherCacheList:	LISTHEAD		; other cache list

; -----------------------------------------------------------------------------
;                            Uninitialized data
; -----------------------------------------------------------------------------

		BSS_SECTION

; ------------- Hash list of cache buffers (size 16 KB or 8 KB)

		align	4,resb 1
CacheHash:	resb	HASHH_size*CACHE_HASHSIZE
