
///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                                  Main                                     //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
// (c) 2021 Miroslav Nemecek, Panda38@seznam.cz, panda38.sweb.cz
// This source code may be used freely without restrictions for any purpose,
// including commercial.

#include "include.h"

// ============================================================================
//                              CRC tables
// ============================================================================

// CRC-32 table (1 KB)
const u32 crc32_tab[256] = {
	0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
	0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
	0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
	0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
	0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
	0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
	0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
	0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
	0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
	0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
	0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
	0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
	0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
	0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
	0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
	0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,

	0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
	0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
	0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
	0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
	0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
	0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
	0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
	0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
	0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
	0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
	0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
	0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
	0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
	0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
	0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
	0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d,
};

// CRC-16 table (512 B), used in disk controllers and in Dallas Maxim 1-Wire bus
const u16 crc16_tab[256] = {
	0x0000, 0xc0c1, 0xc181, 0x0140, 0xc301, 0x03c0, 0x0280, 0xc241,
	0xc601, 0x06c0, 0x0780, 0xc741, 0x0500, 0xc5c1, 0xc481, 0x0440,
	0xcc01, 0x0cc0, 0x0d80, 0xcd41, 0x0f00, 0xcfc1, 0xce81, 0x0e40,
	0x0a00, 0xcac1, 0xcb81, 0x0b40, 0xc901, 0x09c0, 0x0880, 0xc841,
	0xd801, 0x18c0, 0x1980, 0xd941, 0x1b00, 0xdbc1, 0xda81, 0x1a40,
	0x1e00, 0xdec1, 0xdf81, 0x1f40, 0xdd01, 0x1dc0, 0x1c80, 0xdc41,
	0x1400, 0xd4c1, 0xd581, 0x1540, 0xd701, 0x17c0, 0x1680, 0xd641,
	0xd201, 0x12c0, 0x1380, 0xd341, 0x1100, 0xd1c1, 0xd081, 0x1040,
	0xf001, 0x30c0, 0x3180, 0xf141, 0x3300, 0xf3c1, 0xf281, 0x3240,
	0x3600, 0xf6c1, 0xf781, 0x3740, 0xf501, 0x35c0, 0x3480, 0xf441,
	0x3c00, 0xfcc1, 0xfd81, 0x3d40, 0xff01, 0x3fc0, 0x3e80, 0xfe41,
	0xfa01, 0x3ac0, 0x3b80, 0xfb41, 0x3900, 0xf9c1, 0xf881, 0x3840,
	0x2800, 0xe8c1, 0xe981, 0x2940, 0xeb01, 0x2bc0, 0x2a80, 0xea41,
	0xee01, 0x2ec0, 0x2f80, 0xef41, 0x2d00, 0xedc1, 0xec81, 0x2c40,
	0xe401, 0x24c0, 0x2580, 0xe541, 0x2700, 0xe7c1, 0xe681, 0x2640,
	0x2200, 0xe2c1, 0xe381, 0x2340, 0xe101, 0x21c0, 0x2080, 0xe041,

	0xa001, 0x60c0, 0x6180, 0xa141, 0x6300, 0xa3c1, 0xa281, 0x6240,
	0x6600, 0xa6c1, 0xa781, 0x6740, 0xa501, 0x65c0, 0x6480, 0xa441,
	0x6c00, 0xacc1, 0xad81, 0x6d40, 0xaf01, 0x6fc0, 0x6e80, 0xae41,
	0xaa01, 0x6ac0, 0x6b80, 0xab41, 0x6900, 0xa9c1, 0xa881, 0x6840,
	0x7800, 0xb8c1, 0xb981, 0x7940, 0xbb01, 0x7bc0, 0x7a80, 0xba41,
	0xbe01, 0x7ec0, 0x7f80, 0xbf41, 0x7d00, 0xbdc1, 0xbc81, 0x7c40,
	0xb401, 0x74c0, 0x7580, 0xb541, 0x7700, 0xb7c1, 0xb681, 0x7640,
	0x7200, 0xb2c1, 0xb381, 0x7340, 0xb101, 0x71c0, 0x7080, 0xb041,
	0x5000, 0x90c1, 0x9181, 0x5140, 0x9301, 0x53c0, 0x5280, 0x9241,
	0x9601, 0x56c0, 0x5780, 0x9741, 0x5500, 0x95c1, 0x9481, 0x5440,
	0x9c01, 0x5cc0, 0x5d80, 0x9d41, 0x5f00, 0x9fc1, 0x9e81, 0x5e40,
	0x5a00, 0x9ac1, 0x9b81, 0x5b40, 0x9901, 0x59c0, 0x5880, 0x9841,
	0x8801, 0x48c0, 0x4980, 0x8941, 0x4b00, 0x8bc1, 0x8a81, 0x4a40,
	0x4e00, 0x8ec1, 0x8f81, 0x4f40, 0x8d01, 0x4dc0, 0x4c80, 0x8c41,
	0x4400, 0x84c1, 0x8581, 0x4540, 0x8701, 0x47c0, 0x4680, 0x8641,
	0x8201, 0x42c0, 0x4380, 0x8341, 0x4100, 0x81c1, 0x8081, 0x4040,
};

// CRC-Kermit table (512 B)
const u16 crc_kermit_tab[256] = {
	0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
	0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
	0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
	0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
	0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
	0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
	0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
	0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
	0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
	0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
	0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
	0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
	0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
	0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
	0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
	0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,

	0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
	0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
	0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
	0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
	0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
	0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
	0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
	0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
	0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
	0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
	0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
	0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
	0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
	0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
	0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
	0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78,
};

// CRC-DNP table (512 B)
const u16 crc_dnp_tab[256] = {
	0x0000, 0x365e, 0x6cbc, 0x5ae2, 0xd978, 0xef26, 0xb5c4, 0x839a,
	0xff89, 0xc9d7, 0x9335, 0xa56b, 0x26f1, 0x10af, 0x4a4d, 0x7c13,
	0xb26b, 0x8435, 0xded7, 0xe889, 0x6b13, 0x5d4d, 0x07af, 0x31f1,
	0x4de2, 0x7bbc, 0x215e, 0x1700, 0x949a, 0xa2c4, 0xf826, 0xce78,
	0x29af, 0x1ff1, 0x4513, 0x734d, 0xf0d7, 0xc689, 0x9c6b, 0xaa35,
	0xd626, 0xe078, 0xba9a, 0x8cc4, 0x0f5e, 0x3900, 0x63e2, 0x55bc,
	0x9bc4, 0xad9a, 0xf778, 0xc126, 0x42bc, 0x74e2, 0x2e00, 0x185e,
	0x644d, 0x5213, 0x08f1, 0x3eaf, 0xbd35, 0x8b6b, 0xd189, 0xe7d7,
	0x535e, 0x6500, 0x3fe2, 0x09bc, 0x8a26, 0xbc78, 0xe69a, 0xd0c4,
	0xacd7, 0x9a89, 0xc06b, 0xf635, 0x75af, 0x43f1, 0x1913, 0x2f4d,
	0xe135, 0xd76b, 0x8d89, 0xbbd7, 0x384d, 0x0e13, 0x54f1, 0x62af,
	0x1ebc, 0x28e2, 0x7200, 0x445e, 0xc7c4, 0xf19a, 0xab78, 0x9d26,
	0x7af1, 0x4caf, 0x164d, 0x2013, 0xa389, 0x95d7, 0xcf35, 0xf96b,
	0x8578, 0xb326, 0xe9c4, 0xdf9a, 0x5c00, 0x6a5e, 0x30bc, 0x06e2,
	0xc89a, 0xfec4, 0xa426, 0x9278, 0x11e2, 0x27bc, 0x7d5e, 0x4b00,
	0x3713, 0x014d, 0x5baf, 0x6df1, 0xee6b, 0xd835, 0x82d7, 0xb489,

	0xa6bc, 0x90e2, 0xca00, 0xfc5e, 0x7fc4, 0x499a, 0x1378, 0x2526,
	0x5935, 0x6f6b, 0x3589, 0x03d7, 0x804d, 0xb613, 0xecf1, 0xdaaf,
	0x14d7, 0x2289, 0x786b, 0x4e35, 0xcdaf, 0xfbf1, 0xa113, 0x974d,
	0xeb5e, 0xdd00, 0x87e2, 0xb1bc, 0x3226, 0x0478, 0x5e9a, 0x68c4,
	0x8f13, 0xb94d, 0xe3af, 0xd5f1, 0x566b, 0x6035, 0x3ad7, 0x0c89,
	0x709a, 0x46c4, 0x1c26, 0x2a78, 0xa9e2, 0x9fbc, 0xc55e, 0xf300,
	0x3d78, 0x0b26, 0x51c4, 0x679a, 0xe400, 0xd25e, 0x88bc, 0xbee2,
	0xc2f1, 0xf4af, 0xae4d, 0x9813, 0x1b89, 0x2dd7, 0x7735, 0x416b,
	0xf5e2, 0xc3bc, 0x995e, 0xaf00, 0x2c9a, 0x1ac4, 0x4026, 0x7678,
	0x0a6b, 0x3c35, 0x66d7, 0x5089, 0xd313, 0xe54d, 0xbfaf, 0x89f1,
	0x4789, 0x71d7, 0x2b35, 0x1d6b, 0x9ef1, 0xa8af, 0xf24d, 0xc413,
	0xb800, 0x8e5e, 0xd4bc, 0xe2e2, 0x6178, 0x5726, 0x0dc4, 0x3b9a,
	0xdc4d, 0xea13, 0xb0f1, 0x86af, 0x0535, 0x336b, 0x6989, 0x5fd7,
	0x23c4, 0x159a, 0x4f78, 0x7926, 0xfabc, 0xcce2, 0x9600, 0xa05e,
	0x6e26, 0x5878, 0x029a, 0x34c4, 0xb75e, 0x8100, 0xdbe2, 0xedbc,
	0x91af, 0xa7f1, 0xfd13, 0xcb4d, 0x48d7, 0x7e89, 0x246b, 0x1235,
};

// CRC-CCITT table (512 B)
const u16 crc_ccitt_tab[256] = {
	0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
	0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
	0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
	0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
	0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
	0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
	0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
	0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
	0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
	0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
	0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
	0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
	0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
	0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
	0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
	0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,

	0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
	0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
	0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
	0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
	0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
	0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
	0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
	0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
	0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
	0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
	0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
	0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
	0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
	0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
	0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
	0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0,
};

// CRC-SHT table (256 B), SHT1x and SHT7x series of temperature and humidity sensors
const u8 crc_sht_tab[256] = {
	0,   49,  98,  83,  196, 245, 166, 151, 185, 136, 219, 234, 125, 76,  31,  46,
	67,  114, 33,  16,  135, 182, 229, 212, 250, 203, 152, 169, 62,  15,  92,  109,
	134, 183, 228, 213, 66,  115, 32,  17,  63,  14,  93,  108, 251, 202, 153, 168,
	197, 244, 167, 150, 1,   48,  99,  82,  124, 77,  30,  47,  184, 137, 218, 235,
	61,  12,  95,  110, 249, 200, 155, 170, 132, 181, 230, 215, 64,  113, 34,  19,
	126, 79,  28,  45,  186, 139, 216, 233, 199, 246, 165, 148, 3,   50,  97,  80,
	187, 138, 217, 232, 127, 78,  29,  44,  2,   51,  96,  81,  198, 247, 164, 149,
	248, 201, 154, 171, 60,  13,  94,  111, 65,  112, 35,  18,  133, 180, 231, 214,
	122, 75,  24,  41,  190, 143, 220, 237, 195, 242, 161, 144, 7,   54,  101, 84,
	57,  8,   91,  106, 253, 204, 159, 174, 128, 177, 226, 211, 68,  117, 38,  23,
	252, 205, 158, 175, 56,  9,   90,  107, 69,  116, 39,  22,  129, 176, 227, 210,
	191, 142, 221, 236, 123, 74,  25,  40,  6,   55,  100, 85,  194, 243, 160, 145,
	71,  118, 37,  20,  131, 178, 225, 208, 254, 207, 156, 173, 58,  11,  88,  105,
	4,   53,  102, 87,  192, 241, 162, 147, 189, 140, 223, 238, 121, 72,  27,  42,
	193, 240, 163, 146, 5,   52,  103, 86,  120, 73,  26,  43,  188, 141, 222, 239,
	130, 179, 224, 209, 70,  119, 36,  21,  59,  10,  89,  104, 255, 206, 157, 172,
};

// CRC8-Dallas table (256 B), Dallas Maxim 1-Wire bus (used for CRC8-RDALLAS, too)
const u8 crc8_dallas_tab[256] = {
	0x00, 0x5e, 0xbc, 0xe2, 0x61, 0x3f, 0xdd, 0x83,
	0xc2, 0x9c, 0x7e, 0x20, 0xa3, 0xfd, 0x1f, 0x41,
	0x9d, 0xc3, 0x21, 0x7f, 0xfc, 0xa2, 0x40, 0x1e,
	0x5f, 0x01, 0xe3, 0xbd, 0x3e, 0x60, 0x82, 0xdc,
	0x23, 0x7d, 0x9f, 0xc1, 0x42, 0x1c, 0xfe, 0xa0,
	0xe1, 0xbf, 0x5d, 0x03, 0x80, 0xde, 0x3c, 0x62,
	0xbe, 0xe0, 0x02, 0x5c, 0xdf, 0x81, 0x63, 0x3d,
	0x7c, 0x22, 0xc0, 0x9e, 0x1d, 0x43, 0xa1, 0xff,
	0x46, 0x18, 0xfa, 0xa4, 0x27, 0x79, 0x9b, 0xc5,
	0x84, 0xda, 0x38, 0x66, 0xe5, 0xbb, 0x59, 0x07,
	0xdb, 0x85, 0x67, 0x39, 0xba, 0xe4, 0x06, 0x58,
	0x19, 0x47, 0xa5, 0xfb, 0x78, 0x26, 0xc4, 0x9a,
	0x65, 0x3b, 0xd9, 0x87, 0x04, 0x5a, 0xb8, 0xe6,
	0xa7, 0xf9, 0x1b, 0x45, 0xc6, 0x98, 0x7a, 0x24,
	0xf8, 0xa6, 0x44, 0x1a, 0x99, 0xc7, 0x25, 0x7b,
	0x3a, 0x64, 0x86, 0xd8, 0x5b, 0x05, 0xe7, 0xb9,
	0x8c, 0xd2, 0x30, 0x6e, 0xed, 0xb3, 0x51, 0x0f,
	0x4e, 0x10, 0xf2, 0xac, 0x2f, 0x71, 0x93, 0xcd,
	0x11, 0x4f, 0xad, 0xf3, 0x70, 0x2e, 0xcc, 0x92,
	0xd3, 0x8d, 0x6f, 0x31, 0xb2, 0xec, 0x0e, 0x50,
	0xaf, 0xf1, 0x13, 0x4d, 0xce, 0x90, 0x72, 0x2c,
	0x6d, 0x33, 0xd1, 0x8f, 0x0c, 0x52, 0xb0, 0xee,
	0x32, 0x6c, 0x8e, 0xd0, 0x53, 0x0d, 0xef, 0xb1,
	0xf0, 0xae, 0x4c, 0x12, 0x91, 0xcf, 0x2d, 0x73,
	0xca, 0x94, 0x76, 0x28, 0xab, 0xf5, 0x17, 0x49,
	0x08, 0x56, 0xb4, 0xea, 0x69, 0x37, 0xd5, 0x8b,
	0x57, 0x09, 0xeb, 0xb5, 0x36, 0x68, 0x8a, 0xd4,
	0x95, 0xcb, 0x29, 0x77, 0xf4, 0xaa, 0x48, 0x16,
	0xe9, 0xb7, 0x55, 0x0b, 0x88, 0xd6, 0x34, 0x6a,
	0x2b, 0x75, 0x97, 0xc9, 0x4a, 0x14, 0xf6, 0xa8,
	0x74, 0x2a, 0xc8, 0x96, 0x15, 0x4b, 0xa9, 0xf7,
	0xb6, 0xe8, 0x0a, 0x54, 0xd7, 0x89, 0x6b, 0x35,
};

// ===== STM32 CRC32 unit - uses HAL interface
/*
// test pattern (bytes HEX: 01 23 45 67 89 AB CD EF)
const u32 crc32_pattern[2] = { 0x67452301, 0xEFCDAB89 };

// ============================================================================
//                      Start new polynom
// ============================================================================

void crc32_start()
{
	CRC->CR |= CRC_CR_RESET;
}

// ============================================================================
//              Add one DWORD to CRC32 accumulator
// ============================================================================

void crc32_add(u32 d)
{
	CRC->DR = reverse(d);
}

// ============================================================================
//                        Get CRC result
// ============================================================================

u32 crc32_get()
{
	return ~reverse(CRC->DR);
}

// ============================================================================
//  Calculate CRC32 (address must be aligned to DWORD, num is number of DWORDs)
// ============================================================================

u32 crc32_calc(const u32* addr, int num)
{
	crc32_start();
	for (; num > 0; num--) crc32_add(*addr++);
	return crc32_get();
}

// ============================================================================
//             Initialize CRC service (returns False on check error)
// ============================================================================

Bool crc32_init()
{
	// enable clock
	__HAL_RCC_CRC_CLK_ENABLE();

	// setup crc unit
	CRC_HandleTypeDef c;
	c.Instance = CRC;
	c.Init.DefaultPolynomialUse = DEFAULT_POLYNOMIAL_ENABLE;
	c.Init.DefaultInitValueUse = DEFAULT_INIT_VALUE_ENABLE;
	c.Init.CRCLength = CRC_POLYLENGTH_32B;
	c.Init.InputDataInversionMode = CRC_INPUTDATA_INVERSION_NONE;
	c.Init.OutputDataInversionMode = CRC_OUTPUTDATA_INVERSION_DISABLE;
	HAL_CRC_Init(&c);

	// check pattern
	return crc32_calc(crc32_pattern, 2) == CRC32CHECK;
}
*/

// ============================================================================
//                Calculate CRC-32, 1 byte, inverted result
// ============================================================================

#define CRC32_POLY 0xEDB88320 // reversed 0x04C11DB7
u32 crc32_1(u32 crc, u8 data)
{
	return crc32_tab[(crc ^ data) & 0xff] ^ (crc >> 8);
}

u32 crc32_slow_1(u32 crc, u8 data)
{
	int i;
	crc ^= data;
	for (i = 8; i > 0; i--)
	{
		if ((crc & 1) == 0)
			crc >>= 1;
		else
			crc = (crc >> 1) ^ CRC32_POLY;
	}
	return crc;
}

// ============================================================================
//       Calculate CRC-32 (set flag USEHWCRC32 to use hardware CRC32)
// ============================================================================
// Sample "123456789" -> 0xCBF43926
// Sample 0xFC 0x05 0x4A -> 0xA8E10F6D
// Sample CRC32 1KB -> 0x6FCF9E13

u32 crc32(const void* buf, int len)
{
	u32 crc = CRC32_INIT;
	const u8* s = (const u8*)buf;

	// hardware CRC
#ifdef USEHWCRC32	// use hardware CRC32 unit
	if (len >= 12)
	{
		crc32_start();
		if (((u32)s & 3) == 0)
		{
			for (; len > 3; len -= 4)
			{
				crc32_add(*(u32*)s);
				s += 4;
			}
		}
		else
		{
			for (; len > 3; len -= 4)
			{
				crc32_add(s[0] | (s[1] << 8) | (s[2] << 16) | (s[3] << 24));
				s += 4;
			}
		}
		crc = ~crc32_get();
	}
#endif // USEHWCRC32

	// software CRC
	for (; len > 0; len--)
	{
		crc = crc32_tab[(crc ^ *s++) & 0xff] ^ (crc >> 8);
	}

	return ~crc;
}

u32 crc32_slow(const void* buf, int len)
{
	u32 crc = CRC32_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc32_slow_1(crc, *s++);
	}

	return ~crc;
}

// ============================================================================
//       Calculate CRC-32 - Raspberry Pico alternative
// ============================================================================
// Sample "123456789" -> 0x0376E6E7
// Sample 0xFC 0x05 0x4A -> 0x8E10D720
// Sample CRC32 1KB -> 0x5796333D

/*
// Each byte digested MSB-first
// r0: start pointer
// r1: length in bytes
// r2: checksum seed value

.global crc32_small
.type crc32_small,%function
.thumb_func
crc32_small:
    push {r4, r5, lr}
    // r1 now end
    add r1, r0
    ldr  r5, =#0x4c11db7

    b byte_loop_test
byte_loop:
    ldrb r4, [r0]

    rev  r3, r2
    eor  r4, r3
    lsl  r4, #24

    mov r3, #8
bit_loop:
    lsl r4, #1
    bcc no_bit
    eor r4, r5
no_bit:
    sub r3, #1
    bne bit_loop

    lsl r2, #8
    eor r2, r4

    add r0, #1
byte_loop_test:
    cmp r0, r1
    blt byte_loop

    mov r0, r2
    pop {r4, r5, pc}
*/

u32 rev(u32 n)
{
	return ((n & 0xff) << 24) | ((n & 0xff00) << 8) | ((n & 0xff0000) >> 8) | ((n & 0xff000000) >> 24);
}

u32 crc32_Pico(const u8* r0, u32 r1)
{
	u32 r2 = 0xffffffff;
	u32 r3;
	u32 r4;
	u32 r5 = 0x04c11db7;
	for (; r1 > 0; r1--)
	{
		r4 = *r0;
		r3 = rev(r2);
		r4 ^= r3;
		r4 <<= 24;

		for (r3 = 8; r3 > 0; r3--)
		{
			if ((r4 & 0x80000000) != 0)
			{
				r4 <<= 1;
				r4 ^= r5;
			}
			else
				r4 <<= 1;
		}

		r2 <<= 8;
		r2 ^= r4;
		r0++;
	}

	return r2;
}

// ============================================================================
//                  Calculate CRC-16, 1 byte (IBM, Dallas Maxim)
// ============================================================================

#define CRC16_POLY 0xA001 // reversed 0x8005
u16 crc16_1(u16 crc, u8 data)
{
	return crc16_tab[(crc ^ data) & 0xff] ^ (crc >> 8);
}

u16 crc16_slow_1(u16 crc, u8 data)
{
	int i;
	crc ^= data;
	for (i = 8; i > 0; i--)
	{
		if ((crc & 1) == 0)
			crc >>= 1;
		else
			crc = (crc >> 1) ^ CRC16_POLY;
	}
	return crc;
}

// ============================================================================
//                       Calculate CRC-IBM and Dallas
//        used in disk controllers and in Dallas Maxim 1-Wire bus
// ============================================================================
// Sample "123456789" -> 0xBB3D
// Sample: 0xFC 0x05 0x4A -> 0x9742
// Sample CRC32 1KB -> 0x0407

u16 crc_ibm(const void* buf, int len)
{
	u16 crc = CRCIBM_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc16_tab[(crc ^ *s++) & 0xff] ^ (crc >> 8);
	}
	return crc;
}

u16 crc_ibm_slow(const void* buf, int len)
{
	u16 crc = CRCIBM_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc16_slow_1(crc, *s++);
	}
	return crc;
}

// ============================================================================
//                       Calculate CRC-Modbus
// ============================================================================
// Sample "123456789" -> 0x4B37
// Sample: 0xFC 0x05 0x4A -> 0x5733
// Sample CRC32 1KB -> 0xD0B9

u16 crc_modbus(const void* buf, int len)
{
	u16 crc = CRCMODBUS_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc16_tab[(crc ^ *s++) & 0xff] ^ (crc >> 8);
	}
	return crc;
}

u16 crc_modbus_slow(const void* buf, int len)
{
	u16 crc = CRCMODBUS_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc16_slow_1(crc, *s++);
	}
	return crc;
}

// ============================================================================
//             Calculate CRC-Kermit, 1 byte, swap bytes of result
// ============================================================================

#define CRCKERMIT_POLY 0x8408 // reversed 0x1021
u16 crc_kermit_1(u16 crc, u8 data)
{
	return crc_kermit_tab[(crc ^ data) & 0xff] ^ (crc >> 8);
}

u16 crc_kermit_slow_1(u16 crc, u8 data)
{
	int i;
	crc ^= data;
	for (i = 8; i > 0; i--)
	{
		if ((crc & 1) == 0)
			crc >>= 1;
		else
			crc = (crc >> 1) ^ CRCKERMIT_POLY;
	}
	return crc;
}

// ============================================================================
//                  Calculate CRC-Kermit (=original CRC-CCITT)
// ============================================================================
// Sample "123456789" -> 0x8921
// Sample: 0xFC 0x05 0x4A -> 0x71BA
// Sample CRC32 1KB -> 0x5005

u16 crc_kermit(const void* buf, int len)
{
	u16 crc = CRCKERMIT_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_kermit_tab[(crc ^ *s++) & 0xff] ^ (crc >> 8);
	}
	return (u16)((crc >> 8) | (crc << 8));
}

u16 crc_kermit_slow(const void* buf, int len)
{
	u16 crc = CRCKERMIT_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_kermit_slow_1(crc, *s++);
	}
	return (u16)((crc >> 8) | (crc << 8));
}

// ============================================================================
//        Calculate CRC-DNP, 1 byte, invert and swap bytes of result
// ============================================================================
#define CRCDNP_POLY 0xA6BC // reversed 0x3D65

u16 crc_dnp_1(u16 crc, u8 data)
{
	return crc_dnp_tab[(crc ^ data) & 0xff] ^ (crc >> 8);
}

u16 crc_dnp_slow_1(u16 crc, u8 data)
{
	int i;
	crc ^= data;
	for (i = 8; i > 0; i--)
	{
		if ((crc & 1) == 0)
			crc >>= 1;
		else
			crc = (crc >> 1) ^ CRCDNP_POLY;
	}
	return crc;
}

// ============================================================================
//                          Calculate CRC-DNP
// ============================================================================
// Sample "123456789" -> 0x82EA
// Sample: 0xFC 0x05 0x4A -> 0x8AE7
// Sample CRC32 1KB -> 0x69EE

u16 crc_dnp(const void* buf, int len)
{
	u16 crc = CRCDNP_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_dnp_tab[(crc ^ *s++) & 0xff] ^ (crc >> 8);
	}
	crc = ~crc;
	return (u16)((crc >> 8) | (crc << 8));
}

u16 crc_dnp_slow(const void* buf, int len)
{
	u16 crc = CRCDNP_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_dnp_slow_1(crc, *s++);
	}
	crc = ~crc;
	return (u16)((crc >> 8) | (crc << 8));
}

// ============================================================================
//                 Calculate CRC-CCITT, 1 byte (XModem, CCITT)
// ============================================================================
// http://www.embeddedrelated.com/groups/msp430/show/29689.php
// http://www.lammertbies.nl/comm/info/crc-calculation.html

#define CRCCCITT_POLY 0x1021
u16 crc_ccitt_1(u16 crc, u8 data)
{
	return crc_ccitt_tab[(crc >> 8) ^ data] ^ (crc << 8);
}

u16 crc_ccitt_fast_1(u16 crc, u8 data) // without table
{
	crc = (crc >> 8) | (crc << 8);
	crc ^= data;
	crc ^= (crc & 0xff) >> 4;
	crc ^= crc << 12;
	crc ^= (crc & 0xff) << 5;
	return crc;
}

u16 crc_ccitt_slow_1(u16 crc, u8 data)
{
	int i;
	crc ^= ((u16)data << 8);
	for (i = 8; i > 0; i--)
	{
		if ((crc & 0x8000) == 0)
			crc <<= 1;
		else
			crc = (u16)((crc << 1) ^ CRCCCITT_POLY);
	}
	return crc;
}

/*
CRC XModem at ATmega AVR assembler
; ----------------------------------------------------------------------------
;                   Calculate CRC of 1 byte (CCITT or XModem)
; ----------------------------------------------------------------------------
; INPUT: R25:R24 = CRC accumulator
;	 R22 = input byte
; OUTPUT: R25:R24 = result CRC
; DESTROYS: R23
; ----------------------------------------------------------------------------

.global Crc1
Crc1:
; u16 Crc_CCITT_1(u16 crc, u8 data)
;{
;	crc = (crc >> 8) | (crc << 8);
	mov	r23,r25		; swap registers
	mov	r25,r24
	mov	r24,r23

;	crc ^= data;
	eor	r24,r22		; XOR with data byte

;	crc ^= (crc & 0xf0) >> 4;
	mov	r23,r24
	swap	r23
	andi	r23,0x0f
	eor	r24,r23		; XOR with bits 4..7

;	crc ^= crc << 12;
	mov	r23,r24
	swap	r23
	andi	r23,0xf0
	eor	r25,r23		; XOR with bits 0..3

;	crc ^= (crc & 0xff) << 5;
	mov	r23,r24
	lsr	r23
	lsr	r23
	lsr	r23
	eor	r25,r23		; XOR with bits 0..4

	mov	r23,r24
	swap	r23
	lsl	r23
	andi	r23,0xe0
	eor	r24,r23		; XOR with bits 5..7
	ret

; ----------------------------------------------------------------------------
;                   Calculate CRC of the ROM (XModem)
; ----------------------------------------------------------------------------
; OUTPUT: R25:R24 = result CRC
; DESTROYS: R31, R30, R23, R22
; ----------------------------------------------------------------------------
;#define CRCXMODEM_INIT 0
;u16 Crc_XModem(const u8* buf, int len)
;{
;	u16 crc = CRCXMODEM_INIT;
;	for (; len > 0; len--)
;	{
;		crc = Crc1(crc, *buf++);
;	}
;	return crc;
;}

.global GetCrc
GetCrc:
	clr	r31		; start address 0
	clr	r30
	clr	r25		; initialize CRC accumulator to 0
	clr	r24

2:	lpm	r22,Z+		; load one data byte
	rcall	Crc1		; add byte to CRC accumulator
	cpi	r31,hi8(Crc)	; check end address
	brne	2b
	cpi	r30,lo8(Crc)
	brne	2b
	ret
*/

// ============================================================================
//                        Calculate CRC-XModem
// ============================================================================
// Sample "123456789" -> 0x31C3
// Sample: 0xFC 0x05 0x4A -> 0x8048
// Sample CRC32 1KB -> 0xF982

u16 crc_xmodem(const void* buf, int len)
{
	u16 crc = CRCXMODEM_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_tab[(crc >> 8) ^ *s++] ^ (crc << 8);
	}
	return crc;
}

u16 crc_xmodem_fast(const void* buf, int len) // without table
{
	u16 crc = CRCXMODEM_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_fast_1(crc, *s++);
	}
	return crc;
}

u16 crc_xmodem_slow(const void* buf, int len)
{
	u16 crc = CRCXMODEM_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_slow_1(crc, *s++);
	}
	return crc;
}

// ============================================================================
//                       Calculate CRC-CCITT (0xFFFF)
// ============================================================================
// Sample "123456789" -> 0x29B1
// Sample: 0xFC 0x05 0x4A -> 0x4CD4
// Sample CRC32 1KB -> 0x4EED

u16 crc_ccitt(const void* buf, int len)
{
	u16 crc = CRCCCITT_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_tab[(crc >> 8) ^ *s++] ^ (crc << 8);
	}
	return crc;
}

u16 crc_ccitt_fast(const void* buf, int len) // without table
{
	u16 crc = CRCCCITT_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_fast_1(crc, *s++);
	}
	return crc;
}

u16 crc_ccitt_slow(const void* buf, int len)
{
	u16 crc = CRCCCITT_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_slow_1(crc, *s++);
	}
	return crc;
}

// ============================================================================
//                   Calculate CRC-CCITT (0x1D0F)
// ============================================================================
// Sample "123456789" -> 0xE5CC
// Sample: 0xFC 0x05 0x4A -> 0x9144
// Sample CRC32 1KB -> 0xE351

u16 crc_ccitt_b(const void* buf, int len)
{
	u16 crc = CRCCCITTB_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_tab[(crc >> 8) ^ *s++] ^ (crc << 8);
	}
	return crc;
}

u16 crc_ccitt_b_fast(const void* buf, int len) // without table
{
	u16 crc = CRCCCITTB_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_fast_1(crc, *s++);
	}
	return crc;
}

u16 crc_ccitt_b_slow(const void* buf, int len)
{
	u16 crc = CRCCCITTB_INIT;
	const u8* s = (const u8*)buf;

	for (; len > 0; len--)
	{
		crc = crc_ccitt_slow_1(crc, *s++);
	}
	return crc;
}

// ============================================================================
//                      Calculate CRC-Sick, 1 byte
// ============================================================================
// used for Sick devices, electronic devices used for measurement
// and detection in industrial situations.

#define CRCSICKPOLY 0x8005
u16 crc_sick_1(u16 crc, u8 data, u8 prev)
{
	if (crc & 0x8000)
		crc = (crc << 1) ^ CRCSICKPOLY;
	else
		crc <<= 1;
	crc ^= (data | (prev << 8));
	return crc;
}

// ============================================================================
//                         Calculate CRC-Sick
// ============================================================================
// Sample "123456789" -> 0x56A6
// Sample: 0xFC 0x05 0x4A -> 0xB57E
// Sample CRC32 1KB -> 0xC7FE

u16 crc_sick(const void* buf, int len)
{
	u16 crc = CRCSICK_INIT;
	u8 data;
	u8 prev = 0;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--)
	{
		data = *s++;
		crc = crc_sick_1(crc, data, prev);
		prev = data;
	}
	return (u16)((crc >> 8) | (crc << 8));
}

// ============================================================================
//                        Calculate CRC8-RDallas, 1 byte
// ============================================================================

#define CRC8RDALLAS_POLY 0x31 // reversed 0x8c
u8 crc8_rdallas_1(u8 crc, u8 data) // CRC is reversed
{
	return crc8_dallas_tab[crc] ^ data;
}	

u8 crc8_rdallas_slow_1(u8 crc, u8 data)
{
	int i;
	u8 carry;

	for (i = 8; i > 0; i--)
	{
		carry = (crc & 0x80);
		crc <<= 1;
		if ((data & 1) != 0) crc |= 1;
		data >>= 1;
		if (carry != 0) crc ^= CRC8RDALLAS_POLY;
	}

	return crc;
}

// ============================================================================
//                         Calculate CRC8-RDallas
// ============================================================================
// Sample "123456789" -> 0x7C
// Sample: 0xFC 0x05 0x4A -> 0x85
// Sample CRC32 1KB -> 0xE8

u8 crc8_rdallas(const void* buf, int len)
{
	u8 crc = CRC8RDALLAS_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc = crc8_dallas_tab[crc] ^ *s++;
	return Reverse8(crc);
}

u8 crc8_rdallas_slow(const void* buf, int len)
{
	u8 crc = CRC8RDALLAS_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc = crc8_rdallas_slow_1(crc, *s++);
	return crc;
}

// ============================================================================
//                      Calculate CRC8-Dallas, 1 byte
// ============================================================================

#define CRC8DALLAS_POLY 0x8c // reversed 0x31
u8 crc8_dallas_1(u8 crc, u8 data)
{
	return crc8_dallas_tab[crc ^ data];
}	

u8 crc8_dallas_slow_1(u8 crc, u8 data)
{
	int i;
	crc ^= data;

	for (i = 8; i > 0; i--)
	{
		if ((crc & 1) != 0)
			crc = (crc >> 1) ^ CRC8DALLAS_POLY;
		else
			crc >>= 1;
	}
	return crc;
}

// ============================================================================
//                         Calculate CRC8-Dallas
// ============================================================================
// Sample "123456789" -> 0xA1
// Sample: 0xFC 0x05 0x4A -> 0xF1
// Sample CRC32 1KB -> 0x1E

u8 crc8_dallas(const void* buf, int len)
{
	u8 crc = CRC8DALLAS_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc = crc8_dallas_tab[crc ^ *s++];
	return crc;
}

u8 crc8_dallas_slow(const void* buf, int len)
{
	u8 crc = CRC8DALLAS_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc = crc8_dallas_slow_1(crc, *s++);
	return crc;
}

// ============================================================================
//                       Calculate CRC8-SUM, 1 byte
// ============================================================================

u8 crc8_sum_1(u8 crc, u8 data)
{
	return (u8)(crc + data);
}

// ============================================================================
//                         Calculate CRC8-SUM
// ============================================================================
// Sample "123456789" -> 0xDD
// Sample: 0xFC 0x05 0x4A -> 0x4B
// Sample CRC32 1KB -> 0x00

u8 crc8_sum(const void* buf, int len)
{
	u8 crc = CRCSUM8_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc += *s++;
	return crc;
}

// ============================================================================
//                      Calculate CRC16-SUM, 1 byte
// ============================================================================

u16 crc16_sum_1(u16 crc, u8 data)
{
	return (u16)(crc + data);
}

// ============================================================================
//                         Calculate CRC16-SUM
// ============================================================================
// Sample "123456789" -> 0x01DD
// Sample: 0xFC 0x05 0x4A -> 0x014B
// Sample CRC32 1KB -> 0xFE00

u16 crc16_sum(const void* buf, int len)
{
	u16 crc = CRCSUM16_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc += *s++;
	return crc;
}

// ============================================================================
//                      Calculate CRC32-SUM, 1 byte
// ============================================================================

u32 crc32_sum_1(u32 crc, u8 data)
{
	return (u32)(crc + data);
}

// ============================================================================
//                         Calculate CRC32-SUM
// ============================================================================
// Sample "123456789" -> 0x000001DD
// Sample: 0xFC 0x05 0x4A -> 0x0000014B
// Sample CRC32 1KB -> 0x0000FE00

u32 crc32_sum(const void* buf, int len)
{
	u32 crc = CRCSUM32_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc += *s++;
	return crc;
}

// ============================================================================
//                         Calculate CRC-SHT, 1 byte
// ============================================================================

u8 crc_sht_1(u8 crc, u8 data)
{
	return crc_sht_tab[data ^ crc];
}

// ============================================================================
//                            Calculate CRC-SHT
//         (SHT1x and SHT7x series of temperature and humidity sensors)
// ============================================================================
// Sample "123456789" -> 0xA2
// Sample: 0xFC 0x05 0x4A -> 0x10
// Sample CRC32 1KB -> 0x48

u8 crc_sht(const void* buf, int len)
{
	u8 crc = CRCSHT_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--) crc = crc_sht_tab[(*s++) ^ crc];
	return crc;
}

// ============================================================================
//                         Calculate CRC-XOR, 1 byte
// ============================================================================

u16 crc_xor_1(u16 crc, u8 data)
{
	crc ^= data;
	return rol16(crc, 1);
}

// ============================================================================
//                           Calculate CRC-XOR
// ============================================================================
// Sample "123456789" -> 0x406A
// Sample: 0xFC 0x05 0x4A -> 0x0760
// Sample CRC32 1KB -> 0x0000

u16 crc_xor(const void* buf, int len)
{
	u16 crc = CRCXOR_INIT;
	const u8* s = (const u8*)buf;
	for (; len > 0; len--)
	{
		crc ^= *s++;
		crc = rol16(crc, 1);
	}
	return crc;
}

// ============================================================================
//                 Check CRC-32 table (returns False on error)
// ============================================================================
// Polynom: x32 + x26 + x23 + x22 + x16 + x12 + x11 + x10 + x8 + x7 + x5 + x4 + x2 + x + 1
//		used in PNG LZ77 chunks, http://www.w3.org/TR/PNG/#D-CRCAppendix

Bool crc32_check()
{
	int i, j;
	u32 k;

	for(i = 0; i < 256; i++)
	{
		k = i;
		for (j = 8; j > 0; j--)
		{
			if ((k & 1) == 0)
				k >>= 1;
			else
				k = (k >> 1) ^ CRC32_POLY;
		}
		if (crc32_tab[i] != k) return False;
	}
	return True;
}

// ============================================================================
//                Check CRC-16 table (returns False on error)
// ============================================================================
// Polynom: x16 + x15 + x2 + 1
// http://www.lammertbies.nl/comm/software/index.html
// used in disk controllers and in Dallas Maxim 1-Wire bus

Bool crc16_check()
{
	int i, j;
	u16 crc, c;

	for(i = 0; i < 256; i++)
	{
		crc = 0;
		c = (u16)i;
		for (j = 8; j > 0; j--)
		{
			if (((crc ^ c) & 1) == 0)
				crc >>= 1;
			else
				crc = (u16)((crc >> 1) ^ CRC16_POLY);
			c >>= 1;
		}
		if (crc16_tab[i] != crc) return False;
	}
	return True;
}

// ============================================================================
//              Check CRC-Kermit table (returns False on error)
// ============================================================================
// Polynom: x16 + x12 + x5 + 1

Bool crc_kermit_check()
{
	int i, j;
	u16 crc, c;

	for(i = 0; i < 256; i++)
	{
		crc = 0;
		c = i;
		for (j = 8; j > 0; j--)
		{
			if (((crc ^ c) & 1) == 0)
				crc >>= 1;
			else
				crc = (u16)((crc >> 1) ^ CRCKERMIT_POLY);
			c >>= 1;
		}
		if (crc_kermit_tab[i] != crc) return False;
	}
	return True;
}

// ============================================================================
//              Check CRC-DNP table (returns False on error)
// ============================================================================
// Polynom: x16 + x13 + x12 + x11 + x10 + x8 + x6 + x5 + x2 + 1

Bool crc_dnp_check()
{
	int i, j;
	u16 crc, c;

	for(i = 0; i < 256; i++)
	{
		crc = 0;
		c = i;
		for (j = 8; j > 0; j--)
		{
			if (((crc ^ c) & 1) == 0)
				crc >>= 1;
			else
				crc = (u16)((crc >> 1) ^ CRCDNP_POLY);
			c >>= 1;
		}
		if (crc_dnp_tab[i] != crc) return False;
	}
	return True;
}

// ============================================================================
//             Check CRC-CCITT table (returns False on error)
// ============================================================================
// Polynom: x16 + x12 + x5 + 1

Bool crc_ccitt_check()
{
	int i, j;
	u16 crc, c;

	for(i = 0; i < 256; i++)
	{
		crc = 0;
		c = (u16)(i << 8);
		for (j = 8; j > 0; j--)
		{
			if (((crc ^ c) & 0x8000) == 0)
				crc <<= 1;
			else
				crc = (u16)((crc << 1) ^ CRCCCITT_POLY);
			c <<= 1;
		}
		if (crc_ccitt_tab[i] != crc) return False;
	}
	return True;
}

// Check CRC-CCITT fast method (return False on error)
/*
Bool crc_ccitt_fast_check()
{
	u16 crc = 0;
	u8 data = 0;
	for (;;)
	{
		if (crc_ccitt_1(crc, data) != (crc_ccitt_fast_1(crc, data))) return False;
		data++;
		if (data == 0)
		{
			crc++;
			if (crc == 0) break;
		}
	}
	return True;
}
*/
// ============================================================================
//               Check CRC8-Dallas table (returns False on error)
// ============================================================================
// Polynom: x^8 + x^5 + x^4 + 1 (Dallas Maxim 1-Wire bus, used for CRC8-RDALLAS, too)

Bool crc8_dallas_check()
{
	int i, j;
	u8 crc;

	for (i = 0; i < 256; i++)
	{
		crc = (u8)i;

		for (j = 8; j > 0; j--)
		{
			if ((crc & 1) != 0)
				crc = (crc >> 1) ^ CRC8DALLAS_POLY;
			else
				crc >>= 1;
		}
		if (crc8_dallas_tab[i] != crc) return False;
	}
	return True;
}

// ============================================================================
//   Check all CRC methods (takes few seconds, returns error code or 0 if OK)
// ============================================================================
// Online calculator: http://www.lammertbies.nl/comm/info/crc-calculation.html
// http://crccalc.com/

#define CRCPATN1 9	// length of CRC pattern 1
const char CrcPattern1[] = "123456789";

#define CRCPATN2 3	// length of CRC pattern 2
const u8 CrcPattern2[CRCPATN2] = { 0xFC, 0x05, 0x4A };

#define CRCPATN3 1024	// length of CRC pattern 3
#define CrcPattern3 crc32_tab

int crc_check_all()
{
	// Check CRC-32 table
	if (!crc32_check()) return 10;

	// Check CRC-16 table
	if (!crc16_check()) return 20;

	// Check CRC-Kermit table
	if (!crc_kermit_check()) return 30;

	// Check CRC-DNP table
	if (!crc_dnp_check()) return 40;

	// Check CRC-CCITT table
	if (!crc_ccitt_check()) return 50;

	// Check CRC-CCITT fast method (takes too long, a few seconds)
//	if (!crc_ccitt_fast_check()) return 60;

	// Check CRC8-Dallas table
	if (!crc8_dallas_check()) return 70;

	// Check CRC-32 pattern
	if (crc32(CrcPattern1, CRCPATN1) != 0xCBF43926) return 11;
	if (crc32(CrcPattern2, CRCPATN2) != 0xA8E10F6D) return 12;
	if (crc32(CrcPattern3, CRCPATN3) != 0x6FCF9E13) return 13;

	if (crc32_slow(CrcPattern1, CRCPATN1) != 0xCBF43926) return 14;
	if (crc32_slow(CrcPattern2, CRCPATN2) != 0xA8E10F6D) return 15;
	if (crc32_slow(CrcPattern3, CRCPATN3) != 0x6FCF9E13) return 16;

	// Check CRC-16 IBM pattern
	if (crc_ibm(CrcPattern1, CRCPATN1) != 0xBB3D) return 21;
	if (crc_ibm(CrcPattern2, CRCPATN2) != 0x9742) return 21;
	if (crc_ibm(CrcPattern3, CRCPATN3) != 0x0407) return 21;

	if (crc_ibm_slow(CrcPattern1, CRCPATN1) != 0xBB3D) return 22;
	if (crc_ibm_slow(CrcPattern2, CRCPATN2) != 0x9742) return 22;
	if (crc_ibm_slow(CrcPattern3, CRCPATN3) != 0x0407) return 22;

	// Check CRC-16 Modbus pattern
	if (crc_modbus(CrcPattern1, CRCPATN1) != 0x4B37) return 23;
	if (crc_modbus(CrcPattern2, CRCPATN2) != 0x5733) return 23;
	if (crc_modbus(CrcPattern3, CRCPATN3) != 0xD0B9) return 23;

	if (crc_modbus_slow(CrcPattern1, CRCPATN1) != 0x4B37) return 24;
	if (crc_modbus_slow(CrcPattern2, CRCPATN2) != 0x5733) return 24;
	if (crc_modbus_slow(CrcPattern3, CRCPATN3) != 0xD0B9) return 24;

	// Check CRC-Kermit pattern
	if (crc_kermit(CrcPattern1, CRCPATN1) != 0x8921) return 31;
	if (crc_kermit(CrcPattern2, CRCPATN2) != 0x71BA) return 32;
	if (crc_kermit(CrcPattern3, CRCPATN3) != 0x5005) return 33;

	if (crc_kermit_slow(CrcPattern1, CRCPATN1) != 0x8921) return 34;
	if (crc_kermit_slow(CrcPattern2, CRCPATN2) != 0x71BA) return 35;
	if (crc_kermit_slow(CrcPattern3, CRCPATN3) != 0x5005) return 36;

	// Check CRC-DNP pattern
	if (crc_dnp(CrcPattern1, CRCPATN1) != 0x82EA) return 41;
	if (crc_dnp(CrcPattern2, CRCPATN2) != 0x8AE7) return 42;
	if (crc_dnp(CrcPattern3, CRCPATN3) != 0x69EE) return 43;

	if (crc_dnp_slow(CrcPattern1, CRCPATN1) != 0x82EA) return 44;
	if (crc_dnp_slow(CrcPattern2, CRCPATN2) != 0x8AE7) return 45;
	if (crc_dnp_slow(CrcPattern3, CRCPATN3) != 0x69EE) return 46;

	// Check CRC-XModem pattern
	if (crc_xmodem(CrcPattern1, CRCPATN1) != 0x31C3) return 51;
	if (crc_xmodem(CrcPattern2, CRCPATN2) != 0x8048) return 52;
	if (crc_xmodem(CrcPattern3, CRCPATN3) != 0xF982) return 53;

	if (crc_xmodem_fast(CrcPattern1, CRCPATN1) != 0x31C3) return 54;
	if (crc_xmodem_fast(CrcPattern2, CRCPATN2) != 0x8048) return 55;
	if (crc_xmodem_fast(CrcPattern3, CRCPATN3) != 0xF982) return 56;

	if (crc_xmodem_slow(CrcPattern1, CRCPATN1) != 0x31C3) return 57;
	if (crc_xmodem_slow(CrcPattern2, CRCPATN2) != 0x8048) return 58;
	if (crc_xmodem_slow(CrcPattern3, CRCPATN3) != 0xF982) return 59;

	// Check CRC-CCITT (0xFFFF) pattern
	if (crc_ccitt(CrcPattern1, CRCPATN1) != 0x29B1) return 61;
	if (crc_ccitt(CrcPattern2, CRCPATN2) != 0x4CD4) return 62;
	if (crc_ccitt(CrcPattern3, CRCPATN3) != 0x4EED) return 63;

	if (crc_ccitt_fast(CrcPattern1, CRCPATN1) != 0x29B1) return 64;
	if (crc_ccitt_fast(CrcPattern2, CRCPATN2) != 0x4CD4) return 65;
	if (crc_ccitt_fast(CrcPattern3, CRCPATN3) != 0x4EED) return 66;

	if (crc_ccitt_slow(CrcPattern1, CRCPATN1) != 0x29B1) return 67;
	if (crc_ccitt_slow(CrcPattern2, CRCPATN2) != 0x4CD4) return 68;
	if (crc_ccitt_slow(CrcPattern3, CRCPATN3) != 0x4EED) return 69;

	// Check CRC-CCITT (0x1D0F) pattern
	if (crc_ccitt_b(CrcPattern1, CRCPATN1) != 0xE5CC) return 71;
	if (crc_ccitt_b(CrcPattern2, CRCPATN2) != 0x9144) return 72;
	if (crc_ccitt_b(CrcPattern3, CRCPATN3) != 0xE351) return 73;

	if (crc_ccitt_b_fast(CrcPattern1, CRCPATN1) != 0xE5CC) return 74;
	if (crc_ccitt_b_fast(CrcPattern2, CRCPATN2) != 0x9144) return 75;
	if (crc_ccitt_b_fast(CrcPattern3, CRCPATN3) != 0xE351) return 76;

	if (crc_ccitt_b_slow(CrcPattern1, CRCPATN1) != 0xE5CC) return 77;
	if (crc_ccitt_b_slow(CrcPattern2, CRCPATN2) != 0x9144) return 78;
	if (crc_ccitt_b_slow(CrcPattern3, CRCPATN3) != 0xE351) return 79;

	// Check CRC-Sick pattern
	if (crc_sick(CrcPattern1, CRCPATN1) != 0x56A6) return 81;
	if (crc_sick(CrcPattern2, CRCPATN2) != 0xB57E) return 82;
	if (crc_sick(CrcPattern3, CRCPATN3) != 0xC7FE) return 83;

	// Check CRC8-RDallas pattern
	if (crc8_rdallas(CrcPattern1, CRCPATN1) != 0x7C) return 91;
	if (crc8_rdallas(CrcPattern2, CRCPATN2) != 0x85) return 92;
	if (crc8_rdallas(CrcPattern3, CRCPATN3) != 0xE8) return 93;

	if (crc8_rdallas_slow(CrcPattern1, CRCPATN1) != 0x7C) return 94;
	if (crc8_rdallas_slow(CrcPattern2, CRCPATN2) != 0x85) return 95;
	if (crc8_rdallas_slow(CrcPattern3, CRCPATN3) != 0xE8) return 96;

	// Check CRC8-Dallas pattern
	if (crc8_dallas(CrcPattern1, CRCPATN1) != 0xA1) return 101;
	if (crc8_dallas(CrcPattern2, CRCPATN2) != 0xF1) return 102;
	if (crc8_dallas(CrcPattern3, CRCPATN3) != 0x1E) return 103;

	if (crc8_dallas_slow(CrcPattern1, CRCPATN1) != 0xA1) return 104;
	if (crc8_dallas_slow(CrcPattern2, CRCPATN2) != 0xF1) return 105;
	if (crc8_dallas_slow(CrcPattern3, CRCPATN3) != 0x1E) return 106;

	// Check CRC8-SUM pattern
	if (crc8_sum(CrcPattern1, CRCPATN1) != 0xDD) return 111;
	if (crc8_sum(CrcPattern2, CRCPATN2) != 0x4B) return 112;
	if (crc8_sum(CrcPattern3, CRCPATN3) != 0x00) return 113;

	// Check CRC16-SUM pattern
	if (crc16_sum(CrcPattern1, CRCPATN1) != 0x01DD) return 114;
	if (crc16_sum(CrcPattern2, CRCPATN2) != 0x014B) return 115;
	if (crc16_sum(CrcPattern3, CRCPATN3) != 0xFE00) return 116;

	// Check CRC32-SUM pattern
	if (crc32_sum(CrcPattern1, CRCPATN1) != 0x000001DD) return 117;
	if (crc32_sum(CrcPattern2, CRCPATN2) != 0x0000014B) return 118;
	if (crc32_sum(CrcPattern3, CRCPATN3) != 0x0001FE00) return 119;

	// Check CRC-SHT pattern
	if (crc_sht(CrcPattern1, CRCPATN1) != 0xA2) return 121;
	if (crc_sht(CrcPattern2, CRCPATN2) != 0x10) return 122;
	if (crc_sht(CrcPattern3, CRCPATN3) != 0x48) return 123;

	// Check CRC-XOR pattern
	if (crc_xor(CrcPattern1, CRCPATN1) != 0x406A) return 131;
	if (crc_xor(CrcPattern2, CRCPATN2) != 0x0760) return 132;
	if (crc_xor(CrcPattern3, CRCPATN3) != 0x0000) return 133;

	// Check CRC-32 - Raspberry Pico alternative
	if (crc32_Pico((const u8*)CrcPattern1, CRCPATN1) != 0x0376E6E7) return 141;
	if (crc32_Pico((const u8*)CrcPattern2, CRCPATN2) != 0x8E10D720) return 142;
	if (crc32_Pico((const u8*)CrcPattern3, CRCPATN3) != 0x5796333D) return 143;

	return 0;
}
